/****************************************************************************/
/*                                                                          */
/*  VolVis is a volume visualization system for investigating, manipulating */
/*  and rendering geometric and volumetric data.                            */
/*                                                                          */
/*  Copyright (C) 1993 by the Research Foundation of the State University   */
/*                            of New York                                   */
/*                                                                          */
/*  This program is free software; you can redistribute it and/or modify    */
/*  it under the terms of the GNU General Public License as published by    */
/*  the Free Software Foundation; either version 1, or (at your option)     */
/*  any later version.                                                      */
/*                                                                          */
/*  This program is distributed in the hope that it will be useful,         */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/*  GNU General Public License for more details.                            */
/*                                                                          */
/*  You should have received a copy of the GNU General Public License       */
/*  along with this program; if not, write to the Free Software             */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.               */
/*                                                                          */
/*  For information on VolVis, contact us at:                               */
/*                                                                          */
/*                volvis@cs.sunysb.edu                         (email)      */
/*                                                                          */
/*                Lisa Sobierajski & Ricardo Avila             (US Mail)    */
/*                Department of Computer Science                            */
/*                State University of New York at Stony Brook               */
/*                Stony Brook, New York  11794-4400                         */
/*                                                                          */
/****************************************************************************/




/*
 *			File: MOTIF_oco_ui.c
 *		      Author: Lisa Sobierajski 
 *			Date: 03/07/92
 *		 Description: Motif Windowing Routines Of The Object Properties 
 *	Modification History:
 *
 *		Who?		When?		Why?
 *	--------------------------------------------------------------------
 *
 */

#include <stdio.h>
#include <math.h>
#include <Xm/Xm.h>
#include <Xm/RowColumn.h>
#include <X11/StringDefs.h>
#include <Xm/PushB.h>
#include <Xm/PushBG.h>
#include <Xm/BulletinB.h>
#include <Xm/LabelG.h>
#include <Xm/Text.h>
#include <Xm/TextF.h>
#include <Xm/DrawingA.h>
#include <Xm/Frame.h>
#include <Xm/Scale.h>
#include <Xm/ToggleB.h>


#include "C_volvis.h"
#include "MOTIF_windefs.h"
#include "C_help.h"
#include "C_oco.h"


void oco_create_scandata8bit_prop_display( volume_index )
int			volume_index;
{
	extern void	close_callback();
	extern void	help_callback();
	extern void	oco_s_opacity_input();
	extern void	oco_s_opacity_expose();

	extern Widget			vv_toplevel;	
	extern C_WindowDefs		window_defs;
	extern C_World			world;
	extern C_ObjectControlInfo	oco_info;

	void		oco_update_seg_type();
	void		oco_set_data_props();
	void		oco_reset_data_props();
	void		oco_set_max_opacity();
	void		oco_seg_function_toggle_callback();
	void		oco_set_modified();
	extern void 	oco_display_x_slices_callback();
	extern void 	oco_display_y_slices_callback();
	extern void 	oco_display_z_slices_callback();
	

	Arg		args[20];
	int		n;
	int		i;

	Widget		label;
	Widget		slider;
	Widget		frame;
	Widget		rowcolumn;
	Widget		bboard;
	Widget		seg_bboard;
	Widget		parc_bboard;
	Widget		button;
	Widget		pulldown;
	Widget		radiobox;
	XmString	label_string;
	char		string[30];
	OcoSegInfo	*seg_info;
	OcoVolumeInfo	*vinfo;
	C_Volume	*volume;
	Dimension	y_val;
	Dimension	height;
	C_Color		color;
	int		pixel;


	vinfo = oco_info.volume_info[volume_index];
	volume = world.volume[volume_index];

	seg_info = C_New( OcoSegInfo );
	seg_info->volume_index = volume_index;

	vinfo->data.seg_info = seg_info;

	n=0;
        XtSetArg( args[n], XmNautoUnmanage, False ); n++;
	sprintf( string, "Modify %s Segmentation", volume->volume_name );
        label_string = XmStringCreate(string, XmSTRING_DEFAULT_CHARSET );
	XtSetArg( args[n], XmNdialogTitle, label_string ); n++;
        C_Set_Color_Dialog( args, n, window_defs );
        vinfo->data.volume_data_displayed = 
		XmCreateBulletinBoardDialog( 
			vv_toplevel, "dataprops", args, n );

	n = 0;
	XtSetArg( args[n], XmNx, 0 ); n++;
	XtSetArg( args[n], XmNy, 0 ); n++;
	C_Set_Color_Frame( args, n, window_defs );
	frame = XmCreateFrame(vinfo->data.volume_data_displayed, 
			"frame", args, n );
	XtManageChild( frame );

        n = 0;
	C_Set_Color_Bboard( args, n, window_defs );
        seg_bboard = 
		XmCreateBulletinBoard(frame, "bboard", args, n);
        XtManageChild (seg_bboard);

        n = 0;
        C_Set_Color_PullDown( args, n, window_defs );
        pulldown =
                XmCreatePulldownMenu(seg_bboard, "pulldown", args, n);

        n = 0;
        label_string = XmStringCreate ("IsoSurface", XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNlabelString, label_string); n++;
        seg_info->seg_type[C_ISO_SURFACE] =
                XmCreatePushButtonGadget(pulldown, "IsoSurface", args, n);
        XmStringFree (label_string);
        XtAddCallback (seg_info->seg_type[C_ISO_SURFACE], XmNactivateCallback,
                       oco_update_seg_type, seg_info);

        n = 0;
        label_string = XmStringCreate ("Opacity Array (Scalar)", 
			XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNlabelString, label_string); n++;
        seg_info->seg_type[C_SCALAR_OPACITY] =
                XmCreatePushButtonGadget(pulldown, "SOpacity Array",args,n);
        XmStringFree (label_string);
        XtAddCallback (seg_info->seg_type[C_SCALAR_OPACITY], 
		       XmNactivateCallback,
                       oco_update_seg_type, seg_info);

	XtManageChildren( seg_info->seg_type, 2 );

        n = 0;
        label_string = XmStringCreate ("Segmentation Type:",
                                      XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNlabelString, label_string); n++;
        XtSetArg (args[n], XmNsubMenuId, pulldown); n++;
        XtSetArg( args[n], XmNx, 10 ); n++;
        XtSetArg( args[n], XmNy, 10 ); n++;
        C_Set_Color_OptionMenu( args, n, window_defs );

	XtSetArg (args[n], XmNmenuHistory, 
	    seg_info->seg_type[volume->data.scalar_data_8bit->seg_type]); n++;

        seg_info->option_menu = XmCreateOptionMenu(seg_bboard,
                                "oco_option_menu", args, n);

        XtManageChild(seg_info->option_menu);


	/********************************************************/		
	/*	Create a Frame For Isosurface Segmentation 	*/
	/********************************************************/		
	n = 0;
	XtSetArg( args[n], XmNx, 50 ); n++;
	XtSetArg( args[n], XmNy, 50 ); n++;
	C_Set_Color_Frame( args, n, window_defs );
	seg_info->isosurface_frame = 
			XmCreateFrame(seg_bboard, "frame", args, n );

        n = 0;
	C_Set_Color_Bboard( args, n, window_defs );
        bboard = 
		XmCreateBulletinBoard(seg_info->isosurface_frame, 
				      "bboard", args, n);
        XtManageChild (bboard);

       	label_string = XmStringCreate("Isovalue: ", 
		              XmSTRING_DEFAULT_CHARSET );
       	n=0;
       	XtSetArg( args[n], XmNx, 10 ); n++;
       	XtSetArg( args[n], XmNy, 20 ); n++;
       	XtSetArg( args[n], XmNlabelString, label_string ); n++;
       	label = XmCreateLabelGadget( bboard, "label", args, n );
       	XtManageChild( label );
       	XmStringFree( label_string );

       	n=0;
       	XtSetArg( args[n], XmNx, 80 ); n++;
       	XtSetArg( args[n], XmNy, 20 ); n++;
       	XtSetArg( args[n], XmNcolumns, 10 ); n++;
	sprintf( string, "%f", volume->data.scalar_data_8bit->seg.isovalue );
       	XtSetArg( args[n], XmNvalue, string ); n++;
       	C_Set_Color_Text( args, n, window_defs );
       	seg_info->iso_text = 
			XmCreateTextField( bboard, 
					"text", args, n );
	XtManageChild(  seg_info->iso_text );

	XtAddCallback( seg_info->iso_text, XmNvalueChangedCallback, 
		       oco_set_modified, seg_info );
		       
 
	/********************************************************/		
	/*	Create a Frame For Scalar Opacity Segmentation 	*/
	/********************************************************/		

	seg_info->s_opacity_max = 0.1;

	for (i = 0; i < 256; i++)
	{
	    seg_info->s_opacity[i] = 
	       volume->data.scalar_data_8bit->seg.s_opacity->opacity[i] * 
			(255.0 / seg_info->s_opacity_max);	
	}

        if ( !vinfo->color.volume_color_displayed )
        {
          for (i = 0; i < 256; i++)
          {
            vinfo->color.color_array_red[i] =
               volume->color_data.color_array->color[i].red;
            vinfo->color.color_array_green[i] =
               volume->color_data.color_array->color[i].green;
            vinfo->color.color_array_blue[i] =
               volume->color_data.color_array->color[i].blue;
          }
        }
 
	n = 0;
	XtSetArg( args[n], XmNx, 0 ); n++;
	XtSetArg( args[n], XmNy, 50 ); n++;
	C_Set_Color_Frame( args, n, window_defs );
	seg_info->s_opacity_frame = 
			XmCreateFrame(seg_bboard, "frame", args, n );

        n = 0;
	C_Set_Color_Bboard( args, n, window_defs );
        bboard = 
		XmCreateBulletinBoard(seg_info->s_opacity_frame, 
				      "bboard", args, n);
	XtManageChild( bboard );


       	label_string = XmStringCreate("Max Opacity: ", 
		              XmSTRING_DEFAULT_CHARSET );
       	n=0;
       	XtSetArg( args[n], XmNx, 10 ); n++;
       	XtSetArg( args[n], XmNy, 20 ); n++;
       	XtSetArg( args[n], XmNlabelString, label_string ); n++;
       	label = XmCreateLabelGadget( bboard, "Max", args, n );
       	XtManageChild( label );
       	XmStringFree( label_string );

       	n=0;
       	XtSetArg( args[n], XmNx, 110 ); n++;
       	XtSetArg( args[n], XmNy, 10 ); n++;
       	XtSetArg( args[n], XmNcolumns, 8 ); n++;
	sprintf( string, "%f", seg_info->s_opacity_max );
       	XtSetArg( args[n], XmNvalue, string ); n++;
       	C_Set_Color_Text( args, n, window_defs );
       	seg_info->s_opacity_max_text = 
			XmCreateTextField( bboard, "text", args, n );
	XtManageChild(  seg_info->s_opacity_max_text );

	XtAddCallback( seg_info->s_opacity_max_text, XmNactivateCallback,
		       oco_set_max_opacity, seg_info );

        label_string = XmStringCreate("Edit Function: ",
                        XmSTRING_DEFAULT_CHARSET );
        n=0;
        XtSetArg( args[n], XmNx, 5 ); n++;
        XtSetArg( args[n], XmNy, 50 ); n++;
        XtSetArg( args[n], XmNlabelString, label_string ); n++;
        label = XmCreateLabelGadget( bboard, "label",
                                                args, n );
        XtManageChild( label );
        XmStringFree( label_string );
 
        n = 0;
        XtSetArg( args[n], XmNorientation, XmHORIZONTAL ); n++;
        C_Set_Color_RadioBox( args, n, window_defs );
        XtSetArg( args[n], XmNx, 120 ); n++;
        XtSetArg( args[n], XmNy, 40 ); n++;
        radiobox = XmCreateRadioBox( bboard, "radiobox", args, n );
        XtManageChild( radiobox );
 
        n = 0;
        label_string = XmStringCreate ("O", XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNlabelString, label_string); n++;
        XtSetArg( args[n], XmNset, TRUE ); n++;
        C_Set_Color_ToggleButton( args, n, window_defs );
        seg_info->s_opacity_edit_toggle[C_OPACITY_EDIT] =
                        XtCreateWidget ("R", xmToggleButtonWidgetClass,
                                    radiobox, args, n);
        XtManageChild(
                seg_info->s_opacity_edit_toggle[C_OPACITY_EDIT] );
        XmStringFree (label_string);
        XtAddCallback (seg_info->s_opacity_edit_toggle[C_OPACITY_EDIT],
                       XmNvalueChangedCallback,
                       oco_seg_function_toggle_callback, seg_info);
	seg_info->s_opacity_edit_type = C_OPACITY_EDIT;
 
 
        n = 0;
        label_string = XmStringCreate ("R", XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNlabelString, label_string); n++;
        XtSetArg( args[n], XmNset, FALSE ); n++;
        C_Set_Color_ToggleButton( args, n, window_defs );
        seg_info->s_opacity_edit_toggle[C_RED_COLOR_EDIT] =
                        XtCreateWidget ("R", xmToggleButtonWidgetClass,
                                    radiobox, args, n);
        XtManageChild(
                seg_info->s_opacity_edit_toggle[C_RED_COLOR_EDIT] );
        XmStringFree (label_string);
        XtAddCallback (seg_info->s_opacity_edit_toggle[C_RED_COLOR_EDIT],
                       XmNvalueChangedCallback,
                       oco_seg_function_toggle_callback, seg_info);
 
        n = 0;
        label_string = XmStringCreate ("G", XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNlabelString, label_string); n++;
        XtSetArg( args[n], XmNset, FALSE ); n++;
        C_Set_Color_ToggleButton( args, n, window_defs );
        seg_info->s_opacity_edit_toggle[C_GREEN_COLOR_EDIT] =
                        XtCreateWidget ("G", xmToggleButtonWidgetClass,
                                    radiobox, args, n);
        XtManageChild(
                seg_info->s_opacity_edit_toggle[C_GREEN_COLOR_EDIT] );
        XmStringFree (label_string);
        XtAddCallback (seg_info->s_opacity_edit_toggle[C_GREEN_COLOR_EDIT],
 
                       XmNvalueChangedCallback,
                       oco_seg_function_toggle_callback, seg_info);
 
        n = 0;
        label_string = XmStringCreate ("B", XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNlabelString, label_string); n++;
        XtSetArg( args[n], XmNset, FALSE ); n++;
        C_Set_Color_ToggleButton( args, n, window_defs );
        seg_info->s_opacity_edit_toggle[C_BLUE_COLOR_EDIT] =
                        XtCreateWidget ("B", xmToggleButtonWidgetClass,
                                    radiobox, args, n);
        XtManageChild(
                seg_info->s_opacity_edit_toggle[C_BLUE_COLOR_EDIT] );
        XmStringFree (label_string);
        XtAddCallback (seg_info->s_opacity_edit_toggle[C_BLUE_COLOR_EDIT],
                       XmNvalueChangedCallback,
                       oco_seg_function_toggle_callback, seg_info);
 

       	label_string = XmStringCreate("X Location:", 
		              XmSTRING_DEFAULT_CHARSET );
       	n=0;
       	XtSetArg( args[n], XmNx, 10 ); n++;
       	XtSetArg( args[n], XmNy, 90 ); n++;
       	XtSetArg( args[n], XmNlabelString, label_string ); n++;
       	label = XmCreateLabelGadget( bboard, "label", args, n );
       	XtManageChild( label );
       	XmStringFree( label_string );

       	label_string = XmStringCreate("---", 
		              XmSTRING_DEFAULT_CHARSET );
       	n=0;
       	XtSetArg( args[n], XmNx, 100 ); n++;
       	XtSetArg( args[n], XmNy, 90 ); n++;
       	XtSetArg( args[n], XmNlabelString, label_string ); n++;
       	seg_info->s_opacity_x_label = 
			XmCreateLabelGadget( bboard, "label", args, n );
       	XtManageChild( seg_info->s_opacity_x_label );
       	XmStringFree( label_string );



       	label_string = XmStringCreate("Y Location:", 
		              XmSTRING_DEFAULT_CHARSET );
       	n=0;
       	XtSetArg( args[n], XmNx, 10 ); n++;
       	XtSetArg( args[n], XmNy, 110 ); n++;
       	XtSetArg( args[n], XmNlabelString, label_string ); n++;
       	label = XmCreateLabelGadget( bboard, "label", args, n );
       	XtManageChild( label );
       	XmStringFree( label_string );

       	label_string = XmStringCreate("---------------------------", 
		              XmSTRING_DEFAULT_CHARSET );
       	n=0;
       	XtSetArg( args[n], XmNx, 100 ); n++;
       	XtSetArg( args[n], XmNy, 110 ); n++;
       	XtSetArg( args[n], XmNlabelString, label_string ); n++;
       	seg_info->s_opacity_y_label = 
			XmCreateLabelGadget( bboard, "label", args, n );
       	XtManageChild( seg_info->s_opacity_y_label );
       	XmStringFree( label_string );



       	label_string = XmStringCreate("Opacity: ", 
		              XmSTRING_DEFAULT_CHARSET );
       	n=0;
       	XtSetArg( args[n], XmNx, 10 ); n++;
       	XtSetArg( args[n], XmNy, 130 ); n++;
       	XtSetArg( args[n], XmNlabelString, label_string ); n++;
       	label = XmCreateLabelGadget( bboard, "label", args, n );
       	XtManageChild( label );
       	XmStringFree( label_string );

       	label_string = XmStringCreate("------", 
		              XmSTRING_DEFAULT_CHARSET );
       	n=0;
       	XtSetArg( args[n], XmNx, 80 ); n++;
       	XtSetArg( args[n], XmNy, 130 ); n++;
       	XtSetArg( args[n], XmNlabelString, label_string ); n++;
       	seg_info->s_opacity_opacity_label = 
			XmCreateLabelGadget( bboard, "label", args, n );
       	XtManageChild( seg_info->s_opacity_opacity_label );
       	XmStringFree( label_string );



       	label_string = XmStringCreate("R: ", 
		              XmSTRING_DEFAULT_CHARSET );
       	n=0;
       	XtSetArg( args[n], XmNx, 140 ); n++;
       	XtSetArg( args[n], XmNy, 130 ); n++;
       	XtSetArg( args[n], XmNlabelString, label_string ); n++;
       	label = XmCreateLabelGadget( bboard, "label", args, n );
       	XtManageChild( label );
       	XmStringFree( label_string );

       	label_string = XmStringCreate("----", 
		              XmSTRING_DEFAULT_CHARSET );
       	n=0;
       	XtSetArg( args[n], XmNx, 160 ); n++;
       	XtSetArg( args[n], XmNy, 130 ); n++;
       	XtSetArg( args[n], XmNlabelString, label_string ); n++;
       	seg_info->s_opacity_r_label = 
			XmCreateLabelGadget( bboard, "label", args, n );
       	XtManageChild( seg_info->s_opacity_r_label );
       	XmStringFree( label_string );



       	label_string = XmStringCreate("G: ", 
		              XmSTRING_DEFAULT_CHARSET );
       	n=0;
       	XtSetArg( args[n], XmNx, 200 ); n++;
       	XtSetArg( args[n], XmNy, 130 ); n++;
       	XtSetArg( args[n], XmNlabelString, label_string ); n++;
       	label = XmCreateLabelGadget( bboard, "label", args, n );
       	XtManageChild( label );
       	XmStringFree( label_string );

       	label_string = XmStringCreate("----", 
		              XmSTRING_DEFAULT_CHARSET );
       	n=0;
       	XtSetArg( args[n], XmNx, 220 ); n++;
       	XtSetArg( args[n], XmNy, 130 ); n++;
       	XtSetArg( args[n], XmNlabelString, label_string ); n++;
       	seg_info->s_opacity_g_label = 
			XmCreateLabelGadget( bboard, "label", args, n );
       	XtManageChild( seg_info->s_opacity_g_label );
       	XmStringFree( label_string );



       	label_string = XmStringCreate("B: ", 
		              XmSTRING_DEFAULT_CHARSET );
       	n=0;
       	XtSetArg( args[n], XmNx, 260 ); n++;
       	XtSetArg( args[n], XmNy, 130 ); n++;
       	XtSetArg( args[n], XmNlabelString, label_string ); n++;
       	label = XmCreateLabelGadget( bboard, "label", args, n );
       	XtManageChild( label );
       	XmStringFree( label_string );

       	label_string = XmStringCreate("----", 
		              XmSTRING_DEFAULT_CHARSET );
       	n=0;
       	XtSetArg( args[n], XmNx, 280 ); n++;
       	XtSetArg( args[n], XmNy, 130 ); n++;
       	XtSetArg( args[n], XmNlabelString, label_string ); n++;
       	seg_info->s_opacity_b_label = 
			XmCreateLabelGadget( bboard, "label", args, n );
       	XtManageChild( seg_info->s_opacity_b_label );
       	XmStringFree( label_string );



	color.red = 0;
	color.green = 0;
	color.blue = 0;
	
	if ( window_defs.depth == 24 )
		pixel = C_Get_Pixel( color );
	else
		pixel = window_defs.first_color_index;

        n = 0;
        XtSetArg( args[n], XmNwidth, 256 ); n++;
        XtSetArg( args[n], XmNheight, 286 ); n++;
        XtSetArg( args[n], XmNx, 50 ); n++;
        XtSetArg( args[n], XmNy, 160 ); n++;
	XtSetArg( args[n], XmNbackground, pixel ); n++;
        seg_info->s_opacity_draw_area = 
		XmCreateDrawingArea( bboard, "draw_area", args, n );

	XtManageChild( seg_info->s_opacity_draw_area );	

	XtAddCallback( seg_info->s_opacity_draw_area, XmNexposeCallback, 
		       oco_s_opacity_expose, seg_info );

	XtAddEventHandler( seg_info->s_opacity_draw_area,
			   ButtonPressMask | ButtonReleaseMask | 
		           PointerMotionMask, False,
			   oco_s_opacity_input, seg_info );

	seg_info->s_opacity_gc = NULL;

	
	/********************************************************/		
	/*	Manage the right frame according to seg type    */
	/********************************************************/		

	switch ( volume->data.scalar_data_8bit->seg_type ) 
	{
	  case C_ISO_SURFACE: 
		XtManageChild( seg_info->isosurface_frame );
		break;

	  case C_SCALAR_OPACITY: 
		XtManageChild( seg_info->s_opacity_frame );
		break;
	}

	/********************************************************/		
	/*	Create a Frame for PARC Subdivision Level	*/
	/********************************************************/		
	n = 0;
	XtSetArg( args[n], XmNx, 0 ); n++;
	XtSetArg( args[n], XmNy, 0 ); n++;
	C_Set_Color_Frame( args, n, window_defs );
	seg_info->parc_slider_frame = 
			XmCreateFrame(vinfo->data.volume_data_displayed, 
				      "frame", args, n );
	XtManageChild( seg_info->parc_slider_frame );

        n = 0;
	C_Set_Color_Bboard( args, n, window_defs );
        parc_bboard = 
		XmCreateBulletinBoard(seg_info->parc_slider_frame, 
				      "bboard", args, n);
        XtManageChild (parc_bboard);

        label_string = XmStringCreate("PARC Subdivision Level: ",
                                      XmSTRING_DEFAULT_CHARSET );
        n=0;
        XtSetArg( args[n], XmNx, 10 ); n++;
        XtSetArg( args[n], XmNy, 25 ); n++;
        XtSetArg( args[n], XmNlabelString, label_string ); n++;
        label = XmCreateLabelGadget( parc_bboard, "label", args, n );
        XtManageChild( label );
        XmStringFree( label_string );
 
        n = 0;
        XtSetArg( args[n], XmNorientation, XmHORIZONTAL ); n++;
        XtSetArg( args[n], XmNminimum, 0 ); n++;
        XtSetArg( args[n], XmNmaximum, 7 ); n++;
        XtSetArg( args[n], XmNdecimalPoints, 0 ); n++;
        XtSetArg( args[n], XmNshowValue, TRUE ); n++;
	if ( volume->p_info )
	{
        	XtSetArg(args[n], XmNvalue, volume->p_info->sub_level); n++;
	}
	else
	{
        	XtSetArg( args[n], XmNvalue, 0 ); n++;
	}
        XtSetArg( args[n], XmNscaleMultiple, 1 ); n++;
        XtSetArg( args[n], XmNscaleWidth, 100 ); n++;
        XtSetArg( args[n], XmNx, 250 ); n++;
        XtSetArg( args[n], XmNy, 0 ); n++;
        C_Set_Color_Scale( args, n, window_defs );
        seg_info->parc_slider = XtCreateManagedWidget("slider",
                        xmScaleWidgetClass, parc_bboard, args, n );
 
	XtAddCallback( seg_info->parc_slider, XmNvalueChangedCallback,
		       oco_set_modified, seg_info );

	/********************************************************/		
	/*	Create a Frame for Slice Buttons   		*/
	/********************************************************/		
	n = 0;
	XtSetArg( args[n], XmNx, 0 ); n++;
	XtSetArg( args[n], XmNy, 0 ); n++;
	C_Set_Color_Frame( args, n, window_defs );
	seg_info->slice_buttons_frame = 
			XmCreateFrame(vinfo->data.volume_data_displayed, 
				      "frame", args, n );
	XtManageChild( seg_info->slice_buttons_frame );

	/********************************************************/		
	/*	Create a RowColumn for Slice Buttons   		*/
	/********************************************************/		
        n = 0;
	XtSetArg( args[n], XmNorientation, XmHORIZONTAL ); n++;
	C_Set_Color_RowColumn( args, n, window_defs );
        rowcolumn = 
		XmCreateRowColumn(seg_info->slice_buttons_frame, 
				  "rowcolumn", args, n);
        XtManageChild (rowcolumn);

	n = 0;
	C_Set_Color_Button( args, n, window_defs );
	button = XtCreateManagedWidget("X Slices", 
			xmPushButtonWidgetClass, rowcolumn, args, n );

	XtAddCallback( button, XmNactivateCallback, 
		       oco_display_x_slices_callback, volume_index );

	n = 0;
	C_Set_Color_Button( args, n, window_defs );
	button = XtCreateManagedWidget("Y Slices", 
			xmPushButtonWidgetClass, rowcolumn, args, n );

	XtAddCallback( button, XmNactivateCallback, 
		       oco_display_y_slices_callback, volume_index );

	n = 0;
	C_Set_Color_Button( args, n, window_defs );
	button = XtCreateManagedWidget("Z Slices", 
			xmPushButtonWidgetClass, rowcolumn, args, n );

	XtAddCallback( button, XmNactivateCallback, 
		       oco_display_z_slices_callback, volume_index );

	n = 0;
	XtSetArg( args[n], XmNx, 0 ); n++;
	XtSetArg( args[n], XmNy, 300 ); n++;
	C_Set_Color_Frame( args, n, window_defs );
	seg_info->bottom_buttons_frame = 
			XmCreateFrame(vinfo->data.volume_data_displayed, 
				      "frame", args, n );
	XtManageChild( seg_info->bottom_buttons_frame );

	/********************************************************/		
	/*	Create a RowColumn for Buttons   		*/
	/********************************************************/		
        n = 0;
	XtSetArg( args[n], XmNorientation, XmHORIZONTAL ); n++;
	C_Set_Color_RowColumn( args, n, window_defs );
        rowcolumn = 
		XmCreateRowColumn(seg_info->bottom_buttons_frame, 
				  "rowcolumn", args, n);
        XtManageChild (rowcolumn);

	n = 0;
	C_Set_Color_Button( args, n, window_defs );
	button = XtCreateManagedWidget("Accept", 
			xmPushButtonWidgetClass, rowcolumn, args, n );

	XtAddCallback( button, XmNactivateCallback, 
		       oco_set_data_props, seg_info );

	n = 0;
	C_Set_Color_Button( args, n, window_defs );
	button = XtCreateManagedWidget("Cancel", 
			xmPushButtonWidgetClass, rowcolumn, args, n );

	XtAddCallback( button, XmNactivateCallback, 
		       oco_reset_data_props, seg_info );

	n = 0;
	C_Set_Color_Button( args, n, window_defs );
	button = XtCreateManagedWidget("Close", 
			xmPushButtonWidgetClass, rowcolumn, args, n );

	XtAddCallback( button, XmNactivateCallback, 
		       oco_reset_data_props, seg_info );

	XtAddCallback( button, XmNactivateCallback, close_callback, 
		       vinfo->data.volume_data_displayed );

	n = 0;
	C_Set_Color_Button( args, n, window_defs );
	button = XtCreateManagedWidget("Help", 
			xmPushButtonWidgetClass, rowcolumn, args, n );

	XtAddCallback( button, XmNactivateCallback, help_callback, 
		       C_OCO_MODIFY_VOLUME_DATA_HELP );


        XtManageChild( vinfo->data.volume_data_displayed ); 

	switch ( volume->data.scalar_data_8bit->seg_type ) 
	{
	  case C_ISO_SURFACE: 
		oco_update_seg_type( seg_info->seg_type[C_ISO_SURFACE],
				     seg_info, NULL );
		break;

	  case C_SCALAR_OPACITY: 
		oco_update_seg_type( seg_info->seg_type[C_SCALAR_OPACITY],
				     seg_info, NULL );
		break;
	}
}


void oco_set_modified( w, seg_info, call_data )
Widget		w;
OcoSegInfo	*seg_info;
XtPointer	call_data;
{
	extern C_World			world;
	extern C_ObjectControlInfo	oco_info;

	Arg		args[1];
	char		mesg[256];
	OcoVolumeInfo	*vinfo;


	vinfo = oco_info.volume_info[seg_info->volume_index];

        sprintf( mesg, "Modify %s Segmentation -- MODIFIED",
                 world.volume[seg_info->volume_index]->volume_name );
        XtSetArg( args[0], XmNdialogTitle, 
                  XmStringCreate( mesg, XmSTRING_DEFAULT_CHARSET ) );
        XtSetValues( vinfo->data.volume_data_displayed, args, 1 );
}

void oco_update_seg_type( w, seg_info, call_data )
Widget		w;
OcoSegInfo	*seg_info;
XtPointer	call_data;
{
	extern void			C_update_system();
	extern C_ObjectControlInfo	oco_info;
	extern C_World			world;
	
	Dimension	y_val;
	Dimension	height;
	Arg		args[2];
	C_Volume	*volume;
	char		string[10];

	if ( !seg_info ) return;

	volume = world.volume[seg_info->volume_index];

	if ( w == seg_info->seg_type[C_ISO_SURFACE] )
		volume->data.scalar_data_8bit->seg_type = 
		                seg_info->curr_seg_type = C_ISO_SURFACE;
	else if ( w == seg_info->seg_type[C_SCALAR_OPACITY] )
		volume->data.scalar_data_8bit->seg_type = 
		                seg_info->curr_seg_type = C_SCALAR_OPACITY;

	if ( XtIsManaged( seg_info->isosurface_frame ) )
		XtUnmanageChild( seg_info->isosurface_frame );

	if ( XtIsManaged( seg_info->s_opacity_frame ) )
		XtUnmanageChild( seg_info->s_opacity_frame );

	XtSetArg( args[0], XmNy, &y_val );
	XtSetArg( args[1], XmNheight, &height );

	switch ( seg_info->curr_seg_type )
	{
	    case C_ISO_SURFACE:
		XtManageChild( seg_info->isosurface_frame );
		XtGetValues( seg_info->isosurface_frame, args, 2 );

		XtSetArg( args[0], XmNsensitive, TRUE );
		XtSetValues( seg_info->parc_slider, args, 1 );

		volume->shade_type = C_SIMPLE_SHADE;

		break;
	    case C_SCALAR_OPACITY:
		XtManageChild( seg_info->s_opacity_frame );
		XtGetValues( seg_info->s_opacity_frame, args, 2 );

		volume->shade_type = C_NO_SHADE;

		if ( volume->color_type == C_SINGLE_COLOR )
		{
			XtSetArg( args[0], XmNset, TRUE );
			XtSetValues( 
			    seg_info->s_opacity_edit_toggle[C_OPACITY_EDIT], 
			    args, 1 );
			XtSetArg( args[0], XmNset, FALSE );
			XtSetArg( args[1], XmNsensitive, FALSE );
			XtSetValues( 
			    seg_info->s_opacity_edit_toggle[C_RED_COLOR_EDIT], 
			    args, 2 );
			XtSetValues( 
			    seg_info->s_opacity_edit_toggle[C_GREEN_COLOR_EDIT],
			    args, 2 );
			XtSetValues( 
			    seg_info->s_opacity_edit_toggle[C_BLUE_COLOR_EDIT], 
			    args, 2 );

                        sprintf(string, "----" ); 
                        XtSetArg( args[0], XmNlabelString,
                          XmStringCreate( string, XmSTRING_DEFAULT_CHARSET ) );
                        XtSetValues( seg_info->s_opacity_r_label, args, 1 );
                        XtSetValues( seg_info->s_opacity_g_label, args, 1 );
                        XtSetValues( seg_info->s_opacity_b_label, args, 1 );

		}
		else if ( volume->color_type == C_COLOR_ARRAY )
		{
			XtSetArg( args[0], XmNsensitive, TRUE );
			XtSetValues( 
			    seg_info->s_opacity_edit_toggle[C_RED_COLOR_EDIT], 
			    args, 1 );
			XtSetValues( 
			    seg_info->s_opacity_edit_toggle[C_GREEN_COLOR_EDIT],
			    args, 1 );
			XtSetValues( 
			    seg_info->s_opacity_edit_toggle[C_BLUE_COLOR_EDIT], 
			    args, 1 );
		}

		XtSetArg( args[0], XmNvalue, 0 );
		XtSetArg( args[1], XmNsensitive, FALSE );
		XtSetValues( seg_info->parc_slider, args, 2 );
		C_parc_subdivide( volume, 0 );

		C_update_system( C_VOLUME_MODIFIED );

		break;
	}
	XtSetArg( args[0], XmNy, y_val + height + 40 );
	XtSetValues( seg_info->parc_slider_frame, args, 1 );
	XtSetArg( args[0], XmNy, y_val + height + 110 );
	XtSetValues( seg_info->slice_buttons_frame, args, 1 );
	XtSetArg( args[0], XmNy, y_val + height + 160 );
	XtSetValues( seg_info->bottom_buttons_frame, args, 1 );


}

void oco_seg_function_toggle_callback( w, seg_info, call_data )
Widget                          w;
OcoSegInfo                      *seg_info;
XmToggleButtonCallbackStruct    *call_data;
{
 
        if ( call_data->set )
        {
                if ( w ==
                     seg_info->s_opacity_edit_toggle[C_OPACITY_EDIT] )
                        seg_info->s_opacity_edit_type =
                                        C_OPACITY_EDIT;
                else if ( w ==
                     seg_info->s_opacity_edit_toggle[C_RED_COLOR_EDIT] )
                        seg_info->s_opacity_edit_type =
                                        C_RED_COLOR_EDIT;
                else if ( w ==
                     seg_info->s_opacity_edit_toggle[C_GREEN_COLOR_EDIT] )
                        seg_info->s_opacity_edit_type =
                                        C_GREEN_COLOR_EDIT;
                else if ( w ==
                     seg_info->s_opacity_edit_toggle[C_BLUE_COLOR_EDIT] )
                        seg_info->s_opacity_edit_type =
                                        C_BLUE_COLOR_EDIT;
        }
}
 

void oco_set_max_opacity( w, seg_info, call_data )
Widget		w;
OcoSegInfo	*seg_info;
XtPointer	call_data;
{
	char		*string;
	char		mesg[100];
	float		max;

	max = 0;
	string = (char *)XmTextFieldGetString( seg_info->s_opacity_max_text );
	sscanf( string, "%f", &max );
	XtFree(string);

	if ( max <= 0.0 || max > 1.0 )
	{
		C_error_message(
		   "Max must be greater than 0 and less than or equal to 1");
	}
	else
		seg_info->s_opacity_max = max;

	sprintf( mesg, "%f", seg_info->s_opacity_max);
	XmTextSetString( seg_info->s_opacity_max_text, mesg );

}
void oco_set_data_props( w, seg_info, call_data )
Widget		w;
OcoSegInfo	*seg_info;
XtPointer	call_data;
{
	extern void			C_update_system();

	extern C_World			world;
	extern C_ObjectControlInfo	oco_info;

	C_Volume	*volume;
	char		*string;
	char		mesg[256];
	int		i;
	OcoVolumeInfo	*vinfo;
	float		isoval;
	float		max;
	Arg		args[1];
	int		parc_level;

	vinfo = oco_info.volume_info[seg_info->volume_index];

	volume = world.volume[seg_info->volume_index];

	volume->data.scalar_data_8bit->seg_type = seg_info->curr_seg_type;

	switch ( seg_info->curr_seg_type )
	{
	  case C_ISO_SURFACE:

		string = (char *)XmTextFieldGetString( seg_info->iso_text );
		sscanf( string, "%f", &isoval );
		XtFree(string);

		if ( isoval <= 0.0 || isoval > 255.0 )
		{
			C_error_message(
			   "Isovalue must be greater than 0 and less than 255");
		}
		else
		    volume->data.scalar_data_8bit->seg.isovalue = isoval;

		sprintf( mesg, "%f",
		   volume->data.scalar_data_8bit->seg.isovalue );
		XmTextSetString( seg_info->iso_text, mesg );

		if ( volume->shade_type == C_NO_SHADE )
			volume->shade_type = C_SIMPLE_SHADE;

		volume->data.scalar_data_8bit->seg_type = C_ISO_SURFACE;

		XmScaleGetValue( seg_info->parc_slider, &parc_level );

		if (!(volume->p_info) || 
		     (volume->p_info->sub_level != parc_level) ||
		     (volume->p_info->sub_thresh_low != isoval ) )	
			C_parc_subdivide( volume, parc_level );

		break;

	  case C_SCALAR_OPACITY:

		max = 0;
		string = (char *)XmTextFieldGetString( 
				seg_info->s_opacity_max_text );
		sscanf( string, "%f", &max );
		XtFree(string);
	
		if ( max <= 0.0 || max > 1.0 )
		{
		    C_error_message(
	   	      "Max must be greater than 0 and less than or equal to 1");
		}
		else
			seg_info->s_opacity_max = max;
	
		sprintf( mesg, "%f", seg_info->s_opacity_max);
		XmTextSetString( seg_info->s_opacity_max_text, mesg );

		for ( i = 0; i < 256; i++ )
		{
		    volume->data.scalar_data_8bit->seg.s_opacity->opacity[i] = 
			( (float) (seg_info->s_opacity[i]) / 255.0 ) * 
			seg_info->s_opacity_max;
		}

		volume->shade_type = C_NO_SHADE;
		volume->data.scalar_data_8bit->seg_type = C_SCALAR_OPACITY;
		
		if ( volume->color_type == C_COLOR_ARRAY )
		{
	                for ( i = 0; i < 256; i++ )
                	{
                            volume->color_data.color_array->color[i].red =
                                vinfo->color.color_array_red[i];
                            volume->color_data.color_array->color[i].green =
                                vinfo->color.color_array_green[i];
                            volume->color_data.color_array->color[i].blue =
                                vinfo->color.color_array_blue[i];
                        }

		}
		
		break;
	
	}
	
        sprintf( mesg, "Modify %s Segmentation",
                 world.volume[seg_info->volume_index]->volume_name );
        XtSetArg( args[0], XmNdialogTitle, 
                  XmStringCreate( mesg, XmSTRING_DEFAULT_CHARSET ) );
        XtSetValues( vinfo->data.volume_data_displayed, args, 1 );
        
        if ( vinfo->color.volume_color_displayed )
        {
               sprintf( mesg, "Modify %s Color",
                       world.volume[seg_info->volume_index]->volume_name );
               XtSetArg( args[0], XmNdialogTitle, 
                       XmStringCreate( mesg, XmSTRING_DEFAULT_CHARSET ) );
               XtSetValues(vinfo->color.volume_color_displayed,args,1);
         }
 
	C_update_system( C_VOLUME_DATAPROPS_MODIFIED );

}

void oco_reset_data_props( w, seg_info, call_data )
Widget		w;
OcoSegInfo	*seg_info;
XtPointer	call_data;
{

	extern void			C_update_system();
	extern void			oco_draw_s_opacity();
	extern void			oco_draw_color_array();

	extern C_World			world;
	extern C_ObjectControlInfo	oco_info;

	C_Volume	*volume;
	char		*string;
	char		mesg[256];
	int		i;
	OcoVolumeInfo	*vinfo;
	float		isoval;
	float		max;
	Arg		args[1];
	int		parc_level;

	vinfo = oco_info.volume_info[seg_info->volume_index];

	volume = world.volume[seg_info->volume_index];

	seg_info->curr_seg_type = volume->data.scalar_data_8bit->seg_type;

	sprintf( mesg, "%f",
	   volume->data.scalar_data_8bit->seg.isovalue );
	XmTextFieldSetString( seg_info->iso_text, mesg );

	if ( volume->p_info )
		XmScaleSetValue( seg_info->parc_slider, 
			volume->p_info->sub_level );


	sprintf( mesg, "%f", seg_info->s_opacity_max);
	XmTextFieldSetString( seg_info->s_opacity_max_text, mesg );

	for ( i = 0; i < 256; i++ )
	{
	  seg_info->s_opacity[i] = 
	    volume->data.scalar_data_8bit->seg.s_opacity->opacity[i] *
		(255.0 / seg_info->s_opacity_max );
	}

		
	if ( volume->color_type == C_COLOR_ARRAY )
	{
	  for ( i = 0; i < 256; i++ )
	  {
              vinfo->color.color_array_red[i] =
               	volume->color_data.color_array->color[i].red; 
              vinfo->color.color_array_green[i] =
               	volume->color_data.color_array->color[i].green; 
              vinfo->color.color_array_blue[i] =
               	volume->color_data.color_array->color[i].blue; 
          }

	}
		
	
        sprintf( mesg, "Modify %s Segmentation",
                 world.volume[seg_info->volume_index]->volume_name );
        XtSetArg( args[0], XmNdialogTitle, 
                  XmStringCreate( mesg, XmSTRING_DEFAULT_CHARSET ) );
        XtSetValues( vinfo->data.volume_data_displayed, args, 1 );
        
        if ( vinfo->color.volume_color_displayed )
        {
               sprintf( mesg, "Modify %s Color",
                       world.volume[seg_info->volume_index]->volume_name );
               XtSetArg( args[0], XmNdialogTitle, 
                       XmStringCreate( mesg, XmSTRING_DEFAULT_CHARSET ) );
               XtSetValues(vinfo->color.volume_color_displayed,args,1);
        }
 
	oco_draw_s_opacity( seg_info, 0, 255 );
	oco_draw_color_array( seg_info->volume_index, 0, 255 );

	C_update_system( C_VOLUME_DATAPROPS_MODIFIED );

}

