/**
 ** sipp - SImple Polygon Processor
 **
 **  A general 3d graphic package
 **
 **  Copyright Jonas Yngvesson  (jonas-y@isy.liu.se) 1988/89/90/91
 **            Inge Wallin      (ingwa@isy.liu.se)         1990/91
 **
 ** This program is free software; you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation; either version 1, or any later version.
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 ** You can receive a copy of the GNU General Public License from the
 ** Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 **/

/**
 ** rendering.c - Functions that handles rendering of the scene.
 **/

#include <stdio.h>
#ifndef NOMEMCPY
#include <memory.h>
#endif

#include <xalloca.h>
#include <smalloc.h>

#include <lightsource.h>
#include <geometric.h>
#include <rendering.h>
#include <objects.h>
#include <sipp.h>
#include <sipp_bitmap.h>
#include <viewpoint.h>
#include <patchlevel.h>

char *SIPP_VERSION = "2.1";

/*
 * Static global variables.
 */
static bool          show_backfaces;  /* Don't do backface culling */
static Edge        **y_bucket;        /* Y-bucket for edge lists. */
static FILE         *image_file;      /* File to store image in      */
                                      /* when rendering into a file. */
static void         *image_pm;        /* Pixmap to store image in when */
                                      /* rendering into a pix/bitmap.  */
static void        (*pixmap_set)();   /* Pointer to function for setting    */
                                      /* a pixel or draw a line in image_pm */


/*
 * Stack of transformation matrices used
 * when traversing an object hierarchy.
 */
static struct tm_stack_t {
    Transf_mat         mat;
    struct tm_stack_t *next;
} *tm_stack;

static Transf_mat      curr_mat;     /* Current transformation matrix */



/*
 * Calculate the normal vector for all polygons in the polygon list PSTART.
 *
 * Check if the polygon is backfacing with respect to the current
 * viewpoint.
 *
 * The normalized normal is added to a normal kept at each vertex
 * in the polygon. This will produce, at each vertex, an average of the
 * normals of the adjectent plygons.
 */
static void
calc_normals(pstart, eyepoint)
    Polygon *pstart;    /* Head of polygon list */
    Vector   eyepoint;  /* Viewpoint transformed to local coordinate system */
{
    Vector      normal;
    Vertex_ref *vref1, *vref2;
    Polygon    *polyref;
    double      plane_const;

    for (polyref = pstart; polyref != NULL; polyref = polyref->next) {
        vref1 = polyref->vertices;
        vref2 = vref1->next;

        normal.x = normal.y = normal.z = 0.0;
        do {
            normal.x += ((vref1->vertex->y - vref2->vertex->y)
                         * (vref1->vertex->z + vref2->vertex->z));
            normal.y += ((vref1->vertex->z - vref2->vertex->z)
                         * (vref1->vertex->x + vref2->vertex->x));
            normal.z += ((vref1->vertex->x - vref2->vertex->x)
                         * (vref1->vertex->y + vref2->vertex->y));
            vref1 = vref1->next;
            vref2 = ((vref2->next == NULL)?polyref->vertices:vref2->next);
        } while (vref1 != NULL);
        vecnorm(&normal);

        /*
         * Take care of backfacing polygons.
         */
        plane_const = -(normal.x * vref2->vertex->x
                        + normal.y * vref2->vertex->y
                        + normal.z * vref2->vertex->z);
        if (VecDot(eyepoint, normal) + plane_const <= 0.0) {
            if (show_backfaces) {
                polyref->backface = FALSE;
                VecNegate(normal);
            } else {
                polyref->backface = TRUE;
            }
        } else {
            polyref->backface = FALSE;
        }
            
        /*
         * Add the calculated normal to all vertices
         * in the poygon. This will result in an avaraged normal
         * at each vertex after all polygons have been pprocessed.
         */
        for (vref1 = polyref->vertices; vref1 != NULL; vref1 = vref1->next) {
            vref1->vertex->a += normal.x;
            vref1->vertex->b += normal.y;
            vref1->vertex->c += normal.z;
        }
    }
}



/*
 * Walk around a polygon, create the surrounding
 * edges and sort them into the y-bucket.
 */
static void
create_edges(view_vert, polygon, surface, render_mode)
    View_coord *view_vert;
    int         polygon;
    Surface    *surface;
    int         render_mode;
{
    Edge       *edge;
    View_coord *view_ref, *last;
    int         nderiv, y1, y2;
    double      deltay;
    double      x1, x2, xstep;
    double      z1, z2, zstep;
    double      nx1, nx2, nxstep;
    double      ny1, ny2, nystep;
    double      nz1, nz2, nzstep;
    double      u1, u2, ustep;
    double      v1, v2, vstep;
    double      w1, w2, wstep;


    view_ref = last = view_vert;

    do {
        view_ref = view_ref->next;

        /*
         * If we are drawing a line image we dont need
         * to build a complete edgelist. We draw the
         * lines directly instead.
         *
         * Since many lines are drawn twice (edges shared between
         * two polygons) and many line drawing algorithms are unsymmetrical
         * we need to make sure lines are always drawn in the same
         * direction
         */
        if (render_mode == LINE) {
            if (view_ref->y < view_ref->next->y) {
                (*pixmap_set)(image_pm, 
                              (int)(view_ref->x + 0.5), 
                              (int)(view_ref->y + 0.5),
                              (int)(view_ref->next->x + 0.5), 
                              (int)(view_ref->next->y + 0.5));
            } else {
                (*pixmap_set)(image_pm, 
                              (int)(view_ref->next->x + 0.5), 
                              (int)(view_ref->next->y + 0.5), 
                              (int)(view_ref->x + 0.5), 
                              (int)(view_ref->y + 0.5));
            }
            continue;
        }

        /*
         * Check if the slope of the edge is positive or negative
         * or zero.
         */
        y1 = (int)(view_ref->y + 0.5);
        y2 = (int)(view_ref->next->y + 0.5);
        deltay = (double)(y2 - y1);

        if (deltay > 0.0)
            nderiv = 1;
        else if (deltay < 0.0)
            nderiv = -1;
        else
            nderiv = 0;

        /*
         * Check if the edge is horizontal. In that case we
         * just skip it.
         */
        if (nderiv != 0) {

            edge = (Edge *)smalloc(sizeof(Edge));

            x1 = view_ref->x;
            x2 = view_ref->next->x;
            z1 = view_ref->z;
            z2 = view_ref->next->z;
            nx1 = view_ref->nx;
            nx2 = view_ref->next->nx;
            ny1 = view_ref->ny;
            ny2 = view_ref->next->ny;
            nz1 = view_ref->nz;
            nz2 = view_ref->next->nz;
            u1 = view_ref->u;
            u2 = view_ref->next->u;
            v1 = view_ref->v;
            v2 = view_ref->next->v;
            w1 = view_ref->w;
            w2 = view_ref->next->w;

            deltay = fabs(deltay);
            xstep = (x2 - x1) / deltay;
            zstep = (z2 - z1) / deltay;
            if (render_mode != FLAT) {
                nxstep = (nx2 - nx1) / deltay;
                nystep = (ny2 - ny1) / deltay;
                nzstep = (nz2 - nz1) / deltay;
                if (render_mode == PHONG) {
                    ustep = (u2 - u1) / deltay;
                    vstep = (v2 - v1) / deltay;
                    wstep = (w2 - w1) / deltay;
                }
            }

            if (nderiv > 0) {       

                /*
                 * The edge has positive slope
                 */
                edge->y = y2;
                edge->y_stop = y1;
                edge->x = x2;
                edge->z = z2;
                edge->nx = nx2;
                edge->ny = ny2;
                edge->nz = nz2;
                edge->u = u2;
                edge->v = v2;
                edge->w = w2;
                edge->xstep = -xstep;
                edge->zstep = -zstep;
                if (render_mode != FLAT) {
                    edge->nxstep = -nxstep;
                    edge->nystep = -nystep;
                    edge->nzstep = -nzstep;
                    if (render_mode == PHONG) {
                        edge->ustep = -ustep;
                        edge->vstep = -vstep;
                        edge->wstep = -wstep;
                    }
                }

            } else {

                /*
                 * The edge has negative slope.
                 */
                edge->y = y1;
                edge->y_stop = y2;
                edge->x = x1;
                edge->z = z1;
                edge->nx = nx1;
                edge->ny = ny1;
                edge->nz = nz1;
                edge->u = u1;
                edge->v = v1;
                edge->w = w1;
                edge->xstep = xstep;
                edge->zstep = zstep;
                if (render_mode != FLAT) {
                    edge->nxstep = nxstep;
                    edge->nystep = nystep;
                    edge->nzstep = nzstep;
                    if (render_mode == PHONG) {
                        edge->ustep = ustep;
                        edge->vstep = vstep;
                        edge->wstep = wstep;
                    }
                }
            }
            edge->polygon = polygon;
            edge->surface = surface;
            edge->next = y_bucket[edge->y];
            y_bucket[edge->y] = edge;
        }
    } while (view_ref != last);
}



/*
 * Calculate a new vertex by interpolation between
 * V1 and V2.
 */
static View_coord *
interpolate(v1, v2, ratio)
    View_coord *v1, *v2;
    double      ratio;
{
    View_coord *tmp;

    tmp = (View_coord *)smalloc(sizeof(View_coord));

    tmp->x = v1->x + ratio * (v2->x - v1->x);
    tmp->y = v1->y + ratio * (v2->y - v1->y);
    tmp->z = v1->z + ratio * (v2->z - v1->z);
    tmp->nx = v1->nx + ratio * (v2->nx - v1->nx);
    tmp->ny = v1->ny + ratio * (v2->ny - v1->ny);
    tmp->nz = v1->nz + ratio * (v2->nz - v1->nz);
    tmp->u = v1->u + ratio * (v2->u - v1->u);
    tmp->v = v1->v + ratio * (v2->v - v1->v);
    tmp->w = v1->w + ratio * (v2->w - v1->w);
    tmp->next = NULL;

    return tmp;
}



/*
 * Clip a polygon using the Sutherland-Hodgeman algorithm for
 * reentrant clipping;
 */
#define XMIN 0
#define XMAX 1
#define YMIN 2
#define YMAX 3
#define ZMIN 4
#define ZMAX 5

static View_coord *
polygon_clip(vlist, plane, first_vert)
    View_coord *vlist;
    int         plane;
    bool        first_vert;
{
    static View_coord   *first;
    static View_coord   *curr;
    View_coord          *out1;
    View_coord          *out2;
    double               curr_limit;
    double               first_limit;
    double               vlist_limit;
    double               ratio;
    bool                 visible;

    out1 = out2 = NULL;

    if (vlist == NULL) {

        /*
         * Did we get an empty list from the start?
         */
        if (first_vert) {
            return NULL;
        }

        /*
         * Last vertex, close the polygon.
         */
        ratio = 0.0;
        curr_limit = curr->z * camera.focal_ratio;
        first_limit = first->z * camera.focal_ratio;

        switch (plane) {

          case XMIN:
            if ((curr->x < -curr_limit && first->x >= -first_limit)
                || (curr->x >= -curr_limit && first->x < -first_limit)) {
                ratio = fabs(curr->x + curr_limit);
                ratio /= (ratio + fabs(first->x + first_limit)); 
            }
            break;

          case XMAX:
            if ((curr->x <= curr_limit && first->x > first_limit)
                || (curr->x > curr_limit && first->x <= first_limit)) {
                ratio = fabs(curr->x - curr_limit);
                ratio /= (ratio + fabs(first->x - first_limit));
            }
            break;

          case YMIN:
            if ((curr->y < -curr_limit && first->y >= -first_limit)
                || (curr->y >= -curr_limit && first->y < -first_limit)) {
                ratio = fabs(curr->y + curr_limit);
                ratio /= (ratio + fabs(first->y + first_limit));
            }
            break;

          case YMAX:
            if ((curr->y <= curr_limit && first->y > first_limit)
                || (curr->y > curr_limit && first->y <= first_limit)) {
                ratio = fabs(curr->y - curr_limit);
                ratio /= (ratio + fabs(first->y - first_limit));
            }
            break;

          case ZMIN:
            if ((curr->z < hither && first->z >= hither)
                || (curr->z >= hither && first->z < hither)) {
                ratio = fabs(curr->z - hither);
                ratio = ratio / (ratio + fabs(first->z - hither));
            }
            break;

          case ZMAX:
            if ((curr->z <= yon && first->z > yon)
                || (curr->z > yon && first->z <= yon)) {
                ratio = fabs(curr->z - yon);
                ratio = ratio / (ratio + fabs(first->z - yon));
            }
            break;
        }

        if (ratio != 0.0) {
            out1 = interpolate(curr, first, ratio);
            return out1;
        } else {
            return NULL;
        }
    }

    vlist_limit = vlist->z * camera.focal_ratio;
    
    if (first_vert) {
        first = vlist;
    } else {
        ratio = 0.0;
        curr_limit = curr->z * camera.focal_ratio;

        switch (plane) {

          case XMIN:
            if ((curr->x < -curr_limit && vlist->x >= -vlist_limit)
                || (curr->x >= -curr_limit && vlist->x < -vlist_limit)) {
                ratio = fabs(curr->x + curr_limit);
                ratio /= (ratio + fabs(vlist->x + vlist_limit));
            }
            break;

          case XMAX:
            if ((curr->x <= curr_limit && vlist->x > vlist_limit)
                || (curr->x > curr_limit && vlist->x <= vlist_limit)) {
                ratio = fabs(curr->x - curr_limit);
                ratio /= (ratio + fabs(vlist->x - vlist_limit));
            }
            break;

          case YMIN:
            if ((curr->y < -curr_limit && vlist->y >= -vlist_limit)
                || (curr->y >= -curr_limit && vlist->y < -vlist_limit)) {
                ratio = fabs(curr->y + curr_limit);
                ratio /= (ratio + fabs(vlist->y + vlist_limit));
            }
            break;

          case YMAX:
            if ((curr->y <= curr_limit && vlist->y > vlist_limit)
                || (curr->y > curr_limit && vlist->y <= vlist_limit)) {
                ratio = fabs(curr->y - curr_limit);
                ratio /= (ratio + fabs(vlist->y - vlist_limit));
            }
            break;

          case ZMIN:
            if ((curr->z < hither && vlist->z >= hither)
                || (curr->z >= hither && vlist->z < hither)) {
                ratio = fabs(curr->z - hither);
                ratio = ratio / (ratio + fabs(vlist->z - hither));
            }
            break;

          case ZMAX:
            if ((curr->z <= yon && vlist->z > yon)
                || (curr->z > yon && vlist->z <= yon)) {
                ratio = fabs(curr->z - yon);
                ratio = ratio / (ratio + fabs(vlist->z - yon));
            }
            break;
        }

        if (ratio != 0.0) {
            out1 = interpolate(curr, vlist, ratio);
            out1->next = vlist;
        }
    }

    curr = vlist;
    visible = FALSE;
    switch (plane) {

      case XMIN:
        visible = (curr->x >= -vlist_limit);
        break;

      case XMAX:
        visible = (curr->x <= vlist_limit);
        break;

      case YMIN:
        visible = (curr->y >= -vlist_limit);
        break;

      case YMAX:
        visible = (curr->y <= vlist_limit);
        break;

      case ZMIN:
        visible = (curr->z >= hither);
        break;

      case ZMAX:
        visible = (curr->z <= yon);
        break;
    }

    if (visible) {
        out2 = curr;
        out2->next = polygon_clip(curr->next, plane, FALSE);
        return ((out1) ? (out1) : (out2));

    } else {
        if (out1) {
            out1->next = polygon_clip(curr->next, plane, FALSE);
        } else {
            out1 = polygon_clip(curr->next, plane, FALSE);
        }
        free(vlist);
        return out1;
    }
}



/*
 * Transform vertices into view coordinates. The transform is
 * defined in MATRIX. Store the transformed vertices in a
 * temporary list, create edges in the y_bucket.
 */
static void
transf_vertices(vertex_list, surface, view_mat, tr_mat, 
                xsiz, ysiz, render_mode)
    Vertex_ref *vertex_list;
    Surface    *surface;
    Transf_mat *view_mat;
    Transf_mat *tr_mat;
    double      xsiz, ysiz;
    int         render_mode;
{
    static int  polygon = 0;        /* incremented for each call to provide */
                                    /* unique polygon id numbers */
    Vertex_ref *vref;
    View_coord *nhead;
    View_coord *view_ref;
    View_coord *mark;
    Color       color;                    
    double      minsize;
    double      tmp;

    vref = vertex_list;
    nhead = NULL;

    minsize = ((xsiz > ysiz) ? ysiz : xsiz);

    while (vref != NULL) {

        view_ref = (View_coord *)smalloc(sizeof(View_coord));


        /*
         * Transform the normal (world coordinates).
         */
        view_ref->nx = (vref->vertex->a * tr_mat->mat[0][0] 
                        + vref->vertex->b * tr_mat->mat[1][0] 
                        + vref->vertex->c * tr_mat->mat[2][0]);
        view_ref->ny = (vref->vertex->a * tr_mat->mat[0][1] 
                        + vref->vertex->b * tr_mat->mat[1][1] 
                        + vref->vertex->c * tr_mat->mat[2][1]);
        view_ref->nz = (vref->vertex->a * tr_mat->mat[0][2] 
                        + vref->vertex->b * tr_mat->mat[1][2] 
                        + vref->vertex->c * tr_mat->mat[2][2]);

        /*
         * Transform the vertex into view coordinates.
         */
        view_ref->x = (vref->vertex->x * view_mat->mat[0][0] 
                       + vref->vertex->y * view_mat->mat[1][0] 
                       + vref->vertex->z * view_mat->mat[2][0]
                       + view_mat->mat[3][0]);
        view_ref->y = (vref->vertex->x * view_mat->mat[0][1] 
                       + vref->vertex->y * view_mat->mat[1][1] 
                       + vref->vertex->z * view_mat->mat[2][1]
                       + view_mat->mat[3][1]);
        view_ref->z = (vref->vertex->x * view_mat->mat[0][2] 
                       + vref->vertex->y * view_mat->mat[1][2] 
                       + vref->vertex->z * view_mat->mat[2][2]
                       + view_mat->mat[3][2]);

        /*
         * Texture coordinates is not affected by transformations.
         */
        view_ref->u = vref->vertex->u;
        view_ref->v = vref->vertex->v;
        view_ref->w = vref->vertex->w;

        view_ref->next = nhead;
        nhead = view_ref;

        vref = vref->next;
    }

    /* 
     * Clip the resulting polygon. We need to do this
     * before the perpective transformation to keep texture
     * coordinates correct.
     */
    nhead = polygon_clip(nhead, ZMIN, TRUE);
    nhead = polygon_clip(nhead, ZMAX, TRUE);
    if (xsiz > ysiz) {
        tmp = camera.focal_ratio;
        camera.focal_ratio *= xsiz / ysiz;
        nhead = polygon_clip(nhead, XMIN, TRUE);
        nhead = polygon_clip(nhead, XMAX, TRUE);
        camera.focal_ratio = tmp;
        nhead = polygon_clip(nhead, YMIN, TRUE);
        nhead = polygon_clip(nhead, YMAX, TRUE);
    } else {
        tmp = camera.focal_ratio;
        camera.focal_ratio *= ysiz / xsiz;
        nhead = polygon_clip(nhead, YMIN, TRUE);
        nhead = polygon_clip(nhead, YMAX, TRUE);
        camera.focal_ratio = tmp;
        nhead = polygon_clip(nhead, XMIN, TRUE);
        nhead = polygon_clip(nhead, XMAX, TRUE);
    }

    if (nhead == NULL) {    /* Nothing left? */
        return;
    }


    
    /*
     * If we are flat shading, we need a color for the polygon.
     * We call the shader at the first vertex to get this.
     * (This is not quite correct since the normal here is
     * an averaged normal of the surrounding polygons)
     */
    if (render_mode == FLAT) {
        (*surface->shader)
            (nhead->nx, nhead->ny, nhead->nz, 
             nhead->u, nhead->v, nhead->w,
             camera.vec, lightsrc_stack, 
             surface->surface, &color);
    }


    /*
     * Walk around the new (clipped and transformed) polygon and 
     * transform it into perspective screen coordinates.
     * We have to transform the texture coordinates too in order
     * to interpolate them linearly in "screen space". This
     * transformation is inverted in render_line().
     * If we are doing gouraud shading we call the shader at each
     * vertex.
     * Last we tie the head and tail together forming a cirkular
     * list, this simplifies edge creation.
     */
    for (view_ref = nhead;; view_ref = view_ref->next) {
        view_ref->z *= camera.focal_ratio;
        view_ref->x  = view_ref->x * minsize / view_ref->z + xsiz;
        view_ref->y  = view_ref->y * minsize / view_ref->z + ysiz;
        view_ref->u /= view_ref->z;
        view_ref->v /= view_ref->z;
        view_ref->w /= view_ref->z;

        if (render_mode == GOURAUD) {
            (*surface->shader)
                (view_ref->nx, view_ref->ny, view_ref->nz, 
                 view_ref->u, view_ref->v, view_ref->w,
                 camera.vec, lightsrc_stack, 
                 surface->surface, &color);
        } 

        if (render_mode == GOURAUD || render_mode == FLAT) {
            view_ref->nx = color.red;
            view_ref->ny = color.grn;
            view_ref->nz = color.blu;
        } 

        if (view_ref->next == NULL) {
            view_ref->next = nhead;
            break;
        }
    }

    create_edges(nhead, polygon++, surface, render_mode);

    /*
     * Free the memory used by the transformed polygon.
     */
    mark = nhead;
    do {
        view_ref = nhead;
        nhead = nhead->next;
        free(view_ref);
    } while (nhead != mark);
}



/*
 * Initialize the scanline z-buffer and the actual picture
 * scanline buffer.
 */
static void
init_buffers(res, z_buffer, scanline)
    int            res;
    double        *z_buffer;
    unsigned char *scanline;
{
    int i;
    
#ifdef NOMEMCPY
    bzero(scanline, res * 3);
#else
    memset(scanline, 0, res * 3);
#endif
    for (i = 0; i < res; i++)
        z_buffer[i] = 1.0e100;
}
    

/*
 * Read edge pairs from the edge list EDGE_LIST. Walk along the scanline
 * and interpolate z value, texture coordinates and normal vector as 
 * we go. Call the shader and write into scanline buffer according to 
 * result on z-buffer test.
 */
static void
render_line(res, z_buffer, scanline, edge_list, render_mode)
    int            res;
    double        *z_buffer;
    unsigned char *scanline;
    Edge          *edge_list;
    int            render_mode;
{
    double z, zstep;
    double nx, nxstep;
    double ny, nystep;
    double nz, nzstep;
    double u, ustep;
    double v, vstep;
    double w, wstep;
    double ur, vr, wr;
    double ratio;
    Color  color;
    int    i, j, x, xstop;
    Edge  *startedge, *stopedge;
    
    startedge = edge_list;
    stopedge = NULL;
    while (startedge != NULL) {
        stopedge = startedge->next;
        x = (int)(startedge->x + 0.5);
        xstop = (int)(stopedge->x + 0.5);
        z = startedge->z;
        nx = startedge->nx;
        ny = startedge->ny;
        nz = startedge->nz;
        u = startedge->u;
        v = startedge->v;
        w = startedge->w;
        if (x < xstop) {
            ratio = (double)(xstop - x);
            zstep = (stopedge->z - z) / ratio;
            if (render_mode != FLAT) {
                nxstep = (stopedge->nx - nx) / ratio;
                nystep = (stopedge->ny - ny) / ratio;
                nzstep = (stopedge->nz - nz) / ratio;
                if (render_mode == PHONG) {
                    ustep = (stopedge->u - u) / ratio;
                    vstep = (stopedge->v - v) / ratio;
                    wstep = (stopedge->w - w) / ratio;
                }
            }
        } else {
            zstep = 0.0;
            nxstep = nystep = nzstep = 0.0;
            ustep = vstep = wstep = 0.0;
        }

        for (i = x, j = i * 3; i <= xstop; i++) {

            if (z < z_buffer[i]) {

                if (render_mode == PHONG) {
                    ur = u * z;  /* Transform texture coordinates back */
                    vr = v * z;  /* from their homogenouos form */
                    wr = w * z;
                    (*startedge->surface->shader)
                        (nx, ny, nz, ur, vr, wr,
                         camera.vec, lightsrc_stack, 
                         startedge->surface->surface, &color);
                    scanline[j++] = (unsigned char)(color.red * 255.0 + 0.5);
                    scanline[j++] = (unsigned char)(color.grn * 255.0 + 0.5);
                    scanline[j++] = (unsigned char)(color.blu * 255.0 + 0.5);
                    
                } else if (render_mode == FLAT || render_mode == GOURAUD) {
                    scanline[j++] = (unsigned char)(nx * 255.0 + 0.5);
                    scanline[j++] = (unsigned char)(ny * 255.0 + 0.5);
                    scanline[j++] = (unsigned char)(nz * 255.0 + 0.5);
                }
                
                z_buffer[i] = z;
                
            } else if (i >= res) {
                break;
                
            } else {
                j += 3;
            }
            
            z += zstep;
            if (render_mode != FLAT) {
                nx += nxstep;
                ny += nystep;
                nz += nzstep;
                if (render_mode == PHONG) {
                    u += ustep;
                    v += vstep;
                    w += wstep;
                }
            }
        }
        startedge = stopedge->next;
    }
}




/*
 * Insert an edge into an edge list. Edges belonging to the same
 * polygon must be inserted sorted in x, so that edge pairs are
 * created.
 */
static Edge *
insert_edge(edge_list, edge, poly_found)
    Edge *edge_list, *edge;
    bool  poly_found;
{
    if (edge_list == NULL) {
        edge_list = edge;
	edge->next = NULL;
    } else if (edge_list->polygon == edge->polygon) {
        if (edge_list->x > edge->x) {
	    edge->next = edge_list;
	    edge_list = edge;
	} else if ((((int)(edge_list->x + 0.5)) == ((int)(edge->x + 0.5)))
                   && (edge_list->xstep > edge->xstep)) {
	    edge->next = edge_list;
	    edge_list = edge;
        } else {
	    edge_list->next = insert_edge(edge_list->next, edge, TRUE);
        }
    } else if (poly_found) {
        edge->next = edge_list;
        edge_list = edge;
    } else {
        edge_list->next = insert_edge(edge_list->next, edge, FALSE);
    }

    return edge_list;
}
        


/*
 * Merge two edge lists.
 */
static Edge *
merge_edge_lists(list1, list2)
    Edge *list1, *list2;
{
    Edge *eref1, *eref2, *next;
    
    if (list2 == NULL)
        return list1;

    eref1 = list1;
    eref2 = list2;
    do {
        next = eref2->next;
        eref1 = insert_edge(eref1, eref2, FALSE);
	eref2 = next;
    } while (eref2 != NULL);

    return eref1;
}



/*
 * Store a rendered line on the place indicated by STORAGE_MODE.
 */
static void
store_line(buf, npixels, line, storage_mode)
    unsigned char *buf;
    int            npixels;
    int            line;
    int            storage_mode;
{
    int i, j;

    switch (storage_mode) {
      case PBM_FILE:
        fwrite(buf, sizeof(unsigned char), npixels, image_file);
        fflush(image_file);
        break;
        
      case PPM_FILE:
        fwrite(buf, sizeof(unsigned char), npixels * 3, image_file);
        fflush(image_file);
        break;

      case PIXMAP:
        for (i = 0, j = 0; j < npixels; j++, i += 3) {
            (*pixmap_set)(image_pm, j, line, buf[i], buf[i + 1], buf[i + 2]);
        }
        break;

      default:
        break;
    }
}


/*
 * Allocate the needed buffers. Create a list of active edges and
 * move down the y-bucket, inserting and deleting edges from this
 * active list as we go. Call render_line for each scanline and
 * do an average filtering before storing the scanline.
 */
static void
scan_and_render(xres, yres, storage_mode, render_mode, oversampl)
    int   xres, yres;
    int   storage_mode;
    int   render_mode;
    int   oversampl;
{
    Edge           *active_list, *edgep, *edgep2;
    double         *z_buffer;
    unsigned char **linebuf;
    int             curr_line;
    int             y, next_edge;
    int             sum;
    int             i, j, k, l;
    
    z_buffer = (double *)scalloc(xres, sizeof(double));
    linebuf  = (unsigned char **)alloca(oversampl * sizeof(unsigned char *));
    for (i = 0; i < oversampl; i++) {
        linebuf[i] = (unsigned char *)scalloc(xres * 3, sizeof(unsigned char));
    }

    if (storage_mode == PPM_FILE) {
        fprintf(image_file, "P6\n");
        fprintf(image_file, "#Image rendered with SIPP %s%s\n", 
                SIPP_VERSION, PATCHLEVEL);
        fprintf(image_file, "%d\n%d\n255\n", xres / oversampl, 
                                             yres / oversampl);
    }

    y = yres - 1;
    active_list = NULL;
    curr_line = 0;
 
    while (y >= 0) {

        active_list = merge_edge_lists(active_list, y_bucket[y]);
        next_edge = y - 1;

        while (next_edge >=0 && y_bucket[next_edge] == NULL)
            next_edge--;

        while (y > next_edge) {

            init_buffers(xres, z_buffer, linebuf[curr_line]);
            render_line(xres, z_buffer, linebuf[curr_line], active_list,
                        render_mode); 

            if (++curr_line == oversampl) {

                /*
                 * Average the pixel.
                 */
                for (i = 0; i < ((xres / oversampl)); i++) {
                    for (l = 0; l < 3; l++) {
                        sum = 0;
                        for (j = i * 3 * oversampl + l; 
                             j < (i * 3 * oversampl + l + 3 * oversampl); 
                             j += 3) {
                            for (k = 0; k < oversampl; k++) {
                                sum += *(linebuf[k] + j);
                            }
                        }
                        *(linebuf[0] + i * 3 + l)
                            = sum / (oversampl * oversampl);
                    }
                }
                store_line(linebuf[0], xres / oversampl, 
                           (yres - y - 1) / oversampl, storage_mode);
                curr_line = 0;
            }

	    if (active_list != NULL) {

	        edgep2 = active_list;
	        edgep = active_list->next;
	        while (edgep != NULL) {
	            if (edgep->y <= (edgep->y_stop + 1)) {
                        edgep2->next = edgep->next;
		        free(edgep);
	                edgep = edgep2->next;
		    } else {
		        edgep2 = edgep;
		        edgep = edgep->next;
		    }
                }

  	        if (active_list->y <= (active_list->y_stop + 1)) {
	            edgep = active_list;
		    active_list = active_list->next;
	            free(edgep);
	        }

	        edgep = active_list;
	        while (edgep != NULL) {
	            edgep->y--;
		    edgep->x += edgep->xstep;
		    edgep->z += edgep->zstep;
                    if (render_mode != FLAT) {
                        edgep->nx += edgep->nxstep;
                        edgep->ny += edgep->nystep;
                        edgep->nz += edgep->nzstep;
                        if (render_mode == PHONG) {
                            edgep->u += edgep->ustep;
                            edgep->v += edgep->vstep;
                            edgep->w += edgep->wstep;
                        }
                    }
		    edgep = edgep->next;
	        }
	    }
	    y--;
	}
    }
    free(z_buffer);
    for (i = 0; i < oversampl; i++) {
        free(linebuf[i]);
    }
}



/*
 * Reset the averaged normals in the vertex tree P.
 */
static void
reset_normals(vref)
    Vertex *vref;
{
    if (vref != NULL) {
        vref->a = 0;
        vref->b = 0;
        vref->c = 0;
        reset_normals(vref->big);
        reset_normals(vref->sml);
    }
}



/*
 * Push the current transformation matrix on the matrix stack.
 */
static void
matrix_push()
{
    struct tm_stack_t *new_tm;

    new_tm = (struct tm_stack_t *)smalloc(sizeof(struct tm_stack_t));
    MatCopy(&new_tm->mat, &curr_mat);
    new_tm->next = tm_stack;
    tm_stack     = new_tm;
}


/*
 * Pop the top of the matrix stack and make
 * it the new current transformation matrix.
 */
static void
matrix_pop()
{
    struct tm_stack_t *tmp;

    MatCopy(&curr_mat, &tm_stack->mat);
    tmp = tm_stack;
    tm_stack = tm_stack->next;
    free(tmp);
}



/*
 * Traverse an object hierarchy, transform each object
 * according to its transformation matrix.
 * Transform all polygons in the object to view coordinates.
 * Build the edge lists in y_bucket.
 */
static void
traverse_object_tree(object, view_mat, xres, yres, render_mode)
    Object      *object;
    Transf_mat  *view_mat;
    int          xres, yres;
    int          render_mode;
{
    Object      *objref;
    Surface     *surfref;
    Polygon     *polyref;
    Vector       eyepoint, tmp;
    Transf_mat   loc_view_mat;
    double       m[3][4], dtmp;
    int          i, j;


    if (object == NULL) {
        return;
    }

    for (objref = object; objref != NULL; objref = objref->next) {

        matrix_push();
        mat_mul(&curr_mat, &objref->transf, &curr_mat);
        mat_mul(&loc_view_mat, &curr_mat, view_mat);

        tmp.x = camera.x0;
        tmp.y = camera.y0;
        tmp.z = camera.z0;

        /*
         * Do an inverse transformation of the viewpoint to use
         * when doing backface culling (in calc_normals()).
         */
        tmp.x -= curr_mat.mat[3][0];
        tmp.y -= curr_mat.mat[3][1];
        tmp.z -= curr_mat.mat[3][2];
        m[0][0] = curr_mat.mat[0][0] ; m[0][1] = curr_mat.mat[1][0];
        m[0][2] = curr_mat.mat[2][0] ; m[0][3] = tmp.x;
        m[1][0] = curr_mat.mat[0][1] ; m[1][1] = curr_mat.mat[1][1];
        m[1][2] = curr_mat.mat[2][1] ; m[1][3] = tmp.y;
        m[2][0] = curr_mat.mat[0][2] ; m[2][1] = curr_mat.mat[1][2];
        m[2][2] = curr_mat.mat[2][2] ; m[2][3] = tmp.z;

        if (m[0][0] == 0.0) {
            if (m[1][0] != 0.0)
                j = 1;
            else
                j = 2;
            for (i = 0; i < 4; i++) {
                dtmp     = m[0][i];
                m[0][i] = m[j][i];
                m[j][i] = dtmp;
            }
        }
        
        for (j = 1; j < 3; j++) {
            m[j][0] /= (-m[0][0]);
            for (i = 1; i < 4; i++)
                m[j][i] += m[0][i] * m[j][0];
        }
        
        if (m[1][1] == 0.0)
            for (i = 1; i < 4; i++) {
                dtmp     = m[1][i];
                m[1][i] = m[2][i];
                m[2][i] = dtmp;
            }
        
        if (m[1][1] != 0.0) {
            m[2][1] /= (-m[1][1]);
            m[2][2] += m[1][2] * m[2][1];
            m[2][3] += m[1][3] * m[2][1];
        }
        
        eyepoint.z = m[2][3] / m[2][2];
        eyepoint.y = (m[1][3] - eyepoint.z * m[1][2]) / m[1][1];
        eyepoint.x = (m[0][3] - eyepoint.z * m[0][2] 
                      - eyepoint.y * m[0][1]) / m[0][0];

        for (surfref = objref->surfaces; surfref != NULL; 
             surfref = surfref->next) {

            calc_normals(surfref->polygons, eyepoint);

            for (polyref = surfref->polygons; polyref != NULL; 
                 polyref = polyref->next) {

                if (!polyref->backface) {
                    transf_vertices(polyref->vertices, surfref, 
                                    &loc_view_mat, 
                                    &curr_mat, (double)xres / 2.0,
                                    (double)yres / 2.0, render_mode);
                }

            }
            reset_normals(surfref->vertices);

        }
        traverse_object_tree(objref->sub_obj, view_mat, xres, yres,
                             render_mode); 
        matrix_pop();
    }
}



/*
 * Recursively traverse the rendering database (preorder).
 * Call traverse_object_tree for each object.
 */
static void
traverse_object_db(obj_root, view_mat, xres, yres, render_mode)
    Inst_object  *obj_root;
    Transf_mat   *view_mat;
    int           xres, yres;
    int           render_mode;
{
    if (obj_root != NULL) {
        traverse_object_tree(obj_root->object, view_mat,
                             xres, yres, render_mode); 
        traverse_object_db(obj_root->sml, view_mat,
                           xres, yres, render_mode);
        traverse_object_db(obj_root->big, view_mat,
                           xres, yres, render_mode);
    }
}




/*
 * "Main" functions in rendering. Allocate y-bucket, transform vertices
 * into viewing coordinates, make edges and sort them into the y-bucket.
 * Call scan_and_render to do the real work.
 */
static void
render_main(xres, yres, storage_mode, render_mode, oversampling)
    int   xres, yres;
    int   storage_mode;
    int   render_mode;
    int   oversampling;
{
    Transf_mat      view_mat;

    if (render_mode != LINE) {
        xres *= oversampling;
        yres *= oversampling;
        y_bucket = (Edge **)scalloc(yres, sizeof(Edge *));

    } else if (storage_mode == PBM_FILE) {
        image_pm = sipp_bitmap_create(xres, yres);
        pixmap_set = sipp_bitmap_line;
    }

    get_view_transf(&view_mat);
    MatCopy(&curr_mat, &ident_matrix);
    
    vecnorm(&camera.vec);

    traverse_object_db(object_db, &view_mat,
                       xres - 1, yres - 1, render_mode);

    if (render_mode != LINE) {
        scan_and_render(xres, yres, storage_mode, render_mode, oversampling);
        free(y_bucket);

    } else if (storage_mode == PBM_FILE) {
        sipp_bitmap_write(image_file, image_pm);
        sipp_bitmap_destruct(image_pm);
    }

    view_vec_eval();

}
    

void
render_image_file(xres, yres, im_file, render_mode, oversampling)
    int   xres, yres;
    FILE *im_file;
    int   render_mode;
    int   oversampling;
{
    image_file = im_file;

    if (render_mode == LINE) {
        render_main(xres, yres, PBM_FILE, render_mode, oversampling);
    } else {
        render_main(xres, yres, PPM_FILE, render_mode, oversampling);
    }
}


void
render_image_pixmap(xres, yres, pixmap, pixmap_func, render_mode, oversampling)
    int     xres, yres;
    void   *pixmap;
    void  (*pixmap_func)();
    int     render_mode;
    int     oversampling;
{
    image_pm = pixmap;
    pixmap_set = pixmap_func;
    render_main(xres, yres, PIXMAP, render_mode, oversampling);
}



/*============= Functions that handles global initializations==============*/

/*
 * If called with TRUE as argument,  no backface culling will 
 * be performed. If a polygon is backfacing it will be rendered
 * as facing in the opposit direction.
 */
void
sipp_show_backfaces(flag)
    bool flag;
{
    show_backfaces = flag;
}



/*
 * Necessary initializations.
 */
void
sipp_init()
{
    objects_init();
    lightsource_init();
    sipp_show_backfaces(FALSE);
    viewpoint(0.0, 0.0, 10.0,  0.0, 0.0, 0.0,  0.0, 1.0, 0.0,  0.25);
}
