/**
 ** sipp - SImple Polygon Processor
 **
 **  A general 3d graphic package
 **
 **  Copyright Jonas Yngvesson  (jonas-y@isy.liu.se) 1988/89/90/91
 **            Inge Wallin      (ingwa@isy.liu.se)         1990/91
 **
 ** This program is free software; you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation; either version 1, or any later version.
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 ** You can receive a copy of the GNU General Public License from the
 ** Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 **/

/**
 ** ellipsoid.c - Creating ellipsiods and spheres as sipp objects.
 **/

#include <xalloca.h>
#include <math.h>

#include <sipp.h>


Object *
sipp_ellipsoid(x_rad, y_rad, z_rad, res, surface, shader)
    double  x_rad;
    double  y_rad;
    double  z_rad;
    int     res;
    void   *surface;
    Shader *shader;
{
    int      i, j;
    double   factor;
    double   factor1;
    double   factor2;
    double   zradprim;
    double   zradprim1;
    double   zradprim2;
    double  *x_arr;
    double  *y_arr;
    Object  *ellipsoid;
    
    /* Odd resolutions make ugly spheres since the poles will be */
    /* different in size. */
    if (res & 1) {
        res++;
    }

    /* Create two arrays with the coordinates of the points */
    /* around the perimeter at Z = 0 */
    x_arr = (double *) alloca(res * sizeof(double));
    y_arr = (double *) alloca(res * sizeof(double));
    for (i = 0; i < res; i++) {
        x_arr[i] = x_rad * cos(i * 2.0 * M_PI / res);
        y_arr[i] = y_rad * sin(i * 2.0 * M_PI / res);
    }

    /* Create the top pole */
    factor = sin(2.0 * M_PI / res);
    zradprim = z_rad * cos(2.0 * M_PI / res);
    for (i = 0; i < res; i++) {
        vertex_tx_push(x_arr[i] * factor, y_arr[i] * factor, zradprim, 
                       x_arr[i] * factor, y_arr[i] * factor, zradprim);
        vertex_tx_push(factor * x_arr[(i + 1) % res],
                       factor * y_arr[(i + 1) % res],
                       zradprim, 
                       factor * x_arr[(i + 1) % res],
                       factor * y_arr[(i + 1) % res],
                       zradprim);
        vertex_tx_push(0.0, 0.0, z_rad, 0.0, 0.0, z_rad);
        polygon_push();
    }

    /* Create the surface between the poles. */
    factor2 = factor;
    zradprim2 = zradprim;
    for (j = 1; j < res / 2 - 1; j++) {
        factor1 = factor2;
        factor2 = sin((j + 1) * M_PI / (res / 2));
        zradprim1 = zradprim2;
        zradprim2 = z_rad * cos((j + 1) * M_PI / (res / 2));

        for (i = 0; i < res; i++) {
            vertex_tx_push(factor1 * x_arr[i], factor1 * y_arr[i], zradprim1, 
                           factor1 * x_arr[i], factor1 * y_arr[i], zradprim1);
            vertex_tx_push(factor2 * x_arr[i], factor2 * y_arr[i], zradprim2, 
                           factor2 * x_arr[i], factor2 * y_arr[i], zradprim2);
            vertex_tx_push(factor2 * x_arr[(i + 1) % res], 
                           factor2 * y_arr[(i + 1) % res], zradprim2, 
                           factor2 * x_arr[(i + 1) % res],
                           factor2 * y_arr[(i + 1) % res], zradprim2);
            vertex_tx_push(factor1 * x_arr[(i + 1) % res], 
                           factor1 * y_arr[(i + 1) % res], zradprim1, 
                           factor1 * x_arr[(i + 1) % res],
                           factor1 * y_arr[(i + 1) % res], zradprim1);
            polygon_push();
        }
    }

    /* Create the bottom pole */
    factor = sin(2.0 * M_PI / res);
    zradprim = -z_rad * cos(2.0 * M_PI / res);
    for (i = 0; i < res; i++) {
        vertex_tx_push(x_arr[(i + 1) % res] * factor,
                       y_arr[(i + 1) % res] * factor,
                       zradprim, 
                       x_arr[(i + 1) % res] * factor,
                       y_arr[(i + 1) % res] * factor,
                       zradprim);
        vertex_tx_push(x_arr[i] * factor, y_arr[i] * factor, zradprim, 
                       x_arr[i] * factor, y_arr[i] * factor, zradprim);
        vertex_tx_push(0.0, 0.0, -z_rad, 0.0, 0.0, -z_rad);
        polygon_push();
    }

    ellipsoid = object_create();
    object_add_surface(ellipsoid, surface_create(surface, shader));

    return ellipsoid;
}


Object *
sipp_sphere(radius, res, surface, shader)
    double  radius;
    int     res;
    void   *surface;
    Shader *shader;
{
    return sipp_ellipsoid(radius, radius, radius, res, surface, shader);
}
