/*      PROGRAM
 *              2dhist
 *
 *      SYNOPSIS
 *              2dhist [<x_image><y_image>] [-zhb] | pipeout
 *
 *      DESCRIPTION
 *              Creates a 256x256 hipl image output representing
 *              the two dimensional histogram of the image pair.
 *		Input images may be from a pipe or 2 specified files.
 *              Input images may be any size but must match. For
 *              example, input images may be in register views of
 *              the same source, masked to a region of interest.
 *              Output image is in integer format, ready to be
 *              SCALEd as desired.  By default, bin (0,0), which 
 *              represents paired zeros from background, is set
 *              to zero so as not to tip the scaling.  The [-z]
 *              option activates bin (0,0) when zero suppression
 *              is not desired.  A border is automatically added
 *		around the outside, inless the [-b] option is specified.
 *
 *      AUTHOR
 *              Justin D. Pearlman, M.D. M.E.
 *		Stuart Ware & Charles Carman
 *              for
 *              Merickel Imaging Laboratory
 *              Biomedical Engineering
 *              University of Virginia
 *              Charlottesville, VA 22908
 *
 */

#include <ctype.h>
#include <hipl_format.h>
#include <stdio.h>
#include <fcntl.h>

#define SIDE	256
#define HSIZE   SIDE * SIDE
char *Progname;

main(ac,av)
int ac;
char *av[];
{
	char *strchr();
	int sum, max;
	int zeroflag=0, helpflag=0, sequence=1, boardflag=0;
    	int fd1, fd2, h[SIDE][SIDE];
   	int row, col;
   	unsigned int size;
	register int i, *ph;
    	register unsigned char *buf1, *buf2;
    	struct header hd;
	void exit();

	Progname = strsave(*av);
	for (i=1;i<ac;i++)
	{
		if(av[i][0]=='-')
		{
			boardflag = (int)strchr(av[i],'b');
			helpflag = (int)strchr(av[i],'h');
			zeroflag = (int)strchr(av[i],'z');
		}
		else
			sequence = 0;
	}	

    	if (helpflag)
	{
		fprintf(stderr,
		"%s [<x_image> <y_image>] [-zh] | pipe\n",av[0]);
		fprintf(stderr,"-z allow zero bin in histogram\n");
		fprintf(stderr,"-h gives this help message\n");
		exit(1);
	}

	/* Using standard input */
	if (sequence)
	{
		read_header(&hd);
		if (hd.pixel_format != PFBYTE)
			perr("pixel_format must be BYTE");
		if (hd.num_frame<2)
			perr("must have a sequence of images greater than 1");
		row = hd.rows;
		col = hd.cols;
		size = row*col;
		buf1 = (unsigned char *)halloc(size,1);
		buf2 = (unsigned char *)halloc(size,1);
		if (pread(0,buf1,row*col*sizeof(unsigned char)) !=
			row*col*sizeof(unsigned char))
				perr("cannot read images");
		if (pread(0,buf2,row*col*sizeof(unsigned char)) !=
			row*col*sizeof(unsigned char))
				perr("cannot read images");
	}

	/* Using file names */
	else
	{
    		if ((fd1 = open(av[1],O_RDONLY)) == -1)
			perr("error opening %s",av[1]);

    		fread_header(fd1, &hd);

		/* Make sure this is correct format		*/
    		if (hd.pixel_format!=PFBYTE)
        		perr("must use PFBYTE format %s",av[1]);

    		row = hd.rows; 
   		col = hd.cols;

    		if ((fd2 = open(av[2],O_RDONLY)) == -1)
			perr("error opening %s",av[2]);

    		fread_header(fd2, &hd);

    		if (hd.pixel_format!=PFBYTE)
        		perr("must use PFBYTE format: %s",av[2]);

		if (hd.cols != col || hd.rows != row)
			perr("unequal image size");

		size = row*col;
		buf1 = (unsigned char *)halloc(size,1);
		buf2 = (unsigned char *)halloc(size,1);
		if (pread(fd1,buf1,row*col*sizeof(unsigned char)) !=
			row*col*sizeof(unsigned char))
				perr("cannot read images");
		if (pread(fd2,buf2,row*col*sizeof(unsigned char)) !=
			row*col*sizeof(unsigned char))
				perr("cannot read images");
	}

	/* Init bins */
	for (ph=h[0], i=0; i<HSIZE; i++) *ph++ = 0;

    	/*Fill bins*/
    	for (max=0,i=0;i<size;i++,buf1++,buf2++) 
	{
        	sum = ++h[*buf2][*buf1];
        	if (*buf1 != 0 || *buf2 != 0)
            		max = MAX(max, sum);
    	}

	if (zeroflag)
        	max = MAX(max, h[0][0]);
    	else 
		h[0][0]=0;

	/* add a border */
	if (!boardflag) {
	for (i=0; i<SIDE; i++) {
		if (h[i][0] == 0) h[i][0]++;
		if (h[0][i] == 0) h[0][i]++;
		if (h[i][SIDE-1] == 0) h[i][SIDE-1]++;
		if (h[SIDE-1][i] == 0) h[SIDE-1][i]++;
	} }

    	/*Unload results*/
	if (!sequence)
    		init_header(&hd,"","",1,"",SIDE,SIDE,8*sizeof(int),0,PFINT,"");
	else {
		for (i=2; i<hd.num_frame; i++) {
			pread(0,buf1,row*col*sizeof(unsigned char));
		}
		hd.num_frame = 1;
		hd.rows = hd.cols = SIDE;
		hd.pixel_format = PFINT;
	}
    	update_header(&hd,ac,av);
    	write_header(&hd);

	if (write(1,h,SIDE*SIDE*sizeof(int)) != SIDE*SIDE*sizeof(int))
		perr("error writing 2d histogram file");
	return(0);
}
