/*	Copyright (c) 1989 Michael Landy

Disclaimer:  No guarantees of performance accompany this software,
nor is any responsibility assumed on the part of the authors.  All the
software has been tested extensively and every effort has been made to
insure its reliability.   */

/* powerpix.c - raise to a power and normalize
 *		 each pixel of a frame.
 *
 * usage: powerpix [power] <frame >new_frame
 *
 * default power:  .5
 *
 * For byte images, pixels are renormalized to lie between 0 and 255.  For
 * integer and float images, the output is a float image and no
 * renormalization is performed.
 *
 *
 *
 * to load: cc -o powerpix powerpix.c -lhips -lm
 *
 * Yoav Cohen 2/16/82
 * added int/float - Mike Landy - 3/16/89
 *
 *  modified to use look-up table for byte and short images:
 *     Brian Tierney, LBL 10/90
 */

#include <hipl_format.h>
#include <math.h>
#include <stdio.h>
#include <sys/types.h>

char     *Progname;

u_char   *pic;
short    *spic;
int      *ipic;
float    *fpic;

#define MAXSHORT 65536

u_char    byte_lut[256];
float     short_lut[MAXSHORT];

main(argc, argv)
    int       argc;
    char     *argv[];
{
    double    power;
    struct header hd;
    int       form;

    Progname = strsave(*argv);
    if (argc == 1 || argv[1][0] == '-')
	power = .5;
    else
	power = atof(argv[1]);

    read_header(&hd);
    form = hd.pixel_format;
    if (form == PFBYTE) {
	hd.bits_per_pixel = 8;
	pic = (u_char *) halloc(hd.rows * hd.cols, sizeof(char));
    } else if (form == PFSHORT) {
	hd.bits_per_pixel = 32;
	hd.pixel_format = PFFLOAT;
	spic = (short *) halloc(hd.rows * hd.cols, sizeof(short));
	fpic = (float *) halloc(hd.rows * hd.cols, sizeof(float));
    } else if (form == PFINT) {
	hd.bits_per_pixel = 32;
	hd.pixel_format = PFFLOAT;
	ipic = (int *) halloc(hd.rows * hd.cols, sizeof(int));
	fpic = (float *) halloc(hd.rows * hd.cols, sizeof(float));
    } else if (form == PFFLOAT) {
	hd.bits_per_pixel = 32;
	fpic = (float *) halloc(hd.rows * hd.cols, sizeof(float));
    } else
	perr("input format must be byte, short, int, or float");

    update_header(&hd, argc, argv);
    write_header(&hd);

    if (form == PFBYTE)
	make_byte_lut(power);
    if (form == PFSHORT)
	make_short_lut(power);

    if (form == PFBYTE)
	bpowerpix(hd.num_frame, hd.rows, hd.cols);
    else if (form == PFSHORT)
	spowerpix(hd.num_frame, hd.rows, hd.cols, power);
    else if (form == PFINT)
	ipowerpix(hd.num_frame, hd.rows, hd.cols, power);
    else
	fpowerpix(hd.num_frame, hd.rows, hd.cols, power);

    return (0);
}

/*******************************************************************/
make_byte_lut(power)
    double    power;
{
    int       i, j;

    byte_lut[0] = 0;

    for (i = 1; i < 256; i++) {
	/* re-normalize to 0 to 255 */
	j = (u_char) (255. * (pow((double) (i & 0377) / 255., power)) + 0.5);
	byte_lut[i] = j;
    }
}

/*******************************************************************/
make_short_lut(power)
    double    power;
{
    int       i;

    short_lut[0] = 0.;

    for (i = 1; i < MAXSHORT; i++) {
	short_lut[i] = (float) (pow((double) i, power));
    }
}

/*******************************************************************/
bpowerpix(fr, r, c)
    int       fr, r, c;
{
    int       j, i, rc;

    rc = r * c;
    for (j = 0; j < fr; j++) {
	if (pread(0, pic, rc * sizeof(u_char)) != rc * sizeof(u_char))
	    perr("error during read");

	for (i = 0; i < rc; i++)
	    pic[i] = byte_lut[pic[i]];

	if (write(1, pic, rc * sizeof(char)) != rc * sizeof(char))
	    perr("error during write");
    }
    return (0);
}

/*******************************************************************/
spowerpix(fr, r, c, f)
    int       fr, r, c;
    float     f;
{
    int       j, i, rc;

    rc = r * c;
    for (j = 0; j < fr; j++) {
	if (pread(0, spic, rc * sizeof(short)) != rc * sizeof(short))
	    perr("error during read");

	for (i = 0; i < rc; i++) {
	    if (spic[i] >= 0)
		fpic[i] = short_lut[spic[i]];
	    else
		fpic[i] = pow((double) spic[i], f);
	}

	if (write(1, fpic, rc * sizeof(float)) != rc * sizeof(float))
	    perr("error during write");

    }
    return (0);
}

/*******************************************************************/
ipowerpix(fr, r, c, f)
    int       fr, r, c;
    double    f;
{
    int       j, i, rc;

    rc = r * c;
    for (j = 0; j < fr; j++) {
	if (pread(0, ipic, rc * sizeof(int)) != rc * sizeof(int))
	    perr("error during read");

	for (i = 0; i < rc; i++) {
	    if (ipic[i] == 0) {
		fpic[i] = 0;
	    } else
		fpic[i] = pow((double) ipic[i], f);
	}
	if (write(1, fpic, rc * sizeof(float)) != rc * sizeof(float))
	    perr("error during write");
    }
    return (0);
}

/*******************************************************************/
fpowerpix(fr, r, c, f)
    int       fr, r, c;
    double    f;
{
    int       j, i, rc;

    rc = r * c;
    for (j = 0; j < fr; j++) {
	if (pread(0, fpic, rc * sizeof(float)) != rc * sizeof(float))
	    perr("error during read");

	for (i = 0; i < rc; i++) {
	    if (fpic[i] != 0) {
		fpic[i] = pow((double) fpic[i], f);
	    }
	}
	if (write(1, fpic, rc * sizeof(float)) != rc * sizeof(float))
	    perr("error during write");
    }
    return (0);
}
