
/* addmask.c                                 Brian Tierney, LBL   4/90
 *
 *   usage:   addmask [-n] mask_image < image > new_image
 *
 *   creates new image which is the input image where the mask
 *    value is > 0, and zero everywhere else.
 *
 *  Works with data types: Byte, short, int, float, and complex.
 */

/*   This program is copyright (C) 1990, Regents  of  the
University  of  California.   Anyone may reproduce this software,
in whole or in part, provided that:
(1)  Any copy  or  redistribution  must  show  the
     Regents  of  the  University of California, through its
     Lawrence Berkeley Laboratory, as the source,  and  must
     include this notice;
(2)  Any use of this software must reference this  distribu-
     tion,  state that the software copyright is held by the
     Regents of the University of California, and  that  the
     software is used by their permission.

     It is acknowledged that the U.S. Government has  rights
to this software under  Contract DE-AC03-765F00098 between the U.S.
Department of Energy and the University of California.

     This software is provided as a professional  academic  contribu-
tion  for  joint exchange.  Thus it is experimental, is pro-
vided ``as is'', with no warranties of any kind  whatsoever,
no  support,  promise  of updates, or printed documentation.
Bug reports or fixes may be sent to the author, who may or may
not act on them as he desires. 
*/

/*   Author:  Brian L. Tierney
 *            Lawrence Berkeley Laboratory
 *            Imaging and Distributed Computing Group
 *            email: bltierney@lbl.gov
*/

#include <stdio.h>
#include <sys/types.h>
#include <math.h>

#include <hipl_format.h>

#define Calloc(a,b) (b *)calloc((unsigned)(a), sizeof(b))

char     *Progname;

u_char *mask_image;
char  *image, *new_image;
short  *image_s, *new_image_s;
int  *image_i, *new_image_i;
float  *image_f, *new_image_f;

int       nrow, ncol, nf;	/* number of rows, columns in the image */
int  fmt;  /* image format */
int  neg = 0;
char *mask_file;

/******************************************************/
main(argc, argv)
    int       argc;
    char    **argv;
{
    int       fr;
    int  rw_size;  /* number of bytes in a frame */
    FILE     *fp;

    struct header hd, bhd;	/* hips header */

    void      stat_objects(), usageterm();
    void      write_script_head(), write_script_tail();

    Progname = strsave(*argv);
    parse_args(argc, argv);

    read_header(&hd);

    fmt = hd.pixel_format;
    if (fmt != PFBYTE && fmt != PFSHORT && fmt != PFINT &&
	fmt != PFFLOAT && fmt != PFCOMPLEX)
	perr("image must be in byte, short, int, float, or complex format");

    nrow = hd.rows;		/* y axis */
    ncol = hd.cols;		/* x axis */
    nf = hd.num_frame;
    fprintf(stderr, "\n rows: %d,  cols: %d,  frames: %d \n", nrow, ncol, nf);

    if ((fp = fopen(mask_file, "r")) == NULL) {
	fprintf(stderr, "\n error opening binary image file \n\n");
	exit(-1);
    }

    fread_header(fileno(fp), &bhd);
    if (bhd.pixel_format != PFBYTE ||
	bhd.rows != hd.rows || bhd.cols != hd.cols)
	perr("binary mask image not correct format or size");

    update_header(&hd, argc, argv);
    write_header(&hd);

    mask_image = (u_char *) halloc(hd.rows * hd.cols, sizeof(u_char));

    if (fmt == PFBYTE) {
        image = (char *) halloc(hd.rows * hd.cols, sizeof(char));
        new_image = (char *) halloc(hd.rows * hd.cols, sizeof(char));
    } else  if (fmt == PFSHORT) {
        image_s = (short *) halloc(hd.rows * hd.cols, sizeof(short));
        new_image_s = (short *) halloc(hd.rows * hd.cols, sizeof(short));
    } else  if (fmt == PFINT) {
        image_i = (int *) halloc(hd.rows * hd.cols, sizeof(int));
        new_image_i = (int *) halloc(hd.rows * hd.cols, sizeof(int));
    } else  if (fmt == PFFLOAT) {
        image_f = (float *) halloc(hd.rows * hd.cols, sizeof(float));
        new_image_f = (float *) halloc(hd.rows * hd.cols, sizeof(float));
    } else  if (fmt == PFCOMPLEX) {
        image_f = (float *) halloc(2 * hd.rows * hd.cols, sizeof(float));
        new_image_f = (float *) halloc(2 * hd.rows * hd.cols, sizeof(float));
    }

    for (fr = 0; fr < nf; fr++) { /* process each frame in the sequence */

	if (nf > 1)
	    fprintf(stderr, "\n processing frame: %d ", fr);

	/* read mask image */
        if (pread(fileno(fp), mask_image, nrow * ncol) != nrow * ncol)
            perr("error during read");

	/* read image */
	if (fmt == PFBYTE) {
	    rw_size = hd.rows * hd.cols * sizeof(char);
	    if (pread(0, image, rw_size) != rw_size)
		perr("error during read");
	} else  if (fmt == PFSHORT) {
	    rw_size = hd.rows * hd.cols * sizeof(short);
	    if (pread(0, image_s, rw_size) != rw_size)
		perr("error during read");
	} else  if (fmt == PFINT) {
	    rw_size = hd.rows * hd.cols * sizeof(int);
	    if (pread(0, image_i, rw_size) != rw_size)
		perr("error during read");
	} else  if (fmt == PFFLOAT) {
	    rw_size = hd.rows * hd.cols * sizeof(float);
	    if (pread(0, image_f, rw_size) != rw_size)
		perr("error during read");
	} else  if (fmt == PFCOMPLEX) {
	    rw_size = 2 * hd.rows * hd.cols * sizeof(float);
	    if (pread(0, image_f, rw_size) != rw_size)
		perr("error during read");
	}

	combine_images();

	/* write image */
	if (fmt == PFBYTE) {
	    if (write(1, new_image, rw_size) != rw_size)
		perr("error during write");
	    bzero((char *)new_image, rw_size);
	} else  if (fmt == PFSHORT) {
	    if (write(1, new_image_s, rw_size) != rw_size)
		perr("error during write");
	    bzero((char *)new_image_s, rw_size);
	} else  if (fmt == PFINT) {
	    if (write(1, new_image_i, rw_size) != rw_size)
		perr("error during write");
	    bzero((char *)new_image_i, rw_size);
	} else  if (fmt == PFFLOAT) {
	    if (write(1, new_image_f, rw_size) != rw_size)
		perr("error during write");
	    bzero((char *)new_image_f, rw_size);
	} else  if (fmt == PFCOMPLEX) {
	    if (write(1, new_image_f, rw_size) != rw_size)
		perr("error during write");
	    bzero((char *)new_image_f, rw_size);
	}
    }	     

    fclose(fp);
    fprintf(stderr, "addmask done. \n");
    return (0);
}

/******************************************************/
combine_images()
{
    register int i,j;

    for (i=0; i< nrow * ncol; i++) {
	if ((neg == 0 && mask_image[i] > 0) ||
	    (neg > 0 && mask_image[i] == 0)) {
	    if (fmt == PFBYTE) {
		new_image[i] = image[i];
	    } else  if (fmt == PFSHORT) {
		new_image_s[i] = image_s[i];
	    } else  if (fmt == PFINT) {
		new_image_i[i] = image_i[i];
	    } else  if (fmt == PFFLOAT) {
		new_image_f[i] = image_f[i];
	    } else  if (fmt == PFCOMPLEX) {
		j = 2 * i;
		new_image_f[j] = image_f[j];
		j++;
		new_image_f[j] = image_f[j];
	    }
	}
    }
}

/******************************************************/
void
usageterm()
{
    fprintf(stderr, "Usage: addmask [-n] mask_image <  inseq > outseq  \n ");
    fprintf(stderr, "      [-n]  uses the negative of the mask \n\n");
    exit(0);
}

/****************************************************************/

parse_args(argc, argv)
    int       argc;
    char     *argv[];
{
    if (argc < 2)
	usageterm();

    /* Interpret options  */
    while (--argc > 0 && (*++argv)[0] == '-') {
        char     *s;
        for (s = argv[0] + 1; *s; s++)
            switch (*s) {
            case 'n':
		neg++;
                break;
            case 'h':
                usageterm();
                break;
            default:
                usageterm();
                break;
            }
    }
    mask_file = *argv;

    if ( mask_file == NULL)
	usageterm();
}
