/**********************************************************************/
/* help.c :                                                           */
/*                                                                    */
/* Help window for program.                                           */
/*                                                                    */
/* Copyright (C) 1992, Bernard Kwok                                   */
/* All rights reserved.                                               */
/* Revision 1.0                                                       */
/* May, 1992                                                          *
/**********************************************************************/
#include <stdio.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <string.h>
#include <gl/gl.h>
#include <gl/device.h>
#include <fmclient.h>
#ifdef RELEASE3.2
#include <malloc.h>
#else
#include <stdlib.h>
#endif
#define IRIS4D 1
#include "geo.h"
#include "walk.h"

/* Text fonts */
extern fmfonthandle FontScreen, FontScreen5, FontScreen8, FontScreen10;
extern fmfontinfo FontScreenInfo, FontScreen5Info, FontScreen8Info, 
    FontScreen10Info;

/* Size of help window, window id, and title */
static long HelpYsize;
static long HelpXsize;
long HelpWid = -1;
static char *HelpTitle = "Help for Walk";

/* Help text */
static char **Lines = 0;
char HelpText[] = "This program uses the properties of polygons\ninput in order to render the model.\n\nThe following describes the functionality of each mouse button.\n\nLeft              : Rotate the viewer about axis of the camera.\nMiddle            : Translate the viewer about axis of the camera.\nLeft+Right        : Zoom in or out of the model.\nLeft+Middle+Right : Scale in the z plane.\n\nThe following describes the functionality of each button in the main menu.\n\nHelp              : Click the HELP option for this help.\nDebug             : Print or suppress debugging messages.\nScene             : Options menu for scene.\nLog               : Options menu for logging movement.\n\nThe Scene menu allows options for:\n\nSave Image        : Save current image as RGB or black and white.\nGouraud or Flat   : Change to/from Gouraud to flat shading.\nTriangle or\nQuadralateral     : Change to/from quadralateral to trianguler meshed\n                    shading.\nScale             : Scale RGB values of polygons.\n\nThe Log menu allows options for:\n\nClear             : Clear log of movements.\nStart             : Start logging camera movements.\nStop              : Stop logging camera movements.\nSave              : Save current log to output file\nAutomate          : Read and exectute movements saved in file.\n\nCurrently only saving to IRIS imagelib image format is implemented\n";

static int NumLines = 0;          /* Number of lines of help */

/**********************************************************************/
void InitHelp();
long InitHelpWindow();
void DoEventsHelp();
void DrawHelp();
long CloseHelpWindow();
void DoExitHelp();

/**********************************************************************/
/* InitHelpWindow() - Display help window                             */
/**********************************************************************/
long InitHelpWindow(Wname)
     char *Wname;
{
  long Wid;

  foreground();
  prefsize(HelpXsize, HelpYsize);
  Wid = winopen(HelpTitle);
  if (Wid == -1) {
    fprintf(stderr, "No additional graphics windows are available\n");
    return -1;
  }
  gconfig();
  return Wid;
}

/**********************************************************************/
/* CloseHelpWindow() - Close help window, retrun window down.         */
/**********************************************************************/
long CloseHelpWindow(Wid)
     long Wid;
{
  winclose(Wid);
  return -1;
}

/**********************************************************************/
/* Allocate space for help text */
/**********************************************************************/
void InitHelp(Text)
     char Text[];
{
  char *c;
  int line;
  int num_chars, count;
  int file;
  FILE *fptr;
  int length;
  static help_initialized = 0;

  if (help_initialized) return;
    
  /* Find number of characters and number of lines, allocate space */
  for(num_chars = 0; Text[num_chars] != 0; num_chars++);
  for (c = Text; *c != 0; c++) {
    if (*c == '\n')
      NumLines++;
  }
  NumLines++;
  
  Lines = (char **) malloc(NumLines * sizeof(char *));
  
  line = 0;
  Lines[line++] = Text;
  
  for (c = Text; *c != 0; c++)
    if (*c == '\n') {
      *c = 0;
      Lines[line++] = c + 1;
    }
  
  /* Calculate x,y window size from text and font size */
  HelpYsize = NumLines * FontScreen10Info.height + 4;
  HelpXsize = 0;
  
  for (line = 0; line < NumLines; line++) {
    length = fmgetstrwidth(FontScreen10, Lines[line]);
    if (HelpXsize < length)
      HelpXsize = length;
  }
  HelpXsize += 4;

  help_initialized = 1;
}

/**********************************************************************/
/* Draw help text                                                     */
/**********************************************************************/
void DrawHelp()
{
  int line;
  long x, y, space;

  color(BLUE); clear();           /* Clear background                 */  
  color(YELLOW);                  /* Draw help text                   */
  fmsetfont(FontScreen10);
  space = FontScreen10Info.height;
  x = 2;
  y = HelpYsize - space;

  for (line = 0; line < NumLines; line++) {
    cmov2i(x, y);
    fmprstr(Lines[line]);
    y -= space;
  }
}

/**********************************************************************/
/* Exit help window */
/**********************************************************************/
void DoExitHelp(rc)
    int rc;
{
  if (Lines) free(Lines);
  rc = CloseHelpWindow(HelpWid);
}

/**********************************************************************/
/* Handle events for help window */
/**********************************************************************/
void DoEventsHelp(dev, val)
     long dev;
     short val;
{
  /* On redraw, redraw help text */
  winset(HelpWid);
  switch(dev) {
  case REDRAW:
    reshapeviewport();
    DrawHelp();
    break;
  default: break;
  }
}

/**********************************************************************/
/* Pop/unpop help */
/**********************************************************************/
void DoHelp()
{
  if (HelpWid == -1) {
    InitHelp();
    HelpWid = InitHelpWindow(HelpTitle);
    DrawHelp();
  } else
    HelpWid = CloseHelpWindow(HelpWid);
}
