
/*  @(#)expr.y 1.3 92/03/05
 *
 *  Yacc grammar used by the popi program.
 *
 *  Popi was originally written by Gerard J. Holzmann - AT&T Bell Labs.
 *  This version is based on the code in his Prentice Hall book,
 *  "Beyond Photography - the digital darkroom," ISBN 0-13-074410-7,
 *  which is copyright (c) 1988 by Bell Telephone Laboratories, Inc.
 *
 *  Permission is given to distribute these extensions, as long as these
 *  introductory messages are not removed, and no monies are exchanged.
 *
 *  No responsibility is taken for any errors or inaccuracies inherent
 *  either to the comments or the code of this program, but if reported
 *  (see README file) then an attempt will be made to fix them.
 */

%{

#include <stdio.h>
#include <sys/param.h>
#include "popi.h"
#include "expr.h"

#define  RET(x)          return(emit((x)))

enum word_type { W_Special, W_Function, W_Keyword } ;

static int check_buf     P((void)) ;
static int emit          P((int)) ;
static int Getch         P((void)) ;
static int GetNumber     P((int)) ;
static int is_word       P((enum word_type, char *)) ;
static int parse_default P((int)) ;
static int yylex         P((void)) ;
int yydebug = 0 ;

static void pushback     P((int)) ;
static void yyerror      P((char *)) ;

struct NameToken {
  char *name ;
  int token ;
} ;

static double lastfnum ;       /* Last floating point number parsed. */
static int lastinum ;          /* Last integer number parsed. */

static char buf[MAXLINE] ;

%}

%union {
  Tree *node ;
}

%token TOK_COLOR TOK_DEBUG TOK_DISPLAY TOK_FREE TOK_GENEPSON TOK_GENPS
%token TOK_GRAY TOK_HELP TOK_LIST TOK_LOGFILE TOK_MATTE TOK_MELT TOK_MONO
%token TOK_OFMT TOK_OIL TOK_QUIT TOK_READ TOK_SHEAR TOK_SIGNED TOK_SLICE
%token TOK_STORE TOK_TILE TOK_TRUNC TOK_UNDO TOK_VERBOSE TOK_VERSION TOK_WRITE

%token TOK_ABS TOK_ATAN TOK_COS TOK_HYPOT TOK_LOG TOK_RAND TOK_SIN TOK_SQRT

%token TOK_ICON TOK_FCON TOK_FNAME TOK_IMAGE TOK_NEW TOK_POW
%token TOK_OR TOK_AND TOK_EQ TOK_NE TOK_GE TOK_LE TOK_LSHIFT TOK_RSHIFT

%type <node> popi cmd special trans expr term
%type <node> or xor and eq rel shift plus times pow
%type <node> factor unaryop fncall0 fncall1 fncall2
%type <node> coord display dir logfile number
%type <node> file image

%start popi

%%

popi    : /* empty */                       { tree = NULL ; YYACCEPT ;     }
        | '\n'                              { tree = NULL ; YYACCEPT ;     }
        | error '\n'                        { tree = NULL ; YYABORT ;      }
        | cmd '\n'                          { tree = $1 ;   YYACCEPT ;     }
        ;

cmd     : special
        | '?'                               { $$ = mkempty(T_Help) ;       }
        | 'q'                               { $$ = mkempty(T_Quit) ;       }
        ;

special : trans
        | TOK_COLOR                         { $$ = mkempty(T_Color) ;      }
        | TOK_DEBUG      logfile            { $$ = mkun(T_Debug, $2) ;     }
        | TOK_DISPLAY    display            { $$ = mkun(T_Display, $2) ;   }
        | TOK_FREE       image              { $$ = mkun(T_Free, $2) ;      }
        | TOK_GENEPSON   file image         { $$ = mk(T_Genep, $2, $3) ;   }
        | TOK_GENPS      file image         { $$ = mk(T_Genps, $2, $3) ;   }
        | TOK_GRAY                          { $$ = mkempty(T_Gray) ;       }
        | TOK_HELP                          { $$ = mkempty(T_Help) ;       }
        | TOK_LIST                          { $$ = mkempty(T_List) ;       }
        | TOK_LOGFILE    logfile            { $$ = mkun(T_Logfile, $2) ;   }
        | TOK_MATTE      image number       { $$ = mk(T_Matte, $2, $3) ;   }
        | TOK_MELT       image              { $$ = mkun(T_Melt, $2) ;      }
        | TOK_MONO                          { $$ = mkempty(T_Mono) ;       }
        | TOK_OFMT       dir                { $$ = mkun(T_Ofmt, $2) ;      }
        | TOK_OIL        image              { $$ = mkun(T_Oil, $2) ;       }
        | TOK_QUIT                          { $$ = mkempty(T_Quit) ;       }
        | TOK_READ       file image         { $$ = mk(T_Read, $2, $3) ;    }
        | TOK_SHEAR      image              { $$ = mkun(T_Shear, $2) ;     }
        | TOK_SIGNED     dir                { $$ = mkun(T_Signed, $2) ;    }
        | TOK_SLICE      image              { $$ = mkun(T_Slice, $2) ;     }
	| TOK_STORE      image              { $$ = mkun(T_Store, $2) ;     }
        | TOK_TILE       image              { $$ = mkun(T_Tile, $2) ;      }
        | TOK_TRUNC      dir                { $$ = mkun(T_Trunc, $2) ;     }
        | TOK_UNDO                          { $$ = mkempty(T_Undo) ;       }
        | TOK_VERBOSE    dir                { $$ = mkun(T_Verbose, $2) ;   }
        | TOK_VERSION                       { $$ = mkempty(T_Ver) ;        }
        | TOK_WRITE      file image         { $$ = mk(T_Write, $2, $3) ;   }
        ;

trans   : expr
        | TOK_NEW coord '=' expr            { $$ = mk(T_New, $2, $4) ;     }
        ;

expr    : term
        | term '?' expr ':' expr            { $$ = mkCond($1, $3, $5) ;    }
        ;

term    : or
        | term TOK_OR or                    { $$ = mk(T_Lor, $1, $3) ;     }
        | term TOK_AND or                   { $$ = mk(T_Land, $1, $3) ;    }
        ;

or      : xor
        | or '|' xor                        { $$ = mk(T_Or, $1, $3) ;      }
        ;

xor     : and
        | xor '^' and                       { $$ = mk(T_Xor, $1, $3) ;     }
        ;

and     : eq
        | and '&' eq                        { $$ = mk(T_And, $1, $3) ;     }
        ;

eq      : rel
        | eq TOK_EQ rel                     { $$ = mk(T_Eq, $1, $3) ;      }
        | eq TOK_NE rel                     { $$ = mk(T_Ne, $1, $3) ;      }
        ;

rel     : shift
        | rel '>' shift                     { $$ = mk(T_Gt, $1, $3) ;      }
        | rel '<' shift                     { $$ = mk(T_Lt, $1, $3) ;      }
        | rel TOK_GE shift                  { $$ = mk(T_Ge, $1, $3) ;      }
        | rel TOK_LE shift                  { $$ = mk(T_Le, $1, $3) ;      }
        ;

shift   : plus
        | shift TOK_LSHIFT plus             { $$ = mk(T_Lshift, $1, $3) ;  }
        | shift TOK_RSHIFT plus             { $$ = mk(T_Rshift, $1, $3) ;  }
        ;

plus    : times
        | plus '+' times                    { $$ = mk(T_Add, $1, $3) ;     }
        | plus '-' times                    { $$ = mk(T_Sub, $1, $3) ;     }
        ;

times   : pow
        | times '*' pow                     { $$ = mk(T_Mul, $1, $3) ;     }
        | times '/' pow                     { $$ = mk(T_Div, $1, $3) ;     }
        | times '%' pow                     { $$ = mk(T_Mod, $1, $3) ;     }
        ;

pow     : factor
        | pow TOK_POW factor                { $$ = mk(T_Pow, $1, $3) ;     }
        ;

factor  : fncall2
        | 'x'                               { $$ = mkempty(T_Xcoord) ;     }
        | 'y'                               { $$ = mkempty(T_Ycoord) ;     }
        | 'a'                               { $$ = mkempty(T_Pang) ;       }
        | 'r'                               { $$ = mkempty(T_Prad) ;       }
        | TOK_ICON                          { $$ = mkInum(lastinum) ;      }
	| TOK_FCON                          { $$ = mkFnum(lastfnum) ;      }
        | unaryop factor                    { $$ = mkun((enum tree) $1, $2) ; }
        | image coord                       { $$ = mk(T_File, $1, $2) ;    }
        | '(' expr ')'                      { $$ = $2 ;                    }
        ;

fncall2 : fncall1
        | TOK_HYPOT '(' expr ',' expr ')'   { $$ = mk(T_Hypot, $3, $5) ;   }
        | TOK_ATAN  '(' expr ',' expr ')'   { $$ = mk(T_Atan, $3, $5) ;    }
        ;

fncall1 : fncall0
        | TOK_ABS  '(' expr ')'             { $$ = mkun(T_Abs, $3) ;       }
        | TOK_COS  '(' expr ')'             { $$ = mkun(T_Cos, $3) ;       }
        | TOK_LOG  '(' expr ')'             { $$ = mkun(T_Log, $3) ;       }
        | TOK_SIN  '(' expr ')'             { $$ = mkun(T_Sin, $3) ;       }
        | TOK_SQRT '(' expr ')'             { $$ = mkun(T_Sqrt, $3) ;      }
        ;

fncall0 : TOK_RAND '(' ')'                  { $$ = mkempty(T_Rand) ;       }
        ;

unaryop : '!'                               { $$ = mkempty(T_Bang) ;       }
        | '-'                               { $$ = mkempty(T_Neg) ;        }
        | '~'                               { $$ = mkempty(T_Not) ;        }
        ;

coord   : /* empty */                       { $$ = NULL ;                  }
        | '[' expr ',' expr ']'             { $$ = mk(T_Coord, $2, $4) ;   }
        | '{' expr ',' expr '}'             { $$ = mk(T_Polar, $2, $4) ;   }
        ;

display : /* empty */                       { $$ = NULL ;                  }
        | dir
        | image
        ;

logfile : /* empty */                       { $$ = NULL ;                  }
        | dir
        | file
        ;

number  : /* empty */                       { $$ = NULL ;                  }
        | TOK_ICON                          { $$ = mkInum(lastinum) ;      }
        | TOK_FCON                          { $$ = mkFnum(lastfnum) ;      }
        ;

image   : /* empty */                       { $$ = NULL ;                  }
        | TOK_IMAGE                { $$ = mkInum(getImageNo(lastimage)) ;  }
	| '$' TOK_ICON	                    { $$ = mkInum(lastinum + 1) ;  }
        ;

file    : TOK_FNAME                         { $$ = mkempty(T_File) ;       }
        ;

dir     : /* empty */                       { $$ = NULL ;                  }
        | '+'                               { $$ = mkempty(T_Plus) ;       }
        | '-'                               { $$ = mkempty(T_Minus) ;      }
        ;

%%

static struct NameToken SpecialToks[] = {   /* Also update popi.man. */
  { "color",       TOK_COLOR    },
  { "debug",       TOK_DEBUG    },
  { "display",     TOK_DISPLAY  },
  { "exit",        TOK_QUIT     },
  { "free",        TOK_FREE     },
  { "genepson",    TOK_GENEPSON },
  { "genps",       TOK_GENPS    },
  { "grayscale",   TOK_GRAY     },
  { "help",        TOK_HELP     },
  { "list",        TOK_LIST     },
  { "logfile",     TOK_LOGFILE  },
  { "matte",       TOK_MATTE    },
  { "melt",        TOK_MELT     },
  { "monochrome",  TOK_MONO     },
  { "ofmt",        TOK_OFMT     },
  { "oil",         TOK_OIL      },
  { "quit",        TOK_QUIT     },
  { "read",        TOK_READ     },
  { "shear",       TOK_SHEAR    },
  { "signed",      TOK_SIGNED   },
  { "slice",       TOK_SLICE    },
  { "store",       TOK_STORE    },
  { "tile",        TOK_TILE     },
  { "truncate",    TOK_TRUNC    },
  { "undo",        TOK_UNDO     },
  { "verbose",     TOK_VERBOSE  },
  { "version",     TOK_VERSION  },
  { "write",       TOK_WRITE    },
  { (char *) 0,    0            }
} ;

static struct NameToken FuncToks[] = {         /* Also update popi.man. */
  { "abs",         TOK_ABS   },
  { "atan",        TOK_ATAN  },
  { "cos",         TOK_COS   },
  { "hypot",       TOK_HYPOT },
  { "log",         TOK_LOG   },
  { "rand",        TOK_RAND  },
  { "sin",         TOK_SIN   },
  { "sqrt",        TOK_SQRT  },
  { (char *) 0,    0         }
} ;

static struct NameToken KeywordToks[] = {      /* Also update popi.man. */
  { "new",         TOK_NEW   },
  { (char *) 0,    0         }
} ;


static int
Getch()
{
  int c ;

  if (InputStream == (FILE *) 0)
    {
      if (SaveChar == '\0') c = disp_getchar() ;
      else
        { 
          c = SaveChar ;
          SaveChar = '\0' ;
        }
    }    
  else c = getc(InputStream) ;

  OldPos = CharPos ;

       if (c == '\t') CharPos = (CharPos - 1) % 8 + 8 ;
  else if (c == '\n') CharPos = 0 ;
  else                ++CharPos ;

  DEBUG((Debug, "Getch() => '%c'\n", c)) ;

  if (LogStr && CharPos != OldPos) PUTC(c, LogStr) ;

  return(c) ;
}


static int
GetNumber(first)
int first ;
{
  double fract ;
  int c, num ;

  num = first - '0' ; 
  fract = 0.0 ; 
  while (isdigit(c = Getch())) 
    num = 10 * num + c - '0' ; 
 
  if (c == '.') 
    {
      double div = 10.0 ; 
 
      while (isdigit(c = Getch())) 
        { 
          fract += (c - '0') / div ; 
          div *= 10.0 ; 
        }    
    }     
  pushback(c) ; 
  if (fract == 0.0)  
    { 
      lastinum = num ; 
      return(TOK_ICON) ; 
    }
  lastfnum = (double) num + fract ;
  return(TOK_FCON) ;
}


static int
emit(token)
int token ;
{
  HaveToken = 1 ;
  return(token) ;
}


static void
pushback(c)
int c ;
{
  DEBUG((Debug, "pushback('%c')\n", c)) ;
  if (InputStream == (FILE *) 0) SaveChar = c ;
  else
    UNGETC(c, InputStream) ;
  CharPos = OldPos ;
}


static int
is_word(w, buf)
enum word_type w ;
char *buf ;
{
  struct NameToken *nt, *start ;
  int cnt, len, token ;

       if (w == W_Special)  start = nt = &SpecialToks[0] ;
  else if (w == W_Function) start = nt = &FuncToks[0] ;
  else if (w == W_Keyword)  start = nt = &KeywordToks[0] ;
  for (cnt = 0; nt->name; nt++)
    if (strncmp(buf, nt->name, strlen(buf)) == 0)
      {
        token = nt->token ;
        cnt++ ;
      }
  if (!cnt) return(0) ;
  if (cnt > 1)
    {
      SPRINTF(ErrBuf, "Operation '%s' ambiguous (", buf) ;
      for (nt = start; nt->name; nt++)
        if (strncmp(buf, nt->name, strlen(buf)) == 0)
          {
            STRCAT(ErrBuf, nt->name) ;
            STRCAT(ErrBuf, ", ") ;
          }
      len = strlen(ErrBuf) ;
      ErrBuf[len-2] = ')';
      ErrBuf[len-1] = '\n';
      error(ERR_PARSE) ;
      return(0) ;
    }
  return(token) ;
}


static int
check_buf()
{
  int i ;

  if (strlen(lastimage) == 1)
    switch (*lastimage)
      {
        case 'x' :
        case 'y' :
        case 'a' :
        case 'r' :
        case 'q' : return(*lastimage) ;
        case 'A' : lastinum = 360 ;
                   return(TOK_ICON) ;
        case 'R' : lastinum = (int)
                     hypot(Xsize / 2.0, Ysize / 2.0) ;
                   return(TOK_ICON) ;
        case 'X' : lastinum = Xsize-1 ;
                   return(TOK_ICON) ;
        case 'Y' : lastinum = Ysize-1 ;
                   return(TOK_ICON) ;
        case 'Z' : lastinum = Zmax ;
                   return(TOK_ICON) ;
      }
  if ((i = is_word(W_Keyword,  lastimage))) return(i) ;
  if ((i = is_word(W_Function, lastimage))) return(i) ;
  return(TOK_IMAGE) ;
}


static int
parse_default(c)
int c ;
{
  char *l ;
 
  if (isdigit(c)) return(GetNumber(c)) ;
  else
    { 
      l = lastimage ;
      for (;;)
        {
          *l++ = c ;
          switch (c = Getch())
            {
              case EOF  :
              case ' '  :
              case '\t' :
              case '\n' :
              case '?'  :
              case ':'  :
              case '-'  :
              case '+'  :
              case '*'  :
              case '/'  :
              case '%'  :
              case '~'  :
              case '^'  :
              case '('  :
              case ')'  :
              case '<'  :
              case '>'  :
              case '{'  :
              case '}'  :
              case '['  :
              case ']'  :
              case '&'  :
              case '|'  :
              case '='  :
              case '!'  :
              case '\"' :
              case ','  : pushback(c) ;
                          *l = '\0' ;
                          return(check_buf()) ;
            }
        }
    }
}


static int
yylex()
{
  char *l ;
  int c, i, special ;

  while ((c = Getch()) == ' ' || c == '\t') continue ;
  TokPos = CharPos ;
  switch (c)
    {
      case EOF  : saw_eof = 1 ;
                  RET(0) ;
      case ':'  :
      case '#'  : special = c ;
                  if (HaveToken) RET(c) ;
                  l = buf ;
                  while (isalnum(c = Getch())) *l++ = c ;
                  pushback(c) ;
                  *l = '\0' ;
                  if ((i = is_word(W_Special, buf))) RET(i) ;
                  RET(special) ;
      case '\n' :
      case '?'  :
      case '-'  :
      case '+'  :
      case '/'  :
      case '%'  :
      case '~'  :
      case '^'  :
      case '('  :
      case ')'  :
      case '{'  :
      case '}'  :
      case '['  :
      case ']'  :
      case ','  :
      case '$'  : RET(c) ;
      case '*'  : if ((c = Getch()) == '*') RET(TOK_POW) ;
                  pushback(c) ;
                  RET('*') ;
      case '&'  : if ((c = Getch()) == '&') RET(TOK_AND) ;
                  pushback(c) ;
                  RET('&') ;
      case '|'  : if ((c = Getch()) == '|') RET(TOK_OR) ;
                  pushback(c) ;
                  RET('|') ;
      case '>'  : if ((c = Getch()) == '>') RET(TOK_RSHIFT) ;
                  else if (c == '=')        RET(TOK_GE) ;
                  pushback(c) ;
                  RET('>') ;
      case '<'  : if ((c = Getch()) == '<') RET(TOK_LSHIFT) ;
                  else if (c == '=')        RET(TOK_LE) ;
                  pushback(c) ;
                  RET('<') ;
      case '='  : if ((c = Getch()) == '=') RET(TOK_EQ) ;
                  pushback(c) ;
                  RET('=') ;
      case '!'  : if ((c = Getch()) == '=') RET(TOK_NE) ;
                  pushback(c) ;
                  RET('!') ;
      case '\"' : l = lastfname ;
                  while ((c = Getch()) != '\"' && c != '\n' && c != EOF)
                    *l++ = c ;
                  if (c != '\"') pushback(c) ;
                  *l = '\0' ;
                  RET(TOK_FNAME) ;
      default   : RET(parse_default(c)) ;
    }
}


static void
yyerror(err)
char *err ;
{
  STRCPY(ErrBuf, err) ;
  error(ERR_PARSE) ;
}
