
/*  @(#)alloc.c 1.2 91/12/30
 *
 *  A simple single-arena allocator for command-line-lifetime allocation.
 *
 *  Popi was originally written by Gerard J. Holzmann - AT&T Bell Labs.
 *  This version is based on the code in his Prentice Hall book,
 *  "Beyond Photography - the digital darkroom," ISBN 0-13-074410-7,
 *  which is copyright (c) 1988 by Bell Telephone Laboratories, Inc.
 *
 *  Permission is given to distribute these extensions, as long as these
 *  introductory messages are not removed, and no monies are exchanged.
 *
 *  No responsibility is taken for any errors or inaccuracies inherent
 *  either to the comments or the code of this program, but if reported
 *  (see README file) then an attempt will be made to fix them.
 */

#include "popi.h"
#include "expr.h"

typedef struct Block Block ;

static struct Block {
  size_t used, size ;
  char *mem ;
  Block *n ;
} *fl, *ul ;

/*  alignto() works only with power of 2 blocks and assumes 2's complement
 *  arithmetic.
 */

#define  alignto(m, n)  ((m + n - 1) & ~(n - 1))
#define  BLOCKSIZE      ((size_t) 4096)

void *
ealloc(n)
size_t n ;
{
  void *p = (void *) LINT_CAST(malloc((unsigned int) n)) ;
  if (p == NULL)
    {
      perror("malloc") ;
      exit(1) ;
    }
  return(p) ;
}


/*  Gets a block from malloc space and places it at the head of the
 *  used-list.
 */

static void
getblock(n)
size_t n ;
{
  Block *r, *p ;

  for (r = fl, p = NULL; r != NULL; p = r, r = r->n)
    if (n <= r->size)
      break ;

  if (r != NULL)
    {
      if (p != NULL) p->n = r->n ;
      else           fl = r->n ;
    }
  else
    {
      r       = (Block *) LINT_CAST(ealloc(sizeof *r)) ;
      r->mem  = (char *) LINT_CAST(ealloc(alignto(n, BLOCKSIZE))) ;
      r->size = alignto(n, BLOCKSIZE) ;
    }

  r->used = 0 ;
  r->n = ul ;
  ul = r ;
}


/*  A fast single-arena allocator. Looks at the current block, and if there
 *  is not enough room, it goes to getblock() for more. "ul" stands for
 *  "used list", and the head of the list is the current block.
 */

void *
alloc(n)
size_t n ;
{
  char *ret ;

  n = alignto(n, sizeof(size_t)) ; /* A good guess about how to align data? */
  if (ul == NULL || n + ul->used >= ul->size)
    getblock(n) ;
  ret = ul->mem + ul->used ;
  ul->used += n ;
  return((void *) LINT_CAST(ret)) ;
}


/*  Frees memory from nalloc space by putting it on the freelist. Returns
 *  free blocks to the system, retaining at least MAXMEM bytes worth of
 *  blocks for nalloc.
 */

#define  MAXMEM  500000

void
afree()
{
  Block *r ;
  size_t count ;
  if (ul == NULL) return ;
  for (r = ul; r->n != NULL; r = r->n) continue ;
  r->n = fl ;
  fl = ul ;
  ul = NULL ;
  for (r = fl, count = r->size; r->n != NULL; r = r->n, count += r->size)
    {
      if (count >= MAXMEM)
        {
          Block *tmp = r ;

          r = r->n ;
          tmp->n = NULL ;      /* Terminate the freelist. */
          while (r != NULL)    /* Free memory off the tail of the freelist. */
            {
              tmp = r->n ;
              FREE(r->mem) ;
              FREE((char *) r) ;
              r = tmp ;
            }
          return ;
        }
    }
}
