/* Copyright (c) 1994 Gregory P. Ward.  All rights reserved.
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose, without fee, and without written
 * agreement is hereby granted, provided that the above copyright
 * notice and the following two paragraphs appear in all copies of
 * this software.  
 *
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE
 * AUTHOR HAS BEEN ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * THE AUTHOR SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT
 * NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER
 * IS ON AN "AS IS" BASIS, AND THE AUTHOR HAS NO OBLIGATION TO PROVIDE
 * MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 */

/* ----------------------------- MNI Header -----------------------------------
@NAME       : buffers.c
@INPUT      : 
@OUTPUT     : 
@RETURNS    : 
@DESCRIPTION: Functions and variables relating to frame-buffering 
              in the GL MPEG player.
@METHOD     : 
@GLOBALS    : BufferInitialized
              TempFile
	      FrameBufferLimit
	      NumBuffered
	      FrameBuffer
@CALLS      : 
@CREATED    : 94/6/??, Greg Ward.
@MODIFIED   : 
---------------------------------------------------------------------------- */
#undef DEBUG

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include "gl_mpeg.h"


/* Global variables used for buffering images */

Boolean  BufferInitialized = FALSE;
FILE    *TempFile;
int      FrameBufferLimit;
int      NumBuffered = 0;
char   **FrameBuffer;


/*
 * Functions for disk buffering (only called by SetupBuffer and SaveFrame).
 */

/* ----------------------------- MNI Header -----------------------------------
@NAME       : CreateDiskBuffer
@INPUT      : 
@OUTPUT     : 
@RETURNS    : FALSE if unable to create temporary file 
              (error message written to stderr)
	      TRUE if successful
@DESCRIPTION: Generates a temporary filename and creates the file.
              Unlinks the file so that it will be removed when closed.
@METHOD     : 
@GLOBALS    : KeepTempFlag
@CALLS      : 
@CREATED    : 94/6/20, Greg Ward
@MODIFIED   : 
---------------------------------------------------------------------------- */
Boolean CreateDiskBuffer (void)
{
   char     *TempFilename;

   TempFilename = tempnam (NULL, "MPEG");
   if (TempFilename == NULL)
   {
      fprintf (stderr, "Error devising name for temporary file\n");
      return (FALSE);
   }
   dprintf ("Creating temporary file %s\n", TempFilename);
   TempFile = fopen (TempFilename, "w+");

   if (TempFile == NULL)
   {
      fprintf (stderr, "Unable to open temporary file: %s\n",
	       strerror (errno));
      return (FALSE);
   }
   else
   {
      if (!KeepTempFlag)
      {
	 unlink (TempFilename);	 /* file will be deleted when closed */
      }
      return (TRUE);
   }
}


/* ----------------------------- MNI Header -----------------------------------
@NAME       : WriteFrame
@INPUT      : ImgInfo - struct used to determine the image size
              ImageData - the actual image (should be ImgInfo->Size bytes
	                  long!)
@OUTPUT     : the image is written to the disk buffer
@RETURNS    : FALSE if fwrite() fails (error message printed on stderr)
              TRUE otherwise
@DESCRIPTION: Writes a frame at the current location in the temporary 
              file created by CreateDiskBuffer().
@METHOD     : 
@GLOBALS    : TempFile
@CALLS      : 
@CREATED    : 94/6/20, Greg Ward
@MODIFIED   : 
---------------------------------------------------------------------------- */
Boolean WriteFrame (ImageDesc *ImgInfo, char *ImageData)
{
   int	NumItems;
   
   dprintf (" writing at offset %d in temp file\n", 
	    ftell (TempFile));
   NumItems = fwrite ((void *) ImageData, sizeof(char), 
		      ImgInfo->Size, TempFile);

   if (NumItems != ImgInfo->Size)
   {
      fprintf (stderr, "Error writing to temp file (probably disk full)\n(try -buffer none, or free up some space on /tmp)\n");
      return (FALSE);
   }
   return (TRUE);
}


/*
 * Functions for memory buffering (only called by SetupBuffer and SaveFrame).
 */


Boolean CreateMemoryBuffer (void)
{
   NumBuffered = 32;		/* somewhat arbitrary number */
   FrameBuffer = (char **) calloc (NumBuffered, sizeof (char *));
   return (FrameBuffer != NULL);
}


Boolean ExtendMemoryBuffer (void)
{
   NumBuffered *= 2;
   dprintf ("  extending memory buffer: now can hold up to %d frames\n",
	    NumBuffered);
   FrameBuffer = (char**) realloc (FrameBuffer, NumBuffered * sizeof (char *));
   return (FrameBuffer != NULL);
}


Boolean AllocAndCopyFrame (ImageDesc *ImageInfo, char **New, char *Old)
{
   *New = (char *) malloc (ImageInfo->Size);
   if (*New == NULL) return (FALSE);
   dprintf (" allocated %d bytes at %08X, copying image data from %08X\n",
	    ImageInfo->Size, *New, Old);
   memcpy (*New, Old, ImageInfo->Size);
   return (TRUE);
}




/* ----------------------------- MNI Header -----------------------------------
@NAME       : FallBack
@INPUT      : 
@OUTPUT     : 
@RETURNS    : 
@DESCRIPTION: 
@METHOD     : 
@GLOBALS    : 
@CALLS      : 
@CREATED    : 94/7/20, Greg Ward (partly taken from SaveFrame())
@MODIFIED   : 
---------------------------------------------------------------------------- */
Boolean FallBack (int FrameNum, ImageDesc *Image, char *ImageData)
{
   int   Frame;

   switch (BufferFallback)
   {
      case NO_BUFFER:
      {
	 qprintf ("\nFrame buffer limit exceeded, abandoning buffering\n");
	 if (BufferType == DISK_BUFFER)
	 {
	    fclose (TempFile);
	 }
	 else
	 {
	    for (Frame = 0; Frame < FrameNum; Frame++)
	    {
	       free (FrameBuffer [Frame]);
	    }
	 }
	 
	 BufferType = NO_BUFFER;
	 break;
      }
	 
      case DISK_BUFFER:
      {
	 qprintf ("\nFrame buffer limit exceeded, switching to disk buffering\n");

	 BufferType = DISK_BUFFER;
	 if (!CreateDiskBuffer ()) return (FALSE);
	 
	 /* Save the frames already in FrameBuffer[] */
	 
	 for (Frame = 0; Frame < FrameNum; Frame++)
	 {
	    if (!WriteFrame (Image, FrameBuffer [Frame])) 
	       return (FALSE);
	    free (FrameBuffer [Frame]);
	 }
	 
	 /* And now the current frame (the one that WAS supposed to
	  * go into memory)
	  */
	 
	 if (!WriteFrame (Image, ImageData))
	    return (FALSE);
	 
	 break;
      }	 
   }     /* switch on BufferFallback */
}     /* FallBack () */




/* 
 * Functions to interface to either type of buffering (called by 
 * the outside world).
 */



/* ----------------------------- MNI Header -----------------------------------
@NAME       : SetupBuffer
@INPUT      : BufType - the type of buffering, either DISK_BUFFER 
                        or MEMORY_BUFFER
	      ImageInfo - struct to tell the size of images (possibly 
	                  needed to compute frame limit from memory limit)
	      MaxMemory - the maximum amount of memory, in megabytes, that
	                  we are allowed to allocate for memory-buffered
			  images.  MaxMemory is ignored unless MaxFrames
			  is negative.
	      MaxFrames - the maximum number of frames we are allowed to
	                  buffer in memory.  If MaxFrames is negative,
			  it is ignored and SetupBuffer() falls back on
			  MaxMemory to compute the maximum allowed number
			  of frames.  If both MaxFrames and MaxMemory
			  are negative or zero, then there will be no
			  limit to the memory allocated.
@OUTPUT     : 
@RETURNS    : FALSE if any error creating either type of buffer
              TRUE otherwise
@DESCRIPTION: Initializes frame buffers, either a temporary file or
              an array of char *'s.  Note that no significant amount
	      of memory or disk space is actually allocated here -- that
	      is done by SaveFrame(), once images are actually available.
@METHOD     : 
@GLOBALS    : TempFile, TempFilename (implicitly via CreateDiskBuffer)
              NumBuffered, FrameBuffer (implicity via CreateMemoryBuffer)
@CALLS      : CreateDiskBuffer()
              CreateMemoryBuffer()
@CREATED    : 94/6/21, Greg Ward
@MODIFIED   : 
---------------------------------------------------------------------------- */
Boolean SetupBuffer (BufferEnum BufType, 
		     ImageDesc *ImageInfo, 
		     int MaxMemory, int MaxFrames)
{
   switch (BufType)
   {
      case DISK_BUFFER:
      {
	 if (!CreateDiskBuffer ()) return (FALSE);
	 break;
      }
      case MEMORY_BUFFER:
      {
	 if (!CreateMemoryBuffer ()) return (FALSE);

	 /* 
	  * Calculate FrameBufferLimit, the maximum number of frames we're
	  * allowed to buffer in memory: 
	  *   - if MaxFrames is negative and MaxMemory is positive then
	  *     FrameBufferLimit will be calculated from MaxMemory;
	  *   - if both are negative, FrameBufferLimit will be 0 (unlimited);
	  *   - if MaxFrames is non-negative, FrameBufferLimit will be
	  *     set to MaxFrames (this allows 0, or unlimited)
	  */

	 if (MaxFrames < 0)
	    if (MaxMemory > 0)
	       FrameBufferLimit = (MaxMemory*1024*1024) / ImageInfo->Size;
	    else
	       FrameBufferLimit = 0;
		  
	 else
	    FrameBufferLimit = MaxFrames;

	 dprintf ("SetupBuffer: frame limit is %d frames (%d MB)\n",
		  FrameBufferLimit, MaxMemory);
	 break;
      }
   }
   BufferInitialized = TRUE;
   return (TRUE);
}     /* SetupBuffer () */



/* ----------------------------- MNI Header -----------------------------------
@NAME       : CloseBuffer
@INPUT      : BufType - the type of buffering (either DISK_BUFFER 
                        or MEMORY_BUFFER)
	      NumFrames - the number of frames stored in the buffer
                          (actually only used for memory buffering)
@OUTPUT     : (none)
@RETURNS    : (void)
@DESCRIPTION: Frees up the disk or memory space used to buffer the frames
              of a movie.
@METHOD     : 
@GLOBALS    : TempFile, FrameBuffer
@CALLS      : 
@CREATED    : 94/6/22, Greg Ward
@MODIFIED   : 
---------------------------------------------------------------------------- */
void CloseBuffer (BufferEnum BufType, int NumFrames)
{
   switch (BufType)
   {
      case DISK_BUFFER:
      {
	 fclose (TempFile);	/* file already unlinked - closing */
	 break;			/* will remove i */
      }
      case MEMORY_BUFFER:
      {
	 int   i;

	 for (i = 0; i < NumFrames; i++)
	 {
	    free (FrameBuffer [i]);
	 }
	 break;
      }
   }
}     /* CloseBuffer () */





/* ----------------------------- MNI Header -----------------------------------
@NAME       : SaveFrame
@INPUT      : FrameNum - zero-based frame counter
	      ImageInfo - pointer to struct describing the size (and
	                  other pertinent info) of the image
	      ImageData - pointer to the actual image data, as returned
	                  by the MPEG library routine GetMPEGFrame().
@OUTPUT     : 
@RETURNS    : FALSE if writing to a disk buffer fails (eg. disk full)
              FALSE if not enough memory to either buffer an image or
	         extend the memory buffer frame list
	      FALSE (with *BufType reset to NO_BUFFER) if FrameBufferLimit
	         is exceeded
	      TRUE otherwise
@DESCRIPTION: Saves a frame to the currently-active buffer type.  If
	      the buffer type is DISK_BUFFER, then WriteFrame() is
	      called to write the frame to disk.  If MEMORY_BUFFER, a
	      new element of FrameBuffer[] is allocated (N.B. the
	      caller must assume responsibility for incrementing
	      FrameNum with each frame!), and the image is copied to
	      the new area.

	      Note that in the case of MEMORY_BUFFER, if
	      FrameBufferLimit is exceeded, then SaveFrame will return
	      FALSE, but *BufType will be reset to NO_BUFFER.  All
	      other error conditions will simply return FALSE without
	      chaning *BufType.  (An alternate approach to this would
	      be to have SaveFrame() silently switch to disk
	      buffering, save all memory-buffered images to disk, and
	      deallocate the buffer memory -- thus it would look as
	      though disk buffering had been used all along.)

	      Note that for either type of buffering, the caller 
	      should call SetupBuffer() before any calls to SaveFrame().
	      If this is not done, SaveFrame() will call SetupBuffer()
	      to do the required tasks, but the user will not be able
	      to specify a maximum allowed amount of memory to allocate
	      (it will be set to unlimited).	      

@METHOD     : 
@GLOBALS    : BufferInitialized
              NumBuffered
	      FrameBufferLimit
	      BufferType
	      TempFile, TempFilename (implicitly, via WriteFrame)
	      FrameBuffer (implicitly, via AllocAndCopyFrame)
@CALLS      : WriteFrame (for DISK_BUFFER)
              AllocAndCopyFrame (for MEMORY_BUFFER)
@CREATED    : 94/6/21, Greg Ward
@MODIFIED   : 94/7/20, GW: used global BufferType rather than passing argument
---------------------------------------------------------------------------- */
Boolean SaveFrame (int FrameNum, ImageDesc *ImageInfo, char *ImageData)
{
   if (!BufferInitialized && FrameNum == 0)
   {
      SetupBuffer (BufferType, ImageInfo, 0, 0);
   }

   dprintf ("SaveFrame: saving frame %d: ", FrameNum);
   switch (BufferType)
   {
      case DISK_BUFFER:
      {
	 if (!WriteFrame (ImageInfo, ImageData))
	    return (FALSE);
	 break;
      }
      case MEMORY_BUFFER:
      {
	 if (FrameNum >= NumBuffered)
	    ExtendMemoryBuffer ();

	 if (FrameBufferLimit > 0 && FrameNum >= FrameBufferLimit)
	 {
	    return (FallBack (FrameNum, ImageInfo, ImageData));
	 }

	 AllocAndCopyFrame (ImageInfo, FrameBuffer+FrameNum, ImageData);
	 break;
      }
   }

   return (TRUE);
	 
}



/* ----------------------------- MNI Header -----------------------------------
@NAME       : LoadFrame
@INPUT      : FrameNum - which frame to retrieve (zero-based!)
	      ImageInfo - struct telling the size of the movie's frames
@OUTPUT     : (none)
@RETURNS    : Pointer to a static data area containing the desired 
              movie frame.  Caller should not modify this area; it will
	      be overwritten on the next call to LoadFrame.
@DESCRIPTION: Load a movie frame that was previously saved with SaveFrame().
@METHOD     : 
@GLOBALS    : TempFile, FrameBuffer, BufferType
@CALLS      : 
@CREATED    : 94/6/21, Greg Ward
@MODIFIED   : 94/7/20, GW: used global BufferType rather than passing argument
@COMMENTS   : memory leak in case of disk buffering, but not memory
              buffering (ie. CloseBuffer() won't take care of free()'ing
	      the local static ImageData)
---------------------------------------------------------------------------- */
char *LoadFrame (int FrameNum, ImageDesc *ImageInfo)
{
   static char  *ImageData = NULL;

   dprintf ("LoadFrame: loading frame %d ", FrameNum);
   switch (BufferType)
   {
      case DISK_BUFFER:
      {
	 int   NumItems;

	 if (ImageData == NULL)	         /* first retrieved image? */
	 {
	    ImageData = (char *) malloc (ImageInfo->Size);
	 }
	 
	 fseek (TempFile, ImageInfo->Size*FrameNum, SEEK_SET);
	 dprintf ("from offset %d in file\n",
		 ftell (TempFile));

	 NumItems = fread (ImageData, sizeof(char), ImageInfo->Size, TempFile);
	 if (NumItems != ImageInfo->Size)
	 {
	    fprintf (stderr, "Error reading from temporary file\n");
	    return (NULL);
	 }
	 break;
      }
      case MEMORY_BUFFER:
      {
	 if (FrameNum < NumBuffered)
	 {
	    ImageData = FrameBuffer [FrameNum];
	    dprintf ("from address %08X in memory\n", ImageData);
	 }
	 else
	 {
	    fprintf (stderr, "INTERNAL ERROR: Attempt to retrieve invalid frame number (%d) from buffer\n", FrameNum);
	    return (NULL);
	 }
	 break;
      }     /* case MEMORY_BUFFER */
   }     /* switch on BufferType */
   return (ImageData);
}     /* LoadFrame () */
