/*
 *----------------------------------------------------------------------
 *
 * Copyright 1991, University of New Mexico.  All rights reserved.
 * Permission to copy and modify this software and its documen-
 * tation only for internal use in your organization is hereby
 * granted, provided that this notice is retained thereon and
 * on all copies.  UNM makes no representations as to the sui-
 * tability and operability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 * 
 * UNM DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FIT-
 * NESS.  IN NO EVENT SHALL UNM BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY OTHER DAMAGES WHAT-
 * SOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PER-
 * FORMANCE OF THIS SOFTWARE.
 * 
 * No other rights, including, for example, the right to redis-
 * tribute this software and its documentation or the right to
 * prepare derivative works, are granted unless specifically
 * provided in a separate license agreement.
 *---------------------------------------------------------------------
 */

#include "unmcopyright.h"        /* Copyright 1991 by UNM */

/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>  <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
 >>>>
 >>>>         File Name: lvconic.c
 >>>>
 >>>>      Program Name: vconic
 >>>>
 >>>> Date Last Updated: Tue Apr 23 01:36:49 1991 
 >>>>
 >>>>          Routines: lvconic - the library call for vconic
 >>>>
 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>   <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<*/


#include "vinclude.h"


/* -library_includes */
/* -library_includes_end */


/****************************************************************
*
* Routine Name: lvconic - library call for vconic
*
* Purpose:
*    
*    Creates an image of a conic section of revolution
*    
*    
* Input:
*    
*    image          input image structure
*    
*    rows           the desired number of rows in output image
*    
*    cols           the desired number of columns in output image
*    
*    type           the data storage type of the resulting image
*    
*    radius         the radius parameter (meters)
*    
*    conic          the conic constant
*    
*    cx             the x pixel coordinate of the origin of the physi-
*                   cal coordinate system
*    
*    cy             the y pixel coordinate of the origin of the physi-
*                   cal coordinate system
*    
*    x              the X offset of the center of symmetry  (meters)
*    
*    y              the Y offset of the center of symmetry (meters)
*    
*    s              the size if a pixel (meters)
*    
*    cutoff         cutoff radius (meters)
*    
*    piston         piston offset (meters)
*    
*    branch         if 0, take negative branch of quadratic else  take
*                   positive branch
*    
*    
* Output:
*    
*    image          holds the resulting output image.
*    
*    
*
* Written By: Scott Wilson
*    
*    
****************************************************************/


/* -library_def */
int
lvconic(image,rows,cols,type,radius,conic,cx,cy,x,y,s,cutoff,piston,branch)
struct xvimage **image;
int rows,cols,type,branch,cx,cy;
float radius,conic,x,y,s,cutoff,piston;
/* -library_def_end */

/* -library_code */
{
  int i,j;
  unsigned char *c;
  float *f,xhat,yhat;
  double *d,z,eval_conic();
  struct xvimage *img, *createimage();

  img   = createimage((unsigned long) rows,  /* number of rows */
                     (unsigned long) cols,   /* number of columns */
                     (unsigned long) type,   /* data storage type */
                     (unsigned long) 1,      /* num_of_images */
                     (unsigned long) 1,      /* num_data_bands */
                     "created by vconic",    /* comment */
                     (unsigned long) 0,      /* map_row_size */
                     (unsigned long) 0,      /* map_col_size */
                     (unsigned long)
                     VFF_MS_NONE,            /* map_scheme */
                     (unsigned long)
                     VFF_MAPTYP_NONE,        /* map_storage_type */
                     (unsigned long)
                     VFF_LOC_IMPLICIT,       /* location_type */
                     (unsigned long) 0);     /* location_dim */
  if (img == NULL)
    {
      fprintf(stderr,"lvconic: Unable to allocate new image!\n");
      *image = NULL;
      return(0);
    }
  img->pixsizx = s;
  img->pixsizy = s;

  *image = img;

  switch(type)
    {
      case VFF_TYP_1_BYTE:
        c = (unsigned char *)(img->imagedata);
        for (i=0; i<rows; i++)
          {
            for (j=0; j<cols; j++)
              {
                xhat = (j-cx)*s-x;
                yhat = (i-cy)*s-y;
                z = xhat*xhat+yhat*yhat;
                if (cutoff == 0.0)
                  c[PIXEL(j,i,rows,cols)] = 
                    eval_conic(branch,radius,conic,piston,z);
                else {
                  if (z <= cutoff*cutoff)
                    c[PIXEL(j,i,rows,cols)] = 
                      eval_conic(branch,radius,conic,piston,z);
                  else
                    c[PIXEL(j,i,rows,cols)]= 0.0;
                }
              }
          }
        break;
      case VFF_TYP_FLOAT:
        f = (float *)(img->imagedata);
        for (i=0; i<rows; i++)
          {
            for (j=0; j<cols; j++)
              {
                xhat = (j-cx)*s-x;
                yhat = (i-cy)*s-y;
                z = xhat*xhat+yhat*yhat;
                if (cutoff == 0.0)
                  f[PIXEL(j,i,rows,cols)] = 
                    eval_conic(branch,radius,conic,piston,z);
                else {
                  if (z <= cutoff*cutoff)
                    f[PIXEL(j,i,rows,cols)] = 
                      eval_conic(branch,radius,conic,piston,z);
                  else
                    f[PIXEL(j,i,rows,cols)]= 0.0;
                }
              }
          }
        break;
      case VFF_TYP_DOUBLE:
        d = (double *)(img->imagedata);
        for (i=0; i<rows; i++)
          {
            for (j=0; j<cols; j++)
              {
                xhat = (j-cx)*s-x;
                yhat = (i-cy)*s-y;
                z = xhat*xhat+yhat*yhat;
                if (cutoff == 0.0)
                  d[PIXEL(j,i,rows,cols)] = 
                    eval_conic(branch,radius,conic,piston,z);
                else {
                  if (z <= cutoff*cutoff)
                    d[PIXEL(j,i,rows,cols)] = 
                      eval_conic(branch,radius,conic,piston,z);
                  else
                    d[PIXEL(j,i,rows,cols)]= 0.0;
                }
              }
          }
        break;
      default:
        fprintf(stderr,"lvconic: Unsupported data storage type requested\n");
        fprintf(stderr,"         Supported types: BYTE, FLOAT, and DOUBLE\n");
        return(0);
    }

  return(1);
}

double
eval_conic(branch,radius,conic,piston,y2)
int branch;
float conic,radius,piston;
double y2;
  {
    double z;

    if (conic == 0.0) z = y2/(2.0*radius) + piston;
    else
      {
        if (!branch) z =  piston + 
                          (radius - sqrt(radius*radius-conic*y2))/conic;
        else z = piston + (radius + sqrt(radius*radius-conic*y2))/conic;
      }
    return(z);
  }
/* -library_code_end */
