/*  GUBI - Gtk+ User Interface Builder
 *  Copyright (C) 1997	Tim Janik	<timj@psynet.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
#include	"rcs.h"
RCS_ID("$Id: c_source.c,v 1.9 1997/08/18 03:30:09 timj Exp $")


#define		__c_source_c__

#include	"builder.h"
#include	"gsi.h"
#include	"structures.h"
#include	"widdata.h"
#include	"misc.h"
#include	"defines.h"
#include	<errno.h>


/* --- defines --- */
#define	STATIC_STRUCTS	1	/* 0 gives us c++ compatible code */



/* --- global variables --- */
const	builder_templ_S	C_builder_Files[]={
/*
{ base		s-postf	tpl-postfix	dynamic	trunc	source_type		}
*/
{ "files",	"doc",	".tpl",		FALSE,	FALSE,	GB_SOURCE_MAIN_DESC	},
{ "gbuild",	"h",	"h.tpl",	FALSE,	FALSE,	GB_SOURCE_MAIN		},
{ "gbuild", 	"c",	"c.tpl",	FALSE,	FALSE,	GB_SOURCE_MAIN		},
{ "widgets",	"h",	"h.tpl",	TRUE,	TRUE,	GB_SOURCE_MAIN		},
{ "widgets", 	"c",	"c.tpl",	TRUE,	TRUE,	GB_SOURCE_MAIN		},
{ "ufunc", 	"h",	"h.tpl",	TRUE,	FALSE,	GB_SOURCE_MAIN		},
{ "ufunc",	"c",	"c.tpl",	TRUE,	FALSE,	GB_SOURCE_MAIN		},
{ "main", 	"c",	".tpl",		FALSE,	FALSE,	GB_SOURCE_MAIN		},
{ "rcs",	"h",	".tpl",		FALSE,	FALSE,	GB_SOURCE_MAIN		},
{ "Makefile",	NULL,	".tpl",		FALSE,	FALSE,	GB_SOURCE_MAIN		},
};
const	guint		C_builder_Files_count=sizeof(C_builder_Files)/sizeof(builder_templ_S);



/* --- variable --- */
static	gboolean	Skip_Handler=TRUE;


/* --- static prototypes --- */
static	gboolean	ExpandMacro	(const	gchar	*macro);


/* write @c_widget_data_protos@ or @c_widget_data_decls@
 * as described in templates/macros.doc for a whole widget tree.
*/
static	void	Write_widget_tree	(gboolean	protos
					 /* write prototypes?
					  * contents otherwise.
					 */,
					 gboolean	internal
					 /* write internal prototypes?
					 */);


/* write @c_widget_data_protos@ or @c_widget_data_decls@
 * as described in templates/macros.doc.
*/
static	void	Write_widget_data_R	(gb_wdat_base_S	*WidDat,
					 gboolean	protos,
					 gboolean	internal);


/* write @c_window_data_pointers@
 * as described in templates/macros.doc.
*/
static	void	Write_window_data_pointers	(void);


/* write the handler array of a gb_wdat_base_S
*/
static	void	Write_handler_stack	(gb_wdat_base_S		*WidDat);


/* write the linkage information of a gb_wdat_base_S
*/
static	void	Write_linkage		(gb_wdat_base_S		*WidDat);


/* write the fields of gb_wdat_base_S only
*/
static	void	Write_widget_fields	(gb_wdat_base_S		*WidDat,
					 gboolean		protos,
					 gboolean		internal);


/* write the fields of a gb_wdat_*_S excluding
 * the ones from gb_wdat_base_S
*/
static	void	Write_generic_field	(const gsi_field_info_S	*field_info,
					 gb_any_S		*Struct);



/* --- functions --- */
gboolean
ExpandMacro	(const	char	*macro)
{
	ChBuf[0]=0;
	
	if (strcmp(macro,				"c_window_data_pointers")==0)
		Write_window_data_pointers();
	else if (strcmp(macro,				"c_widget_data_pointer_protos")==0)
		Write_widget_tree(TRUE , FALSE);
	else if (strcmp(macro,				"c_widget_data_iprotos")==0)
		Write_widget_tree(TRUE , TRUE);
	else if (strcmp(macro,				"c_widget_data_decls")==0)
		Write_widget_tree(FALSE, FALSE);
	else
		return FALSE;

	FLUSH_TARGET();
	
	return TRUE;
}


gint
Write_c_source	(const	gchar			*target_dir,
		 const	gboolean		dynamic_only,
		 const	gboolean		skip_handler,
		 const	gboolean		trunc_override,
		 const	guint			source_mask)
{
	register const builder_templ_S	*source_set;
	register guint			i;
	register gint			error;
	
	g_assert(target_dir);
	
	source_set=C_builder_Files;
	
	error=FALSE;
	
	Skip_Handler=skip_handler;
	
	for (i=0; i<C_builder_Files_count; i++) {
		register gchar		*err_name;
		
		g_assert(source_set[i].base);
		g_assert(source_set[i].template_postfix);
		g_assert(source_set[i].dynamic || !source_set[i].trunc);
		
		
		if ( source_set[i].source_type&source_mask &&
		     (source_set[i].dynamic || !dynamic_only) ) {
			snprintf(ChBuf, ChBufL, "%s%s", source_set[i].base, source_set[i].template_postfix);
			err_name=NULL;
		
			err_name=builder_init(DIR_TEMPLATE,
						ChBuf,
						target_dir,
						source_set[i].base,
						source_set[i].postfix,
						dynamic_only,
						source_set[i].dynamic,
						source_set[i].trunc | trunc_override);
			if (err_name) {
				error=TRUE;
				g_warning("%s\n%s:\n%s:\n%s",
					FUNCNAME,
					err_name,
					g_strerror(EIO),
					g_strerror(errno));
				g_free(err_name);
			} else
				builder_expand_file(ExpandMacro);
			builder_reset();
		}
	}
	
	return error ? EIO : FALSE;
}


void
Write_window_data_pointers	(void)
{
	register GList	*list;
	register guint	i;
	
	list=UD_tree_list;
	
	i=0;
	while (list) {
		register tree_S			*tree;
		register gb_wdat_window_S	*WinDat;
		
		g_assert((tree=list->data));
		g_assert(tree->widget_data_list);
		WinDat=tree->widget_data_list->data;
		g_assert(GB_IS_WIDDAT_WINDOW(WinDat));
		
		snprintf(ChBuf, ChBufL, "\t%s&%s%s%s%s,\t/* \"%s\" */\n",
			WinDat->type==GB_WIDGET_WINDOW ? "" : "GB_wCAST(window,\n\t  ",
			WIDGET_DATA_NAME_PREFIX,
			widget_data_symbol_name_get(GB_wCAST(base, WinDat)),
			WIDGET_DATA_NAME_POSTFIX,
			WinDat->type==GB_WIDGET_WINDOW ? "" : ")",
			WinDat->title);
		FLUSH_TARGET();
		
		i++;
		list=list->next;
	}
	
	if (i==0) {
		snprintf(ChBuf, ChBufL, "\tNULL, /* ANSI C forbids empty initializer braces */\n");
		FLUSH_TARGET();
	}
	
	ChBuf[0]=0;
}


void
Write_widget_tree	(gboolean	protos,
			 gboolean	internal)
{
	register GList		*tree_list;
	
	
	/* for each tree write it's widgets
	*/
	tree_list=UD_tree_list;
	while (tree_list) {
		register gb_wdat_base_S	*WinDat;
		register tree_S		*tree;
		
		g_assert((tree=tree_list->data));
		g_assert(tree->widget_data_list);
		WinDat=tree->widget_data_list->data;
		g_assert(GB_IS_WIDDAT_WINDOW(WinDat));
		
		Write_widget_data_R(WinDat, protos, internal);
		
		tree_list=tree_list->next;
	}
}


void
Write_widget_data_R	(gb_wdat_base_S	*WidDat,
			 gboolean	protos,
			 gboolean	internal)
{
	register GList	*list;
	
	g_assert(GB_IS_WIDDAT(WidDat));
	
	
	/* write prototype or structure
	*/
	Write_widget_fields(WidDat, protos, internal);
	
	
	/* write children
	*/
	list=GUBI_DATA(WidDat)->children;
	while (list) {
		register gb_wdat_base_S	*ChildDat;
		
		ChildDat=list->data;
		g_assert(GB_IS_WIDDAT(ChildDat));
		
		Write_widget_data_R(ChildDat, protos, internal);
		
		list=list->next;
	}
}


void
Write_widget_fields	(gb_wdat_base_S	*WidDat,
			 gboolean	protos,
			 gboolean	internal)
{
	register GList				*list, *free_list;
	
	g_assert(GB_IS_WIDDAT(WidDat));
	
	
	/* if we should write the gb_wdat_*_S contents
	 * let's first write auxillary structures
	*/
	if (!protos) {
		Write_handler_stack(WidDat);
		Write_linkage(WidDat);
	}
	
	
	/* write prototype part of a c-structure
	 * or it's begining if the fields follow
	*/
	if (protos) {
		if (internal)
			snprintf(ChBuf, ChBufL, "%sgb_wdat_%s_S\t%s%s%s;\n",
				STATIC_STRUCTS ? "static\t" : "extern\t",
				GUBI_DATA(WidDat)->struct_info->struct_name,
				WIDGET_DATA_NAME_PREFIX,
				widget_data_symbol_name_get(WidDat),
				WIDGET_DATA_NAME_POSTFIX);
		else
			snprintf(ChBuf, ChBufL, "extern\tgb_wdat_%s_S\t*const\t%s;\n",
				GUBI_DATA(WidDat)->struct_info->struct_name,
				widget_data_symbol_name_get(WidDat));
	} else
		snprintf(ChBuf, ChBufL, "%sgb_wdat_%s_S\t%s%s%s = {\n",
			STATIC_STRUCTS ? "static\t" : "",
			GUBI_DATA(WidDat)->struct_info->struct_name,
			WIDGET_DATA_NAME_PREFIX,
			widget_data_symbol_name_get(WidDat),
			WIDGET_DATA_NAME_POSTFIX);
	FLUSH_TARGET();
	ChBuf[0]=0;

	if (protos)
		return;
	
	
	/* non-generic object fields
	*/
	snprintf(ChBuf, ChBufL, "\tGB_WIDGET_%s\n\t\t\t/* type */,\n",
		to_UPCASE(GUBI_DATA(WidDat)->struct_info->struct_name));
	FLUSH_TARGET();
	
	snprintf(ChBuf, ChBufL, "\tNULL\t\t/* widget */,\n");
	FLUSH_TARGET();

	snprintf(ChBuf, ChBufL, "\tFALSE\t\t/* gfree_on_destroy */,\n");
	FLUSH_TARGET();
	snprintf(ChBuf, ChBufL, "\tNULL\t\t/* clone */,\n");
	FLUSH_TARGET();
	
	if (WidDat->next)
		snprintf(ChBuf, ChBufL, "\tGB_CAST(wdat_base, &%s%s%s)\n\t\t\t/* next */,\n",
			WIDGET_DATA_NAME_PREFIX,
			widget_data_symbol_name_get(WidDat->next),
			WIDGET_DATA_NAME_POSTFIX);
	else
		snprintf(ChBuf, ChBufL, "\tNULL\t\t/* next */,\n");
	FLUSH_TARGET();
	
	snprintf(ChBuf, ChBufL, "\tNULL\t\t/* keyed_data_stack */,\n");
	FLUSH_TARGET();
	
	if (WidDat->signal_handler_stack && !Skip_Handler)
		snprintf(ChBuf, ChBufL, "\t%s%s%s\n\t\t\t/* signal_handler_stack */,\n",
			HANDLER_NAME_PREFIX,
			widget_data_symbol_name_get(WidDat),
			HANDLER_NAME_POSTFIX);
	else
		snprintf(ChBuf, ChBufL, "\tNULL\t\t/* signal_handler_stack */,\n");
	FLUSH_TARGET();
	
	
	/* generic object fields
	*/
	free_list=list=gsi_struct_field_list(gsi_struct_info(GB_OBJECT));
	while (list) {
		register const gsi_field_info_S	*field_info;
		
		field_info=list->data;
		Write_generic_field(field_info, GB_CAST(any, WidDat));
		
		list=list->next;
	}
	g_list_free(free_list);
	
	
	/* non-generic widget fields:
	 *	*parent,
	 *	*linkage,
	*/
	if (WidDat->parent)
		snprintf(ChBuf, ChBufL, "\tGB_CAST(wdat_base, &%s%s%s)\n\t\t\t/* parent */,\n",
			WIDGET_DATA_NAME_PREFIX,
			widget_data_symbol_name_get(WidDat->parent),
			WIDGET_DATA_NAME_POSTFIX);
	else
		snprintf(ChBuf, ChBufL, "\tNULL\t\t/* parent */,\n");
	FLUSH_TARGET();
	if (!widget_data_linkage_check(WidDat))
		snprintf(ChBuf, ChBufL, "\tGB_CAST(any, &%s%s%s)\n\t\t\t/* linkage */,\n",
			LINKAGE_NAME_PREFIX,
			widget_data_symbol_name_get(WidDat),
			LINKAGE_NAME_POSTFIX);
	else
		snprintf(ChBuf, ChBufL, "\tNULL\t\t/* linkage */,\n");
	FLUSH_TARGET();
	
	
	/* rest of generic fields
	*/
	free_list=list=gsi_struct_field_list_till(GUBI_DATA(WidDat)->struct_info, GB_OBJECT);
	while (list) {
		register const gsi_field_info_S	*field_info;
		
		field_info=list->data;
		Write_generic_field(field_info, GB_CAST(any, WidDat));
		
		list=list->next;
	}
	g_list_free(free_list);
	
	
	/* trailer
	*/
	snprintf(ChBuf, ChBufL, "};\n");
	FLUSH_TARGET();
	snprintf(ChBuf, ChBufL, "gb_wdat_%s_S\t*const %s\t= ",
		GUBI_DATA(WidDat)->struct_info->struct_name,
		widget_data_symbol_name_get(WidDat));
	FLUSH_TARGET();
	snprintf(ChBuf, ChBufL, "&%s%s%s;\n",
			WIDGET_DATA_NAME_PREFIX,
			widget_data_symbol_name_get(WidDat),
			WIDGET_DATA_NAME_POSTFIX);
	FLUSH_TARGET();

	ChBuf[0]=0;
}


void
Write_generic_field	(const gsi_field_info_S	*field_info,
			 gb_any_S		*Struct)
{
	g_assert(field_info);
	g_assert(GB_IS_STRUCT(Struct));
	
	switch (field_info->field_type) {
	register const gchar		*string;
	register gb_any_S		*link_p;
	
	case	GSI_FIELD_STRING:
		string=gsi_field_get_value_as_string(Struct, field_info);
		if (string[0]==0)
			string="NULL";
		if (strlen(string)>15)
			snprintf(ChBuf, ChBufL, "\t%s\n\t\t\t/* %s */,\n",
				string,
				field_info->field_name);
		else
			snprintf(ChBuf, ChBufL, "\t%s%s\t/* %s */,\n",
				string,
				(strlen(string)<8) ? "\t" : "",
				field_info->field_name);
		break;
	
	case	GSI_FIELD_STRUCT_LINK:
		link_p=gsi_field_get_value(Struct, field_info).v_struct_link;
		if (link_p)
			string=widget_data_symbol_name_get(GB_wCAST(base, link_p));
		else
			string=NULL;
		if (string) {
			register guint	len;
			
			len= strlen(string) +
			     strlen(WIDGET_DATA_NAME_PREFIX) +
			     strlen(WIDGET_DATA_NAME_POSTFIX);
			
			if (len>14)
				snprintf(ChBuf, ChBufL, "\t&%s%s%s\n\t\t\t/* %s */,\n",
					WIDGET_DATA_NAME_PREFIX,
					string,
					WIDGET_DATA_NAME_POSTFIX,
					field_info->field_name);
			else
				snprintf(ChBuf, ChBufL, "\t&%s%s%s%s\t/* %s */,\n",
					WIDGET_DATA_NAME_PREFIX,
					string,
					WIDGET_DATA_NAME_POSTFIX,
					len<7 ? "\t" : "",
					field_info->field_name);
		} else
			snprintf(ChBuf, ChBufL, "\tNULL\t\t/* %s */,\n",
				field_info->field_name);
		break;
	
	default:
		string=gsi_field_get_value_as_string(Struct, field_info);
		snprintf(ChBuf, ChBufL, "\t%s%s\t/* %s */,\n",
			string,
			strlen(string)>15 ? "\n\t\t" : strlen(string)<8 ? "\t" : "",
			field_info->field_name);
		break;
	}
	
	FLUSH_TARGET();
}


void
Write_handler_stack	(gb_wdat_base_S	*WidDat)
{
	register gb_signal_handler_S	*h_next;
	
	g_assert(GB_IS_WIDDAT(WidDat));
	
	if (Skip_Handler)
		return;
	
	
	/* there is nothing to write,
	 * if the list is empty
	*/
	if (!(h_next=WidDat->signal_handler_stack))
		return;
	
	
	/* write prototype part of an array
	 * or it's begining, if the elements follow
	*/
	snprintf(ChBuf, ChBufL, "gb_%s_S\t%s%s%s[] = {\n",
		gb_signal_handler_struct_name,
		HANDLER_NAME_PREFIX,
		widget_data_symbol_name_get(WidDat),
		HANDLER_NAME_POSTFIX);
	FLUSH_TARGET();
	ChBuf[0]=0;

	
	/* write the handler values
	*/
	while (h_next->signal_name) {
		snprintf(ChBuf, ChBufL, "\t{\t%u\t/* %s */,\n",
			h_next->connect_options,
			gb_signal_handler_field_info[GB_SIGH_CONNECT_OPTIONS_INDX].field_name);
		FLUSH_TARGET();
		snprintf(ChBuf, ChBufL, "\t\t0\t/* handler_id */,\n");
		FLUSH_TARGET();
		snprintf(ChBuf, ChBufL, "\t\t\"%s\"\t/* %s */,\n",
			h_next->signal_name,
			gb_signal_handler_field_info[GB_SIGH_SIGNAL_NAME_INDX].field_name);
		FLUSH_TARGET();
		snprintf(ChBuf, ChBufL, "\t\tGTK_SIGNAL_FUNC(%s)\n\t\t\t/* %s */,\n",
			(char*)h_next->handler_func,
			gb_signal_handler_field_info[GB_SIGH_FUNC_NAME_INDX].field_name);
		FLUSH_TARGET();
		snprintf(ChBuf, ChBufL, "\t\t{ %s }\t/* %s */,\n",
			(char*)h_next->data.func_data,
			gb_signal_handler_field_info[GB_SIGH_DATA_NAME_INDX].field_name);
		FLUSH_TARGET();
		snprintf(ChBuf, ChBufL, "\t},\n");
		FLUSH_TARGET();
		h_next++;
	}
	snprintf(ChBuf, ChBufL, "\t{\t0, 0, NULL, NULL, { NULL }\t}\n");
	FLUSH_TARGET();
	snprintf(ChBuf, ChBufL, "};\n");
	FLUSH_TARGET();

	ChBuf[0]=0;
}


void
Write_linkage		(gb_wdat_base_S		*WidDat)
{
	register const gsi_struct_info_S	*struct_info;
	register guint				i;
	
	g_assert(GB_IS_WIDDAT(WidDat));
	
	if (widget_data_linkage_check(WidDat))
		return;
	
	struct_info=gsi_struct_info(WidDat->linkage->type);
	
	/* write linkage structure begining
	*/
	snprintf(ChBuf, ChBufL, "gb_%s_S\t%s%s%s = {\n",
		struct_info->struct_name,
		LINKAGE_NAME_PREFIX,
		widget_data_symbol_name_get(WidDat),
		LINKAGE_NAME_POSTFIX);
	FLUSH_TARGET();
	
	snprintf(ChBuf, ChBufL, "\tGB_%s\n\t\t\t/* type */,\n",
		to_UPCASE(struct_info->struct_name));
	FLUSH_TARGET();
	
	/* write the linkage values
	*/
	ChBuf[0]=0;
	for (i=0; i<struct_info->n_fields; i++)
		Write_generic_field(&struct_info->fields[i], GB_CAST(any, WidDat->linkage));
	
	snprintf(ChBuf, ChBufL, "};\n");
	FLUSH_TARGET();

	ChBuf[0]=0;
}
