/*  GUBI - Gtk+ User Interface Builder
 *  Copyright (C) 1997	Tim Janik	<timj@psynet.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/* builder.h: functions used for source building
 * RCS_ID("$Id: builder.h,v 1.4 1997/08/18 03:24:10 timj Exp $")
*/


#ifndef		__builder_h__
#define		__builder_h__

#ifdef		__cplusplus
  extern	"C"	{
#endif

#include	"wdtree.h"
#include	"config.h"



/* --- defines --- */


/* short hand macro for lazy programmers ;)
*/
#define	FLUSH_TARGET()		builder_put_string(ChBuf)



/* --- typedefs --- */


/* the gb_builder_file_E is used to qualify the different
 * template/target files.
*/
typedef	enum
{
	GB_SOURCE_MAIN		= 1 <<  0,
	GB_SOURCE_MAIN_DESC	= 1 <<  1,
	GB_SOURCE_PLUGIN	= 1 <<  2,
	GB_SOURCE_PLUGIN_DESC	= 1 <<  3
} gb_builder_flag_E;

typedef	struct	builder_templ_S		builder_templ_S;


/* callback for language specific macro processing.
 * this must be analog to builder_expand_macro().
*/
typedef	gboolean	(*builder_expand_macro_func) (const gchar *macro);



/* --- structures --- */


/* structure specifying the template and target files.
*/
struct	builder_templ_S {
	const	gchar		*base
				/* *valid* basename for templates and
				 * target files
				*/;
	const	gchar		*postfix
				/* target files postfix, NULL indicates no
				 * postfix and no '.'
				*/;
	const	gchar		*template_postfix
				/* *valid* postfix for template file
				*/;
	gboolean		dynamic
				/* does this file reflect changes in the
				 * widget tree?
				*/;
	gboolean		trunc
				/* are we allowed to overwrite an existing
				 * dynamic file?
				*/;
	gb_builder_flag_E	source_type;
};



/* --- global variables --- */
extern	const	gchar	*Template_Name;
extern	guint		Template_Line;
extern	const	gchar	*Target_Base;
extern	const	gchar	*Target_Postfix;
extern	gboolean	Target_Dynamic;
extern	FILE		*Target_Stream;
extern	FILE		*Template_Stream;
extern	struct	tm	*Target_tm;
extern	gchar		ChBuf[];
extern	guint		ChBufL;


/* template and destination file names for C-source.
*/
extern	const	builder_templ_S		C_builder_Files[];



/* --- prototypes --- */



/* the Build_c_source() function builds C-source out of the
 * given widget tree. this function reads from template files
 * and expands any macros it finds. the result is writen into
 * the destination file.
 * the destination file name is made up of dest_dir + "/" +
 * dest_base + "." + dest_postfix except when dest_postfix
 * is NULL, then it is made up of dest_dir + "/" +
 * dest_base only.
 * returns:
 * this returns possible errno on error.
*/
int	Write_c_source	(const	gchar			*target_dir
			 /* *valid* string pointer containing the
			  * directory part of the destination files.
			 */,
			 const	gboolean		dynamic_only
			 /* this specifies wether to write dynamic
			  * files only or dynamic and static files.
			 */,
			 const	gboolean		skip_handler
			 /* this specifies wether to write the widget
			  * specific handler connections.
			 */,
			 const	gboolean		trunc_override
			 /* this specifies wether the truncate flag of
			  * a file is always assumed to be true.
			 */,
			 const	guint			source_mask
			 /* specify the file set to be written.
			 */);


/* the SigH_Builder_Build_clicked() function is invoked
 * to actualy build the source.
*/
void	SigH_Builder_Build_clicked	(GtkWidget	*widget,
					 gpointer	func_data);


/* the Builder_create() function pops up a dialog window
 * with a few building options and a button to perform
 * the widget building.
 * if widget is not NULL, it's sensitivity is reset while
 * the buiilder dialog exists.
*/
void	Builder_create			(GtkObject	*parent_object);



/* --- helper functions --- */


/* the builder_expand_file() is the main char copy loop
 * for source production.
 * call builder_init() before and builder_reset() afterwards
 * and give your own builder_expand_macro() like expander
 * function as argument.
*/
void	builder_expand_file	(builder_expand_macro_func	MacroFunc);


/* the builder_reset() function resets the global variables
 * and closes open files if neccessary.
*/
void	builder_reset	(void);


/* the builder_init() function opens the target and template
 * files and sets up the global variables.
 * returns:
 * on error the filename (g_malloc()ed) is returned and errno
 * is set appropriatly. NULL is returned otherwise.
*/
gchar	*builder_init	(const	gchar		*template_dir
			 /* *valid* string pointer containing the
			  * directory part of the template file.
			 */,
			 const gchar *template_name
			 /* *valid* string pointer containing the
			  * filename of the template file.
			 */,
			 const gchar *target_dir
			 /* *valid* string pointer containing the
			  * directory part of the destination file.
			 */,
			 const gchar *target_base
			 /* *valid* string pointer containing the
			  * filename of the template file.
			 */,
			 const gchar *target_postfix
			 /* *valid* string pointer containing the
			  * filename of the template file.
			 */,
			 gboolean	update,
			 gboolean	dynamic,
			 gboolean	truncate);


/* the template_get_char() function gets a char from
 * template_stream, and invokes g_error() if check_eof is
 * TRUE and EOF is reached.
 * this also increments template_line.
*/
gint	template_get_char	(gboolean	check_eof);


/* the builder_put_string() function writes string
 * into target_stream.
*/
void	builder_put_string	(const	gchar	*string);


/* the builder_expand_macro() function writes the
 * substitution of macro to target_stream, if it
 * is known.
 * returns:
 * this returns FALSE on unknown macros. in this case
 * a warning will be printed.
 *
 * MACROS:
 * macros are enclosed in '@' signs with one exception.
 * an '@' sign immediatly followed by a '#' will eat all
 * chars up to (and including) the next newline.
 * this gives one the possibility to include comments
 * into the template files that are not shown in the output.
 * macros can consist of up to MAX_MACRO_LEN characters.
 * for a description on the various supported macros
 * see templates/macros.doc.
 * a special case is an empty macro "@@": it is simply
 * replaced by a single '@' sign.
*/
gboolean	builder_expand_macro	(const	gchar	*macro);



/*
*/
void		builder_flags_get	(gboolean	*dynamic_only,
					 gboolean	*skip_handler,
					 gboolean	*trunc_override);


/*
*/
void		builder_flags_set	(gboolean	dynamic_only,
					 gboolean	skip_handler,
					 gboolean	trunc_override);






#ifdef		__cplusplus
  }
#endif
#endif		/*__builder_h__*/
