/*  GUBI - Gtk+ User Interface Builder
 *  Copyright (C) 1997  Tim Janik
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
#include	"RCS.h"
RCS_ID("$Id: widdata.c,v 1.6 1997/05/09 16:17:57 tim Exp $")


#define		__widdata_c__

#include	"widdata.h"
#include	"structures.h"
#include	"misc.h"
#include	"editor.h"
#include	"defines.h"


/* --- external global variables --- */
GList		*Gubi_auxillary_list=NULL;



/* --- variables --- */
static	GHashTable	*symbol_name_hash_table=NULL;



/* --- prototypes --- */
static	void	symbol_name_hash_table_init	(void);
static	void	widget_data_delete_foreach	(gpointer	data,
						 gpointer	user_data);
static	void	SigH_widget_data_item_destroy	(GtkWidget	*item,
						 gpointer	func_data);



/* --- functions --- */
void
symbol_name_hash_table_init	(void)
{
	if (!symbol_name_hash_table) {
		symbol_name_hash_table=g_hash_table_new((GHashFunc)g_string_hash,
							(GCompareFunc)g_string_equal);
	}
	
	g_assert(symbol_name_hash_table);
}


gb_wdat_base_S*
widget_data_new	(gb_struct_type_E	struct_type,
		 const gchar		*symbol_name,
		 gboolean		auxillary)
{
	register gb_wdat_base_S	*WidDat;
	
	g_assert(GB_TYPE_IS_WIDGET(struct_type));
	
	
	/* allocate a gb_wdat_*_S
	*/
	WidDat=(gb_wdat_base_S*)g_malloc0(gb_struct_size(struct_type));
	
	
	/* initialize private fields of base widget data
	*/
	WidDat->type=struct_type;
	WidDat->parent=NULL;
	WidDat->gfree_on_destroy=FALSE;
	 WidDat->clone=NULL;
	 WidDat->user_data=NULL;
	WidDat->next=NULL;
	WidDat->widget=NULL;
	WidDat->handler_stack=NULL;
	WidDat->linkage=NULL;
	
	
	/* allocate and pre-initialize gubi_data_S
	*/
	WidDat->GUBI_DATA_FIELD=(gpointer)g_new(gubi_data_S, 1);
	GUBI_DATA(WidDat)->type=GB_STRUCT_NONE;
	GUBI_DATA(WidDat)->widget_data=WidDat;
	GUBI_DATA(WidDat)->symbol_name=NULL;
	GUBI_DATA(WidDat)->category=WID_CAT_NORMAL | (auxillary ? WID_CAT_AUXILLARY : 0);
	GUBI_DATA(WidDat)->children=NULL;
	GUBI_DATA(WidDat)->max_child_count=structure_info(WidDat->type)->widget_child_count;
	GUBI_DATA(WidDat)->child_count=0;
	GUBI_DATA(WidDat)->tree=NULL;
	GUBI_DATA(WidDat)->editor=NULL;
	GUBI_DATA(WidDat)->browser=NULL;
	GUBI_DATA(WidDat)->item_refs=NULL;
	GUBI_DATA(WidDat)->link_refs=NULL;
	
	
	/* set symbol name,
	 * right now we need an unregistered symbol name!
	*/
	widget_data_symbol_name_set(WidDat, symbol_name);
	widget_data_symbol_name_unregister(WidDat);
	
	
	/* initialize widget specific fields
	*/
	structure_fields_initialize(GB_CAST(any, WidDat), structure_info(GB_WIDGET_BASE));
	if (GB_TYPE_IS_CONTAINER_WIDGET(WidDat->type))
		structure_fields_initialize(GB_CAST(any, WidDat), structure_info(GB_WIDGET_CONTAINER));
	structure_fields_initialize(GB_CAST(any, WidDat), structure_info(WidDat->type));
	
	
	/* reset the child count
	*/
	UPDATE_CHILD_COUNT(WidDat);
	
	return WidDat;
}


void
widget_data_delete_foreach	(gpointer	data,
				 gpointer	user_data)
{
	register gb_wdat_base_S	*WidDat;
	
	WidDat=data;
	
	g_assert(GB_IS_WIDGET(WidDat));
	
	widget_data_delete_R(WidDat);
}


void
widget_data_delete_R	(gb_wdat_base_S	*WidDat)
{
	g_assert(GB_IS_WIDGET(WidDat));
	g_assert(GUBI_DATA(WidDat));
	g_assert(!WidDat->widget);
	g_assert(!GUBI_DATA(WidDat)->editor);
	g_assert(!GUBI_DATA(WidDat)->browser);
	g_assert(GUBI_DATA(WidDat)->tree==NULL);
	g_assert(!widget_data_lookup(widget_data_symbol_name_get(WidDat)));
	if (GUBI_DATA(WidDat)->category & WID_CAT_AUXILLARY) {
		g_assert(WidDat->parent=NULL);
		g_assert(WidDat->next=NULL);
        }
        
	if (GB_TYPE_IS_WINDOW_WIDGET(WidDat->type))
		g_assert(g_list_length(GUBI_DATA(WidDat)->children)==0);
	
	
	/* delete all list items that might refer to us
	*/
	while (GUBI_DATA(WidDat)->item_refs)
		widget_data_item_destroy(WidDat, GUBI_DATA(WidDat)->item_refs->data);
	
	
	/* delete references from other widgets in widget data
	*/
	while (GUBI_DATA(WidDat)->link_refs) {
		register gb_wdat_base_S	*RefDat;
		
		RefDat=GUBI_DATA(WidDat)->link_refs->data;
		g_assert(GB_IS_WIDGET(RefDat));
		
		widget_data_link_remove(RefDat, WidDat);
		
		if (GUBI_DATA(RefDat)->editor)
			Editor_refresh(RefDat);
	}
	
	
	/* delete children and children list
	*/
	g_list_foreach(GUBI_DATA(WidDat)->children,
			widget_data_delete_foreach,
			NULL);
	g_list_free(GUBI_DATA(WidDat)->children);
	
	
	/* free handler_stack string fields
	*/
	if (WidDat->handler_stack) {
		register gb_handler_S	*next;
		
		next=WidDat->handler_stack;
		
		while (next->handler_func) {
			g_free((void*)next->handler_func);
			g_free(next->signal_name);
			g_free(next->data.func_data);
			next++;
		}
	}
	
	
	/* free gubi_data_S
	*/
	g_free(GUBI_DATA(WidDat)->symbol_name);
	g_free(GUBI_DATA(WidDat));
	
	
	/* free other widget field specific memory allocations
	*/
	structure_fields_free(GB_CAST(any, WidDat), structure_info(GB_WIDGET_BASE));
	if (GB_TYPE_IS_CONTAINER_WIDGET(WidDat->type))
		structure_fields_free(GB_CAST(any, WidDat), structure_info(GB_WIDGET_CONTAINER));
	structure_fields_free(GB_CAST(any, WidDat), structure_info(WidDat->type));
	
	
	/* free gb_wdat_base_S
	 * this also frees the linkage and handler_stack
	*/
	gb_sigh_structure_free(NULL, GB_CAST(any, WidDat));
}


const	gchar*
widget_data_symbol_name_get	(gb_wdat_base_S	*WidDat)
{
	g_assert(GB_IS_WIDGET(WidDat));
	
	return GUBI_DATA(WidDat)->symbol_name ? GUBI_DATA(WidDat)->symbol_name : "";
}


void
widget_data_symbol_name_set	(gb_wdat_base_S	*WidDat,
				 const gchar	*name)
{
	register gchar		*buffer;
	
	g_assert(GB_IS_WIDGET(WidDat));
	
	symbol_name_hash_table_init();
	
	buffer=g_new(gchar, DFL_BUFFER_SIZE);
	
	if (GUBI_DATA(WidDat)->symbol_name) {
		if (widget_data_lookup(GUBI_DATA(WidDat)->symbol_name))
			widget_data_symbol_name_unregister(WidDat);
		g_free(GUBI_DATA(WidDat)->symbol_name);
	}
	
	
	/* if the given name already exists or is invalid
	 * we make it generic
	*/
	if (!name || strlen(name)==0 || widget_data_lookup(name))
		name=NULL;
	
	
	/* if no name is given, we genrate one
	*/
	if (!name) {
		GUBI_DATA(WidDat)->category|=WID_CAT_GENERIC;
		sprintf(buffer, "%s_%%d",
			to_UpCase(structure_info(WidDat->type)->struct_name));
	} else {
		GUBI_DATA(WidDat)->category&=~WID_CAT_GENERIC;
		sprintf(buffer, "%s", name);
	}
	
	
	/* if name is generic, check for existence
	*/
	if (GUBI_DATA(WidDat)->category & WID_CAT_GENERIC) {
		register guint	i;
		
		GUBI_DATA(WidDat)->symbol_name=g_strdup(buffer);
		i=1;
		do {
			sprintf(buffer, GUBI_DATA(WidDat)->symbol_name, i);
			i++;
		} while (widget_data_lookup(buffer));
		g_free(GUBI_DATA(WidDat)->symbol_name);
	}
	GUBI_DATA(WidDat)->symbol_name=g_strdup(buffer);
	
	
	/* we don't allow generic names in auxillary pool
	*/
	if (GUBI_DATA(WidDat)->category & WID_CAT_AUXILLARY &&
	    GUBI_DATA(WidDat)->category & WID_CAT_GENERIC) {
		GUBI_DATA(WidDat)->category&=~WID_CAT_GENERIC;
		g_warning("%s\ngeneric name generation forced for auxillary widget: `%s'",
			FUNCNAME,
			GUBI_DATA(WidDat)->symbol_name);
	}
	
	widget_data_symbol_name_register(WidDat);
}


gb_wdat_base_S*
widget_data_lookup	(const gchar	*name)
{
	register gpointer	Ret;
	
	g_assert(name);
	
	Ret=g_hash_table_lookup(symbol_name_hash_table, (gpointer)name);
	
	return GB_wCAST(base, Ret);
}


void
widget_data_symbol_name_register	(gb_wdat_base_S	*WidDat)
{
	g_assert(GB_IS_WIDGET(WidDat));
	g_assert(!widget_data_lookup(widget_data_symbol_name_get(WidDat)));
	if (GUBI_DATA(WidDat)->category & WID_CAT_AUXILLARY)
		g_assert(!GUBI_DATA(WidDat)->category&WID_CAT_GENERIC);
	
	g_hash_table_insert(symbol_name_hash_table,
			    (gpointer)widget_data_symbol_name_get(WidDat),
			    WidDat);
}


void
widget_data_symbol_name_unregister	(gb_wdat_base_S	*WidDat)
{
	g_assert(GB_IS_WIDGET(WidDat));
	g_assert(widget_data_lookup(widget_data_symbol_name_get(WidDat)));
	
	g_hash_table_remove(symbol_name_hash_table,
			    (gpointer)widget_data_symbol_name_get(WidDat));
}


GtkWidget*
widget_data_item_new		(gb_wdat_base_S	*WidDat,
				 guint		indent_level,
				 GtkList	*List,
				 gint		position)
{
	register GtkWidget	*item;
	register gchar		*name;
	register const gchar	*symbol_name;
	register guint		i;
	
	g_assert(GB_IS_WIDGET(WidDat));
	
	symbol_name=widget_data_symbol_name_get(WidDat);
	name=g_new(gchar, strlen(symbol_name)+indent_level+1);
	for (i=0; i<indent_level; i++)
		name[i]=INDENT_CHAR;
	strcpy(&name[i], symbol_name);
	
	item=gtk_list_item_new_with_label(name);
	g_free(name);
	
	gtk_object_set_user_data(GTK_OBJECT(item), (gpointer)WidDat);
	
	GUBI_DATA(WidDat)->item_refs=g_list_append(GUBI_DATA(WidDat)->item_refs, item);
	
	gtk_signal_connect(GTK_OBJECT(item),
			   "destroy",
			   GTK_SIGNAL_FUNC(SigH_widget_data_item_destroy),
			   NULL);
	
	if (List) {
		register GList	*item_list;
	
		item_list=g_list_alloc();
		item_list->data=item;
		gtk_list_insert_items(List, item_list, position);
	}
	
	gtk_widget_show(item);
	
	return item;
}


void
SigH_widget_data_item_destroy	(GtkWidget	*item,
				 gpointer	func_data)
{
	register gb_wdat_base_S	*WidDat;
	
	WidDat=gtk_object_get_user_data(GTK_OBJECT(item));
	
	g_assert(GB_IS_WIDGET(WidDat));
	
	GUBI_DATA(WidDat)->item_refs=g_list_remove(GUBI_DATA(WidDat)->item_refs, item);
}


void
widget_data_item_destroy	(gb_wdat_base_S	*WidDat,
				 GtkWidget	*item)
{
	g_assert(GB_IS_WIDGET(WidDat));
	g_assert(g_list_find(GUBI_DATA(WidDat)->item_refs, item)); /* FIXME */
	
	if (item->parent) {
		register GList		*list;
	
		g_assert(GTK_IS_LIST(item->parent));
		
		list=g_list_append(NULL, item);
		
		gtk_list_remove_items(GTK_LIST(item->parent), list);
	}
	
	g_assert(GTK_IS_ITEM(item));
	gtk_widget_destroy(item);
}


guint
widget_data_parent_level	(gb_wdat_base_S	*WidDat)
{
	register guint	level;
	
	g_assert(GB_IS_WIDGET(WidDat));
	
	level=0;
	WidDat=WidDat->parent;
	while (WidDat) {
		level++;
		WidDat=WidDat->parent;
	}
	
	return level;
}


guint
widget_data_handler_add(gb_wdat_base_S *WidDat,
			const guint16 connect_options,
			const gchar *func_name,
			const gchar *signal_name,
			const gchar *data_name)
{
	register gb_handler_S	*last;
	register guint		count;
	
	g_assert(GB_IS_WIDGET(WidDat));
	g_assert(func_name);
	g_assert(signal_name);
	g_assert(data_name);
	
	count=0;
	if (!WidDat->handler_stack)
		WidDat->handler_stack=g_new(gb_handler_S, 2);
	else {
		register gb_handler_S	*old_list;
		
		old_list=WidDat->handler_stack;
		
		for (count=0; old_list[count].handler_func; count++);
		
		WidDat->handler_stack=g_new(gb_handler_S, count+1+1);
		memcpy(WidDat->handler_stack, old_list, count*sizeof(gb_handler_S));
		g_free(old_list);
	}

	last=&WidDat->handler_stack[count];
	
	last->connect_options=connect_options;
	last->handler_func=(GtkSignalFunc)g_strdup(func_name);
	last->signal_name=g_strdup(signal_name);
	last->data.func_data=g_strdup(data_name);
	
	last++;
	
	last->connect_options=0;
	last->handler_id=0;
	last->handler_func=NULL;
	last->signal_name=NULL;
	last->data.func_data=NULL;
	
	return count;
}


void
widget_data_linkage_new	(gb_wdat_base_S	*WidDat)
{
	g_assert(GB_IS_WIDGET(WidDat));
	g_assert(!WidDat->linkage);
	g_assert(!(GUBI_DATA(WidDat)->category & WID_CAT_AUXILLARY));
	
	if (!WidDat->parent)
		return;
	
	
	/* we only need the linkage information on
	 * special parent widgets
	*/
	switch (WidDat->parent->type) {
	
	case	GB_WIDGET_H_BOX:
	case	GB_WIDGET_V_BOX:
	case	GB_WIDGET_FILE_SELECTION:
		WidDat->linkage=(gb_any_S*)g_malloc0(gb_struct_size(GB_LINKAGE_BOX));
		WidDat->linkage->type=GB_LINKAGE_BOX;
		break;
	
	case	GB_WIDGET_TABLE:
		WidDat->linkage=(gb_any_S*)g_malloc0(gb_struct_size(GB_LINKAGE_TABLE));
		WidDat->linkage->type=GB_LINKAGE_TABLE;
		break;
	
	default:
		break;
	}
	
	if (WidDat->linkage)
		structure_fields_initialize(WidDat->linkage, structure_info(WidDat->linkage->type));
	widget_data_linkage_check(WidDat);
}


gboolean
widget_data_linkage_check	(gb_wdat_base_S	*WidDat)
{
	register const struct_info_S	*LnkInf;
	register gboolean		has_default;
	register guint			i;
	
	g_assert(GB_IS_WIDGET(WidDat));
	g_assert(!(GUBI_DATA(WidDat)->category & WID_CAT_AUXILLARY));
	
	
	/* do we have a linkage to check?
	*/
	if (!WidDat->linkage)
		return TRUE;
	
	
	/* we only need to check the linkage validity on
	 * special parent widgets
	*/
	switch (WidDat->parent->type) {
		register gb_linkage_table_S	*l_table;
		register gb_wdat_table_S	*Table;
		
	case	GB_WIDGET_H_BOX:
	case	GB_WIDGET_V_BOX:
	case	GB_WIDGET_FILE_SELECTION:
		g_assert(WidDat->linkage->type==GB_LINKAGE_BOX);
		break;
		
	case	GB_WIDGET_TABLE:
		g_assert(WidDat->linkage->type==GB_LINKAGE_TABLE);
		l_table=(gb_linkage_table_S*)WidDat->linkage;
		Table=GB_wCAST(table, WidDat->parent);
		
		g_assert(Table->columns>=1 && Table->rows>=1);
		l_table->left_attach=CLAMP(l_table->left_attach, 0, Table->columns-1);
		l_table->right_attach=CLAMP(l_table->right_attach, 1, Table->columns);
		l_table->top_attach=CLAMP(l_table->top_attach, 0, Table->rows-1);
		l_table->bottom_attach=CLAMP(l_table->bottom_attach, 1, Table->rows);
		l_table->left_attach=MIN(l_table->left_attach, l_table->right_attach-1);
		l_table->top_attach=MIN(l_table->top_attach, l_table->bottom_attach-1);
		break;
	
	default:
		g_assert_not_reached();
		break;
	}
	
	
	/* let's determine wether we have default values
	*/
	g_assert((LnkInf=structure_info(WidDat->linkage->type)));
	has_default=TRUE;
	for (i=0; i<LnkInf->field_count; i++)
		has_default&=structure_field_has_default(WidDat->linkage, field_info(LnkInf, i));
	
	return has_default;
}


gboolean
widget_data_children_linkage_check	(gb_wdat_base_S	*WidDat)
{
	register gboolean	something_changed=FALSE;
	
	g_assert(GB_IS_WIDGET(WidDat));
	
	
	/* if we are a table we check our childrens
	 * linkage
	*/
	if (WidDat->type==GB_WIDGET_TABLE) {
		register GList	*list;
		
		list=GUBI_DATA(WidDat)->children;
		if (list)
			something_changed=TRUE;
		while (list) {
			register gb_wdat_base_S	*ChildDat=list->data;
			
			g_assert(GB_IS_WIDGET(ChildDat));
			
			widget_data_linkage_check(ChildDat);
			
			list=list->next;
		}
	}
	
	return something_changed;
}
