/* The GIMP -- an image manipulation program
 * Copyright (C) 1995 Spencer Kimball and Peter Mattis
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "gimp.h"

typedef struct {
    long interval;
    long offset;
} ContourData;

/* Declare a local function */
static void int_callback(int, void *, void *);
static void ok_callback(int, void *, void *);
static void cancel_callback(int, void *, void *);
static void contour(Image, Image);

static char *prog_name;
static int dialog_ID;
static ContourData vals;

int main(int argc, char *argv[]) {
    Image input, output;
    int group_ID, interval_ID, offset_ID;
    char buf[16];
    void *data;

    prog_name = argv[0];
    if (gimp_init(argc, argv)) {
        input = gimp_get_input_image(0);
        if (input) {
            if (gimp_image_type(input) == GRAY_IMAGE) {
                data = gimp_get_params();
                if (data)
                    vals = *((ContourData *)data);
                else {
                    vals.interval = 10;
                    vals.offset = 0;
                }
                dialog_ID = gimp_new_dialog("Contours");
                group_ID = gimp_new_column_group(dialog_ID, DEFAULT,
                                                 NORMAL, "");
                gimp_new_label(dialog_ID, group_ID, "Interval:");
                sprintf(buf, "%ld", vals.interval);
                interval_ID = gimp_new_text(dialog_ID, group_ID, buf);
                gimp_add_callback(dialog_ID, interval_ID, int_callback,
                                  &vals.interval);
                gimp_new_label(dialog_ID, group_ID, "Offset:");
                sprintf(buf, "%ld", vals.offset);
                offset_ID = gimp_new_text(dialog_ID, group_ID, buf);
                gimp_add_callback(dialog_ID, offset_ID, int_callback,
                                  &vals.offset);
                /* OK button */
                gimp_add_callback(dialog_ID, gimp_ok_item_id(dialog_ID),
                                  ok_callback, 0);
                /* Cancel button */
                gimp_add_callback(dialog_ID, gimp_cancel_item_id(dialog_ID),
                                  cancel_callback, 0);
                /* Now show the dialog */
                if (gimp_show_dialog(dialog_ID)) {
                    output = gimp_new_image(0,
                                            gimp_image_width(input),
                                            gimp_image_height(input),
                                            gimp_image_type(input));
                    if (output) {
                        gimp_set_params(sizeof(ContourData), &vals);
                        contour(input, output);
                        gimp_display_image(output);
                        gimp_update_image(output);
                        gimp_free_image(output);
                    }
                }
            } else {
                gimp_message("contour: can only operate on grayscale images");
            }
        }
        gimp_free_image(input);
        gimp_quit();
    }
    return 0;
}

/* Callbacks */

static void int_callback(int item_ID, void *client_data, void *call_data) {
    *((long *)client_data) = atoi(call_data);
}

static void ok_callback (item_ID, client_data, call_data)
     int item_ID;
     void *client_data;
     void *call_data;
{
  gimp_close_dialog (dialog_ID, 1);
}

static void cancel_callback (item_ID, client_data, call_data)
     int item_ID;
     void *client_data;
     void *call_data;
{
  gimp_close_dialog (dialog_ID, 0);
}

static void contour(Image input, Image output) {
    long width, height;
    long channels, rowstride;
    unsigned char *src, *dest;
    long n, m, src_n, neighbor;

    width = gimp_image_width(input);
    height = gimp_image_height(input);
    channels = gimp_image_channels(input);
    rowstride = width * channels;
    src = gimp_image_data(input);
    dest = gimp_image_data(output);
    for (n = 0; n < width*height; n++) {
        src_n = ((src[n] + vals.offset) / vals.interval) * vals.interval;
        dest[n] = 0;
        if (n > width) {
            /* Top neighbors */
            for (m = -1; m <= 1; m++) {
                neighbor = ((src[n-width+m] + vals.offset) / vals.interval)
                    * vals.interval;
                if (( neighbor > src_n) && (neighbor > dest[n])) {
                    dest[n] = neighbor;
                }
            }
        }
        if (n < width*height - width - 1) {
            /* Bottom neighbors */
            for (m = -1; m <= 1; m++) {
                neighbor = ((src[n+width+m] + vals.offset) / vals.interval)
                    * vals.interval;
                if ((neighbor > src_n) && (neighbor > dest[n])) {
                    dest[n] = neighbor;
                }
            }
        }
        /* Side neighbors */
        if (n % width != 0) {
            neighbor = ((src[n-1] + vals.offset) / vals.interval)
                * vals.interval;
            if ((neighbor > src_n) && (neighbor > dest[n])) {
                dest[n] = neighbor;
            }
        }
        if ((n+1) % width != 0) {
            neighbor = ((src[n+1] + vals.offset) / vals.interval)
                * vals.interval;
            if ((neighbor > src_n) && (neighbor > dest[n])) {
                dest[n] = neighbor;
            }
        }
        if (dest[n] == 0) {
            dest[n] = 255;
        }
    }
}
