/*ScianDougFiles.c
  HDF File reading routine for scian
  Doug Lee
  June 27, 1991
  based on work done by
  Eric Pepke
  August 17, 1990
*/

#define MAX_RANK 1024

#include "Scian.h"
#include "ScianTypes.h"
#include "ScianArrays.h"
#include "ScianIcons.h"
#include "ScianFileSystem.h"
#include "ScianLists.h"
#include "ScianPictures.h"
#include "ScianErrors.h"
#include "ScianTimers.h"
#include "ScianWindows.h"
#include "ScianObjWindows.h"
#include "ScianVisWindows.h"
#include "ScianDatasets.h"
#include "ScianScripts.h"
#include "ScianIDs.h"
#include "ScianVisContours.h"
#include "ScianFiles.h"

#ifdef HDFDEF
ObjPtr ReadHDFDFile(reader, fileName)
ObjPtr reader;
char *fileName;
/*Reads an HDF scientific data set file.*/
{
    int nDataSets;		/*Number of data sets*/
    int whichSet;	/*Number of current data set*/
    char dimsLabel[256], dimsUnit[256], dimsFormat[256];
    ObjPtr outOfBoundsVar;
    int outOfBoundsAction;

#ifdef DEBUG
    printf("ReadHDFDFile: HDF filename = %s\n", fileName);
#endif

    nDataSets = DFSDnumber(fileName);

    if (nDataSets == -1)
    {
        FileFormatError("ReadHDFDFile", "Not an HDF file or file does not exist.");
        return NULLOBJ;
    }

    
#ifdef DEBUG
    printf("ReadHDFDFile: nDataSets = %d\n", nDataSets);
#endif

    if (nDataSets == 0)
    {
        FileFormatError("ReadHDFDFile", "No datasets to read.");
        return NULLOBJ;
    }

    if ((outOfBoundsVar = GetIntVar("ReadHDFDFile", reader, OUTOFBOUNDS)) == NULLOBJ)
        outOfBoundsAction = 0;
    else
        outOfBoundsAction = GetInt(outOfBoundsVar);

    for (whichSet = 0; whichSet < nDataSets; ++whichSet)
    {
        int rank;
        int k;
        int dimSizes[MAX_RANK];
        long arraySize;			/*Total size of the array*/
        float min = PLUSINF, max = MINUSINF;
        real intMissing = missingData;		/*Internal missing data*/
        ObjPtr curField;
	Bool minMaxSet = false;

        /*Start off with null field*/
        curField = 0;

        /*Get dimensions of data set*/

        if (-1 == DFSDgetdims(fileName, &rank, dimSizes, MAX_RANK))
        {
            FileFormatError("ReadHDFDFile", "Unable to get dimensions.");
            return NULLOBJ;
        }

#ifdef DEBUG
        {
            int i;

            printf("ReadHDFDFile: rank = %d\n", rank);
            for (i = 0; i < rank; ++i)
                printf("ReadHDFDFile: dimSizes[%d] = %d\n", i, dimSizes[i]);
        }
#endif

        /*See if there's a min and max*/
        if (0 == DFSDgetmaxmin(&max, &min))
        {
#ifdef DEBUG
    printf("ReadHDFDFile: max = %f, min = %f\n", max, min);
#endif
	    minMaxSet = true;
        }
        else
        {
#ifdef DEBUG
    printf("ReadHDFDFile: No min and max\n");
#endif
            min = MINUSINF; 
            max = PLUSINF;
	    minMaxSet = false;
        }

        /*Read and create an object based on this data set*/
        {
            long *dims_L;
            real *dims;
            real *bounds;
            ObjPtr dataForm;
            ObjPtr dimsArray;
            ObjPtr fieldData;
            ObjPtr boundsArray;
            ObjPtr *scales;
            char label[256];
            char *s;		/*Temporary string pointer*/
            char dummy[256];
            real *dataMeat;
            ObjPtr vectorDataset, vectorField, *elements;
            long temp_long;
            ObjPtr indices;
            int mem_ok;

            /*Make data form*/
            dataForm = NewObject(dataFormClass, 0);
            dimsArray = NewRealArray(1, (long) rank);

            mem_ok = 0;

            if (!(dims_L = malloc(sizeof(long) * rank)))
            {
malloc_error:
                if (!mem_ok)
                    FileFormatError("ReadHDFDFile", "Can't allocate memory.");
                return NULLOBJ;
            }

            if (!(dims = malloc(sizeof(real) * rank)))
            {
malloc_error_1:
                free(dims_L);
                goto malloc_error;
            }

            if (!(bounds = malloc(sizeof(real) * rank * 2)))
            {
malloc_error_2:
                free(dims);
                goto malloc_error_1;
            }

            if (!(scales = malloc(sizeof(ObjPtr) * rank)))
            {
malloc_error_3:
                free(bounds);
                goto malloc_error_2;
            }

            mem_ok = 1;

            /*Make DIMENSIONS (convert from int to long to real in the process)*/

            for (k = 0; k < rank; ++k)
                dims[k] = dims_L[k] = dimSizes[k];
            CArray2Array(dimsArray, dims);
            SetVar(dataForm, DIMENSIONS, dimsArray);
		
            /*Find bounds for the file*/

            for (k = 0; k < rank; ++k)
            {
                if (DFSDgetdimstrs(k + 1, dimsLabel, dimsUnit, dimsFormat) != -1)
                {
#ifdef DEBUG
    printf("ReadHDFDFile: dimsLabel = %s, DimsUnit = %s, dimsFormat = %s\n", dimsLabel, dimsUnit, dimsFormat);
#endif
                }
                scales[k] = NewRealArray(1, (long) dimSizes[k]);
                indices = NewRealArray(1, 1L);
                *(real *)ELEMENTS(indices) = k;
                SetVar(scales[k], INDICES, indices);

                if (DFSDgetdimscale(k + 1, dimSizes[k], ArrayMeat(scales[k])) != -1)
                {
                    switch (k)
                    {
                        case 0:	    SetVar(dataForm, XSCALE, scales[k]); break;
                        case 1:	    SetVar(dataForm, YSCALE, scales[k]); break;
                        default:    SetVar(dataForm, ZSCALE, scales[k]); break;
                    }
                    bounds[k * 2] = *ArrayMeat(scales[k]);
                    bounds[(k * 2) + 1] = *(ArrayMeat(scales[k]) + dimSizes[k] - 1);
                }
                else
                {
/*                    AddToReferenceList(scales[k]);
                    DeleteThing(scales[k]);*/
                    printf("Guessing scale %d from %d to %d\n", k, 0, dimSizes[k]);
                    bounds[k * 2] = 0.0;
                    bounds[(k * 2) + 1] = dims[k] - 1.0;

                    {
                        /*EMP fill in the scale*/
                        int i;
                        real *elements;
                        elements = ELEMENTS(scales[k]);
                        for (i = 0; i < dimSizes[k]; ++i)
                        {
                            elements[i] = i;
                        }
                    }
                }
#ifdef DEBUG
                {
                    int i;
                    real *elements;

                    elements = ELEMENTS(scales[k]);

                    for (i = 0; i < dimSizes[k]; ++i)
                        printf("ReadHDFDFile: elements[%d][%d] = %f\n", k, i, elements[i]);
                }

    printf("ReadHDFDFile: lower bound = %f, upper bound = %f\n", bounds[k * 2], bounds[(k * 2) + 1]);
#endif
            }
            boundsArray = NewRealArray(1, (long) (rank * 2));
            CArray2Array(boundsArray, bounds);
            SetVar(dataForm, BOUNDS, boundsArray);
            SetVar(dataForm, NCOMPONENTS, NewInt(rank));

            label[0] = '\0';    /* Init label[] to null string */

            /*Get the name*/
            if ((-1 == DFSDgetdatastrs(label, dummy, dummy, dummy)) || (label[0] == '\0'))
            {
                strcpy(label, fileName);
            }
		
#ifdef DEBUG
    printf("ReadHDFDFile: label = %s\n", label);
#endif
            /*Read in the data*/
            fieldData = NewArray(AT_REAL, rank, dims_L);
            if (-1 == DFSDgetdata(fileName, rank, dimSizes, ArrayMeat(fieldData)))
            {
                FileFormatError("ReadHDFDFile", "Can't read field data.");
                goto malloc_error_3;
            }

            if (outOfBoundsAction < 2)
            {   /*Search for out of bounds data*/
                arraySize = dimSizes[0];
                if (rank > 1)
                    for (k = 1; k < rank; ++k)
                        arraySize *= dimSizes[k];
                dataMeat = ArrayMeat(fieldData);
                for (k = 0; k < arraySize; ++k)
                {
                    if (*dataMeat < min)
                        *dataMeat = (outOfBoundsAction == 1) ? min : missingData;
                    else
                        if (*dataMeat > max)
                            *dataMeat = (outOfBoundsAction == 1) ? max : missingData;
                    ++dataMeat;
                }
            }

            /* Create "vector field" dataset of scales */

            temp_long = rank;

            vectorDataset = NewObject(datasetClass, 0);
            vectorField = NewArray(AT_OBJECT, 1, &temp_long);

            elements = ELEMENTS(vectorField);
            for (k = 0; k < rank; ++k)
                elements[k] = scales[k];

            SetVar(vectorDataset, DATA, vectorField);
            SetVar(vectorDataset, NCOMPONENTS, NewInt(rank));

            SetVar(dataForm, DATA, vectorDataset);

            /*Create the field*/

            curField = NewObject(datasetClass, 0);
            SetVar(curField, DATA, fieldData);
            SetVar(curField, DATAFORM, dataForm);
            switch (rank)
            {
                case 1:     SetVar(curField, DEFAULTICON, icon1DScalar); break;
                case 2:     SetVar(curField, DEFAULTICON, icon2DScalar); break;
                case 3:     SetVar(curField, DEFAULTICON, icon3DScalar); break;
                default:    SetVar(curField, DEFAULTICON, icon4DScalar); break;
            }

            /*Remove leading blanks on label*/
            s = label;
            while (*s && *s == ' ') ++s;
            SetVar(curField, NAME, NewString(s));

            /*Set the min and max*/
            if (minMaxSet)
            {
                real minMax[2];
                ObjPtr minMaxArray;
                minMax[0] = min;
                minMax[1] = max;
                minMaxArray = NewRealArray(1, 2L);
                CArray2Array(minMaxArray, minMax);
                SetVar(curField, MINMAX, minMaxArray);
            }

            free(bounds);
            free(dims);
            free(dims_L);
        }

        /*Add the current field*/
        if (curField)
        {
            RegisterDataset(curField);
        }
    }

    DFSDrestart();

    return NULLOBJ;
} /* ReadHDFDFile */
#endif
