/*ScianPictures.c
  Eric Pepke
  June 25, 1990
  Code to draw 3-D pictures in scian.

  12/4/91	Fixed bug with cylinders along x axis
*/

#include "Scian.h"
#include "ScianTypes.h"
#include "ScianColors.h"
#include "ScianPictures.h"
#include "ScianIDs.h"
#include "ScianErrors.h"
#include "ScianWindows.h"
#include "ScianDatasets.h"
#include "ScianArrays.h"
#include "ScianPreferences.h"
#include "ScianSpaces.h"

#define MAXCYLSIDES	512
#define CYLSIDES	16	/*Sides around cylinders, must be 2^n, 4 or more*/
#define MAXSUB		8	/*Maximum number of subdivisions*/
#ifndef SPHERESUB
#define SPHERESUB	2	/*Number of subdivisions of a sphere*/
#endif

Linestyle dashedLine = 0xF0F0;

double cefs[MAXSUB];		/*Chord extension factor for each subdivision*/
double cefs_2[MAXSUB];		/*cefs / 2*/

ObjPtr picClass = 0;		/*Class for all pictures*/

int curColorMode;		/*The current color mode, CMODECMAP or CMODERGB*/
int curColorShading;		/*The current color shading*/
int curLightShading;		/*The current light shading*/ 
Bool curIsTransparent;		/*True iff transparent*/

unsigned short greyPattern[16] = 
	{
	    0x5555,
	    0xAAAA,
	    0x5555,
	    0xAAAA,
	    0x5555,
	    0xAAAA,
	    0x5555,
	    0xAAAA,
	    0x5555,
	    0xAAAA,
	    0x5555,
	    0xAAAA,
	    0x5555,
	    0xAAAA,
	    0x5555,
	    0xAAAA
	};

Matrix cardBasis=
    {
        {-0.5,  1.5, -1.5,  0.5},
        { 1.0, -2.5,  2.0, -0.5},
        {-0.5,  0.0,  0.5,  0.0},
        { 0.0,  1.0,  0.0,  0.0}
    };

static ObjPtr MakePictureColored(picture)
ObjPtr picture;
/*Makes a picture's PICCOLORED variable.  Really doesn't have to do anything*/
{
    SetVar(picture, PICCOLORED, ObjTrue);
}

void InitPictures()
/*Initializes the pictures system*/
{
    int k;
    double theta;

    deflinestyle(DASHEDLINE, dashedLine);

    /*Define the grey pattern*/
    defpattern(GREYPAT, 16, greyPattern);

    /*Define the cardinal spline basis*/
    defbasis(CARDBASIS, cardBasis);

    picClass = NewObject(NULLOBJ, 0);
    AddToReferenceList(picClass);
    DeclareIndirectDependency(picClass, PICCOLORED, REPOBJ, PICCOLORED);
    SetMethod(picClass, PICCOLORED, MakePictureColored);

    /*Initialize cef*/
    theta = M_PI_4;
    for (k = 0; k < MAXSUB; ++k)
    {
	cefs[k] = 1.0 / cos(theta);
	cefs_2[k] = cefs[k] * 0.5;
	theta *= 0.5;
    }
}

void KillPictures()
/*Kills the pictures system*/
{
    DeleteThing(picClass);
}

static ObjPtr CleanupPicture(pic)
PicPtr pic;
/*Cleans up pic*/
{
    PicItemPtr next;
    while (pic -> items)
    {
	if (pic -> items -> type == POLYGONS)
	{
	    PolyPtr runner, nr;
	    runner = ((PolysPtr) pic -> items) -> polygons;
	    while (runner)
	    {
		nr = (PolyPtr) runner -> item . next;
		free(runner);
		runner = nr;
	    }
	}
	next = pic -> items -> next;
	free(pic -> items);
	pic -> items = next;
    }
    pic -> lastItem = 0;
    return ObjTrue;
}

ObjPtr NewPicture()
/*Returns a new empty picture or 0 if it could not make one.*/
{
    PicPtr retVal;
    retVal = (PicPtr) NewObject(picClass, sizeof(Picture) - sizeof(Thing));
    if (retVal)
    {
        retVal -> thing . flags = PICTURE;
	SetMethod((ObjPtr) retVal, CLEANUP, CleanupPicture);

	retVal -> items = 0;
	retVal -> lastItem = 0;
	return (ObjPtr) retVal;
    }
    else
    {
	return 0;
    }
}

PolysPtr NewPolygons()
/*Returns a new empty set of polygons 0 if it could not make one.*/
{
    PolysPtr item;
    item = (PolysPtr) malloc(sizeof(Polygons));
    if (item)
    {
	item -> item . type = POLYGONS;
	item -> item . colorShading = SMOOTHCOLOR;
	item -> item . lightShading = MONOLIGHT;

	item -> polygons = 0;
	item -> lastItem = 0;
	item -> enclosed = false;
	return item;
    }
    else
    {
	return 0;
    }
}

Bool AppendPolyToPolys(polys, nVertices, vertices)
PolysPtr polys;
int nVertices;
VertexTuple vertices[];
/*Append a polygon with nVertices where vertices is an array of VertexTuples
  to polygons polys.  Returns true iff successful.*/
{
    PolyPtr item;
    int k, d;

    item = (PolyPtr)
	   malloc(sizeof(Polygon) + (nVertices - 1) * sizeof(VertexTuple));
    if (item)
    {
	item -> item . type = POLYGON;
	item -> item . colorShading = SMOOTHCOLOR;
	item -> item . lightShading = MONOLIGHT;
	item -> nVertices = nVertices;
	for (k = 0; k < nVertices; ++k)
	{
	    for (d = 0; d < 3; ++d)
	    {
		item -> vertices[k] . position[d] = vertices[k] . position[d];
		item -> vertices[k] . normal[d] = vertices[k] . normal[d];
	    }
	    item -> vertices[k] . colorIndex = vertices[k] . colorIndex;
	}
	if (polys -> polygons)
	{
	    /*It's not the first*/
	    polys -> lastItem -> item . next = (PicItemPtr) item;
	}
	else
	{
	    /*It is the first*/
	    polys -> polygons = item;
	}
	item -> item . next = 0;
	polys -> lastItem = item;
	return true;
    }
    else
    {
	OMErr();
	return false;
    }
}

void AppendItemToPicture(pic, item)
PicPtr pic;
PicItemPtr item;
/*Appends item to the end of pic*/
{
    if (pic -> items)
    {
	/*It's not the first*/
	pic -> lastItem -> next = item;
    }
    else
    {
	/*It is the first*/
	pic -> items = item;
    }
    item -> next = 0;
    pic -> lastItem = item;
}

#ifdef PROTO
RectMeshPtr NewRectMesh(long xDim, long yDim, Bool inCenter)
#else
RectMeshPtr NewRectMesh(xDim, yDim, inCenter)
long yDim, xDim;
Bool inCenter;
#endif
/*Returns a new rectangular mesh.  inCenter if nodes in center*/
{
    RectMeshPtr item;

    item = (RectMeshPtr)
	   malloc(sizeof(RectMesh) + (xDim * (2 * yDim - 1)) *
			sizeof(VertexTuple));
    item -> item . type = RECTMESH;
    item -> item . colorShading = SMOOTHCOLOR;
    item -> item . lightShading = SMOOTHLIGHT;
    item -> inCenter = inCenter;
    item -> xDim = xDim;
    item -> yDim = yDim;

    return item;
}

#define RECTMESHVERTEX(rectMesh, i, j) ((j) * 2 + (i) * (2 * (rectMesh) -> yDim - 1))
#define RECTMESHCENTER(rectMesh, i, j) ((j) * 2 + 1 + (i) * (2 * (rectMesh) -> yDim - 1))

#ifdef PROTO
void SetRectMeshVPosition(RectMeshPtr rectMesh, long i, long j, real x, real y, real z)
#else
void SetRectMeshVPosition(rectMesh, i, j, x, y, z)
RectMeshPtr rectMesh;
long i, j;
real x, y, z; 
#endif
/*Sets the position of vertex i, j to x, y, z*/
{
    long offset;
    offset = RECTMESHVERTEX(rectMesh, i, j);
    rectMesh -> vertices[offset] . position[0] = x; 
    rectMesh -> vertices[offset] . position[1] = y;
    rectMesh -> vertices[offset] . position[2] = z;
}

#ifdef PROTO
void GetRectMeshVPosition(RectMeshPtr rectMesh, long i, long j, real *x, real *y, real *z)
#else
void GetRectMeshVPosition(rectMesh, i, j, x, y, z)
RectMeshPtr rectMesh;
long i, j;
real *x, *y, *z; 
#endif
/*Gets the position of vertex i, j to x, y, z*/
{
    long offset;
    offset = RECTMESHVERTEX(rectMesh, i, j);
    *x = rectMesh -> vertices[offset] . position[0]; 
    *y = rectMesh -> vertices[offset] . position[1];
    *z = rectMesh -> vertices[offset] . position[2];
}

#ifdef PROTO
void SetRectMeshVNormal(RectMeshPtr rectMesh, long i, long j, real x, real y, real z)
#else
void SetRectMeshVNormal(rectMesh, i, j, x, y, z)
RectMeshPtr rectMesh;
long i, j;
real x, y, z; 
#endif
/*Sets the normal of vertex i, j to x, y, z*/
{
    long offset;
    offset = RECTMESHVERTEX(rectMesh, i, j);
    rectMesh -> vertices[offset] . normal[0] = x;
    rectMesh -> vertices[offset] . normal[1] = y;
    rectMesh -> vertices[offset] . normal[2] = z;
}

#ifdef PROTO
void GetRectMeshVNormal(RectMeshPtr rectMesh, long i, long j, real *x, real *y, real *z)
#else
void GetRectMeshVNormal(rectMesh, i, j, x, y, z)
RectMeshPtr rectMesh;
long i, j;
real *x, *y, *z; 
#endif
/*Sets the normal of vertex i, j to x, y, z*/
{
    long offset;
    offset = RECTMESHVERTEX(rectMesh, i, j);
    *x = rectMesh -> vertices[offset] . normal[0];
    *y = rectMesh -> vertices[offset] . normal[1];
    *z = rectMesh -> vertices[offset] . normal[2];
}

#ifdef PROTO
void SetRectMeshVColorIndex(RectMeshPtr rectMesh, long i, long j, int index)
#else
void SetRectMeshVColorIndex(rectMesh, i, j, index)
RectMeshPtr rectMesh;
long i, j;
int index; 
#endif
/*Sets the normal of vertex i, j to x, y, z*/
{
    long offset;
    offset = RECTMESHVERTEX(rectMesh, i, j);
    rectMesh -> vertices[offset] . colorIndex = index;
}

#ifdef PROTO
void SetRectMeshVAlpha(RectMeshPtr rectMesh, long i, long j, real alpha)
#else
void SetRectMeshVAlpha(rectMesh, i, j, alpha)
RectMeshPtr rectMesh;
long i, j;
real alpha; 
#endif
/*Sets the normal of vertex i, j to x, y, z*/
{
    long offset;
    offset = RECTMESHVERTEX(rectMesh, i, j);
    rectMesh -> vertices[offset] . alpha = alpha;
}

#ifdef PROTO
void SetRectMeshCPosition(RectMeshPtr rectMesh, long i, long j, real x, real y, real z)
#else
void SetRectMeshCPosition(rectMesh, i, j, x, y, z)
RectMeshPtr rectMesh;
long i, j;
real x, y, z; 
#endif
/*Sets the position of center i, j to x, y, z*/
{
    long offset;
    offset = RECTMESHCENTER(rectMesh, i, j);
    rectMesh -> vertices[offset] . position[0] = x; 
    rectMesh -> vertices[offset] . position[1] = y;
    rectMesh -> vertices[offset] . position[2] = z;
}

#ifdef PROTO
void GetRectMeshCPosition(RectMeshPtr rectMesh, long i, long j, real *x, real *y, real *z)
#else
void GetRectMeshCPosition(rectMesh, i, j, x, y, z)
RectMeshPtr rectMesh;
long i, j;
real *x, *y, *z; 
#endif
/*Sets the position of center i, j to x, y, z*/
{
    long offset;
    offset = RECTMESHCENTER(rectMesh, i, j);
    *x = rectMesh -> vertices[offset] . position[0]; 
    *y = rectMesh -> vertices[offset] . position[1];
    *z = rectMesh -> vertices[offset] . position[2];
}

#ifdef PROTO
void SetRectMeshCNormal(RectMeshPtr rectMesh, long i, long j, real x, real y, real z)
#else
void SetRectMeshCNormal(rectMesh, i, j, x, y, z)
RectMeshPtr rectMesh;
long i, j;
real x, y, z; 
#endif
/*Sets the normal of center i, j to x, y, z*/
{
    long offset;
    offset = RECTMESHCENTER(rectMesh, i, j);
    rectMesh -> vertices[offset] . normal[0] = x;
    rectMesh -> vertices[offset] . normal[1] = y;
    rectMesh -> vertices[offset] . normal[2] = z;
}

#ifdef PROTO
void GetRectMeshCNormal(RectMeshPtr rectMesh, long i, long j, real *x, real *y, real *z)
#else
void GetRectMeshCNormal(rectMesh, i, j, x, y, z)
RectMeshPtr rectMesh;
long i, j;
real *x, *y, *z; 
#endif
/*Sets the normal of center i, j to x, y, z*/
{
    long offset;
    offset = RECTMESHCENTER(rectMesh, i, j);
    *x = rectMesh -> vertices[offset] . normal[0];
    *y = rectMesh -> vertices[offset] . normal[1];
    *z = rectMesh -> vertices[offset] . normal[2];
}

#ifdef PROTO
void SetRectMeshCColorIndex(RectMeshPtr rectMesh, long i, long j, int index)
#else
void SetRectMeshCColorIndex(rectMesh, i, j, index)
RectMeshPtr rectMesh;
long i, j;
int index; 
#endif
/*Sets the normal of center i, j to x, y, z*/
{
    long offset;
    offset = RECTMESHCENTER(rectMesh, i, j);
    rectMesh -> vertices[offset] . colorIndex = index;
}

#ifdef PROTO
void SetRectMeshCAlpha(RectMeshPtr rectMesh, long i, long j, real alpha)
#else
void SetRectMeshCAlpha(rectMesh, i, j, alpha)
RectMeshPtr rectMesh;
long i, j;
real alpha; 
#endif
/*Sets the normal of center i, j to x, y, z*/
{
    long offset;
    offset = RECTMESHCENTER(rectMesh, i, j);
    rectMesh -> vertices[offset] . alpha = alpha;
}

#ifdef PROTO
void CalcRectNormals(RectMeshPtr rectMesh)
#else
void CalcRectNormals(rectMesh)
RectMeshPtr rectMesh;
#endif
/*Calculates the normals in rectMesh and copies them to the vertices.
  Obviously, do this after the positions have been set.*/
{
    register long i, j, k, offset, offset2, iDim, jDim, nX, nY;
    register VertexPtr vertices;
    float normal[3];		/*Normal at center vertex*/
    float position1[3];		/*Position of base vertex*/
    float vI[3], vJ[3];		/*Vectors for cross product*/

    iDim = rectMesh -> xDim;
    jDim = rectMesh -> yDim;
    vertices = rectMesh -> vertices;
    nX = jDim * 2 - 1;
#define nY 2

    for (j = 0; j < jDim - 1; ++j)
    {
	offset = RECTMESHVERTEX(rectMesh, 0, j);
	for (i = 0; i < iDim - 1; ++i)
	{
	    /*Get position here at lower left*/
	    position1[0] = vertices[offset] . position[0];
	    position1[1] = vertices[offset] . position[1];
	    position1[2] = vertices[offset] . position[2];

	    /*Get vector in j direction*/
	    offset2 = offset + nY;
	    vJ[0] = vertices[offset2] . position[0] - position1[0];
	    vJ[1] = vertices[offset2] . position[1] - position1[1];
	    vJ[2] = vertices[offset2] . position[2] - position1[2];

	    if (vJ[0] == 0.0 && vJ[1] == 0.0 && vJ[2] == 0.0)
	    {
		/*Have to try another way to get vJ*/
		long offset3;
		offset3 = offset + nX;
		offset2 = offset2 + nX;
		vJ[0] = vertices[offset2] . position[0] - 
			vertices[offset3] . position[0];
		vJ[1] = vertices[offset2] . position[1] - 
			vertices[offset3] . position[1];
	 	vJ[2] = vertices[offset2] . position[2] - 
			vertices[offset3] . position[2];
	    }

	    /*Get vector in i direction*/
	    offset2 = offset + nX;
	    vI[0] = vertices[offset2] . position[0] - position1[0];
	    vI[1] = vertices[offset2] . position[1] - position1[1];
	    vI[2] = vertices[offset2] . position[2] - position1[2];

	    if (vI[0] == 0.0 && vI[1] == 0.0 && vI[2] == 0.0)
	    {
		/*Have to try another way to get vJ*/
		long offset3;
		offset3 = offset + nY;
		offset2 = offset2 + nY;
		vI[0] = vertices[offset2] . position[0] - 
			vertices[offset3] . position[0];
		vI[1] = vertices[offset2] . position[1] - 
			vertices[offset3] . position[1];
		vI[2] = vertices[offset2] . position[2] - 
			vertices[offset3] . position[2];
	    }

	    /*Set normal to cross product*/
	    normal[0] = vI[1] * vJ[2] - vI[2] * vJ[1];
	    normal[1] = vI[2] * vJ[0] - vI[0] * vJ[2];
	    normal[2] = vI[0] * vJ[1] - vI[1] * vJ[0];

	    /*Normalize it*/
	    NORMALIZE(normal);

	    /*Set it to center*/
	    vertices[offset + 1] . normal[0] = normal[0];
	    vertices[offset + 1] . normal[1] = normal[1];
	    vertices[offset + 1] . normal[2] = normal[2];

	    offset = offset2;
	}
    }
#undef nY

    /*Interpolate the centers with four corners*/
    for (i = 1; i < iDim - 1; ++i)
    {
	for (j = 1; j < jDim - 1; ++j)
	{
	    /*Lower left corner*/
	    offset = RECTMESHCENTER(rectMesh, i - 1, j - 1);
	    normal[0] = vertices[offset] . normal[0];
	    normal[1] = vertices[offset] . normal[1];
	    normal[2] = vertices[offset] . normal[2];

	    /*Upper left corner*/
	    offset = RECTMESHCENTER(rectMesh, i - 1, j);
	    normal[0] += vertices[offset] . normal[0];
	    normal[1] += vertices[offset] . normal[1];
	    normal[2] += vertices[offset] . normal[2];

	    /*Lower right corner*/
	    offset = RECTMESHCENTER(rectMesh, i, j - 1);
	    normal[0] += vertices[offset] . normal[0];
	    normal[1] += vertices[offset] . normal[1];
	    normal[2] += vertices[offset] . normal[2];

	    /*Upper right corner*/
	    offset = RECTMESHCENTER(rectMesh, i, j);
	    normal[0] += vertices[offset] . normal[0];
	    normal[1] += vertices[offset] . normal[1];
	    normal[2] += vertices[offset] . normal[2];

	    /*Normalize results*/
	    NORMALIZE(normal);

	    /*Stuff them*/
	    offset = RECTMESHVERTEX(rectMesh, i, j);
	    vertices[offset] . normal[0] = normal[0];
	    vertices[offset] . normal[1] = normal[1];
	    vertices[offset] . normal[2] = normal[2];
	}
    }

    /*Copy to left and right sides*/
    for (j = 0; j < jDim - 1; ++j)
    {
	/*Left side*/
	offset = RECTMESHCENTER(rectMesh, 0, j);
	for (k = 0; k < 3; ++k)
	{
	    normal[k] = vertices[offset] . normal[k];
	}
	/*Stuff them back*/
	offset = RECTMESHVERTEX(rectMesh, 0, j);
	for (k = 0; k < 3; ++k)
	{
	    vertices[offset] . normal[k] = normal[k];
	}

	/*Right side*/
	offset = RECTMESHCENTER(rectMesh, iDim - 2, j);
	for (k = 0; k < 3; ++k)
	{
	    normal[k] = vertices[offset] . normal[k];
	}
	/*Stuff them back*/
	offset = RECTMESHVERTEX(rectMesh, iDim - 1, j);
	for (k = 0; k < 3; ++k)
	{
	    vertices[offset] . normal[k] = normal[k];
	}
    }	

    /*Copy to bottom and top sides*/
    for (i = 1; i < iDim - 1; ++i)
    {
	/*Bottom side*/
	offset = RECTMESHCENTER(rectMesh, i, 0);
	for (k = 0; k < 3; ++k)
	{
	    normal[k] = vertices[offset] . normal[k];
	}
	/*Stuff them back*/
	offset = RECTMESHVERTEX(rectMesh, i, 0);
	for (k = 0; k < 3; ++k)
	{
	    vertices[offset] . normal[k] = normal[k];
	}

	/*Top side*/
	offset = RECTMESHCENTER(rectMesh, i, jDim - 2);
	for (k = 0; k < 3; ++k)
	{
	    normal[k] = vertices[offset] . normal[k];
	}
	/*Stuff them back*/
	offset = RECTMESHVERTEX(rectMesh, i, jDim - 1);
	for (k = 0; k < 3; ++k)
	{
	    vertices[offset] . normal[k] = normal[k];
	}
    }

    /*Do upper left corner*/
    offset = RECTMESHCENTER(rectMesh, 0, jDim - 2);
    for (k = 0; k < 3; ++k)
    {
	normal[k] = vertices[offset] . normal[k];
    }
    offset = RECTMESHVERTEX(rectMesh, 0, jDim - 1);
    for (k = 0; k < 3; ++k)
    {
	vertices[offset] . normal[k] = normal[k];
    }

    /*Do lower right corner*/
    offset = RECTMESHCENTER(rectMesh, iDim - 2, 0);
    for (k = 0; k < 3; ++k)
    {
	normal[k] = vertices[offset] . normal[k];
    }
    offset = RECTMESHVERTEX(rectMesh, iDim - 1, 0);
    for (k = 0; k < 3; ++k)
    {
	vertices[offset] . normal[k] = normal[k];
    }

    /*Do upper right corner*/
    offset = RECTMESHCENTER(rectMesh, iDim - 2, jDim - 2);
    for (k = 0; k < 3; ++k)
    {
	normal[k] = vertices[offset] . normal[k];
    }
    offset = RECTMESHVERTEX(rectMesh, iDim - 1, jDim - 1);
    for (k = 0; k < 3; ++k)
    {
	vertices[offset] . normal[k] = normal[k];
    }
}

#ifdef PROTO
void InterpRectCenters(RectMeshPtr rectMesh)
#else
void InterpRectCenters(rectMesh)
RectMeshPtr rectMesh;
#endif
/*Interpolates rect centers from vertices*/
{
    register long i, j, offset, iDim, jDim, nX;
    register VertexPtr vertices;
    float position[3];		/*Position of center vertex*/
    float alpha;		/*Alpha transparency value*/
    int colorIndex;		/*Color within color table*/

    iDim = rectMesh -> xDim;
    jDim = rectMesh -> yDim;
    vertices = rectMesh -> vertices;
    nX = jDim * 2 - 1;

    for (i = 0; i < iDim - 1; ++i)
    {
	for (j = 0; j < jDim - 1; ++j)
	{
	    /*Lower left corner*/
	    offset = RECTMESHVERTEX(rectMesh, i, j);
	    position[0] = vertices[offset] . position[0];
	    position[1] = vertices[offset] . position[1];
	    position[2] = vertices[offset] . position[2];
	    colorIndex = vertices[offset] . colorIndex;
	    alpha = vertices[offset] . alpha;

	    /*Upper left corner*/
	    offset = RECTMESHVERTEX(rectMesh, i, j + 1);
	    position[0] += vertices[offset] . position[0];
	    position[1] += vertices[offset] . position[1];
	    position[2] += vertices[offset] . position[2];
	    colorIndex += vertices[offset] . colorIndex;
	    alpha += vertices[offset] . alpha;

	    /*Lower right corner*/
	    offset = RECTMESHVERTEX(rectMesh, i + 1, j);
	    position[0] += vertices[offset] . position[0];
	    position[1] += vertices[offset] . position[1];
	    position[2] += vertices[offset] . position[2];
	    colorIndex += vertices[offset] . colorIndex;
	    alpha += vertices[offset] . alpha;

	    /*Upper right corner*/
	    offset = RECTMESHVERTEX(rectMesh, i + 1, j + 1);
	    position[0] += vertices[offset] . position[0];
	    position[1] += vertices[offset] . position[1];
	    position[2] += vertices[offset] . position[2];
	    colorIndex += vertices[offset] . colorIndex;
	    alpha += vertices[offset] . alpha;

	    /*Normalize results*/
	    position[0] *= 0.25;
	    position[1] *= 0.25;
	    position[2] *= 0.25;
	    colorIndex /= 4;
	    alpha *= 0.25;

	    /*Stuff them*/
	    offset = RECTMESHCENTER(rectMesh, i, j);
	    vertices[offset] . position[0] = position[0];
	    vertices[offset] . position[1] = position[1];
	    vertices[offset] . position[2] = position[2];
	    vertices[offset] . colorIndex = colorIndex;
	    vertices[offset] . alpha = alpha;
	}
    }
}

#ifdef PROTO
void InterpRectVertices(RectMeshPtr rectMesh)
#else
void InterpRectVertices(rectMesh)
RectMeshPtr rectMesh;
#endif
/*Interpolates and extrapolates rect vertices from centers*/
{
    register long i, j, k, offset, iDim, jDim, nX;
    register VertexPtr vertices;
    float position[3];		/*Position of center vertex*/
    float alpha;		/*Alpha transparency value*/
    int colorIndex;		/*Color within color table*/

    iDim = rectMesh -> xDim;
    jDim = rectMesh -> yDim;

    vertices = rectMesh -> vertices;
    nX = jDim * 2 - 1;

    /*Interpolate the centers with four corners*/
    for (i = 1; i < iDim - 1; ++i)
    {
	for (j = 1; j < jDim - 1; ++j)
	{
	    /*Lower left corner*/
	    offset = RECTMESHCENTER(rectMesh, i - 1, j - 1);
	    position[0] = vertices[offset] . position[0];
	    position[1] = vertices[offset] . position[1];
	    position[2] = vertices[offset] . position[2];
	    colorIndex = vertices[offset] . colorIndex;
	    alpha = vertices[offset] . alpha;

	    /*Upper left corner*/
	    offset = RECTMESHCENTER(rectMesh, i - 1, j);
	    position[0] += vertices[offset] . position[0];
	    position[1] += vertices[offset] . position[1];
	    position[2] += vertices[offset] . position[2];
	    colorIndex += vertices[offset] . colorIndex;
	    alpha += vertices[offset] . alpha;

	    /*Lower right corner*/
	    offset = RECTMESHCENTER(rectMesh, i, j - 1);
	    position[0] += vertices[offset] . position[0];
	    position[1] += vertices[offset] . position[1];
	    position[2] += vertices[offset] . position[2];
	    colorIndex += vertices[offset] . colorIndex;
	    alpha += vertices[offset] . alpha;

	    /*Upper right corner*/
	    offset = RECTMESHCENTER(rectMesh, i, j);
	    position[0] += vertices[offset] . position[0];
	    position[1] += vertices[offset] . position[1];
	    position[2] += vertices[offset] . position[2];
	    colorIndex += vertices[offset] . colorIndex;
	    alpha += vertices[offset] . alpha;

	    /*Normalize results*/
	    position[0] *= 0.25;
	    position[1] *= 0.25;
	    position[2] *= 0.25;
	    colorIndex /= 4;
	    alpha *= 0.25;

	    /*Stuff them*/
	    offset = RECTMESHVERTEX(rectMesh, i, j);
	    vertices[offset] . position[0] = position[0];
	    vertices[offset] . position[1] = position[1];
	    vertices[offset] . position[2] = position[2];
	    vertices[offset] . colorIndex = colorIndex;
	    vertices[offset] . alpha = alpha;
	}
    }

    /*Extrapolate left and right sides*/
    for (j = 1; j < jDim - 1; ++j)
    {
	/*Left side*/
	offset = RECTMESHVERTEX(rectMesh, 1, j);
	for (k = 0; k < 3; ++k)
	{
	    position[k] = vertices[offset] . position[k];
	}
	colorIndex = vertices[offset] . colorIndex;
	alpha = vertices[offset] . alpha;
	if (iDim >= 3)
	{
	    /*Extend position only*/
	    offset = RECTMESHVERTEX(rectMesh, 2, j);
	    for (k = 0; k < 3; ++k)
	    {
		position[k] += position[k] - vertices[offset] . position[k];
	    }
	}
	/*Stuff them back*/
	offset = RECTMESHVERTEX(rectMesh, 0, j);
	for (k = 0; k < 3; ++k)
	{
	    vertices[offset] . position[k] = position[k];
	}
	vertices[offset] . colorIndex = colorIndex;
	vertices[offset] . alpha = alpha;

	/*Right side*/
	offset = RECTMESHVERTEX(rectMesh, iDim - 2, j);
	for (k = 0; k < 3; ++k)
	{
	    position[k] = vertices[offset] . position[k];
	}
	colorIndex = vertices[offset] . colorIndex;
	alpha = vertices[offset] . alpha;
	if (iDim >= 3)
	{
	    /*Extend position only*/
	    offset = RECTMESHVERTEX(rectMesh, iDim - 3, j);
	    for (k = 0; k < 3; ++k)
	    {
		position[k] += position[k] - vertices[offset] . position[k];
	    }
	}
	/*Stuff them back*/
	offset = RECTMESHVERTEX(rectMesh, iDim - 1, j);
	for (k = 0; k < 3; ++k)
	{
	    vertices[offset] . position[k] = position[k];
	}
	vertices[offset] . colorIndex = colorIndex;
	vertices[offset] . alpha = alpha;
    }	

    /*Extrapolate bottom and top sides*/
    for (i = 1; i < iDim - 1; ++i)
    {
	/*Left side*/
	offset = RECTMESHVERTEX(rectMesh, i, 1);
	for (k = 0; k < 3; ++k)
	{
	    position[k] = vertices[offset] . position[k];
	}
	colorIndex = vertices[offset] . colorIndex;
	alpha = vertices[offset] . alpha;
	if (jDim >= 3)
	{
	    /*Extend position only*/
	    offset = RECTMESHVERTEX(rectMesh, i, 2);
	    for (k = 0; k < 3; ++k)
	    {
		position[k] += position[k] - vertices[offset] . position[k];
	    }
	}
	/*Stuff them back*/
	offset = RECTMESHVERTEX(rectMesh, i, 0);
	for (k = 0; k < 3; ++k)
	{
	    vertices[offset] . position[k] = position[k];
	}
	vertices[offset] . colorIndex = colorIndex;
	vertices[offset] . alpha = alpha;

	/*Right side*/
	offset = RECTMESHVERTEX(rectMesh, i, jDim - 2);
	for (k = 0; k < 3; ++k)
	{
	    position[k] = vertices[offset] . position[k];
	}
	colorIndex = vertices[offset] . colorIndex;
	alpha = vertices[offset] . alpha;
	if (jDim >= 3)
	{
	    /*Extend position only*/
	    offset = RECTMESHVERTEX(rectMesh, i, jDim - 3);
	    for (k = 0; k < 3; ++k)
	    {
		position[k] += position[k] - vertices[offset] . position[k];
	    }
	}
	/*Stuff them back*/
	offset = RECTMESHVERTEX(rectMesh, i, jDim - 1);
	for (k = 0; k < 3; ++k)
	{
	    vertices[offset] . position[k] = position[k];
	}
	vertices[offset] . colorIndex = colorIndex;
	vertices[offset] . alpha = alpha;
    }

    /*Do lower left corner*/
    offset = RECTMESHCENTER(rectMesh, 0, 0);
    for (k = 0; k < 3; ++k)
    {
	position[k] = vertices[offset] . position[k];
    }
    colorIndex = vertices[offset] . colorIndex;
    alpha = vertices[offset] . alpha;

    offset = RECTMESHVERTEX(rectMesh, 1, 1);
    for (k = 0; k < 3; ++k)
    {
	position[k] += position[k] - vertices[offset] . position[k];
    }
    offset = RECTMESHVERTEX(rectMesh, 0, 0);
    for (k = 0; k < 3; ++k)
    {
	vertices[offset] . position[k] = position[k];
    }
    vertices[offset] . colorIndex = colorIndex;
    vertices[offset] . alpha = alpha;

    /*Do upper left corner*/
    offset = RECTMESHCENTER(rectMesh, 0, jDim - 2);
    for (k = 0; k < 3; ++k)
    {
	position[k] = vertices[offset] . position[k];
    }
    colorIndex = vertices[offset] . colorIndex;
    alpha = vertices[offset] . alpha;

    offset = RECTMESHVERTEX(rectMesh, 1, jDim - 2);
    for (k = 0; k < 3; ++k)
    {
	position[k] += position[k] - vertices[offset] . position[k];
    }
    offset = RECTMESHVERTEX(rectMesh, 0, jDim - 1);
    for (k = 0; k < 3; ++k)
    {
	vertices[offset] . position[k] = position[k];
    }
    vertices[offset] . colorIndex = colorIndex;
    vertices[offset] . alpha = alpha;

    /*Do lower right corner*/
    offset = RECTMESHCENTER(rectMesh, iDim - 2, 0);
    for (k = 0; k < 3; ++k)
    {
	position[k] = vertices[offset] . position[k];
    }
    colorIndex = vertices[offset] . colorIndex;
    alpha = vertices[offset] . alpha;

    offset = RECTMESHVERTEX(rectMesh, iDim - 2, 1);
    for (k = 0; k < 3; ++k)
    {
	position[k] += position[k] - vertices[offset] . position[k];
    }
    offset = RECTMESHVERTEX(rectMesh, iDim - 1, 0);
    for (k = 0; k < 3; ++k)
    {
	vertices[offset] . position[k] = position[k];
    }
    vertices[offset] . colorIndex = colorIndex;
    vertices[offset] . alpha = alpha;

    /*Do upper left corner*/
    offset = RECTMESHCENTER(rectMesh, iDim - 2, jDim - 2);
    for (k = 0; k < 3; ++k)
    {
	position[k] = vertices[offset] . position[k];
    }
    colorIndex = vertices[offset] . colorIndex;
    alpha = vertices[offset] . alpha;

    offset = RECTMESHVERTEX(rectMesh, iDim - 2, jDim - 2);
    for (k = 0; k < 3; ++k)
    {
	position[k] += position[k] - vertices[offset] . position[k];
    }
    offset = RECTMESHVERTEX(rectMesh, iDim - 1, jDim - 1);
    for (k = 0; k < 3; ++k)
    {
	vertices[offset] . position[k] = position[k];
    }
    vertices[offset] . colorIndex = colorIndex;
    vertices[offset] . alpha = alpha;
}

Bool AppendPolyToPicture(pic, nVertices, vertices)
ObjPtr pic;
int nVertices;
VertexTuple vertices[];
/*Append a polygon with nVertices where vertices is an array of VertexTuples
  to picture pic.  Returns true iff successful.*/
{
    PolyPtr item;
    int k, d;

    item = (PolyPtr)
	   malloc(sizeof(Polygon) + (nVertices - 1) * sizeof(VertexTuple));
    if (item)
    {
	item -> item . type = POLYGON;
	item -> item . colorShading = SMOOTHCOLOR;
	item -> item . lightShading = MONOLIGHT;
	item -> nVertices = nVertices;
	for (k = 0; k < nVertices; ++k)
	{
	    for (d = 0; d < 3; ++d)
	    {
		item -> vertices[k] . position[d] = vertices[k] . position[d];
		item -> vertices[k] . normal[d] = vertices[k] . normal[d];
	    }
	    item -> vertices[k] . colorIndex = vertices[k] . colorIndex;
	}
	AppendItemToPicture((PicPtr) pic, (PicItemPtr) item);
	return true;
    }
    else
    {
	return false;
    }
}

#ifdef PROTO
Bool AppendFrustumToPicture(ObjPtr pic, float end1[3], float rad1,
		float end2[3], float rad2, int colorIndex)
#else
Bool AppendFrustumToPicture(pic, end1, rad1, end2, rad2, colorIndex)
ObjPtr pic;
float end1[3];
float rad1;
float end2[3];
float rad2;
int colorIndex;
#endif
/*Appends a conical frustum with endpoints end1 and end2 and radii rad1 and
  rad2 to picture pic*/
{
    FrustumPtr item;
    int k;

    item = new(Frustum);
    if (item)
    {
	item -> item . type = FRUSTUM;
	item -> item . colorShading = SMOOTHCOLOR;
	item -> item . lightShading = SMOOTHLIGHT;
	item -> rad1 = rad1;
	item -> rad2 = rad2;
	item -> colorIndex = colorIndex;
	for (k = 0; k < 3; ++k)
	{
	    item -> end1[k] = end1[k];
	    item -> end2[k] = end2[k];
	}
	AppendItemToPicture((PicPtr) pic, (PicItemPtr) item);
	return true;
    }
    else
    {
	return false;
    }
}

#ifdef PROTO
Bool AppendSphereToPicture(ObjPtr pic, float center[3], float radius, int colorIndex)
#else
Bool AppendSphereToPicture(pic, center, radius, colorIndex)
ObjPtr pic;
float center[3];
float radius;
int colorIndex;
#endif
/*Appends a sphere with center center and radius radius to pic*/
{
    SpherePtr item;
    int k;

    item = new(Sphere);
    if (item)
    {
	item -> item . type = SPHERE;
	item -> item . colorShading = SMOOTHCOLOR;
	item -> item . lightShading = SMOOTHLIGHT;
	item -> radius = radius;
	item -> colorIndex = colorIndex;
	for (k = 0; k < 3; ++k)
	{
	    item -> center[k] = center[k];
	}
	AppendItemToPicture((PicPtr) pic, (PicItemPtr) item);
	return true;
    }
    else
    {
	return false;
    }
}

static float curCenter[3];	/*Current center*/
static float curRadius;		/*Current radius*/
static int curSub;		/*Current number of subdivisions*/
static int curNSides;		/*Current number of sides*/

void DrawQuadrant(r1, r2, r3, level)
float r1[3];
float r2[3];
float r3[3];
int level;
/*Draws a quadrant of sphere at center curCenter with radius curRadius given 
  radial normal vectors r1, r2, and r3, using current color table, at recursion 
  level level.  Assumes that the appropriate color table is current*/
{
    if (level >= SPHERESUB)
    {
	/*Maximum number of subdivisions has been reached*/
	float v1[3], v2[3], v3[3];
	v1[0] = curCenter[0] + r1[0] * curRadius;
	v1[1] = curCenter[1] + r1[1] * curRadius;
	v1[2] = curCenter[2] + r1[2] * curRadius;
	v2[0] = curCenter[0] + r2[0] * curRadius;
	v2[1] = curCenter[1] + r2[1] * curRadius;
	v2[2] = curCenter[2] + r2[2] * curRadius;
	v3[0] = curCenter[0] + r3[0] * curRadius;
	v3[1] = curCenter[1] + r3[1] * curRadius;
	v3[2] = curCenter[2] + r3[2] * curRadius;

        if (rgbp)
        {
            bgnpolygon();
	    n3f(r1);
	    v3f(v1);
	    n3f(r2);
	    v3f(v2);
	    n3f(r3);
	    v3f(v3);
	    endpolygon();
        }
	else
        {
	    bgnpolygon();
	    SetRealColor(r1[0]);
	    v3f(v1);
	    SetRealColor(r2[0]);
	    v3f(v2);
	    SetRealColor(r3[0]);
	    v3f(v3);
            endpolygon();
        }
    }
    else
    {
	/*Do another subdivision*/
	float r12[3], r23[3], r31[3];	/*Inner triangle subdivisions*/
#if 0
	double cef_2;			/*Chord extension factor / 2*/

	cef_2 = cefs_2[level];

	r12[0] = (r1[0] + r2[0]) * cef_2;
	r12[1] = (r1[1] + r2[1]) * cef_2;
	r12[2] = (r1[2] + r2[2]) * cef_2;
	r23[0] = (r2[0] + r3[0]) * cef_2;
	r23[1] = (r2[1] + r3[1]) * cef_2;
	r23[2] = (r2[2] + r3[2]) * cef_2;
	r31[0] = (r3[0] + r1[0]) * cef_2;
	r31[1] = (r3[1] + r1[1]) * cef_2;
	r31[2] = (r3[2] + r1[2]) * cef_2;
#else
	r12[0] = (r1[0] + r2[0]);
	r12[1] = (r1[1] + r2[1]);
	r12[2] = (r1[2] + r2[2]);
	r23[0] = (r2[0] + r3[0]);
	r23[1] = (r2[1] + r3[1]);
	r23[2] = (r2[2] + r3[2]);
	r31[0] = (r3[0] + r1[0]);
	r31[1] = (r3[1] + r1[1]);
	r31[2] = (r3[2] + r1[2]);
	NORMALIZE(r12);
	NORMALIZE(r23);
	NORMALIZE(r31);
#endif

	/*Draw the subdivisions*/
	DrawQuadrant(r1, r12, r31, level + 1);
	DrawQuadrant(r12, r2, r23, level + 1);
	DrawQuadrant(r23, r3, r31, level + 1);
	DrawQuadrant(r12, r23, r31, level + 1);
    }
}

#ifdef PROTO
void SetPicColorShading(ObjPtr pic, int shading)
#else
void SetPicColorShading(pic, int shading)
ObjPtr pic;
#endif
/*Sets the shadecolor value in pic to shading*/
{
    PicItemPtr curItem;
    ObjPtr repObj;
    curItem = ((PicPtr) pic) -> items;

    while (curItem)
    {
	curItem -> colorShading = shading;
	curItem = curItem -> next;
    }
}

#ifdef PROTO
void SetPicLightShading(ObjPtr pic, int shading)
#else
void SetPicLightShading(pic, int shading)
ObjPtr pic;
#endif
/*Sets the shadecolor value in pic to shading*/
{
    PicItemPtr curItem;
    ObjPtr repObj;
    curItem = ((PicPtr) pic) -> items;
    while (curItem)
    {
	curItem -> lightShading = shading;
	curItem = curItem -> next;
    }
}

#ifdef PROTO
void UnColorPicture(ObjPtr pic)
#else
void UnColorPicture(pic)
ObjPtr pic;
#endif
/*Uncolors picture pic*/
{
    PicItemPtr curItem;
    ObjPtr repObj;

    curItem = ((PicPtr) pic) -> items;

    while (curItem)
    {
	curItem = curItem -> next;
    }
}

#ifdef PROTO
void ColorItemByObject(PicItemPtr curItem,ObjPtr colorObj, Bool interpolate)
#else
void ColorItemByObject(curItem, colorObj, interpolate)
ObjPtr pic;
PicItemPtr curItem;
Bool interpolate;
#endif
/*Colors item curItem by colorObj*/
{
    long k;
	long nVertices;
	VertexPtr vertices;

	switch(curItem -> type)
	{
	    case POLYGON:
	    case PLANARPOLYGON:
		nVertices = ((PolyPtr) curItem) -> nVertices;
		vertices = ((PolyPtr) curItem) -> vertices;
		break;
	    case POLYGONS:
		ColorItemsByObject((PicItemPtr) ((PolysPtr) curItem) -> polygons, 
			colorObj, interpolate);
		return;
		break;
	    case RECTMESH:
		{
		    int i, j;
		    real sample;
		    int index;
		    long offset;
		    real position[3];


		    vertices = ((RectMeshPtr) curItem) -> vertices;
		    if (((RectMeshPtr) curItem) -> inCenter)
		    {
			for (i = 0; i < ((RectMeshPtr) curItem) -> xDim - 1; ++i)
			{
			    for (j = 0; j < ((RectMeshPtr) curItem) -> yDim - 1; ++j)
			    {
				offset = RECTMESHCENTER((RectMeshPtr) curItem, i, j);
				position[0] = vertices[offset] . position[0];
				position[1] = vertices[offset] . position[1];
				position[2] = vertices[offset] . position[2];

				if (SampleSpatField(FIELD1, FIELD2, 1, &sample,
				    3, position, interpolate));

				index = GetRealColorIndex(sample);
				vertices[offset] . colorIndex = index;
			    }
			}
			InterpRectVertices((RectMeshPtr) curItem);
		    }
		    else
		    {
			for (i = 0; i < ((RectMeshPtr) curItem) -> xDim; ++i)
			{
			    for (j = 0; j < ((RectMeshPtr) curItem) -> yDim; ++j)
			    {
				offset = RECTMESHVERTEX((RectMeshPtr) curItem, i, j);

				position[0] = vertices[offset] . position[0];
				position[1] = vertices[offset] . position[1];
				position[2] = vertices[offset] . position[2];

				if (SampleSpatField(FIELD1, FIELD2, 1, &sample,
				    3, position, interpolate));

				index = GetRealColorIndex(sample);
				vertices[offset] . colorIndex = index;
			    }
			}
			InterpRectCenters((RectMeshPtr) curItem);
		    }
		}
		return;
		break;
	    default:
		return;
	}

	/*Make it colored*/
	for (k = 0; k < nVertices; ++k)
	{
	    real position[3];
	    real sample;
	    int index;
	    int j;

	    position[0] = vertices[k] . position[0];
	    position[1] = vertices[k] . position[1];
	    position[2] = vertices[k] . position[2];

	    if (SampleSpatField(FIELD1, FIELD2, 1, &sample,
				    3, position, interpolate));

	    index = GetRealColorIndex(sample);
	    vertices[k] . colorIndex = index;
	}
}

#ifdef PROTO
void ColorItemWithIndex(PicItemPtr curItem, int index)
#else
void ColorItemWithIndex(curItem, index)
PicItemPtr curItem;
int index;
#endif
/*Colors item curItem with index*/
{
    long k;
	long nVertices;
	VertexPtr vertices;

	switch(curItem -> type)
	{
	    case POLYGON:
	    case PLANARPOLYGON:
		nVertices = ((PolyPtr) curItem) -> nVertices;
		vertices = ((PolyPtr) curItem) -> vertices;
		break;
	    case POLYGONS:
		ColorItemsWithIndex((PicItemPtr) ((PolysPtr) curItem) -> polygons, 
			index);
		return;
		break;
	    case RECTMESH:
		{
		    int i, j;
		    real sample;
		    long offset;

		    vertices = ((RectMeshPtr) curItem) -> vertices;
		    if (((RectMeshPtr) curItem) -> inCenter)
		    {
			for (i = 0; i < ((RectMeshPtr) curItem) -> xDim - 1; ++i)
			{
			    for (j = 0; j < ((RectMeshPtr) curItem) -> yDim - 1; ++j)
			    {
				offset = RECTMESHCENTER((RectMeshPtr) curItem, i, j);

				vertices[offset] . colorIndex = index;
			    }
			}
			InterpRectVertices((RectMeshPtr) curItem);
		    }
		    else
		    {
			for (i = 0; i < ((RectMeshPtr) curItem) -> xDim; ++i)
			{
			    for (j = 0; j < ((RectMeshPtr) curItem) -> yDim; ++j)
			    {
				offset = RECTMESHVERTEX((RectMeshPtr) curItem, i, j);

				vertices[offset] . colorIndex = index;
			    }
			}
			InterpRectCenters((RectMeshPtr) curItem);
		    }
		}
		return;
		break;
	    default:
		return;
	}

	/*Make it colored*/
	for (k = 0; k < nVertices; ++k)
	{
	    vertices[k] . colorIndex = index;
	}
}

#ifdef PROTO
void ColorItemsByObject(PicItemPtr curItem,ObjPtr colorObj, Bool interpolate)
#else
void ColorItemsByObject(curItem, colorObj, interpolate)
ObjPtr pic;
PicItemPtr curItem;
Bool interpolate;
#endif
/*Colors items curItem by colorObj*/
{
    ObjPtr repObj;
    ObjPtr colors;
    long k;

    MakeVar(colorObj, CPALETTE);
    colors = GetPaletteVar("ColorItemsByObject", colorObj, CPALETTE);
    if (!colors) return;
    SetPalette(colors);

    while (curItem)
    {
	ColorItemByObject(curItem, colorObj, interpolate);

	curItem = curItem -> next;
    }
}

#ifdef PROTO
void ColorItemsWithIndex(PicItemPtr curItem, int index)
#else
void ColorItemsWithIndex(curItem, index)
PicItemPtr curItem;
int index;
#endif
/*Colors items starting at curItem with index*/
{
    ObjPtr repObj;
    ObjPtr colors;
    long k;

    while (curItem)
    {
	ColorItemWithIndex(curItem, index);

	curItem = curItem -> next;
    }
}

#ifdef PROTO
void ColorPictureByObject(ObjPtr pic,ObjPtr colorObj, Bool interpolate)
#else
void ColorPictureByObject(pic, colorObj, interpolate)
ObjPtr pic;
ObjPtr colorObj;
Bool interpolate;
#endif
/*Colors picture pic by colorObj.  if interpolate, interpolates on lookup*/
{
    SetCurField(FIELD1, colorObj);
    SetCurForm(FIELD2, colorObj);

    ColorItemsByObject(((PicPtr) pic) -> items, colorObj, interpolate);
}

/*Shading models*/
int shadeModels[2][3][3] =
    {
	{{    FLAT,    FLAT, GOURAUD},
	 {    FLAT,    FLAT, GOURAUD},
	 { GOURAUD,    FLAT, GOURAUD}},
	{{    FLAT,    FLAT, GOURAUD},
	 {    FLAT,    FLAT, GOURAUD},
	 { GOURAUD, GOURAUD, GOURAUD}}
    };

#define PrepareToDrawC(item)						\
	if (item -> colorShading != curColorShading ||			\
	    item -> lightShading != curLightShading)			\
	{								\
	    curColorShading = item -> colorShading;			\
	    curLightShading = item -> lightShading;			\
	    shademodel(shadeModels[curColorMode][curLightShading][curColorShading]); \
	}								\
	if (curColorShading == NOCOLOR && curLightShading == NOLIGHT)	\
	{								\
	    color(curBeg + curNColors - 4);				\
	}
	
#define PrepareToDrawRGB(item)						\
	if (item -> colorShading != curColorShading ||			\
	    item -> lightShading != curLightShading)			\
	{								\
	    curColorShading = item -> colorShading;			\
	    curLightShading = item -> lightShading;			\
	    shademodel(shadeModels[curColorMode][curLightShading][curColorShading]); \
	    if (curLightShading == NOLIGHT)				\
	    {								\
		lmcolor(LMC_COLOR);					\
	    }								\
	    else							\
	    {								\
		lmcolor(LMC_DIFFUSE);					\
	    }								\
	}								\
	if (curColorShading == NOCOLOR && curLightShading == NOLIGHT)	\
	{								\
	     c3s(curColors[curNColors - 2]);				\
	}

/*Vertex drawing macros.  Hate to do it like this, but...

  The name is by this convention:

  PRNS(vertexPointer, nVertices)
  ^^^^
  ||||
  |||+- Color shading, either (N)one, (M)ono, or (S)mooth
  ||+-- Light shading, either (N)one, (M)ono, or (S)mooth
  |+--- Color mode, either (R)GB or (C)MAP
  +---- This is always a P, for prepare to draw vertices

  VRNS(vertex)
  ^^^^
  ||||
  |||+- Color shading, either (N)one, (M)ono, or (S)mooth
  ||+-- Light shading, either (N)one, (M)ono, or (S)mooth
  |+--- Color mode, either (R)GB or (C)MAP
  +---- This is always a V, for draw vertex

  They all have to be separate for efficiency's sake
*/

#define ZERO3(d) {(d)[0] = 0.0; (d)[1] = 0.0; (d)[2] = 0.0;}
#define COPY3(d, s) {(d)[0] = (s)[0]; (d)[1] = (s)[1]; (d)[2] = (s)[2];}
#define ADD3(d, s) {(d)[0] += (s)[0]; (d)[1] += (s)[1]; (d)[2] += (s)[2];}
#define DIV3SC(d, s) {(d)[0] /= (s); (d)[1] /= (s); (d)[2] /= (s);}
static float tC[3];
static long tCI;
static float tN[3];

/*Color map macros*/
#define PCNN(vP, nV)
#define VCNN(v) v3f(v -> position);

#define PCNM(vP, nV) {int k; VertexPtr v; v = vP; tCI = 0; for(k=0;k<nV;++k) {tCI+=v->colorIndex; ++v;} color(curBeg + (int) ((float)(tCI) / nV + 0.5));}
#define VCNM(v) v3f(v -> position);

#define PCNS(vP, nV)
#define VCNS(v) color(curBeg + v -> colorIndex); v3f(v -> position);


#define PCMN(vP, nV)
#define VCMN(v) color((int) (curBeg + 2 + (v -> normal[0] * 0.5 + 0.5) * (curNColors - 5))); v3f(v -> position);

#define PCMM PCNM
#define VCMM VCNM

#define PCMS PCNS
#define VCMS VCNS


#define PCSN PCMN
#define VCSN VCMN

#define PCSM PCMM
#define VCSM VCMM

#define PCSS PCMS
#define VCSS VCMS

/*RGB macros*/
#define PRNN(vP, nV)
#define VRNN(v) v3f(v -> position);

#define PRNM(vP, nV) {int k; VertexPtr v; v = vP; tCI = 0; for(k=0;k<nV;++k) {tCI+=v->colorIndex; ++v;} c3s(curColors[curBeg + (int) ((float)(tCI) / nV + 0.5)]);}
#define VRNM(v) v3f(v -> position);

#define PRNS(vP, nV)
#define VRNS(v) c3s(curColors[v -> colorIndex]); v3f(v -> position);


#define PRMN PRNN
#define VRMN(v) n3f(v -> normal); v3f(v -> position);

#define PRMM PRNM
#define VRMM(v) n3f(v -> normal); v3f(v -> position);

#define PRMS PRMM
#define VRMS(v) c3s(curColors[v -> colorIndex]); n3f(v -> normal); v3f(v -> position);

#define PRSN PRMN
#define VRSN VRMN

#define PRSM PRMM
#define VRSM VRMM

#define PRSS PRMM
#define VRSS VRMS



/*Traversal macros to make life a little easier*/

/*Polygon traversal macro.  Decomposes simply into triangles*/
#define TRAVERSEPOLY {							\
    register VertexTuple *v1, *v2;					\
    v1 = ((PolyPtr) curItem) -> vertices;				\
	    v2 = ((PolyPtr) curItem) -> vertices + 			\
	    (((PolyPtr) curItem) -> nVertices - 1);			\
    bgntmesh();								\
    P(v1, ((PolyPtr) curItem) -> nVertices);				\
									\
    while (v2 >= v1)							\
    {									\
	/*Emit v1 and increment*/					\
	V(v1);								\
	++v1;								\
	if (v2 >= v1)							\
	{								\
	    /*Emit v2 and decrement*/					\
	    V(v2);							\
	    --v2;							\
	}								\
    }									\
    endtmesh();}

/*Planar traversal macro*/
#define TRAVERSEPLANARPOLY {						\
    register VertexPtr v1;						\
    register int k;							\
    v1 = ((PolyPtr) curItem) -> vertices;				\
									\
    bgnpolygon();							\
    P(v1, ((PolyPtr) curItem) -> nVertices);				\
    for (k = 0; k < ((PolyPtr) curItem) -> nVertices; ++k)		\
    {									\
	/*Emit v1 and increment*/					\
	V(v1);								\
	++v1;								\
    }									\
    endpolygon();}



/*Rectangular mesh traversal macro*/
#define TRAVERSERECTMESH						\
    {									\
	register int i, j;						\
	int xDim, yDim;							\
	register VertexPtr vertexPtr;					\
	int nX, nY;							\
									\
	vertexPtr = ((RectMeshPtr) curItem) -> vertices;		\
	xDim = ((RectMeshPtr) curItem) -> xDim;				\
	yDim = ((RectMeshPtr) curItem) -> yDim;				\
	nY = 2;								\
	nX = (yDim * 2 - 1);						\
	for (i = 0; i < xDim - 1; ++i)					\
	{								\
	    bgntmesh();							\
	    P(vertexPtr, 2);						\
	    V(vertexPtr);						\
	    vertexPtr += nX;						\
	    V(vertexPtr);						\
		for (j = 1; j < yDim; ++j)				\
		{							\
		    vertexPtr += nY - nX;				\
		    V(vertexPtr);					\
		    vertexPtr += nX;					\
		    V(vertexPtr);					\
		}							\
	    vertexPtr -= nX - 1;					\
	    endtmesh();							\
	}								\
    }

/*Versions for rect mesh with center nodes*/
#define MCRPCNM(vP) color(curBeg + (vP + 1) -> colorIndex);
#define MCRPRNM(vP) c3s(curColors[(vP + 1) -> colorIndex]);

/*Monocolor Rectangular mesh traversal macro*/
#define TRAVERSEMCRECTMESH						\
    {									\
	register int i, j;						\
	int xDim, yDim;							\
	register VertexPtr vertexPtr;					\
	int nX, nY;							\
									\
	vertexPtr = ((RectMeshPtr) curItem) -> vertices;		\
	xDim = ((RectMeshPtr) curItem) -> xDim;				\
	yDim = ((RectMeshPtr) curItem) -> yDim;				\
	nY = 2;								\
	nX = (yDim * 2 - 1);						\
	for (i = 0; i < xDim - 1; ++i)					\
	{								\
	    for (j = 0; j < yDim - 1; ++j)				\
	    {								\
		bgntmesh();						\
		P(vertexPtr);						\
		V(vertexPtr);						\
		vertexPtr += nX;					\
		V(vertexPtr);						\
		vertexPtr += nY - nX;					\
		V(vertexPtr);						\
		vertexPtr += nX;					\
		V(vertexPtr);						\
		endtmesh();						\
		vertexPtr -= nX;					\
	    }								\
	    vertexPtr += nY - 1;  /*-1 for missing center*/		\
	}								\
    }

static void DrawWPolygon(curItem)
PicItemPtr curItem;
/*Draws a wire frame of a polygon*/
{
    register VertexPtr vertex;
    int k;

    bgnclosedline();
    vertex = ((PolyPtr) curItem) -> vertices;
    for (k = 0; k < ((PolyPtr) curItem) -> nVertices; ++k)
    {
	v3f(vertex -> position);
	++vertex;
    }
    endclosedline();
}

static void DrawCPolygon(curItem)
PicItemPtr curItem;
/*Draws a color map shaded polygon*/
{
#define TRAVERSE TRAVERSEPOLY
#include "ScianStdCDraw.h"
#undef TRAVERSE
}

static void DrawRGBPolygon(curItem)
PicItemPtr curItem;
{
#define TRAVERSE TRAVERSEPOLY
#include "ScianStdRGBDraw.h"
#undef TRAVERSE
}

static void DrawCPPolygon(curItem)
PicItemPtr curItem;
/*Draws a color map shaded planarpolygon*/
{
#define TRAVERSE TRAVERSEPLANARPOLY
#include "ScianStdCDraw.h"
#undef TRAVERSE
}

static void DrawRGBPPolygon(curItem)
PicItemPtr curItem;
/*Draws an RGB shaded planar polygon*/
{
#define TRAVERSE TRAVERSEPLANARPOLY
#include "ScianStdRGBDraw.h"
#undef TRAVERSE
}

static void DrawSRectMesh(curItem)
PicItemPtr curItem;
/*Draws a skeleton of a rectangular mesh*/
{
    /*Draw a wire mesh*/
    register int i, j;
    int xDim, yDim;
    register VertexPtr vertexPtr;
    int nX, nY;

    vertexPtr = ((RectMeshPtr) curItem) -> vertices;
    xDim = ((RectMeshPtr) curItem) -> xDim;
    yDim = ((RectMeshPtr) curItem) -> yDim;
    nX = (2 * yDim - 1);
    nY = 2;

    if (xDim + yDim < 200)
    {
    /*Draw the fast moving axes*/
    for (i = 0; i < xDim; ++i)
    {
	vertexPtr = ((RectMeshPtr) curItem) -> vertices +
		RECTMESHVERTEX(((RectMeshPtr) curItem), i, 0);
	bgnline();
	for (j = 0; j < yDim; ++j)
	{
	    v3f(vertexPtr -> position);
	    vertexPtr += nY;
	}
	endline();
    }

    /*Draw the slow moving axes*/
    for (j = 0; j < yDim; ++j)
    {
	vertexPtr = ((RectMeshPtr) curItem) -> vertices +
			RECTMESHVERTEX(((RectMeshPtr) curItem), 0, j);
	bgnline();
	for (i = 0; i < xDim; ++i)
	{
	    v3f(vertexPtr -> position);
	    vertexPtr += nX;
	}
	endline();
    }
    }
    else
    {
	/*Just draw the outline*/
	i = 0;
	vertexPtr = ((RectMeshPtr) curItem) -> vertices +
		RECTMESHVERTEX(((RectMeshPtr) curItem), i, 0);
	bgnline();
	for (j = 0; j < yDim; ++j)
	{
	    v3f(vertexPtr -> position);
	    vertexPtr += nY;
	}
	endline();

	i = xDim - 1;
	vertexPtr = ((RectMeshPtr) curItem) -> vertices +
		RECTMESHVERTEX(((RectMeshPtr) curItem), i, 0);
	bgnline();
	for (j = 0; j < yDim; ++j)
	{
	    v3f(vertexPtr -> position);
	    vertexPtr += nY;
	}
	endline();

	j = 0;
	vertexPtr = ((RectMeshPtr) curItem) -> vertices +
			RECTMESHVERTEX(((RectMeshPtr) curItem), 0, j);
	bgnline();
	for (i = 0; i < xDim; ++i)
	{
	    v3f(vertexPtr -> position);
	    vertexPtr += nX;
	}
	endline();

	j = yDim - 1;
	vertexPtr = ((RectMeshPtr) curItem) -> vertices +
			RECTMESHVERTEX(((RectMeshPtr) curItem), 0, j);
	bgnline();
	for (i = 0; i < xDim; ++i)
	{
	    v3f(vertexPtr -> position);
	    vertexPtr += nX;
	}
	endline();
    }
}

static void DrawWRectMesh(curItem)
PicItemPtr curItem;
/*Draws a wire frame of a rectangular mesh*/
{
    /*Draw a wire mesh*/
    register int i, j;
    int xDim, yDim;
    register VertexPtr vertexPtr;
    int nX, nY;

    vertexPtr = ((RectMeshPtr) curItem) -> vertices;
    xDim = ((RectMeshPtr) curItem) -> xDim;
    yDim = ((RectMeshPtr) curItem) -> yDim;
    nX = (2 * yDim - 1);
    nY = 2;

    /*Draw the fast moving axes*/
    for (i = 0; i < xDim; ++i)
    {
	vertexPtr = ((RectMeshPtr) curItem) -> vertices +
		RECTMESHVERTEX(((RectMeshPtr) curItem), i, 0);
	bgnline();
	for (j = 0; j < yDim; ++j)
	{
	    v3f(vertexPtr -> position);
	    vertexPtr += nY;
	}
	endline();
    }

    /*Draw the slow moving axes*/
    for (j = 0; j < yDim; ++j)
    {
	vertexPtr = ((RectMeshPtr) curItem) -> vertices +
			RECTMESHVERTEX(((RectMeshPtr) curItem), 0, j);
	bgnline();
	for (i = 0; i < xDim; ++i)
	{
	    v3f(vertexPtr -> position);
	    vertexPtr += nX;
	}
	endline();
    }
}

static void DrawCRectMesh(curItem)
PicItemPtr curItem;
/*Draws a rectangular mesh in color map mode*/
{
	switch (curLightShading)
	{
	    case NOLIGHT:
		switch (curColorShading)
		{
		    case NOCOLOR:
#define P PCNN
#define V VCNN
			TRAVERSERECTMESH;
#undef P
#undef V
			break;
		    case MONOCOLOR:
#define P MCRPCNM
#define V VCNM
			TRAVERSEMCRECTMESH;
#undef P
#undef V
			break;
		    case SMOOTHCOLOR:
#define P PCNS
#define V VCNS
			TRAVERSERECTMESH;
#undef P
#undef V
			break;
		}
		break;
	    case MONOLIGHT:
		switch (curColorShading)
		{
		    case NOCOLOR:
#define P PCMN
#define V VCMN
			TRAVERSERECTMESH;
#undef P
#undef V
			break;
		    case MONOCOLOR:
#define P MCRPCNM
#define V VCMM
			TRAVERSEMCRECTMESH;
#undef P
#undef V
			break;
		    case SMOOTHCOLOR:
#define P PCMS
#define V VCMS
			TRAVERSERECTMESH;
#undef P
#undef V
			break;
		}
		break;
	    case SMOOTHLIGHT:
		switch (curColorShading)
		{
		    case NOCOLOR:
#define P PCSN
#define V VCSN
			TRAVERSERECTMESH;
#undef P
#undef V
			break;
		    case MONOCOLOR:
#define P MCRPCNM
#define V VCSM
			TRAVERSEMCRECTMESH;
#undef P
#undef V
			break;
		    case SMOOTHCOLOR:
#define P PCSS
#define V VCSS
			TRAVERSERECTMESH;
#undef P
#undef V
			break;
		}
		break;
	}
}

int yaga = 0;

static void DrawRGBRectMesh(curItem)
PicItemPtr curItem;
/*Draws a rectangular mesh in RGB mode*/
{
	switch (curLightShading)
	{
	    case NOLIGHT:
		switch (curColorShading)
		{
		    case NOCOLOR:
#define P PRNN
#define V VRNN
			TRAVERSERECTMESH;
#undef P
#undef V
			break;
		    case MONOCOLOR:
#define P MCRPRNM
#define V VRNM
			TRAVERSEMCRECTMESH;
#undef P
#undef V
			break;
		    case SMOOTHCOLOR:
#define P PRNS
#define V VRNS
			TRAVERSERECTMESH;
#undef P
#undef V
			break;
		}
		break;
	    case MONOLIGHT:
		switch (curColorShading)
		{
		    case NOCOLOR:
#define P PRMN
#define V VRMN
			TRAVERSERECTMESH;
#undef P
#undef V
			break;
		    case MONOCOLOR:
#define P MCRPRNM
#define V VRMM
			TRAVERSEMCRECTMESH;
#undef P
#undef V
			break;
		    case SMOOTHCOLOR:
#define P PRMS
#define V VRMS
			TRAVERSERECTMESH;
#undef P
#undef V
			break;
		}
		break;
	    case SMOOTHLIGHT:
		switch (curColorShading)
		{
		    case NOCOLOR:
#define P PRSN
#define V VRSN
			TRAVERSERECTMESH;
#undef P
#undef V
			break;
		    case MONOCOLOR:
#define P MCRPRNM
#define V VRSM
			TRAVERSEMCRECTMESH;
#undef P
#undef V
			break;
		    case SMOOTHCOLOR:
#define P PRSS
#define V VRSS
			TRAVERSERECTMESH;
#undef P
#undef V
			break;
		}
		break;
	}
}

void DrawCSphere(curItem)
PicItemPtr curItem;
/*Draws a sphere in cmap mode*/
{
			{
			    Sphere curSphere;
			    float vx[3], vy[3], vz[3]; /*Vertices stuck out on those axes*/
			    
			    curSphere = *((SpherePtr) curItem);
	
	        	    if (!curIsTransparent)
	        	    {
	       			backface(TRUE);
	        	    }
			    /*Draw the eight quadrants of the sphere*/
			    /* +x +y +z */
			    vx[0] = 1.0;
			    vx[1] = 0.0;
			    vx[2] = 0.0;
			    vy[0] = 0.0;
			    vy[1] = 1.0;
			    vy[2] = 0.0;
			    vz[0] = 0.0;
			    vz[1] = 0.0;
			    vz[2] = 1.0;
	
			    curRadius = curSphere. radius;
			    curCenter[0] = curSphere . center[0];
			    curCenter[1] = curSphere . center[1];
			    curCenter[2] = curSphere . center[2];
			    DrawQuadrant(vx, vy, vz, 0);
	
			    /* +y -x +z*/
			    vx[0] = -1.0;
			    DrawQuadrant(vy, vx, vz, 0);
	
			    /* -x -y -z*/
			    vy[1] = -1.0;
			    DrawQuadrant(vx, vy, vz, 0);
	
			    /* -y +x +z*/
			    vx[0] = 1.0;
			    DrawQuadrant(vy, vx, vz, 0);
	
			    /* +x -y -z*/
			    vz[2] = -1.0;
			    DrawQuadrant(vx, vy, vz, 0);
	
			    /* +y +x -z*/
			    vy[1] = 1.0;
			    DrawQuadrant(vy, vx, vz, 0);
	
			    /* -x +y -z*/
			    vx[0] = -1.0;
			    DrawQuadrant(vx, vy, vz, 0);
	
			    /* -y, -x, -z*/
			    vy[1] = -1.0;
			    DrawQuadrant(vy, vx, vz, 0);
	
	        	    if (!curIsTransparent)
	        	    {
	       			backface(FALSE);
	        	    }
			}
}

void DrawSFrustum(curItem)
PicItemPtr curItem;
/*Draws a skeletal frustum*/
{
    bgnline();
    v3f(((FrustumPtr) curItem) -> end1);
    v3f(((FrustumPtr) curItem) -> end2);
    endline();
}

void DrawSSphere(curItem)
PicItemPtr curItem;
/*Draws a skeletal frustum*/
{
    pnt(((SpherePtr) curItem) -> center[0],
	((SpherePtr) curItem) -> center[1],
	((SpherePtr) curItem) -> center[2]);
}

void DrawCFrustum(curItem)
PicItemPtr curItem;
/*Draws a color map frustum*/
{
    Frustum curFrustum;
    float a[3];		/*Axial unit vector*/
    float r[CYLSIDES][3];
    double length;	/*Length of cylinder*/
    double nf;		/*Normalization factor*/
			/*Radial units around cylinder*/
    int end;		/*Counter for which end*/
    int b;		/*Counter around radius*/
    int d;		/*Delta for subdivision*/
    int k;		/*Random counter*/

    if (!curIsTransparent)
    {
	backface(TRUE);
    }

    /*Copy the frustum into curFrustum for speed*/
    curFrustum = *((FrustumPtr) curItem);

    /*First make an axial unit vector*/
    a[0] = curFrustum . end2[0] -
	   curFrustum . end1[0];
    a[1] = curFrustum . end2[1] -
	   curFrustum . end1[1];
    a[2] = curFrustum . end2[2] -
	   curFrustum . end1[2];
    length = sqrt(a[0]*a[0] + a[1]*a[1] + a[2]*a[2]);
    nf = 1.0 / length;
    a[0] *= nf; a[1] *= nf; a[2] *= nf;

    /*See if it's nearly colinear with i*/
    if (ABS(a[0]) > 0.8)
    {
	/*It is, so cross by j to get first r*/
	r[0][0] = -a[2];
	r[0][1] = 0.0;
	r[0][2] = a[0];
    }
    else
    {
	/*It isn't, so cross by i to get first r*/
	r[0][0] = 0.0;
	r[0][1] = a[2];
	r[0][2] = -a[1];
    }
    
    /*Normalize r*/
    length = sqrt(r[0][0]*r[0][0] + r[0][1]*r[0][1] + r[0][2]*r[0][2]);
    nf = 1.0 / length;
    r[0][0] *= nf; r[0][1] *= nf, r[0][2] *= nf;

    /*Cross a with first radial unit to get orthogonal unit*/
    CROSS(a, r[0], r[CYLSIDES / 4]);

    /*Fill out point radii 3 and 4 by reversing 1 and 2*/
    r[CYLSIDES / 2][0] = -r[0][0];
    r[CYLSIDES / 2][1] = -r[0][1];
    r[CYLSIDES / 2][2] = -r[0][2];

    r[CYLSIDES / 4 * 3][0] = -r[CYLSIDES / 4][0];
    r[CYLSIDES / 4 * 3][1] = -r[CYLSIDES / 4][1];
    r[CYLSIDES / 4 * 3][2] = -r[CYLSIDES / 4][2];

    /*Subdivide the sides of the cylinder*/
    k = 0;
    for (d = CYLSIDES / 4; d > 1; d /= 2)
    {
	register double cef_2;	/*Chord extension factor/2*/
	
	cef_2 = cefs_2[k];
	++k;
	
	for (b = 0; b < CYLSIDES; b += d)
	{
	    register int m, e;	/*Midpoint and endpoint*/

	    /*Interpolate center of chord*/
	    e = (b + d) & (CYLSIDES - 1);	/*Clip circular*/
	    m = b + d / 2;
	    r[m][0] = (r[b][0] + r[e][0]) * cef_2;
	    r[m][1] = (r[b][1] + r[e][1]) * cef_2;
	    r[m][2] = (r[b][2] + r[e][2]) * cef_2;
	}
    }

    /*Draw the cylinder*/
    
#ifndef PSYCHEDELIC
    if (rgbp == false)
    {
	float v[3];
	
	bgntmesh();
	SetRealColor(r[CYLSIDES - 1][0]);
	v[0] = curFrustum . rad2 * r[CYLSIDES - 1][0] + 
	       curFrustum . end2[0];
	v[1] = curFrustum . rad2 * r[CYLSIDES - 1][1] + 
	       curFrustum . end2[1];
	v[2] = curFrustum . rad2 * r[CYLSIDES - 1][2] + 
	       curFrustum . end2[2];
	v3f(v);
	v[0] = curFrustum . rad1 * r[CYLSIDES - 1][0] + 
	       curFrustum . end1[0];
	v[1] = curFrustum . rad1 * r[CYLSIDES - 1][1] + 
	       curFrustum . end1[1];
	v[2] = curFrustum . rad1 * r[CYLSIDES - 1][2] + 
	       curFrustum . end1[2];
	v3f(v);

	for (k = 0; k < CYLSIDES; ++k)
	{
	    SetRealColor(r[k][0]);
	    v[0] = curFrustum . rad2 * r[k][0] + 
		   curFrustum . end2[0];
	    v[1] = curFrustum . rad2 * r[k][1] + 
		   curFrustum . end2[1];
	    v[2] = curFrustum . rad2 * r[k][2] + 
		   curFrustum . end2[2];
	    v3f(v);
	    v[0] = curFrustum . rad1 * r[k][0] + 
		   curFrustum . end1[0];
	    v[1] = curFrustum . rad1 * r[k][1] + 
		   curFrustum . end1[1];
	    v[2] = curFrustum . rad1 * r[k][2] + 
		   curFrustum . end1[2];
	    v3f(v);
	}
	endtmesh();

	/*Do top cap*/
	SetRealColor(a[0]);
	bgnpolygon();
	for (k = 0; k < CYLSIDES; ++k)
	{
	    v[0] = curFrustum . rad2 * r[k][0] + 
		   curFrustum . end2[0];
	    v[1] = curFrustum . rad2 * r[k][1] + 
		   curFrustum . end2[1];
	    v[2] = curFrustum . rad2 * r[k][2] + 
		   curFrustum . end2[2];
	    v3f(v);
	}
	endpolygon();

	/*Do bottom cap*/
	SetRealColor(-a[0]);
	bgnpolygon();
	for (k = CYLSIDES - 1; k >= 0; --k)
	{
	    v[0] = curFrustum . rad1 * r[k][0] + 
		   curFrustum . end1[0];
	    v[1] = curFrustum . rad1 * r[k][1] + 
		   curFrustum . end1[1];
	    v[2] = curFrustum . rad1 * r[k][2] + 
		   curFrustum . end1[2];
	    v3f(v);
	}
	endpolygon();
    }
    else
#endif
    {
	float v[3];

	bgntmesh();
	n3f(r[CYLSIDES - 1]);
	v[0] = curFrustum . rad2 * r[CYLSIDES - 1][0] + 
	       curFrustum . end2[0];
	v[1] = curFrustum . rad2 * r[CYLSIDES - 1][1] + 
	       curFrustum . end2[1];
	v[2] = curFrustum . rad2 * r[CYLSIDES - 1][2] + 
	       curFrustum . end2[2];
	v3f(v);
	v[0] = curFrustum . rad1 * r[CYLSIDES - 1][0] + 
	       curFrustum . end1[0];
	v[1] = curFrustum . rad1 * r[CYLSIDES - 1][1] + 
	       curFrustum . end1[1];
	v[2] = curFrustum . rad1 * r[CYLSIDES - 1][2] + 
	       curFrustum . end1[2];
	v3f(v);

	for (k = 0; k < CYLSIDES; ++k)
	{
	    n3f(r[k]);
	    v[0] = curFrustum . rad2 * r[k][0] + 
		   curFrustum . end2[0];
	    v[1] = curFrustum . rad2 * r[k][1] + 
		   curFrustum . end2[1];
	    v[2] = curFrustum . rad2 * r[k][2] + 
		   curFrustum . end2[2];
	    v3f(v);
	    v[0] = curFrustum . rad1 * r[k][0] + 
		   curFrustum . end1[0];
	    v[1] = curFrustum . rad1 * r[k][1] + 
		   curFrustum . end1[1];
	    v[2] = curFrustum . rad1 * r[k][2] + 
		   curFrustum . end1[2];
	    v3f(v);
	}
	endtmesh();

	/*Do top cap*/
	n3f(a);
	bgnpolygon();
	for (k = 0; k < CYLSIDES; ++k)
	{
	    v[0] = curFrustum . rad2 * r[k][0] + 
		   curFrustum . end2[0];
	    v[1] = curFrustum . rad2 * r[k][1] + 
		   curFrustum . end2[1];
	    v[2] = curFrustum . rad2 * r[k][2] + 
		   curFrustum . end2[2];
	    v3f(v);
	}
	endpolygon();

	/*Do bottom cap*/
	a[0] = -a[0];
	a[1] = -a[1];
	a[2] = -a[2];
	n3f(a);
	bgnpolygon();
	for (k = CYLSIDES - 1; k >= 0; --k)
	{
	    v[0] = curFrustum . rad1 * r[k][0] + 
		   curFrustum . end1[0];
	    v[1] = curFrustum . rad1 * r[k][1] + 
		   curFrustum . end1[1];
	    v[2] = curFrustum . rad1 * r[k][2] + 
		   curFrustum . end1[2];
	    v3f(v);
	}
	endpolygon();
    }
    if (!curIsTransparent)
    {
	backface(FALSE);
    }
}

void DrawCPolygons(item)
PicItemPtr item;
/*Draws a bunch of polygons in cmode*/
{
    PolyPtr curItem;
int nPolys;
nPolys = 0;
    curItem = ((PolysPtr) item) -> polygons;
    while (curItem)
    {
	DrawCPolygon(curItem);
	curItem = (PolyPtr) curItem -> item . next;
	++nPolys;
    }
}

void DrawRGBPolygons(item)
PicItemPtr item;
/*Draws a bunch of polygons in RGB mode*/
{
    PolyPtr curItem;

    if (!curIsTransparent && ((PolysPtr) item) -> enclosed)	
    {
	backface(TRUE);
    }
    curItem = ((PolysPtr) item) -> polygons;
    while (curItem)
    {
	DrawRGBPolygon(curItem);
	curItem = (PolyPtr) curItem -> item . next;
    }
    if (!curIsTransparent && ((PolysPtr) item) -> enclosed)	
    {
	backface(FALSE);
    }
}

void DrawWPolygons(item)
PicItemPtr item;
/*Draws a bunch of polygons as wire frames*/
{
    PolyPtr curItem;
    curItem = ((PolysPtr) item) -> polygons;
    while (curItem)
    {
	DrawWPolygon(curItem);
	curItem = (PolyPtr) curItem -> item . next;
    }
}

static void DrawSPolygon(curItem)
PicItemPtr curItem;
/*Draws a point of a polygon*/
{
    register VertexPtr vertex;
    int k;

    vertex = ((PolyPtr) curItem) -> vertices;
    pnt(vertex -> position[0], vertex -> position[1], vertex -> position[2]);
}

void DrawSPolygons(item)
PicItemPtr item;
/*Draws a bunch of polygons as skeleton*/
{
    PolyPtr curItem;
    curItem = ((PolysPtr) item) -> polygons;
    while (curItem)
    {
	DrawSPolygon(curItem);
	curItem = (PolyPtr) curItem -> item . next;
    }
}

typedef void (* DrawFunc)();

DrawFunc drawFuncs[NPICITEMTYPES][DQ_NQUALITIES + 1] =
    {
	{DrawWPolygon,  DrawWPolygon,  DrawCPolygon,  DrawRGBPolygon},
	{DrawSFrustum,  DrawSFrustum,  DrawCFrustum,  DrawCFrustum},
	{DrawSSphere,   DrawCSphere,   DrawCSphere,   DrawCSphere},
	{DrawSRectMesh, DrawWRectMesh, DrawCRectMesh, DrawRGBRectMesh},
	{DrawCPPolygon, DrawCPPolygon, DrawCPPolygon, DrawRGBPPolygon},
	{DrawSPolygons, DrawWPolygons, DrawCPolygons, DrawRGBPolygons}
    };


#ifdef PROTO
void DrawPicture(ObjPtr pic, Bool isTransparent, ObjPtr colors)
#else
void DrawPicture(pic, isTransparent, colors)
ObjPtr pic;
Bool isTransparent;
ObjPtr colors;
#endif
/*Draws picture pic.  isTransparent if transparent*/
{
    PicItemPtr curItem;
    ObjPtr repObj;
    DrawFunc drawFunc;
    int qualIndex;
    int nPictures;
    nPictures = 0;

    /*Make the picture colored*/
    MakeVar(pic, PICCOLORED);

    curIsTransparent = isTransparent;
    SetPalette(colors);

    /*Set the color mode*/
    if (rgbp == false)
    {
	curColorMode = CMODECMAP;
    }
    else
    {
	curColorMode = CMODERGB;
    }

    /*Set the current color and light shading to silly values*/
    curColorShading = curLightShading = -1;

    /*Calculate the quality index*/
    qualIndex = drawingQuality;
    if (qualIndex == DQ_FULL && curColorMode == CMODERGB) ++qualIndex;

    if (curSpace)
    {
	ObjPtr panel, color, var;
	Bool hasBackground;
	real rgb[3];

	panel = GetVar(curSpace, BACKPANEL);
	if (panel)
	{
	    var = GetVar(panel, BACKGROUND);
	    if (var && IsArray(var) && RANK(var) == 1 && DIMS(var)[0] == 3)
	    {
		Array2CArray(rgb, var);
		hasBackground = true;
	    }
	    else if (var && IsInt(var))
	    {
		rgb[0] = uiColors[GetInt(var)][0] / 255.0;
		rgb[1] = uiColors[GetInt(var)][1] / 255.0;
		rgb[2] = uiColors[GetInt(var)][2] / 255.0;
		hasBackground = true;
	    }
	    else
	    {
		rgb[0] = rgb[1] = rgb[2] = 0.0;
		hasBackground = false;
	    }
	}
	if (rgb[0] + rgb[1] + rgb[2] > 1.5)
	{
	    SetUIColor(UIBLACK);
	}
	else
	{
	    SetUIColor(UIWHITE);
	}
    }

    curItem = ((PicPtr) pic) -> items;

    switch (curColorMode)
    {
	case CMODECMAP:
	    while (curItem)
	    {
		PrepareToDrawC(curItem);
		drawFunc = drawFuncs[curItem -> type][qualIndex];
		(*drawFunc)(curItem);
		curItem = curItem -> next;
		++nPictures;
	    }
	    break;
	case CMODERGB:
	    while (curItem)
	    {
		PrepareToDrawRGB(curItem);
		drawFunc = drawFuncs[curItem -> type][qualIndex];
		(*drawFunc)(curItem);
		curItem = curItem -> next;
	    }
	    break;
    }
}

static PolysPtr curPolys;

void QuadrantPolys(r1, r2, r3, level)
float r1[3];
float r2[3];
float r3[3];
int level;
/*Emits a quadrant of sphere at center curCenter with radius curRadius given 
  radial normal vectors r1, r2, and r3, using current color table, at recursion 
  level level into curPolys*/
{
    if (level >= curSub)
    {
	/*Maximum number of subdivisions has been reached*/
	VertexTuple v[3];
	v[2] . position[0] = curCenter[0] + r1[0] * curRadius;
	v[2] . position[1] = curCenter[1] + r1[1] * curRadius;
	v[2] . position[2] = curCenter[2] + r1[2] * curRadius;
	v[1] . position[0] = curCenter[0] + r2[0] * curRadius;
	v[1] . position[1] = curCenter[1] + r2[1] * curRadius;
	v[1] . position[2] = curCenter[2] + r2[2] * curRadius;
	v[0] . position[0] = curCenter[0] + r3[0] * curRadius;
	v[0] . position[1] = curCenter[1] + r3[1] * curRadius;
	v[0] . position[2] = curCenter[2] + r3[2] * curRadius;

	v[2] . normal[0] = r1[0];
	v[2] . normal[1] = r1[1];
	v[2] . normal[2] = r1[2];
	v[1] . normal[0] = r2[0];
	v[1] . normal[1] = r2[1];
	v[1] . normal[2] = r2[2];
	v[0] . normal[0] = r3[0];
	v[0] . normal[1] = r3[1];
	v[0] . normal[2] = r3[2];

	AppendPolyToPolys(curPolys, 3, v);
    }
    else
    {
	/*Do another subdivision*/
	float r12[3], r23[3], r31[3];	/*Inner triangle subdivisions*/
	r12[0] = (r1[0] + r2[0]);
	r12[1] = (r1[1] + r2[1]);
	r12[2] = (r1[2] + r2[2]);
	r23[0] = (r2[0] + r3[0]);
	r23[1] = (r2[1] + r3[1]);
	r23[2] = (r2[2] + r3[2]);
	r31[0] = (r3[0] + r1[0]);
	r31[1] = (r3[1] + r1[1]);
	r31[2] = (r3[2] + r1[2]);
	NORMALIZE(r12);
	NORMALIZE(r23);
	NORMALIZE(r31);

	/*Draw the subdivisions*/
	QuadrantPolys(r1, r12, r31, level + 1);
	QuadrantPolys(r12, r2, r23, level + 1);
	QuadrantPolys(r23, r3, r31, level + 1);
	QuadrantPolys(r12, r23, r31, level + 1);
    }
}

PolysPtr SphereToPolys(sphere, nSubdivisions)
PicItemPtr sphere;
int nSubdivisions;
/*Returns a polys object of sphere*/
{
    Sphere curSphere;
    float vx[3], vy[3], vz[3]; /*Vertices stuck out on those axes*/
	
    curSub = nSubdivisions;	    
    curSphere = *((SpherePtr) sphere);
    curPolys = NewPolygons();
    curPolys -> enclosed = true;

    if (!curIsTransparent)
    {
	backface(TRUE);
    }
    /*Make the eight quadrants of the sphere*/
    /* +x +y +z */
    vx[0] = 1.0;
    vx[1] = 0.0;
    vx[2] = 0.0;
    vy[0] = 0.0;
    vy[1] = 1.0;
    vy[2] = 0.0;
    vz[0] = 0.0;
    vz[1] = 0.0;
    vz[2] = 1.0;
	
    curRadius = curSphere. radius;
    curCenter[0] = curSphere . center[0];
    curCenter[1] = curSphere . center[1];
    curCenter[2] = curSphere . center[2];
    QuadrantPolys(vx, vy, vz, 0);
	
    /* +y -x +z*/
    vx[0] = -1.0;
    QuadrantPolys(vy, vx, vz, 0);
	
    /* -x -y -z*/
    vy[1] = -1.0;
    QuadrantPolys(vx, vy, vz, 0);
	
    /* -y +x +z*/
    vx[0] = 1.0;
    QuadrantPolys(vy, vx, vz, 0);
	
    /* +x -y -z*/
    vz[2] = -1.0;
    QuadrantPolys(vx, vy, vz, 0);
	
    /* +y +x -z*/
    vy[1] = 1.0;
    QuadrantPolys(vy, vx, vz, 0);
	
    /* -x +y -z*/
    vx[0] = -1.0;
    QuadrantPolys(vx, vy, vz, 0);
	
    /* -y, -x, -z*/
    vy[1] = -1.0;
    QuadrantPolys(vy, vx, vz, 0);

    return curPolys;
}

#ifdef PROTO
void ConvertFrustumOntoPicture(ObjPtr picture, float end1[3], float rad1,
		float end2[3], float rad2, int nSubdivisions, Bool capEnds)
#else
void ConvertFrustumOntoPicture(picture, end1, rad1, end2, rad2, nSubdivisions, capEnds)
float end1[3], end2[3];
float rad1, rad2;
ObjPtr picture;
int nSubdivisions;
Bool capEnds;
#endif
/*Converts a frustum onto a picture*/
{
    float a[3];		/*Axial unit vector*/
    float r[MAXCYLSIDES][3];
    VertexTuple v[MAXCYLSIDES + 1];
    double length;	/*Length of cylinder*/
    double nf;		/*Normalization factor*/
			/*Radial units around cylinder*/
    int end;		/*Counter for which end*/
    int b;		/*Counter around radius*/
    int d;		/*Delta for subdivision*/
    int k;		/*Random counter*/
    RectMeshPtr rectMesh;

    curNSides = 4;
    while (nSubdivisions--) curNSides *= 2;

    /*First make an axial unit vector*/
    a[0] = end2[0] -
	   end1[0];
    a[1] = end2[1] -
	   end1[1];
    a[2] = end2[2] -
	   end1[2];
    length = sqrt(a[0]*a[0] + a[1]*a[1] + a[2]*a[2]);
    nf = 1.0 / length;
    a[0] *= nf; a[1] *= nf; a[2] *= nf;

    /*See if it's nearly colinear with i*/
    if (ABS(a[0]) > 0.8)
    {
	/*It is, so cross by j to get first r*/
	r[0][0] = -a[2];
	r[0][1] = 0.0;
	r[0][2] = a[0];
    }
    else
    {
	/*It isn't, so cross by i to get first r*/
	r[0][0] = 0.0;
	r[0][1] = a[2];
	r[0][2] = -a[1];
    }
    
    /*Normalize r*/
    length = sqrt(r[0][0]*r[0][0] + r[0][1]*r[0][1] + r[0][2]*r[0][2]);
    nf = 1.0 / length;
    r[0][0] *= nf; r[0][1] *= nf, r[0][2] *= nf;

    /*Cross a with first radial unit to get orthogonal unit*/
    CROSS(a, r[0], r[curNSides / 4]);

    /*Fill out point radii 3 and 4 by reversing 1 and 2*/
    r[curNSides / 2][0] = -r[0][0];
    r[curNSides / 2][1] = -r[0][1];
    r[curNSides / 2][2] = -r[0][2];

    r[curNSides / 4 * 3][0] = -r[curNSides / 4][0];
    r[curNSides / 4 * 3][1] = -r[curNSides / 4][1];
    r[curNSides / 4 * 3][2] = -r[curNSides / 4][2];

    /*Subdivide the sides of the cylinder*/
    k = 0;
    for (d = curNSides / 4; d > 1; d /= 2)
    {
	register double cef_2;	/*Chord extension factor/2*/
	
	cef_2 = cefs_2[k];
	++k;
	
	for (b = 0; b < curNSides; b += d)
	{
	    register int m, e;	/*Midpoint and endpoint*/

	    /*Interpolate center of chord*/
	    e = (b + d) & (curNSides - 1);	/*Clip circular*/
	    m = b + d / 2;
	    r[m][0] = (r[b][0] + r[e][0]) * cef_2;
	    r[m][1] = (r[b][1] + r[e][1]) * cef_2;
	    r[m][2] = (r[b][2] + r[e][2]) * cef_2;
	}
    }

    /*Convert the cylinder*/
    rectMesh = NewRectMesh(2, curNSides + 1, false);
    SetRectMeshVPosition(rectMesh, 0, 0,
	rad2 * r[curNSides - 1][0] + end2[0],
	rad2 * r[curNSides - 1][1] + end2[1],
	rad2 * r[curNSides - 1][2] + end2[2]);
    SetRectMeshVPosition(rectMesh, 1, 0,
	rad1 * r[curNSides - 1][0] + end1[0],
	rad1 * r[curNSides - 1][1] + end1[1],
	rad1 * r[curNSides - 1][2] + end1[2]);

    for (k = 0; k < curNSides; ++k)
    {
	SetRectMeshVPosition(rectMesh, 0, k + 1,
	    rad2 * r[k][0] + end2[0],
	    rad2 * r[k][1] + end2[1],
	    rad2 * r[k][2] + end2[2]);
	SetRectMeshVPosition(rectMesh, 1, k + 1,
	    rad1 * r[k][0] + end1[0],
	    rad1 * r[k][1] + end1[1],
	    rad1 * r[k][2] + end1[2]);
    }

    InterpRectCenters(rectMesh);

    CalcRectNormals(rectMesh);

    AppendItemToPicture((PicPtr) picture, (PicItemPtr) rectMesh);

    if (capEnds)
    {
    /*Do top cap*/
    for (k = 0; k < curNSides; ++k)
    {
	v[k] . position[0] = rad2 * r[k][0] + 
		   end2[0];
	v[k] . position[1] = rad2 * r[k][1] + 
		   end2[1];
	v[k] . position[2] = rad2 * r[k][2] + 
		   end2[2];
	v[k] . normal[0] = a[0];
	v[k] . normal[1] = a[1];
	v[k] . normal[2] = a[2];
    }
    AppendPolyToPicture((ObjPtr) picture, curNSides, v);

    /*Do bottom cap*/
    for (k = 0; k < curNSides; ++k)
    {
	v[k] . position[0] = rad1 * r[k][0] + 
		   end1[0];
	v[k] . position[1] = rad1 * r[k][1] + 
		   end1[1];
	v[k] . position[2] = rad1 * r[k][2] + 
		   end1[2];
	v[k] . normal[0] = -a[0];
	v[k] . normal[1] = -a[1];
	v[k] . normal[2] = -a[2];
    }
    AppendPolyToPicture((ObjPtr) picture, curNSides, v);
    }
}

#ifdef PROTO
PicItemPtr ColorItemsByItems(PicItemPtr destItem, ObjPtr owner, PicItemPtr curItem)
#else
PicItemPtr ColorItemsByItems(dest, owner, curItem)
ObjPtr dest, pic;
ObjPtr owner;
PicItemPtr curItem;
#endif
/*Colors a picture with another picture, which must be the source picture*/
{
    ObjPtr var;
    Bool capEnds;

    capEnds = GetPredicate(owner, CAPENDSP);

    while (curItem)
    {
	if (!destItem)
	{
	    ReportError("ColorItemsByItems", "Pictures don't match");
	}
	switch (curItem -> type)
	{
	    case POLYGON:
	    case PLANARPOLYGON:

		ColorItemWithIndex(destItem, 
			((PolyPtr) curItem) -> vertices[0] . colorIndex);
		break;
	    case FRUSTUM:

		ColorItemWithIndex(destItem,
			((FrustumPtr) curItem) -> colorIndex);
		if (capEnds)
		{
		    /*Cap the ends*/
		    destItem = destItem -> next;
		    if (!destItem) break;
		    ColorItemWithIndex(destItem,
			((FrustumPtr) curItem) -> colorIndex);
		    destItem = destItem -> next;
		    if (!destItem) break;
		    ColorItemWithIndex(destItem,
			((FrustumPtr) curItem) -> colorIndex);
		}
		break;
	    case SPHERE:

		ColorItemWithIndex(destItem,
			((SpherePtr) curItem) -> colorIndex);
		break;
	    case POLYGONS:

		{
		    PolyPtr polygons;
		    polygons = ((PolysPtr) curItem) -> polygons;
		    if (polygons)
		    {
			ColorItemWithIndex((PicItemPtr) polygons,
				polygons -> vertices[0] . colorIndex);
		    }
		}
		break;
	    case RECTMESH:

		ColorItemWithIndex(destItem,
			((RectMeshPtr) curItem) -> vertices[0] . colorIndex);
		break;

	}
	curItem = curItem -> next;
	destItem = destItem -> next;
    }
    return destItem;
}

#ifdef PROTO
void ColorPictureByPicture(ObjPtr dest, ObjPtr owner, ObjPtr pic)
#else
void ColorPictureByPicture(dest, owner, pic)
ObjPtr dest, pic;
ObjPtr owner;
#endif
/*Colors a picture with another picture, which must be the source picture*/
{
    PicItemPtr curItem, destItem;

    curItem = ((PicPtr) pic) -> items;
    destItem = ((PicPtr) dest) -> items;
    ColorItemsByItems(destItem, owner, curItem);
}

#ifdef PROTO
ObjPtr ConvertOntoPicture(ObjPtr retVal, ObjPtr pic, ObjPtr owner)
#else
ObjPtr ConvertOntoPicture(retVal, pic, owner)
ObjPtr retVal;
ObjPtr pic;
ObjPtr owner;
#endif
/*Returns a copy of pic converted into internal form on the end of retVal.  
  Owner can give info on conversion*/
{
    PicItemPtr curItem;
    ObjPtr var;
    int sphereSub;
    int frustumSub;
    Bool capEnds;

    var = GetIntVar("ConvertOntoPicture", owner, SPHERESUBDIV);
    if (var)
    {
	sphereSub = GetInt(var);
    }
    else
    {
	sphereSub = 2;
    }

    var = GetIntVar("ConvertOntoPicture", owner, FRUSTUMSUBDIV);
    if (var)
    {
	frustumSub = GetInt(var);
    }
    else
    {
	frustumSub = 2;
    }

    capEnds = GetPredicate(owner, CAPENDSP);

    curItem = ((PicPtr) pic) -> items;
    while(curItem)
    {
	switch (curItem -> type)
	{
	    case POLYGON:
	    case PLANARPOLYGON:
		AppendPolyToPicture(retVal,
			((PolyPtr) curItem) -> nVertices,
			((PolyPtr) curItem) -> vertices);
		break;
	    case FRUSTUM:
		ConvertFrustumOntoPicture(retVal, 
			((FrustumPtr) curItem) -> end1,
			((FrustumPtr) curItem) -> rad1,
			((FrustumPtr) curItem) -> end2,
			((FrustumPtr) curItem) -> rad2,
			frustumSub, capEnds);
		break;
	    case SPHERE:
		AppendItemToPicture((PicPtr) retVal,
			(PicItemPtr) SphereToPolys(curItem, sphereSub));
		break;
	    case POLYGONS:
		{
		    PolyPtr polygons;
		    PolysPtr polySet;
		    polySet = NewPolygons();
		    polygons = ((PolysPtr) curItem) -> polygons;
		    while (polygons)
		    {
			AppendPolyToPolys(polySet, polygons -> nVertices, polygons -> vertices);
			polygons = (PolyPtr) (polygons -> item . next);
		    }
		    AppendItemToPicture((PicPtr) retVal, (PicItemPtr) polygons);
		}
		break;
	    case RECTMESH:
		{
		    RectMeshPtr rectMesh;
		    long k;
		    rectMesh = NewRectMesh(
			((RectMeshPtr) curItem) -> xDim, 
			((RectMeshPtr) curItem) -> yDim,
			((RectMeshPtr) curItem) -> inCenter);
		    ((RectMeshPtr) rectMesh) -> item . colorShading =
		    ((RectMeshPtr) curItem) -> item . colorShading;
		    for (k = 0; k < (((RectMeshPtr) curItem) -> xDim * (2 * ((RectMeshPtr) curItem) -> yDim - 1)); ++k)
		    {
			((RectMeshPtr) rectMesh) -> vertices[k] . normal[0] =
			((RectMeshPtr) curItem) -> vertices[k] . normal[0];
			((RectMeshPtr) rectMesh) -> vertices[k] . normal[1] =
			((RectMeshPtr) curItem) -> vertices[k] . normal[1];
			((RectMeshPtr) rectMesh) -> vertices[k] . normal[2] =
			((RectMeshPtr) curItem) -> vertices[k] . normal[2];
			((RectMeshPtr) rectMesh) -> vertices[k] . position[0] =
			((RectMeshPtr) curItem) -> vertices[k] . position[0];
			((RectMeshPtr) rectMesh) -> vertices[k] . position[1] =
			((RectMeshPtr) curItem) -> vertices[k] . position[1];
			((RectMeshPtr) rectMesh) -> vertices[k] . position[2] =
			((RectMeshPtr) curItem) -> vertices[k] . position[2];
			((RectMeshPtr) rectMesh) -> vertices[k] . alpha =
			((RectMeshPtr) curItem) -> vertices[k] . alpha;
			((RectMeshPtr) rectMesh) -> vertices[k] . colorIndex =
			((RectMeshPtr) curItem) -> vertices[k] . colorIndex;
		    }
		    AppendItemToPicture((PicPtr) retVal, ((PicItemPtr) rectMesh));
		}
		break;
	}
	curItem = curItem -> next;
    }
    return retVal;
}

#ifdef PROTO
ObjPtr ConvertPicture(ObjPtr pic, ObjPtr owner)
#else
ObjPtr ConvertPicture(pic, owner)
ObjPtr pic;
ObjPtr owner;
#endif
/*Returns a copy of pic converted into internal form.  Owner can give info on conversion*/
{
    ObjPtr retVal;
    retVal = NewPicture();
    ConvertOntoPicture(retVal, pic, owner);
    return retVal;
}

void GetPictureBounds(pic, bounds)
ObjPtr pic;
real bounds[6];
/*Returns the bounds of pic in bounds*/
{
    PicItemPtr curItem;
    bounds[0] = bounds[2] = bounds[4] = PLUSINF;
    bounds[1] = bounds[3] = bounds[5] = MINUSINF;

    curItem = ((PicPtr) pic) -> items;

    while (curItem)
    {
	switch(curItem -> type)
	{
	    case SPHERE:
		if (((SpherePtr) curItem) -> center[0] -
		    ((SpherePtr) curItem) -> radius < bounds[0])
		{
		    bounds[0] = ((SpherePtr) curItem) -> center[0] -
				((SpherePtr) curItem) -> radius;
		}
		if (((SpherePtr) curItem) -> center[0] +
		    ((SpherePtr) curItem) -> radius > bounds[1])
		{
		    bounds[1] = ((SpherePtr) curItem) -> center[0] +
				((SpherePtr) curItem) -> radius;
		}
		if (((SpherePtr) curItem) -> center[1] -
		    ((SpherePtr) curItem) -> radius < bounds[2])
		{
		    bounds[2] = ((SpherePtr) curItem) -> center[1] -
				((SpherePtr) curItem) -> radius;
		}
		if (((SpherePtr) curItem) -> center[1] +
		    ((SpherePtr) curItem) -> radius > bounds[3])
		{
		    bounds[3] = ((SpherePtr) curItem) -> center[1] +
				((SpherePtr) curItem) -> radius;
		}
		if (((SpherePtr) curItem) -> center[2] -
		    ((SpherePtr) curItem) -> radius < bounds[4])
		{
		    bounds[4] = ((SpherePtr) curItem) -> center[2] -
				((SpherePtr) curItem) -> radius;
		}
		if (((SpherePtr) curItem) -> center[2] +
		    ((SpherePtr) curItem) -> radius > bounds[5])
		{
		    bounds[5] = ((SpherePtr) curItem) -> center[2] +
				((SpherePtr) curItem) -> radius;
		}
		break;
	    case FRUSTUM:
		if (((FrustumPtr) curItem) -> end1[0] - 
		    ((FrustumPtr) curItem) -> rad1 < bounds[0])
		{
		    bounds[0] = ((FrustumPtr) curItem) -> end1[0] - 
		    		((FrustumPtr) curItem) -> rad1;
		}
		if (((FrustumPtr) curItem) -> end1[0] + 
		    ((FrustumPtr) curItem) -> rad1 > bounds[1])
		{
		    bounds[1] = ((FrustumPtr) curItem) -> end1[0] + 
		    		((FrustumPtr) curItem) -> rad1;
		}
		if (((FrustumPtr) curItem) -> end1[1] - 
		    ((FrustumPtr) curItem) -> rad1 < bounds[2])
		{
		    bounds[2] = ((FrustumPtr) curItem) -> end1[1] - 
		    		((FrustumPtr) curItem) -> rad1;
		}
		if (((FrustumPtr) curItem) -> end1[1] + 
		    ((FrustumPtr) curItem) -> rad1 > bounds[3])
		{
		    bounds[3] = ((FrustumPtr) curItem) -> end1[1] + 
		    		((FrustumPtr) curItem) -> rad1;
		}
		if (((FrustumPtr) curItem) -> end1[2] - 
		    ((FrustumPtr) curItem) -> rad1 < bounds[4])
		{
		    bounds[4] = ((FrustumPtr) curItem) -> end1[2] - 
		    		((FrustumPtr) curItem) -> rad1;
		}
		if (((FrustumPtr) curItem) -> end1[2] + 
		    ((FrustumPtr) curItem) -> rad1 > bounds[5])
		{
		    bounds[5] = ((FrustumPtr) curItem) -> end1[2] + 
		    		((FrustumPtr) curItem) -> rad1;
		}

		if (((FrustumPtr) curItem) -> end2[0] - 
		    ((FrustumPtr) curItem) -> rad2 < bounds[0])
		{
		    bounds[0] = ((FrustumPtr) curItem) -> end2[0] - 
		    		((FrustumPtr) curItem) -> rad2;
		}
		if (((FrustumPtr) curItem) -> end2[0] + 
		    ((FrustumPtr) curItem) -> rad2 > bounds[1])
		{
		    bounds[1] = ((FrustumPtr) curItem) -> end2[0] + 
		    		((FrustumPtr) curItem) -> rad2;
		}
		if (((FrustumPtr) curItem) -> end2[1] - 
		    ((FrustumPtr) curItem) -> rad2 < bounds[2])
		{
		    bounds[2] = ((FrustumPtr) curItem) -> end2[1] - 
		    		((FrustumPtr) curItem) -> rad2;
		}
		if (((FrustumPtr) curItem) -> end2[1] + 
		    ((FrustumPtr) curItem) -> rad2 > bounds[3])
		{
		    bounds[3] = ((FrustumPtr) curItem) -> end2[1] + 
		    		((FrustumPtr) curItem) -> rad2;
		}
		if (((FrustumPtr) curItem) -> end2[2] - 
		    ((FrustumPtr) curItem) -> rad2 < bounds[4])
		{
		    bounds[4] = ((FrustumPtr) curItem) -> end2[2] - 
		    		((FrustumPtr) curItem) -> rad2;
		}
		if (((FrustumPtr) curItem) -> end2[2] + 
		    ((FrustumPtr) curItem) -> rad2 > bounds[5])
		{
		    bounds[5] = ((FrustumPtr) curItem) -> end2[2] + 
		    		((FrustumPtr) curItem) -> rad2;
		}
		break;
	    case POLYGON:
		{
		    int k;
		    VertexTuple *curVertex;

		    curVertex = ((PolyPtr) curItem) -> vertices;
		    for (k = 0; k < ((PolyPtr) curItem) -> nVertices; ++k)
		    {
			if (curVertex -> position[0] < bounds[0])
			{
			    bounds[0] = curVertex -> position[0];
			}
			if (curVertex -> position[0] > bounds[1])
			{
			    bounds[1] = curVertex -> position[0];
			}
			if (curVertex -> position[1] < bounds[2])
			{
			    bounds[2] = curVertex -> position[1];
			}
			if (curVertex -> position[1] > bounds[3])
			{
			    bounds[3] = curVertex -> position[1];
			}
			if (curVertex -> position[2] < bounds[4])
			{
			    bounds[4] = curVertex -> position[2];
			}
			if (curVertex -> position[2] > bounds[5])
			{
			    bounds[5] = curVertex -> position[2];
			}
			++curVertex;
		    }
		}
		break;
	    case POLYGONS:
		{
		    PolyPtr polygons;
		    polygons = ((PolysPtr) curItem) -> polygons;
		    while (polygons)
		    {
			int k;
			VertexTuple *curVertex;

			curVertex = polygons -> vertices;
			for (k = 0; k < polygons -> nVertices; ++k)
			{
			    if (curVertex -> position[0] < bounds[0])
			    {
				bounds[0] = curVertex -> position[0];
			    }
			    if (curVertex -> position[0] > bounds[1])
			    {
				bounds[1] = curVertex -> position[0];
			    }
			    if (curVertex -> position[1] < bounds[2])
			    {
				bounds[2] = curVertex -> position[1];
			    }
			    if (curVertex -> position[1] > bounds[3])
			    {
				bounds[3] = curVertex -> position[1];
			    }
			    if (curVertex -> position[2] < bounds[4])
			    {
				bounds[4] = curVertex -> position[2];
			    }
			    if (curVertex -> position[2] > bounds[5])
			    {
				bounds[5] = curVertex -> position[2];
			    }
			    ++curVertex;
			}
			polygons = (PolyPtr) (polygons -> item . next);
		    }
		}
		break;
	    case RECTMESH:
		{
		    register long i, j, offset;
		    VertexTuple *vertices;

		    vertices = ((RectMeshPtr) curItem) -> vertices;

		    for (i = 0; i < ((RectMeshPtr) curItem) -> xDim; ++i)
		    {
			for (j = 0; j < ((RectMeshPtr) curItem) -> yDim; ++j)
			{
			    offset = RECTMESHVERTEX((RectMeshPtr) curItem, i, j);
			    if (vertices[offset] . position[0] < bounds[0])
			    {
				bounds[0] = vertices[offset] . position[0];
			    }
			    if (vertices[offset] . position[0] > bounds[1])
			    {
				bounds[1] = vertices[offset] . position[0];
			    }
			    if (vertices[offset] . position[1] < bounds[2])
			    {
				bounds[2] = vertices[offset] . position[1];
			    }
			    if (vertices[offset] . position[1] > bounds[3])
			    {
				bounds[3] = vertices[offset] . position[1];
			    }
			    if (vertices[offset] . position[2] < bounds[4])
			    {
				bounds[4] = vertices[offset] . position[2];
			    }
			    if (vertices[offset] . position[2] > bounds[5])
			    {
				bounds[5] = vertices[offset] . position[2];
			    }
			}
		    }
		}
		break;
	}
	curItem = curItem -> next;
    }
}
