/*	ScianFont.c
	Routines for font system and string utilities

	Jim Lyons
	7/26/91

	9/24/91  added support for raster fonts
*/

#include "Scian.h"
#include "ScianStyle.h"
#include "ScianTypes.h"
#include "ScianFontSystem.h"
#include "ScianDraw.h"
#include "ScianPalatino18.h"
#include "ScianTitleFont.h"


#define NRASTERFONTS	4	/* NOT including system font or icon fonts */
#define SCIANTITLE	1

/*
	The font system handles font manager routines, if available, as
	well as raster fonts. When SetupFont is called, the font name
	is first looked up using fmfindfont. If found, the font is scaled
	to size and currentFontHandle is set for use by subsequent calls 
	to DrawString, StringWidth, etc. If fmfindfont fails (or if not
	FONTS4D) currentFontHandle is set to NULL and the name is looked
	up in the raster font name table, rasterFont[]. (The size parameter
	is ignored.) If found, curFont is set to the number; if not found, 
	curFont is set to 0, the system font.

	Note that icon fonts are not treated like text fonts, although they
	are initialized here in InitFonts.
*/

/* globals for font system */

char *rasterFont[NRASTERFONTS + 1];	/* table of raster font name pointers */
static int curFont = 0;	/* current raster font number (0 = "System") */

#ifdef FONTS4D

fmfonthandle currentFontHandle = NULL;
/*** local font cache system? ***/

#endif



void InitFonts(void)
{
	int i;

#ifdef FONTS4D
	fminit();
	/*** count and register fonts for menu (from ScianMain.c) ***/
	/*** set up caching? ***/
#endif
	rasterFont[0] = "System"; /* default for curFont = 0 */
	for (i=1; i<NRASTERFONTS+1; ++i) rasterFont[i] = "\0";

	rasterFont[TITLEFONT] = "ScianTitle";
	defrasterfont(TITLEFONT, ScianTitleFontHT, ScianTitleFontNC, ScianTitleFontchars, 
			ScianTitleFontRS, (void *) ScianTitleFontrasters);

	rasterFont[SUBTITLEFONT] = "Palatino18";
	defrasterfont(SUBTITLEFONT, palatino18HT, palatino18NC, stChars,
			palatino18RS, (void *) stRasters);
	/*** add raster fonts to menu? ***/
}

void KillFonts(void)
{
	/*** free memory ***/
}

#ifdef PROTO
Bool SetupFont(char *name, int size)
#else
Bool SetupFont(name, size)
char *name;
int size;
#endif
{
	int i;
#ifdef FONTS4D
	currentFontHandle = fmfindfont(name);
	if (currentFontHandle)
	{
		currentFontHandle = fmscalefont(currentFontHandle, size);
		fmsetfont(currentFontHandle);
		return TRUE;
	}
	else
	{
		currentFontHandle = NULL;
		for (i=1; i<NRASTERFONTS+1; ++i)
		{
			if (strcmp(name, rasterFont[i]) != 0) continue;
			font(curFont = i);
			return TRUE;
		}
		font(curFont = 0);
		return FALSE;
	}
#else
	for (i=1; i<NRASTERFONTS+1; ++i)
	{
		if (strcmp(name, rasterFont[i]) != 0) continue;
		font(curFont = i);
		return TRUE;
	}
	font(curFont = 0);
	return FALSE;
#endif
}

#ifdef PROTO
int ChrWidth(char c)
#else
int ChrWidth(c)
char c;
#endif
{
	if (c < ' ') return 0;
#ifdef FONTS4D
	if (currentFontHandle)
	{
		return fmgetchrwidth(currentFontHandle, c);
	}
	else
	{
		char buf[2];

		buf[0] = c;
		buf[1] = '\0';
		return strwidth(buf);
	}
#else
	{	char buf[2];

	buf[0] = c;
	buf[1] = '\0';
	return strwidth(buf);
	}
#endif
}

#ifdef PROTO
int StrWidth(char *s)
#else
int StrWidth(s)
char *s;
#endif
{
#ifdef FONTS4D
	if (currentFontHandle)
	{
		return fmgetstrwidth(currentFontHandle, s);
	}
	else
	{
		return strwidth(s);
	}
#else
	return strwidth(s);
#endif
}

#ifdef PROTO
void DrawString(long x, long y, char *s)
#else
void DrawString(x, y, s)
long x, y;
char *s;
#endif
{
	Screencoord left, right, bottom, top;
	int xx,yy;

	/* first check to see if string starts out of bounds */
	getviewport(&left, &right, &bottom, &top);
	CurOffset(&xx, &yy);
	left -= xx;
	while (x < left)
	{
		if (*s == '\0') return;
		x += ChrWidth(*s++);
	}
	cmov2i(x, y);

#ifdef FONTS4D
	if (currentFontHandle)
	{
		fmprstr(s);
	}
	else
	{
		charstr(s);
	}
#else
	charstr(s);
#endif
}
