/*
  File: smalltalk2wff.c
  Author: John Maloney
  Last Modified: 1 December 1988
  Purpose: Converts a Smalltalk "Form" file into a binary .wff image file.

  A Smalltalk Form object is stored as follows:
    <ID><width><height><horizonalOffset><verticalOffset>[bits...]
  where:
    <ID> is the sixteen bit integer constant 1
    <width> and <height> are sixteen bit unsigned integers
    <horizonalOffset> and <verticalOffset> are sixteen bit signed integers
    [bits...] is the bitmap, packed as sixteen bit words and stored in
       row-major order. The width is rounded up to the nearest multiple
       of sixteen; the extra bits are ignored. The first row is the top
       scanline of the image.

  Because forms are stored top-to-bottom whereas .wff files are stored
  bottom-to-top, we buffer the entire bitmap.
*/
#include <stdio.h>
#include <strings.h>
#include "wff.h"

/* Smalltalk Form header */
unsigned char Header[10] =
 {                    /* Header for a Smalltalk Form file */
  0, 1,               /* [0,1] ID field = 1               */
  0, 0,               /* [2,3] unsigned width             */
  0, 0,               /* [4,5] unsigned height            */
  0, 0,               /* [6,7] signed horizontal offset   */
  0, 0                /* [8,9] signed vertical offset     */
 };

static char *RoutineName;

static void usage()
 {
  fprintf(stderr,"Usage is\n\t%s <formFile >wffFile\n", RoutineName);
 }

static void
ReadFormHeader(smalltalkFile, width, height, hOffset, vOffset)
 FILE *smalltalkFile;
 int *width, *height, *hOffset, *vOffset;
 {
  if (10 != fread(Header, 1, 10, smalltalkFile))
   {
    fprintf(stderr, "%s: Error reading Smalltalk Form header\n",
                    RoutineName);
    exit(1);
   }
  if ((0 != Header[0]) || (1 != Header[1]))
   {
    fprintf(stderr, "%s: bad Smalltalk Form header\n", RoutineName);
    exit(1);
   }
  *width   = Header[3] | (Header[2] << 8);
  *height  = Header[5] | (Header[4] << 8);
  *hOffset = Header[7] | (Header[6] << 8);
  *vOffset = Header[9] | (Header[8] << 8);
 }

static void
ReadFormBitmap(smalltalkFile, width, height, bitmap)
 FILE *smalltalkFile;
 int width, height;
 unsigned short *bitmap;
 {
  int columnCount = (width + 15) / 16;
  int column, row;
  unsigned char inWord[2];
  unsigned short word;

  for (row = (height - 1); row >= 0; row--)
   {
    for (column = 0; column < columnCount; column++)
     {
      if (2 != fread(&inWord[0], 1, 2, smalltalkFile))
       {
        fprintf(stderr, "%s: File error reading Smalltalk bitmap\n",
                         RoutineName);
        exit(1);
       }
      word = ((inWord[0] << 8) + (inWord[1]));
      bitmap[(row * columnCount) + column] = word;
     }
   }
 }

static int
GetBit(bitmap, columnCount, x, y)
 unsigned short *bitmap;
 int columnCount, x, y;
 {
  int column, row, word;
  unsigned short mask;

  row = y;
  column = x / 16;
  mask = 1 << (15 - (x % 16));
  word = bitmap[(row * columnCount) + column];
  if (0 == (word & mask))
   { return 1; }
  else
   { return 0; }
 }

static void
WriteWffPixels(FBout, width, height, bitmap)
 FrameBufferType *FBout;
 int width, height;
 unsigned short *bitmap;
 {
  int columnCount = (width + 15) / 16;
  int x, y;

  for (y = 0; y < height; y++)
   {
    for (x = 0; x < width; x++)
     {
      if (FAILURE == NextPixelOut(FBout, GetBit(bitmap, columnCount, x, y)))
       {
        fprintf(stderr, "%s: File error writing Wff bitmap.\n",
                         RoutineName);
        (void)CloseFB(&FBout);
        exit(1);
       }
     }
   }
 }

static void
Pass(fdIn, fdOut)
 FILE *fdIn, *fdOut;
 {
  FrameBufferType *FBout = NULL;
  int Bottom, Left, Top, Right;
  char Name[NameLength], Value[ValueLength];
  char WhatBands[ValueLength];
  int BitsPerBand;
  int width, height, hOffset, vOffset, columnCount, bitmapSize;
  unsigned short *bitmap;	/* points to a dynamically allocated bitmap */

  /* read the Smalltalk Form header */
  ReadFormHeader(fdIn, &width, &height, &hOffset, &vOffset);
  columnCount = (width + 15) / 16;

  /* allocate the bitmap */
  bitmapSize = columnCount * height * sizeof(unsigned short);
  bitmap = (unsigned short *) malloc(bitmapSize);
  if (NULL == bitmap)
   {
    fprintf(stderr, "Insufficient memory for scratch bitmap\n" );
    (void) CloseFB(&FBout);
    exit(1);
   }

  /* read the Smalltalk Form bitmap */
  ReadFormBitmap(fdIn, width, height, bitmap);

  /* create and write the .wff header */
  if (FAILURE == OpenFB(&FBout))  { return; }
  Bottom = vOffset;
  Left = hOffset;
  Top = vOffset + height - 1;
  Right = hOffset + width - 1;
  SetBounds(FBout, Bottom, Left, Top, Right);
  (void) strcpy(WhatBands,"I"); BitsPerBand = 1;
  SetColorSystem(FBout, WhatBands, BitsPerBand);
  (void) strcpy(Name,"X-CreatedBy");
  (void) strcpy(Value, RoutineName);
  SetDescriptor(FBout, Name, Value);
  if (FAILURE == PassImageOut(fdOut, FBout))
   {
    (void)CloseFB(&FBout);
    exit(1);
   }

  /* write the bitmap to the WFF file */
  WriteWffPixels(FBout, width, height, bitmap);

  /* free the bitmap storage and close the WFF file */
  free(bitmap);
  (void)CloseFB(&FBout);
 }

int main(argc, argv)
 int argc;
 char *argv[];
 {
  RoutineName = argv[0];
  if (argc != 1)
   { usage(); exit (-1); }
  Pass(stdin, stdout);
  exit(0);
 }
