/*
 *----------------------------------------------------------------------
 *  File:  a2wff.c
 *	Convert an ascii representation of a bitmap to a wff file.
 *	The file format is assume to use two characters, one for
 *	white, one for black.  A line in the file represents a scanline
 *	of the image.
 *
 *	Flags:
 *		-b c	: the character that represents black
 *		-w c	: the character that represents white
 *		-r	: reverse the characters used for black and white
 *		-B n	: the number of bits of intensity the output
 *			  image should have
 *	Defaults:
 *		The defaults for black and white are # and -.  These
 *		where chosen because they're what bmtoa (the converter
 *		for X) uses.
 *		The default number of intensity bits is 1.
 *----------------------------------------------------------------------
 */
#include <stdio.h>
#include <math.h>
#include <strings.h>
#include "wff.h"

#define MAX_SIZE 1300

FILE* fd = stdin;
char black = '#';
char white = '-';
int bits=1;

ReadABitmap(fd, bm, h, w)
FILE* fd;
char bm[MAX_SIZE][MAX_SIZE];
int* h;
int* w;
{
	char buf[MAX_SIZE+1];
	int sl;
	int first=1;

	*h = 0;
	while ( fgets(buf, MAX_SIZE+1, fd) ) {
		sl = strlen(buf);
		if ( sl <= 1 ) {
			continue;
		}
		if ( sl == MAX_SIZE  &&  buf[sl-1] != '\n' ) {
			fprintf(stderr, "ReadABitmap: bitmap too wide.\n");
			exit(1);
		}
		buf[sl-1] = NULL; sl--;
		if ( first ) {
			*w = sl;
			first=0;
		} else if ( *w != sl ) {
			fprintf(stderr, 
				"ReadABitmap: rows have different lengths\n");
			exit(1);
		}
		if ( *h >= MAX_SIZE ) {
			fprintf(stderr, "ReadABitmap: bitmap too tall.\n");
			exit(1);
		}
		strcpy(bm[*h], buf);
		(*h)++;
	}
}


GetBitmapPixel(bm, i, j, p)
char bm[MAX_SIZE][MAX_SIZE];
int i;
int j;
short* p;
{
	if ( bm[i][j] == white ) {
		*p = 0xFF;
	} else if ( bm[i][j] == black ) {
		*p = 0;
	} else {
		fprintf(stderr, 
			"Warning: pixel %d %d '%c' not black or white\n",
			i,j,bm[i][j]);
		fprintf(stderr, "	Using black.\n");
		*p = 0;
	}
}


main(argc,argv)
int argc;
char* argv[];
{
	int height;
	int width;
	char bm[MAX_SIZE][MAX_SIZE];
	FrameBufferType* fb;
	short p;
	int i,j;

	ParseArgs(argc, argv);

	ReadABitmap(fd, bm,&height,&width);

	fb = NULL;
	OpenFB(&fb);
	fprintf(stderr,"Height = %d, width = %d\n",height,width);
	SetBounds(fb, 0, 0, height-1, width-1);
	SetColorSystem(fb, "I", bits);
	SetDescriptor(fb, "Title", "X Bitmap");
	for(i=0; i<height; i++) {
		for(j=0; j<width; j++) {
			GetBitmapPixel(bm, i,j, &p);
			PutPixel(fb, j, height-1-i, &p);
		}
	}

	WriteImage(stdout, fb);
	CloseFB(&fb);
}

FormatError()
{
	fprintf(stderr,"Format: a2wff [-b c] [w c] [-r] [-B n]\n");
	exit(1);
}


ParseArgs(argc, argv)
int argc;
char* argv[];
{
	int i;
	char tmp;

	for ( i=1; i<argc; i++ ) {
		if ( argv[i][0] != '-' ) {
			FormatError();
		}
		switch(argv[i][1]) {
		      case 'b':
			black = argv[i+1][0];
			i++;
			break;
		      case 'w':
			white = argv[i+1][0];
			i++;
			break;
		      case 'r':
			tmp = black;
			black = white;
			white = tmp;
			break;
		      case 'B':
			bits = atoi(argv[i+1]);
			if ( bits <= 0 ) bits = 1;
			if ( bits > 8 ) bits = 8;
			i++;
			break;
		      default:
			FormatError();
		}
	}
}	
