/*
  File: Stretch.c 
  Authors: K.R. Sloan
	   S. Mann
  Last Modified: January 29, 1992
  Purpose: Stretch a wff INTENSITY image, to enhance constrast
 */

#include <stdio.h>
#include <wff.h>
#include <math.h>

int VERBOSE = 0;

static char *RoutineName;
static void usage()
 {
  fprintf(stderr, "Usage is\n\t%s [-v][-h][-MQE -p pct]\n",
              RoutineName);
 }

static int histogram[1<<16], table[1<<16];
static void
Stretch(fdIn, fdOut, pct, Equalize, UseQuartiles, UseMedian)
 FILE *fdIn, *fdOut;
 int pct, Equalize, UseQuartiles, UseMedian;
 {
  FrameBufferType *FrameBuffer;
  int Left, Bottom, Top, Right, BitsPerBand, BandsPerPixel;
  char WhatBands[10], Name[NameLength], Value[ValueLength];
  int x,y,n;
  int passed;
  int Levels,i,low,high;
  unsigned short pixel[99];
  double N, Sum, SumSq, StdDev, Mean;
  double S,Limit;
  int pct25, pct50, pct75, pct1, pct99;
  int peak, mode;

  FrameBuffer  = (FrameBufferType *)0;
  if (FAILURE == OpenFB(&FrameBuffer))         return;
  if (FAILURE == ReadImage(fdIn, FrameBuffer)) return;
  if (FAILURE == GetBounds(FrameBuffer, &Bottom, &Left, &Top, &Right)) 
   { (void)CloseFB(&FrameBuffer); return; }
  if (FAILURE == GetColorSystem(FrameBuffer, WhatBands, &BitsPerBand))
   { (void)CloseFB(&FrameBuffer); return; }
  if (FAILURE == GetDescriptor(FrameBuffer,"BitsPerBand",Value))
   { (void)CloseFB(&FrameBuffer); return; }
  BitsPerBand = atoi(Value);

  BandsPerPixel = strlen(WhatBands);
  if (   (0 != strcmp(WhatBands,"I"))
      && (0 != strcmp(WhatBands,"IA")) )
   {
    fprintf(stderr,"Stretch: can't handle ColorSystem %s\n",WhatBands);
    if (1 == BandsPerPixel)
     {
      fprintf(stderr,"Stretch: treating it as ColorSystem = I\n");
     }
    else return;
   }

  /* 
     Scan existing NV pairs - change "X-PassedBy"
     and leave the rest alone
   */
  for (n=0;;n++)
   {
    GetDescriptorN(FrameBuffer, n, Name, Value);
    if (Name[0] == '\0') break;
    if (0 == strcmp(Name,"X-PassedBy"))
     {
      if ( (strlen(Value)+strlen(RoutineName)+3) > ValueLength)
       strcpy(Value,"...");
      strcat(Value,", "); strcat(Value,RoutineName);
      passed = 1;
      SetDescriptor(FrameBuffer, Name, Value);
     }
   }

  /*  if necessary, add "X-PassedBy" */
  if (0 == passed)
   {
    strcpy(Name,"X-PassedBy");
    strcpy(Value,RoutineName);
    SetDescriptor(FrameBuffer, Name, Value);
   }

  /*  
    Now that the header is taken care of, on with the real work
   */

  Levels = 1<<BitsPerBand;

  for (i=0;i<Levels;i++) histogram[i] = 0;

  for (y=Bottom;y<=Top;y++)
   for (x=Left;x<=Right;x++)
    {
     GetPixel(FrameBuffer,x,y,pixel);
     histogram[pixel[0]]++;
    }

  for (low =   0;     (low<Levels) && (histogram[low] == 0); low++);
  for (high = Levels; (high>  0) && (histogram[high] == 0); high--);

  N = 0.0;  Sum = 0.0;  SumSq = 0.0;  peak = 0;  mode = 0;
  for (i = high; i >= low; i--)
   {
    if (peak < histogram[i]) { peak = histogram[i]; mode = i; }
    N     += (double)histogram[i];
    Sum   += (double)histogram[i]*(double)i;
    SumSq += (double)histogram[i]*(double)i*(double)i;
   }
  
  Mean = Sum/N;
  StdDev = sqrt((SumSq-(Sum*Sum/N))/(N-1.0));

  Limit = pct*N*0.01;
  for (S = 0.0, pct1 = low;  (pct1<Levels) && (S<Limit);pct1++)
   S += histogram[pct1];
  pct1--;
  for (S = 0.0, pct99 = high; (pct99>0) && (S<Limit);pct99--)
   S += histogram[pct99];
  pct99++;
  Limit = 0.25*N;
  for (S = 0.0, pct25 = low; (pct25<Levels) && (S<Limit);pct25++)
   S += histogram[pct25];
  pct25--;
  for (S = 0.0, pct75 = high; (pct75>0) && (S<Limit);pct75--)
   S += histogram[pct75];
  pct75++;
  Limit = 0.50*N;
  for (S = 0.0, pct50 = low; (pct50<Levels) && (S<Limit);pct50++)
   S += histogram[pct50];
  pct50--;

  fprintf(stderr,
   "Stretch: low = %d, mode = %d high = %d, Mean = %lf, StdDev = %lf\n",
               low, mode, high, Mean, StdDev);
  fprintf(stderr,
   "Stretch: pct[%d]=%d, pct[25]=%d, pct[50]=%d, pct[75]=%d, pct[%d]=%d\n",
                pct, pct1, pct25, pct50, pct75, 100-pct, pct99);

  if (Equalize)
   {
    fprintf(stderr, "Stretch: Equalizing the histogram\n");
    S = 0.0;
    for (i=0; i<Levels; i++)
     {
      S += histogram[i];
      table[i] = S*Levels/N;
     }
   }
  else if (UseQuartiles)
   {
    double Quarter;

    Quarter = Levels/4.0;
    fprintf(stderr,
    "Stretch: placing the Median at %d, pct[25] at %d, pct[75] at %d\n,");
    fprintf(stderr,
    "                                 pct[%d] at 0, and pct[%d] at %d\n",
    (int)Quarter, (int)(2.0*Quarter), (int)(3.0*Quarter),
                   pct, 100-pct, (int)(Levels-1.0));

    for (i=0; i<pct1; i++)
     table[i] =   0.0;
    for (i=pct1; i<pct25; i++) 
     table[i] =   0.0 + (Quarter*(i-pct1)/(pct25-pct1));
    for (i=pct25; i<pct50;i++) 
     table[i] = Quarter + (Quarter*(i-pct25)/(pct50-pct25));
    for (i=pct50; i<pct75;i++) 
     table[i] = 2.0*Quarter + (Quarter*(i-pct50)/(pct75-pct50));
    for (i=pct75; i<pct99;i++) 
     table[i] = 3.0*Quarter + (Quarter*(i-pct75)/(pct99-pct75));
    for (i=pct99; i<Levels; i++)
     table[i] = Levels-1.0;
   }
  else if (UseMedian)
   {
    double Half;

    Half = Levels/2.0;
    fprintf(stderr,
    "Stretch: placing the Median at %d, pct[%d]at 0, and pct[%d] at %d\n",
                     (int)Half,                    
                     pct, 100-pct,(int)Levels-1.0);

    for (i=0; i<pct1; i++)
     table[i] =   0.0;
    for (i=pct1; i<pct50; i++) 
     table[i] =   0.0 + (Half*(i-pct1)/(pct50-pct1));
    for (i=pct50; i<pct99;i++) 
     table[i] = Half + (Half*(i-pct50)/(pct99-pct50));
    for (i=pct99; i<256; i++)
     table[i] = Levels-1.0;
   }
  else
   {
    double Half;

    Half = Levels/2.0;
    fprintf(stderr,
      "Stretch: placing the Mean at %d, pct[%d]at 0, and pct[%d] at %d\n",
                     (int)Half,
                     pct, 100-pct, (int)(Levels-1.0));

    for (i=0; i<pct1; i++)
     table[i] =   0.0;
    for (i=pct1; i<Mean; i++) 
     table[i] =   0.0 + (Half*(i-pct1)/(Mean-pct1));
    for (i=Mean; i<pct99;i++) 
     table[i] = Half + (Half*(i-Mean)/(pct99-Mean));
    for (i=pct99; i<256; i++)
     table[i] = Levels-1.0;
   }

  /*
    clamp table to Levels
   */
  for(i=Levels;i>0;i--)
   {
    if(table[i] > Levels-1.0) table[i] = Levels-1.0;
    else                      break;
   }

  /*
    transform the pixels
   */
  for (y=Bottom;y<=Top;y++)
   for (x=Left;x<=Right;x++)
    {
     GetPixel(FrameBuffer,x,y,pixel);
     pixel[0] = table[pixel[0]];
     PutPixel(FrameBuffer,x,y,pixel);
    }

  /*
    write out the image
   */

  (void)WriteImage(fdOut, FrameBuffer);

  return;
 }

int main(argc, argv)
 int argc;
 char *argv[]; 
 {
  int ArgsParsed = 0;
  int pct = 0;
  int UseMedian = 0;
  int UseQuartiles = 0;
  int Equalize = 0;

  RoutineName = argv[ArgsParsed++];

  while (ArgsParsed < argc)
   {
    if ('-' != argv[ArgsParsed][0])  { usage(); exit (-1); }
    switch (argv[ArgsParsed++][1])
     {
      case 'M': UseMedian = -1; break;
      case 'Q': UseQuartiles = -1; break;
      case 'E': Equalize = -1; break;
      case 'p': if (1 < (argc-ArgsParsed)){ usage(); exit (-1); }
                pct = atoi(argv[ArgsParsed++]);
                break;
      case 'v': VERBOSE = -1; break;
      default:
      case 'h': {usage();exit(-1); }
     }
   }
 
  Stretch(stdin, stdout, pct,Equalize,UseQuartiles,UseMedian);

  exit (0);
 }
