/*
  File: Gaussian.c 
  Authors: K.R. Sloan
  Last Modified: 29 June 1989
  Purpose: Convolve with a Gaussian, parameterized by:
             s = std deviation (default 1.0 pixel)
             g = gain          (default 1.0)
             d = dc            (default 0.0)  
           Gain is the area under EACH of the two 1D Gaussians.
           DC is the amount added to EACH of the two 1D Gaussians. 

           An important special case is when the images are offset
           so that middle-gray is considered to be the "zero" value.

           In this case, when we apply a gain factor g, we should
           automatically apply a DC correction of (g-1.0)/2.0.

           To do this, we offer a composite parameter:
             G = gain          (default 1.0)
                 dc = (gain-1.0)/2.0

           Yet another special case is (very) scattered data points
           in a field of gray.  Here, we want to set the gain (and DC)
           so that the peak value of the kernal is 1.0.  So...another
           parameter (-P).  In a fit of generality, we allow the Peak
           value to be a variable, thus:
             P = peak [gain and dc derived from s]

 */

#include <stdio.h>
#include <math.h>
#include "wff.h"

#define ROUND(x) ((int)(0.5+(x)))

#define SwitchChar '-'
#define MaxFilterWidth 1000

double atof();


static int VERBOSE = 0;
static char *RoutineName;
static void usage()
 {
  fprintf(stderr, "Usage is\n\t%s [-h][-v][-s sigma][-g gain][-d DC][-G gain][-P]\n",
              RoutineName);
 }

int main(argc, argv)
 int argc;
 char *argv[]; 
 {
  int ArgsParsed = 0;
  double sigma = 1.0;
  double gain = 1.0;
  double DC = 0.0;
  double Peak = 0.0;
  int n, HalfN, k, v;
  int Kernel[MaxFilterWidth];
  double c, u;
  int Sum;
  double Correction;
  int Correction2;

  RoutineName = argv[ArgsParsed++];

  while (argc > ArgsParsed)
   {
    if (SwitchChar == argv[ArgsParsed][0])
     switch (argv[ArgsParsed++][1])
      {
       case 's': sigma = atof(argv[ArgsParsed++]); break;
       case 'g': gain  = atof(argv[ArgsParsed++]); break;
       case 'd': DC    = atof(argv[ArgsParsed++]); break;
       case 'G': gain  = atof(argv[ArgsParsed++]);
                 DC    = 0.5*(1.0-gain);
                 break;
       case 'P': Peak  = atof(argv[ArgsParsed++]); break;
       case 'v': VERBOSE = 1; break;
       case 'h':
       default:  usage(); exit(-1);  
      }
    else  { usage(); exit (-1); }	
   }

  HalfN = ROUND(2.0*sigma);
  n = 2+HalfN+HalfN;
  if (n>MaxFilterWidth)
    {
     n = MaxFilterWidth;
     HalfN = (n-2)/2;      /* belt */
     n = 2+HalfN+HalfN;    /* suspenders */
    } 

  if (0.0 != Peak)
   {
    gain = Peak*sqrt(3.1415926)*sigma;
    DC   = 0.5*(1.0-gain);
    c    = 2048.0*Peak;
   }
  else
   c = 2048.0*gain/(sqrt(3.1415926)*sigma);

  Kernel[0] = ROUND(2048.0*DC);
  for(k=0;k<=HalfN;k++)
   {
    u = k/sigma; 
    v = ROUND(c*exp(-u*u));
    Kernel[1+HalfN+k] = v;
    Kernel[1+HalfN-k] = v;
   }

  /* correct errors due to truncating the kernel */
   
  Sum = 0;  for(k=1;k<n;k++) Sum += Kernel[k];
  Correction = ROUND(2048.0*gain)/(double)Sum;
  if (VERBOSE)
   fprintf(stderr,
          "sigma = %f; gain = %f; DC = %f\nSum = %d; Correction = %f\n",
           sigma, gain, DC, Sum, Correction);

  /* correct round-off errors */

  for (k=1;k<n;k++) Kernel[k] = (int)((double)Kernel[k]*Correction);
  Sum = 0;  for(k=1;k<n;k++) Sum += Kernel[k];
  Correction2 = ROUND(2048.0*gain)-Sum;
  Kernel[1+HalfN] += Correction2;

  if (VERBOSE)
   {
    fprintf(stderr,"Secondary Correction = %f\n",Correction2/2048.0);
    fprintf(stderr,"DC  : %15.12f\n",(double)Kernel[0]/2048.0);
    for(k=1;k<n;k++)
     fprintf(stderr,"%4d: %15.12f\n", k-HalfN-1,(double)Kernel[k]/2048.0);
   } 

  (void)wffConvHV(stdin,stdout,n,n,Kernel,Kernel);

  exit (0);
 }
