/*
   File: Composite.c 
   Authors: Mike Schmidt,
            K.R. Sloan
	    James Painter
   Last Modified: 20 May 1988
   Purpose: Take two (or more) WFF input files and create an output file which
            contains:

             (...(A over (B over (C .... over Z) over zero background

 */
#include <stdio.h>
#include <strings.h>
#include <math.h>
#include <wff.h>

static char *RoutineName;
static FrameBufferType *FBin = NULL, *FBout = NULL;
 
static void usage()
 {
  fprintf(stderr,"Usage is\n\t%s InputFile1 InputFile2\n",
           RoutineName);
 }

static void ExitErr()
 {
  if ((FrameBufferType *)0 != FBin)
   (void)CloseFB(&FBin);
  if ((FrameBufferType *)0 != FBout)
   (void)CloseFB(&FBout);
  exit(-1);
 }

/*
 *  Composite using the Pixel1 OVER Pixel2.  See Porter & Cook
 *   "Commpositing Digitial Images", Sigraph '84 for details.
 *  Output in Pixel1.
 */

static void
MixPixels (Pixel1,Pixel2,MaxValue,Bands)
 short *Pixel1, *Pixel2;
 int MaxValue, Bands;
 {
  register short *p1, *end1, *p2;
  register int Fraction, value;
 
  Fraction = (MaxValue - Pixel1[Bands-1]); /* Over Maxvalue */
  for (p1 = &Pixel1[0], p2 = &(Pixel2[0]), end1 = p1+Bands; 
       p1 < end1; p1++, p2++)
     {
       value = ( (*p1)*MaxValue + (*p2)*Fraction) / MaxValue;
       if (value > MaxValue) value = MaxValue;
       *p1 = value;
     }
 }
 
static void
Pass(inCount, fnIn,fdOut)
 int  inCount;
 char *fnIn[];
 FILE *fdOut;
 {
  int i, passed=0;
  int Bottom, Left, Top, Right, Bottom1, Left1, Top1, Right1;
  char Name[NameLength], Value1[ValueLength], Value2[ValueLength];
  int BitsPerBand1,BitsPerBand2,MaxValue,x,y,n;
  short Pixel1[99],Pixel2[99];  /* that ought to be enough... */
  int Bands, Opacity; /* TRUE iff ColorSystem == .....A */
  FILE *fdIn;

  /* Open all the input files and compute bounds */
  Left = Bottom = 2048;
  Top  = Right  = 0;
  for(i=0; i< inCount; i++) 
    {
      fdIn = fopen (fnIn[i],"r");
      if ((FILE *)0 == fdIn)
	{
	  fprintf(stderr,"Can't open input file: %s\n",fnIn[i]);
	  exit(-1);
	}
      if (FAILURE == OpenFB(&FBin)) ExitErr();
      if (FAILURE == PassImageIn( fdIn, FBin )) ExitErr();
      if (i == 0)
	{
	  GetColorSystem (FBin,Value1,&BitsPerBand1);
	}
      else
	{
	  GetColorSystem (FBin,Value2,&BitsPerBand2);
	  if ((0 != strcmp(Value1,Value2)) || (BitsPerBand1 != BitsPerBand2))
	    {
	      fprintf (stderr,"Input files are not compatible!\n");
	      ExitErr();
	    }
	}
      GetBounds ( FBin, &Bottom1, &Left1, &Top1, &Right1 );
      if (Bottom > Bottom1)  Bottom = Bottom1;
      if (Left   > Left1)    Left   = Left1;
      if (Top    < Top1)     Top    = Top1;
      if (Right  < Right1)   Right  = Right1;
      CloseFB( &FBin );
      fclose( fdIn );
    }


  /* Reopen the first file */
  fdIn = fopen (fnIn[0],"r");
  if ((FILE *)0 == fdIn)
    {
      fprintf(stderr,"Can't open input file: %s\n",fnIn[0]);
      exit(-1);
    }
  if (FAILURE == OpenFB(&FBin)) ExitErr();
  if (FAILURE == PassImageIn( fdIn, FBin )) ExitErr();


  if (FAILURE == OpenFB(&FBout)) ExitErr();
  /* Make a new header for the output file. */
  /* Copy over existing NV pairs - watch for "X-PassedBy". */
  for (n=0; ;n++)
   {
    GetDescriptorN(FBin,n,Name,Value1);
    if (Name[0] == '\0') break;
    if (0 == strcmp(Name,"X-PassedBy"))
     {
      if ( (strlen(Value1)+strlen(RoutineName)+3) > ValueLength)
       strcpy(Value1,"...");
      strcat(Value1,", "); 
      strcat(Value1,RoutineName);
      passed = 1;
     }
    SetDescriptor(FBout,Name,Value1);
   }
  

  GetColorSystem (FBin,Value1,&BitsPerBand1);
  Bands = strlen(Value1);
  Opacity = ('A' == Value1[Bands-1]);
  if (Opacity) MaxValue = (1 << BitsPerBand1) - 1;
  SetColorSystem (FBout,Value1,BitsPerBand1);
  GetDescriptor(FBin,"Encoding",Value1);
  SetDescriptor(FBout,"Encoding",Value1);
  SetBounds(FBout, Bottom, Left, Top, Right );

  /* If necessary, add passed by. */
  if (0 == passed)
   {
    strcpy (Name,"X-PassedBy");
    strcpy (Value1,RoutineName);
    SetDescriptor(FBout, Name, Value1);
   }

  CloseFB( &FBin );
  fclose(  fdIn );
  /* Header operations over, now we can start the output stream */

  for(i=inCount-1; i>=0; i--)  /* Loop through input files */
    {
      fdIn = fopen (fnIn[i],"r");
      if ((FILE *)0 == fdIn)
	{
	  fprintf(stderr,"Can't open input file: %s\n",fnIn[i]);
	  exit(-1);
	}
      if (FAILURE == OpenFB(&FBin)) ExitErr();
      if (FAILURE == PassImageIn( fdIn, FBin )) ExitErr();

      GetBounds( FBin, &Bottom, &Left, &Top, &Right );
      for (y=Bottom; y<=Top; y++)
	{
	  for(x=Left; x<=Right; x++)
	    {
	      if (FAILURE == NextPixelIn( FBin, Pixel1 )) ExitErr();
	      if (i != inCount-1 && Opacity)
		{
		  if (FAILURE == GetPixel( FBout, x, y, Pixel2 )) ExitErr();
		  MixPixels (Pixel1,Pixel2,MaxValue,Bands);
		}
	      if (FAILURE == PutPixel( FBout, x, y, Pixel1 )) ExitErr();
	    }
	}
      /* Close the file  */
      CloseFB( &FBin );
      fclose( fdIn );
    }

  if (FAILURE == WriteImage( fdOut, FBout )) ExitErr();      
  (void)CloseFB(&FBout);

}

main(argc,argv)
 int argc;
 char *argv[];
 {
  int ArgsParsed = 0;
 
  RoutineName = argv[ArgsParsed++];

  if (argc-ArgsParsed < 2) 
    {
      fprintf( stderr, "Must have at least 2 input files to composite.\n" );
      exit(-1);
    }
  Pass( argc-ArgsParsed, argv+ArgsParsed, stdout);

  exit(0);			/* Success */
}

