/*
  File: StandardMaps.c
  Author: K.R. Sloan,
  Last Modified: 30 January 1991
  Purpose: Establish a standard color map for XShow to use.
           This program should be run:
              After the window manager has fired up
              Before you run XShow
           It should be run in the background, forever! 

           In this version, we do one map per invocation:
             -g : grayscale
             -c : color  [DEFAULT]

           So, you ought to invoke it TWICE - once per map.

           do the color invocation first, to get a good RGB map,
           then, let the grayscale map pick up the rest of the color cells.

           Further options:

            -G gamma : modify the colormap to gamma correct images
                       This is essentially a CONTRAST control.

            -R min max : specify the range of output values
                         This is essentially a BRIGHTNESS control.

 */

#include <math.h>
#include <errno.h>
#include <sys/file.h>
#include <signal.h>
#include <stdio.h>
#include <wff.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <X11/Xutil.h>
#include <X11/Intrinsic.h>
#include <X11/Shell.h>
#include <X11/Form.h>
#include <X11/StringDefs.h>
#include <X11/Box.h>
#include <X11/Command.h>

static int VERBOSE = 0;
static char *RoutineName;
static void usage()
{
 fprintf(stderr,"Usage is:\n\t%s [-h][-v][-g][-c][-G gamma][-R min max]\n",
                  RoutineName);
}

static Display *display;
static Screen  *screen;
static Window  window;
static XStandardColormap cmap;
static int grayscale = 0;
static double Gamma = 1.0;  /* looks OK */ 
static double min = 0.25;   /* raise the black level */
static double max = 1.0;    /* full white */

void term_handler()
 {
  fflush(stdout);
  /*  XGrabServer(display); */
  XDeleteProperty(display, DefaultRootWindow(display), 
                  grayscale? XA_RGB_GRAY_MAP :  XA_RGB_DEFAULT_MAP);
  /*  XUngrabServer(display); */
  XSync(display,False);
  fprintf(stderr,"%s: -30-\n", RoutineName); fflush(stderr);
  exit(0);
 }

int InitializeColormaps()
 {
  int i,r,g,b;
  Colormap cm;
  Status cmstatus;
  XColor colors[256];
  unsigned long pixels[256];
  int ncolors, oncolors;
  double Exponent, Scale, R, G, B;

  /* 
    terminate, with prejudice, existing map!

    We do this to protect against lossage.  Sometimes, the map may
    be defined, but the program which defined it has vanished.  This
    is bad.  

    On the other hand, with this design, we may have many "StandardMaps"
    running - only one (per map) of which is necessary.  It's up to you
    to make sure that one, and only one, StandardMaps is allowed to run
    at a time.  

    Or, you could fix the #$%^&*( window managers to do this for you...

   */

  XDeleteProperty(display, DefaultRootWindow(display),
                  grayscale? XA_RGB_GRAY_MAP :  XA_RGB_DEFAULT_MAP);
  XSync(display,False);

  /* Sanity Check - if the above didn't work, punt! */

  XGetStandardColormap(display, DefaultRootWindow(display),
                       &cmap, 
                  grayscale? XA_RGB_GRAY_MAP :  XA_RGB_DEFAULT_MAP);

  if ((Colormap)0 != cmap.colormap)
   {
    fprintf(stderr,
            "%s: Could not kill the existing standard map! - giving up\n",
             RoutineName);  
    exit(-1);
   }   

  ncolors = grayscale? 256 : 216; /* assume depth = 8 */

  cm = DefaultColormapOfScreen(screen);
  oncolors = ncolors;
  do 
   {
    cmstatus = XAllocColorCells(display, cm,
                                TRUE,                   /* contiguous     */
                                (unsigned long *)0, 0,  /* no plane masks */
                                pixels, ncolors);       /* pixel values   */
    if ((Status)0 == cmstatus)
     { 
      if (grayscale)  ncolors--; /* try for fewer */
      else break; /* insist on all 216 for color */
     }
   } while ((ncolors > 1) && ((Status)0 == cmstatus));

  if ((Status)0 == cmstatus) 
   {
    fprintf(stderr, "%s: Unable to allocate colormap cells - giving up\n",
                    RoutineName);
    exit(-1);
   }

    /* Load the color map */

  cmap.colormap = cm;
  cmap.base_pixel = pixels[0];

  if (grayscale)
   {
    Exponent = 1.0 / Gamma;
    Scale = (max-min)/pow((double)ncolors-1.0,Exponent);
    for(i=0; i<ncolors; i++) 
     {
      colors[i].pixel = pixels[i];
      colors[i].red = colors[i].green = colors[i].blue
                    = 65535.0 * (min + Scale * pow((double)i, Exponent));
      colors[i].flags = DoRed | DoGreen | DoBlue;
      if (VERBOSE) fprintf(stderr,"%s: LUT[%d] = {%d,%d,%d}\n",
            RoutineName, i, colors[i].red, colors[i].green, colors[i].blue);
     }
    cmap.red_max  = ncolors-1; cmap.green_max  = 0; cmap.blue_max  = 0;
    cmap.red_mult = 1;         cmap.green_mult = 0; cmap.blue_mult = 0;
   }
  else
   {
    i = 0;
    Exponent = 1.0 / Gamma;
    Scale = (max-min)/pow(5.0, Exponent);
    for(b=0; b<6; b++)
     {
      B = 65535.0 * (min +  Scale * pow((double)b, Exponent));
      for(g=0; g<6; g++)
       {
        G = 65535.0 * (min +  Scale * pow((double)g, Exponent));
        for(r=0; r<6; r++) 
         {
          R = 65535.0 *(min +  Scale * pow((double)r, Exponent));
          colors[i].pixel = pixels[i];
          colors[i].red   = R;
          colors[i].green = G;
          colors[i].blue  = B;
          colors[i++].flags = DoRed | DoGreen | DoBlue;
          if (VERBOSE) fprintf(stderr,"%s: LUT[%d] = {%d,%d,%d}\n",
               RoutineName, i, colors[i].red, colors[i].green, colors[i].blue);
         }
       }
     }
    cmap.red_max  = 5; cmap.green_max  = 5; cmap.blue_max  =  5;
    cmap.red_mult = 1; cmap.green_mult = 6; cmap.blue_mult = 36;
   }

  XStoreColors(display, cm, colors, ncolors);
  XInstallColormap(display, cm);

  XChangeProperty(display, DefaultRootWindow(display), 
                  grayscale? XA_RGB_GRAY_MAP :  XA_RGB_DEFAULT_MAP,
                  XA_RGB_COLOR_MAP, 32, PropModeReplace,
                  (char *) &cmap, 
                  sizeof(cmap)/sizeof(long));

  return (ncolors);
 }

int main(argc,argv)
 int argc;
 char *argv[];
 {
  int ArgsParsed = 0;
  Window root;
  int x,y,width, height, border, depth;
  int ncolors;
  Widget toplevel;
  int NoArgc;
  char *NoArgv;

  RoutineName = argv[ArgsParsed++];


  for(; ArgsParsed < argc; )
   {
    if ('-' != argv[ArgsParsed][0]) { usage(); exit(1); }
    switch (argv[ArgsParsed++][1])
     {
      case 'G': if (1 > (argc-ArgsParsed)){usage(); exit(-1);}
                Gamma = atof(argv[ArgsParsed++]);
                break;
      case 'R': if (2 > (argc-ArgsParsed)){usage(); exit(-1);}
                min = atof(argv[ArgsParsed++]);
                max = atof(argv[ArgsParsed++]);
                break;
      case 'c': grayscale =  0; break;
      case 'g': grayscale = -1; break;
      case 'v': VERBOSE = -1; break;
      default:
      case 'h':  { usage(); exit(1); }
     }
   }

  /* Initialize, no args for X */
  NoArgc = 0; NoArgv = (char *)0;
  toplevel = XtInitialize(RoutineName, "StandardMaps", NULL, 0,
                           &NoArgc, NoArgv);

  display = XtDisplay(toplevel);
  window = DefaultRootWindow(display);
  if ((Window)0 == window)
   {
    fprintf(stderr, "%s: Unable to open X window - giving up.\n",RoutineName); 
    exit(1);
   }

  screen = XtScreen(toplevel);
  XGetGeometry(display,window,&root,&x,&y,&width,&height,&border,&depth);
  if (depth != 8) 
   {
    fprintf(stderr,"%s: we don't do shallow windows - giving up.",RoutineName);
    exit(2);
   } 

  /* XGrabServer(display); */  
  ncolors = InitializeColormaps();
  /*  XUngrabServer(display);  */

  fprintf(stderr,
    "%s: %s Color Map Established with %d colors.\n",
          RoutineName, grayscale ? "Grayscale" : "RGB", ncolors);

  XtMainLoop();
  term_handler(); /* will he ever return...     */
  exit(0);        /* I suppose it's possible... */
 }
