/*
  File: Warnock.c
  Author: K.R. Sloan
  Last Modified: 9 October 1989
  Purpose: hidden-surface rendering of triangles

           This is an alternative to "Triangles".  The interface
           is virtually identical.

           DiceTriangles displays the entire collection of triangles,
           with hidden surfaces removed. 

           The user program should allocate an array of TriangleType3D,
           and fill in the geometry and color information for each vertex.

           The Triangles array is used for scratch storage, may be
           extended, and is meaningless after DiceTriangles returns.
          
           If the user program doesn't cull back-facing triangles, we
           do it here (and that ought to provide enough extra space).
           If not, the user program must allocate an array large enough
           to handle some expansion.  10-20% should be enough...

           Actually, this version doesn't need any extra space - but it does
           change things.

           Unless...the global TwoSided indicates that we should
           reverse all normals on back-facing triangles.  This may
           not do what you want - tough.   It's purely kludge-control.
           The tasteful user will not find it necessary.

           Similarly, the global LeftHanded indicates that triangles
           follow the left-hand rule for cross-product calculations.
           Adroit users won't need this feature.

           Yet another global, Facets, indicate that the normal to the
           plane containing the vertices should be used INSTEAD OF the
           vertex normals.

           CAUTION: Facets, TwoSided and LeftHanded are defined in 
           "Triangles.c".
           If you build a program which uses Warnock, but NOT Triangles,
           then you need to define them, and initialize them, perhaps in your
           main program.


           MORE CAUTION: this program is not know to work.....



 */
#include <stdio.h>
#include <math.h>
#include "3D.h"
#include "Lighting.h"
#include "Triangles.h"
#include "Warnock.h"

extern int VERBOSE;
double fabs();

int Looks, ReClassify, DominateS, Guesses, PANIC, PaintOne;

#define EPSILON (0.0000000000001)

static void TraceTriangle(s,T)
 FILE *s;
 TriangleType3D T; 
 {
  fprintf(s,"Vertices:\n[%f %f %f]\n[%f %f %f]\n[%f %f %f]\n",
          T.Vertex[0].x,T.Vertex[0].y,T.Vertex[0].z,
          T.Vertex[1].x,T.Vertex[1].y,T.Vertex[1].z,
          T.Vertex[2].x,T.Vertex[2].y,T.Vertex[2].z);
  fprintf(s,"Normals:\n[%f %f %f]\n[%f %f %f]\n[%f %f %f]\n",
          T.Normal[0].dx,T.Normal[0].dy,T.Normal[0].dz,
          T.Normal[1].dx,T.Normal[1].dy,T.Normal[1].dz,
          T.Normal[2].dx,T.Normal[2].dy,T.Normal[2].dz);
  fprintf(s,"Face Normal:[%f %f %f]\n",
          T.N.dx,T.N.dy,T.N.dz);
 }
/*
   Is the point P inside triangle [A,B,C]?
   If so, fill in the barycentric coordinates and return -1
   If not, return 0
 */
static int BaryCentric(P,A,B,C,a,b,c)
 sgpPointType P,A,B,C;
 double *a, *b, *c;
 {
  double abx, aby, acx, acy, abCROSSac;
  double pax, pay, pbx, pby, pcx, pcy;
  double paCROSSpb, pbCROSSpc, pcCROSSpa; 

  /* check for tiny triangles - nothing is inside them... */
  abx = B.x-A.x; aby  = B.y-A.y;
  acx = C.x-A.x; acy  = C.y-A.y;
  abCROSSac = abx*acy - aby*acx;
  if ((-EPSILON < abCROSSac) && (abCROSSac < EPSILON)) return(0);

  pax = A.x-P.x; pay = A.y-P.y;
  pbx = B.x-P.x; pby = B.y-P.y;
  pcx = C.x-P.x; pcy = C.y-P.y;

  paCROSSpb = pax*pby - pay*pbx;
  pbCROSSpc = pbx*pcy - pby*pcx;
  pcCROSSpa = pcx*pay - pcy*pax;

  *a = pbCROSSpc / abCROSSac;
  *b = pcCROSSpa / abCROSSac;
  *c = 1.0 - *a - *b;

  /* don't insist on being strictly inside */
  return(  (*a>=0.0) && (*b>=0.0) && (*c>0.0) );
 }

/*
   Does the line segment between A1 and A2 intersect
        the line segment between B1 and B2?
   Reference: Newman&Sproull p. 494-5
 */

static int LineSegmentIntersectionP(A1,A2,B1,B2)
 sgpPointType A1,A2,B1,B2;
 {
  double u1,u2,u3, v1,v2,v3;
  double w1,w2,w3;
  double d;

  /* represent the line between A1 and A2 as u = A1 x A2  */
  /* A1 and A2 specify x and y - augment to add w = 1.0, and optimize */
  u1 = A1.y - A2.y;
  u2 = A2.x - A1.x;
  u3 = A1.x*A2.y - A1.y*A2.x;
  /* represent the line between B1 and B2 as v = B1 x B2*/
  /* B1 and B2 specify x and y - augment to add w = 1.0, and optimize */
  v1 = B1.y - B2.y;
  v2 = B2.x - B1.x;
  v3 = B1.x*B2.y - B1.y*B2.x;
  /* the intersection of the lines is  w = u x v */
  w1 = u2*v3 - u3*v2;
  w2 = u3*v1 - u1*v3;
  w3 = u1*v2 - u2*v1;
  /* check for degeneracy, and convert back to 2D */
  if ( (-EPSILON <= w3) && (w3 <= EPSILON) ) return(0); /* can't tell */
  w1 /= w3; w2 /= w3;
  /*
    [w1,w2] is the point of intersection of the lines.
    check BOTH line segments to see that the point is INSIDE
    the segment.

   */
  /* My version... */
  /*
    To fend off numerical instability (especially for horizontal
    and vertical lines, reject an intersection only if it
    is outside BOTH the x and y extents of the segment    
   */
/*
  if (  ( ((w1<=A1.x) && (w1<=A2.x)) || ((w1>=A1.x) && (w1>=A2.x)) )
      &&( ((w2<=A1.y) && (w2<=A2.y)) || ((w2>=A1.y) && (w2>=A2.y)) )
     ) return(0);
  if (  ( ((w1<=B1.x) && (w1<=B2.x)) || ((w1>=B1.x) && (w1>=B2.x)) )
      &&( ((w2<=B1.y) && (w2<=B2.y)) || ((w2>=B1.y) && (w2>=B2.y)) )
     ) return(0);
  return(1);
 */
  /*
    Tony's version
   */
  /*
    check the length from every endpoint to the intersection point
    if they are all less than the length of the segment involved, accept
    if any one is too big, reject
   */
  d = (A2.x-A1.x)*(A2.x-A1.x) + (A2.y-A1.y)*(A2.y-A1.y);
  if (((w1-A1.x)*(w1-A1.x) + (w2-A1.y)*(w2-A1.y)) > d) return(0);
  if (((w1-A2.x)*(w1-A2.x) + (w2-A2.y)*(w2-A2.y)) > d) return(0);
  d = (B2.x-B1.x)*(B2.x-B1.x) + (B2.y-B1.y)*(B2.y-B1.y);
  if (((w1-B1.x)*(w1-B1.x) + (w2-B1.y)*(w2-B1.y)) > d) return(0);
  if (((w1-B2.x)*(w1-B2.x) + (w2-B2.y)*(w2-B2.y)) > d) return(0);
  return(1);
 }

static int PreprocessTriangles(n, T, EyePoint)
 int n;
 TriangleType3D T[];
 PointType3D EyePoint;
 {
  int s,t;
  VectorType3D A, B, C;
  int v;
  double back;

  /* t consumes triangles, s produces them */
  for(s=0,t=0;t<n;t++)
   {
    /*
      cull triangles which have any vertex on wrong side of hither 
      this ought to be fixed to clip the triangle against the hither
      plane - producing either 0, 1, or 2 triangles

      but then, we wouldn't be able to do this in place...
     */

    if (0 == ClipLineHither(&T[t].Vertex[0], &T[t].Vertex[0])) continue;
    if (0 == ClipLineHither(&T[t].Vertex[1], &T[t].Vertex[1])) continue;
    if (0 == ClipLineHither(&T[t].Vertex[2], &T[t].Vertex[2])) continue;

    /* calculate normal */
    
    A.dx = (T[t].Vertex[1].x) - (T[t].Vertex[0].x);
    A.dy = (T[t].Vertex[1].y) - (T[t].Vertex[0].y);
    A.dz = (T[t].Vertex[1].z) - (T[t].Vertex[0].z);
    Normalize3D(&A);

    B.dx = (T[t].Vertex[2].x) - (T[t].Vertex[0].x);
    B.dy = (T[t].Vertex[2].y) - (T[t].Vertex[0].y);
    B.dz = (T[t].Vertex[2].z) - (T[t].Vertex[0].z);
    Normalize3D(&B);

    if (LeftHanded) T[t].N = Cross3D(B,A); else T[t].N = Cross3D(A,B);
    Normalize3D(&T[t].N);


    /*
      cull back-facing (and on-edge) triangles 
      by comparing the face normal with the vector from the EyePoint
      to any vertex

      or, perhaps flip the normals

     */

    C.dx = T[t].Vertex[0].x - EyePoint.x;
    C.dy = T[t].Vertex[0].y - EyePoint.y;
    C.dz = T[t].Vertex[0].z - EyePoint.z;
    Normalize3D(&C);

    if (-EPSILON <= Dot3D(T[t].N, C))
     if (!TwoSided) continue;
     else
      {
       T[t].N.dx = -T[t].N.dx;
       T[t].N.dy = -T[t].N.dy;
       T[t].N.dz = -T[t].N.dz;
       for(v=0;v<3;v++)
        {
         T[t].Normal[v].dx = -T[t].Normal[v].dx;
         T[t].Normal[v].dy = -T[t].Normal[v].dy;
         T[t].Normal[v].dz = -T[t].Normal[v].dz;
	}
      }     

    /* solve the rest of the plane equation */
    A.dx = T[t].Vertex[0].x; A.dy = T[t].Vertex[0].y; A.dz = T[t].Vertex[0].z;
    T[t].d = -Dot3D(A,T[t].N);

    /* does ANY vertex normal face away from us? */
    /* if so, make it perpendicular to the line of sight to that vertex */
    for(v=0;v<3;v++)
     {
      C.dx = T[t].Vertex[v].x - EyePoint.x;
      C.dy = T[t].Vertex[v].x - EyePoint.y;
      C.dz = T[t].Vertex[v].x - EyePoint.z;
      Normalize3D(&C);
      back = Dot3D(C,T[t].Normal[v]);
      if (back>0.0)
       {
        /* it does!, subtract out the away-facing component */
        T[t].Normal[v].dx -= back*C.dx;
        T[t].Normal[v].dy -= back*C.dy;
        T[t].Normal[v].dz -= back*C.dz;
       } 
     }

    /* If we want facets, replace the Vertex normals with the face normal */
    if (Facets) 
     for(v=0;v<3;v++)
      T[t].Normal[v] = T[t].N;

    /* copy forward */
    T[s++] = T[t];
   } 
  return(s); /* number of surviving triangles */
 }

/*
  We've reduced the size of the Window/Viewport so that only one triangle
  needs to be painted.  Let SGP worry about clipping to the viewport!
 */
void OneLeft(t, Window, Viewport)
 TriangleType3D *t;
 sgpRectangleType Window;
 sgpRectangleType Viewport;
 {
  int v;  
  sgpColorType Colors[3];

  PaintOne++;
  sgpSetWindow(Window); sgpSetViewport(Viewport);
  for(v=0;v<3;v++)
   {
    SetSurfaceReflectance(t->Diffuse[v], 
                          t->Specular[v],
                          t->Specularity[v]); 
    Colors[v] = ColorAtPoint(t->Vertex[v], t->Normal[v]);
   } 
  ShadeTriangle3D(t->Vertex, Colors);
 }

/*
  Triangles are kept as follows:
          
                0
                1    Surrounders        
                .
                .                  <--FirstIntersector
                .    Intersectors
                .                  <--LastIntersector
                .    Disjoint
                .                  <--m-1

   ReClassifyIntersectors moves triangles which were "Intersectors"
   of the parent viewport to either the "Surrounder" of "Disjoint"
   classes, by swapping with the First/LastIntersector and moving the
   boundary.  Thus, the parent classifications are still valid (using
   the parent's First/LastIntersector cursors.

   LastIntersector always points at the last Triangle which is either a 
   Surrounder or an Intersector.  If LastIntersector<FirstIntersector,
   then there are NO Intersectors - just Surrounders and Disjoint.

 */
void ReClassifyIntersectors(T,FirstIntersector,LastIntersector,
                            Window)
 TriangleType3D T[];
 int *FirstIntersector, *LastIntersector;
 sgpRectangleType Window;
 {
  int t;
  sgpPointType A0,A1,A2;
  sgpPointType NE,NW,SW,SE;
  double b0,b1,b2;
  int InNE, InNW, InSW, InSE;
  TriangleType3D TempT;
  int DJ;

  NE.x = Window.Right; NE.y = Window.Top;
  NW.x = Window.Left;  NW.y = Window.Top;
  SW.x = Window.Left;  SW.y = Window.Bottom;
  SE.x = Window.Right; SE.y = Window.Bottom;

  for(t = *FirstIntersector; t <= *LastIntersector; t++)
   {
    ReClassify++;
    /*
      does this triangle surround the viewport, miss it, or still
      intersect?

      find out by trying to find the barycentric coordinates of
      the four corners.
     */

    World3DTosgpWorld(T[t].Vertex[0],&A0);
    World3DTosgpWorld(T[t].Vertex[1],&A1);
    World3DTosgpWorld(T[t].Vertex[2],&A2);

    InNE = BaryCentric(NE,A0,A1,A2,&b0,&b1,&b2);
    InNW = BaryCentric(NW,A0,A1,A2,&b0,&b1,&b2);
    InSW = BaryCentric(SW,A0,A1,A2,&b0,&b1,&b2);
    InSE = BaryCentric(SE,A0,A1,A2,&b0,&b1,&b2);

    if (InNE && InNW && InSW && InSE)
     { /* a new surrounder - swap with first intersector and move boundary */
      TempT = T[t];
      T[t] = T[*FirstIntersector];
      T[(*FirstIntersector)++] = TempT;
     }
    else if (!(InNE || InNW || InSW || InSE))
     { /* perhaps disjoint - let's check further */
      DJ = -1;
       /* if any vertex is inside the Window, not disjoint */
      if     (  (  ((Window.Left   <= A0.x) && (A0.x <= Window.Right ))
                 ||((Window.Right  <= A0.x) && (A0.x <= Window.Left  )) )
              &&(  ((Window.Bottom <= A0.y) && (A0.y <= Window.Top   ))
                 ||((Window.Top    <= A0.y) && (A0.y <= Window.Bottom)) )
             )
       DJ = 0;
      else if(  (  ((Window.Left   <= A1.x) && (A1.x <= Window.Right ))
                 ||((Window.Right  <= A1.x) && (A1.x <= Window.Left  )) )
              &&(  ((Window.Bottom <= A1.y) && (A1.y <= Window.Top   ))
                 ||((Window.Top    <= A1.y) && (A1.y <= Window.Bottom)) )
             )
       DJ = 0;
      else if(  (  ((Window.Left   <= A2.x) && (A2.x <= Window.Right ))
                 ||((Window.Right  <= A2.x) && (A2.x <= Window.Left  )) )
              &&(  ((Window.Bottom <= A2.y) && (A2.y <= Window.Top   ))
                 ||((Window.Top    <= A2.y) && (A2.y <= Window.Bottom)) )
             )
       DJ = 0;
       /* if any edge intersects a Window side, then not disjoint */ 
      else if (LineSegmentIntersectionP(A0,A1,NE,NW)) DJ = 0;
      else if (LineSegmentIntersectionP(A1,A2,NE,NW)) DJ = 0;
      else if (LineSegmentIntersectionP(A2,A0,NE,NW)) DJ = 0;
      else if (LineSegmentIntersectionP(A0,A1,NW,SW)) DJ = 0;
      else if (LineSegmentIntersectionP(A1,A2,NW,SW)) DJ = 0;
      else if (LineSegmentIntersectionP(A2,A0,NW,SW)) DJ = 0;
      else if (LineSegmentIntersectionP(A0,A1,SW,SE)) DJ = 0;
      else if (LineSegmentIntersectionP(A1,A2,SW,SE)) DJ = 0;
      else if (LineSegmentIntersectionP(A2,A0,SW,SE)) DJ = 0;
      else if (LineSegmentIntersectionP(A0,A1,SE,NE)) DJ = 0;
      else if (LineSegmentIntersectionP(A1,A2,SE,NE)) DJ = 0;
      else if (LineSegmentIntersectionP(A2,A0,SE,NE)) DJ = 0;
      if (DJ)
       { /* now disjoint - swap with last intersector and move boundary */
        TempT = T[t];
        T[t--] = T[*LastIntersector];     /* back up t to look at this one */
        T[(*LastIntersector)--] = TempT; 
       }
     }
   }
 }

/*
  return the index of the surrounder which is in front at all four
  corners.  If none, return -1;
 */
int DominantSurrounder(T, FirstIntersector, Window)
 TriangleType3D T[];
 int FirstIntersector;
 sgpRectangleType Window;
 {
  int t;
  sgpPointType NE,NW,SW,SE;
  sgpPointType A0,A1,A2;
  double b0,b1,b2;
  double zNE, zNW, zSW, zSE;
  double mNE, mNW, mSW, mSE;
  int    tNE, tNW, tSW, tSE;


  NE.x = Window.Right; NE.y = Window.Top;
  NW.x = Window.Left;  NW.y = Window.Top;
  SW.x = Window.Left;  SW.y = Window.Bottom;
  SE.x = Window.Right; SE.y = Window.Bottom;

  tNE = -1; tNW = -1; tSW = -1; tSE = -1;
  for(t = 0; t < FirstIntersector; t++)
   {
    DominateS++;
    /*
      evaluate the depth of this triangle at the four corners.
      Remember the winners.
     */

    World3DTosgpWorld(T[t].Vertex[0],&A0);
    World3DTosgpWorld(T[t].Vertex[1],&A1);
    World3DTosgpWorld(T[t].Vertex[2],&A2);

    if(!BaryCentric(NE,A0,A1,A2,&b0,&b1,&b2))
     fprintf(stderr,"PANIC: this can't happen\n");
    zNE  = b0*T[t].Vertex[0].z
          +b1*T[t].Vertex[1].z
          +b2*T[t].Vertex[2].z;
    if (!BaryCentric(NW,A0,A1,A2,&b0,&b1,&b2))
     fprintf(stderr,"PANIC: this can't happen\n");
    zNW  = b0*T[t].Vertex[0].z
          +b1*T[t].Vertex[1].z
          +b2*T[t].Vertex[2].z;
    if (!BaryCentric(SW,A0,A1,A2,&b0,&b1,&b2))
     fprintf(stderr,"PANIC: this can't happen\n");
    zSW  = b0*T[t].Vertex[0].z
          +b1*T[t].Vertex[1].z
          +b2*T[t].Vertex[2].z;
    if (!BaryCentric(SE,A0,A1,A2,&b0,&b1,&b2))
     fprintf(stderr,"PANIC: this can't happen\n");
    zSE  = b0*T[t].Vertex[0].z
          +b1*T[t].Vertex[1].z
          +b2*T[t].Vertex[2].z;

    if (0 == t)
     {
      mNE = zNE; mNW = zNW; mSW = zSW; mSE = zSE; 
      tNE = 0;   tNW = 0;   tSW = 0;   tSE = 0;
     }     
    else
     {
      if (zNE<mNE) { mNE = zNE; tNE = t; }
      if (zNW<mNW) { mNW = zNW; tNW = t; }
      if (zSW<mSW) { mSW = zSW; tSW = t; }
      if (zSE<mSE) { mSE = zSE; tSE = t; }
     } 
   }
  if ( (tNE == tNW) && (tNE == tSW) && (tNE == tSE) )
   {
    return(tNE);
   }
  return(-1);
 }

/*
  Guess which triangle to paint by casting a ray into the center of the
  viewport.  Pick the closest PLANE.

  At this point, we are desparate, and will settle for any reasonable 
  answer.  More than likely we are at (or very near)
  the intersection of two triangles -
  it doesn't matter which one we pick.

  There are lots of oddball cases to consider here,.... but we won't. 

  Well, alright, lets' try a bit harder...cast rays into the four corners,
  as well as the center, and pick the closest point that's actually INSIDE
  a triangle.
 */
int Guess(T, LastIntersector, Window)
 TriangleType3D T[];
 int LastIntersector;
 sgpRectangleType Window;
 {
  int t;
  sgpPointType P;
  sgpPointType A0,A1,A2;
  double b0,b1,b2;
  double zP;
  double mP;
  int    tP;

  Guesses++;
  P.x = (Window.Right+Window.Left)/2.0;
  P.y = (Window.Top+Window.Bottom)/2.0;

  tP = -1;
  for(t = 0; t <= LastIntersector; t++)
   {
    World3DTosgpWorld(T[t].Vertex[0],&A0);
    World3DTosgpWorld(T[t].Vertex[1],&A1);
    World3DTosgpWorld(T[t].Vertex[2],&A2);

    if(BaryCentric(P,A0,A1,A2,&b0,&b1,&b2))
     {
      zP  = b0*T[t].Vertex[0].z
           +b1*T[t].Vertex[1].z
           +b2*T[t].Vertex[2].z;

      if ( (-1 ==  tP) || (zP < mP) )
       { mP = zP;  tP = t; }
     }
   }

  P.x = Window.Left; P.y = Window.Bottom;
  for(t = 0; t <= LastIntersector; t++)
   {
    World3DTosgpWorld(T[t].Vertex[0],&A0);
    World3DTosgpWorld(T[t].Vertex[1],&A1);
    World3DTosgpWorld(T[t].Vertex[2],&A2);

    if(BaryCentric(P,A0,A1,A2,&b0,&b1,&b2))
     {
      zP  = b0*T[t].Vertex[0].z
           +b1*T[t].Vertex[1].z
           +b2*T[t].Vertex[2].z;

      if ( (-1 ==  tP) || (zP < mP) )
       { mP = zP;  tP = t; }
     }
   }
  P.x = Window.Left; P.y = Window.Top;
  for(t = 0; t <= LastIntersector; t++)
   {
    World3DTosgpWorld(T[t].Vertex[0],&A0);
    World3DTosgpWorld(T[t].Vertex[1],&A1);
    World3DTosgpWorld(T[t].Vertex[2],&A2);

    if(BaryCentric(P,A0,A1,A2,&b0,&b1,&b2))
     {
      zP  = b0*T[t].Vertex[0].z
           +b1*T[t].Vertex[1].z
           +b2*T[t].Vertex[2].z;

      if ( (-1 ==  tP) || (zP < mP) )
       { mP = zP;  tP = t; }
     }
   }
  P.x = Window.Right; P.y = Window.Top;
  for(t = 0; t <= LastIntersector; t++)
   {
    World3DTosgpWorld(T[t].Vertex[0],&A0);
    World3DTosgpWorld(T[t].Vertex[1],&A1);
    World3DTosgpWorld(T[t].Vertex[2],&A2);

    if(BaryCentric(P,A0,A1,A2,&b0,&b1,&b2))
     {
      zP  = b0*T[t].Vertex[0].z
           +b1*T[t].Vertex[1].z
           +b2*T[t].Vertex[2].z;

      if ( (-1 ==  tP) || (zP < mP) )
       { mP = zP;  tP = t; }
     }
   }
  P.x = Window.Right; P.y = Window.Bottom;
  for(t = 0; t <= LastIntersector; t++)
   {
    World3DTosgpWorld(T[t].Vertex[0],&A0);
    World3DTosgpWorld(T[t].Vertex[1],&A1);
    World3DTosgpWorld(T[t].Vertex[2],&A2);

    if(BaryCentric(P,A0,A1,A2,&b0,&b1,&b2))
     {
      zP  = b0*T[t].Vertex[0].z
           +b1*T[t].Vertex[1].z
           +b2*T[t].Vertex[2].z;

      if ( (-1 ==  tP) || (zP < mP) )
       { mP = zP;  tP = t; }
     }
   }
  return(tP);
 }

/*
  This is the classic Divide-and-Conquer "Warnock Algorithm".  If the
  Window is simple (or small) then do the obvious thing.  Otherwise,
  subdivide into 4 quadrants and hope that THEY are simple...

  Keep track of both the window AND the viewport.  Most routines only
  need to see the window.  The routines which actually draw something
  need the viewport as well.

  xstep, ystep are the smallest increment in x,y that we care about.
 */
void Look(Window, Viewport, T, m, FirstIntersector, LastIntersector,
          xstep, ystep
         )
 sgpRectangleType Window;
 sgpRectangleType Viewport;
 TriangleType3D T[];
 int m;
 int FirstIntersector, LastIntersector;
 double xstep, ystep; 
 {
  sgpRectangleType wNE, wNW, wSW, wSE;
  sgpRectangleType vNE, vNW, vSW, vSE;
  int t;
 
  Looks++;
  /* try to reduce the number of "Intersectors" */ 
  ReClassifyIntersectors(T,&FirstIntersector,&LastIntersector,Window);

  /* if none, or one, then we know what to do */
  if (-1 == LastIntersector) return;  /* no one home */
  if ( 0 == LastIntersector)
   {
    OneLeft(&T[0],Window,Viewport);   /* only one    */
    return;
   }

  /* if no Intersectors, but several Surrounders, see if one dominates */
  if (LastIntersector < FirstIntersector)
   {
    if (0 <= (t = DominantSurrounder(T, FirstIntersector, Window)))
     {
      OneLeft(&T[t],Window,Viewport);
      return;
     }
   }    
  /* Oh well, subdivide, subdivide... */
  if (   (fabs(Window.Right-Window.Left) > xstep)
      && (fabs(Window.Top-Window.Bottom) > ystep) )
   { /* subdivide in x and y */
    wNE = Window;
    wNE.Left   = 0.5*(Window.Right+Window.Left);
    wNE.Bottom = 0.5*(Window.Top+Window.Bottom);
    vNE = Viewport;
    vNE.Left   = 0.5*(Viewport.Right+Viewport.Left);
    vNE.Bottom = 0.5*(Viewport.Top+Viewport.Bottom);
    Look(wNE,vNE, T, m, FirstIntersector, LastIntersector, xstep, ystep);

    wNW = wNE;
    wNW.Left = Window.Left; wNW.Right = wNE.Left;
    vNW = vNE;
    vNW.Left = Viewport.Left; vNW.Right = vNE.Left;
    Look(wNW,vNW, T, m, FirstIntersector, LastIntersector, xstep, ystep);

    wSW = wNW;
    wSW.Bottom = Window.Bottom; wSW.Top = wNW.Bottom;
    vSW = vNW;
    vSW.Bottom = Viewport.Bottom; vSW.Top = vNW.Bottom;
    Look(wSW,vSW, T, m, FirstIntersector, LastIntersector, xstep, ystep);

    wSE = wSW;
    wSE.Right = Window.Right; wSE.Left = wSW.Right;
    vSE = vSW;
    vSE.Right = Viewport.Right; vSE.Left = vSW.Right;
    Look(wSE,vSE, T, m, FirstIntersector, LastIntersector, xstep, ystep);
   }
  else if ((fabs(Window.Right-Window.Left) > xstep))
   { /* subdivide only in x */
    wNE = Window;
    wNE.Left = 0.5*(Window.Right+Window.Left);
    vNE = Viewport;
    vNE.Left = 0.5*(Viewport.Right+Viewport.Left);
    Look(wNE,vNE, T, m, FirstIntersector, LastIntersector, xstep, ystep);

    wNW = wNE;
    wNW.Left = Window.Left; wNW.Right = wNE.Left;
    vNW = vNE;
    vNW.Left = Viewport.Left; vNW.Right = vNE.Left;
    Look(wNW,vNW, T, m, FirstIntersector, LastIntersector, xstep, ystep);
   }
  else if ((fabs(Window.Top-Window.Bottom) > ystep))
   { /* subdivide only in y */
    wNE = Window;
    wNE.Bottom = 0.5*(Window.Top+Window.Bottom);
    vNE = Viewport;
    vNE.Bottom = 0.5*(Viewport.Top+Viewport.Bottom);
    Look(wNE,vNE, T, m, FirstIntersector, LastIntersector, xstep, ystep);

    wSE = wNE;
    wSE.Bottom = Window.Bottom; wSE.Top = wNE.Bottom;
    vSE = vNE;
    vSE.Bottom = Viewport.Bottom; vSE.Top = vNE.Bottom;
    Look(wSE,vSE, T, m, FirstIntersector, LastIntersector, xstep, ystep);
   }
  else
   {
    /*
       sigh - we'e down to the pixel level, and still don't know
       what to do.  So - GUESS!
     */
    t = Guess(T,LastIntersector,Window);
    if(0 <= t)
     OneLeft(&T[t],Window,Viewport);
    else
     {
       /*
         this is ridiculous!  Even guessing doesn't work.
         Let's paint ALL the triangles, and let them fight it out!   
        */
      PANIC++;
      for(t=0;t<=LastIntersector;t++) OneLeft(&T[t],Window,Viewport);
     }
   }
 }

void DiceTriangles(n, T, Tmax)
 int n;
 TriangleType3D T[];
 int Tmax;
 {
  PointType3D EyePoint;
  VectorType3D OpticalAxis, Up;
  int m;
  double y, ylo, ystep, yhi, x, xlo, xstep, xhi;
  sgpStateType sgpState;

  Looks = 0; ReClassify = 0; DominateS = 0; Guesses = 0;
  PANIC = 0; PaintOne = 0;

  GetCamera3D(&EyePoint, &OpticalAxis, &Up);

  m = PreprocessTriangles(n, T, EyePoint);

  if (m <= 0) return; /* nothing to see */

  /*
    find the resolution of the screen, in World Coordinates
   */
  sgpInquire(&sgpState);
  xstep =  ( (sgpState.Window.Right-sgpState.Window.Left)
            *(sgpState.Normalized.Right-sgpState.Normalized.Left))
          /( (sgpState.Viewport.Right-sgpState.Viewport.Left)
            *(sgpState.Physical.Right-sgpState.Physical.Left));
  if (sgpState.Window.Right<sgpState.Window.Left)
   { 
    xlo = sgpState.Window.Right;
    xhi = sgpState.Window.Left;
   }
  else
   {
    xlo = sgpState.Window.Left;
    xhi = sgpState.Window.Right;
   } 
  if (0.0  > xstep) xstep = -xstep;
  if (0.0 == xstep) xstep = 1.0;

  ystep =  ( (sgpState.Window.Top-sgpState.Window.Bottom)
            *(sgpState.Normalized.Top-sgpState.Normalized.Bottom))
          /( (sgpState.Viewport.Top-sgpState.Viewport.Bottom)
            *(sgpState.Physical.Top-sgpState.Physical.Bottom));
  if (sgpState.Window.Bottom<sgpState.Window.Top)
   { 
    ylo = sgpState.Window.Bottom;
    yhi = sgpState.Window.Top;
   }
  else
   {
    ylo = sgpState.Window.Top;
    yhi = sgpState.Window.Bottom;
   } 
  if (0.0  > ystep) ystep = -ystep;
  if (0.0 == ystep) ystep = 1.0;

fprintf(stderr,"calling Look\n");
fprintf(stderr,"Window   = (%e %e %e %e)\n",
 sgpState.Window.Left,sgpState.Window.Bottom,
 sgpState.Window.Right,sgpState.Window.Top);
fprintf(stderr,"Viewport = (%e %e %e %e)\n",
 sgpState.Viewport.Left,sgpState.Viewport.Bottom,
 sgpState.Viewport.Right,sgpState.Viewport.Top); 
fprintf(stderr,"m = %d, Intersectors = %d,%d\n",m,0,m-1); 
fprintf(stderr,"pixel size = (%e %e)\n",xstep,ystep);
  Look(sgpState.Window,      /* window to render */
       sgpState.Viewport,    /* into this viewport */
       T, m,                 /* Triangles, and number */
       0, m-1,               /* first Intersector, last Intersector */
       xstep, ystep          /* pixel dimensions */
      );
  /*
    restore Window/Viewport (Look trashed it)
   */
  sgpSetWindow(sgpState.Window); sgpSetViewport(sgpState.Viewport);
  /*
    report on our good deeds...
   */
  fprintf(stderr,"N              = %10d\n",n);
  fprintf(stderr,"M              = %10d\n",m-1);
  fprintf(stderr,"Looks          = %10d\n",Looks);
  fprintf(stderr,"ReClassify     = %10d\n",ReClassify);
  fprintf(stderr,"DominateS      = %10d\n",DominateS);
  fprintf(stderr,"Guesses        = %10d\n",Guesses);
  fprintf(stderr,"PaintOne       = %10d\n",PaintOne);  
  fprintf(stderr,"PANIC          = %10d\n",PANIC);  
 }
