/*
  File: 2D.c
  Authors: C.A. Curcio
           K.R. Sloan
  Last Modified: 7 December 1989
  Purpose: a set of procedures for two-dimensional transformations,
           implemented via 3x3 matrixes.
 */

#include <stdio.h>
#include <math.h>
#include <2D.h>

/*
  The following routines support 3-dimensional homogeneous coordinate
  transforms (for doing actual 2D transforms--see Newman & Sproull,
  "Principles of interactive computer graphics". 
 */

void Print3DT(s,T)
 FILE *s;
 TransformType3D T;
 {
  int i,j;
  for (i=0;i<3;i++)
   {
    fprintf(s,"| ");
    for (j=0;j<3;j++)
     {
      fprintf(s," %8g ",T[i][j]);
     }
    fprintf(s," |\n");
   }   	
 }
/*
  Project implements  [x',y', 1.0] = [x, y, 1.0] T
 */

void Project (x, y, T)
 double *x, *y;
 TransformType3D T;
 {
  double x0, y0, x1, y1, w1;

  x0 = *x;
  y0 = *y;
  x1 = (x0 * T[0][0]) + (y0 * T[1][0]) + (T[2][0]);
  y1 = (x0 * T[0][1]) + (y0 * T[1][1]) + (T[2][1]);
  w1 = (x0 * T[0][2]) + (y0 * T[1][2]) + (T[2][2]);
  *x = x1/w1;
  *y = y1/w1;
 } /* Project */
 
/* Identity makes T the identity transform matrix. */

void Identity (T)
 TransformType3D T;
 {
  int i, j;
  for (i=0;i<3;i++)
   {
    for (j=0;j<3;j++) T[i][j] = 0.0;
    T[i][i] = 1.0;
   }
 } /* Identity */
  
/*
  The following routines change the current transformation matrix
  T by composing it with the appropriate new transform M to TM = T'.
  First, the compose and copy transforms.
 */

void Compose (T0, T1, T2)
 TransformType3D T0, T1, T2;
 {
  int i,j;
  for (i=0;i<3;i++)
   for (j=0;j<3;j++)
    T2[i][j] = (T0[i][0]*T1[0][j]) + (T0[i][1]*T1[1][j]) + (T0[i][2]*T1[2][j]);
 } /* Compose */
 
void Copy (T0, T1)
 TransformType3D T0, T1;
 {
  int i,j;
  for (i=0;i<3;i++)
   for (j=0;j<3;j++)
    T1[i][j] = T0[i][j];
 } /* Copy */
 
 /* Now the 2D manipulation routines */
 
void RotateZ (angle, T)
 double angle;
 TransformType3D T;  
 {
  TransformType3D T0, T1;
  Identity (T0);
  T0[0][0] = cos (angle);
  T0[1][1] = T0[0][0];
  T0[1][0] = sin (angle);
  T0[0][1] = -T0[1][0];
  Compose (T, T0, T1);
  Copy (T1, T);
 } /* RotateZ */
  
void Translate (x, y, T)
 double x, y;
 TransformType3D T;
 {
  TransformType3D T0, T1;
  Identity (T0);
  T0[2][0] = x;
  T0[2][1] = y;
  Compose (T, T0, T1);
  Copy (T1, T);
 } /* Translate */
  
void Scale (s, T)
 double s;
 TransformType3D T;
 {
  TransformType3D T0, T1;
  Identity (T0);
  T0[0][0] = s;
  T0[1][1] = s;
  Compose (T, T0, T1);
  Copy (T1, T);
 } /* Scale */

/*
  compute a conformal map that takes 
            (x11,y11) -> (x21,y21)
            (x12,y12) -> (x22,y22)
 */
void Similar (x11, y11, x12, y12, x21, y21, x22, y22, T)
 double x11, y11, x12, y12, x21, y21, x22, y22;
 TransformType3D T;
 {
  double dx1, dx2, dy1, dy2, l1, l2, a1, a2, s;
  dx1 = x12-x11; dy1 = y12-y11;
  dx2 = x22-x21; dy2 = y22-y21;
  l1 = dx1*dx1 + dy1*dy1;
  l2 = dx2*dx2 + dy2*dy2;
  a1 = atan2(dy1,dx1);
  a2 = atan2(dy2,dx2);
  s = sqrt(l2/l1); 
  Identity (T);
  Translate (-x11, -y11, T);
  RotateZ (a1-a2, T);
  Scale (s, T);
  Translate (x21, y21, T); 
 } /* Similar */
