/*
  File: Sweeps.c
  Author: K.R. Sloan
  Last Modified: 21 May 1991
  Purpose: Construct a surface, using "sgpTriangle"s.

  INPUT: nx ny
         z00  z01  ... z0nx
         z10  z11  ... z1nx
          .    .   ...  .
          .    .   ...  .
         zny0 zny1 ... znynx

   Output : triangulated surface, suitable for "Render".

  */ 

#include <stdio.h>
#include <3D.h>
#include <Triangles.h>

double atof();

int VERBOSE = 0;
static int SCALEZ = -1;
static double minZ = -1.0;
static double maxZ = 1.0;
static char *RoutineName;
static void usage()
 {
  fprintf(stderr,"Usage is\n\t%s [-h][-v][-z minZ maxZ]\n",
                 RoutineName);
 }

static void FatalError(s)
 {
  fprintf(stderr,"%s: FatalError(%s)\n",RoutineName,s);
 }

static void ReadSweep(Z,nx)
 double Z[];
 int nx; 
 {
  int x;

  if (VERBOSE) fprintf(stderr,"%s: ReadSweep(Z,%d)\n",RoutineName,nx);
  for(x=0;x<nx;x++)
   {
    if (1 != fscanf(stdin," %lf",&Z[x])) FatalError("reading sweep");
    Z[x] = -1.0 + (2.0 * (Z[x]-minZ) / (maxZ-minZ));
    if (SCALEZ) Z[x] /= (double)nx;   
   }
 } 

static void NormalizedCrossProduct(Adx,Ady,Adz, Bdx,Bdy,Bdz, Cdx,Cdy,Cdz)
 double Adx,Ady,Adz, Bdx,Bdy,Bdz;
 double *Cdx,*Cdy,*Cdz;
 {
  double dx,dy,dz,l;
  dx = Ady*Bdz - Adz*Bdx;
  dy = Adz*Bdx - Adx*Bdz;
  dz = Adx*Bdy - Ady*Bdx;

  l = dx*dx + dy*dy + dz*dz;
  if (l < 0.0000000001) l = 0.0000000001;

  *Cdx = dx/l; *Cdy = dy/l; *Cdz = dz/l; 
 }

static void OutputTriangles(Z0,Z1,nx,y,ny)
 double Z0[], Z1[];
 int nx;
 int y, ny; 
 {
  int x0,x1;
  double u0,u1,v0,v1;
  double Ndx, Ndy, Ndz;
  TriangleType3D T;

  if (VERBOSE)
   fprintf(stderr,
          "%s: OutputTriangles(Z0,Z1,%d,%d,%d)\n",RoutineName,nx,y,ny);

  v0 = (double)y/(double)ny; v1 = ((double)y+1.0)/(double)ny;

  for(x0=0,x1=1;x1<nx;x0++,x1++)
   {
    double Ndx00, Ndy00, Ndz00;
    double Ndx01, Ndy01, Ndz01;
    double Ndx10, Ndy10, Ndz10;
    double Ndx11, Ndy11, Ndz11;

    u0 = (double)x0/(double)nx; u1 = (double)x1/(double)nx;

    /* create a quadrilateral patch by painting two triangles */
    /*
       here, we triangulate all quadrilaterals the same.  A possible
       improvement is to find the SHORTER diagonal (in 3D) and
       split that way.  Or perhaps, the longer?  Or perhaps, the pair
       of corners which are closest in Z? or, ...or,... possibilities 
       abound! 
     */


    /* first - estimate normals at all four vertices */
    /*
       at (u0,v0,Z0[x0]) compute Ndx00,Ndy00,Ndz00
       at (u1,v0,Z0[x1]) compute Ndx10,Ndy10,Ndz10
       at (u1,v1,Z1[x1]) compute Ndx11,Ndy11,Ndz11
       at (u0,v1,Z1[x0]) compute Ndx01,Ndy01,Ndz01
     */

    NormalizedCrossProduct(u1-u0, v0-v0, Z0[x1]-Z0[x0],
                           u0-u0, v1-v0, Z1[x0]-Z1[x0],
                           &Ndx00, &Ndy00, &Ndz00);

    NormalizedCrossProduct(u1-u1, v1-v0, Z1[x1]-Z0[x1],
                           u0-u1, v0-v0, Z0[x0]-Z0[x1],
                           &Ndx10, &Ndy10, &Ndz10);

    NormalizedCrossProduct(u0-u1, v1-v1, Z1[x0]-Z1[x1],
                           u1-u1, v0-v1, Z0[x1]-Z1[x1],
                           &Ndx11, &Ndy11, &Ndz11);

    NormalizedCrossProduct(u0-u0, v0-v1, Z0[x0]-Z1[x0],
                           u1-u0, v1-v1, Z1[x1]-Z1[x0],
                           &Ndx01, &Ndy01, &Ndz01);

    /* perhaps these should be blended, somehow - you try it */


    /* The first triangle...*/

    /* vertices */
    T.Vertex[0].x = u0; T.Vertex[0].y = v0; T.Vertex[0].z = Z0[x0];
    T.Vertex[1].x = u1; T.Vertex[1].y = v0; T.Vertex[1].z = Z0[x1];
    T.Vertex[2].x = u0; T.Vertex[2].y = v1; T.Vertex[2].z = Z1[x0];

    T.Normal[0].dx = Ndx00; T.Normal[0].dy = Ndy00; T.Normal[0].dz = Ndz00;
    T.Normal[1].dx = Ndx10; T.Normal[1].dy = Ndy10; T.Normal[1].dz = Ndz10;
    T.Normal[2].dx = Ndx01; T.Normal[2].dy = Ndy01; T.Normal[2].dz = Ndz01;

    /*
      colors - this is a good place to do clever things  such as
               false-color by z.  But, we just paint everything green, with
               moderate specularity
     */
 
    T.Diffuse[0].r = 0.2;
    T.Diffuse[0].g = 0.8; 
    T.Diffuse[0].b = 0.4;

    T.Diffuse[1].r = 0.2;
    T.Diffuse[1].g = 0.8; 
    T.Diffuse[1].b = 0.4;

    T.Diffuse[2].r = 0.2;
    T.Diffuse[2].g = 0.8; 
    T.Diffuse[2].b = 0.4;
 
    T.Specular[0].r = 1.0; T.Specular[0].g = 1.0; T.Specular[0].b = 1.0;
    T.Specular[1].r = 1.0; T.Specular[1].g = 1.0; T.Specular[1].b = 1.0;
    T.Specular[2].r = 1.0; T.Specular[2].g = 1.0; T.Specular[2].b = 1.0;

    T.Specularity[0] = 6.0; T.Specularity[1] = 6.0; T.Specularity[2] = 6.0;

    PrintTriangle(stdout, &T);

    /* the second triangle */

    /* vertices */
    T.Vertex[0].x = u0; T.Vertex[0].y = v1; T.Vertex[0].z = Z1[x0];
    T.Vertex[1].x = u1; T.Vertex[1].y = v0; T.Vertex[1].z = Z0[x1];
    T.Vertex[2].x = u1; T.Vertex[2].y = v1; T.Vertex[2].z = Z1[x1];

    T.Normal[0].dx = Ndx01; T.Normal[0].dy = Ndy01; T.Normal[0].dz = Ndz01;
    T.Normal[1].dx = Ndx10; T.Normal[1].dy = Ndy10; T.Normal[1].dz = Ndz10;
    T.Normal[2].dx = Ndx11; T.Normal[2].dy = Ndy11; T.Normal[2].dz = Ndz11;
 
    /*
      one idea is to make the alternate triangles out of different material
      but - that doesn't always look good, so we leave the material properties
      the same here as in the previous triangle.

      there are lots of opportunites to "play" here.  Go ahead, play!

     */

    PrintTriangle(stdout, &T);

   }

 }

int main(argc, argv)
 int argc; 
 char *argv[];
 {
  int ArgsParsed = 0;
  double *Z[2];
  int nx, ny, y;

  RoutineName = argv[ArgsParsed++];

  while (ArgsParsed < argc)
   {
    if ('-' != argv[ArgsParsed][0]) { usage(); exit (-1); }
    switch (argv[ArgsParsed++][1])
     {
      case 'z': if (2 < (argc - ArgsParsed)) { usage(); exit (-1); }
                minZ = atof(argv[ArgsParsed++]);
                maxZ = atof(argv[ArgsParsed++]);
                break;
      case 'v': VERBOSE = -1; break;
      case 'h': 
      default:
       usage(); exit(-1);
     }
   }   

  if (2 != fscanf(stdin," %d %d", &nx, &ny)) FatalError("reading nx ny");

  if (VERBOSE) fprintf(stderr,"%s: Input is %d by %d\n",RoutineName,nx,ny);

  Z[0] = (double *)malloc(nx*sizeof(double));
  if ((double *)0 == Z[0]) FatalError("allocating Z0");

  Z[1] = (double *)malloc(nx*sizeof(double));
  if ((double *)0 == Z[1]) FatalError("allocating Z1");

  ReadSweep(Z[1],nx);
  for (y=1;y<ny;y++)
   {
    double *t;

    t = Z[0]; Z[0] = Z[1]; Z[1] = t;

    ReadSweep(Z[1],nx);
    OutputTriangles(Z[0],Z[1],nx,y,ny);

   }
  exit(0);
 }


