
/*************************************************************************
**									**
**	MODULE NAME 	:	fit_cubic.c				**
**									**
**	MODULE TYPE 	:	Piecewise cubic fitting code		**
**									**
**	MODULE AUTHOR 	:	Philip J. Schneider			**
**      Last Modified   :       March 18, 1991                          **
**      By              :       Lian Wang                               **
**************************************************************************
*************************************************************************/
#include <stdio.h>
#include <malloc.h>
#include <math.h>
#include "3d.h"

typedef Point3 *BezierCurve;

/* Forward declarations */
extern	void		FitCurve();
static	void		FitCubic();
static	double		*Reparameterize();
static	double		NewtonRaphsonRootFind();
static	Point3		Bezier();
static	double 		B0(), B1(), B2(), B3();
static	Vector3		ComputeLeftTangent();
static	Vector3		ComputeRightTangent();
static	Vector3		ComputeCenterTangent();
static	Vector3		ComputeInitialTangent();
static	double		ComputeMaxError();
static	double		*ChordLengthParameterize();
static	BezierCurve	GenerateBezier();
extern	Vector3		V3AddII();
extern	Vector3		V3ScaleII();
extern	Vector3		V3Sub();

#define MAXPOINTS	1000		/*  The most points you can have   */

#ifdef TESTMODE
/*
 *  main:
 *	Example of how to use the curve-fitting code.  Given an array
 *  	of points and a tolerance (squared error between points and 
 *	fitted curve), the algorithm will generate a piecewise
 *	cubic Bezier representation that approximates the points.
 *	When a cubic is generated, the routine "DrawBezierCurve"
 *	is called, which outputs the Bezier curve just created
 *	(arguments are the degree and the control points, respectively).
 *	Users will have to implement this function themselves 
 *	(e.g., X11 drawing calls, PostScript (tm) calls, ascii output, etc.)
 *
 */	
main()
{
    static Point3 d[7] = {	/*  Digitized points */
	{ 0.0, 0.0, 1.0 },
	{ 0.0, 0.5, 1.0},
	{ 1.1, 1.4, 1.0},
	{ 2.1, 1.6, 1.0},
	{ 3.2, 1.1, 1.0 },
	{ 4.0, 0.2, 1.0 },
	{ 4.0, 0.0, 1.0 },
    };
    double	error = 4.0;	/*  Squared error */
    FitCurve(d, 7, error);	/*  Fit the Bezier curves */
}
#endif /* TESTMODE */


/*
 *  FitCurve :
 *  	Fit a Bezier curve to a set of digitized points
 * 
 */
void FitCurve(d, nPts, error)
    Point3	*d;		/*  Array of digitized points	*/
    int		nPts;		/*  Number of digitized points	*/
    double	error;		/*  User-defined error squared	*/
{
#define OPEN_CURVE

#ifdef OPEN_CURVE
    Vector3	tHat1, tHat2;	/*  Unit tangent vectors at endpoints */

    /*
     *  Estimate tangent vectors at the endpoints of the curve
     *  Note that tangents face "inward", i.e., towards the
     *  interior of the set of sampled points
     */
    tHat1 = ComputeLeftTangent(d, 0);
    tHat2 = ComputeRightTangent(d, nPts - 1);


    /*
     *  Begin recursive curve-fitting algorithm
     */
    FitCubic(d, 	/*  Array of points				*/
	     0,		/*  First point in region of curve to be fit	*/
	     nPts - 1,	/*  Last    "    "    "    "    "   "  "  "	*/
	     tHat1,	/*  Beginning endpoint tangent vector		*/
	     tHat2,	/*  Last endpoint tangent vector		*/
	     error);	/*  Squared error term (in arbitrary units	*/

#else /* CLOSED_CURVE */

    Vector3	tHat1, tHat2;	/*  Unit tangent vectors at endpoints */
    Point3	*dNew;

    /*
     *  Compute endpoint tangents as if curve is closed --
     *  "beginning" and "end" vectors are just reversals of each other
     *
     */
    tHat1 = ComputeInitialTangent(d, nPts);
    tHat2 = tHat1;
    V3Negate(&tHat2);

    /*
     *  Make a new point vector, and make another copy of the first point
     *  at the end of the list (this makes the rest of the code work the
     *  same for closed and open curves :-)
     */
    nPts++;
    dNew = malloc((nPts) * sizeof(Point3));
    bcopy((char *)d, (char *)dNew, (nPts-1) * sizeof(Point3));
    dNew[nPts-1] = d[0];

    /*
     *  See above
     */
    FitCubic(dNew, 0, nPts - 1, tHat1, tHat2, error);

#endif /* OPEN_CURVE */
}



/*
 *  FitCubic :
 *  	Fit a Bezier curve to a (sub)set of digitized points
 *
 */
static void FitCubic(d, first, last, tHat1, tHat2, error)
    Point3	*d;		/*  Array of digitized points		    */
    int		first, last;	/*  Indices of first and last pts in region */
    Vector3	tHat1, tHat2;	/*  Unit tangent vectors at endpoints	    */
    double	error;		/*  User-defined error squared		    */
{
    BezierCurve	bezCurve;	/*  Control points of fitted Bezier curve */
    double	*u;		/*  Parameter values for points		  */
    double	*uPrime;	/*  Improved parameter values		  */
    double	maxError;	/*  Maximum fitting error		  */
    int		splitPoint;	/*  Point to split point set at		  */
    int		nPts;		/*  Number of points in subset		  */
    double	iterationError;	/*  Error below which you try iterating	  */
    int		maxIterations = 4; /*  Max times to try iterating	  */
    Vector3	tHatCenter;     /*  Unit tangent vector at splitPoint	  */
    int		i;		

    iterationError = error * error;
    nPts = last - first + 1;

    /*  Use heuristic if region only has two points in it */
    if (nPts == 2) {
	double	dist = V3DistanceBetween2Points(&d[last], &d[first]) / 3.0;

	bezCurve = (Point3 *)malloc(4 * sizeof(Point3));
	bezCurve[0] = d[first];
	bezCurve[3] = d[last];
	V3Add(&bezCurve[0], V3Scale(&tHat1, dist), &bezCurve[1]);
	V3Add(&bezCurve[3], V3Scale(&tHat2, dist), &bezCurve[2]);
	DrawBezierCurve(3, bezCurve);
	return;
    }

    /*  Parameterize points, and attempt to fit curve */
    u = ChordLengthParameterize(d, first, last);
    bezCurve = GenerateBezier(d, first, last, u, tHat1, tHat2);

    /*  Find max deviation of points to fitted curve */
    maxError = ComputeMaxError(d, first, last, bezCurve, u, &splitPoint);
    if (maxError < error) {
	/*
	 *  "bezCurve" contains the control points -- this is
	 *   where to put the output routine for "real" applications
	 */
	DrawBezierCurve(3, bezCurve);
	return;
    }

    /*  If error not too large, try some reparameterization and iteration */
    if (maxError < iterationError) {
	for (i = 0; i < maxIterations; i++) {
	    uPrime = Reparameterize(d, first, last, u, bezCurve);
	    bezCurve = GenerateBezier(d, first, last, uPrime, tHat1, tHat2);
	    maxError = ComputeMaxError(d, first, last,
				       bezCurve, uPrime, &splitPoint);
	    if (maxError < error) {
		DrawBezierCurve(3, bezCurve);
		return;
	    }
	    (void)free((void *)u);
	    u = uPrime;
	}
    }

    /*  Fitting failed -- split at max error point and fit recursively */
    tHatCenter = ComputeCenterTangent(d, splitPoint);
    FitCubic(d, first, splitPoint, tHat1, tHatCenter, error);
    V3Negate(&tHatCenter);
    FitCubic(d, splitPoint, last, tHatCenter, tHat2, error);
}


/*
 *  GenerateBezier :
 *  	Use least-squares method to find Bezier control points for region.
 *
 */
static BezierCurve  GenerateBezier(d, first, last, uPrime, tHat1, tHat2)
    Point3	*d;			/*  Array of digitized points	*/
    int		first, last;		/*  Indices defining region	*/
    double	*uPrime;		/*  Parameter values for region */
    Vector3	tHat1, tHat2;		/*  Unit tangents at endpoints	*/
{
    int 	i;
    Vector3 	A[MAXPOINTS][2];	/* Precomputed rhs for eqn	*/
    int 	nPts;			/* Number of pts in sub-curve	*/
    double 	C[2][2];		/* Matrix C			*/
    double 	X[2];			/* Matrix X			*/
    double 	det_C0_C1,		/* Determinants of matrices	*/
    	   	det_C0_X,
	   	det_X_C1;
    double 	alpha_l,		/* Alpha values, left and right	*/
    	   	alpha_r;
    Vector3 	tmp;			/* Utility variable		*/
    BezierCurve	bezCurve;		/* RETURN bezier curve ctl pts	*/

    bezCurve = (Point3 *)malloc(4 * sizeof(Point3));
    nPts = last - first + 1;
 
    /* Compute the A's	*/
    for (i = 0; i < nPts; i++) {
	Vector3		v1, v2;
	v1 = tHat1;
	v2 = tHat2;
	V3Scale(&v1, B1(uPrime[i]));
	V3Scale(&v2, B2(uPrime[i]));
	A[i][0] = v1;
	A[i][1] = v2;
    }

    /* Create the C and X matrices	*/
    C[0][0] = 0.0;
    C[0][1] = 0.0;
    C[1][0] = 0.0;
    C[1][1] = 0.0;
    X[0]    = 0.0;
    X[1]    = 0.0;

    for (i = 0; i < nPts; i++) {
        C[0][0] += V3Dot(&A[i][0], &A[i][0]);
	C[0][1] += V3Dot(&A[i][0], &A[i][1]);
/*	C[1][0] += V3Dot(&A[i][0], &A[i][1]);*/	
	C[1][0] = C[0][1];
	C[1][1] += V3Dot(&A[i][1], &A[i][1]);

	tmp = V3Sub(d[first + i],
	        V3AddII(
	          V3ScaleII(d[first], B0(uPrime[i])),
		    V3AddII(
		      V3ScaleII(d[first], B1(uPrime[i])),
		        V3AddII(
	                  V3ScaleII(d[last], B2(uPrime[i])),
	                    V3ScaleII(d[last], B3(uPrime[i]))))));
	

	X[0] += V3Dot(&A[i][0], &tmp);
	X[1] += V3Dot(&A[i][1], &tmp);
    }

    /* Compute the determinants of C and X	*/
    det_C0_C1 = C[0][0] * C[1][1] - C[1][0] * C[0][1];
    det_C0_X  = C[0][0] * X[1]    - C[0][1] * X[0];
    det_X_C1  = X[0]    * C[1][1] - X[1]    * C[0][1];

    /* Finally, derive alpha values	*/
    if (det_C0_C1 == 0.0) {
	det_C0_C1 = (C[0][0] * C[1][1]) * 10e-12;
    }
    alpha_l = det_X_C1 / det_C0_C1;
    alpha_r = det_C0_X / det_C0_C1;

    /*  If alpha negative, use the Wu/Barsky heuristic (see text) */
    if (alpha_l < 0.0 || alpha_r < 0.0) {
	double	dist = V3DistanceBetween2Points(&d[last], &d[first]) / 3.0;

	bezCurve[0] = d[first];
	bezCurve[3] = d[last];
	V3Add(&bezCurve[0], V3Scale(&tHat1, dist), &bezCurve[1]);
	V3Add(&bezCurve[3], V3Scale(&tHat2, dist), &bezCurve[2]);
	return (bezCurve);
    }

    /*  First and last control points of the Bezier curve are		*/
    /*  positioned exactly at the first and last data points		*/
    /*  Control points 1 and 2 are positioned an alpha distance out 	*/
    /*  on the tangent vectors, left and right, respectively		*/
    bezCurve[0] = d[first];
    bezCurve[3] = d[last];
    V3Add(&bezCurve[0], V3Scale(&tHat1, alpha_l), &bezCurve[1]);
    V3Add(&bezCurve[3], V3Scale(&tHat2, alpha_r), &bezCurve[2]);
    return (bezCurve);
}


/*
 *  Reparameterize:
 *	Given set of points and their parameterization, try to find
 *      a better parameterization.
 *
 */
static double *Reparameterize(d, first, last, u, bezCurve)
    Point3	*d;			/*  Array of digitized points	*/
    int		first, last;		/*  Indices defining region	*/
    double	*u;			/*  Current parameter values	*/
    BezierCurve	bezCurve;		/*  Current fitted curve	*/
{
    int 	nPts = last-first+1;	
    int 	i;
    double	*uPrime;		/*  New parameter values	*/

    uPrime = (double *)malloc(nPts * sizeof(double));
    for (i = first; i <= last; i++) {
	uPrime[i-first] = NewtonRaphsonRootFind(bezCurve, d[i], u[i-first]);
    }
    return (uPrime);
}


/*
 *  NewtonRaphsonRootFind :
 *	Use Newton-Raphson iteration to find better root.
 */
static double NewtonRaphsonRootFind(Q, P, u)
    BezierCurve	Q;			/*  Current fitted curve	*/
    Point3 	P;			/*  Digitized point		*/
    double 	u;			/*  Parameter value for "P"	*/
{
    double 	numerator, denominator;
    Point3 	Q1[3], Q2[2];		/*  Q' and Q''			*/
    Point3	Q_u, Q1_u, Q2_u;	/*  u evaluated at Q, Q', & Q''	*/
    double 	uPrime;			/*  Improved u			*/
    int 	i;
    
    /* Compute Q(u)	*/
    Q_u = Bezier(3, Q, u);
    
    /* Generate control vertices for Q'	*/
    for (i = 0; i <= 2; i++) {
	Q1[i].x = (Q[i+1].x - Q[i].x) * 3.0;
	Q1[i].y = (Q[i+1].y - Q[i].y) * 3.0;
	Q1[i].z = (Q[i+1].z - Q[i].z) * 3.0;
    }
    
    /* Generate control vertices for Q'' */
    for (i = 0; i <= 1; i++) {
	Q2[i].x = (Q1[i+1].x - Q1[i].x) * 2.0;
	Q2[i].y = (Q1[i+1].y - Q1[i].y) * 2.0;
	Q2[i].z = (Q1[i+1].z - Q1[i].z) * 2.0;
    }
    
    /* Compute Q'(u) and Q''(u)	*/
    Q1_u = Bezier(2, Q1, u);
    Q2_u = Bezier(1, Q2, u);
    
    /* Compute f(u)/f'(u) */
    numerator = (Q_u.x - P.x) * (Q1_u.x) + (Q_u.y - P.y) * (Q1_u.y) 
                + (Q_u.z - P.z) * (Q1_u.z);
    denominator = (Q1_u.x) * (Q1_u.x) + (Q1_u.y) * (Q1_u.y) +
                  (Q1_u.z) * (Q1_u.z) + (Q_u.x - P.x) * (Q2_u.x) 
                  + (Q_u.y - P.y) * (Q2_u.y) + (Q_u.z - P.z)*(Q2_u.z);
    
    /* u = u - f(u)/f'(u) */
    uPrime = u - (numerator/denominator);
    return (uPrime);
}

			       
/*
 *  Bezier :
 *  	Evaluate a Bezier curve at a particular parameter value
 * 
 */
static Point3 Bezier(degree, V, t)
    int		degree;		/* The degree of the bezier curve	*/
    Point3 	*V;		/* Array of control points		*/
    double 	t;		/* Parametric value to find point for	*/
{
    int 	i, j;		
    Point3 	Q;	        /* Point on curve at parameter t	*/
    Point3 	*Vtemp;		/* Local copy of control points		*/

    /* Copy array	*/
    Vtemp = (Point3 *)malloc((unsigned)((degree+1) * sizeof(Point3)));
    for (i = 0; i <= degree; i++) {
	Vtemp[i] = V[i];
    }

    /* Triangle computation	*/
    for (i = 1; i <= degree; i++) {	
	for (j = 0; j <= degree-i; j++) {
	    Vtemp[j].x = (1.0 - t) * Vtemp[j].x + t * Vtemp[j+1].x;
	    Vtemp[j].y = (1.0 - t) * Vtemp[j].y + t * Vtemp[j+1].y;
	    Vtemp[j].z = (1.0 - t) * Vtemp[j].z + t * Vtemp[j+1].z;
	}
    }

    Q = Vtemp[0];
    (void)free((void *)Vtemp);
    return Q;
}


/*
 *  B0, B1, B2, B3 :
 *	Bezier multipliers
 */
static double B0(u)
    double	u;
{
    double tmp = 1.0 - u;
    return (tmp * tmp * tmp);
}

static double B1(u)
    double	u;
{
    double tmp = 1.0 - u;
    return (3 * u * (tmp * tmp));
}

static double B2(u)
    double	u;
{
    double tmp = 1.0 - u;
    return (3 * u * u * tmp);
}

static double B3(u)
    double	u;
{
    return (u * u * u);
}



/*
 *  ComputeLeftTangent, ComputeRightTangent, ComputeCenterTangent :
 *	Approximate unit tangents at endpoints and "center" of digitized curve
 */
static Vector3 ComputeLeftTangent(d, end)
    Point3	*d;		/*  Digitized points			*/
    int		end;		/*  Index to "left" end of region	*/
{
    Vector3	tHat1;
    tHat1 = V3Sub(d[end+1], d[end]);
    tHat1 = *V3Normalize(&tHat1);
    return tHat1;
}

static Vector3 ComputeRightTangent(d, end)
    Point3	*d;		/*  Digitized points			*/
    int		end;		/*  Index to "right" end of region	*/
{
    Vector3	tHat2;
    tHat2 = V3Sub(d[end-1], d[end]);
    tHat2 = *V3Normalize(&tHat2);
    return tHat2;
}

static Vector3 ComputeCenterTangent(d, center)
    Point3	*d;		/*  Digitized points			*/
    int		center;		/*  Index to point inside region	*/
{
    Vector3	V1, V2, tHatCenter;

    V1 = V3Sub(d[center-1], d[center]);
    V2 = V3Sub(d[center], d[center+1]);
    tHatCenter.x = (V1.x + V2.x)/2.0;
    tHatCenter.y = (V1.y + V2.y)/2.0;
    tHatCenter.z = (V1.z + V2.z)/2.0;
    tHatCenter = *V3Normalize(&tHatCenter);

    return tHatCenter;
}


static Vector3 ComputeInitialTangent(d, nPts)
    Point3	*d;		/*  Digitized points */
    int		nPts;		/*  Number of points */
{
    Vector3	V1, V2, tHatInitial;

    V1 = V3Sub(d[0], d[nPts - 1]);
    V2 = V3Sub(d[1], d[0]);
    tHatInitial.x = (V1.x + V2.x)/2.0;
    tHatInitial.y = (V1.y + V2.y)/2.0;
    tHatInitial.y = (V1.z + V2.z)/2.0;
    tHatInitial = *V3Normalize(&tHatInitial);
    return tHatInitial;
}




/*
 *  ChordLengthParameterize :
 *	Assign parameter values to digitized points 
 *	using relative distances between points.
 */
static double *ChordLengthParameterize(d, first, last)
    Point3	*d;			/*  Array of digitized points	*/
    int		first, last;		/*  Indices defining region	*/
{
    int		i;	
    double	*u;			/*  Parameterization		*/

    u = (double *)malloc((unsigned)(last-first+1) * sizeof(double));

    u[0] = 0.0;
    for (i = first+1; i <= last; i++) {
	u[i-first] = u[i-first-1] +
	  V3DistanceBetween2Points(&d[i], &d[i-1]);
    }

    for (i = first + 1; i <= last; i++) {
	u[i-first] = u[i-first] / u[last-first];
    }

    return(u);
}



/*
 *  ComputeMaxError :
 *	Find the maximum squared distance of digitized points
 *	to fitted curve.
*/
static double ComputeMaxError(d, first, last, bezCurve, u, splitPoint)
    Point3	*d;			/*  Array of digitized points	*/
    int		first, last;		/*  Indices defining region	*/
    BezierCurve	bezCurve;		/*  Fitted Bezier curve		*/
    double	*u;			/*  Parameterization of points	*/
    int		*splitPoint;		/*  Point of maximum error	*/
{
    int		i;
    double	maxDist;		/*  Maximum error		*/
    double	dist;			/*  Current error		*/
    Point3	P;			/*  Point on curve		*/
    Vector3	v;			/*  Vector from point to curve	*/

    *splitPoint = (last - first + 1)/2;
    maxDist = 0.0;
    for (i = first + 1; i < last; i++) {
	P = Bezier(3, bezCurve, u[i-first]);
	v = V3Sub(P, d[i]);
	dist = V3SquaredLength(&v);
	if (dist >= maxDist) {
	    maxDist = dist;
	    *splitPoint = i;
	}
    }

    return (maxDist);
}

