;; erc.el --- An Emacs Internet Relay Chat client

;; Author: Alexander L. Belikoff (alexander@belikoff.net)
;; Contributors: Sergey Berezin (sergey.berezin@cs.cmu.edu),
;;               Mario Lang (mlang@delysid.org),
;;               Alex Schroeder (alex@gnu.org)
;;               Andreas Fuchs (afs@void.at)
;;               Gergely Nagy (algernon@midgard.debian.net)
;;               David Edmondson (dme@dme.org)
;; Maintainer: Mario Lang (mlang@delysid.org)
;; Keywords: IRC, chat, client, Internet

;; Copyright (C) 1997  Alexander L. Belikoff
;; Copyright (C) 2001, 2002, 2003, 2004  Mario Lang

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2 of the License, or
;; (at your option) any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program; if not, write to the Free Software
;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

;;; Commentary:

;; ERC is an IRC client for Emacs.

;; For more information, see the following URLs:
;; * http://sf.net/projects/erc/
;; * http://www.emacswiki.org/cgi-bin/wiki.pl?EmacsIRCClient

;; Jul-26-2001. erc.el is now in CVS on SourceForge.  I invite everyone
;; who wants to hack it to contact me <mlang@delysid.org> in order to
;; get write access on the CVS.

;; Installation:

;; Put erc.el in your load-path, and put (require 'erc) in your .emacs.

;; Configuration:

;; Use M-x customize-group RET erc RET to get an overview
;; of all the variables you can tweak.

;; Usage:

;; To connect to an IRC server, do
;;
;; M-x erc-select RET
;;
;; After you are connected to a server, you can use C-h m or have a look at
;; the IRC menu.

;;; History:
;;

;;; Code:

(require 'cl)
(require 'font-lock)
(require 'format-spec)
(require 'pp)
(require 'thingatpt)
(require 'erc-compat)

(defconst erc-version-string "Version 4.0 $Revision: 1.657 $"
  "ERC version.  This is used by function `erc-version'.")

(defvar erc-official-location
  "http://erc.sf.net (comments mailto://mlang@delysid.org)"
  "Location of the ERC client on the Internet.")

(defgroup erc nil
  "Emacs Internet Relay Chat client."
  :link '(url-link "http://www.emacswiki.org/cgi-bin/wiki.pl?EmacsIRCClient")
  :prefix "erc-"
  :group 'processes)

;; tunable connection and authentication parameters

(defcustom erc-server nil
  "IRC server to use.
See function `erc-compute-server' for more details on connection
parameters and authentication."
  :group 'erc
  :type '(choice (const nil) string))

(defcustom erc-port nil
  "IRC port to use."
  :group 'erc
  :type '(choice (const nil) number string))

(defcustom erc-nick nil
  "Nickname to use.

Can be either a string, or a list of strings.
In the latter case, if the first nick in the list is already in use,
other nicks are tried in the list order.

See function `erc-compute-nick' for more details on connection
parameters and authentication."
  :group 'erc
  :type '(choice (const nil)
		 (string :tag "Nickname")
		 (repeat string)))

(defcustom erc-nick-uniquifier "`"
  "The character to append to the nick if it is already in use."
  :group 'erc
  :type 'string)

(defcustom erc-manual-set-nick-on-bad-nick-p nil
  "If the nickname you chose isn't available, ERC should not automatically
attempt to set another nickname.  You can manually set another nickname with
the /NICK command."
  :group 'erc
  :type 'boolean)

(defcustom erc-user-full-name nil
  "User full name.

See function `erc-compute-full-name' for more details on connection
parameters and authentication."
  :group 'erc
  :type '(choice (const nil) string function)
  :set (lambda (sym val)
	 (if (functionp val)
	     (set sym (funcall val))
	   (set sym val))))

(defvar erc-password nil
  "ERC password to use in authentication (not necessary).")

(defcustom erc-user-mode nil
  "Initial user modes to be set after a connection is established."
  :group 'erc
  :type '(choice (const nil) string function))


(defcustom erc-prompt-for-password t
  "Asks before using the default password, or whether to enter a new one."
  :group 'erc
  :type 'boolean)

(defcustom erc-warn-about-blank-lines t
  "Warn the user if they attempt to send a blank line."
  :group 'erc
  :type 'boolean)

(defcustom erc-send-whitespace-lines nil
  "If set to non-nil, send lines consisting of only whitespace."
  :group 'erc
  :type 'boolean)

(defcustom erc-hide-prompt nil
  "If non-nil, do not display the prompt for commands.

\(A command is any input starting with a '/').

See also the variables `erc-prompt' and `erc-command-indicator'."
  :group 'erc
  :type 'boolean)

;; tunable GUI stuff

(defcustom erc-show-my-nick t
  "If non-nil, display one's own nickname when sending a message.

If non-nil, \"<nickname>\" will be shown.
If nil, only \"> \" will be shown."
  :group 'erc
  :type 'boolean)

(define-widget 'erc-message-type 'set
  "A set of standard IRC Message types."
  :args '((const "JOIN")
	  (const "KICK")
	  (const "NICK")
	  (const "PART")
	  (const "QUIT")
	  (repeat :tag "Others" (string :tag "IRC Message Type"))))

(defcustom erc-hide-list nil
  "*List of IRC type messages to hide.
A typical value would be '(\"JOIN\" \"PART\" \"QUIT\")."
  :group 'erc
  :type 'erc-message-type)

(defvar erc-session-server nil
  "The server name used to connect to for this session.")
(make-variable-buffer-local 'erc-session-server)

(defvar erc-session-port nil
  "The port used to connect to.")
(make-variable-buffer-local 'erc-session-port)

(defvar erc-session-password nil
  "The password used for the current session.")
(make-variable-buffer-local 'erc-session-password)

(defvar erc-announced-server-name nil
  "The name the server announced to use.")
(make-variable-buffer-local 'erc-announced-server-name)

(defvar erc-server-parameters nil

  "Alist listing the supported server parameters.

This is only set if the server sends 005 messages saying what is
supported on the server.

Entries are of the form:
  (PARAMETER . VALUE)
or
  (PARAMETER) if no value is provided.

Some examples of possible parameters sent by servers:
CHANMODES=b,k,l,imnpst - list of supported channel modes
CHANNELLEN=50 - maximum length of channel names
CHANTYPES=#&!+ - supported channel prefixes
CHARMAPPING=rfc1459 - character mapping used for nickname and channels
KICKLEN=160 - maximum allowed kick message length
MAXBANS=30 - maximum number of bans per channel
MAXCHANNELS=10 - maximum number of channels allowed to join
NETWORK=EFnet -  the network identifier
NICKLEN=9 - maximum allowed length of nicknames
PREFIX=(ov)@+ - list of channel modes and the user prefixes if user has mode
RFC2812 - server supports RFC 2812 features
SILENCE=10 - supports the SILENCE command, maximum allowed number of entries
TOPICLEN=160 - maximum allowed topic length
WALLCHOPS - supports sending messages to all operators in a channel")
(make-variable-buffer-local 'erc-server-parameters)


(defcustom erc-disconnected-hook nil
  "Run this hook with arguments (NICK IP REASON) when disconnected.
This happens before automatic reconnection.  Note, that `erc-server-QUIT-hook'
might not be run when we disconnect, simply because we do not necessarily
receive the QUIT event."
  :group 'erc-hooks
  :type 'hook)

(defcustom erc-complete-functions nil
  "These functions get called when the user hits TAB in ERC.
Each function in turn is called until one returns non-nil to
indicate it has handled the input."
  :group 'erc-hooks
  :type 'hook)

(defcustom erc-join-hook nil
  "Hook run when we join a channel.  Hook functions are called
without arguments, with the current buffer set to the buffer of
the new channel.

See also `erc-server-JOIN-hook', `erc-part-hook'."
  :group 'erc-hooks
  :type 'hook)

(defcustom erc-quit-hook nil
  "Hook run when processing a quit command directed at our nick.

The hook receives one argument, the current PROCESS.
See also `erc-server-QUIT-hook' and `erc-disconnected-hook'."
  :group 'erc-hooks
  :type 'hook)

(defcustom erc-part-hook nil
  "Hook run when processing a PART message directed at our nick.

The hook receives one argument, the current BUFFER.
See also `erc-server-QUIT-hook', `erc-quit-hook' and
`erc-disconnected-hook'."
  :group 'erc-hooks
  :type 'hook)

(defcustom erc-kick-hook nil
  "Hook run when processing a KICK message directed at our nick.

The hook receives one argument, the current BUFFER.
See also `erc-server-PART-hook' and `erc-part-hook'."
  :group 'erc-hooks
  :type 'hook)

(defcustom erc-nick-changed-functions nil
  "List of functions run when your nick was successfully changed.

Each function should accept two arguments, NEW-NICK and OLD-NICK."
  :group 'erc-hooks
  :type 'hook)

(defcustom erc-connect-pre-hook '(erc-initialize-log-marker)
  "Hook called just before `erc' calls `erc-connect'.
Functions are run in the buffer-to-be."
  :group 'erc-hooks
  :type 'hook)

(defvar erc-process nil
  "The process object of the corresponding server connection.")
(make-variable-buffer-local 'erc-process)


(defvar erc-connected nil
  "Non-nil if the `current-buffer' is associated with an open IRC connection.
This variable is buffer-local.")
(make-variable-buffer-local 'erc-connected)

(defvar erc-channel-users nil
  "A hash table of members in the current channel, which
associates nicknames with cons cells of the form:
\(USER . MEMBER-DATA) where USER is a pointer to an
erc-server-user struct, and MEMBER-DATA is a pointer to an
erc-channel-user struct.")
(make-variable-buffer-local 'erc-channel-users)

(defvar erc-server-users nil
  "A hash table of users on the current server, which associates
nicknames with erc-server-user struct instances.")
(make-variable-buffer-local 'erc-server-users)

(defun erc-downcase (string)
  "Convert STRING to IRC standard conforming downcase."
  (let ((s (downcase string))
	(c '((?\[ . ?\{)
	     (?\] . ?\})
	     (?\\ . ?\|)
	     (?~  . ?^))))
    (save-match-data
      (while (string-match "[]\\[~]" s)
	(aset s (match-beginning 0)
	      (cdr (assq (aref s (match-beginning 0)) c)))))
    s))

(defstruct (erc-server-user (:type vector) :named)
  
  ;; User data
  nickname host login full-name info

  ;; Buffers
  ;;
  ;; This is an alist of the form (BUFFER . CHANNEL-DATA), where
  ;; CHANNEL-DATA is either nil or an erc-channel-user struct.
  (buffers nil)
  )

(defstruct (erc-channel-user (:type vector) :named)
  
  op voice
  
  ;; Last message time (in the form of the return value of
  ;; (current-time)
  ;;
  ;; This is useful for ordered name completion.
  (last-message-time nil))

(defsubst erc-get-channel-user (nick)
  "Finds the (USER . CHANNEL-DATA) element corresponding to NICK
in the current buffer's `erc-channel-users' hash table."
  (gethash (erc-downcase nick) erc-channel-users))

(defsubst erc-get-server-user (nick)
  "Finds the USER corresponding to NICK in the current server's
`erc-server-users' hash table."
  (with-current-buffer (process-buffer erc-process)
    (gethash (erc-downcase nick) erc-server-users)))

(defsubst erc-add-server-user (nick user)
  "This function is for internal use only.

Adds USER with nickname NICK to the `erc-server-users' hash table."
  (with-current-buffer (process-buffer erc-process)
    (puthash (erc-downcase nick) user erc-server-users)))

(defsubst erc-remove-server-user (nick)
  "This function is for internal use only.

Removes the user with nickname NICK from the `erc-server-users'
hash table.  This user is not removed from the
`erc-channel-users' lists of other buffers.

See also: `erc-remove-user'."
  (with-current-buffer (process-buffer erc-process)
    (remhash (erc-downcase nick) erc-server-users)))

(defun erc-change-user-nickname (user new-nick)
  "This function is for internal use only.

Changes the nickname of USER to NEW-NICK in the
`erc-server-users' hash table.  The `erc-channel-users' lists of
other buffers are also changed."
  (let ((nick (erc-server-user-nickname user)))
    (setf (erc-server-user-nickname user) new-nick)
    (with-current-buffer (process-buffer erc-process)
      (remhash (erc-downcase nick) erc-server-users)
      (puthash (erc-downcase new-nick) user erc-server-users))
    (dolist (buf (erc-server-user-buffers user))
      (if (buffer-live-p buf)
	  (with-current-buffer buf
	    (let ((cdata (erc-get-channel-user nick)))
	      (remhash (erc-downcase nick) erc-channel-users)
	      (puthash (erc-downcase new-nick) cdata
		       erc-channel-users)))))))

(defun erc-remove-channel-user (nick)
  "This function is for internal use only.

Removes the user with nickname NICK from the `erc-channel-users'
list for this channel.  If this user is not in the
`erc-channel-users' list of any other buffers, the user is also
removed from the server's `erc-server-users' list.

See also: `erc-remove-server-user' and `erc-remove-user'."
  (let ((channel-data (erc-get-channel-user nick)))
    (when channel-data
      (let ((user (car channel-data)))
	(setf (erc-server-user-buffers user)
	      (delq (current-buffer)
		    (erc-server-user-buffers user)))
	(remhash (erc-downcase nick) erc-channel-users)
	(if (null (erc-server-user-buffers user))
	    (erc-remove-server-user nick))))))

(defun erc-remove-user (nick)
  "This function is for internal use only.

Removes the user with nickname NICK from the `erc-server-users'
list as well as from all `erc-channel-users' lists.

See also: `erc-remove-server-user' and
`erc-remove-channel-user'."
  (let ((user (erc-get-server-user nick)))
    (when user
      (let ((buffers (erc-server-user-buffers user)))
	(dolist (buf buffers)
	  (if (buffer-live-p buf)
	      (with-current-buffer buf
		(remhash (erc-downcase nick) erc-channel-users)
		(run-hooks 'erc-channel-members-changed-hook)))))
      (erc-remove-server-user nick))))

(defun erc-remove-channel-users ()
  "This function is for internal use only.

Removes all users in the current channel.  This is called by
`erc-server-PART' and `erc-server-QUIT'."
  (when (and erc-connected
	     (erc-process-alive)
	     (hash-table-p erc-channel-users))
    (maphash (lambda (nick cdata)
	       (erc-remove-channel-user nick))
	     erc-channel-users)
    (clrhash erc-channel-users)))

(defsubst erc-channel-user-op-p (nick)
  "Return `t' if NICK is an operator in the current channel."
  (and (hash-table-p erc-channel-users)
       (let ((cdata (erc-get-channel-user nick)))
	 (and cdata (cdr cdata)
	      (erc-channel-user-op (cdr cdata))))))

(defsubst erc-channel-user-voice-p (nick)
  "Return `t' if NICK has voice in the current channel."
  (and (hash-table-p erc-channel-users)
       (let ((cdata (erc-get-channel-user nick)))
	 (and cdata (cdr cdata)
	      (erc-channel-user-voice (cdr cdata))))))

(defun erc-get-channel-user-list ()
  "Returns a list of users in the current channel.  Each element
of the list is of the form (USER . CHANNEL-DATA), where USER is
an erc-server-user struct, and CHANNEL-DATA is either `nil' or an
erc-channel-user struct.

See also: `erc-sort-channel-users-by-activity'"
  (let (users)
    (if (hash-table-p erc-channel-users)
      (maphash (lambda (nick cdata)
		 (setq users (cons cdata users)))
	       erc-channel-users))
    users))

(defun erc-get-server-nickname-list ()
  "Returns a list of known nicknames on the current server."
    (if (erc-process-alive)
	(with-current-buffer (erc-server-buffer)
	  (let (nicks)
	    (when (hash-table-p erc-server-users)
	      (maphash (lambda (n user)
			 (setq nicks
			       (cons (erc-server-user-nickname user)
				     nicks)))
		       erc-server-users)
	      nicks)))))

(defun erc-get-channel-nickname-list ()
  "Returns a list of known nicknames on the current channel."
  (let (nicks)
    (when (hash-table-p erc-channel-users)
      (maphash (lambda (n cdata)
		 (setq nicks
		       (cons (erc-server-user-nickname (car cdata))
			     nicks)))
	       erc-channel-users)
      nicks)))

(defun erc-get-server-nickname-alist ()
  "Returns an alist of known nicknames on the current server."
    (if (erc-process-alive)
	(with-current-buffer (erc-server-buffer)
	  (let (nicks)
	    (when (hash-table-p erc-server-users)
	      (maphash (lambda (n user)
			 (setq nicks
			       (cons (cons (erc-server-user-nickname user) nil)
				     nicks)))
		       erc-server-users)
	      nicks)))))

(defun erc-get-channel-nickname-alist ()
  "Returns an alist of known nicknames on the current channel."
  (let (nicks)
    (when (hash-table-p erc-channel-users)
      (maphash (lambda (n cdata)
		 (setq nicks
		       (cons (cons (erc-server-user-nickname (car cdata)) nil)
			     nicks)))
	       erc-channel-users)
      nicks)))

(defun erc-sort-channel-users-by-activity (list)
  "Sorts LIST such that users which have spoken most recently are
listed first.  LIST must be of the form (USER . CHANNEL-DATA).

See also: `erc-get-channel-user-list'."
  (sort list
	(lambda (x y)
	  (when (and
		 (cdr x) (cdr y))
	    (let ((tx (erc-channel-user-last-message-time (cdr x)))
		  (ty (erc-channel-user-last-message-time (cdr y))))
	      (if tx
		  (if ty
		      (time-less-p ty tx)
		    t)
		nil))))))

(defun erc-sort-channel-users-alphabetically (list)
  "Sort LIST so that users' nicknames are in alphabetical order.
LIST must be of the form (USER . CHANNEL-DATA).

See also: `erc-get-channel-user-list'."
  (sort list
	(lambda (x y)
	  (when (and
		 (cdr x) (cdr y))
	    (let ((nickx (downcase (erc-server-user-nickname (car x))))
		  (nicky (downcase (erc-server-user-nickname (car y)))))
	      (if nickx
		  (if nicky
		      (string-lessp nickx nicky)
		    t)
		nil))))))

(defvar channel-topic nil
  "A topic string for the channel.  Should only be used in channel-buffers.")
(make-variable-buffer-local 'channel-topic)

(defvar channel-modes nil
  "List of strings representing channel modes.
E.g. '(\"i\" \"m\" \"s\" \"b Quake!*@*\")
\(not sure the ban list will be here, but why not)")
(make-variable-buffer-local 'channel-modes)

(defvar erc-insert-marker nil
  "The place where insertion of new text in erc buffers should happen.")
(make-variable-buffer-local 'erc-insert-marker)

(defun erc-string-no-properties (string)
  "Return a copy of STRING will all text-properties removed."
  (let ((newstring (copy-sequence string)))
    (set-text-properties 0 (length newstring) nil newstring)
    newstring))

(if (not (fboundp 'propertize))
    (defun erc-propertize (string &rest props)
      (while props
	(put-text-property 0 (length string)
			   (nth 0 props) (nth 1 props) string)
	(setq props (cddr props)))
      string)
  (defalias 'erc-propertize 'propertize))

(defcustom erc-prompt "ERC>"
  "Prompt used by ERC.  Trailing whitespace is not required."
  :group 'erc
  :type '(choice string function))

(defun erc-prompt ()
  "Return the input prompt as a string.

See also the variable `erc-prompt'."
  (let ((prompt (if (functionp erc-prompt)
		    (funcall erc-prompt)
		  erc-prompt)))
    (if (> (length prompt) 0)
	(concat prompt " ")
      prompt)))

(defcustom erc-input-line-position nil
  "Specify where to position the input line when using `erc-scroll-to-bottom'.

This should be an integer specifying the line of the buffer on which
the input line should stay.  A value of \"-1\" would keep the input
line positioned on the last line in the buffer.  This is passed as an
argument to `recenter'."
  :group 'erc
  :type '(choice integer (const nil)))

(defcustom erc-command-indicator nil
  "Indicator used by ERC for showing commands.

If non-nil, this will be used in the ERC buffer to indicate
commands (i.e., input starting with a '/').

If nil, the prompt will be constructed from the variable `erc-prompt'."
  :group 'erc
  :type '(choice string function))

(defun erc-command-indicator ()
  "Return the command indicator prompt as a string.

This only has any meaning if the variable `erc-command-indicator' is non-nil."
  (and erc-command-indicator
       (let ((prompt (if (functionp erc-command-indicator)
			 (funcall erc-command-indicator)
			 erc-command-indicator)))
	 (if (> (length prompt) 0)
	     (concat prompt " ")
	     prompt))))

;; ; Hmm, is this useful at all. If so, we would need to do it better, because
;; ; one looses nickname completion when turning this variable on.
;; Removed -- Lawrence 2004-01-08
;; (defcustom erc-prompt-interactive-input nil
;;   "*If non-nil, input can be typed in the minibuffer instead.
;; This uses a local-map text-property to detect you started typing
;; at the prompt, and copies the last key as initial input into the minibuffer.
;; Obsolete as of 1.348."
;;   :group 'erc
;;   :type 'boolean)
;; (make-obsolete-variable 'erc-prompt-interactive-input "Input in the
;;  minibuffer is deprecated now because it breaks some stuff,
;;  e.g. nick completion")

(defcustom erc-notice-prefix "*** "
  "*Prefix for all notices."
  :group 'erc
  :type 'string)

(defcustom erc-notice-highlight-type 'all
  "*Determines how to highlight notices.
See `erc-notice-prefix'.

The following values are allowed:

    'prefix - highlight notice prefix only
    'all    - highlight the entire notice

Any other value disables notice's highlighting altogether."
  :group 'erc
  :type '(choice (const :tag "highlight notice prefix only" prefix)
		 (const :tag "highlight the entire notice" all)
		 (const :tag "don't highlight notices at all" nil)))

(defcustom erc-echo-notice-hook nil
  "*Specifies a list of functions to call to echo a private
notice.  Each function is called with four arguments, the string
to display, the parsed server message, the target buffer (or
nil), and the sender.  The functions are called in order, until a
function evaluates to non-nil.  These hooks are called after
those specified in `erc-echo-notice-always-hook'.

See also: `erc-echo-notice-always-hook',
`erc-echo-notice-in-default-buffer',
`erc-echo-notice-in-target-buffer',
`erc-echo-notice-in-minibuffer',
`erc-echo-notice-in-server-buffer',
`erc-echo-notice-in-active-non-server-buffer',
`erc-echo-notice-in-active-buffer',
`erc-echo-notice-in-user-buffers',
`erc-echo-notice-in-user-and-target-buffers',
`erc-echo-notice-in-first-user-buffer'"
  :group 'erc-hooks
  :type 'hook
  :options '(erc-echo-notice-in-default-buffer
	     erc-echo-notice-in-target-buffer
	     erc-echo-notice-in-minibuffer
	     erc-echo-notice-in-server-buffer
	     erc-echo-notice-in-active-non-server-buffer
	     erc-echo-notice-in-active-buffer
	     erc-echo-notice-in-user-buffers
	     erc-echo-notice-in-user-and-target-buffers
	     erc-echo-notice-in-first-user-buffer))

(defcustom erc-echo-notice-always-hook
  '(erc-echo-notice-in-default-buffer)
  "*Specifies a list of functions to call to echo a private
notice.  Each function is called with four arguments, the string
to display, the parsed server message, the target buffer (or
nil), and the sender.  The functions are called in order, and all
functions are called.  These hooks are called before those
specified in `erc-echo-notice-hook'.

See also: `erc-echo-notice-hook',
`erc-echo-notice-in-default-buffer',
`erc-echo-notice-in-target-buffer',
`erc-echo-notice-in-minibuffer',
`erc-echo-notice-in-server-buffer',
`erc-echo-notice-in-active-non-server-buffer',
`erc-echo-notice-in-active-buffer',
`erc-echo-notice-in-user-buffers',
`erc-echo-notice-in-user-and-target-buffers',
`erc-echo-notice-in-first-user-buffer'"
  :group 'erc-hooks
  :type 'hook
  :options '(erc-echo-notice-in-default-buffer
	     erc-echo-notice-in-target-buffer
	     erc-echo-notice-in-minibuffer
	     erc-echo-notice-in-server-buffer
	     erc-echo-notice-in-active-non-server-buffer
	     erc-echo-notice-in-active-buffer
	     erc-echo-notice-in-user-buffers
	     erc-echo-notice-in-user-and-target-buffers
	     erc-echo-notice-in-first-user-buffer))

(defcustom erc-interpret-controls-p t
  "*Whether to interpret the colors and other highlighting info or not.
The interpreting can require a lot of resources and in chatty
channels, or in an emergency (message flood) it can be turned off to
save processing time."
  :group 'erc
  :type 'boolean)

(defcustom erc-interpret-mirc-color nil
  "*If non-nil, erc will interpret mIRC color codes."
  :group 'erc
  :type 'boolean)

(defcustom erc-use-info-buffers nil
  "*If non-nil, erc will open INFO buffers.
INFO buffers contain information about channels members."
  :group 'erc
  :type 'boolean)

;; other tunable parameters

(defcustom erc-whowas-on-nosuchnick nil
  "*If non-nil, do a whowas on a nick if no such nick."
  :group 'erc
  :type 'boolean)

(defcustom erc-verbose-server-ping nil
  "*If non-nil, show every time you get a PING or PONG from the server."
  :group 'erc
  :type 'boolean)

(defcustom erc-public-away-p nil
  "*Let others know you are back when you are no longer marked away.
This happens in this form:
* <nick> is back (gone for <time>)

Many consider it impolite to do so automatically."
  :group 'erc
  :type 'boolean)

(defcustom erc-away-nickname nil
  "*The nickname to take when you are marked as being away."
  :group 'erc
  :type '(choice (const nil)
		 string))

(defcustom erc-paranoid nil
  "If non-nil, then all incoming CTCP requests will be shown."
  :group 'erc
  :type 'boolean)

(defcustom erc-disable-ctcp-replies nil
  "Disable replies to CTCP requests that require a reply.
If non-nil, then all incoming CTCP requests that normally require
an automatic reply (like VERSION or PING) will be ignored.  Good to
set if some hacker is trying to flood you away."
  :group 'erc
  :type 'boolean)

(defcustom erc-anonymous-login t
  "Be paranoid, don't give away your machine name."
  :group 'erc
  :type 'boolean)

(defcustom erc-auto-reconnect t
  "Non-nil means that ERC will attempt to reestablish broken connections.

Reconnection will happen automatically for any unexpected disconnection."
  :group 'erc
  :type 'boolean)

(defcustom erc-prompt-for-channel-key nil
  "Prompt for channel key when using `erc-join-channel' interactively"
  :group 'erc
  :type 'boolean)

(defcustom erc-email-userid "user"
  "Use this as your email user ID."
  :group 'erc
  :type 'string)

(defcustom erc-ignore-list nil
  "*List of regexps matching user identifiers to ignore.

A user identifier has the form \"nick!login@host\".  If an
identifier matches, the message from the person will not be
processed."
  :group 'erc
  :type '(repeat regexp))
(make-variable-buffer-local 'erc-ignore-list)

(defcustom erc-ignore-reply-list nil
  "*List of regexps matching user identifiers to ignore completely.

This differs from `erc-ignore-list' in that it also ignores any
messages directed at the user.

A user identifier has the form \"nick!login@host\".

If an identifier matches, or a message is addressed to a nick
whose identifier matches, the message will not be processed.

CAVEAT: ERC doesn't know about the user and host of anyone who
was already in the channel when you joined, but never said
anything, so it won't be able to match the user and host of those
people. You can update the ERC internal info using /WHO *."
  :group 'erc
  :type '(repeat regexp))

(defvar erc-flood-protect 'normal
  "*If non-nil, flood protection is enabled.
Flooding is sending too much information to the server in too short time,
which may result in loosing connection.

If the value is 'strict, use a more strict limits provided in
`erc-flood-limit2', otherwise use \"normal\" limits from `erc-flood-limit'.")

(defvar erc-flood-limit '(1000 25 5)
  "Is a 3-element list (BYTES LINES SEC), defining the flood threshold:
at most BYTES bytes or LINES lines in messages within SEC seconds from
each other.  When either byte or line limit is exceeded, ERC stops
sending anything to the server, except for pings and one-line manual
user's commands.")

(defvar erc-flood-limit2 '(300 10 5)
  "Similar to `erc-flood-protect', but normally much more strict.
It will be used instead of `erc-flood-protect' in critical situations
\(detected flood, explicit user request, etc.).  Currently, it is
turned on when the flood limit is exceeded for the first time.")

;; Script parameters

(defcustom erc-startup-file-list
  '("~/.ercrc.el" "~/.ercrc" ".ercrc.el" ".ercrc")
  "List of files to try for a startup script.
The first existent and readable one will get executed.

If the filename ends with `.el' it is presumed to be an emacs-lisp
script and it gets (load)ed.  Otherwise is is treated as a bunch of
regular IRC commands"
  :group 'erc
  :type '(repeat file))

(defcustom erc-script-path nil
  "List of directories to look for a script in /load command.
The script is first searched in the current directory, then in each
directory in the list."
  :group 'erc
  :type '(repeat directory))

(defcustom erc-script-echo t
  "*If not-NIL, echo the IRC script commands locally."
  :group 'erc
  :type 'boolean)

(defvar erc-last-saved-position nil
  "A marker containing the position the current buffer was last saved at.")
(make-variable-buffer-local 'erc-last-saved-position)

(defcustom erc-kill-buffer-on-part nil
  "Kill the channel buffer on PART.
This variable should probably stay nil, as ERC can reuse buffers if
you rejoin them later."
  :group 'erc
  :type 'boolean)

(defcustom erc-kill-queries-on-quit nil
  "Kill all query (also channel) buffers of this server on QUIT.
See the variable `erc-kill-buffer-on-part' for details."
  :group 'erc
  :type 'boolean)

(defcustom erc-quit-reason-various-alist nil
  "Alist of possible arguments to the /quit command.

Each element has the form:
  (REGEXP RESULT)

If REGEXP matches the argument to /quit, then its relevant RESULT
will be used.  RESULT may be either a string, or a function.  If
a function, it should return the quit message as a string.

If no elements match, then the empty string is used.

As an example:
  (setq erc-quit-reason-various-alist
      '((\"zippy\" erc-quit-reason-zippy)
	(\"xmms\" dme:now-playing)
	(\"version\" erc-quit-reason-normal)
	(\"home\" \"Gone home !\")
	(\"\" \"Default Reason\")))
If the user types \"/quit zippy\", then a Zippy the Pinhead quotation
will be used as the quit message."
  :group 'erc
  :type '(repeat (list regexp (choice (string) (function)))))

(defcustom erc-part-reason-various-alist nil
  "Alist of possible arguments to the /part command.

Each element has the form:
  (REGEXP RESULT)

If REGEXP matches the argument to /part, then its relevant RESULT
will be used.  RESULT may be either a string, or a function.  If
a function, it should return the part message as a string.

If no elements match, then the empty string is used.

As an example:
  (setq erc-part-reason-various-alist
      '((\"zippy\" erc-part-reason-zippy)
	(\"xmms\" dme:now-playing)
	(\"version\" erc-part-reason-normal)
	(\"home\" \"Gone home !\")
	(\"\" \"Default Reason\")))
If the user types \"/part zippy\", then a Zippy the Pinhead quotation
will be used as the part message."
  :group 'erc
  :type '(repeat (list regexp (choice (string) (function)))))

(defcustom erc-quit-reason 'erc-quit-reason-normal
  "*A function which returns the reason for quitting.

The function is passed a single argument, the string typed by the
user after \"/quit\"."
  :group 'erc
  :type '(choice (const erc-quit-reason-normal)
		 (const erc-quit-reason-zippy)
		 (const erc-quit-reason-various)
		 (symbol)))

(defcustom erc-part-reason 'erc-part-reason-normal
  "A function which returns the reason for parting a channel.

The function is passed a single argument, the string typed by the
user after \"/PART\"."
  :group 'erc
  :type '(choice (const erc-part-reason-normal)
		 (const erc-part-reason-zippy)
		 (const erc-part-reason-various)
		 (symbol)))

(defvar erc-grab-buffer-name "*erc-grab*"
  "The name of the buffer created by `erc-grab-region'.")

;; variables available for IRC scripts

(defvar erc-user-information "ERC User"
  "USER_INFORMATION IRC variable.")

;; Hooks

(defgroup erc-hooks nil
  "Hook variables for fancy customizations of ERC."
  :group 'erc)

(defcustom erc-mode-hook nil
  "Hook run after `erc-mode' setup is finished."
  :group 'erc-hooks
  :type 'hook
  :options '(erc-add-scroll-to-bottom))

(defcustom erc-timer-hook nil
  "Put functions which should get called more or less periodically here.
The idea is that servers always play ping pong with the client, and so there
is no need for any idle-timer games with Emacs."
  :group 'erc-hooks
  :type 'hook)

(defcustom erc-insert-pre-hook nil
  "Hook called first when some text is inserted through `erc-display-line'.
It gets called with one argument, STRING.
To be able to modify the inserted text, use `erc-insert-modify-hook' instead.
Filtering functions can set `erc-insert-this' to nil to avoid
display of that particular string at all."
  :group 'erc-hooks
  :type 'hook)

(defcustom erc-send-pre-hook '(erc-send-distinguish-noncommands)
  "Hook called first when some text is sent through `erc-send-current-line'.
It gets called with one argument, STRING.

To change the text that will be sent, set the variable STR which is
used in `erc-send-current-line'.

To change the text inserted into the buffer without changing the text
that will be sent, use `erc-send-modify-hook' instead.

Filtering functions can set `erc-send-this' to nil to avoid sending of
that particular string at all and `erc-insert-this' to prevent
inserting that particular string into the buffer.

Note that it's useless to set `erc-send-this' to nil and
`erc-insert-this' to t.  ERC is sane enough to not insert the text
anyway."
  :group 'erc-hooks
  :type 'hook)

(defvar erc-insert-this t
  "Insert the text into the target buffer or not.
Functions on `erc-insert-pre-hook' can set this variable to nil
if they wish to avoid insertion of a particular string.")

(defvar erc-send-this t
  "Send the text to the target or not.
Functions on `erc-send-pre-hook' can set this variable to nil
if they wish to avoid sending of a particular string.")

(defcustom erc-insert-modify-hook ()
  "Insertion hook for functions that will change the text's appearance.
This hook is called just after `erc-insert-pre-hook' when the value
of `erc-insert-this' is t.
While this hook is run, narrowing is in effect and `current-buffer' is
the buffer where the text got inserted.  One possible value to add here
is `erc-fill'."
  :group 'erc-hooks
  :type 'hook)

(defcustom erc-insert-post-hook nil
  "This hook is called just after `erc-insert-modify-hook'.
At this point, all modifications from prior hook functions are done."
  :group 'erc-hooks
  :type 'hook
  :options '(erc-truncate-buffer
	     erc-make-read-only
	     erc-save-buffer-in-logs))

(defcustom erc-send-modify-hook nil
  "Sending hook for functions that will change the text's appearance.
This hook is called just after `erc-send-pre-hook' when the values
of `erc-send-this' and `erc-insert-this' are both t.
While this hook is run, narrowing is in effect and `current-buffer' is
the buffer where the text got inserted.

Note that no function in this hook can change the appearance of the
text that is sent.  Only changing the sent text's appearance on the
sending user's screen is possible.  One possible value to add here
is `erc-fill'."
  :group 'erc-hooks
  :type 'hook)

(defcustom erc-send-post-hook nil
  "This hook is called just after `erc-send-modify-hook'.
At this point, all modifications from prior hook functions are done.
NOTE: The functions on this hook are called _before_ sending a command
to the server.

This function is called with narrowing, ala `erc-send-modify-hook'"
  :group 'erc-hooks
  :type 'hook
  :options '(erc-make-read-only))

(defcustom erc-send-completed-hook
  (when (featurep 'emacspeak)
    (list (byte-compile
	   (lambda (str)
	     (emacspeak-auditory-icon 'select-object)))))
  "Hook called after a message is successfully sent to the server.

The single argument to the functions is the unmodified string
which the local user typed."
  :group 'erc-hooks
  :type 'hook)
;; mode-specific tables

(defvar erc-mode-syntax-table
  (let ((syntax-table (make-syntax-table)))
    (modify-syntax-entry ?\" ".   " syntax-table)
    (modify-syntax-entry ?\\ ".   " syntax-table)
    (modify-syntax-entry ?' "w   " syntax-table)
    ;; Make dabbrev-expand useful for nick names
    (modify-syntax-entry ?< "." syntax-table)
    (modify-syntax-entry ?> "." syntax-table)
    syntax-table)
  "Syntax table used while in ERC mode.")

(defvar erc-mode-abbrev-table nil
  "Abbrev table used while in ERC mode.")
(define-abbrev-table 'erc-mode-abbrev-table ())

(defvar erc-mode-map
  (let ((map (make-sparse-keymap)))
    (define-key map "\C-m" 'erc-send-current-line)
    (define-key map "\C-a" 'erc-bol)
    (define-key map "\C-c\C-a" 'erc-bol)
    (define-key map "\C-c\C-b" 'erc-iswitchb)
    (define-key map "\C-c\C-c" 'erc-toggle-interpret-controls)
    (define-key map "\C-c\C-d" 'erc-input-action)
    (define-key map "\C-c\C-e" 'erc-toggle-ctcp-autoresponse)
    (define-key map "\C-c\C-f" 'erc-toggle-flood-control)
    (define-key map "\C-c\C-i" 'erc-invite-only-mode)
    (define-key map "\C-c\C-j" 'erc-join-channel)
    (define-key map "\C-c\C-n" 'erc-channel-names)
    (define-key map "\C-c\C-o" 'erc-get-channel-mode-from-keypress)
    (define-key map "\C-c\C-p" 'erc-part-from-channel)
    (define-key map "\C-c\C-q" 'erc-quit-server)
    (define-key map "\C-c\C-r" 'erc-remove-text-properties-region)
    (define-key map "\C-c\C-t" 'erc-set-topic)
    (define-key map "\C-c\C-u" 'erc-kill-input)
    (define-key map "\M-\t" 'ispell-complete-word)
    (define-key map "\t" 'erc-complete-word)
    map)
  "ERC keymap.")

(defvar erc-info-mode-map (make-sparse-keymap)
  "Keymap used in `erc-info-mode'.")

;; Faces

; Honestly, I have a horrible sense of color and the "defaults" below
; are supposed to be really bad. But colors ARE required in IRC to
; convey different parts of conversation. If you think you know better
; defaults - send them to me.

;; Now colors are a bit nicer, at least to my eyes.
;; You may still want to change them to better fit your background.-- S.B.

(defgroup erc-faces nil
  "Faces for ERC."
  :group 'erc)

(defface erc-default-face '((t))
  "ERC default face."
  :group 'erc-faces)
(defface erc-direct-msg-face '((t (:foreground "IndianRed")))
  "ERC face used for messages you receive in the main erc buffer."
  :group 'erc-faces)
(defface erc-input-face '((t (:foreground "brown")))
  "ERC face used for your input."
  :group 'erc-faces)
(defface erc-bold-face '((t (:bold t)))
  "ERC bold face."
  :group 'erc-faces)
(defface erc-inverse-face
  '((t (:foreground "White" :background "Black")))
  "ERC inverse face."
  :group 'erc-faces)
(defface erc-underline-face '((t (:underline t)))
  "ERC underline face."
  :group 'erc-faces)
(defface erc-prompt-face
  '((t (:bold t :foreground "Black" :background"lightBlue2")))
  "ERC face for the prompt."
  :group 'erc-faces)
(defface erc-command-indicator-face
    '((t (:bold t)))
  "ERC face for the command indicator.  See the variable
`erc-command-indicator'."
  :group 'erc-faces)
(defface erc-notice-face '((t (:bold t :foreground "SlateBlue")))
  "ERC face for notices."
  :group 'erc-faces)
(defface erc-action-face '((t (:bold t)))
  "ERC face for actions generated by /ME."
  :group 'erc-faces)
(defface erc-error-face '((t (:foreground "red")))
  "ERC face for errors."
  :group 'erc-faces)
(defface erc-nick-default-face '((t (:bold t)))
  "ERC nickname default face."
  :group 'erc-faces)
(defface erc-nick-msg-face '((t (:bold t :foreground "IndianRed")))
  "ERC nickname face for private messages."
  :group 'erc-faces)

(defface fg:erc-color-face0 '((t (:foreground "White")))
  "ERC face."
  :group 'erc-faces)
(defface fg:erc-color-face1 '((t (:foreground "black")))
  "ERC face."
  :group 'erc-faces)
(defface fg:erc-color-face2 '((t (:foreground "blue4")))
  "ERC face."
  :group 'erc-faces)
(defface fg:erc-color-face3 '((t (:foreground "green4")))
  "ERC face."
  :group 'erc-faces)
(defface fg:erc-color-face4 '((t (:foreground "red")))
  "ERC face."
  :group 'erc-faces)
(defface fg:erc-color-face5 '((t (:foreground "brown")))
  "ERC face."
  :group 'erc-faces)
(defface fg:erc-color-face6 '((t (:foreground "purple")))
  "ERC face."
  :group 'erc-faces)
(defface fg:erc-color-face7 '((t (:foreground "orange")))
  "ERC face."
  :group 'erc-faces)
(defface fg:erc-color-face8 '((t (:foreground "yellow")))
  "ERC face."
  :group 'erc-faces)
(defface fg:erc-color-face9 '((t (:foreground "green")))
  "ERC face."
  :group 'erc-faces)
(defface fg:erc-color-face10 '((t (:foreground "lightblue1")))
  "ERC face."
  :group 'erc-faces)
(defface fg:erc-color-face11 '((t (:foreground "cyan")))
  "ERC face."
  :group 'erc-faces)
(defface fg:erc-color-face12 '((t (:foreground "blue")))
  "ERC face."
  :group 'erc-faces)
(defface fg:erc-color-face13 '((t (:foreground "deeppink")))
  "ERC face."
  :group 'erc-faces)
(defface fg:erc-color-face14 '((t (:foreground "gray50")))
  "ERC face."
  :group 'erc-faces)
(defface fg:erc-color-face15 '((t (:foreground "gray90")))
  "ERC face."
  :group 'erc-faces)

(defface bg:erc-color-face0 '((t (:background "White")))
  "ERC face."
  :group 'erc-faces)
(defface bg:erc-color-face1 '((t (:background "black")))
  "ERC face."
  :group 'erc-faces)
(defface bg:erc-color-face2 '((t (:background "blue4")))
  "ERC face."
  :group 'erc-faces)
(defface bg:erc-color-face3 '((t (:background "green4")))
  "ERC face."
  :group 'erc-faces)
(defface bg:erc-color-face4 '((t (:background "red")))
  "ERC face."
  :group 'erc-faces)
(defface bg:erc-color-face5 '((t (:background "brown")))
  "ERC face."
  :group 'erc-faces)
(defface bg:erc-color-face6 '((t (:background "purple")))
  "ERC face."
  :group 'erc-faces)
(defface bg:erc-color-face7 '((t (:background "orange")))
  "ERC face."
  :group 'erc-faces)
(defface bg:erc-color-face8 '((t (:background "yellow")))
  "ERC face."
  :group 'erc-faces)
(defface bg:erc-color-face9 '((t (:background "green")))
  "ERC face."
  :group 'erc-faces)
(defface bg:erc-color-face10 '((t (:background "lightblue1")))
  "ERC face."
  :group 'erc-faces)
(defface bg:erc-color-face11 '((t (:background "cyan")))
  "ERC face."
  :group 'erc-faces)
(defface bg:erc-color-face12 '((t (:background "blue")))
  "ERC face."
  :group 'erc-faces)
(defface bg:erc-color-face13 '((t (:background "deeppink")))
  "ERC face."
  :group 'erc-faces)
(defface bg:erc-color-face14 '((t (:background "gray50")))
  "ERC face."
  :group 'erc-faces)
(defface bg:erc-color-face15 '((t (:background "gray90")))
  "ERC face."
  :group 'erc-faces)

(defun erc-get-bg-color-face (n)
  "Fetches the right face for background color N (0-15)."
  (if (stringp n) (setq n (string-to-number n)))
  (if (not (numberp n))
      (progn
	(message "erc-get-bg-color-face: n is NaN: %S" n)
	(beep)
	'default)
    (when (> n 16)
      (erc-log (format "   Wrong color: %s" n))
      (setq n (mod n 16)))
    (cond
     ((and (>= n 0) (< n 16))
      (intern (concat "bg:erc-color-face" (number-to-string n))))
     (t (erc-log (format "   Wrong color: %s" n)) 'default))))

(defun erc-get-fg-color-face (n)
  "Fetches the right face for foreground color N (0-15)."
  (if (stringp n) (setq n (string-to-number n)))
  (if (not (numberp n))
      (progn
	(message "erc-get-fg-color-face: n is NaN: %S" n)
	(beep)
	'default)
    (when (> n 16)
      (erc-log (format "   Wrong color: %s" n))
      (setq n (mod n 16)))
    (cond
     ((and (>= n 0) (< n 16))
      (intern (concat "fg:erc-color-face" (number-to-string n))))
     (t (erc-log (format "   Wrong color: %s" n)) 'default))))

;; Debugging support

(defvar erc-log-p nil
  "When set to t, generate debug messages in a separate debug buffer.")

(defvar erc-debug-log-file (concat (expand-file-name ".") "/ERC.debug")
  "Debug log file name.")

(defvar erc-dbuf nil)
(make-variable-buffer-local 'erc-dbuf)

(defmacro define-erc-module (name alias doc enable-body disable-body
			     &optional local-p)
  "Define a new minor mode using ERC conventions.
Symbol NAME is the name of the module.
Symbol ALIAS is the alias to use, or nil.
DOC is the documentation string to use for the minor mode.
ENABLE-BODY is a list of expressions used to enable the mode.
DISABLE-BODY is a list of expressions used to disable the mode.
If LOCAL-P is non-nil, the mode will be created as a buffer-local
mode.  Rather than a global one.

This will define a minor mode called erc-NAME-mode, possibly
an alias erc-ALIAS-mode, as well as the helper functions
erc-NAME-enable, and erc-NAME-disable.

Example:

  ;;;###autoload (autoload 'erc-replace-mode \"erc-replace\")
  (define-erc-module replace nil
    \"This mode replaces incoming text according to `erc-replace-alist'.\"
    ((add-hook 'erc-insert-modify-hook
	       'erc-replace-insert))
    ((remove-hook 'erc-insert-modify-hook
		  'erc-replace-insert)))"
  (let* ((sn (symbol-name name))
	 (mode (intern (format "erc-%s-mode" (downcase sn))))
	 (group (intern (format "erc-%s" (downcase sn))))
	 (enable (intern (format "erc-%s-enable" (downcase sn))))
	 (disable (intern (format "erc-%s-disable" (downcase sn)))))
    `(progn
       (erc-define-minor-mode
	,mode
	,(format "Toggle ERC %S mode.
With arg, turn ERC %S mode on if and only if arg is positive.
%s" name name doc)
	nil nil nil
	:global ,(not local-p) :group (quote ,group)
	(if ,mode
	    (,enable)
	  (,disable)))
       (defun ,enable ()
	 ,(format "Enable ERC %S mode."
		  name)
	 (interactive)
	 (add-to-list 'erc-modules (quote ,name))
	 (setq ,mode t)
	 ,@enable-body)
       (defun ,disable ()
	 ,(format "Disable ERC %S mode."
		  name)
	 (interactive)
	 (setq erc-modules (delq (quote ,name) erc-modules))
	 (setq ,mode nil)
	 ,@disable-body)
       ,(when (and alias (not (eq name alias)))
	  `(defalias
	     (quote
	      ,(intern
		(format "erc-%s-mode"
			(downcase (symbol-name alias)))))
	     (quote
	      ,mode))))))

;; We need these functions in erc-once-with-server-event, so we need
;; them when compiling.
(eval-and-compile
  (defun erc-event-to-hook-name (event)
    "Return the name of the hook handling EVENT as stringp."
    (concat "erc-server-"
	    (cond
	     ((symbolp event)
	      (symbol-name event))
	     ((numberp event)
	      (format "%03i" event))
	     ((stringp event)
	      event)
	     (t
	      (error
	       "EVENT should be a symbol or a number")))
	    "-hook"))

  (defun erc-event-to-hook (event)
    "Return the hook handling EVENT as symbolp."
    (intern (erc-event-to-hook-name event)))
  ) ; eval-and-compile


(defun erc-once-with-server-event (event &rest forms)
  "Execute FORMS the next time EVENT occurs in the `current-buffer'.

You should make sure that `current-buffer' is a server buffer.

This function temporarily adds a function to EVENT's hook to
execute FORMS.  After FORMS are run, the function is removed from
EVENT's hook.  The last expression of FORMS should be either nil
or t.  nil indicates that the other functions on EVENT's hook
should be run too, and t indicates that other functions should
not be run.

Please be sure to use this function in server-buffers.  In
channel-buffers it may not work at all, as it uses the LOCAL
argument of `add-hook' and `remove-hook' to ensure multiserver
capabilities."
  (unless (erc-server-buffer-p)
    (error
     "You should only run `erc-once-with-server-event' in a server buffer"))
  (let ((fun (gensym))
	(hook (erc-event-to-hook event)))
     (put fun 'erc-original-buffer (current-buffer))
     (fset fun `(lambda (proc parsed)
		  (with-current-buffer (get ',fun 'erc-original-buffer)
		    (remove-hook ',hook ',fun t))
		  (fmakunbound ',fun)
		  ,@forms))
     (add-hook hook fun nil t)
     fun))

(defun erc-once-with-server-event-global (event &rest forms)
  "Execute FORMS the next time EVENT occurs in any server buffer.

This function temporarily prepends a function to EVENT's hook to
execute FORMS.  After FORMS are run, the function is removed from
EVENT's hook.  The last expression of FORMS should be either nil
or t.  nil indicates that the other functions on EVENT's hook
should be run too, and t indicates that other functions should
not be run.

When FORMS execute, the current buffer is the server buffer associated with the
connection over which the data was received that triggered EVENT."
  (let ((fun (gensym))
	(hook (erc-event-to-hook event)))
     (fset fun `(lambda (proc parsed)
		  (remove-hook ',hook ',fun)
		  (fmakunbound ',fun)
		  ,@forms))
     (add-hook hook fun nil nil)
     fun))

(defmacro erc-log (string)
  "Logs STRING if logging is on (see `erc-log-p')."
  `(when erc-log-p
     (erc-log-aux ,string)))

(defun erc-server-buffer ()
  "Return the server buffer for the current buffer's process.
The buffer-local variable `erc-process' is used to find the process buffer."
  (and (boundp 'erc-process)
       (processp erc-process)
       (process-buffer erc-process)))

(defun erc-server-buffer-p (&optional buffer)
  "Return non-nil if argument BUFFER is an ERC server buffer.

If BUFFER is nil, the current buffer is used."
  (with-current-buffer (or buffer (current-buffer))
    (and (eq major-mode 'erc-mode)
	 (null (erc-default-target)))))

(defun erc-query-buffer-p (&optional buffer)
  "Return non-nil if BUFFER is an ERC query buffer.
If BUFFER is nil, the current buffer is used."
  (with-current-buffer (or buffer (current-buffer))
    (let ((target (erc-default-target)))
      (and (eq major-mode 'erc-mode)
	   target
	   (not (memq (aref target 0) '(?# ?& ?+ ?!)))))))

(defun erc-ison-p (nick)
  "Return non-nil if NICK is online."
  (interactive "sNick: ")
  (with-current-buffer (erc-server-buffer)
    (let ((erc-online-p 'unknown))
      (erc-once-with-server-event
       303
       `(let ((ison (split-string (aref parsed 3))))
	  (setq erc-online-p (car (erc-member-ignore-case ,nick ison)))
	  t))
      (erc-send-command (format "ISON %s" nick))
      (while (eq erc-online-p 'unknown) (accept-process-output))
      (if (interactive-p)
	  (message "%s is %sonline"
		   (or erc-online-p nick)
		   (if erc-online-p "" "not "))
	erc-online-p))))

(defun erc-log-aux (string)
  "Do the debug logging of STRING."
  (let ((cb (current-buffer))
	(point 1)
	(was-eob nil)
	(session-buffer (erc-server-buffer)))
    (if session-buffer
	(progn
	  (set-buffer session-buffer)
	  (if (not (and erc-dbuf (bufferp erc-dbuf) (buffer-live-p erc-dbuf)))
	      (progn
		(setq erc-dbuf (get-buffer-create
				(concat "*ERC-DEBUG: "
					erc-session-server "*")))))
	  (set-buffer erc-dbuf)
	  (setq point (point))
	  (setq was-eob (eobp))
	  (goto-char (point-max))
	  (insert (concat "** " string "\n"))
	  (if was-eob (goto-char (point-max))
	    (goto-char point))
	  (set-buffer cb))
      (message "ERC: ** %s" string))))

;; Last active buffer, to print server messages in the right place

(defvar erc-active-buffer nil
  "The current active buffer, the one where the user typed the last command.")

;; Mode activation routines

(defun erc-mode ()
  "Major mode for Emacs IRC.
Special commands:

\\{erc-mode-map}

Turning on `erc-mode' runs the hook `erc-mode-hook'."
  (kill-all-local-variables)
  (use-local-map erc-mode-map)
  (setq mode-name "ERC"
	major-mode 'erc-mode
	local-abbrev-table erc-mode-abbrev-table)
  (set-syntax-table erc-mode-syntax-table)
  ;; This seems to work only in CVS emacs. "GNU Emacs 21.2.2" has
  ;; problems with this:
  ;; (string-match "[^ ]" "foo bar") returns nil.
  ;; And XEmacs doesn't have set-case-syntax-pair :(...
;;  (when (and (>= emacs-major-version 21)
;;	     (>= emacs-minor-version 3))
;;    (set-case-table
;;     (let ((table (copy-case-table (standard-case-table))))
;;       (set-case-syntax-pair ?\[ ?\{ table)
;;       (set-case-syntax-pair ?\] ?\} table)
;;       (set-case-syntax-pair ?\\ ?\| table)
;;       (set-case-syntax-pair ?~ ?^ table)
;;       table)))
  (when (boundp 'next-line-add-newlines)
    (set (make-local-variable 'next-line-add-newlines) nil))
  (make-variable-buffer-local 'paragraph-separate)
  (make-variable-buffer-local 'paragraph-start)
  (setq line-move-ignore-invisible t)
  (setq paragraph-separate (concat "\C-l\\|\\(^" (regexp-quote (erc-prompt))
				   "\\)"))
  (setq paragraph-start (concat "\\(" (regexp-quote (erc-prompt)) "\\)"))
  ;; Run the mode hooks
  (run-hooks 'erc-mode-hook))

(defun erc-info-mode ()
  "Major mode for Emacs IRC Channel Info buffers.
Special commands:

\\{erc-mode-map}."
  (kill-all-local-variables)

  (use-local-map erc-info-mode-map)
  (setq mode-name "ERC Info")
  (setq major-mode 'erc-info-mode)
  (toggle-read-only 1))


;; activation

(defconst erc-default-server "irc.freenode.net"
  "IRC server to use if it cannot be detected otherwise.")

(defconst erc-default-port "ircd"
  "IRC port to use if it cannot be detected otherwise.")

(defcustom erc-join-buffer 'buffer
  "Determines how to display the newly created IRC buffer.
'window - in another window,
'window-noselect - in another window, but don't select that one,
'frame - in another frame,
'bury - bury it in a new buffer,
any other value - in place of the current buffer"
  :group 'erc
  :type '(choice (const window)
		 (const window-noselect)
		 (const frame)
		 (const bury)
		 (const buffer)))

(defcustom erc-join-info-buffer 'disable
  "Determines how to display the INFO buffer for a channel on join.
Values can be 'frame - in another frame, 'window - another window in
the same frame, 'split - split the current window in two and display
it in the lower subwindow.  Any other value will leave the info buffer
invisible."
  :group 'erc
  :type '(choice (const frame)
		 (const window)
		 (const split)
		 (const disable)))

(defcustom erc-frame-alist nil
  "*Alist of frame parameters for creating erc frames.
A value of `nil means to use `default-frame-alist'."
  :group 'erc
  :type '(repeat (cons :format "%v"
		       (symbol :tag "Parameter")
		       (sexp :tag "Value"))))

(defcustom erc-frame-dedicated-flag nil
  "*Non-nil means the erc frames are dedicated to that buffer.
This only has effect when `erc-join-buffer' and `erc-join-info-buffer'
are set to `frame'."
  :group 'erc
  :type 'boolean)

(defun erc-channel-p (channel)
  "Return non-nil if CHANNEL seems to be an IRC channel name."
  (cond ((stringp channel)
	 (memq (aref channel 0) '(?# ?& ?+ ?!)))
	((and (bufferp channel) (buffer-live-p channel))
	 (with-current-buffer channel
	   (erc-channel-p (erc-default-target))))
	(t nil)))

(defcustom erc-reuse-buffers t
  "*If nil, create new buffers on joining a channel/query.
If non-nil, a new buffer will only be created when you join
channels with same names on different servers, or have query buffers
open with nicks of the same name on different servers.  Otherwise,
the existing buffers will be reused."
  :group 'erc
  :type 'boolean)

(defun erc-normalize-port (port)
  "Normalize the port specification PORT to integer form.
PORT may be an integer, a string or a symbol.  If it is a string or a
symbol, it may have these values:
* irc         -> 194
* ircs        -> 994
* ircd        -> 6667
* ircd-dalnet -> 7000"
  (cond
   ((symbolp port)
    (erc-normalize-port (symbol-name port)))
   ((stringp port)
    (let ((port-nr (string-to-number port)))
      (cond
       ((> port-nr 0)
	port-nr)
       ((string-equal port "irc")
	194)
       ((string-equal port "ircs")
	994)
       ((string-equal port "ircd")
	6667)
       ((string-equal port "ircd-dalnet")
	7000)
       (t
	nil))))
   ((numberp port)
    port)
   (t
    nil)))

(defun erc-port-equal (a b)
  "Check whether ports A and B are equal."
  (= (erc-normalize-port a) (erc-normalize-port b)))

(defun erc-generate-new-buffer-name (server port target &optional proc)
  "Create a new buffer name based on the arguments."
  (when (numberp port) (setq port (number-to-string port)))
  (let* ((buf-name (or target
		       (or (let ((name (concat server ":" port)))
			     (when (> (length name) 1)
			       name))
			   ; This fallback should in fact never happen
			   "*erc-server-buffer*"))))
    ;; Reuse existing buffers, but not if the buffer is a connected server
    ;; buffer and not if its associated with a different server than the
    ;; current ERC buffer.
    (if (and erc-reuse-buffers
	     (get-buffer buf-name)
	     (or target
		 (with-current-buffer (get-buffer buf-name)
		   (and (erc-server-buffer-p)
			(not erc-connected))))
	     (with-current-buffer (get-buffer buf-name)
	       (and (string= erc-session-server server)
		    (erc-port-equal erc-session-port port))))
	buf-name
      (generate-new-buffer-name buf-name))))

(defun erc-get-buffer-create (server port target &optional proc)
  "Create a new buffer based on the arguments."
  (get-buffer-create (erc-generate-new-buffer-name server port target proc)))


(defun erc-member-ignore-case (string list)
  "Return non-nil if STRING is a member of LIST.

All strings are compared according to IRC protocol case rules, see
`erc-downcase'."
  (setq string (erc-downcase string))
  (catch 'result
    (while list
      (if (string= string (erc-downcase (car list)))
	  (throw 'result list) (setq list (cdr list))))))

(defmacro erc-with-buffer (spec &rest body)
  "Execute BODY in the buffer associated with SPEC.

SPEC should have the form

 (TARGET [PROCESS])

The buffer used is the buffer associated with TARGET.
If PROCESS is not specified, the current process is used.

See also `with-current-buffer' and `erc-get-buffer'.

\(fn (TARGET [PROCESS]) BODY...)"
  (let ((buf (gensym))
	(proc (gensym)))
    `(let ((,buf (if (bufferp ,(car spec))
		     ,(car spec)
		   (let ((,proc (or ,(cadr spec)
				     (and (processp erc-process)
					  erc-process))))
		      (if (and ,(car spec) ,proc)
			  (erc-get-buffer ,(car spec) ,proc))))))
       (when ,buf
	 (with-current-buffer ,buf ,@body)))))

(defun erc-get-buffer (target &optional proc)
  "Return the buffer matching TARGET in the process PROC.
If PROC is not supplied, all processes are searched."
  (let ((downcased-target (erc-downcase target)))
    (catch 'buffer
      (erc-buffer-filter
       (lambda ()
	 (let ((current (erc-default-target)))
	   (and (stringp current)
		(string-equal downcased-target (erc-downcase current))
		(throw 'buffer (current-buffer)))))
       proc))))

(defun erc-buffer-filter (predicate &optional proc)
  "Return a list of `erc-mode' buffers matching certain criteria.
PREDICATE is a function executed with each buffer, if it returns t, that buffer
is considered a valid match.
PROC is either an `erc-process', identifying a certain server connection,
or nil which means all open connections."
  (save-excursion
    (delq
     nil
     (mapcar (lambda (buf)
	       (with-current-buffer buf
		 (and (eq major-mode 'erc-mode)
		      (or (not proc)
			  (eq proc erc-process))
		      (funcall predicate)
		      buf)))
	     (buffer-list)))))

(defun erc-buffer-list (&optional predicate proc)
  "Return a list of ERC buffers.
PREDICATE is a function which executes with every buffer satisfying
the predicate.  If PREDICATE is passed as nil, return a list of all ERC
buffers.  If PROC is given, the buffers local variable `erc-process'
needs to match PROC."
  (unless predicate
    (setq predicate (lambda () t)))
  (erc-buffer-filter predicate proc))

(defmacro erc-with-all-buffers-of-server (process pred &rest forms)
  "Execute FORMS in all buffers which have same process as this server.
FORMS will be evaluated in all buffers having the process PROCESS and
where PRED matches or in all buffers of the server process if PRED is
nil."
  `(mapcar (lambda (buffer)
	    (with-current-buffer buffer
	      ,@forms))
	   (erc-buffer-list ,pred
			    ,process)))

(defun erc-iswitchb (arg)
  "Use `iswitchb-read-buffer' to prompt for a ERC buffer to switch to.
When invoked with prefix argument, use all erc buffers.  Without prefix
ARG, allow only buffers related to same session server.
If `erc-track-mode' is in enabled, put the last element of
`erc-modified-channels-alist' in front of the buffer list.

Due to some yet unresolved reason, global function `iswitchb-mode'
needs to be active for this function to work."
  (interactive "P")
  (require 'iswitchb)
  (let ((iswitchb-make-buflist-hook
	 (lambda ()
	   (setq iswitchb-temp-buflist
		 (mapcar 'buffer-name
			 (erc-buffer-list
			  nil
			  (when (and arg (boundp 'erc-process))
			    erc-process)))))))
    (switch-to-buffer
     (iswitchb-read-buffer
      "Switch-to: "
      (buffer-name (caar (last erc-modified-channels-alist)))
      t))))

(defun erc-channel-list (proc)
  "Return a list of channel buffers.
PROC is the process for the server connection.  If PROC is nil, return
all channel buffers on all servers."
  (erc-buffer-filter
   (lambda ()
     (and (erc-default-target)
	  (erc-channel-p (erc-default-target))))
   proc))

(defun erc-buffer-list-with-nick (nick proc)
  "Return buffers containing NICK in the `erc-channel-users' list."
  (with-current-buffer (process-buffer proc)
    (let ((user (gethash (erc-downcase nick) erc-server-users)))
      (if user
	  (erc-server-user-buffers user)
	nil))))
		

;; Some local variables

(defvar erc-default-recipients nil
  "List of default recipients of the current buffer.")
(make-variable-buffer-local 'erc-default-recipients)

(defvar erc-session-user-full-name nil
  "Full name of the user on the current server.")
(make-variable-buffer-local 'erc-session-user-full-name)

(defvar current-nick nil
  "Nickname on the current server.
Use `erc-current-nick' to access this.")
(make-variable-buffer-local 'current-nick)

(defvar channel-user-limit nil
  "Limit of users per channel.")
(make-variable-buffer-local 'channel-user-limit)

(defvar channel-key nil
  "Key needed to join channel.")
(make-variable-buffer-local 'channel-key)

(defvar last-peers '(nil . nil)
  "Last peers used, both sender and receiver.
Those are used for /MSG destination shortcuts.")
(make-variable-buffer-local 'last-peers)

(defvar invitation nil
  "Last invitation channel.")
(make-variable-buffer-local 'invitation)

(defvar away nil
  "Non-nil indicates that we are away.")
(make-variable-buffer-local 'away)

(defvar channel-list nil
  "Server channel list.")
(make-variable-buffer-local 'channel-list)

(defvar last-sent-time nil
  "Time the message was sent.
This is useful for flood protection.")
(make-variable-buffer-local 'last-sent-time)

(defvar last-ping-time nil
  "Time the last ping was sent.
This is useful for flood protection.")
(make-variable-buffer-local 'last-ping-time)

(defvar last-ctcp-time nil
  "Time the last CTCP response/request was sent.
This is useful for flood protection.")
(make-variable-buffer-local 'last-ctcp-time)

(defvar erc-lines-sent nil
  "Line counter.")
(make-variable-buffer-local 'erc-lines-sent)

(defvar erc-bytes-sent nil
  "Byte counter.")
(make-variable-buffer-local 'erc-bytes-sent)

(defvar quitting nil
  "Non-nil if the user requests a quit.")
(make-variable-buffer-local 'quitting)

(defvar bad-nick nil
  "Non-nil indicates that we got a `nick in use' error while connecting.")
(make-variable-buffer-local 'bad-nick)

(defvar erc-logged-in nil
  "Non-nil indicates that we are logged in.")
(make-variable-buffer-local 'erc-logged-in)

(defvar erc-default-nicks nil
  "The local copy of `erc-nick' - the list of nicks to choose from.")
(make-variable-buffer-local 'erc-default-nicks)

(defvar erc-nick-change-attempt-count 0
  "Used to keep track of how many times an attempt at changing nick is made.")
(make-variable-buffer-local 'erc-nick-change-attempt-count)

(defcustom erc-modules '(netsplit fill track completion ring button autojoin)
  "A list of modules which erc should enable.
Use \\[customize-set-variable] to set this value, or remember to call
\(erc-update-modules) after you change it.  Removing entries from the list will
not cause them to be unloaded at this time."
  :set (lambda (sym val) (set-default sym val)
	 ;; this test is for the case where erc hasn't been loaded yet
	 (when (fboundp 'erc-update-modules)
	   (erc-update-modules)))
  :type '(set :greedy t
	      (const :tag "Automatically mark as being away" autoaway)
	      (const :tag "Automatically join channels" autojoin)
	      (const :tag "Integrate with Big Brother Database" bbdb)
	      (const :tag "Text buttonization" button)
	      (const :tag "Nick and command completion" completion)
	      (const :tag "Wrap long lines" fill)
	      (const :tag "Save buffers in logs" log)
	      (const :tag "Highlight pals, fools, and other keywords" match)
	      (const :tag "Netsplit avoidance mode" netsplit)
	      (const :tag "Online status notification" notify)
	      (const :tag "Programmable nick and command completion" pcomplete)
	      (const :tag "Replace text" replace)
	      (const :tag "Input ring" ring)
	      (const :tag "Automatic IRC Services handling" services)
	      (const :tag "Convert smileys to pretty icons" smiley)
	      (const :tag "Play sounds when you receive CTCP SOUND requests" sound)
	      (const :tag "Add timestamps to messages" stamp)
	      (const :tag "Track channel activity" track)
	      (const :tag "Truncate buffers to a certain size" truncate)
	      (repeat :tag "Others" :inline t symbol))
  :group 'erc)

;; We use eval-after-load here to ensure recursive dependencies don't
;; break things. E.g. we might want to require erc-foo.el which uses
;; erc.el, so it should be loaded after this file is loaded.
(eval-after-load "erc" '(and
			 ;; Make sure we finished loading erc.
			 (featurep 'erc)
			 (erc-update-modules)))

(defun erc-update-modules ()
  "Run this to enable erc-foo-mode for all modules in `erc-modules'."
  (let (req)
    (dolist (mod erc-modules)
      (setq req (concat "erc-" (symbol-name mod)))
      (cond
       ;; yuck. perhaps we should bring the filenames into sync?
       ((string= req "erc-completion")
	(setq req "erc-pcomplete")
	(setq mod 'pcomplete))
       ((string= req "erc-services")
	(setq req "erc-nickserv")
	(setq mod 'services)))
      (load req t)
      (funcall (or (intern-soft (concat "erc-" (symbol-name mod) "-mode"))
		   (error "Unknown module %s in erc-modules" mod))
	       1))))

(defun erc-setup-buffer (buffer)
  "Consults `erc-join-buffer' to find out how to display `BUFFER'."
  (cond ((eq erc-join-buffer 'window)
	 (if (active-minibuffer-window)
	     (display-buffer buffer)
	   (switch-to-buffer-other-window buffer)))
	((eq erc-join-buffer 'window-noselect)
	 (display-buffer buffer))
	((eq erc-join-buffer 'bury)
	 nil)
	((eq erc-join-buffer 'frame)
	 (funcall '(lambda (frame)
		     (raise-frame frame)
		     (select-frame frame))
		  (make-frame (or erc-frame-alist
				  default-frame-alist)))
	 (switch-to-buffer buffer)
	 (when erc-frame-dedicated-flag
	   (set-window-dedicated-p (selected-window) t)))
	(t
	 (if (active-minibuffer-window)
	     (display-buffer buffer)
	   (switch-to-buffer buffer)))))

(defvar erc-previous-read ""
  "Variable used to cache partially received lines.")
(make-variable-buffer-local 'erc-previous-read)

(defun erc (&optional server port nick full-name
		      connect passwd tgt-list channel process)
  "Run the Emacs Internet Relay Chat Client.

Connect to SERVER on PORT as NICK with FULL-NAME If CONNECT is
non-nil, connect to the server.  Otherwise assume already
connected and just create a separate buffer for the new target
CHANNEL.  Use PASSWD as user password on the server.  If TGT-LIST
is non-nil, use it to initialise `erc-default-recipients'.

Returns the buffer for the given server or channel."
  (let ((announced-server-name (when (and (boundp 'erc-session-server)
					  (string= server erc-session-server)
					  (boundp 'erc-announced-server-name))
				 erc-announced-server-name))
	(connected-p (unless connect erc-connected))
	(buffer (erc-get-buffer-create server port channel)))
    (unless (eq buffer (current-buffer))
      (when erc-log-p
	;; we can't log to debug buffer, it may not exist yet
	(message "erc: current buffer %s, switching to %s"
		 (current-buffer) buffer))
      (erc-setup-buffer buffer))
    (set-buffer buffer)
    (setq erc-active-buffer buffer)
    (erc-mode)
    (setq erc-announced-server-name announced-server-name)
    (setq erc-connected connected-p)
    ;; go to the end of the buffer and open a new line
    ;; (the buffer may have existed)
    (goto-char (point-max))
    (open-line 1)
    (goto-char (point-max))
    ;; connection parameters
    (setq erc-process process)
    (setq erc-insert-marker (make-marker))
    (set-marker erc-insert-marker (point))
    ;; stack of default recipients
    (setq erc-default-recipients tgt-list)
    ;; stack for user's nicknames
    (setq current-nick nil)
    ;; Initialize erc-server-users and erc-channel-users
    (if connect
	(progn ;; server buffer
	  (setq erc-server-users
		(make-hash-table :test 'equal))
	  (setq erc-channel-users nil))
      (progn ;; target buffer
	(setq erc-server-users nil)
	(setq erc-channel-users
	      (make-hash-table :test 'equal))))
    ;; clear last incomplete line read
    (setq erc-previous-read "")
    (setq channel-topic "")
    ;; limit on the number of users on the channel (mode +l)
    (setq channel-user-limit nil)
    (setq channel-key nil)
    ;; last peers (sender and receiver)
    (setq last-peers '(nil . nil))
    ;; last invitation channel
    (setq invitation nil)
    ;; away flag
    ;; Should only be used in session-buffers
    (setq away (let ((serverbuf (erc-server-buffer)))
		 (and serverbuf (with-current-buffer serverbuf away))))
    ;; Server channel list
    (setq channel-list ())
;;;; Some flood protection stuff
    ;; time of last command sent
    ;; time of last CTCP response/request sent
    (setq last-sent-time (erc-current-time))
    (setq last-ping-time (erc-current-time))
    (setq last-ctcp-time (erc-current-time))
    (setq erc-lines-sent 0)
    (setq erc-bytes-sent 0)
    ;; user requested quit
    (setq quitting nil)
    ;; login-time 'nick in use' error
    (setq bad-nick nil)
    ;; whether we have logged in
    (setq erc-logged-in nil)
    ;; The local copy of `erc-nick' - the list of nicks to choose
    (setq erc-default-nicks (if (consp erc-nick) erc-nick (list erc-nick)))
    ;; password stuff
    (setq erc-session-password passwd)
    ;; debug output buffer
    (setq erc-dbuf
	  (when erc-log-p
	    (get-buffer-create (concat "*ERC-DEBUG: " server "*"))))
    (erc-determine-parameters server port nick full-name)
    ;; Saving log file on exit

    (run-hooks 'erc-connect-pre-hook)

    (if connect (erc-connect))
    (erc-update-mode-line)
    (goto-char (point-max))
    (open-line 1)
    (goto-char (point-max))
    (set-marker (process-mark erc-process) (point))
    (set-marker erc-insert-marker (point))
    (erc-display-prompt)
    (goto-char (point-max))

    buffer))

(defun erc-initialize-log-marker ()
  "Initialize the `erc-last-saved-position' marker to a sensible position."
    (setq erc-last-saved-position (make-marker))
    (move-marker erc-last-saved-position
		 (1- (marker-position erc-insert-marker))))

;; interactive startup

(defvar erc-server-history-list nil
  "IRC server interactive selection history list.")

(defvar erc-nick-history-list nil
  "Nickname interactive selection history list.")

(defun erc-already-logged-in (server port nick)
  "Return the buffers corresponding to a NICK on PORT of a session SERVER.
This is determined by looking for the appropriate buffer and checking
whether the connection is still alive.
If no buffer matches, return nil."
  (erc-buffer-list
   (lambda ()
     (and (erc-process-alive)
	  (string= erc-session-server server)
	  (erc-port-equal erc-session-port port)
	  (erc-current-nick-p nick)))))

(if (not (fboundp 'read-passwd))
    (defun read-passwd (prompt)
      "Substitute for read-passwd in early emacsen"
      (read-from-minibuffer prompt)))

(defcustom erc-before-connect nil
  "Hook called before connecting to a server.
This hook gets executed before `erc-select' actually invokes `erc-mode'
with your input data.  The functions in here get called with three
parameters, SERVER, PORT and NICK."
  :group 'erc-hooks
  :type 'hook)

(defcustom erc-after-connect nil
  "Hook called after connecting to a server.
This hook gets executed when an end of MOTD has been received.	All
functions in here get called with the parameters SERVER and NICK."
  :group 'erc-hooks
  :type 'hook)

;;;###autoload
(defun erc-select (&optional server port nick)
  "Interactively select connection parameters and run ERC.
Optional argument SERVER uses server as default for the input query.
Optional argument PORT uses passed port as default for the input query.
Optional argument NICK uses the passed nick as default for the input query."
  (interactive)
  (setq port (or port erc-port))
  (setq server (erc-compute-server server))
  (setq nick (erc-compute-nick nick))
  (let* ((server (read-from-minibuffer
		  "IRC server: " server nil nil 'erc-server-history-list))
	 (port
	  (erc-string-to-port
	   (read-from-minibuffer "IRC port: "
				 (erc-port-to-string (or port "ircd")))))
	 (nick
	  (if (erc-already-logged-in server port nick)
	      (read-from-minibuffer
	       (erc-format-message 'nick-in-use ?n nick)
	       nick
	       nil nil 'erc-nick-history-list)
	    (read-from-minibuffer
	     "Nickname: " nick
	     nil nil 'erc-nick-history-list)))
	 (passwd (if erc-prompt-for-password
		     (if (and erc-password
			      (y-or-n-p "Use the default password? "))
			 erc-password
		       (read-passwd "Password: "))
		   erc-password)))
    (when (and passwd (string= "" passwd))
      (setq passwd nil))
    (while (erc-already-logged-in server port nick)
      ;; hmm, this is a problem when using multiple connections to a bnc
      ;; with the same nick. Currently this code prevents using more than one
      ;; bnc with the same nick. actually it would be nice to have
      ;; bncs transparent, so that erc-compute-buffer-name displays
      ;; the server one is connected to.
      (setq nick (read-from-minibuffer
		  (erc-format-message 'nick-in-use ?n nick)
		  nick
		  nil nil 'erc-nick-history-list)))
    (run-hook-with-args 'erc-before-connect server port nick)
    (erc server port nick erc-user-full-name t passwd)))

(defun erc-select-ssl (&optional server port nick)
  "Interactively select SSL connection parameters and run ERC.
Optional argument SERVER uses server as default for the input query.
Optional argument PORT uses passed port as default for the input query.
Optional argument NICK uses the passed nick as default for the input query."
  (interactive)
  (let ((erc-connect-function 'erc-open-ssl-stream))
    (erc-select server port nick)))

(defun erc-open-ssl-stream (name buffer host port)
  "Open an SSL stream to an IRC server.
The process will be given the name NAME, its target buffer will be
BUFFER.	 HOST and PORT specify the connection target."
  (when (require 'ssl)
    (let ((proc (open-ssl-stream name buffer host port)))
      ;; Ugly hack, but it works for now. Problem is it is
      ;; very hard to detect when ssl is established, because s_client
      ;; doesn't give any CONNECTIONESTABLISHED kind of message, and
      ;; most IRC servers send nothing and wait for you to identify.
      (sit-for 5)
      proc)))

;;; process management

(defcustom erc-connect-function
  (if (and (fboundp 'open-network-stream-nowait)
           ;; CVS Emacs claims to define open-network-stream-nowait on
           ;; windows, however, it does, in fact, not work.
           (not (memq system-type '(windows-nt cygwin ms-dos darwin))))
      'open-network-stream-nowait
    'open-network-stream)
  "Function used to initiate a connection.
It should take same arguments as `open-network-stream' does."
  :group 'erc
  :type 'function)

(defun erc-connect ()
  "Perform the connection and login."
  (let ((msg (erc-format-message 'connect
			     ?S erc-session-server
			     ?p erc-session-port)))
    (message "%s" msg)
    (setq erc-process
	  (funcall erc-connect-function
		   (format "erc-%s-%s" erc-session-server erc-session-port)
		   (current-buffer)
		   erc-session-server
		   erc-session-port))
    (message "%s...done" msg))
  (set-process-sentinel erc-process 'erc-process-sentinel)
  (set-process-filter erc-process 'erc-process-filter)
  ;; we do our own encoding and decoding
  (when (fboundp 'set-process-coding-system)
    (set-process-coding-system erc-process 'raw-text 'raw-text))
  (set-marker (process-mark erc-process) (point))
  (set-marker erc-insert-marker (point))
  (erc-log "\n\n\n********************************************\n")
  (message (erc-format-message 'login ?n (erc-current-nick)))
  ;; wait with script loading until we receive a confirmation (first
  ;; MOTD line)
  (if (eq erc-connect-function 'open-network-stream-nowait)
      ;; it's a bit unclear otherwise that it's attempting to establish a
      ;; connection
      (erc-display-message nil nil (current-buffer)
			   "Opening connection..\n")
    (erc-login)))

(defun erc-split-multiline (string)
  "Split STRING, containing multiple lines and return them in a list.
This function is closely tied in to `erc-process-filter'.
It uses a buffer local variable called `erc-previous-read' to handle partial
strings."
  (let ((i0 0)
	(doit t) l)
    (while doit
      (let ((i (string-match "\r?\n" string i0))
	    (s (substring string i0)))
	(cond (i
	       (setq l (cons (concat erc-previous-read (substring string i0 i))
			     l))
	       (setq erc-previous-read "")
	       (setq i0 (match-end 0)))
	      ((> (length s) 0)
	       (setq erc-previous-read (concat erc-previous-read s))
	       (setq doit nil))
	      (t (setq doit nil)))))
    (nreverse l)))

(defvar erc-debug-irc-protocol nil
  "If non-nil, log all IRC protocol traffic to the buffer \"*erc-protocol*\".

The buffer is created if it doesn't exist.

NOTE: If this variable is non-nil, and you kill the the only
visible \"*erc-protocol*\" buffer, it will be recreated shortly,
but you won't see it.

WARNING: Do not set this variable directly!  Instead, use the
function `erc-toggle-debug-irc-protocol' to toggle its value.")

(defun erc-log-irc-protocol (string &optional outbound)
  "Append STRING to the buffer *erc-protocol*.

This only has any effect if `erc-debug-irc-protocol' is non-nil.

The buffer is created if it doesn't exist.

If OUTBOUND is non-nil, STRING is being sent to the IRC server
and appears in erc-input-face in the buffer."
  (when erc-debug-irc-protocol
    (with-current-buffer (get-buffer-create "*erc-protocol*")
      (save-excursion
	(goto-char (point-max))
	(let ((inhibit-read-only t))
	  (insert (if (not outbound)
		      string
		    (erc-propertize
		     (concat ">>> " string
			     (if (/= ?\n
				     (aref string
					   (1- (length string))))
				 "\n"))
		     'face 'erc-input-face)))))
      (let ((orig-win (selected-window))
	    (debug-buffer-window (get-buffer-window (current-buffer) t)))
	(when debug-buffer-window
	  (select-window debug-buffer-window)
	  (when (= 1 (count-lines (point) (point-max)))
	    (goto-char (point-max))
	    (recenter -1))
	  (select-window orig-win))))))

(defun erc-toggle-debug-irc-protocol (&optional arg)
  "Toggle the value of `erc-debug-irc-protocol'.

If ARG is non-nil, show the *erc-protocol* buffer."
  (interactive "P")
  (let* ((buf (get-buffer-create "*erc-protocol*")))
    (with-current-buffer buf
      (view-mode 1)
      (when (null (current-local-map))
	(use-local-map (make-sparse-keymap))
	(local-set-key (kbd "RET") 'erc-toggle-debug-irc-protocol))
      (add-hook 'kill-buffer-hook
		#'(lambda () (setq erc-debug-irc-protocol nil))
		nil 'local)
      (goto-char (point-max))
      (let ((inhibit-read-only t))
	(insert (erc-make-notice
		 (format "IRC protocol logging %s at %s\n"
			 (if erc-debug-irc-protocol "disabled" "enabled")
			 (current-time-string))))))
    (setq erc-debug-irc-protocol (not erc-debug-irc-protocol))
    (if (and arg
	     (not (get-buffer-window "*erc-protocol*" t)))
	(display-buffer buf t))
    (message "IRC protocol traffic logging %s (see buffer *erc-protocol*)."
	     (if erc-debug-irc-protocol "enabled" "disabled"))))

(defun erc-process-filter (proc string)
  "Filter function for incoming server traffic.
PROC is the process where input came from and
STRING is the string received."
  (erc-log-irc-protocol string)
  (with-current-buffer (process-buffer proc)
    (mapc (lambda (line)
	    (erc-parse-line-from-server proc line))
	  (erc-split-multiline string))))

(defun erc-process-sentinel (cproc event)
  "Sentinel function for ERC process."
  (erc-log (format
	    "SENTINEL: proc: %S	 status: %S  event: %S (quitting: %S)"
	    erc-process (process-status erc-process) event quitting))
  (save-excursion
    (set-buffer (process-buffer cproc))
    (cond
     ((equal event "open\n")
      ;; newly opened connection for a nowait connection
      (erc-login))
     ;; ((eq event 'failed)))
     ;; default to this
     (t
      (erc-with-all-buffers-of-server cproc nil (setq erc-connected nil))
      (when erc-ping-handler
	(with-current-buffer (current-buffer)
	  (progn (cancel-timer erc-ping-handler)
		 (setq erc-ping-handler nil))))
      (run-hook-with-args 'erc-disconnected-hook
			  (erc-current-nick) (system-name) "")
      (if (string= event "exited abnormally with code 256\n")

	  ;; Sometimes (eg on a /LIST command) ERC happens to die with
	  ;; an exit code 256. The icrii client also shows this behavior
	  ;; and it restarts itself. So do I.

	  (cond
	   ((not quitting)
	    (erc-display-message nil 'error (current-buffer)
				 (if erc-auto-reconnect
				     'disconnected
				   'disconnected-noreconnect))
	    (erc-update-mode-line)
	    (setq erc-active-buffer (current-buffer))
	    (setq last-sent-time 0)
	    (setq erc-lines-sent 0)
	    (when erc-auto-reconnect
	      (erc erc-session-server erc-session-port current-nick
		   erc-session-user-full-name t erc-session-password))
	    (goto-char (point-max)))
	   (t
	    (let* ((wd (window-width))
		   (msg "*** ERC finished ***")
		   (off (/ (- wd (length msg)) 2))
		   (s ""))
	      (if (> off 0)
		  (setq s (make-string off ? )))
	      (insert (concat "\n\n" s msg "\n")))))
	(insert (concat "\n\n*** ERC terminated: " event "\n"))))
     (goto-char (point-max))
     (erc-update-mode-line))))


;;; I/O interface

;; send interface

(defun erc-flood-exceeded-p (line)
  "Determines whether the flood limits are exceeded or not by the LINE.
It also maintains all the flood control variables."
  ;; First, clean up if no messages for long enough time
  (let ((flood (cond ((eq erc-flood-protect 'strict) erc-flood-limit2)
		     (erc-flood-protect erc-flood-limit)
		     (t nil))))
    (when (or (not flood)
	      (< (+ last-sent-time (nth 2 flood)) (erc-current-time)))
      (setq erc-lines-sent 0
	    erc-bytes-sent 0))
    ;; Update the variables for the new line
    (setq erc-lines-sent (1+ erc-lines-sent)
	  erc-bytes-sent (+ (length line) erc-bytes-sent)
	  last-sent-time (erc-current-time))
    ;; Now do what they ask
    (and flood
	 (let ((bytes (nth 0 flood))
	       (lines (nth 1 flood)))
	   (or (and lines (> erc-lines-sent lines))
	       (and bytes (> erc-bytes-sent bytes)))))))

(defun erc-send-command (l &optional force target)
  "Send command line L to IRC server.
If the optional FORCE is non-nil, send the command even if the
flood guard is in effect and the limit is exceeded.

Usually this function is called in a server buffer, so
`erc-default-target' will return nil.  In cases where encoding
and decoding of arguments is important, pass the current TARGET
along, so that we can pass it on to
`erc-encode-string-for-target'.

Return non-nil if the line is actually sent, nil otherwise.
The command line must contain neither prefix nor trailing `\\n'"
  (erc-log (concat "erc-send-command: " l "(" (buffer-name) ")"))
  (let ((ob (current-buffer))
	(buf (erc-server-buffer))
	(res t)
	commands exceeded)
    (if buf
	(with-current-buffer buf
	  ;; maybe split the command up into multiple commands, so as to avoid
	  ;; exceeding length restrictions
	  (mapc
	   (lambda (l)
	     (setq exceeded (erc-flood-exceeded-p l))
	     (if (or force (not exceeded))
		 (progn
		   (when exceeded
		     (message "Warning: flood exceeded, but send forced")
		     (erc-log-aux
		      (format (concat "ERC FLOOD PROTECT: flood exceeded,"
				      " but send forced on %S\n")
			      l)))
		   (erc-log-irc-protocol l 'outbound)
		   (process-send-string
		    erc-process
		    (concat (erc-encode-string-for-target l target) "\n")))
	       (unless erc-disable-ctcp-replies
		 (setq erc-disable-ctcp-replies t)
		 (erc-display-message nil 'error ob 'flood-ctcp-off))
	       (unless (eq erc-flood-protect 'strict)
		 (setq erc-flood-protect 'strict)
		 (erc-display-message nil 'error ob 'flood-strict-mode))
	       (message "ERC FLOOD PROTECT: line not sent: %S" l)
	       (erc-log-aux (format "ERC FLOOD PROTECT: line not sent: %S"
	       l))))
	   (erc-split-command l)))
      (message "ERC: No process running")
      (beep))
    res))

(defcustom erc-default-coding-system (if (and (fboundp 'coding-system-p)
					      (coding-system-p 'undecided)
					      (coding-system-p 'utf-8))
					 '(utf-8 . undecided)
				       nil)
  "The default coding system for incoming and outgoing text.
This is either a coding system, a cons, or a function.

If a cons, the encoding system for outgoing text is in the car
and the decoding system for incoming text is in the cdr.  The
most interesting use for this is to put `undecided' in the cdr.
If a function, it is called with no should return a coding system
or a cons as described above.  Note that you can use the
dynamically bound variable `target' to get the current target.
See `erc-coding-system-for-target'.

If you need to send non-ASCII text to people not using a client that
does decoding on its own, you must tell ERC what encoding to use.
Emacs cannot guess it, since it does not know what the people on the
other end of the line are using."
  :group 'erc
  :type '(choice (const :tag "None" nil)
		 coding-system
		 (cons (coding-system :tag "encoding" :value utf-8)
		       (coding-system :tag "decoding" :value undecided))
		 function))

(defcustom erc-encoding-coding-alist nil
  "Alist of target regexp and coding-system pairs to use.
This overrides `erc-default-coding-system' depending on the current target
as returned by `erc-default-target'.

Example: If you know that the channel #linux-ru uses the coding-system
`cyrillic-koi8', then add '(\"#linux-ru\" . cyrillic-koi8) to the
alist."
  :group 'erc
  :type '(repeat (cons (string :tag "Target")
		       coding-system)))

(defun erc-coding-system-for-target (target)
  "Return the coding system or cons cell appropriate for TARGET.
This is determined via `erc-encoding-coding-alist' or
`erc-default-coding-system'."
  (or (cdr (assoc target erc-encoding-coding-alist))
      (and (functionp erc-default-coding-system)
	   (funcall erc-default-coding-system))
      erc-default-coding-system))

(defun erc-encode-string-for-target (str target)
  "Encode STR as appropriate for TARGET.
This is indicated by `erc-encoding-coding-alist', defaulting to the value of
`erc-default-coding-system'."
  (let ((coding (erc-coding-system-for-target target)))
    (when (consp coding)
      (setq coding (car coding)))
    (erc-encode-coding-string str coding)))

(defun erc-decode-string-from-target (str target)
  "Decode STR as appropriate for TARGET.
This is indicated by `erc-encoding-coding-alist', defaulting to the value of
`erc-default-coding-system'."
  (let ((coding (erc-coding-system-for-target target)))
    (when (consp coding)
      (setq coding (cdr coding)))
    (erc-decode-coding-string str coding)))

(defun erc-split-command (line)
  "Split LINE into manageable chunks.

LINE is split so that no single command is longer than 510 bytes (the max
length of a command according to the the IRC RFC).

LINE is returned as a list of strings.

Signals an error if the line is too long and it doesn't make
sense to split it."
  (let ((splittable "^\\(PRIVMSG\\|NOTICE\\)")
	;; IRC RFC says max 510 for the whole command, which means
	;; :nick!user@host COMMAND args :arg
	;; FIXME: We ought to calculate the length of :nick!user@host --
	;; Lawrence 2004-01-08
	;; So we're forced to go with a conservative estimate
	(maxlen 450)
	preamble lines div curline ctcpp)
    (if (> (length line) maxlen)
      (if (string-match splittable line)
	  (progn
	    ;; extract the command info
	    (setq div (1+ (string-match ":" line)))
	    (setq preamble (substring line 0 div))
	    (setq line (substring line div (length line)))
	    (when (string-match "^\\(\C-a[A-Z]* \\)\\(.*\\)\C-a" line)
	      (setq preamble (concat preamble
				     (match-string 1 line)))
	      (setq line (match-string 2 line))
	      (setq ctcpp t))
	    (setq div (- maxlen (length preamble) 1))
	    (while line
	      ;; a splitting algorithm that favoured not breaking words
	      ;; would be preferable here
	      (setq div (min div (length line)))
	      (setq curline (substring line 0 div))
	      (if (= div (length line))
		  (setq line nil)
		(setq line (substring line div (length line))))
	      (add-to-list 'lines (concat preamble
					  curline
					  (if ctcpp
					      "\C-a"
					    ""))
			   t)))
	(error "Oversized non-splittable command: %s" line))
      ;; if the line was smaller, return it as a list
      (setq lines (list line)))
    lines))

(defun erc-send-ctcp-message (tgt l &optional force)
  "Send CTCP message L to TGT.

If TGT is nil the message is not sent.
The command must contain neither a prefix nor a trailing `\\n'.

See also `erc-send-command'."
  (cond
   (tgt
    (erc-log (format "erc-send-CTCP-message: [%s] %s" tgt l))
    (erc-send-command (format "PRIVMSG %s :\C-a%s\C-a" tgt l) force))))

(defun erc-send-ctcp-notice (tgt l &optional force)
  "Send CTCP notice L to TGT.

If TGT is nil the message is not sent.
The command must contain neither a prefix nor a trailing `\\n'.

See also `erc-send-command'."
  (cond
   (tgt
    (erc-log (format "erc-send-CTCP-notice: [%s] %s" tgt l))
    (erc-send-command (format "NOTICE %s :\C-a%s\C-a" tgt l) force))))

(defun erc-send-action (tgt str &optional force)
  "Send CTCP ACTION information described by STR to TGT."
  (erc-send-ctcp-message tgt (format "ACTION %s" str) force)
  (erc-display-message
   nil 'input (current-buffer)
   'ACTION ?n (erc-current-nick) ?a str ?u "" ?h ""))

(defvar erc-noncommands-list '(erc-cmd-ME
			       erc-cmd-COUNTRY
			       erc-cmd-SV
			       erc-cmd-SM
			       erc-cmd-SMV
			       erc-cmd-LASTLOG)
  "List of commands that are aliases for CTCP ACTION or for erc messages.

If a command's function symbol is in this list, the typed command
does not appear in the ERC buffer after the user presses ENTER.")

;; Display interface

(defun erc-string-invisible-p (string)
  "Check whether STRING is invisible or not.
I.e. any char in it has the `invisible' property set."
  (text-property-any 0 (length string) 'invisible t string))

(defun erc-display-line-1 (string buffer)
  "Display STRING in `erc-mode' BUFFER.
Auxiliary function used in `erc-display-line'.	The line gets filtered to
interpret the control characters.  Then, `erc-insert-pre-hook' gets called.
If `erc-insert-this' is still t, STRING gets inserted into the buffer.
Afterwards, `erc-insert-modify' and `erc-insert-post-hook' get called.
If STRING is nil, the function does nothing."
  (when string
    (save-excursion
      (set-buffer (or buffer (process-buffer erc-process)))
      (let ((insert-position (or (marker-position erc-insert-marker)
				 (point-max))))
	(let ((string (erc-interpret-controls string))
	      (buffer-undo-list t)
	      (inhibit-read-only t))
	  (unless (string-match "\n$" string)
	    (setq string (concat string "\n"))
	    (when (erc-string-invisible-p string)
	      (erc-put-text-properties 0 (length string) string
				       '(invisible intangible))))
	  (erc-log (concat "erc-display-line: " string
			   (format "(%S)" string) " in buffer "
			   (format "%s" buffer)))
	  (setq erc-insert-this t)
	  (run-hook-with-args 'erc-insert-pre-hook string)
	  (if (null erc-insert-this)
	      ;; Leave erc-insert-this set to t as much as possible.  Fran
	      ;; Litterio <franl> has seen erc-insert-this set to nil while
	      ;; erc-send-pre-hook is running, which should never happen.  This
	      ;; may cure it.
	      (setq erc-insert-this t)
	    (save-excursion ;; to restore point in the new buffer
	      (goto-char insert-position)
	      (insert-before-markers string)
	      ;; run insertion hook, with point at restored location
	      (save-restriction
		(narrow-to-region insert-position (point))
		(run-hooks 'erc-insert-modify-hook)
		(run-hooks 'erc-insert-post-hook)))))
	(erc-update-undo-list (- (or (marker-position erc-insert-marker)
				     (point-max))
				 insert-position))))))

(defun erc-update-undo-list (shift)
  ;; Translate buffer positions in buffer-undo-list by SHIFT.
  (unless (or (zerop shift) (atom buffer-undo-list))
    (let ((list buffer-undo-list) elt)
      (while list
	(setq elt (car list))
	(cond ((integerp elt)		; POSITION
	       (incf (car list) shift))
	      ((or (atom elt)		; nil, EXTENT
		   ;; (eq t (car elt))	; (t HIGH . LOW)
		   (markerp (car elt)))	; (MARKER . DISTANCE)
	       nil)
	      ((integerp (car elt))	; (BEGIN . END)
	       (incf (car elt) shift)
	       (incf (cdr elt) shift))
	      ((stringp (car elt))	; (TEXT . POSITION)
	       (incf (cdr elt) (* (if (natnump (cdr elt)) 1 -1) shift)))
	      ((null (car elt))	        ; (nil PROPERTY VALUE BEG . END)
	       (let ((cons (nthcdr 3 elt)))
		 (incf (car cons) shift)
		 (incf (cdr cons) shift)))
	      ((and (featurep 'xemacs)
		    (extentp (car elt))) ; (EXTENT START END)
	       (incf (nth 1 elt) shift)
	       (incf (nth 2 elt) shift)))
	(setq list (cdr list))))))

(defvar erc-valid-nick-regexp "[]a-zA-Z^[;\\`_{}|][]^[;\\`_{}|a-zA-Z0-9-]*"
  "Regexp which matches all legal characters in a IRC nickname.")

(defun erc-is-valid-nick-p (nick)
  "Check if NICK is a valid IRC nickname."
  (string-match (concat "^" erc-valid-nick-regexp "$") nick))

(defun erc-scroll-to-bottom (window display-start)
  "Recenter WINDOW so that `point' is on the last line.

This is added to `window-scroll-functions' by `erc-add-scroll-to-bottom'.

You can control which line is recentered to by customizing the
variable `erc-input-line-position'.

DISPLAY-START is ignored."
  (if (and window (window-live-p window))
      ;; Temporarily bind resize-mini-windows to nil so that users who have it
      ;; set to a non-nil value will not suffer from premature minibuffer
      ;; shrinkage due to the below recenter call.  I have no idea why this
      ;; works, but it solves the problem, and has no negative side effects.
      ;; (Fran Litterio, 2003/01/07)
      (let ((resize-mini-windows nil))
	(save-selected-window
	  (select-window window)
	  (save-restriction
	    (widen)
	    (when (> (point) erc-insert-marker) ; we're editing a line. Scroll.
	      (save-excursion
		;; antifuchs removed (goto-char
		;; (point-max)), as it messed up redisplay.
		(recenter (or erc-input-line-position -1))
		(sit-for 0))))))))

(defun erc-add-scroll-to-bottom ()
  "A hook function for `erc-mode-hook' to recenter output at bottom of window.

If you find that ERC hangs when using this function, try customizing
the value of `erc-input-line-position'.

This works whenever scrolling happens, so it's added to
`window-scroll-functions' rather than `erc-insert-post-hook'."
  ;;(make-local-hook 'window-scroll-functions)
  (add-hook 'window-scroll-functions 'erc-scroll-to-bottom nil t))

(defun erc-display-line (string &optional buffer)
  "Display STRING in the ERC BUFFER.
All screen output must be done through this function.  If BUFFER is nil
or omitted, the default ERC buffer for the `erc-session-server' is used.
The BUFFER can be an actual buffer, a list of buffers, 'all or 'active.
If BUFFER = 'all, the string is displayed in all the ERC buffers for the
current session.  'active means the current active buffer
\(`erc-active-buffer').	 If the buffer can't be resolved, the current
buffer is used.	 `erc-display-line-1' is used to display STRING.

If STRING is nil, the function does nothing."
  (let ((inhibit-point-motion-hooks t)
	new-bufs)
    (dolist (buf (cond
		  ((bufferp buffer) (list buffer))
		  ((listp buffer) buffer)
		  ((processp buffer) (list (process-buffer buffer)))
		  ((eq 'all buffer)
		   (and (boundp 'erc-process)
			;; Hmm, or all of the same session server?
			(erc-buffer-list nil erc-process)))
		  ((eq 'active buffer)
		   (and erc-active-buffer (list erc-active-buffer)))
		  ((and (boundp 'erc-process) (processp erc-process))
		   (list (process-buffer erc-process)))
		  (t (list (current-buffer)))))
      (when (buffer-live-p buf)
	(erc-display-line-1 string buf)
	(add-to-list 'new-bufs buf)))
    (when (null new-bufs)
      (if (and (boundp 'erc-process) (processp erc-process))
	  (erc-display-line-1 string (process-buffer erc-process))
	(erc-display-line-1 string (current-buffer))))))

(defun erc-display-message-highlight (type string)
  "Highlight STRING according to TYPE, where erc-TYPE-face is an erc face.

See also `erc-make-notice'"
  (cond ((eq type 'notice)
	 (erc-make-notice string))
	(t
	 (erc-put-text-property
	  0 (length string)
	  'face (or (intern-soft
		     (concat "erc-" (symbol-name type) "-face"))
		    "erc-default-face")
	  string)
	 string)))

(defun erc-display-message (parsed type buffer msg &rest args)
  "Display MSG in BUFFER.

ARGS, PARSED, and TYPE are used to format MSG sensibly.

See also `erc-format-message' and `erc-display-line'."
  (let ((string (if (symbolp msg)
		    (apply 'erc-format-message msg args)
		  msg)))
    (setq string
	  (cond
	   ((listp type)
	    (mapc (lambda (type)
		    (setq string
			  (erc-display-message-highlight type string)))
		  type)
	    string)
	   ((symbolp type)
	    (erc-display-message-highlight type string))))

    (when (vectorp parsed)
      (erc-put-text-property 0 (length string) 'erc-parsed parsed string)
      (erc-put-text-property 0 (length string) 'rear-sticky t string))
    (erc-display-line string buffer)))

(defun erc-message-type-member (position list)
  "Return non-nil if the erc-parsed text-property at POSITION is in LIST.

This function relies on the erc-parsed text-property being
present."
  (let ((prop-val (get-text-property position 'erc-parsed)))
    (and prop-val (member (aref prop-val 0) list))))

(defvar erc-send-input-line-function 'erc-send-input-line)
(make-variable-buffer-local 'erc-send-input-line-function)

(defun erc-send-input-line (target line &optional force)
  "Send LINE to TARGET.

See also `erc-send-command'."
  (erc-send-command (format "PRIVMSG %s :%s"
			    target
			    ;; If the line is empty, we still want to
			    ;; send it - i.e. an empty pasted line.
			    (if (string= line "\n")
				" \n"
			      line))
		    force
		    target))

;; if we're in emacs21 CVS, we use help-function-arglist which is more
;; sophisticated and can handle subrs, etc
(if (fboundp 'help-function-arglist)
    (defalias 'erc-get-arglist 'help-function-arglist)
  (defun erc-get-arglist (fun)
    "Returns the arglist signature of FUN"
    (let ((def (symbol-function fun)))
      (ignore-errors
	;; load an autoloaded function first
	(when (equal 'autoload (car-safe def))
          (load (second def))
          (setq def (symbol-function fun)))
	(if (listp def)
	    (second def)
	  (format "[Arglist not available, try %s instead]"
		  (substitute-command-keys "\\[describe-function]")))))))

(defun erc-process-input-line (line &optional force no-command)
  "Translate LINE to an RFC1459 command and send it based.
Returns non-nil if the command is actually sent to the server, and nil
otherwise.

If the command in the LINE is not boundp as a function `erc-cmd-COMMAND',
it is passed to `erc-cmd-default'.  If LINE is not a command (ie. doesn't
start with /<COMMAND>) then it is sent as a message.

An optional FORCE argument forces sending the line when flood
protection is in effect.  The optional NO-COMMAND argument prohibits
this function from interpreting the line as a command."
  (let ((command-list (erc-extract-command-from-line line)))
    (if	(and command-list
	     (not no-command))
	(let* ((cmd  (nth 0 command-list))
	       (args (nth 1 command-list)))
	  (condition-case nil
	      (if (listp args)
		  (apply cmd args)
		(funcall cmd args))
	    (wrong-number-of-arguments
	     (erc-display-message nil 'error (current-buffer) 'incorrect-args
				  ?c cmd
				  ?u (erc-get-arglist cmd)
				  ?d (format "%s\n"
					     (or (documentation cmd) ""))))))
      (let ((r (erc-default-target)))
	(if r
	    (funcall erc-send-input-line-function r line force)
	  (erc-display-message nil 'error (current-buffer) 'no-target)
	  nil)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;		      Input commands handlers
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun erc-cmd-AMSG (line)
  "Send LINE to all channels of the current server that you are on."
  (interactive "sSend to all channels you're on: ")
  (setq line (erc-trim-string line))
  (erc-with-all-buffers-of-server nil
   (lambda ()
     (erc-channel-p (erc-default-target)))
   (erc-send-message line)))
(put 'erc-cmd-AMSG 'do-not-parse-args t)

(defun erc-cmd-SAY (line)
  "Send LINE to the current query or channel as a message, not a command.

Use this when you want to send a message with a leading '/'.  Note
that since multi-line messages are never a command, you don't
need this when pasting multiple lines of text."
  (if (string-match "^\\s-*$" line)
      nil
    (erc-process-input-line line nil t)))
(put 'erc-cmd-SAY 'do-not-parse-args t)

(defun erc-cmd-SET (line)
  "Set the variable named by the first word in LINE to some VALUE.
VALUE is computed by evaluating the rest of LINE in Lisp."
  (cond
   ((string-match "^\\s-*\\(\\S-+\\)\\s-+\\(.*\\)$" line)
    (let ((var (read (concat "erc-" (match-string 1 line))))
	  (val (read (match-string 2 line))))
      (if (boundp var)
	  (progn
	    (set var (eval val))
	    (erc-display-message
	     nil nil 'active (format "Set %S to %S" var val))
	    t)
	(setq var (read (match-string 1 line)))
	(if (boundp var)
	    (progn
	      (set var (eval val))
	      (erc-display-message
	       nil nil 'active (format "Set %S to %S" var val))
	      t)
	  (erc-display-message nil 'error 'active 'variable-not-bound)
	  nil))))
   ((string-match "^\\s-*$" line)
    (erc-display-line
     (concat "Available user variables:\n"
	     (apply
	      'concat
	      (mapcar
	       (lambda (var)
		 (let ((val (symbol-value var)))
		   (concat (format "%S:" var)
			   (if (consp val)
			       (concat "\n" (pp-to-string val))
			     (format " %S\n" val)))))
	       (apropos-internal "^erc-" 'user-variable-p))))
     (current-buffer)) t)
   (t nil)))
(defalias 'erc-cmd-VAR 'erc-cmd-SET)
(defalias 'erc-cmd-VARIABLE 'erc-cmd-SET)
(put 'erc-cmd-SET 'do-not-parse-args t)

(defun erc-cmd-default (line)
  "Fallback command.

Commands for which no erc-cmd-xxx exists, are tunnelled through
this function.  LINE is sent to the server verbatim, and
therefore has to contain the command itself as well."
  (erc-log (format "cmd: DEFAULT: %s" line))
  (erc-send-command (substring line 1))
  t)

(defun erc-cmd-IGNORE (&optional user)
  "Ignore USER.  This should be a regexp matching nick!user@host.
If no USER argument is specified, list the contents of `erc-ignore-list'."
  (if user
      (progn
	(erc-display-line
	 (erc-make-notice (format "Now ignoring %s" user))
	 'active)
	(with-current-buffer (erc-server-buffer)
	  (add-to-list 'erc-ignore-list user)))
    (if (null (with-current-buffer (erc-server-buffer) erc-ignore-list))
	(erc-display-line (erc-make-notice "Ignore list is empty") 'active)
      (erc-display-line (erc-make-notice "Ignore list:") 'active)
      (mapc #'(lambda (item)
		(erc-display-line (erc-make-notice item)
				  'active))
	    (with-current-buffer (erc-server-buffer) erc-ignore-list))))
  t)

(defun erc-cmd-UNIGNORE (user)
  "Remove the user specified in USER from the ignore list."
  (erc-display-line
   (erc-make-notice (format "Not ignoring %s" user))
   'active)
  (with-current-buffer (erc-server-buffer)
    (setq erc-ignore-list (delete user erc-ignore-list)))
  t)

(defun erc-cmd-CLEAR ()
  "Clear the window content."
  (recenter 0)
  t)

(defun erc-cmd-OPS ()
  "Show the ops in the current channel."
  (interactive)
  (let ((ops nil))
    (if erc-channel-users
	(maphash (lambda (nick user-data)
		   (let ((cuser (cdr user-data)))
		     (if (and cuser
			      (erc-channel-user-op cuser))
			 (setq ops (cons (erc-server-user-nickname (car user-data))
					 ops)))))
		 erc-channel-users))
    (setq ops (sort ops 'string-lessp))
    (setq ops (if ops
		  (format "%d operator%s: %s" (length ops)
			  (if (> (length ops) 1) "s" "")
			  (mapconcat 'identity ops " "))
		"No operators in this channel."))
    (erc-display-line (erc-make-notice ops) (current-buffer)))
  t)

(defun erc-cmd-COUNTRY (tld)
  "Display the country associated with the top level domain TLD."
  (require 'mail-extr)
  (let ((co (ignore-errors (what-domain tld))))
    (erc-display-message
     nil 'notice 'active
     (or co (concat tld ": No such domain")))
  t))

(defun erc-cmd-AWAY (line)
  "Mark the user as being away, the reason being indicated by LINE.
If no reason is given, unset away status."
  (when (string-match "^\\s-*\\(.*\\)$" line)
    (let ((reason (match-string 1 line)))
      (erc-log (format "cmd: AWAY: %s" reason))
      (erc-send-command
       (if (string= reason "")
	   "AWAY"
	 (concat "AWAY :" reason))))
    t))
(put 'erc-cmd-AWAY 'do-not-parse-args t)

(defun erc-cmd-GAWAY (line)
  "Mark the user as being away everywhere, the reason being indicated by LINE."
  ;; on all server buffers.
  (erc-with-all-buffers-of-server nil
   #'erc-server-buffer-p
   (erc-cmd-AWAY line)))
(put 'erc-cmd-GAWAY 'do-not-parse-args t)

(defun erc-cmd-CTCP (nick cmd &rest args)
  "Send a Client To Client Protocol message to NICK.

CMD is the CTCP command, possible values being ECHO, FINGER, CLIENTINFO, TIME,
VERSION and so on.  It is called with ARGS."
  (let ((str (concat cmd
		     (when args
		       (concat " " (mapconcat #'identity args " "))))))
    (erc-log (format "cmd: CTCP [%s]: [%s]" nick str))
    (erc-send-ctcp-message nick str)
    t))

(defun erc-cmd-HELP (&optional func)
  "Popup help information.

If FUNC contains a valid function or variable, help about that
will be displayed.  If FUNC is empty, display an apropos about
erc commands.  Otherwise, do apropos in the erc namespace
\(\"erc-.*LINE\"\).

Examples:
To find out about erc and bbdb, do
  /help bbdb.*

For help about the WHOIS command, do:
  /help whois

For a list of user commands (/join /part, ...):
  /help."
  (if func
    (let* ((sym (or (let ((sym (intern-soft
				(concat "erc-cmd-" (upcase func)))))
		      (if (and sym (or (boundp sym) (fboundp sym)))
			  sym
			nil))
		    (let ((sym (intern-soft func)))
		      (if (and sym (or (boundp sym) (fboundp sym)))
			  sym
			nil))
		    (let ((sym (intern-soft (concat "erc-" func))))
		      (if (and sym (or (boundp sym) (fboundp sym)))
			  sym
			nil)))))
      (if sym
	  (cond
	   ((boundp sym) (describe-variable sym))
	   ((fboundp sym) (describe-function sym))
	   (t nil))
	(apropos-command (concat "erc-.*" func) nil
			 (lambda (x)
			   (or (commandp x)
			       (get x 'custom-type))))
	t))
    (apropos "erc-cmd-")
    (message "Type C-h m to get additional information about keybindings.")
    t))

(defalias 'erc-cmd-H 'erc-cmd-HELP)

(defun erc-cmd-JOIN (channel &optional key)
  "Join the channel given in CHANNEL, optionally with KEY.
If CHANNEL is specified as \"-invite\", join the channel to which you
were most recently invited.  See also `invitation'."
  (let (chnl)
    (if (string= (upcase channel) "-INVITE")
	(if invitation
	    (setq chnl invitation)
	  (erc-display-message nil 'error (current-buffer) 'no-invitation))
      (setq chnl (erc-ensure-channel-name channel)))

    ;; Prevent double joining of same channel on same server.
    (let ((joined-channels
	   (mapcar #'(lambda (chanbuf)
		       (with-current-buffer chanbuf (erc-default-target)))
		   (erc-channel-list erc-process))))
      (if (erc-member-ignore-case chnl joined-channels)
	  (switch-to-buffer (car (member-ignore-case chnl joined-channels)))
	(erc-log (format "cmd: JOIN: %s" chnl))
	(if (and chnl key)
	    (erc-send-command (format "JOIN %s %s" chnl key))
	  (erc-send-command (format "JOIN %s" chnl))))))
  t)

(defalias 'erc-cmd-CHANNEL 'erc-cmd-JOIN)
(defalias 'erc-cmd-J 'erc-cmd-JOIN)

(defvar erc-channel-new-member-names nil
  "If non-nil, a names list is currently being received.

If non-nil, this variable is a hash-table that associates
received nicks with t.")
(make-variable-buffer-local 'erc-channel-new-member-names)

(defun erc-cmd-NAMES (&optional channel)
  "Display the users in CHANNEL.
If CHANNEL is not specified, display the users in the current channel.
This function clears the channel name list first, then sends the
command."
  (let ((tgt (or (and (erc-channel-p channel) channel)
		 (erc-default-target))))
    (if (and tgt (erc-channel-p tgt))
	(progn
	  (erc-log (format "cmd: DEFAULT: NAMES %s" tgt))
	  (erc-with-buffer
	   (tgt)
	   (erc-channel-begin-receiving-names))
	  (erc-send-command (concat "NAMES " tgt)))
      (erc-display-message nil 'error (current-buffer) 'no-default-channel)))
  t)
(defalias 'erc-cmd-N 'erc-cmd-NAMES)

(defun erc-cmd-KICK (target &optional reason-or-nick &rest reasonwords)
  "Kick the user indicated in LINE from the current channel.
LINE has the format: \"#CHANNEL NICK REASON\" or \"NICK REASON\"."
  (let ((reasonstring (mapconcat 'identity reasonwords " ")))
    (if (string= "" reasonstring)
	(setq reasonstring (format "Kicked by %s" (erc-current-nick))))
    (if (erc-channel-p target)
	(let ((nick reason-or-nick))
	  (erc-log (format "cmd: KICK: %s/%s: %s" nick target reasonstring))
	  (erc-send-command (format "KICK %s %s :%s" target nick reasonstring))
	  t)
      (when target
	(let ((ch (erc-default-target)))
	  (setq reasonstring (concat
			      (if reason-or-nick (concat reason-or-nick " "))
			      reasonstring))
	  (if ch
	      (progn
		(erc-log
		 (format "cmd: KICK: %s/%s: %s" target ch reasonstring))
		(erc-send-command
		 (format "KICK %s %s :%s" ch target reasonstring)))
	    (erc-display-message nil 'error (current-buffer)
				 'no-default-channel))
	  t)))))

(defvar erc-script-args nil)

(defun erc-cmd-LOAD (line)
  "Load the script provided in the LINE.
If LINE continues beyond the file name,
the rest of it is put in a (local) variable
`erc-script-args', which can be used in elisp scripts.

The optional FORCE argument is ignored here - you can't force loading
a script after exceeding the flood threshold."
  (cond
   ((string-match "^\\s-*\\(\\S-+\\)\\(.*\\)$" line)
    (let* ((file-to-find (match-string 1 line))
	   (erc-script-args (match-string 2 line))
	   (file (erc-find-file file-to-find erc-script-path)))
      (erc-log (format "cmd: LOAD: %s" file-to-find))
      (cond
       ((not file)
	(erc-display-message nil 'error (current-buffer)
			     'cannot-find-file ?f file-to-find))
       ((not (file-readable-p file))
	(erc-display-message nil 'error (current-buffer)
			     'cannot-read-file ?f file))
       (t
	(message "Loading \'%s\'..." file)
	(erc-load-script file)
	(message "Loading \'%s\'...done" file))))
    t)
   (t nil)))

(defun erc-cmd-WHOIS (user &optional server)
  "Display whois information for USER.

If SERVER is non-nil, use that, rather than the current server."
  ;; FIXME: is the above docstring correct?  -- Lawrence 2004-01-08
  (let ((send (if server
		  (format "WHOIS %s %s" user server)
		(format "WHOIS %s" user))))
    (erc-log (format "cmd: %s" send))
    (erc-send-command send)
  t))
(defalias 'erc-cmd-WI 'erc-cmd-WHOIS)

(defun erc-cmd-WHOAMI ()
  "Display whois information about yourself."
  (erc-cmd-WHOIS (erc-current-nick))
  t)

(defun erc-cmd-IDLE (nick)
  "Show the length of time NICK has been idle."
  (let ((serverbuf (erc-server-buffer))
	(origbuf (current-buffer))
	symlist)
    (with-current-buffer serverbuf
      (add-to-list 'symlist
		   (cons (erc-once-with-server-event
			  311 `(string= ,nick (aref parsed 3)))
			 'erc-server-311-hook))
      (add-to-list 'symlist
		   (cons (erc-once-with-server-event
			  312 `(string= ,nick (aref parsed 3)))
			 'erc-server-312-hook))
      (add-to-list 'symlist
		   (cons (erc-once-with-server-event
			  318 `(string= ,nick (aref parsed 3)))
			 'erc-server-318-hook))
      (add-to-list 'symlist
		   (cons (erc-once-with-server-event
			  319 `(string= ,nick (aref parsed 3)))
			 'erc-server-319-hook))
      (add-to-list 'symlist
		   (cons (erc-once-with-server-event
			  320 `(string= ,nick (aref parsed 3)))
			 'erc-server-320-hook))
      (add-to-list 'symlist
		   (cons (erc-once-with-server-event
			  330 `(string= ,nick (aref parsed 3)))
			 'erc-server-330-hook))
      (add-to-list 'symlist
		   (cons (erc-once-with-server-event
			  317
			  `(let ((idleseconds
				  (string-to-number (aref parsed 4))))
			     (erc-display-line
			      (erc-make-notice
			       (format "%s has been idle for %s."
				       (erc-string-no-properties ,nick)
				       (erc-seconds-to-string idleseconds)))
			      ,origbuf))
			  t)
			 'erc-server-317-hook))

      ;; Send the WHOIS command.
      (erc-cmd-WHOIS nick)

      ;; Remove the uninterned symbols from the server hooks that did not run.
      (run-at-time 20 nil `(lambda ()
			     (with-current-buffer ,(current-buffer)
			       (dolist (sym ',symlist)
				 (let ((hooksym (cdr sym))
				       (funcsym (car sym)))
				   (remove-hook hooksym funcsym t))))))))
  t)

(defun erc-cmd-DESCRIBE (line)
  "Pose some action to a certain user.
LINE has the format \"USER ACTION\"."
  (cond
   ((string-match
     "^\\s-*\\(\\S-+\\)\\s-\\(.*\\)$" line)
    (let ((dst (match-string 1 line))
	  (s (match-string 2 line)))
      (erc-log (format "cmd: DESCRIBE: [%s] %s" dst s))
      (erc-send-action dst s))
    t)
   (t nil)))

(defun erc-cmd-ME (line)
  "Send LINE as an action."
  (cond
   ((string-match "^\\s-\\(.*\\)$" line)
    (let ((s (match-string 1 line)))
      (erc-log (format "cmd: ME: %s" s))
      (erc-send-action (erc-default-target) s))
    t)
   (t nil)))
(put 'erc-cmd-ME 'do-not-parse-args t)

(defun erc-cmd-LASTLOG (line)
  "Show all lines in the current buffer matching the regexp LINE.

If a match spreads across multiple lines, all those lines are shown.

The lines are shown in a buffer named `*Occur*'.
It serves as a menu to find any of the occurrences in this buffer.
\\[describe-mode] in that buffer will explain how.

If LINE contains upper case characters (excluding those preceded by `\'),
the matching is case-sensitive."
  (occur line)
  t)
(put 'erc-cmd-LASTLOG 'do-not-parse-args t)

(defun erc-message (message-command line &optional force)
  "Send LINE to the server as a privmsg or a notice.
MESSAGE-COMMAND should be either \"PRIVMSG\" or \"NOTICE\".
If the target is \",\", the last person you've got a message from will
be used. If the target is \".\", the last person you've sent a message
to will be used."
  (cond
   ((string-match "^\\s-*\\(\\S-+\\)\\(\\s-*$\\|\\s-+.*\\)" line)
    (let ((tgt (match-string 1 line))
	  (s (match-string 2 line)))
      (erc-log (format "cmd: MSG(%s): [%s] %s" message-command tgt s))
      (if (string-match "^ " s) (setq s (substring s 1)))
      (cond
       ((string= tgt ",")
	(if (car last-peers)
	    (setq tgt (car last-peers))
	  (setq tgt nil)))
       ((string= tgt ".")
	(if (cdr last-peers)
	    (setq tgt (cdr last-peers))
	  (setq tgt nil))))
      (cond
       (tgt
	(setcdr last-peers tgt)
	(erc-send-command
	 (format "%s %s :%s" message-command tgt s)) force)
       (t
	(erc-display-message nil 'error (current-buffer) 'no-target))))
    t)
   (t nil)))

(defun erc-send-message (line &optional force)
  "Send LINE to the current channel or user and display it.

See also `erc-message' and `erc-display-line'."
  (erc-message "PRIVMSG" (concat (erc-default-target) " " line) force)
  (erc-display-line
   (concat (erc-format-my-nick) line)
     (current-buffer))
  ;; FIXME - treat multiline, run hooks, or remove me?
  t)

(defun erc-cmd-NOTICE (channel-or-user &rest message)
  "Send a notice to the channel or user given as the first word.
The rest is the message to send."
  (erc-message "NOTICE" (concat channel-or-user " "
				(mapconcat #'identity message " "))))

(defun erc-cmd-MSG (line)
  "Send a message to the channel or user given as the first word in LINE.

The rest of LINE is the message to send."
  (erc-message "PRIVMSG" line))

(defalias 'erc-cmd-M 'erc-cmd-MSG)
(put 'erc-cmd-MSG 'do-not-parse-args t)

(defun erc-cmd-SQUERY (line)
  "Send a Service Query to the service given as the first word in LINE.

The rest of LINE is the message to send."
  (erc-message "SQUERY" line))

(defun erc-cmd-NICK (nick)
  "Change current nickname to NICK."
  (erc-log (format "cmd: NICK: %s (bad-nick: %S)" nick bad-nick))
  (let ((nicklen (cdr (assoc "NICKLEN" (with-current-buffer (erc-server-buffer)
				    erc-server-parameters)))))
    (and nicklen (> (length nick) (string-to-number nicklen))
	 (erc-display-message nil 'notice 'active
	  (format (concat "WARNING: Nick length (%d) "
			  "exceeds max NICKLEN(%s) defined by server")
		  (length nick) nicklen))))
  (erc-send-command (format "NICK %s" nick))
  (cond (bad-nick
	 (erc-set-current-nick nick)
	 (erc-update-mode-line)
	 (setq bad-nick nil)))
  t)

(defun erc-cmd-PART (line)
  "When LINE is an empty string, leave the current channel.
Otherwise leave the channel indicated by LINE."
  (cond
   ((string-match "^\\s-*\\([&#+!]\\S-+\\)\\s-?\\(.*\\)$" line)
    (let* ((ch (match-string 1 line))
	   (msg (match-string 2 line))
	   (reason (funcall erc-part-reason (if (equal msg "") nil msg))))
      (erc-log (format "cmd: PART: %s: %s" ch reason))
      (erc-send-command
       (if (string= reason "")
	   (format "PART %s" ch)
	 (format "PART %s :%s" ch reason))))
    t)
   ((string-match "^\\s-*\\(.*\\)$" line)
    (let* ((ch (erc-default-target))
	   (msg (match-string 1 line))
	   (reason (funcall erc-part-reason (if (equal msg "") nil msg))))
      (if (and ch (erc-channel-p ch))
	  (progn
	    (erc-log (format "cmd: PART: %s: %s" ch reason))
	    (erc-send-command
	     (if (string= reason "")
		 (format "PART %s" ch)
	       (format "PART %s :%s" ch reason))))
	(erc-display-message nil 'error (current-buffer) 'no-target)))
    t)
   (t nil)))
(put 'erc-cmd-PART 'do-not-parse-args t)

(defalias 'erc-cmd-LEAVE 'erc-cmd-PART)

(defun erc-cmd-PING (recipient)
  "Ping RECIPIENT."
  (let ((time (format "%f" (erc-current-time))))
    (erc-log (format "cmd: PING: %s" time))
    (erc-cmd-CTCP recipient "PING" time)))

(defun erc-cmd-QUOTE (line)
  "Send LINE directly to the server.
All the text given as argument is sent to the sever as unmodified,
just as you provided it.  Use this command with care!"
  (cond
   ((string-match "^\\s-\\(.+\\)$" line)
    (erc-send-command (match-string 1 line)))
   (t nil)))
(put 'erc-cmd-QUOTE 'do-not-parse-args t)

(defun erc-cmd-QUERY (&optional user)
  "Open a query with USER.
The type of query window/frame/etc will depend on the value of
`erc-join-buffer'.  If USER is omitted, close the current query buffer if one
exists - except this is broken now ;-)"
  (interactive
   (list (read-from-minibuffer "Start a query with: " nil)))
  (let ((session-buffer (erc-server-buffer)))
    (if user
	(erc-query user session-buffer)
      ;; currently broken, evil hack to display help anyway
      ;(erc-delete-query))))
      (signal 'wrong-number-of-arguments ""))))
(defalias 'erc-cmd-Q 'erc-cmd-QUERY)

(defun erc-quit-reason-normal (&optional s)
  "Normal quit message.

If S is non-nil, it will be used as the quit reason."
  (or s
      (format "\C-bERC\C-b %s (IRC client for Emacs)"; - \C-b%s\C-b"
	      erc-version-string) ; erc-official-location)
  ))

(defun erc-quit-reason-zippy (&optional s)
  "Zippy quit message.

If S is non-nil, it will be used as the quit reason."
  (or s
      (erc-replace-regexp-in-string "\n" "" (yow))))

(defun erc-quit-reason-various (s)
  "Choose a quit reason based on S (a string)."
  (when (featurep 'xemacs) (require 'poe))
  (let ((res (car (assoc-default (or s "")
		   erc-quit-reason-various-alist 'string-match))))
    (cond
     ((functionp res) (funcall res))
     ((stringp res) res)
     ;; hopefully never reached
     (s))))

(defun erc-part-reason-normal (&optional s)
  "Normal part message.

If S is non-nil, it will be used as the quit reason."
  (or s
      (format "\C-bERC\C-b %s (IRC client for Emacs)"; - \C-b%s\C-b"
	      erc-version-string) ; erc-official-location)
  ))

(defun erc-part-reason-zippy (&optional s)
  "Zippy part message.

If S is non-nil, it will be used as the quit reason."
  (or s
      (erc-replace-regexp-in-string "\n" "" (yow))))

(defun erc-part-reason-various (s)
  "Choose a part reason based on S (a string)."
  (when (featurep 'xemacs) (require 'poe))
  (let ((res (car (assoc-default (or s "")
		   erc-part-reason-various-alist 'string-match))))
    (cond
     ((functionp res) (funcall res))
     ((stringp res) res)
     (s))))

(defun erc-cmd-QUIT (reason)
  "Disconnect from the current server.
If REASON is omitted, display a default quit message, otherwise display
the message given by REASON."
  (unless reason
    (setq reason ""))
  (cond
   ((string-match "^\\s-*\\(.*\\)$" reason)
    (let* ((s (match-string 1 reason))
	   (ob (current-buffer))
	   (buffer (erc-server-buffer))
	   (reason (funcall erc-quit-reason (if (equal s "") nil s))))
      (if (and buffer (bufferp buffer)) (set-buffer buffer))
      (erc-log (format "cmd: QUIT: %s" reason))
      (setq quitting t)
      (erc-send-command (format "QUIT :%s" reason))
      (set-buffer ob)
      (run-hook-with-args 'erc-quit-hook erc-process)
      (when erc-kill-queries-on-quit
	(erc-kill-query-buffers erc-process)))
    t)
   (t nil)))

(defun erc-kill-queries-on-quit (process)
  "Kill all query buffers associated with PROCESS.

This only has effect if the variable `erc-kill-queries-on-quit' is non-nil."
  (when erc-kill-queries-on-quit
    ;; FIXME:  This doesn't actually seem to be used anywhere.  Also, is
    ;; `erc-process' intentionally passed to `erc-kill-query-buffers'?  --
    ;; Lawrence 2004-01-08
    (erc-kill-query-buffers erc-process)))

(defalias 'erc-cmd-BYE 'erc-cmd-QUIT)
(defalias 'erc-cmd-EXIT 'erc-cmd-QUIT)
(defalias 'erc-cmd-SIGNOFF 'erc-cmd-QUIT)
(put 'erc-cmd-QUIT 'do-not-parse-args t)

(defun erc-cmd-GQUIT (reason)
  "Disconnect from all servers at once with the same quit REASON."
  (erc-with-all-buffers-of-server nil #'(lambda ()
					  (and (erc-server-buffer-p)
					       (erc-process-alive)))
				  (erc-cmd-QUIT reason)))

(defalias 'erc-cmd-GQ 'erc-cmd-GQUIT)
(put 'erc-cmd-GQUIT 'do-not-parse-args t)

(defun erc-cmd-SERVER (server)
  "Connect to SERVER, leaving existing connection intact."
  (erc-log (format "cmd: SERVER: %s" server))
  (erc-select server nil (erc-current-nick))
  t)

(defun erc-cmd-SV ()
  "Say the current ERC and Emacs version into channel."
  (erc-send-message (format "I'm using ERC %s with %s %s (%s%s%s)!"
			    erc-version-string
			    (if (featurep 'xemacs) "XEmacs" "GNU Emacs")
			    emacs-version
			    system-configuration
			    (concat
			     (cond ((featurep 'motif)
				    (concat ", " (substring
						  motif-version-string 4)))
				   ((featurep 'x-toolkit) ", X toolkit")
				   ((featurep 'gtk)
				    (concat ", GTK+ Version "
					    gtk-version-string))
				   ((featurep 'mac-carbon) ", Mac Carbon")
				   (t ""))
			     (if (and (boundp 'x-toolkit-scroll-bars)
				      (memq x-toolkit-scroll-bars '(xaw xaw3d)))
				 (format ", %s scroll bars"
					 (capitalize (symbol-name
						      x-toolkit-scroll-bars)))
			       "")
			     (if (featurep 'multi-tty) ", multi-tty" ""))
			    (format-time-string ", built %Y-%m-%d"
						erc-emacs-build-time)))
  t)

(defun erc-cmd-SM ()
  "Say the current ERC modes into channel."
  (erc-send-message (format "I'm using the following modules: %s!"
			    (erc-modes)))
  t)

(defun erc-cmd-SMV ()
  "Say the current ERC module versions into channel."
  (erc-send-message (format "I'm using the following module revisions: %s!"
			    (erc-version-modules)))
  t)

(defun erc-cmd-DEOP (&rest people)
  "Remove the operator setting from user(s) given in PEOPLE."
  (when (> (length people) 0)
    (erc-send-command (concat "MODE " (erc-default-target)
			      " -"
			      (make-string (length people) ?o)
			      " "
			      (reduce (lambda (&optional a b)
					(concat a " " b))
				      people)))
    t))

(defun erc-cmd-OP (&rest people)
  "Add the operator setting to users(s) given in PEOPLE."
  (when (> (length people) 0)
    (erc-send-command (concat "MODE " (erc-default-target)
			      " +"
			      (make-string (length people) ?o)
			      " "
			      (reduce (lambda (&optional a b)
					(concat a " " b))
				      people)))
    t))

(defun erc-cmd-TIME ()
  "Request the current time and date from the current server."
  (cond
   ((string-match "^\\s-*\\(.*\\)$" line)
    (let ((args (match-string 1 line)))
      (erc-log (format "cmd: TIME: %s" args))
      (erc-send-command (concat "TIME " args)))
    t)
   (t nil)))
(defalias 'erc-cmd-DATE 'erc-cmd-TIME)

(defun erc-cmd-TOPIC (topic)
  "Set or request the topic for a channel.
LINE has the format: \"#CHANNEL TOPIC\", \"#CHANNEL\", \"TOPIC\"
or the empty string.

If no #CHANNEL is given, the default channel is used.  If TOPIC is
given, the channel topic is modified, otherwise the current topic will
be displayed."
  (cond
   ;; /topic #channel TOPIC
   ((string-match "^\\s-*\\([&#+!]\\S-+\\)\\s-\\(.*\\)$" topic)
    (let ((ch (match-string 1 topic))
	  (topic (match-string 2 topic)))
      (erc-log (format "cmd: TOPIC [%s]: %s" ch topic))
      (erc-send-command (format "TOPIC %s :%s" ch topic)))
    t)
   ;; /topic #channel
   ((string-match "^\\s-*\\([&#+!]\\S-+\\)" topic)
    (let ((ch (match-string 1 topic)))
      (erc-send-command (format "TOPIC %s" ch))
      t))
   ;; /topic
   ((string-match "^\\s-*$" topic)
    (let ((ch (erc-default-target)))
      (erc-send-command (format "TOPIC %s" ch))
      t))
   ;; /topic TOPIC
   ((string-match "^\\s-*\\(.*\\)$" topic)
    (let ((ch (erc-default-target))
	  (topic (match-string 1 topic)))
      (if (and ch (erc-channel-p ch))
	  (progn
	    (erc-log (format "cmd: TOPIC [%s]: %s" ch topic))
	    (erc-send-command (format "TOPIC %s :%s" ch topic)))
	(erc-display-message nil 'error (current-buffer) 'no-target)))
    t)
   (t nil)))
(defalias 'erc-cmd-T 'erc-cmd-TOPIC)
(put 'erc-cmd-TOPIC 'do-not-parse-args t)

(defun erc-cmd-APPENDTOPIC (topic)
  "Append TOPIC to the current channel topic, separated by a space."
  (let ((oldtopic channel-topic))
    ;; display help when given no arguments
    (when (string-match "^\\s-*$" topic)
      (signal 'wrong-number-of-arguments nil))
    ;; strip trailing ^C
    (when (string-match "\\(.*\\)\C-c" oldtopic)
      (erc-cmd-TOPIC (concat (match-string 1 oldtopic) topic)))))
(defalias 'erc-cmd-AT 'erc-cmd-APPENDTOPIC)
(put 'erc-cmd-APPENDTOPIC 'do-not-parse-args t)

(defun erc-cmd-CLEARTOPIC (&optional channel)
  "Clear the topic for a CHANNEL.
If CHANNEL is not specified, clear the topic for the default channel."
  (interactive "sClear topic of channel (RET is current channel): ")
  (let ((chnl (or (and (erc-channel-p channel) channel) (erc-default-target))))
    (when chnl
      (erc-send-command (format "TOPIC %s :" chnl))
      t)))

;;; Banlists

(defvar erc-server-364-hook nil)
(defvar erc-server-365-hook nil)
(defvar erc-server-367-hook nil)
(defvar erc-server-368-hook nil)

(defvar channel-banlist nil
  "A list of bans seen for the current channel.

Each ban is an alist of the form:
  (WHOSET . MASK)

The property `received-from-server' indicates whether
or not the ban list has been requested from the server.")
(make-variable-buffer-local 'channel-banlist)
(put 'channel-banlist 'received-from-server nil)

(defun erc-cmd-BANLIST ()
  "Pretty-print the contents of `channel-banlist'.

The ban list if fetched from the server if necessary."
  (let ((chnl (erc-default-target)))

  (cond
     ((not (erc-channel-p chnl))
      (erc-display-line (erc-make-notice
			 (format "You're not on a channel\n" chnl))
			'active))

     ((not (get 'channel-banlist 'received-from-server))
      (let ((old-367-hook erc-server-367-hook))
	(setq erc-server-367-hook 'erc-banlist-store)
      ;; fetch the ban list then callback
      (with-current-buffer (erc-server-buffer)
	(erc-once-with-server-event
	 368
	 `(with-current-buffer ,chnl
	    (put 'channel-banlist 'received-from-server t)
	      (setq erc-server-367-hook ,old-367-hook)
	    (erc-cmd-BANLIST)
	    t))
	  (erc-send-command (format "MODE %s b" chnl)))))

   ((null channel-banlist)
      (erc-display-line (erc-make-notice
			 (format "No bans for channel: %s\n" chnl))
                      'active))

   (t
      (let* ((erc-fill-column (or (and (boundp 'erc-fill-column)
				       erc-fill-column)
				  (and (boundp 'fill-column)
				       fill-column)
				  (1- (window-width))))
	     (separator (make-string erc-fill-column (string-to-char "=")))
	     (fmt (concat
		   "%-" (number-to-string (/ erc-fill-column 2)) "s"
		   "%" (number-to-string (/ erc-fill-column 2)) "s")))

      (erc-display-line
       (erc-make-notice (format "Ban list for channel: %s\n"
                                (erc-default-target)))
       'active)

	(erc-display-line separator 'active)
	(erc-display-line (format fmt "Ban Mask" "Banned By") 'active)
	(erc-display-line separator 'active)

      (mapc
       (lambda (x)
         (erc-display-line
	    (format fmt
		    (truncate-string-to-width (cdr x) (/ erc-fill-column 2))
		    (if (car x)
			(truncate-string-to-width (car x) (/ erc-fill-column 2))
		      ""))
          'active))
       channel-banlist)

	(erc-display-line (erc-make-notice "End of Ban list")
			  'active)))))
  t)

(defalias 'erc-cmd-BL 'erc-cmd-BANLIST)

(defun erc-cmd-MASSUNBAN ()
  "Mass Unban.

Unban all currently banned users in the current channel."
  (let ((chnl (erc-default-target)))
    (cond

     ((not (erc-channel-p chnl))
      (erc-display-line
       (erc-make-notice (format "You're not on a channel\n" chnl))
       'active))

     ((not (get 'channel-banlist 'received-from-server))
      (let ((old-367-hook erc-server-367-hook))
	(setq erc-server-367-hook 'erc-banlist-store)
      ;; fetch the ban list then callback
      (with-current-buffer (erc-server-buffer)
	(erc-once-with-server-event
	 368
	 `(with-current-buffer ,chnl
	    (put 'channel-banlist 'received-from-server t)
	      (setq erc-server-367-hook ,old-367-hook)
	    (erc-cmd-MASSUNBAN)
	    t))
	  (erc-send-command (format "MODE %s b" chnl)))))

     (t (let ((bans (mapcar 'cdr channel-banlist)))
    (when bans
      ;; Glob the bans into groups of three, and carry out the unban.
      ;; eg. /mode #foo -bbb a*!*@* b*!*@* c*!*@*
      (mapc
       (lambda (x)
         (erc-send-command
          (format "MODE %s -%s %s" (erc-default-target)
                  (make-string (length x) (string-to-char "b"))
			(mapconcat 'identity x " "))))
       (erc-group-list bans 3))))
	t))))

(defalias 'erc-cmd-MUB 'erc-cmd-MASSUNBAN)

;;;; End of IRC commands

(defun erc-ensure-channel-name (channel)
  "Return CHANNEL if it is a valid channel name.
Eventually add a # in front of it, if that turns it into a valid channel name."
  (if (erc-channel-p channel)
      channel
    (concat "#" channel)))

(defun erc-make-property-list (fg bg bold ul inv)
  "Compiles the list of properties from flags.
Compiles the list of properties from the FG, BG colors and BOLD, UL (underline)
and INV (inverse) flags."
  (let (res)
    (if ul (setq res (cons 'erc-underline-face res)))
    (if bold (setq res (cons 'erc-bold-face res)))
    (if fg (setq res (cons (if inv
			       (erc-get-bg-color-face fg)
			     (erc-get-fg-color-face fg))
			   res)))
    (if bg (setq res (cons (if inv
			       (erc-get-fg-color-face bg)
			     (erc-get-bg-color-face bg))
			   res)))
    (if (and inv (not (or fg bg))) (setq res (cons 'erc-inverse-face res)))
    res))

(defun erc-prepend-properties (obj start end fg bg bold ul inv)
  "Prepend text properties found in the flags to OBJ.

Prepend text properties from START to END to OBJ (string or buffer) that
are determined by BG, FG colors (0-15) and BOLD, UL (underline) and INV
\(inverse) flags."
  (let ((props (erc-make-property-list fg bg bold ul inv)))
    (when props
      ;;(erc-log (format "prepending property list: %s" props))
      (if fg (erc-put-text-property start end 'erc-fg fg obj))
      (if bg (erc-put-text-property start end 'erc-bg bg obj))
      (if bold (erc-put-text-property start end 'erc-bold bold obj))
      (if ul (erc-put-text-property start end 'erc-ul ul obj))
      (if inv (erc-put-text-property start end 'erc-inv inv obj))
      (font-lock-prepend-text-property start end 'face props obj))))

(defun erc-strip-controls (string)
  "Strip IRC color control codes (et al.) from STRING.

Returns the string without any of the codes."
  ;; Control codes characters are: ^B, ^C, ^G, ^O, ^V, ^_.  ^C is followed by
  ;; one or two numbers separated by a comma.  The others appear alone.
  (when (stringp string)
    (while (string-match
	    (concat "\C-b\\|\C-_\\|\C-v\\|\C-g"
		    "\\|\C-o\\|\C-c[0-9]?[0-9]?"
		    "\\(,[0-9][0-9]?\\)?")
	    string)
      (setq string (replace-match "" nil nil string))))
  string)

(defun erc-decode-controls (line)
  "Convert properties of LINE into explicit control characters.

Convert all 'erc-* properties of LINE into explicit control
characters that can be sent back to the server.  Useful for
resending a colored line just by pasting it at the prompt, or for
grabbing color pop-ups that other people send."
  (let ((pos 0)
	(next-pos 0)
	fg bg bold ul inv
	;; previous values
	col bg0 bold0 ul0 inv0
	(res ""))
    (while pos
      (setq fg	 (get-text-property pos 'erc-fg line)
	    bg	 (get-text-property pos 'erc-bg line)
	    bold (get-text-property pos 'erc-bold line)
	    ul	 (get-text-property pos 'erc-ul line)
	    inv	 (get-text-property pos 'erc-inv line))
      (setq next-pos (next-property-change pos line))
      ;; put "end of color" only if not at the beginning of line,
      ;; and the color was on.	Also put it if the bg color disappeared
      (if (or (and col (not fg) (not bg) (/= pos 0))
	      (and (not bg) bg0))
	  (setq res (concat res "\C-c")))
      (when fg
	(setq res (concat res (char-to-string ?\^c) (format "%02d" fg)))
	(if bg (setq res (concat res (format ",%02d" bg)))))
      (if (or (and bold (not bold0))
	      (and (not bold) bold0))
	  (setq res (concat res (char-to-string ?\^b))))
      (if (or (and ul (not ul0))
	      (and (not ul) ul0))
	  (setq res (concat res (char-to-string ?\^_))))
      (if (or (and inv (not inv0))
	      (and (not inv) inv0))
	  (setq res (concat res (char-to-string ?\^v))))
      (setq col (or fg bg)
	    bg0 bg
	    bold0 bold
	    ul0 ul
	    inv0 inv)
      (setq res (concat res (substring line pos next-pos)))
      (setq pos next-pos))
    res))

;; FIXME: This really needs fixing up, for maintainability if nothing
;; else :P -- Lawrence 2004-01-11
(defun erc-interpret-controls (line)
  "Translate control characters in the LINE to faces and beeps."
  (cond
   ((and (not (null line)) erc-interpret-controls-p)
    (let ((res "")
	  (i 0)
	  (j 0)
	  (len (length line))
	  (parts nil);; for partitioning the string into segments
	  (bg nil) ;; number 0-15 (color)
	  (fg nil) ;; number 0-15 (color)
	  (bold nil)	;; boolean
	  (ul nil);; boolean (underline)
	  (inv nil);; boolean (inverse) - not sure how it'll work...
	  (hl-start 0);; position in the string where it starts
	  (hl-face nil);; the list of faces ready for
		       ;; "font-lock-prepend-text-property"
	  (prop-list nil));; list of arguments for `erc-prepend-properties'

      ;; First, decompose the string into parts without controls, and
      ;; the controls by themselves, and make a list of corresponding
      ;; substrings.  Then we'll combine them one chunk at a time, not
      ;; by one character, as was before.
      (setq j (string-match "\\(\C-b\\|\C-c\\|\C-_\\|\C-v\\|\C-g\\|\C-o\\)"
			    (substring line i)))
      (while j
	(let* ((rest (substring line j))
	       (s1 (substring line i j)) ;; "clean" part of the string
	       (ctrl (cond ((string-match "\\(\\(\C-c\\([0-9]\\([0-9]\\)?\\(,[0-9]\\([0-9]\\)?\\)?\\)?\\)\\|\C-b\\|\C-_\\|\C-v\\|\C-g\\|\C-o\\)" rest)
			    (setq i (+ j (match-end 1)))
			    (match-string 1 rest))
			   (t (erc-log-aux
			       (concat
				"ERROR: erc-interpret-controls: no match in "
				rest))
			      (setq j nil)
			      ""))))
	  (if (not (string= s1 "")) (setq parts (cons s1 parts)))
	  (setq parts (cons ctrl parts))
	  (setq j (string-match "\\(\C-b\\|\C-c\\|\C-_\\|\C-v\\|\C-g\\|\C-o\\)"
			    (substring line i)))
	  (if j (setq j (+ i j)))))

      (if (< i len) (setq parts (cons (substring line i) parts)))
      (setq parts (nreverse parts))
      (setq i 0 j 0)
      ;;(erc-log (format "computed list of parts: %S" (current-time)))

      ;;(erc-log (format "erc-interpret-controls: parts = %S" parts))

      ;; The parts of the string are ready:
      (while parts
	(let ((c (string-to-char (car parts))))
;;	(erc-log (format "intp-c: i %d c [%c] st %S fc %S"
;;			 i c hl-start hl-face))
	  (cond ((char-equal c ?\^g)	; beep
		 (ding t))

		;; bold
		((char-equal c ?\^b)
;;	       (erc-log	(format "intp-c: BOLD i %d j %d st %S" i j hl-start))
		 (setq prop-list (cons (list hl-start j fg bg bold ul inv)
				       prop-list))
		 ;;(erc-prepend-properties hl-start j res fg bg bold ul inv)
		 (setq bold (not bold)
		       hl-start j))

		;; underline
		((char-equal c ?\^_)
		 (setq prop-list (cons (list hl-start j fg bg bold ul inv)
				       prop-list))
		 ;;(erc-prepend-properties hl-start j res fg bg bold ul inv)
		 (setq ul (not ul)
		       hl-start j))

		;; inverse
		((char-equal c ?\^v)
		 (setq prop-list (cons (list hl-start j fg bg bold ul inv)
				       prop-list))
		 ;;(erc-prepend-properties hl-start j res fg bg bold ul inv)
		 (setq inv (not inv)
		       hl-start j))
		;; mIRC color -- S.B.
		((char-equal c ?\^c)
		 (setq prop-list (cons (list hl-start j fg bg bold ul inv)
				       prop-list))
		 ;;(erc-prepend-properties hl-start j res fg bg bold ul inv)
		 ;; clear the fg color specs by default (keep the bg)
		 (setq hl-start j
		       fg nil
;;		       bg nil
		       )
		 (let ((ccode (substring (car parts) 1)))
		   (if (and erc-interpret-mirc-color
			    (string-match
			     "^\\([0-9]\\([0-9]\\)?\\(,[0-9]\\([0-9]\\)?\\)?\\)"
			     ccode))
		       (let ((cspec (match-string 1 ccode)))
			 (cond ((string-match "^\\([0-9]+\\),\\([0-9]+\\)"
					      cspec)
				(setq fg (string-to-number
					  (match-string 1 cspec))
				      bg (string-to-number
					  (match-string 2 cspec))))
			       (t (setq fg (string-to-number cspec)))))
		     ;; otherwise, reset the colors
		     (setq bg nil))))
		;; End of text property -- S.B.
		((char-equal c ?\^o);; - end of all properties (??)
		 (setq prop-list (cons (list hl-start j fg bg bold ul inv)
				       prop-list))
		 ;;(erc-prepend-properties hl-start j res fg bg bold ul inv)
		 (setq bg nil
		       fg nil
		       bold nil
		       ul nil
		       inv nil
		       hl-start j))
		;; FIXME: deal with these -- Lawrence 2004-01-11
;		((char-equal c ?\^q)
;		 ;; I don't know how to interpret this.	 Just skip it for now.
;		 nil)
;		((char-equal c ?\^r)
;		 ;; I don't know how to interpret this.	 Just skip it for now.
;		 nil)
		(t
		 (setq res (concat res (car parts)))
		 (setq j (+ j (length (car parts)))))))
	(setq parts (cdr parts)))
      (setq prop-list (cons (list hl-start j fg bg bold ul inv)
			    prop-list))
      ;;(erc-prepend-properties hl-start j res fg bg bold ul inv)
      ;; now prepend all the properties
      (while prop-list
	(apply 'erc-prepend-properties (cons res (car prop-list)))
	(setq prop-list (cdr prop-list)))
      ;;(erc-log (format "done: %S" (current-time)))
      ;;(erc-log (concat "erc-interpret-controls: res = " res))
      res))
   ((not (null line)) (erc-strip-controls line))
   (t nil)))

(defun erc-merge-controls (line)
  "Translate and merge control characters and faces in LINE.
Translate control characters in the LINE to faces and beeps, then
converts all properties back into control symbols (to mix both user
entered and existing colors), and removes all the text properties from
the string.

This allows copying a colored line and sending it preserving the
colors, and also to edit it without decoding the colors first."
  ;; FIXME: why was the interpretation removed?  Do we never call
  ;; this with uninterpreted controls? -- Lawrence 2004-01-11
  (erc-decode-controls line)
;  (erc-decode-controls
;   (erc-interpret-controls line))
  )

(defun erc-grab-region (start end)
  "Copy the region between START and END in a recreatable format.

Converts all the IRC text properties in each line of the region
into control codes and writes them to a separate buffer.  The
resulting text may be used directly as a script to generate this
text again."
  (interactive "r")
  (setq erc-active-buffer (current-buffer))
  (save-excursion
    (let* ((cb (current-buffer))
	   (buf (generate-new-buffer erc-grab-buffer-name))
	   (region (buffer-substring start end))
	   (lines (erc-split-multiline-safe region)))
      (set-buffer buf)
      (dolist (line lines)
	(insert (concat (erc-decode-controls line) "\n")))
      (set-buffer cb)
      (switch-to-buffer-other-window buf))))

(defun erc-display-prompt (&optional buffer pos prompt face)
  "Display PROMPT in BUFFER at position POS.
Display an ERC prompt in BUFFER.

If PROMPT is nil, one is constructed with the function `erc-prompt'.
If BUFFER is nil, the `current-buffer' is used.
If POS is nil, PROMPT will be displayed at `point'.
If FACE is non-nil, it will be used to propertize the prompt.  If it is nil,
`erc-prompt-face' will be used."
  (let* ((prompt (or prompt (erc-prompt)))
	 (l (length prompt))
	 (ob (current-buffer)))
    (when (> l 0)
      ;; We cannot use save-excursion because we move point, therefore
      ;; we resort to the ol' ob trick to restore this.
      (when (and buffer (bufferp buffer))
	(set-buffer buffer))

      ;; now save excursion again to store where point and mark are
      ;; in the current buffer
      (save-excursion
	(setq pos (or pos (point)))
	(goto-char pos)
	;; Do not extend the text properties when typing at the end
	;; of the prompt, but stuff typed in front of the prompt
	;; shall remain part of the prompt.
	(setq prompt (erc-propertize prompt
				     'start-open t ; XEmacs
				     'rear-nonsticky t ; Emacs
				     'erc-prompt t
				     'front-sticky t
				     'read-only t))
	(erc-put-text-property 0 (1- (length prompt))
			       'face (or face 'erc-prompt-face)
			       prompt)
	;; Removed -- Lawrence 2004-01-08
	;; (when erc-prompt-interactive-input
	;;   (setq prompt (erc-propertize prompt
	;; 			       'local-map
	;; 			       (erc-interactive-input-map))))
	(insert prompt))

      ;; Now we are back at the old position.  If the prompt was
      ;; inserted here or before us, advance point by the length of
      ;; the prompt.
      (when (or (not pos) (<= (point) pos))
	(forward-char l))
      ;; Clear the undo buffer now, so the user can undo his stuff,
      ;; but not the stuff we did. Sneaky!
      (setq buffer-undo-list nil)
      (set-buffer ob))))

;; interactive operations

;; Removed -- Lawrence 2004-01-08.  If someone wants this, I'll have a look at
;; fixing it.

;; (defun erc-interactive-input-map ()
;;   "Create an input map that called `erc-input-message' when an ASCII
;; character (one between 32 and 127) is entered. This is installed as an
;; input map when `erc-prompt-interactive-input' is set to non-nil.
;; Obsolete as of 1.348."
;;   (let ((lm (make-sparse-keymap)))
;;     (loop for c from 32 to 127
;; 	  do (define-key lm (vector c) 'erc-input-message))
;;     lm))

(defun erc-input-message ()
  "Read input from the minibuffer."
  (interactive)
  (let ((minibuffer-allow-text-properties t)
	(read-map minibuffer-local-map))
    (insert (read-from-minibuffer "Message: "
				  (string last-command-char) read-map))
    (erc-send-current-line)))

(defvar erc-action-history-list ()
  "History list for interactive action input.")

(defun erc-input-action ()
  "Interactively input a user action and send it to IRC."
  (interactive "")
  (setq erc-active-buffer (current-buffer))
  (let ((action (read-from-minibuffer
		 "Action: " nil nil nil erc-action-history-list)))
    (if (not (string-match "^\\s-*$" action))
	(erc-send-action (erc-default-target) action))))

(defun erc-join-channel (channel &optional key)
  "Join CHANNEL.

If `point' is at the beginning of a channel name, use that as default."
  (interactive
   (list
    (let ((chnl (if (looking-at "\\([&#+!][^ ]+\\)") (match-string 1) ""))
	  (table (when (and (boundp 'erc-process) (processp erc-process))
		   (set-buffer (process-buffer erc-process))
		   channel-list)))
      (completing-read "Join channel: " table nil nil nil nil chnl))
    (when erc-prompt-for-channel-key
      (read-from-minibuffer "Channel key (RET for none): " nil))))
  (erc-cmd-JOIN channel (when (>= (length key) 1) key)))

(defun erc-part-from-channel (reason)
  "Part from the current channel and prompt for a REASON."
  (interactive
   (list
    (if (and (boundp 'reason) (stringp reason) (not (string= reason "")))
	reason
      (read-from-minibuffer (concat "Leave " (erc-default-target)
				    ", Reason? ")
			    (cons "No reason" 0)))))
  (erc-cmd-PART (concat (erc-default-target)" " reason)))

(defun erc-set-topic (topic)
  "Prompt for a TOPIC for the current channel."
  (interactive
   (list
    (read-from-minibuffer
     (concat "Set topic of " (erc-default-target) ": ")
			  (when (boundp 'channel-topic)
       (cons (apply 'concat (butlast (split-string channel-topic "\C-c")))
			     0)))))
  (let ((topic-list (split-string topic "\C-c"))) ; strip off the topic setter
    (erc-cmd-TOPIC (concat (erc-default-target) " " (car topic-list)))))

(defun erc-set-channel-limit (&optional limit)
  "Set a LIMIT for the current channel.  Remove limit if nil.
Prompt for one if called interactively."
  (interactive (list (read-from-minibuffer
		      (format "Limit for %s (RET to remove limit): "
			      (erc-default-target)))))
  (let ((tgt (erc-default-target)))
    (if (and limit (>= (length limit) 1))
	(erc-send-command (format "MODE %s +l %s" tgt limit))
      (erc-send-command (format "MODE %s -l" tgt)))))

(defun erc-set-channel-key (&optional key)
  "Set a KEY for the current channel.  Remove key if nil.
Prompt for one if called interactively."
  (interactive (list (read-from-minibuffer
		      (format "Key for %s (RET to remove key): "
			      (erc-default-target)))))
  (let ((tgt (erc-default-target)))
    (if (and key (>= (length key) 1))
	(erc-send-command (format "MODE %s +k %s" tgt key))
      (erc-send-command (format "MODE %s -k" tgt)))))

(defun erc-quit-server (reason)
  "Disconnect from current server after prompting for REASON.
`erc-quit-reason' works with this just like with `erc-cmd-QUIT'."
  (interactive (list (read-from-minibuffer
		      (format "Reason for quitting %s: "
			      erc-announced-server-name))))
  (erc-cmd-QUIT reason))

;; Movement of point

(defun erc-bol ()
  "Move `point' to the beginning of the current line.

This places `point' just after the prompt, or at the beginning of the line."
  (interactive)
  (forward-line 0)
  (when (get-text-property (point) 'erc-prompt)
    (goto-char (or (next-single-property-change (point) 'erc-prompt)
		   (point-at-eol))))
  (point))

(defun erc-kill-input ()
  "Kill current input line using `erc-bol' followed by `kill-line'."
  (interactive)
  (when (and (erc-bol)
	     (/= (point) (point-max))) ;; Prevent a (ding) and an error when
				       ;; there's nothing to kill
    (if (boundp 'erc-input-ring-index)
	(setq erc-input-ring-index nil))
    (kill-line)))

(defun erc-complete-word ()
  "Complete the word before point.

This function uses `erc-complete-functions'."
  (interactive)
  (let ((funs erc-complete-functions))
    (while (and funs
		(not (funcall (car funs))))
      (setq funs (cdr funs)))
    (if (not funs)
	(beep))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;;			  IRC SERVER INPUT HANDLING
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;; New Input parsing

; Stolen from ZenIRC. I just wanna test this code, so here is
; experiment area.

(defcustom erc-default-server-hook '(erc-debug-missing-hooks
				     erc-default-server-handler)
  "*Default for server messages which aren't covered by `erc-server-hooks'."
  :group 'erc-server-hooks
  :type 'hook)

(defun erc-default-server-handler (proc parsed)
  "Default server handler.

Displays PROC and PARSED appropriately using `erc-display-message'."
  (erc-display-message
   parsed 'notice proc
   (mapconcat
    'identity
    (let (res)
      (mapc #'(lambda (x)
		(if (stringp x)
		    (setq res (append res (list x)))))
	    parsed)
      res)
    " ")))

(defvar erc-server-vectors
  '(["msgtype" "sender" "to" "arg1" "arg2" "arg3" "..."])
  "List of received server messages which ERC does not specifically handle.
See `erc-debug-missing-hooks'.")
;(make-variable-buffer-local 'erc-server-vectors)

(defun erc-debug-missing-hooks (proc parsed)
  "Add PARSED server message ERC does not yet handle to `erc-server-vectors'.
These vectors can be helpful when adding new server message handlers to ERC.
See `erc-default-server-hook'."
  (nconc erc-server-vectors (list parsed))
  nil)

(defun erc-call-hooks (proc msg)
  "Call all hooks relevant to MSG with PROC.

Hooks are found by calling `erc-event-to-hook-name' on the first element of
MSG."
  (let ((hook (intern-soft (erc-event-to-hook-name (aref msg 0)))))
    (if (and hook (symbol-value hook))
	(progn
	  (run-hook-with-args-until-success hook proc msg)
	  nil)
      (run-hook-with-args-until-success 'erc-default-server-hook proc msg)
      nil)
    (with-current-buffer (erc-server-buffer)
      (run-hook-with-args 'erc-timer-hook (erc-current-time)))))

(defvar erc-duplicates (make-hash-table :test 'equal)
  "Internal variable used to track duplicate messages.")
(make-variable-buffer-local 'erc-duplicates)

(defvar erc-duplicate-timeout 60
  ;; FIXME: is this docstring correct? -- Lawrence 2004-01-08
  "The time in seconds between duplicate messages.

If two identical messages arrive within this value of one another, the second
isn't displayed.")

(defvar erc-prevent-duplicates '("301")
  "*Either nil or a list of strings.
Each string is a IRC message type, like PRIVMSG or NOTICE.
All Message types in that list of subjected to duplicate prevention.")

(defun erc-parse-line-from-server (proc string)
  "Parse and act upon a complete line from a IRC server.
PROC is the process (connection) from which STRING was received.
PROCs `process-buffer' is `current-buffer' when this function is called."
  (unless (string= string "") ;; Ignore empty strings
    (save-match-data
      (let ((posn (if (eq (aref string 0) ?:)
		      (string-match " " string)
		    0))
	    (msg (make-vector 32 nil))
	    (n 2))
	(aset msg 1 (if (eq posn 0)
			erc-session-server
		      (substring string 1 posn)))

	(aset msg 0 (let* ((bposn (string-match "[^ ]" string posn))
			   (eposn (string-match " " string bposn)))
		      (setq posn (and eposn
				      (string-match "[^ ]" string eposn)))
		      (substring string bposn eposn)))

	(while (and posn
		    (not (eq (aref string posn) ?:)))
	  (aset msg n (let* ((bposn posn)
			     (eposn (string-match " " string bposn)))
			(setq posn (and eposn
					(string-match "[^ ]" string eposn)))
			(substring string bposn eposn)))
	  (setq n (1+ n)))
	(if posn
	    (aset msg n (substring string (1+ posn))))
	(if (member (aref msg 0) erc-prevent-duplicates)
	    (progn
	      (if (< (or
		      (gethash (mapconcat 'identity msg "") erc-duplicates)
		      0)
		     (- (erc-current-time) erc-duplicate-timeout))
		  (erc-call-hooks proc msg))
	      (puthash (mapconcat 'identity msg "")
		       (erc-current-time)
		       erc-duplicates))
	  (erc-call-hooks proc msg))))))

(defun erc-query (target server)
  "Open a query buffer on TARGET, using SERVER.
To change how this query window is displayed, use `let' to bind
`erc-join-buffer' before calling this."
  (unless (and server
	       (buffer-live-p server)
	       (set-buffer server))
    (error "Couldn't switch to server buffer"))
  (let ((buf (erc erc-session-server
		  erc-session-port
		  (erc-current-nick)
		  erc-session-user-full-name
		  nil
		  nil
		  (list target)
		  target
		  erc-process)))
    (unless buf
      (error "Couldn't open query window"))
    (erc-update-mode-line)
    buf))

(defcustom erc-auto-query nil
  "If non-nil, create a query buffer each time you receive a private message.

If the buffer doesn't already exist it is created.  This can be
set to a symbol, to control how the new query window should
appear.  See the documentation for `erc-join-buffer' for
available choices."
  :group 'erc
  :type '(choice (const nil)
		 (const buffer)
		 (const window)
		 (const window-noselect)
		 (const bury)
		 (const frame)))

(defcustom erc-query-on-unjoined-chan-privmsg t
  "If non-nil create query buffer on receiving any PRIVMSG at all.
This includes PRIVMSGs directed to channels.  If you are using an IRC
bouncer, such as dircproxy, to keep a log of channels when you are
disconnected, you should set this option to t."
  :group 'erc
  :type 'boolean)

(defcustom erc-format-query-as-channel-p t
  "If non-nil, format text from others in a query buffer like in a channel,
otherwise format like a private message."
  :group 'erc
  :type 'boolean)

(defcustom erc-minibuffer-notice nil
  "If non-nil, print ERC notices for the user in the minibuffer.
Only happens when the session buffer isn't visible."
  :group 'erc
  :type 'boolean)

(defcustom erc-minibuffer-ignored nil
  "If non-nil, print a message in the minibuffer if we ignored something."
  :group 'erc
  :type 'boolean)

(defun erc-wash-quit-reason (reason nick login host)
  "Remove duplicate text from quit REASON.
Specifically in relation to NICK (user@host) information.  Returns REASON
unmodified if nothing can be removed.
E.g. \"Read error to Nick [user@some.host]: 110\" would be shortened to
\"Read error: 110\". The same applies for \"Ping Timeout\"."
  (setq nick (regexp-quote nick)
	login (regexp-quote login)
	host (regexp-quote host))
  (or (when (string-match (concat "^\\(Read error\\) to "
				  nick "\\[" host "\\]: "
				  "\\(.+\\)$") reason)
	(concat (match-string 1 reason) ": " (match-string 2 reason)))
      (when (string-match (concat "^\\(Ping timeout\\) for "
				  nick "\\[" host "\\]$") reason)
	(match-string 1 reason))
      reason))

(defun erc-nickname-in-use (nick reason)
  "If NICK is unavailable, tell the user the REASON.

See also `erc-display-error-notice'."
  (if (or erc-manual-set-nick-on-bad-nick-p
	  ;; how many default-nicks are left + one more try...
	  (eq erc-nick-change-attempt-count
	      (if (consp erc-nick)
		  (+ (length erc-nick) 1)
		1)))
      (erc-display-error-notice
       nil
       (format "Nickname %s is %s, try another." nick reason))
    (setq erc-nick-change-attempt-count (+ erc-nick-change-attempt-count 1))
    (let ((newnick (nth 1 erc-default-nicks))
	  (nicklen (cdr (assoc "NICKLEN"
			       (with-current-buffer (erc-server-buffer)
				 erc-server-parameters)))))
      (setq bad-nick t)
      ;; try to use a different nick
      (if erc-default-nicks
	  (setq erc-default-nicks (cdr erc-default-nicks)))
      (if (not newnick)
	  (setq newnick (concat (truncate-string-to-width
				 nick
				 (if (and erc-connected nicklen)
				     (- (string-to-int nicklen) 1)
				   ;; rfc2812 max nick length = 9
				   ;; we must assume this is the server's setting
				   ;; if we haven't established a connection yet
				   8))
				erc-nick-uniquifier)))
      (erc-cmd-NICK newnick)
      (erc-display-error-notice
       nil
       (format "Nickname %s is %s, trying %s"
	       nick reason newnick)))))

;;; Server messages


(defgroup erc-server-hooks nil
  "Server event callbacks.
Every server event - like numeric replies - has it's own hook.
Those hooks are all called using `run-hook-with-args-until-success'.
They receive as first argument the process object from where the event
originated from,
and as second argument the event parsed as a vector."
  :group 'erc-hooks)

(defmacro erc-server-hook-list (list)
  "Generate hook definitions for the IRC protocol from LIST."
  `(progn
     ,@(mapcar
	#'(lambda (pair)
	    `(defcustom
	       ,(erc-event-to-hook (car pair))
	       (quote ,(cdr pair))
	       ,(concat (if (numberp (car pair))
			    (concat "Numeric server reply #"
				    (number-to-string (car pair)) "\n")
			  (concat (symbol-name (car pair))
				  " server reply.\n"))
			"Arguments are (PROC PARSED).
PROC is the process object of the server that message came from and
PARSED is a vector containing the server message fields as strings.")
	       :group (quote erc-server-hooks)
	       :type (quote hook)))
	list)))

(erc-server-hook-list
 ((ERROR erc-server-ERROR)
  (INVITE erc-server-INVITE)
  (JOIN erc-server-JOIN)
  (KICK erc-server-KICK)
  (MODE erc-server-MODE erc-banlist-update)
  (NICK erc-server-NICK)
  (NOTICE erc-server-PRIVMSG-or-NOTICE)
  (PART erc-server-PART)
  (PING erc-server-PING)
  (PRIVMSG erc-auto-query erc-server-PRIVMSG-or-NOTICE)
  (QUIT erc-server-QUIT)
  (TOPIC erc-server-TOPIC)
  (WALLOPS erc-server-WALLOPS)
  (001 erc-server-001)
  (002 erc-server-MOTD)
  (003 erc-server-MOTD)
  (004 erc-server-004)
  (005 erc-server-005)
  (221 erc-server-221)
  (250 erc-display-server-message)
  (251 erc-display-server-message)
  (252 erc-server-252)
  (253 erc-server-253)
  (254 erc-server-254)
  (255 erc-display-server-message)
  (256 erc-display-server-message)
  (257 erc-display-server-message)
  (258 erc-display-server-message)
  (259 erc-display-server-message)
  (265 erc-display-server-message)
  (266 erc-display-server-message)
  (301 erc-server-301)
  (303 erc-server-303)
  (305 erc-server-305)
  (306 erc-server-306)
  (311 erc-server-311-or-314)
  (312 erc-server-312)
  (313 erc-server-313)
  (314 erc-server-311-or-314)
  (315 ignore)
  (317 erc-server-317)
  (318 ignore)
  (319 erc-server-319)
  (320 erc-server-320)
  (321 erc-server-321)
  (322 erc-server-322)
  (324 erc-server-324)
  (329 erc-server-329)
  (330 erc-server-330)
  (331 erc-server-331)
  (332 erc-server-332)
  (333 erc-server-333)
  (341 erc-server-341)
  (352 erc-server-352)
  (353 erc-server-353)
  (366 erc-server-366)
  (369 ignore)
  (371 erc-server-MOTD)
  (372 erc-server-MOTD)
  (374 erc-server-MOTD)
  (375 erc-server-MOTD)
  (376 erc-server-MOTD erc-connection-established)
  (377 erc-display-server-message)
  (378 erc-display-server-message)
  (379 erc-server-379)
  (401 erc-server-401)
  (403 erc-server-403)
  (405 erc-server-405)
  (406 erc-server-406)
  (412 erc-server-412)
  (421 erc-server-421)
  (422 erc-server-MOTD erc-connection-established)
  (432 erc-server-432)
  (433 erc-server-433)
  (437 erc-server-437)
  (442 erc-server-442)
  (461 erc-server-461)
  (474 erc-server-474)
  (475 erc-server-475)
  (477 erc-server-477)
  (481 erc-server-481)
  (482 erc-server-482)
  (501 erc-server-501)
))

(defun erc-display-server-message (proc parsed)
  "Display the message sent by the server as a notice."
  (erc-display-message
   parsed 'notice 'active (aref parsed 3)))

(defun erc-server-ERROR (proc parsed)
  "Handle an ERROR command from the server."
  (erc-display-message
   parsed 'error nil
   (format "==> ERROR from %s: %s\n" (aref parsed 1) (aref parsed 2)))
  (erc-update-mode-line))

(defun erc-server-INVITE (proc parsed)
  "Handle invitation messages.
PARSED has the form [\"INVITE\" \"who\" \"whom\" \"where-to\"]."
  (let* ((target (aref parsed 2))
	 (chnl (aref parsed 3))
	 (sndr (erc-parse-user (aref parsed 1)))
	 (nick (nth 0 sndr))
	 (login (nth 1 sndr))
	 (host (nth 2 sndr)))
    (setq invitation chnl)
    (when (string= target (erc-current-nick))
      (erc-display-message
       parsed 'notice 'active
       'INVITE ?n nick ?u login ?h host ?c chnl))))

(defun erc-server-JOIN (proc parsed)
  "Handle join messages.
PARSED has the form [\"JOIN\" \"who\" \"channel\"]."
  (let* ((chnl (aref parsed 2))
	 (sndr (erc-parse-user (aref parsed 1)))
	 (nick (nth 0 sndr))
	 (login (nth 1 sndr))
	 (host (nth 2 sndr))
	 (buffer nil))
    ;; strip the stupid combined JOIN facility (IRC 2.9)
    (if (string-match "^\\(.*\\)?\^g.*$" chnl)
	(setq chnl (match-string 1 chnl)))
    (let* ((ob (current-buffer))
	   (info-buf nil)
	   (str (cond
		 ;; If I have joined a channel
		 ((erc-current-nick-p nick)
		  (setq buffer (erc erc-session-server erc-session-port
				    nick erc-session-user-full-name
				    nil nil
				    erc-default-recipients chnl erc-process))
		  (when buffer
		    (set-buffer buffer)
		    (erc-add-default-channel chnl)
		    (erc-send-command (format "MODE %s" chnl))
		    ;; display the channel info buffer
		    (setq info-buf (erc-find-channel-info-buffer chnl))
		    (cond ((eq erc-join-info-buffer 'frame)
			   (switch-to-buffer-other-frame info-buf))
			  ((eq erc-join-info-buffer 'window)
			   (switch-to-buffer-other-window info-buf))
			  ((eq erc-join-info-buffer 'split)
			   (split-window-vertically)
			   (switch-to-buffer-other-window info-buf)))
		    ;; and return to the channel buffer...
		    ;; boy, how to put the focus back into the channel
		    ;; window now???
		    (set-buffer buffer))
		  (erc-with-buffer
		   (chnl proc)
		   (erc-channel-begin-receiving-names))
		  (erc-update-mode-line)
		  (run-hook-with-args 'erc-join-hook
				      chnl (erc-get-buffer chnl proc))
		  (erc-make-notice
		   (erc-format-message 'JOIN-you ?c chnl)))
		 (t
		  (setq buffer (erc-get-buffer chnl proc))
		  (erc-make-notice
		   (erc-format-message
		    'JOIN ?n nick ?u login ?h host ?c chnl))))))
      (when buffer (set-buffer buffer))
      (erc-update-channel-member chnl nick nick t nil nil host login)
      (erc-update-channel-info-buffer chnl)
      ;; on join, we want to stay in the new channel buffer
      ;;(set-buffer ob)
      (unless (member "JOIN" erc-hide-list)
	(erc-display-message
	 parsed nil buffer
	 str)))))

(defun erc-server-KICK (proc parsed)
  "Handle kick messages received from the server."
  (let* ((ch (aref parsed 2))
	 (tgt (aref parsed 3))
	 (reason (erc-trim-string (aref parsed 4)))
	 (sndr (erc-parse-user (aref parsed 1)))
	 (nick (nth 0 sndr))
	 (login (nth 1 sndr))
	 (host (nth 2 sndr))
	 (buffer (erc-get-buffer ch proc)))
    (erc-remove-channel-member buffer tgt)
    (erc-update-channel-info-buffer ch)
    (cond
     ((string= tgt (erc-current-nick))
      (erc-display-message
       parsed 'notice buffer
       'KICK-you ?n nick ?u login ?h host ?c ch ?r reason)
      (run-hook-with-args 'erc-kick-hook buffer)
      (erc-with-buffer
       (buffer)
       (erc-remove-channel-users))
      (erc-delete-default-channel ch buffer)
      (erc-update-mode-line buffer))
     ((string= nick (erc-current-nick))
      (erc-display-message
       parsed 'notice buffer
       'KICK-by-you ?k tgt ?c ch ?r reason))
     (t (unless (member "KICK" erc-hide-list)
	  (erc-display-message
	   parsed 'notice buffer
	   'KICK ?k tgt ?n nick ?u login ?h host ?c ch ?r reason))))))

(defvar erc-server-KILL-hook nil)

(defun erc-server-MODE (proc parsed)
  "Handle server mode changes."
  (let* ((tgt (aref parsed 2))
	 (mode (mapconcat 'identity (delete* nil (subseq parsed 3)) " "))
	 (sndr (erc-parse-user (aref parsed 1)))
	 (nick (nth 0 sndr))
	 (login (nth 1 sndr))
	 (host (nth 2 sndr)))
    (erc-log (format "MODE: %s -> %s: %s" nick tgt mode))
    ;; dirty hack
    (let ((ob (current-buffer))
	  (buf (cond ((erc-channel-p tgt)
		      (erc-get-buffer tgt proc))
		     ((string= tgt (erc-current-nick)) nil)
		     (erc-active-buffer erc-active-buffer)
		     (t (erc-get-buffer tgt)))))
      (when buf (set-buffer buf))
      (erc-update-modes tgt mode nick host login)
      (set-buffer ob)
      (unless (member "MODE" erc-hide-list)
	(if (or (string= login "") (string= host ""))
	    (erc-display-message
	     parsed 'notice buf
	     'MODE-nick ?n nick ?t tgt ?m mode)
	  (erc-display-message
	   parsed 'notice buf
	   'MODE ?n nick ?u login ?h host ?t tgt ?m mode))))))

(defun erc-server-NICK (proc parsed)
  "Handle nick change messages."
  (let* ((nn (aref parsed 2))
	 (sndr (erc-parse-user (aref parsed 1)))
	 (nick (nth 0 sndr))
	 (login (nth 1 sndr))
	 (host (nth 2 sndr))
	 (bufs (erc-buffer-list-with-nick nick proc)))
    (erc-log (format "NICK: %s -> %s" nick nn))
    ;; if we had a query with this user, make sure future messages will be
    ;; sent to the correct nick. also add to bufs, since the user will want
    ;; to see the nick change in the query, and if it's a newly begun query,
    ;; erc-channel-users won't contain it
    (erc-buffer-filter
     (lambda ()
       (when (equal (erc-default-target) nick)
	 (setq erc-default-recipients
	       (cons nn (cdr erc-default-recipients)))
	 (rename-buffer nn)
	 (erc-update-mode-line)
	 (add-to-list 'bufs (current-buffer)))))
    (erc-update-user-nick nick nn host nil nil login)
    (erc-update-channel-info-buffers bufs)
    (cond
     ((string= nick (erc-current-nick))
      (add-to-list 'bufs (erc-server-buffer))
      (erc-set-current-nick nn)
      (erc-update-mode-line)
      (setq erc-nick-change-attempt-count 0)
      (setq erc-default-nicks (if (consp erc-nick) erc-nick (list erc-nick)))
      (erc-display-message
       parsed 'notice bufs
       'NICK-you ?n nick ?N nn)
      (run-hook-with-args 'erc-nick-changed-functions nn nick))
     (t
      (erc-handle-user-status-change 'nick (list nick login host) (list nn))
      (unless (member "NICK" erc-hide-list)
	(erc-display-message
	 parsed 'notice bufs 'NICK ?n nick ?u login ?h host ?N nn))))))

(defun erc-server-PART (proc parsed)
  "Handle part messages."
  (let* ((chnl (aref parsed 2))
	 (reason (erc-trim-string (aref parsed 3)))
	 (sndr (erc-parse-user (aref parsed 1)))
	 (nick (nth 0 sndr))
	 (login (nth 1 sndr))
	 (host (nth 2 sndr))
	 (buffer (erc-get-buffer chnl proc)))
    (erc-remove-channel-member buffer nick)
    (erc-update-channel-info-buffer chnl)
    (unless (member "PART" erc-hide-list)
      (erc-display-message
       parsed 'notice buffer
       'PART ?n nick ?u login ?h host ?c chnl ?r (or reason "")))
    (when (string= nick (erc-current-nick))
      (run-hook-with-args 'erc-part-hook buffer)
      (erc-with-buffer
       (buffer)
       (erc-remove-channel-users))
      (erc-delete-default-channel chnl buffer)
      (erc-update-mode-line buffer)
      (when erc-kill-buffer-on-part
	(kill-buffer buffer)))))

(defun erc-server-PING (proc parsed)
  "Handle ping messages."
  (let ((pinger (aref parsed 2)))
    (erc-log (format "PING: %s" pinger))
    ;; ping response to the server MUST be forced, or you can lose big
    (erc-send-command (format "PONG :%s" pinger) t)
    (when erc-verbose-server-ping
      (erc-display-message
       parsed 'error proc
       'PING ?s (erc-time-diff last-ping-time (erc-current-time))))
    (setq last-ping-time (erc-current-time))))

(defvar erc-server-PONG-hook '(erc-server-PONG))

(defvar erc-lag nil
  "Calculated server lag time in seconds.
This variable is only set in a server buffer.")
(make-variable-buffer-local 'erc-lag)


(defun erc-server-PONG (proc parsed)
  "Handle pong messages."
  (let ((time (string-to-number (aref parsed 3))))
    (when (> time 0)
      (setq erc-lag (erc-time-diff time (erc-current-time)))
      (when erc-verbose-server-ping
	(erc-display-message
	 parsed 'notice proc
	 (format "PONG from %s (%d second%s)" (aref parsed 2) erc-lag
		 (if (/= erc-lag 1) "s" ""))))
      (erc-update-mode-line)))
  t)

(defcustom erc-send-ping-interval 90
  "*Interval of sending pings to the server, in seconds.
If this is set to nil, pinging the server is disabled."
  :group 'erc
  :type '(choice (const nil) (integer :tag "Seconds")))

(defvar erc-ping-handler nil
  "This variable holds the periodic ping timer.")
(make-variable-buffer-local 'erc-ping-handler)

(defun erc-setup-periodical-server-ping (&rest ignore)
  "Set up a timer to periodically ping the current server."
  (and erc-ping-handler (cancel-timer erc-ping-handler))
  (when erc-send-ping-interval
    (setq erc-ping-handler
	  (run-with-timer
	   4 erc-send-ping-interval
	   (lambda (buf)
	     (when (buffer-live-p buf)
	       (with-current-buffer buf
		 (erc-send-command
		  (format "PING %.0f"
			  (erc-current-time))))))
	   (current-buffer)))))

(defun erc-auto-query (proc parsed)
  ;; FIXME: This needs more documentation, unless it's not a user function --
  ;; Lawrence 2004-01-08
  "Put this on `erc-server-PRIVMSG-hook'."
  (when erc-auto-query
    (let* ((nick (car (erc-parse-user (aref parsed 1))))
	   (target (aref parsed 2))
	   (msg (aref parsed 3))
	   (query  (if (not erc-query-on-unjoined-chan-privmsg)
		       nick
		     (if (erc-current-nick-p target)
			 nick
		       target))))
      (and (not (erc-ignored-user-p (aref parsed 1)))
	   (or erc-query-on-unjoined-chan-privmsg
	       (string= target (erc-current-nick)))
	   (not (erc-get-buffer query proc))
	   (not (erc-is-message-ctcp-and-not-action-p msg))
	   (let ((erc-join-buffer erc-auto-query))
	     (erc-cmd-QUERY query))
	   nil))))

(defun erc-is-message-ctcp-p (message)
  "Check if MESSAGE is a CTCP message or not."
  (string-match "^\C-a\\([^\C-a]*\\)\C-a?$" message))

(defun erc-is-message-ctcp-and-not-action-p (message)
  "Check if MESSAGE is a CTCP message or not."
  (and (erc-is-message-ctcp-p message)
       (not (string-match "^\C-a\\ACTION.*\C-a$" message))))

(defun erc-format-privmessage (nick msg privp msgp)
  "Format a PRIVMSG in an insertible fashion."
  (let* ((mark-s (if msgp (if privp "*" "<") "-"))
	 (mark-e (if msgp (if privp "*" ">") "-"))
	 (str	 (format "%s%s%s %s" mark-s nick mark-e msg))
	 (nick-face (if privp 'erc-nick-msg-face 'erc-nick-default-face))
	 (msg-face (if privp 'erc-direct-msg-face 'erc-default-face)))
    ;; add text properties to text before the nick, the nick and after the nick
    (erc-put-text-property 0 (length mark-s) 'face msg-face str)
    (erc-put-text-property (length mark-s) (+ (length mark-s) (length nick))
			   'face nick-face str)
    (erc-put-text-property (+ (length mark-s) (length nick)) (length str)
			   'face msg-face str)
    str))

(defcustom erc-format-nick-function 'erc-format-nick
  "*Function to format a nickname for message display."
  :group 'erc
  :type 'function)

(defun erc-format-nick (&optional user channel-data)
  "Standard nickname formatting function.  Only returns the value
of NICK."
  (if user
      (erc-server-user-nickname user)))

(defun erc-format-@nick (&optional user channel-data)
  "Format a nickname such that @ or + are prefix for the NICK
if OP or VOICE are t respectively."
  (if user
      (let (op voice)
	(if channel-data
	    (setq op (erc-channel-user-op channel-data)
		  voice (erc-channel-user-voice channel-data)))
    (concat (if voice "+" "")
	    (if op "@" "")
		(erc-server-user-nickname user)))))

(defun erc-format-my-nick ()
  "Return the beginning of this user's message, correctly propertized"
  (if erc-show-my-nick
      (let ((open "<")
	    (close "> ")
	    (nick (erc-current-nick)))
	(concat
	 (erc-propertize open 'face 'erc-default-face)
	 (erc-propertize nick 'face 'erc-nick-default-face)
	 (erc-propertize close 'face 'erc-default-face)))
    (let ((prefix "> "))
      (erc-propertize prefix 'face 'erc-default-face))))

(defun erc-echo-notice-in-default-buffer (s parsed buffer sender)
  "Echos a private notice in the default buffer, namely the
target buffer specified by BUFFER, or there is no target buffer,
the server buffer.  This function is designed to be added to
either `erc-echo-notice-hook' or `erc-echo-notice-always-hook',
and always returns t."
  (erc-display-message parsed nil buffer s)
  t)

(defun erc-echo-notice-in-target-buffer (s parsed buffer sender)
  "Echos a private notice in BUFFER, if BUFFER is non-nil.  This
function is designed to be added to either `erc-echo-notice-hook'
or `erc-echo-notice-always-hook', and returns non-nil iff BUFFER
is non-nil."
  (if buffer
      (progn (erc-display-message parsed nil buffer s) t)
    nil))

(defun erc-echo-notice-in-minibuffer (s parsed buffer sender)
  "Echos a private notice in the minibuffer.  This function is
designed to be added to either `erc-echo-notice-hook' or
`erc-echo-notice-always-hook', and always returns t."
  (message "%s" (concat "NOTICE: " s))
  t)

(defun erc-echo-notice-in-server-buffer (s parsed buffer sender)
  "Echos a private notice in the server buffer.  This function is
designed to be added to either `erc-echo-notice-hook' or
`erc-echo-notice-always-hook', and always returns t."
  (erc-display-message parsed nil nil s)
  t)

(defun erc-echo-notice-in-active-non-server-buffer (s parsed buffer sender)
  "Echos a private notice in the active buffer if the active
buffer is not the server buffer.  This function is designed to be
added to either `erc-echo-notice-hook' or
`erc-echo-notice-always-hook', and returns non-nil iff the active
buffer is not the server buffer."
  (if (not (eq (erc-server-buffer) erc-active-buffer))
      (progn (erc-display-message parsed nil 'active s) t)
    nil))

(defun erc-echo-notice-in-active-buffer (s parsed buffer sender)
  "Echos a private notice in the active buffer.  This function is
designed to be added to either `erc-echo-notice-hook' or
`erc-echo-notice-always-hook', and always returns t."
  (erc-display-message parsed nil 'active s)
  t)

(defun erc-echo-notice-in-user-buffers (s parsed buffer sender)
  "Echos a private notice in all of the buffers for which SENDER
is a member.  This function is designed to be added to either
`erc-echo-notice-hook' or `erc-echo-notice-always-hook', and
returns non-nil iff there is at least one buffer for which the
sender is a member.

See also: `erc-echo-notice-in-first-user-buffer',
`erc-buffer-list-with-nick'"
  (let ((buffers (erc-buffer-list-with-nick sender erc-process)))
    (if buffers
	(progn (erc-display-message parsed nil buffers s) t)
      nil)))

(defun erc-echo-notice-in-user-and-target-buffers (s parsed buffer sender)
  "Echos a private notice in BUFFER and in all of the buffers for
which SENDER is a member.  This function is designed to be added
to either `erc-echo-notice-hook' or
`erc-echo-notice-always-hook', and returns non-nil iff there is
at least one buffer for which the sender is a member or the
default target.

See also: `erc-echo-notice-in-user-buffers',
`erc-buffer-list-with-nick'"
  (let ((buffers (erc-buffer-list-with-nick sender erc-process)))
    (add-to-list 'buffers buffer)
    (if buffers
	(progn (erc-display-message parsed nil buffers s) t)
      nil)))

(defun erc-echo-notice-in-first-user-buffer (s parsed buffer sender)
  "Echos a private notice in one of the buffers for which SENDER
is a member.  This function is designed to be added to either
`erc-echo-notice-hook' or `erc-echo-notice-always-hook', and
returns non-nil iff there is at least one buffer for which the
sender is a member.

See also: `erc-echo-notice-in-user-buffers',
`erc-buffer-list-with-nick'"
  (let ((buffers (erc-buffer-list-with-nick sender erc-process)))
    (if buffers
	(progn (erc-display-message parsed nil (car buffers) s) t)
      nil)))

(defun erc-server-PRIVMSG-or-NOTICE (proc parsed)
  "Format and display a PRIVMSG or NOTICE."
  (let ((sender-spec (aref parsed 1))
	(cmd (aref parsed 0))
	(tgt (aref parsed 2))
	(msg (aref parsed 3)))
    (if (or (erc-ignored-user-p sender-spec)
	    (erc-ignored-reply-p msg tgt proc))
	(when erc-minibuffer-ignored
	  (message "Ignored %s from %s to %s" cmd sender-spec tgt))
      (let* ((sndr (erc-parse-user sender-spec))
	     (nick (nth 0 sndr))
	     (login (nth 1 sndr))
	     (host (nth 2 sndr))
	     (msgp (string= cmd "PRIVMSG"))
	     (noticep (string= cmd "NOTICE"))
	     ;; S.B. downcase *both* tgt and current nick
	     (privp (erc-current-nick-p tgt))
	     s buffer
	     fnick
	     (msg (erc-decode-string-from-target msg
						 (if privp sender-spec tgt))))
	(setf (aref parsed 3) msg)
	(setq buffer (erc-get-buffer (if privp nick tgt) proc))
	(when buffer
	  (with-current-buffer buffer
	    ;; update the chat partner info.  Add to the list if private
	    ;; message.	 We will accumulate private identities indefinitely
	    ;; at this point.
	    (if (erc-update-channel-member (if privp nick tgt) nick nick
					   privp nil nil host login nil nil t)
		(erc-update-channel-info-buffer (if privp nick tgt)))
	    (let ((cdata (erc-get-channel-user nick)))
	      (setq fnick (funcall erc-format-nick-function (car cdata) (cdr cdata))))))
	(cond
	 ((erc-is-message-ctcp-p msg)
	  (setq s (if msgp
		      (erc-process-ctcp-query proc parsed nick login host)
		    (erc-process-ctcp-reply proc parsed nick login host
					    (match-string 1 msg)))))
	 (t
	  (setcar last-peers nick)
	  (setq s (erc-format-privmessage (or fnick nick) msg
					  ;; If buffer is a query buffer,
					  ;; format the nick as for a channel.
					  (and (not
						(and
						 buffer
						 (erc-query-buffer-p buffer)
						 erc-format-query-as-channel-p))
					      privp)
					  msgp))))
	(when s
	  (if (and noticep privp)
	      (progn
		(run-hook-with-args 'erc-echo-notice-always-hook
				    s parsed buffer nick)
		(run-hook-with-args-until-success
		 'erc-echo-notice-hook s parsed buffer nick))
            (erc-display-message parsed nil buffer s)))))))

(defun erc-server-QUIT (proc parsed)
  "Quit handler."
  (let* ((reason (aref parsed 2))
	 (sndr (erc-parse-user (aref parsed 1)))
	 (nick (nth 0 sndr))
	 (login (nth 1 sndr))
	 (host (nth 2 sndr))
	 (bufs (erc-buffer-list-with-nick nick proc)))
    (erc-remove-user nick)
    (erc-update-channel-info-buffers bufs)
    (setq reason (erc-wash-quit-reason reason nick login host))
    (unless (member "QUIT" erc-hide-list)
      (erc-display-message
       parsed 'notice bufs
       'QUIT ?n nick ?u login ?h host ?r reason))))

(defun erc-server-TOPIC (proc parsed)
  "Topic handler."
  (let* ((sender-spec (aref parsed 1))
	 (ch (aref parsed 2))
	 (topic (erc-trim-string (erc-decode-string-from-target
				  (aref parsed 3) ch)))
	 (sndr (erc-parse-user sender-spec))
	 (nick (nth 0 sndr))
	 (login (nth 1 sndr))
	 (host (nth 2 sndr))
	 (time (format-time-string "%T %m/%d/%y" (current-time))))
    (erc-update-channel-member ch nick nick nil nil nil host login)
    (erc-update-channel-topic ch (format "%s\C-c (%s, %s)" topic nick time))
    (erc-update-channel-info-buffer ch)
    (unless (member "TOPIC" erc-hide-list)
      (erc-display-message
       parsed 'notice (erc-get-buffer ch proc)
       'TOPIC ?n nick ?u login ?h host ?c ch ?T topic))))

(defun erc-server-WALLOPS (proc parsed)
  "Display WALLOPS messages."
  (let* ((sender-spec (aref parsed 1))
	 (sndr (erc-parse-user sender-spec))
	 (nick (nth 0 sndr))
	 (login (nth 1 sndr))
	 (host (nth 2 sndr))
	 (message (aref parsed 2)))
    (erc-display-message
     parsed 'notice nil
     'WALLOPS ?n nick ?m message)))

(defvar erc-server-version nil)
(make-variable-buffer-local 'erc-server-version)

(defun erc-server-001 (proc parsed)
  "Set current-nick to reflect the server's settings and display the
welcome message."
  (let ((nick (aref parsed 2)))
    (erc-set-current-nick nick))
  (erc-update-mode-line) ; needed here?
  (setq erc-nick-change-attempt-count 0)
  (setq erc-default-nicks (if (consp erc-nick) erc-nick (list erc-nick)))
  (erc-display-message
   parsed 'notice 'active (aref parsed 3)))

(defun erc-server-004 (proc parsed)
  "Handle the server's information and display the message."
  (let ((server-name (aref parsed 3))
	(server-version (aref parsed 4)))
    (setq erc-server-version server-version)
    (setq erc-announced-server-name server-name)
    (erc-update-mode-line-buffer (process-buffer proc))
    (erc-display-message
     parsed 'notice proc
     's004 ?s server-name ?v server-version
     ?U (aref parsed 5) ?C (aref parsed 6))))

(defun erc-server-005 (proc parsed)
  "Set the variable `erc-server-parameters' and display the received message.

According to RFC 2812, suggests alternate servers on the network.
Many servers, however, use this code to show which parameters they have set,
for example, the network identifier, maximum allowed topic length, whether
certain commands are accepted and more.	 See documentation for
`erc-server-parameters' for more information on the parameters sent.

A server may send more than one 005 message."
  ;; make use of the information being provided here
  (let (line)
    (dotimes (n (length parsed) line)
      (let ((section (aref parsed n)))
	(when (and (> n 2) section)
	  ;; put the line back together
	  (setq line (concat line (unless (eq n 3) " ") section))
	  ;; fill erc-server-parameters
	  (if (string-match "^\\([A-Z]+\\)\=\\(.*\\)$\\|^\\([A-Z]+\\)$"
			    section)
	      (add-to-list 'erc-server-parameters
			   `(,(or (match-string 1 section)
				  (match-string 3 section))
			     .
			     ,(match-string 2 section)))))))
    (erc-display-message parsed 'notice proc line)))

(defvar erc-server-200-hook nil)
(defvar erc-server-201-hook nil)
(defvar erc-server-202-hook nil)
(defvar erc-server-203-hook nil)
(defvar erc-server-204-hook nil)
(defvar erc-server-205-hook nil)
(defvar erc-server-206-hook nil)
(defvar erc-server-208-hook nil)
(defvar erc-server-209-hook nil)
(defvar erc-server-211-hook nil)
(defvar erc-server-212-hook nil)
(defvar erc-server-213-hook nil)
(defvar erc-server-214-hook nil)
(defvar erc-server-215-hook nil)
(defvar erc-server-216-hook nil)
(defvar erc-server-217-hook nil)
(defvar erc-server-218-hook nil)
(defvar erc-server-219-hook nil)

(defun erc-server-221 (proc parsed)
  (let ((modes (mapconcat 'identity (delete* nil (subseq parsed 3)) " "))
	(nick (aref parsed 2)))
    (erc-set-modes nick modes)
    (erc-display-message
     parsed 'notice 'active
     's221 ?n nick ?m modes)))

(defvar erc-server-241-hook nil)
(defvar erc-server-242-hook nil)
(defvar erc-server-243-hook nil)
(defvar erc-server-244-hook nil)
(defvar erc-server-249-hook nil)

(defun erc-server-252 (proc parsed)
  "Display the number of IRC operators."
  (let ((number (aref parsed 3)))
    (erc-display-message
     parsed 'notice 'active
     's252 ?i number)))

(defun erc-server-253 (proc parsed)
  "Display the number of unknown connections."
  (let ((number (aref parsed 3)))
    (erc-display-message
     parsed 'notice 'active
     's253 ?i number)))

(defun erc-server-254 (proc parsed)
  "Display the number of channels formed."
  (let ((number (aref parsed 3)))
    (erc-display-message
     parsed 'notice 'active
     's254 ?i number)))

(defvar erc-server-261-hook nil)
(defvar erc-server-262-hook nil)

(defun erc-server-301 (proc parsed)
  "AWAY notice."
  (let ((nick (aref parsed 3))
	(msg (aref parsed 4)))
    (erc-display-message
     parsed 'notice 'active
      's301 ?n nick ?r msg)))

(defvar erc-server-302-hook nil)

(defun erc-server-303 (proc parsed)
  (let ((str (aref parsed 3)))
    (erc-display-message
     parsed 'notice proc
     's303 ?n str)))


;;; AWAY message

(defun erc-server-305 (proc parsed)
  "You are no longer marked as being away."
  (erc-process-away proc nil)
  (erc-display-message
   parsed 'notice 'active
   's305 ?m (aref parsed 3)))

(defun erc-server-306 (proc parsed)
  "You have been marked as being away."
  (erc-process-away proc t)
  (erc-display-message
   parsed 'notice 'active
   's306 ?m (aref parsed 3)))

(defun erc-server-311-or-314 (proc parsed)
  "WHOIS/WAS notices."
  (let ((nick (aref parsed 3))
	(user (aref parsed 4))
	(host (aref parsed 5))
	(fname (aref parsed 7)))
    (erc-update-user-nick nick nick host nil fname user)
    (erc-update-channel-info-buffers)
    (erc-display-message
     parsed 'notice 'active
      (format "%s %s %s (%s@%s)"
	      nick
	      (if (string= (aref parsed 0) "311")
		  "is"
		"was")
	      fname user host))))

(defun erc-server-312 (proc parsed)
  (let ((nick (aref parsed 3))
	(saddr (aref parsed 4))
	(server (aref parsed 5)))
    (erc-display-message
     parsed 'notice 'active
      (format "%s is/was on server %s (%s)"
	      nick saddr server))))

(defun erc-server-313 (proc parsed)
  ;; Irc Oper in whois
  (let ((nick (aref parsed 3))
	(ircopermsg (aref parsed 4)))
    (erc-display-message
     parsed 'notice 'active
     (format "%s %s" nick ircopermsg))))
	
(defun erc-server-317 (proc parsed)
  ;; IDLE notice
  (let* ((nick (aref parsed 3))
	 (nsec (string-to-number (aref parsed 4)))
	 (time-str (aref parsed 5))
	 (time (when time-str
		 (format-time-string "%T %m/%d/%y"
				     (erc-string-to-emacs-time time-str)))))
    (erc-update-user-nick nick nick nil nil nil
				      (if time
					  (format "on since %s" time) nil))
    (erc-update-channel-info-buffers)
    (erc-display-message
     parsed 'notice 'active
      (format "%s is idle for %s%s"
	      nick
	      (erc-sec-to-time nsec)
	      (if time (format ", on since %s" time) "")))))

(defun erc-server-319 (proc parsed)
  (let ((nick (aref parsed 3))
	(chnls (aref parsed 4)))
    (erc-display-message
     parsed 'notice 'active
      (format "%s is on channel(s): %s" nick chnls))))

(defun erc-server-320 (proc parsed)
  (let ((nick (aref parsed 3))
	(text (aref parsed 4)))
    (erc-display-message
     parsed 'notice 'active
      (format "%s %s" nick text))))

(defun erc-server-321 (proc parsed)
  ;; LIST header
  (setq channel-list nil)
  (erc-display-message
   parsed 'notice 'active
   (concat (aref parsed 3) "  " (aref parsed 4))))

(defun erc-server-322 (proc parsed)
  ;; LIST notice
  (let ((chnl (aref parsed 3))
	(nv (aref parsed 4))
	(topic (aref parsed 5)))
    (add-to-list 'channel-list (list chnl))
    (erc-update-channel-topic chnl topic)
    (erc-update-channel-info-buffer chnl)
    (erc-display-message
     parsed 'notice 'active
      (if (string= topic "")
	  (format "%s [%s]" chnl nv)
	(format "%s [%s]: %s" chnl nv topic)))))

(defvar erc-server-323-hook nil)

(defun erc-server-324 (proc parsed)
  ;; channel or nick modes
  (let ((chnl (aref parsed 3))
	(modes (mapconcat 'identity (delete* nil (subseq parsed 4)) " ")))
    (erc-set-modes chnl modes)
    (erc-display-message
     parsed 'notice (erc-get-buffer chnl proc)
     's324 ?c chnl ?m modes)))

(defun erc-server-329 (proc parsed)
  ;; Channel living time
  (let ((chnl (aref parsed 3))
	(time (erc-string-to-emacs-time (aref parsed 4))))
    (erc-display-message
     parsed 'notice (erc-get-buffer chnl proc)
     's329 ?c chnl ?t (format-time-string "%A %Y-%m-%d %X" time))))

(defun erc-server-330 (proc parsed)
  ;; Authed as in whois (quakenet)
  (let ((nick (aref parsed 3))
	(authaccount (aref parsed 4))
	(authmsg (aref parsed 5)))
    (erc-display-message
     parsed 'notice 'active
      (format "%s %s %s"
	      nick authmsg authaccount))))
    
(defun erc-server-331 (proc parsed)
  ;; Channel topic
  (let ((chnl (aref parsed 3))
	(topic (aref parsed 4)))
    (erc-display-message
     parsed 'notice (erc-get-buffer chnl proc)
     's331 ?c chnl)))

(defun erc-server-332 (proc parsed)
  ;; TOPIC notice
  (let ((chnl (aref parsed 3))
	(topic (aref parsed 4)))
    (erc-update-channel-topic chnl topic)
    (erc-update-channel-info-buffer chnl)
    (erc-display-message
     parsed 'notice (erc-get-buffer chnl proc)
     's332 ?c chnl ?T topic)))

(defun erc-server-333 (proc parsed)
  ;; who set the topic and when
  (let ((chnl (aref parsed 3))
	(nick (aref parsed 4))
	(time (format-time-string "%T %m/%d/%y"
				  (erc-string-to-emacs-time (aref parsed 5)))))
    (erc-update-channel-topic chnl (format "\C-c (%s, %s)" nick time)
			      'append)
    (erc-update-channel-info-buffer chnl)
    (erc-display-message
     parsed 'notice (erc-get-buffer chnl proc)
     's333 ?c chnl ?n nick ?t time)))

(defun erc-server-341 (proc parsed)
  "Let user know when an INVITE attempt has been sent successfully."
  (let ((nick (aref parsed 3))
	(chnl (aref parsed 4)))
    (erc-display-message
     parsed 'notice (erc-get-buffer chnl proc)
     's341 ?n nick ?c chnl)))

(defvar erc-server-342-hook nil)
(defvar erc-server-351-hook nil)

(defun erc-server-352 (proc parsed)
  ;; WHO notice
  (let ((chnl (aref parsed 3))
	(user (aref parsed 4))
	(host (aref parsed 5))
	(server (aref parsed 6))
	(nick (aref parsed 7))
	(flag (aref parsed 8))
	(name (aref parsed 9))	    ; This is not quite right, but ok
	hopcount)
    (when (string-match "\\(^[0-9]+ \\)\\(.*\\)$" name)
      (setq hopcount (match-string 1 name))
      (setq name (match-string 2 name)))
    (erc-update-channel-member chnl nick nick nil nil nil host user name)
    (erc-display-message
     parsed 'notice 'active
      (format "%-11s %-10s %-4s %s@%s (%s)" chnl nick flag user host name))))

(defun erc-server-353 (proc parsed)
  (let ((chnl (aref parsed 4))
	(users (aref parsed 5)))
    (erc-with-buffer (chnl proc)
		     (erc-channel-receive-names users))
    (erc-update-channel-info-buffer chnl)
    (erc-display-message
     parsed 'notice (or (erc-get-buffer chnl proc)
			'active)
     's353 ?c chnl ?u users)))

(defun erc-server-366 (proc parsed)
  (let ((chnl (aref parsed 3)))
    (erc-with-buffer (chnl proc)
		     (erc-channel-end-receiving-names))))


;;; Ban manipulation

(defun erc-banlist-store (proc parsed)
  "Record ban entries for a channel."
  (let ((whoset (aref parsed 5))
	 (channel (aref parsed 3))
	 (mask (aref parsed 4)))

    ;; Determine to which buffer the message corresponds
    (let ((buffer (erc-get-buffer channel proc)))
      (with-current-buffer buffer
	(unless (member (cons whoset mask) channel-banlist)
	  (setq channel-banlist (cons (cons whoset mask) channel-banlist))))))
  nil)

(defun erc-banlist-finished (proc parsed)
  "Record that we have received the banlist."
  (let* ((channel (aref parsed 3))
	 (buffer (erc-get-buffer channel proc)))
    (with-current-buffer buffer
      (put 'channel-banlist 'received-from-server t)))
  t)					; suppress the 'end of banlist' message

(defun erc-banlist-update (proc parsed)
  "Check MODE commands for bans and update the banlist appropriately."
  (let* ((tgt (aref parsed 2))
         (mode (mapconcat 'identity (delete* nil (subseq parsed 3)) " "))
	 (whoset (aref parsed 1))
         (buffer (erc-get-buffer tgt proc)))
    (when buffer
      (with-current-buffer buffer
	(cond ((not (get 'channel-banlist 'received-from-server)) nil)
	      ((string-match "^\\([+-]\\)b" mode)
          ;; This is a ban
          (cond
           ((string-match "^-" mode)
            ;; Remove the unbanned masks from the ban list
            (setq channel-banlist
		       (delete-if
                   (lambda (y)
                     (member (upcase (cdr y))
                             (mapcar #'upcase (cdr (split-string mode)))))
                   channel-banlist)))
           ((string-match "^+" mode)
            ;; Add the banned mask(s) to the ban list
            (mapc
             (lambda (mask)
		    (unless (member (cons whoset mask) channel-banlist)
		      (setq channel-banlist
			    (cons (cons whoset mask) channel-banlist))))
		  (cdr (split-string mode))))))))))
  nil)

;; used for the banlist cmds
(defun erc-group-list (list n)
  "Group LIST into sublists of length N."
  (cond ((null list) nil)
        ((null (nthcdr n list)) (list list))
	(t (cons (subseq list 0 n) (erc-group-list (nthcdr n list) n)))))

;; Is this faster?  Does it matter?  Was I bored? :P -- Lawrence 2004-01-08
;; (defun erc-group-list-1 (list n)
;;   (let ((i 0)
;;         result)
;;     (while list
;;       (setq result
;;             (nconc result (let (res)
;;                             (while (and (car list)
;;                                         (< i n))
;;                               (setq res (nconc res (list (pop list))))
;;                               (setq i (1+ i)))
;;                             (list res))))
;;       (setq i 0))
;;     result))


;;; MOTD numreplies

(defun erc-handle-login ()
  "Handle the logging in process of connection."
  (unless erc-logged-in
    (setq erc-logged-in t)
    (message "Logging in as \'%s\'... done" (erc-current-nick))
    ;; execute a startup script
    (let ((f (erc-select-startup-file)))
      (when f
	(erc-load-script f)))))

(defun erc-server-MOTD (proc parsed)
  "Display the server's message of the day."
  (erc-handle-login)
  (erc-display-message
   parsed 'notice 'active (aref parsed 3)))

(defun erc-connection-established (proc parsed)
  "Run just after connection.

Set user modes and run `erc-after-connect hook'."
  (unless erc-connected ; only once per session
    (let ((server erc-announced-server-name)
	  (nick (aref parsed 2)))
      (setq erc-connected t)
      (erc-set-initial-user-mode nick)
      (erc-setup-periodical-server-ping)
      (run-hook-with-args 'erc-after-connect server nick))))

(defun erc-set-initial-user-mode (nick)
  "If `erc-user-mode' is non-nil for NICK, set the user modes."
  (when erc-user-mode
    (let ((mode (if (functionp erc-user-mode)
		    (funcall erc-user-mode)
		  erc-user-mode)))
      (when (stringp mode)
	(erc-log (format "changing mode for %s to %s" nick mode))
	(erc-send-command (format "MODE %s %s" nick mode))))))

(defun erc-server-379 (proc parsed)
  "Forwarding to another channel."
  (let ((from (aref parsed 3))
	(to (aref parsed 4)))
    (erc-display-message
     parsed 'notice 'active
      's379 ?c from ?f to)))

(defvar erc-server-381-hook nil)
(defvar erc-server-382-hook nil)
(defvar erc-server-391-hook nil)
(defvar erc-server-392-hook nil)
(defvar erc-server-393-hook nil)
(defvar erc-server-394-hook nil)
(defvar erc-server-395-hook nil)

(defun erc-display-error-notice (parsed string)
  "Display STRING as an error notice.

See also `erc-display-message'."
  (erc-display-message
   parsed '(notice error) 'active string))

(defun erc-server-401 (proc parsed)
  "No such nick/channel."
  (let ((nick (aref parsed 3)))
    (when erc-whowas-on-nosuchnick
      (erc-log (format "cmd: WHOWAS: %s" nick))
      (erc-send-command (format "WHOWAS %s 1" nick)))
    (erc-display-message
     parsed '(notice error) 'active
     's401 ?n nick)))

(defvar erc-server-402-hook nil)

(defun erc-server-403 (proc parsed)
  "That channel doesn't exist."
  (let ((channel (aref parsed 3)))
    (erc-display-message
     parsed '(notice error) 'active
     's403 ?c channel)))

(defvar erc-server-404-hook nil)

(defun erc-server-405 (proc parsed)
  "You can't join that many channels."
  (let ((channel (aref parsed 3)))
    (erc-display-message
     parsed '(notice error) 'active
     's405 ?c channel)))

(defun erc-server-406 (proc parsed)
  "There was no such nickname."
  (let ((nick (aref parsed 3)))
    (erc-display-message
     parsed '(notice error) nil
     's406 ?n nick)))

(defvar erc-server-407-hook nil)
(defvar erc-server-409-hook nil)
(defvar erc-server-411-hook nil)

(defun erc-server-412 (proc parsed)
  "Handle \"no text to send\" errors."
  (erc-display-message
   parsed '(notice error) 'active
   's412))

(defvar erc-server-413-hook nil)
(defvar erc-server-414-hook nil)
(defvar erc-server-415-hook nil)

(defun erc-server-421 (proc parsed)
  "Handle \"Unknown command\" errors."
  (let ((command (aref parsed 3)))
    (erc-display-message
     parsed '(notice error) 'active
     's421 ?c command)))

(defvar erc-server-422-hook nil)
(defvar erc-server-423-hook nil)
(defvar erc-server-424-hook nil)
(defvar erc-server-431-hook nil)

(defun erc-server-432 (proc parsed)
  "Handle \"Erroneous Nickname\" errors."
  (let ((nick (aref parsed 3)))
    (erc-display-message
     parsed '(notice error) 'active
     's432 ?n nick)))

(defun erc-server-433 (proc parsed)
  ;; login-time 'nickname in use' message
  (erc-nickname-in-use (aref parsed 3) "already in use"))

(defvar erc-server-436-hook nil)

(defvar erc-server-437-hook nil)

(defun erc-server-437 (proc parsed)
  ;; "Nickname temporarily unavailable" on IRCnet
  (let ((what (aref parsed 3)))
    (unless (erc-channel-p what)
      (erc-nickname-in-use what "temporarily unavailable"))))

(defvar erc-server-441-hook nil)

(defun erc-server-442 (proc parsed)
  "Display \"not on that channel\" error."
  (let ((channel (aref parsed 3)))
    (erc-display-message
     parsed '(notice error) 'active
     's442 ?c channel)))
    
(defvar erc-server-443-hook nil)
(defvar erc-server-444-hook nil)
(defvar erc-server-445-hook nil)
(defvar erc-server-446-hook nil)
(defvar erc-server-451-hook nil)

(defun erc-server-461 (proc parsed)
  (let ((cmd (aref parsed 3))
	(msg (aref parsed 4)))
    (erc-display-message
     parsed '(notice error) 'active
     's461 ?c cmd ?m msg)))

(defvar erc-server-462-hook nil)
(defvar erc-server-463-hook nil)
(defvar erc-server-464-hook nil)
(defvar erc-server-465-hook nil)
(defvar erc-server-467-hook nil)
(defvar erc-server-471-hook nil)
(defvar erc-server-472-hook nil)
(defvar erc-server-473-hook nil)

(defun erc-server-474 (proc parsed)
  "Handle errors when the user is banned from the requested channel."
  (let ((channel (aref parsed 3)))
    (erc-display-message
     parsed '(notice error) nil
     's474 ?c channel)))

(defun erc-server-475 (proc parsed)
  "Handle error messages when a channel key is needed."
  (let ((channel (aref parsed 3)))
    (erc-display-message
     parsed '(notice error) nil
     's475 ?c channel)))

(defun erc-server-477 (proc parsed)
  "Display server's 477 message in the appropriate channel buffer."
  (let* ((channel (aref parsed 3))
	(msg (aref parsed 4))
	(buffer (erc-get-buffer channel proc)))
    (erc-display-message
     parsed 'notice buffer
     ;; freenode uses this to send random "info" messages,
     ;; so just show the message as is.
     (format "%s: %s" channel msg))))

(defun erc-server-481 (proc parsed)
  "Handle errors when the user needs IRC operator privileges."
  (erc-display-message
   parsed '(notice error) nil
   's481))

(defun erc-server-482 (proc parsed)
  (let ((channel (aref parsed 3)) (msg (aref parsed 4)))
    (erc-display-message
     parsed '(error notice) 'active
     's482 ?c channel ?m msg)))

(defvar erc-server-483-hook nil)
(defvar erc-server-491-hook nil)

(defun erc-server-501 (proc parsed)
  (erc-display-error-notice
   parsed
   (aref parsed 3)))

(defvar erc-server-502-hook nil)

(defun erc-process-ctcp-query (proc parsed nick login host)
  ;; FIXME: This needs a proper docstring -- Lawrence 2004-01-08
  "Process a CTCP query."
  (let ((queries (delete "" (split-string (aref parsed 3) "\C-a"))))
    (if (> (length queries) 4)
	(erc-display-message
	 parsed (list 'notice 'error) proc
	 "Too many CTCP queries in single message. Ignoring")
      (if (= 0 (length queries))
	  (erc-display-message
	   parsed (list 'notice 'error) proc
	   (format "Illegal empty CTCP query received from %s. Ignoring."
		   nick))
	(while queries
	  (let* ((type (upcase (car (split-string (car queries)))))
		 (hook (intern-soft (concat "erc-ctcp-query-" type "-hook"))))
	    (if (and hook (boundp hook)
		     erc-paranoid
		     (not (string-equal type "ACTION")))
		(erc-display-message
		 parsed 'error nil
		 (if (erc-current-nick-p (aref parsed 2))
		     (format "==> CTCP request from %s: %s\n"
			     (aref parsed 1) (car queries))
		   (format "==> CTCP request from %s to %s: %s\n"
			   (aref parsed 1) (aref parsed 2) (car queries)))))
	    (if (boundp hook)
		(if (equal type "ACTION")
		    (run-hook-with-args-until-success
		     hook proc parsed nick login host
		     (aref parsed 2) (car queries))
		  (run-hook-with-args-until-success hook proc nick login host
						    (aref parsed 2)
						    (car queries)))
	      (erc-display-message
	       parsed (list 'notice 'error) proc
	       'undefined-ctcp)))
	  (setq queries (cdr queries)))))))

(defvar erc-ctcp-query-ACTION-hook '(erc-ctcp-query-ACTION))

(defun erc-ctcp-query-ACTION (proc parsed nick login host to msg)
  "Respond to a CTCP ACTION query."
  (when (string-match "^ACTION\\s-\\(.*\\)\\s-*$" msg)
    (let ((s (match-string 1 msg))
	  (buf (or (erc-get-buffer to proc)
		   (erc-get-buffer nick proc)
		   (process-buffer proc))))
      (erc-display-message
       parsed 'action buf
       'ACTION ?n nick ?u login ?h host ?a s))))

(defvar erc-ctcp-query-CLIENTINFO-hook '(erc-ctcp-query-CLIENTINFO))

(defun erc-ctcp-query-CLIENTINFO (proc nick login host to msg)
  "Respond to a CTCP CLIENTINFO query."
  (when (string-match "^CLIENTINFO\\(\\s-*\\|\\s-+.*\\)$" msg)
    (let ((s (erc-client-info (erc-trim-string (match-string 1 msg)))))
      (unless erc-disable-ctcp-replies
	  (erc-send-ctcp-notice nick (format "CLIENTINFO %s" s)))))
  nil)

(defvar erc-ctcp-query-ECHO-hook '(erc-ctcp-query-ECHO))
(defun erc-ctcp-query-ECHO (proc nick login host to msg)
  "Respond to a CTCP ECHO query."
  (when (string-match "^ECHO\\s-+\\(.*\\)\\s-*$" msg)
    (let ((s (match-string 1 msg)))
      (unless erc-disable-ctcp-replies
	(erc-send-ctcp-notice nick (format "ECHO %s" s)))))
  nil)

(defvar erc-ctcp-query-FINGER-hook '(erc-ctcp-query-FINGER))
(defun erc-ctcp-query-FINGER (proc nick login host to msg)
  "Respond to a CTCP FINGER query."
  (unless erc-disable-ctcp-replies
    (let ((s (if erc-anonymous-login
		 (format "FINGER I'm %s." (erc-current-nick))
	       (format "FINGER %s (%s@%s)."
		       (user-full-name)
		       (user-login-name)
		       (system-name))))
	  (ns (erc-time-diff last-sent-time (erc-current-time))))
	(when (> ns 0)
	    (setq s (concat s " Idle for " (erc-sec-to-time ns))))
	(erc-send-ctcp-notice nick s)))
  nil)

(defvar erc-ctcp-query-PING-hook '(erc-ctcp-query-PING))
(defun erc-ctcp-query-PING (proc nick login host to msg)
  "Respond to a CTCP PING query."
  (when (string-match "^PING\\s-+\\(.*\\)" msg)
    (unless erc-disable-ctcp-replies
      (let ((arg (match-string 1 msg)))
	(erc-send-ctcp-notice nick (format "PING %s" arg)))))
  nil)

(defvar erc-ctcp-query-TIME-hook '(erc-ctcp-query-TIME))
(defun erc-ctcp-query-TIME (proc nick login host to msg)
  "Respond to a CTCP TIME query."
  (unless erc-disable-ctcp-replies
    (erc-send-ctcp-notice nick (format "TIME %s" (current-time-string))))
  nil)

(defvar erc-ctcp-query-USERINFO-hook '(erc-ctcp-query-USERINFO))
(defun erc-ctcp-query-USERINFO (proc nick login host to msg)
  "Respond to a CTCP USERINFO query."
  (unless erc-disable-ctcp-replies
    (erc-send-ctcp-notice nick (format "USERINFO %s" erc-user-information)))
  nil)

(defvar erc-ctcp-query-VERSION-hook '(erc-ctcp-query-VERSION))
(defun erc-ctcp-query-VERSION (proc nick login host to msg)
  "Respond to a CTCP VERSION query."
  (unless erc-disable-ctcp-replies
    (erc-send-ctcp-notice
     nick (format
	   "VERSION \C-bERC\C-b %s - an IRC client for emacs (\C-b%s\C-b)"
	   erc-version-string
	   erc-official-location)))
  nil)

(defun erc-process-ctcp-reply (proc parsed nick login host msg)
  "Process MSG as a CTCP reply."
  (let* ((type (car (split-string msg)))
	 (hook (intern (concat "erc-ctcp-reply-" type "-hook"))))
    (if (boundp hook)
	(run-hook-with-args-until-success
	 hook proc nick login host (aref parsed 2) msg)
      (erc-display-message
       parsed 'notice nil
       'CTCP-UNKNOWN ?n nick ?u login ?h host ?m msg))))

(defvar erc-ctcp-reply-ECHO-hook '(erc-ctcp-reply-ECHO))
(defun erc-ctcp-reply-ECHO (proc nick login host to msg)
  "Handle a CTCP ECHO reply."
  (when (string-match "^ECHO\\s-+\\(.*\\)\\s-*$" msg)
    (let ((message (match-string 1 msg)))
      (erc-display-message
       nil '(notice action) nil
       'CTCP-ECHO ?n nick ?m message)))
  nil)

(defvar erc-ctcp-reply-CLIENTINFO-hook '(erc-ctcp-reply-CLIENTINFO))
(defun erc-ctcp-reply-CLIENTINFO (proc nick login host to msg)
  "Handle a CTCP CLIENTINFO reply."
  (when (string-match "^CLIENTINFO\\s-+\\(.*\\)\\s-*$" msg)
    (let ((message (match-string 1 msg)))
      (erc-display-message
       nil 'notice nil
       'CTCP-CLIENTINFO ?n nick ?m message)))
  nil)

(defvar erc-ctcp-reply-FINGER-hook '(erc-ctcp-reply-FINGER))
(defun erc-ctcp-reply-FINGER (proc nick login host to msg)
  "Handle a CTCP FINGER reply."
  (when (string-match "^FINGER\\s-+\\(.*\\)\\s-*$" msg)
    (let ((message (match-string 1 msg)))
      (erc-display-message
       nil 'notice nil
       'CTCP-FINGER ?n nick ?m message)))
  nil)

(defvar erc-ctcp-reply-PING-hook '(erc-ctcp-reply-PING))
(defun erc-ctcp-reply-PING (proc nick login host to msg)
  "Handle a CTCP PING reply."
  (if (not (string-match "^PING\\s-+\\([0-9.]+\\)" msg))
      nil
    (let ((time (match-string 1 msg)))
      (condition-case nil
	  (let ((delta (erc-time-diff (string-to-number time)
				      (erc-current-time))))
	    (erc-display-message
	     nil 'notice 'active
	     'CTCP-PING ?n nick
	     ?t (erc-sec-to-time delta)))
	(range-error
	 (erc-display-message
	  nil 'error nil
	  'bad-ping-response ?n nick ?t time))))))

(defvar erc-ctcp-reply-TIME-hook '(erc-ctcp-reply-TIME))
(defun erc-ctcp-reply-TIME (proc nick login host to msg)
  "Handle a CTCP TIME reply."
  (when (string-match "^TIME\\s-+\\(.*\\)\\s-*$" msg)
    (let ((message (match-string 1 msg)))
      (erc-display-message
       nil 'notice nil
       'CTCP-TIME ?n nick ?m message)))
  nil)

(defvar erc-ctcp-reply-VERSION-hook '(erc-ctcp-reply-VERSION))
(defun erc-ctcp-reply-VERSION (proc nick login host to msg)
  "Handle a CTCP VERSION reply."
  (when (string-match "^VERSION\\s-+\\(.*\\)\\s-*$" msg)
    (let ((message (match-string 1 msg)))
      (erc-display-message
       nil 'notice nil
       'CTCP-VERSION ?n nick ?m message)))
  nil)

(defun erc-process-away (proc away-p)
  ;; FIXME: This docstring is AWFUL -- Lawrence 2004-01-08
  "Process the user being away, or returning from an away break."
  (let ((sessionbuf (process-buffer proc)))
    (when sessionbuf
      (with-current-buffer sessionbuf
	(when erc-away-nickname
	  (erc-log (format "erc-process-away: away-nick: %s, away-p: %s"
			   erc-away-nickname away-p))
	  (erc-cmd-NICK (if away-p
			    erc-away-nickname
			  erc-nick)))
	(cond
	 (away-p
	  (erc-with-all-buffers-of-server proc nil
					  (setq away (current-time))))
	 (t
	  (let ((away-time away))
	    ;; away must be set to NIL BEFORE sending anything to prevent
	    ;; an infinite recursion
	    (erc-with-all-buffers-of-server proc nil
					    (setq away nil))
	    (save-excursion
	      (set-buffer erc-active-buffer)
	      (when erc-public-away-p
		(erc-send-action
		 (erc-default-target)
		 (if away-time
		     (format "is back (gone for %s)"
			     (erc-sec-to-time
			      (erc-time-diff
			       (erc-emacs-time-to-erc-time away-time)
			       (erc-current-time))))
		   "is back")))))))))
    (erc-update-mode-line)))

;;;; List of channel members handling

(defun erc-channel-begin-receiving-names ()
  "Internal function.

Used when a channel names list is about to be received.  Should
be called with the current buffer set to the channel buffer.

See also `erc-channel-end-receiving-names'."
  (setq erc-channel-new-member-names (make-hash-table :test 'equal)))

(defun erc-channel-end-receiving-names ()
  "Internal function.

Used to fix `erc-channel-users' after a channel names list has been
received.  Should be called with the current buffer set to the
channel buffer.

See also `erc-channel-begin-receiving-names'."
  (maphash (lambda (nick user)
	     (if (null (gethash nick erc-channel-new-member-names))
		 (erc-remove-channel-user nick)))
	   erc-channel-users)
  (setq erc-channel-new-member-names nil))

(defun erc-channel-receive-names (names-string)
  "This function is for internal use only.

Update `erc-channel-users' according to NAMES-STRING.
NAMES-STRING is a string listing some of the names on the
channel."
  (let (names name op voice)
      ;; We need to delete "" because in XEmacs, (split-string "a ")
      ;; returns ("a" "").
      (setq names (delete "" (split-string names-string)))
      (let ((erc-channel-members-changed-hook nil))
	(dolist (item names)
	  (cond ((string-match "^@\\(.*\\)$" item)
		 (setq name (match-string 1 item)
		       op 'on
		       voice 'off))
		((string-match "^+\\(.*\\)$" item)
		 (setq name (match-string 1 item)
		       op 'off
		       voice 'on))
		(t (setq name item
			 op 'off
			 voice 'off)))
	(puthash (erc-downcase name) t
		 erc-channel-new-member-names)
	(erc-update-current-channel-member
	 name name t op voice)))
    (run-hooks 'erc-channel-members-changed-hook)))

(defcustom erc-channel-members-changed-hook nil
  "*This hook is called every time the variable `channel-members' changes.
The buffer where the change happened is current while this hook is called."
  :group 'erc-hooks
  :type 'hook)

(defun erc-update-user-nick (nick &optional new-nick
				  host login full-name info)
  "Updates the stored user information for the user with nickname
NICK.

See also: `erc-update-user'"
  (erc-update-user (erc-get-server-user nick) new-nick
		   host login full-name info))

(defun erc-update-user (user &optional new-nick
			     host login full-name info)
  "Update user info for USER.  USER must be an erc-server-user
struct.  Any of NEW-NICK, HOST, LOGIN, FULL-NAME, INFO which are
non-nil and not equal to the existing values for USER are used to
replace the stored values in USER.

If, any only if a change is made,
`erc-channel-members-changed-hook' is run for each channel for
which USER is a member, and `t' is returned."
  (let (changed)
    (when user
      (when (and new-nick
		 (not (equal (erc-server-user-nickname user)
			     new-nick)))
	(setq changed t)
	(erc-change-user-nickname user new-nick))
      (when (and host
		 (not (equal (erc-server-user-host user) host)))
	(setq changed t)
	(setf (erc-server-user-host user) host))
      (when (and login
		 (not (equal (erc-server-user-login user) login)))
	(setq changed t)
	(setf (erc-server-user-login user) login))
      (when (and full-name
		 (not (equal (erc-server-user-full-name user)
			     full-name)))
	(setq changed t)
	(setf (erc-server-user-full-name user) full-name))
      (when (and info
		 (not (equal (erc-server-user-info user) info)))
	(setq changed t)
	(setf (erc-server-user-info user) info))
      (if changed
	  (dolist (buf (erc-server-user-buffers user))
	    (if (buffer-live-p buf)
		(with-current-buffer buf
		  (run-hooks 'erc-channel-members-changed-hook))))))
    changed))

(defun erc-update-current-channel-member
  (nick new-nick &optional add op voice host login full-name info
	update-message-time)
  "Updates the stored user information for the user with nickname
NICK.  `erc-update-user' is called to handle changes to nickname,
host, login, full-name, and info.  If `op' or `voice' are
non-nil, they must be equal to either `on' or `off', in which
case the operator or voice status of USER in the current channel
is changed accordingly.  If `update-message-time' is non-nil, the
last-message-time of the user in the current channel is set
to (current-time).

If ADD is non-nil, the user will be added with the specified
information if it is not already present in the user or channel
lists.

If, and only if, changes are made, or the user is added,
`erc-channel-members-updated-hook' is run, and `t' is returned.

See also: `erc-update-user' and `erc-update-channel-member'."
  (let* (changed user-changed
	 (channel-data (erc-get-channel-user nick))
	 (cuser (if channel-data (cdr channel-data)))
	 (user (if channel-data (car channel-data)
		 (erc-get-server-user nick))))
    (if cuser
	(progn
	  (erc-log (format "update-member: user = %S, cuser = %S" user cuser))
	  (when (and op
		     (not (eq (erc-channel-user-op cuser) op)))
	      (setq changed t)
	    (setf (erc-channel-user-op cuser)
		  (cond ((eq op 'on) t)
				   ((eq op 'off) nil)
				   (t op))))
	  (when (and voice
		     (not (eq (erc-channel-user-voice cuser) voice)))
	      (setq changed t)
	    (setf (erc-channel-user-voice cuser)
		  (cond ((eq voice 'on) t)
				      ((eq voice 'off) nil)
				      (t voice))))
	  (when update-message-time
	    (setf (erc-channel-user-last-message-time cuser) (current-time)))
	  (setq user-changed
		(erc-update-user user new-nick
				 host login full-name info)))
      (when add
	(if (null user)
	    (progn 
	      (setq user (make-erc-server-user
			  :nickname nick
			  :host host
			  :full-name full-name
			  :login login
			  :info info
			  :buffers (list (current-buffer))))
	      (erc-add-server-user nick user))
	  (setf (erc-server-user-buffers user)
		(cons (current-buffer)
		      (erc-server-user-buffers user))))
	(setq cuser (make-erc-channel-user
		     :op (cond ((eq op 'on) t)
				       ((eq op 'off) nil)
				       (t op))
		     :voice (cond ((eq voice 'on) t)
				       ((eq voice 'off) nil)
				       (t voice))
		     :last-message-time
		     (if update-message-time (current-time))))
	(puthash (erc-downcase nick) (cons user cuser)
		 erc-channel-users)
	(setq changed t)))
    (when (and changed (null user-changed))
      (run-hooks 'erc-channel-members-changed-hook))
    (or changed user-changed add)))

(defun erc-update-channel-member (channel nick new-nick
				  &optional add op voice host login
				  full-name info update-message-time)
  "Updates user and channel information for the user with
nickname NICK in channel CHANNEL.

See also: `erc-update-current-channel-member'"
  (erc-with-buffer
   (channel)
   (erc-update-current-channel-member nick new-nick add op voice host
				      login full-name info
				      update-message-time)))

(defun erc-remove-current-channel-member (nick)
  "Remove NICK from current channel membership list.  Runs
`erc-channel-members-changed-hook'."
  (let ((channel-data (erc-get-channel-user nick)))
    (when channel-data
      (erc-remove-channel-user nick)
      (run-hooks 'erc-channel-members-changed-hook))))

(defun erc-remove-channel-member (channel nick)
  "Remove NICK from CHANNEL's membership list.

See also `erc-remove-current-channel-member'."
  (erc-with-buffer
   (channel)
   (erc-remove-current-channel-member nick)))

(defun erc-update-channel-topic (channel topic &optional modify)
  "Find a buffer for CHANNEL and set the TOPIC for it.

If optional MODIFY is 'append or 'prepend, then append or prepend the
TOPIC string to the current topic."
  (erc-with-buffer (channel)
    (unless (boundp 'channel-topic)
      (error "`channel-topic' appears to be unbound.  Please report as a bug"))
    (cond ((eq modify 'append)
	   (setq channel-topic (concat channel-topic topic)))
	  ((eq modify 'prepend)
	   (setq channel-topic (concat topic channel-topic)))
	  (t (setq channel-topic topic)))
    (erc-update-mode-line-buffer (current-buffer))))

(defun erc-set-modes (tgt mode-string)
  "Set the modes for the TGT provided as MODE-STRING."
  (let* ((modes (erc-parse-modes mode-string))
	 (add-modes (nth 0 modes))
	 (remove-modes (nth 1 modes))
	 ;; list of triples: (mode-char 'on/'off argument)
	 (arg-modes (nth 2 modes)))
    (cond ((erc-channel-p tgt); channel modes
	   (let ((buf (and (boundp 'erc-process) erc-process
			   (erc-get-buffer tgt erc-process)))
		 (ob (current-buffer)))
	     (when buf
	       (set-buffer buf)
	       (setq channel-modes add-modes)
	       (setq channel-user-limit nil)
	       (setq channel-key nil)
	       (while arg-modes
		 (let ((mode (nth 0 (car arg-modes)))
		       (onoff (nth 1 (car arg-modes)))
		       (arg (nth 2 (car arg-modes))))
		   (cond ((string-match "^[Ll]" mode)
			  (erc-update-channel-limit tgt onoff arg))
			 ((string-match "^[Kk]" mode)
			  (erc-update-channel-key tgt onoff arg))
			 (t nil)))
		 (setq arg-modes (cdr arg-modes)))
	       (erc-update-mode-line-buffer buf)
	       (erc-update-channel-info-buffer tgt)
	       (set-buffer ob))))
	  ;; we do not keep our nick's modes yet
	  ;;(t (setq erc-user-modes add-modes))
	  )
    ))

(defun erc-sort-strings (list-of-strings)
  "Sort LIST-OF-STRINGS in lexicographic order.

Side-effect free."
  (sort (copy-sequence list-of-strings) 'string<))

(defun erc-parse-modes (mode-string)
  "Parse MODE-STRING into a list.

Returns a list of three elements:

  (ADD-MODES REMOVE-MODES ARG-MODES).

The add-modes and remove-modes are lists of single-character strings
for modes without parameters to add and remove respectively.  The
arg-modes is a list of triples of the form:

  (MODE-CHAR ON/OFF ARGUMENT)."
  (if (string-match "^\\s-*\\(\\S-+\\)\\(\\s-.*$\\|$\\)" mode-string)
      (let ((chars (mapcar 'char-to-string (match-string 1 mode-string)))
	    ;; arguments in channel modes
	    (args-str (match-string 2 mode-string))
	    (args nil)
	    (add-modes nil)
	    (remove-modes nil)
	    (arg-modes nil); list of triples: (mode-char 'on/'off argument)
	    (add-p t))
	;; make the argument list
	(while (string-match "^\\s-*\\(\\S-+\\)\\(\\s-+.*$\\|$\\)" args-str)
	  (setq args (cons (match-string 1 args-str) args))
	  (setq args-str (match-string 2 args-str)))
	(setq args (nreverse args))
	;; collect what modes changed, and match them with arguments
	(while chars
	  (cond ((string= (car chars) "+") (setq add-p t))
		((string= (car chars) "-") (setq add-p nil))
		((string-match "^[ovbOVB]" (car chars))
		 (setq arg-modes (cons (list (car chars)
					     (if add-p 'on 'off)
					     (if args (car args) nil))
				       arg-modes))
		 (if args (setq args (cdr args))))
		((string-match "^[LlKk]" (car chars))
		 (setq arg-modes (cons (list (car chars)
					     (if add-p 'on 'off)
					     (if (and add-p args)
						 (car args) nil))
				       arg-modes))
		 (if (and add-p args) (setq args (cdr args))))
		(add-p (setq add-modes (cons (car chars) add-modes)))
		(t (setq remove-modes (cons (car chars) remove-modes))))
	  (setq chars (cdr chars)))
	(setq add-modes (nreverse add-modes))
	(setq remove-modes (nreverse remove-modes))
	(setq arg-modes (nreverse arg-modes))
	(list add-modes remove-modes arg-modes))
    nil))

(defun erc-update-modes (tgt mode-string &optional nick host login)
  "Update the mode information for TGT, provided as MODE-STRING.
Optional arguments: NICK, HOST and LOGIN - the attributes of the
person who changed the modes."
  (let* ((modes (erc-parse-modes mode-string))
	 (add-modes (nth 0 modes))
	 (remove-modes (nth 1 modes))
	 ;; list of triples: (mode-char 'on/'off argument)
	 (arg-modes (nth 2 modes)))
    ;; now parse the modes changes and do the updates
    (cond ((erc-channel-p tgt); channel modes
	   (let ((buf (and (boundp 'erc-process) erc-process
			   (erc-get-buffer tgt erc-process)))
		 (ob (current-buffer)))
	     (when buf
	       (set-buffer buf)
	       (if (not (boundp 'channel-modes))
		   (setq channel-modes nil))
	       (while remove-modes
		 (setq channel-modes (delete (car remove-modes) channel-modes)
		       remove-modes (cdr remove-modes)))
	       (while add-modes
		 (setq channel-modes (cons (car add-modes) channel-modes)
		       add-modes (cdr add-modes)))
	       (setq channel-modes (erc-sort-strings channel-modes))
	       (while arg-modes
		 (let ((mode (nth 0 (car arg-modes)))
		       (onoff (nth 1 (car arg-modes)))
		       (arg (nth 2 (car arg-modes))))
		   (cond ((string-match "^[oO]" mode)
			  (erc-update-channel-member tgt arg arg nil onoff))
			 ((string-match "^[Vv]" mode)
			  (erc-update-channel-member tgt arg arg nil nil
						     onoff))
			 ((string-match "^[Ll]" mode)
			  (erc-update-channel-limit tgt onoff arg))
			 ((string-match "^[Kk]" mode)
			  (erc-update-channel-key tgt onoff arg))
			 (t nil)); only ops are tracked now
		   (setq arg-modes (cdr arg-modes))))
	       (erc-update-mode-line buf)
	       (erc-update-channel-info-buffer tgt))))
	  ;; nick modes - ignored at this point
	  (t nil))))

(defun erc-update-channel-limit (channel onoff n)
  ;; FIXME: what does ONOFF actually do?  -- Lawrence 2004-01-08
  "Update CHANNEL's user limit to N."
  (if (or (not (eq onoff 'on))
	  (and (stringp n) (string-match "^[0-9]+$" n)))
      (erc-with-buffer
       (channel)
      (cond ((eq onoff 'on) (setq channel-user-limit (string-to-number n)))
	    (t (setq channel-user-limit nil))))))

(defun erc-update-channel-key (channel onoff key)
  "Update CHANNEL's key to KEY if ONOFF is 'on or to nil if it's 'off."
  (erc-with-buffer
   (channel)
   (cond ((eq onoff 'on) (setq channel-key key))
	 (t (setq channel-key nil)))))

(defun erc-find-channel-info-buffer (channel)
  "Return CHANNEL's info buffer.

If the info buffer doesn't exist, it is created.

The channel must be a string starting with `#'.  If it is not,
nil is returned and no buffer is created."
  (if erc-use-info-buffers
      (let (name res)
	(erc-with-buffer
	 (channel)
	  ;; if chnl is a real channel, the target must match
	 (when (and channel
		    (or (not (erc-channel-p channel))
			 (and (erc-default-target)
			     (erc-nick-equal-p (erc-default-target)
					       channel))))
	    ;;(message "test is true, chnl = %S" chnl)
	   (setq name (concat (buffer-name) ":INFO"))
	    (setq res (get-buffer name))
	    ;;(message "res (after 1st setq) = %S" res)
	    (unless res
	      (setq res (get-buffer-create name))
	      ;;(message "res (after 2nd setq) = %S" res)
	     (with-current-buffer res
	       (erc-info-mode))))
	 res))))

(defun erc-update-channel-info-buffer (chnl)
  "Update CHANNEL's info buffer with current information.

For example, topic, modes, user limit."
  (if erc-use-info-buffers
      (let* ((chnl-buf (or (and (bufferp chnl) chnl)
			   (and ;;(string-match "^[&#+!]" chnl)
			    (boundp 'erc-process) erc-process
			    (erc-get-buffer chnl erc-process))))
	     (ob (current-buffer))
	     (info-buf (erc-find-channel-info-buffer chnl))
	     limit key names topic modes info-point)
	;;(message "chnl-buf=%s" chnl-buf)
	;;(message "info-buf=%s" info-buf)
	;; if no target, do not update anything
	(when (and chnl chnl-buf info-buf)
	  (set-buffer chnl-buf)
	  (maphash (lambda (nick channel-data)
		     (setq names (cons channel-data names))))
	  (sort names (lambda (x y)
			(string< (erc-server-user-nickname (car x))
				 (erc-server-user-nickname (car y)))))
	  (setq topic (and (boundp 'channel-topic) channel-topic)
		modes (and (boundp 'channel-modes) channel-modes)
		limit (and (boundp 'channel-user-limit) channel-user-limit)
		key (and (boundp 'channel-key) channel-key))
	  ;;(message "info-buf=%s" info-buf)
	  (set-buffer info-buf)
	  (setq info-point (point))
	  ;;(message "info-point = %s" info-point)
	  (toggle-read-only -1)
	  (erase-buffer)
	  (insert (erc-interpret-controls
		   (cond ((erc-channel-p chnl)
			  (format "\C-c3Channel \C-b%s\C-b[%s]:\C-c %s"
				  chnl (length names)
				  (if (and topic (not (string= "" topic)))
				      topic
				    "<no topic>")))
			 (t (format "\C-c5Private from \C-b%s\C-b[%s]\C-c"
				    chnl (length names))))))
	  (if modes
	      (insert (concat "\nmodes: +"
			      (mapconcat 'identity modes ""))))
	  (if limit (insert (format "	user limit = %s" limit)))
	  (if key (insert (format "	key = %s" key)))
	  (insert "\n\n")
	  (dolist (name names)
	    (let* ((user (car name))
		   (cuser (cdr name))
		   (nick (erc-server-user-nickname user))
		   (host (erc-server-user-host user))
		   (login (erc-server-user-login user))
		   (full (erc-server-user-full-name user))
		   (info (erc-server-user-info user))
		   (op (and cuser (erc-channel-user-op cuser)))
		   (voice (and cuser (erc-channel-user-voice cuser))))
		  (insert
		   (concat (if op "@" " ")
			   (if voice "+" " ") nick " "
		       (if login login "")
			   (if host (concat "@" host) "")
			   (if full
			       (concat " "
				       (if (> (length full) 25)
					   (concat (substring full 0 22) "...")
					 full))
			     "")
		       (if info (concat " " info) "") "\n"))))
	  (goto-char info-point)
	  (toggle-read-only 1)
	  (set-buffer ob)))))

(defun erc-update-channel-info-buffers (&optional buffer-list)
  "Update all channels' info buffers.

If BUFFER-LIST is non-nil, only consider channels corresponding to buffers in
BUFFER-LIST.

See also `erc-update-channel-info-buffer'."
  (let ((ob (current-buffer))
	tgt)
    (or buffer-list
	(setq buffer-list (and (boundp 'erc-process)
			       (erc-buffer-list nil erc-process))))
    (dolist (buffer buffer-list)
      (when (buffer-live-p buffer)
	(set-buffer buffer)
	(setq tgt (erc-default-target))
	(if (and tgt (erc-channel-p tgt))
	    (erc-update-channel-info-buffer tgt))))
    (set-buffer ob)))

(defun erc-handle-user-status-change (type nlh &optional l)
  "Handle changes in any user's status.

So far, only nick change is handled.

Generally, the TYPE argument is a symbol describing the change type, NLH is
a list containing the original nickname, login name and hostname for the user,
and L is a list containing additional TYPE-specific arguments.

So far the following TYPE/L pairs are supported:

       event			TYP		       L

    nickname change	       'nick		    (NEW-NICK)"
  (erc-log (format "user-change: type: %S  nlh: %S  l: %S" type nlh l))
  (cond
   ;; nickname change
   ((equal type 'nick)
    t)
   (t
    nil)))

(defun erc-highlight-notice (s)
  "Highlight notice message S and return it.
See also variable `erc-notice-highlight-type'"
  (cond
   ((eq erc-notice-highlight-type 'prefix)
    (erc-put-text-property 0 (length erc-notice-prefix)
			   'face 'erc-notice-face s)
    s)
   ((eq erc-notice-highlight-type 'all)
    (erc-put-text-property 0 (length s) 'face 'erc-notice-face s)
    s)
   (t s)))

(defun erc-make-notice (message)
  "Notify the user of MESSAGE."
  (when erc-minibuffer-notice
    (message "%s" message))
  (erc-highlight-notice (concat erc-notice-prefix message)))

(defun erc-highlight-error (s)
  "Highlight error message S and return it."
  (erc-put-text-property 0 (length s) 'face 'erc-error-face s)
  s)

(defun erc-put-text-property (start end property value &optional object)
  "Set text-property for an object (usually a string).
START and END define the characters covered.
PROPERTY is the text-property set, usually the symbol `face'.
VALUE is the value for the text-property, usually a face symbol such as
the face `bold' or `erc-pal-face'.
OBJECT is a string which will be modified and returned.
OBJECT is modified without being copied first.

You can redefine or `defadvice' this function in order to add
EmacsSpeak support."
  (put-text-property start end property value object))

(defun erc-list (thing)
  "Return THING if THING is a list, or a list with THING as its element."
  (if (listp thing)
      thing
    (list thing)))

(defun erc-parse-user (string)
  "Parse STRING as a user specification (nick!login@host).

Return a list of the three separate tokens."
  (if (string-match "^\\([^!]*\\)!\\([^@]*\\)@\\(.*\\)$" string)
      (list (match-string 1 string)
	    (match-string 2 string)
	    (match-string 3 string))
    (list string "" "")))

(defun erc-extract-nick (string)
  "Return the nick corresponding to a user specification STRING.

See also `erc-parse-user'."
  (car (erc-parse-user string)))

(defun erc-put-text-properties (start end properties
				&optional object value-list)
  "Set text-properties for OBJECT.

START and END describe positions in OBJECT.
If VALUE-LIST is nil, set each property in PROPERTIES to t, else set
each property to the corresponding value in VALUE-LIST."
  (unless value-list
    (setq value-list (mapcar (lambda (x)
			       t)
			     properties)))
  (mapcar* (lambda (prop value)
	     (erc-put-text-property start end prop value object))
	   properties value-list))

;;; Input area handling:

(defun erc-beg-of-input-line ()
  "Return the value of `point' at the beginning of the input line.

This either returns the value of `point' at `erc-insert-marker', or at
`line-beginning-position'."
  (save-excursion
    (if (and (boundp 'erc-insert-marker)
	     (markerp erc-insert-marker))
	(goto-char erc-insert-marker)
      (forward-line 0))
    (point)))

(defun erc-end-of-input-line ()
  "Return the value of `point' at the end of the input line."
  (point-max))

(defun erc-parse-current-line ()
  "Parse current input line.
Returns a pair (PART1 . PART2), where PART1 is the input part before the point
and PART2 is the part after the point."
  (save-excursion
    (let* ((pnt (point))
	   (start (progn (goto-char (erc-beg-of-input-line))
			 (erc-bol)))
	   (end (erc-end-of-input-line)))
      (cons (buffer-substring start pnt)
	    (buffer-substring pnt end)))))

(defun erc-send-distinguish-noncommands (str)
  "If STR is an ERC non-command, set `erc-insert-this' to nil."
  (let* ((command (erc-extract-command-from-line str))
	 (cmd-fun (and command
		       (car command))))
    (when (and cmd-fun
	       (not (string-match "\n.+$" str))
	       (memq cmd-fun erc-noncommands-list))
      (setq erc-insert-this nil))))

;; FIXME: this function is too big.  The way we currently send input
;; is also far too limited.  There's no way of interpreting a line as
;; a command without sending, or displaying without sending, or
;; somesuch, which would be useful for things like erc-cmd-SAY.  --
;; Lawrence 2004-01-11
(defun erc-send-current-line ()
  "Parse current line and send it to IRC."
  (interactive)
  (if (< (point) (erc-beg-of-input-line))
      (beep)
    (setq erc-active-buffer (current-buffer))
    (let* ((inhibit-read-only t)
	   (line (progn
		   (goto-char (point-max))
		   (while (and (> (point) (+ 1 (length (erc-prompt))
					     erc-insert-marker))
			       (or (memq (char-before) '(?\r ?\n))
				   (eq ?\t (char-syntax (char-before)))))
		     (delete-backward-char 1))
		   (erc-parse-current-line)))
	   (str  (concat (car line) (cdr line)))
	   (old-buf (current-buffer)))
      (cond
       ((<= (point) erc-insert-marker)
	(message "Point is not in the input area")
	(ding))
       ((if erc-send-whitespace-lines
	    (string= str "")
	  (string-match "\\`[ \t\r\f\n]*\\'" str))
	(when erc-warn-about-blank-lines
	  (message "Blank line - ignoring...")
	  (ding)))
       (t
	(setq erc-send-this t)
	(setq erc-insert-this t)
	(run-hook-with-args 'erc-send-pre-hook str)
	(delete-region (erc-beg-of-input-line) (erc-end-of-input-line))
	(when erc-send-this
	  (let* ((lines (split-string str "\n"))
		 (multiline-p (< 1 (length lines))))
	    (mapc
	     (lambda (line)
	       (if (null erc-insert-this)
		   ;; Leave erc-insert-this set to t as much as possible.
		   ;; Fran Litterio <franl> has seen erc-insert-this set to
		   ;; nil while erc-send-pre-hook is running, which should
		   ;; never happen.  This may cure it.
		   (setq erc-insert-this t)
		 (let ((insert-position (point)))
		   (if (and (not multiline-p)
			    (char-equal (aref line 0) ?/)) ; is it a non-pasted command?
		       (unless erc-hide-prompt
			 (erc-display-prompt nil nil (erc-command-indicator)
					     (and (erc-command-indicator)
						  'erc-command-indicator-face)))
		     (insert (erc-format-my-nick)))	   ; ok, it's a privmsg.
		   (when (string-match "[\n]+$" line)	   ; remove the \ns at the end.
		     (setq line (substring line 0 (match-beginning 0))))
		   (erc-put-text-property 0 (length line)
					  'face 'erc-input-face line)
		   (insert (erc-interpret-controls line))
		   (goto-char (point-max))
		   (open-line 1)
		   (goto-char (point-max))
		   (set-marker (process-mark erc-process) (point))
		   (set-marker erc-insert-marker (point))
		   (save-excursion
		     (save-match-data
		       (save-restriction
			 (narrow-to-region insert-position (point-max))
			 (run-hook-with-args 'erc-send-modify-hook)
			 (run-hook-with-args 'erc-send-post-hook))))))
	       (erc-process-input-line (concat line "\n") t multiline-p))
	     lines)))
	;; Go to (point-max), but only if the command left an ERC buffer as the
	;; current buffer.
	(when (eq major-mode 'erc-mode)
	    (goto-char (point-max)))
	(when (buffer-live-p old-buf)
	  (set-buffer old-buf)
	  (set-marker (process-mark erc-process) (point))
	  (set-marker erc-insert-marker (point))
	  (let ((buffer-modified (buffer-modified-p)))
	    (erc-display-prompt)
	    (set-buffer-modified-p buffer-modified))
	  (run-hook-with-args 'erc-send-completed-hook str)))))))

(defun erc-extract-command-from-line (line)
  "Extract command and args from the input LINE.
If no command was given, return nil.  If command matches, return a
list of the form: (command args) where both elements are strings."
  (when (string-match "^/\\([A-Za-z]+\\)\\(\\s-+.*\\|\\s-*\\)$" line)
    (let* ((cmd (intern-soft
		 (format "erc-cmd-%s" (upcase (match-string 1 line)))))
	   ;; note: return is nil, we apply this simply for side effects
	   (canon-defun (while (and cmd (symbolp (symbol-function cmd)))
			  (setq cmd (symbol-function cmd))))
	   (cmd-fun (or cmd #'erc-cmd-default))
	   (arg (if cmd
		    (if (get cmd-fun 'do-not-parse-args)
			(format "%s" (match-string 2 line))
		      (delete "" (split-string (erc-trim-string
						(match-string 2 line)) " ")))
		  line)))
      (list cmd-fun arg))))

(defun erc-split-multiline-safe (string)
  "Split STRING, containing multiple lines and return them in a list.
Do it only for STRING as the complete input, do not carry unfinished
strings over to the next call."
  (let ((l ())
	(i0 0)
	(doit t))
    (while doit
      (let ((i (string-match "\r?\n" string i0))
	    (s (substring string i0)))
	(cond (i (setq l (cons (substring string i0 i) l))
		 (setq i0 (match-end 0)))
	      ((> (length s) 0)
		 (setq l (cons s l))(setq doit nil))
	      (t (setq doit nil)))))
    (nreverse l)))

;; nick handling

(defun erc-set-current-nick (nick)
  "Set the current nickname to NICK."
  (with-current-buffer (or (erc-server-buffer)
			   (current-buffer))
    (setq current-nick nick)))

(defun erc-current-nick ()
  "Return the current nickname."
  (with-current-buffer (if (buffer-live-p (erc-server-buffer))
			   (erc-server-buffer)
			 (current-buffer))
    current-nick))

(defun erc-current-nick-p (nick)
  "Return non-nil if NICK is the current nickname."
  (erc-nick-equal-p nick (erc-current-nick)))

(defun erc-nick-equal-p (nick1 nick2)
  "Return non-nil if NICK1 and NICK2 are the same.

This matches strings according to the IRC protocol's case convention.

See also `erc-downcase'."
  (string= (erc-downcase nick1)
	   (erc-downcase nick2)))

;; default target handling

(defun erc-default-target ()
  "Return the current default target (as a character string) or nil if none."
  (let ((tgt (car erc-default-recipients)))
    (cond
     ((not tgt) nil)
     ((listp tgt) (cdr tgt))
     (t tgt))))

(defun erc-add-default-channel (channel)
  "Add CHANNEL to the default channel list."

;;; This is no longer true.  The channel is added to another window
;;; and we don't want to mess the target there.
;"If the current default
;recipient is of QUERY type, then push the new default channel *after*
;the head"

  (let ((d1 (car erc-default-recipients))
	(d2 (cdr erc-default-recipients))
	(chl (downcase channel)))
;    (if (and (listp d1)
;	     (eq (car d1) 'QUERY))
;	(setq erc-default-recipients
;	      (cons d1 (cons chl d2)))
      (setq erc-default-recipients
	    (cons chl erc-default-recipients))
;)
      ))

(defun erc-delete-default-channel (channel &optional buffer)
  "Delete CHANNEL from the default channel list."
  (let ((ob (current-buffer)))
    (if (and buffer (bufferp buffer))
	(set-buffer buffer))
    (setq erc-default-recipients (delete (downcase channel)
					 erc-default-recipients))
    (set-buffer ob)))

(defun erc-add-query (nickname)
  "Add QUERY'd NICKNAME to the default channel list.

The previous default target of QUERY type gets removed"
  (let ((d1 (car erc-default-recipients))
	(d2 (cdr erc-default-recipients))
	(qt (cons 'QUERY (downcase nickname))))
    (if (and (listp d1)
	     (eq (car d1) 'QUERY))
	(setq erc-default-recipients (cons qt d2))
      (setq erc-default-recipients (cons qt erc-default-recipients)))))

(defun erc-delete-query ()
  "Delete the topmost target if it is a QUERY."

  (let ((d1 (car erc-default-recipients))
	(d2 (cdr erc-default-recipients)))
    (if (and (listp d1)
	     (eq (car d1) 'QUERY))
	(setq erc-default-recipients d2)
      (error "Current target is not a QUERY"))))

(defun erc-ignored-user-p (spec)
  "Return non-nil if SPEC matches something in `erc-ignore-list'.

Takes a full SPEC of a user in the form \"nick!login@host\", and
matches against all the regexp's in `erc-ignore-list'.  If any
match, returns that regexp."
  (dolist (ignored (with-current-buffer (erc-server-buffer) erc-ignore-list))
    (if (string-match (concat "\\`"ignored) spec)
	;; We have `require'd cl, so we can return from the block named nil
	(return ignored))))

(defmacro with-erc-channel-buffer (tgt proc &rest body)
  "Execute BODY for channel TGT and process PROC."
  `(if (not (and (stringp ,tgt)
		 (erc-channel-p ,tgt)
		 (processp ,proc)))
       nil
     (let ((buffer (erc-get-buffer ,tgt ,proc)))
       (if (not (bufferp buffer))
	   nil
	 (with-current-buffer buffer
	   ,@body)))))

(defun erc-ignored-reply-p (msg tgt proc)
  ;; FIXME: this docstring needs fixing -- Lawrence 2004-01-08
  "Return non-nil if MSG matches something in `erc-ignore-reply-list'.

Takes a message MSG to a channel and returns non-nil if the addressed
user matches any regexp in `erc-ignore-reply-list'."
  (let ((target-nick (erc-message-target msg)))
    (if (not target-nick)
	nil
      (with-erc-channel-buffer
       tgt proc
       (let ((user (erc-get-server-user target-nick)))
	 (when user
	      (erc-list-match erc-ignore-reply-list
			   (erc-user-spec user))))))))

(defun erc-message-target (msg)
  "Return the addressed target in MSG.

The addressed target is the string before the first colon in MSG."
  (if (string-match "^\\([^: ]*\\):" msg)
      (match-string 1 msg)
    nil))

(defun erc-user-spec (user)
  "Create a nick!user@host spec from a user struct."
  (let ((nick (erc-server-user-nickname user))
	(host (erc-server-user-host user))
	(login (erc-server-user-login user)))
  (concat (if nick
              nick
            "")
          "!"
	    (if login
		login
            "")
          "@"
          (if host
              host
	      ""))))

(defun erc-list-match (lst str)
  "Return non-nil if any regexp in LST matches STR."
  (memq nil (mapcar (lambda (regexp)
		      (not (string-match regexp str)))
		    lst)))

;; other "toggles"

(defun erc-toggle-ctcp-autoresponse (&optional arg)
  "Toggle automatic CTCP replies (like VERSION and PING).

If ARG is positive, turns CTCP replies on.

If ARG is non-nil and not positive, turns CTCP replies off."
  (interactive "P")
  (cond ((and (numberp arg) (> arg 0))
	 (setq erc-disable-ctcp-replies t))
	(arg (setq erc-disable-ctcp-replies nil))
	(t (setq erc-disable-ctcp-replies (not erc-disable-ctcp-replies))))
  (message "ERC CTCP replies are %s" (if erc-disable-ctcp-replies "OFF" "ON")))

(defun erc-toggle-flood-control (&optional arg)
  "Toggle between strict, normal and no flood control.

If ARG is positive, select strict control.
If ARG is non-nil and not positive, select normal control.
If ARG is nil, and `erc-flood-protect' is strict, select no control."
  (interactive "P")
  (cond ((and (numberp arg) (> arg 0))
	 (setq erc-flood-protect 'strict))
	(arg (setq erc-flood-protect 'normal))
	((eq erc-flood-protect 'strict)
	 (setq erc-flood-protect nil))
	(erc-flood-protect (setq erc-flood-protect 'strict))
	(t (setq erc-flood-protect 'normal)))
  (message "ERC flood control is %s"
	   (cond ((eq erc-flood-protect 'strict) "STRICT")
		 (erc-flood-protect "NORMAL")
		 (t "OFF"))))

(defun erc-toggle-interpret-controls (&optional arg)
  "Toggle interpretation of control sequences in messages.

If ARG is positive, interpretation is turned on.
Else interpretation is turned off."
  (interactive "P")
  (cond ((and (numberp arg) (> arg 0))
	 (setq erc-interpret-controls-p t))
	(arg (setq erc-interpret-controls-p nil))
	(t (setq erc-interpret-controls-p (not erc-interpret-controls-p))))
  (message "ERC color interpretation %s"
	   (if erc-interpret-controls-p "ON" "OFF")))

;; Some useful channel and nick commands for fast key bindings

(defun erc-invite-only-mode (&optional arg)
  "Turn on the invite only mode (+i) for the current channel.

If ARG is non-nil, turn this mode off (-i).

This command is sent even if excess flood is detected."
  (interactive "P")
  (setq erc-active-buffer (current-buffer))
  (let ((tgt (erc-default-target))
	(erc-force-send t))
    (cond ((or (not tgt) (not (erc-channel-p tgt)))
	   (erc-display-message nil 'error (current-buffer) 'no-target))
	  (arg (erc-load-irc-script-lines (list (concat "/mode " tgt " -i"))
					  t))
	  (t (erc-load-irc-script-lines (list (concat "/mode " tgt " +i"))
					t)))))

(defun erc-get-channel-mode-from-keypress (key)
  "Read a key sequence and call the corresponding channel mode function.
After doing C-c C-m type in a channel mode letter.

C-g means quit.
RET let's you type more than one mode at a time.
If \"l\" is pressed, `erc-set-channel-limit' gets called.
If \"k\" is pressed, `erc-set-channel-key' gets called.
Anything else will be sent to `erc-toggle-channel-mode'.
Only useful when called with a key binding."
  (interactive "kChannel mode (RET to set more than one): ")
  (when (featurep 'xemacs)
    (setq key (char-to-string (event-to-character (aref key 0)))))
  (cond ((equal key "\007")
	 (keyboard-quit))
	((equal key "\015")
	 (erc-insert-mode-command))
	((equal key "l")
	 (call-interactively 'erc-set-channel-limit))
	((equal key "k")
	 (call-interactively 'erc-set-channel-key))
	(t (erc-toggle-channel-mode key))))

(defun erc-toggle-channel-mode (mode &optional channel)
  "Toggle channel MODE.

If CHANNEL is non-nil, toggle MODE for that channel, otherwise use
`erc-default-target'."
  (interactive "P")
  (setq erc-active-buffer (current-buffer))
  (let ((tgt (or channel (erc-default-target)))
	(erc-force-send t))
    (cond ((or (null tgt) (null (erc-channel-p tgt)))
	   (erc-display-message nil 'error 'active 'no-target))
	  ((member mode channel-modes)
	   (erc-log (format "%s: Toggle mode %s OFF" tgt mode))
	   (message "Toggle channel mode %s OFF" mode)
	   (erc-send-command (format "MODE %s -%s" tgt mode)))
	  (t (erc-log (format "%s: Toggle channel mode %s ON" tgt mode))
	     (message "Toggle channel mode %s ON" mode)
	     (erc-send-command (format "MODE %s +%s" tgt mode))))))

(defun erc-insert-mode-command ()
  "Insert the line \"/mode <current target> \" at `point'."
  (interactive)
  (let ((tgt (erc-default-target)))
    (if tgt (insert (concat "/mode " tgt " "))
      (erc-display-message nil 'error (current-buffer) 'no-target))))

(defun erc-channel-names ()
  "Run \"/names #channel\" in the current channel."
  (interactive)
  (setq erc-active-buffer (current-buffer))
  (let ((tgt (erc-default-target)))
    (if tgt (erc-load-irc-script-lines (list (concat "/names " tgt)))
      (erc-display-message nil 'error (current-buffer) 'no-target))))

(defun erc-remove-text-properties-region (start end &optional object)
  "Clears the region (START,END) in OBJECT from all colors, etc."
  (interactive "r")
  (save-excursion
    (let ((inhibit-read-only t))
      (set-text-properties start end nil object))))

;; script execution and startup

(defun erc-find-file (file &optional path)
  "Search for a FILE in the filesystem.
First the `default-directory' is searched for FILE, then any directories
specified in the list PATH.

If FILE is found, return the path to it."
  (let ((filepath file))
    (if (file-readable-p filepath) filepath
      (progn
	(while (and path
		    (progn (setq filepath (concat (car path) "/" file))
			   (not (file-readable-p filepath))))
	  (setq path (cdr path)))
	(if path filepath nil)))))

(defun erc-select-startup-file ()
  "Select an ERC startup file.
See also `erc-startup-file-list'."
  (let ((l erc-startup-file-list)
	(f nil))
    (while (and (not f) l)
      (if (file-readable-p (car l))
	  (setq f (car l)))
      (setq l (cdr l)))
    f))

(defun erc-find-script-file (file)
  "Search for FILE in `default-directory', and any in `erc-script-path'."
  (erc-find-file file erc-script-path))

(defun erc-load-script (file)
  "Load a script from FILE.

FILE must be the full name, it is not searched in the
`erc-script-path'.  If the filename ends with `.el', then load it
as a emacs-lisp program.  Otherwise, treat it as a regular IRC
script"
  (erc-log (concat "erc-load-script: " file))
  (cond
   ((string-match "\\.el$" file)
    (load file))
   (t
    (erc-load-irc-script file))))

(defun erc-process-script-line (line &optional args)
  "Process an IRC script LINE.

Does script-specific substitutions (script arguments, current nick,
server, etc.)  in LINE and returns it.

Substitutions are: %C and %c = current target (channel or nick),
%S %s = current server, %N %n = my current nick, and %x is x verbatim,
where x is any other character;
$* = the entire argument string, $1 = the first argument, $2 = the second,
end so on."
  (if (not args) (setq args ""))
  (let* ((arg-esc-regexp "\\(\\$\\(\\*\\|[1-9][0-9]*\\)\\)\\([^0-9]\\|$\\)")
	 (percent-regexp "\\(%.\\)")
	 (esc-regexp (concat arg-esc-regexp "\\|" percent-regexp))
	 (tgt (erc-default-target))
	 (server (and (boundp 'erc-session-server) erc-session-server))
	 (nick (erc-current-nick))
	 (res "")
	 (tmp nil)
	 (arg-list nil)
	 (arg-num 0))
    (if (not tgt) (setq tgt ""))
    (if (not server) (setq server ""))
    (if (not nick) (setq nick ""))
    ;; First, compute the argument list
    (setq tmp args)
    (while (string-match "^\\s-*\\(\\S-+\\)\\(\\s-+.*$\\|$\\)" tmp)
      (setq arg-list (cons (match-string 1 tmp) arg-list))
      (setq tmp (match-string 2 tmp)))
    (setq arg-list (nreverse arg-list))
    (setq arg-num (length arg-list))
    ;; now do the substitution
    (setq tmp (string-match esc-regexp line))
    (while tmp
      ;;(message "beginning of while: tmp=%S" tmp)
      (let* ((hd (substring line 0 tmp))
	     (esc "")
	     (subst "")
	     (tail (substring line tmp)))
	(cond ((string-match (concat "^" arg-esc-regexp) tail)
	       (setq esc (match-string 1 tail))
	       (setq tail (substring tail (match-end 1))))
	      ((string-match (concat "^" percent-regexp) tail)
	       (setq esc (match-string 1 tail))
	       (setq tail (substring tail (match-end 1)))))
	;;(message "hd=%S, esc=%S, tail=%S, arg-num=%S" hd esc tail arg-num)
	(setq res (concat res hd))
	(setq subst
	      (cond ((string= esc "") "")
		    ((string-match "^\\$\\*$" esc) args)
		    ((string-match "^\\$\\([0-9]+\\)$" esc)
		     (let ((n (string-to-number (match-string 1 esc))))
		       (message "n = %S, integerp(n)=%S" n (integerp n))
		       (if (<= n arg-num) (nth (1- n) arg-list) "")))
		    ((string-match "^%[Cc]$" esc) tgt)
		    ((string-match "^%[Ss]$" esc) server)
		    ((string-match "^%[Nn]$" esc) nick)
		    ((string-match "^%\\(.\\)$" esc) (match-string 1 esc))
		    (t (erc-log (format "BUG in erc-process-script-line: bad escape sequence: %S\n" esc))
		       (message "BUG IN ERC: esc=%S" esc)
		       "")))
	(setq line tail)
	(setq tmp (string-match esc-regexp line))
	(setq res (concat res subst))
	;;(message "end of while: line=%S, res=%S, tmp=%S" line res tmp)
	))
    (setq res (concat res line))
    res))

(defun erc-load-irc-script (file &optional force)
  "Load an IRC script from FILE."
  (erc-log (concat "erc-load-script: " file))
  (let (str)
    (with-temp-buffer
      (insert-file file)
      (setq str (buffer-string)))
    (erc-load-irc-script-lines (erc-split-multiline-safe str) force)))

(defun erc-load-irc-script-lines (lines &optional force noexpand)
  "Load IRC script LINES (a list of strings).

If optional NOEXPAND is non-nil, do not expand script-specific
sequences, process the lines verbatim.  Use this for multiline
user input."
  (let* ((cb (current-buffer))
	 (pnt (point))
	 (s "")
	 (sp (erc-prompt))
	 (args (and (boundp 'erc-script-args) erc-script-args)))
    (if (and args (string-match "^ " args))
	(setq args (substring args 1)))
    ;; prepare the prompt string for echo
    (erc-put-text-property 0 (1- (length (erc-prompt)))
			   'face 'erc-prompt-face sp)
    (erc-put-text-property (length (erc-prompt)) (length sp)
			   'face 'erc-input-face sp)
    (while lines
      (setq s (car lines))
      (erc-log (concat "erc-load-script: CMD: " s))
      (unless (string-match "^\\s-*$" s)
	(let ((line (if noexpand s (erc-process-script-line s args))))
	  (if (and (erc-process-input-line line force)
		   erc-script-echo)
	      (progn
		(erc-put-text-property 0 (length line)
				       'face 'erc-input-face line)
		(erc-display-line (concat sp line) cb)))))
      (setq lines (cdr lines)))))

;; authentication

(defun erc-login ()
  "Perform user authentication at the IRC server."
  (erc-log (format "login: nick: %s, user: %s %s %s :%s"
		   (erc-current-nick)
		   (user-login-name)
		   (system-name)
		   erc-session-server
		   erc-session-user-full-name))
  (if erc-session-password
      (erc-send-command (format "PASS %s" erc-session-password))
    (message "Logging in without password"))
  (erc-send-command (format "NICK %s" (erc-current-nick)))
  (erc-send-command
   (format "USER %s %s %s :%s"
	   ;; hacked - S.B.
	   (if erc-anonymous-login erc-email-userid (user-login-name))
	   "0" "*"
	   erc-session-user-full-name)))

;; connection properties' heuristics

(defun erc-determine-parameters (&optional server port nick name)
  "Determine the connection and authentication parameters.
Sets the buffer local variables:

- erc-session-server
- erc-session-port
- erc-session-full-name
- current-nick"
  (setq erc-session-server (erc-compute-server server)
	erc-session-port (or port erc-default-port)
	erc-session-user-full-name (erc-compute-full-name name))
  (erc-set-current-nick (erc-compute-nick nick)))

(defun erc-compute-server (server)
  "Return an IRC server name.

Tries a number of increasingly more default methods until a non-nil value is
found:

- SERVER
- `erc-server'
- The value of the IRCSERVER environment variable
- `erc-default-server'."
  (or server
      erc-server
      (getenv "IRCSERVER")
      erc-default-server))

(defun erc-compute-nick (nick)
  "Return user's NICK.

Tries a number of increasingly more default methods until a non-nil value is
found:

- NICK
- `erc-nick'
- The value of the IRCNICK environment variable
- via the function `user-login-name'."
  (or nick
      (if (consp erc-nick) (car erc-nick) erc-nick)
      (getenv "IRCNICK")
      (user-login-name)))


(defun erc-compute-full-name (full-name)
  "Return user's FULL-NAME.

Tries a number of increasingly more default methods until a non-nil value is
found:

- FULL-NAME
- `erc-user-full-name'
- The value of the IRCNAME environment variable
- via the function `user-full-name'."
  (or name
      erc-user-full-name
      (getenv "IRCNAME")
      (if erc-anonymous-login "unknown" nil)
      (user-full-name)))

;; time routines

(defun erc-string-to-emacs-time (string)
  "Convert the long number represented by STRING into an Emacs format.
Returns a list of the form (HIGH LOW), compatible with Emacs time format."
  (let* ((n (string-to-number (concat string ".0"))))
    (list (truncate (/ n 65536))
	  (truncate (mod n 65536)))))

(defun erc-emacs-time-to-erc-time (time)
  "Convert Emacs TIME to a number of seconds since the epoch."
  (when time
    (+ (* (nth 0 time) 65536.0) (nth 1 time))))
;  (round (+ (* (nth 0 tm) 65536.0) (nth 1 tm))))

(defun erc-current-time ()
  "Return the `current-time' as a number of seconds since the epoch.

See also `erc-emacs-time-to-erc-time'."
  (erc-emacs-time-to-erc-time (current-time)))

(defun erc-time-diff (t1 t2)
  "Return the time difference in seconds between T1 and T2."
  (abs (- t2 t1)))

(defun erc-time-gt (t1 t2)
  "Check whether T1 > T2."
  (> t1 t2))

(defun erc-sec-to-time (ns)
  "Convert NS to a time string HH:MM.SS."
  (setq ns (truncate ns))
  (format "%02d:%02d.%02d"
	  (/ ns 3600)
	  (/ (% ns 3600) 60)
	  (% ns 60)))

(defun erc-seconds-to-string (seconds)
  "Convert a number of SECONDS into an English phrase."
  (let (days hours minutes format-args output)
    (setq days		(/ seconds 86400)
	  seconds	(% seconds 86400)
	  hours		(/ seconds 3600)
	  seconds	(% seconds 3600)
	  minutes	(/ seconds 60)
	  seconds	(% seconds 60)
	  format-args	(if (> days 0)
			    `("%d days, %d hours, %d minutes, %d seconds"
			      ,days ,hours ,minutes ,seconds)
			  (if (> hours 0)
			      `("%d hours, %d minutes, %d seconds"
				,hours ,minutes ,seconds)
			    (if (> minutes 0)
				`("%d minutes, %d seconds" ,minutes ,seconds)
			      `("%d seconds" ,seconds))))
	  output	(apply 'format format-args))
    ;; Change all "1 units" to "1 unit".
    (while (string-match "\\([^0-9]\\|^\\)1 \\S-+\\(s\\)" output)
      (setq output (replace-match "" nil nil output 2)))
    output))


;; info

(defconst erc-clientinfo-alist
  '(("ACTION" . "is used to inform about one's current activity")
    ("CLIENTINFO" . "gives help on CTCP commands supported by client")
    ("ECHO" . "echoes its arguments back")
    ("FINGER" . "shows user's name, location, and idle time")
    ("PING" . "measures delay between peers")
    ("TIME" . "shows client-side time")
    ("USERINFO" . "shows information provided by a user")
    ("VERSION" . "shows client type and version"))
  "Alist of CTCP CLIENTINFO for ERC commands.")

(defun erc-client-info (s)
  "Return CTCP CLIENTINFO on command S.
If S is NIL or an empty string then return general CLIENTINFO"
  (if (or (not s) (string= s ""))
      (concat
       (apply #'concat
	      (mapcar (lambda (e)
			(concat (car e) " "))
		      erc-clientinfo-alist))
       ": use CLIENTINFO <COMMAND> to get more specific information")
    (let ((h (assoc (upcase s) erc-clientinfo-alist)))
      (if h
	  (concat s " " (cdr h))
	(concat s ": unknown command")))))

;; Hook functions

(defun erc-directory-writable-p (dir)
  "Determine whether DIR is a writable directory.
If it doesn't exist, create it."
  (unless (file-attributes dir) (make-directory dir))
  (or (file-accessible-directory-p dir) (error "Cannot access %s" dir)))

(defun erc-kill-query-buffers (process)
  "Kill all buffers of PROCESS."
  ;; here, we only want to match the channel buffers, to avoid
  ;; "selecting killed buffers" b0rkage.
  (erc-with-all-buffers-of-server process
				  (lambda ()
				    (not (erc-server-buffer-p)))
				  (kill-buffer (current-buffer))))

(defun erc-nick-at-point ()
  "Give information about the nickname at `point'.

If called interactively, give a human readable message in the
minibuffer.  If called programatically, return the corresponding
entry of `channel-members'."
  (interactive)
  (require 'thingatpt)
  (let* ((word (word-at-point))
	 (channel-data (erc-get-channel-user word))
	 (cuser (cdr channel-data))
	 (user (if channel-data
		   (car channel-data)
		 (erc-get-server-user word)))
	 host login full-name info nick op voice)
    (when user
      (setq nick (erc-server-user-nickname user)
	    host (erc-server-user-host user)
	    login (erc-server-user-login user)
	    full-name (erc-server-user-full-name user)
	    info (erc-server-user-info user))
      (if cuser
	  (setq op (erc-channel-user-op cuser)
		voice (erc-channel-user-voice cuser)))
      (if (interactive-p)
	  (message "%s is %s@%s%s%s"
		   nick login host
		   (if full-name (format " (%s)" full-name) "")
		   (if (or op voice)
			       (format " and is +%s%s on %s"
			       (if op "o" "")
			       (if voice "v" "")
				       (erc-default-target))
			     ""))
	user))))

;; (defun erc-operator-p (nick &optional channel)
;;   "Report if NICK is a channel operator on CHANNEL.

;; If CHANNEL is nil, use `erc-default-target'."
;;   ;; Unfinished yet. This is intended to be a saveguard to prevent
;;   ;; doing things which require op rights client-side.
;;   ;; Interactively it could prevent looking in the info buffer or doing
;;   ;; C-c C-n and looking for the nick.
;;   (interactive (list (let ((nickinfo (erc-nick-at-point)))
;; 		       (if nickinfo
;; 			   nickinfo
;; 			 (read-from-minibuffer "Nick: ")))
;; 		     (erc-default-target)))
;;   (let ((result))
;;     (cond ((and (string= channel (erc-default-target)) (stringp nick))
;; 	   (setq result (car (cdr (assoc nick channel-members)))))
;; 	  ((listp nick)
;; 	   (progn (setq result (car (cdr nick)))
;; 		  (setq nick (car nick))))
;; 	  (t (error "Not implemented")))
;;     (if (interactive-p)
;; 	(message "%s %s operator on %s"
;; 			 nick
;; 			 (if result "is" "is not")
;; 			 channel)
;;       result)))

(defun erc-away-p ()
  "Return t if the current ERC process is set away."
  (save-excursion
    (and (boundp 'erc-process)
         (processp erc-process)
         (set-buffer (process-buffer erc-process))
         away)))

;; Mode line handling

(defcustom erc-mode-line-format
  (list (if (boundp 'mode-line-mule-info)
	    mode-line-mule-info
	  "")
	'mode-line-modified
	" " 'target-and/or-server
	" " 'status 'away
	'global-mode-string
	" %[(" 'mode-name 'mode-line-process 'minor-mode-alist "%n" ")%]--"
	'(line-number-mode "L%l--") '(-3 . "%p") "-%-")
  "The `mode-line-format' in `erc-mode'.

This variable is processed through `erc-prepare-mode-line-format'
first."
  :group 'erc
  :type 'sexp)

(defcustom erc-header-line-format
  '("[IRC] " nick " on " target
    " " channel-modes " " topic)
  "Format of the header-line in `erc-mode'.
Only used in Emacs 21. This variable is processed using
`erc-prepare-mode-line-format'."
  :group 'erc
  :type 'sexp)

(defcustom erc-header-line-uses-help-echo-p t
  "Show the contents of the header line in the echo area or as a tooltip
when you move point into the header line."
  :group 'erc
  :type 'boolean)

(defcustom erc-show-channel-key-p t
  "Show the the channel key in the header line."
  :group 'erc
  :type 'boolean)

(defcustom erc-common-server-suffixes
  '(("openprojects.net$" . "OPN")
    ("freenode.net$" . "OPN"))
  "Alist of common server name suffixes.
This variable is used in mode-line display to save screen
real estate.  Set it to nil if you want to avoid changing
displayed hostnames."
  :group 'erc
  :type 'alist)

(defcustom erc-mode-line-away-status-format
  "(AWAY since %a %b %d %H:%M) "
  "When you're away on a server, this is shown in the mode line.
This should be a string with substitution variables recognized by
format-time-message."
  :group 'erc
  :type 'string)

(defun erc-shorten-server-name (server-name)
  "Shorten SERVER-NAME according to `erc-common-server-suffixes'."
  (if (stringp server-name)
      (with-temp-buffer
	(insert server-name)
	(let ((alist erc-common-server-suffixes))
	  (while alist
	    (goto-char (point-min))
	(if (re-search-forward (caar alist) nil t)
	    (replace-match (cdar alist)))
	(setq alist (cdr alist))))
	(buffer-string))))

(defun erc-prepare-mode-line-format (line)
  "Format LINE with data acquired from the current `erc-mode' buffer.
The following symbols are recognized:
'away: String indicating away status or \"\" if you are not away
'channel-modes: The modes of the channel
'nick: The current nick name
'port: The session port
'status: \" (CLOSED) \" in case the process is no longer open/run
'target: The name of the target (channel or nickname or servername:port)
'target-and/or-server: In the server-buffer, this gets filled with the
		       value of erc-announced-server-name,
		       in a channel, the value of (erc-default-target) also
		       get concatenated.
'topic: The topic of the channel"
  (let ((away (when (and (boundp 'erc-process)
			 (processp erc-process))
		(with-current-buffer (process-buffer erc-process)
		  away)))
	(lag (when (and (boundp 'erc-process)
			(processp erc-process))
	       (with-current-buffer (process-buffer erc-process)
		 erc-lag))))
    (mapcar
     (lambda (sym)
       (cond ((eq sym 'nick)
	      (erc-current-nick))
	     ((eq sym 'target)
	      (let ((target (erc-default-target)))
		(if target
		    target
		  (concat (erc-shorten-server-name
			   (if (boundp 'erc-announced-server-name)
			       erc-announced-server-name
			     erc-session-server))
			  ":" (erc-port-to-string erc-session-port)))))
	     ((eq sym 'port)
	      (erc-port-to-string erc-session-port))
	     ((eq sym 'status)
	      (if (erc-process-alive)
		  ""
		" (CLOSED) "))
	     ((eq sym 'channel-modes)
	      (concat (apply 'concat
			     "(+" channel-modes)
		      (cond ((and channel-user-limit channel-key)
			     (if erc-show-channel-key-p
				 (format "lk %.0f %s" channel-user-limit channel-key)
			       (format "kl %.0f" channel-user-limit)))
			    (channel-user-limit
			     ;; Emacs has no bignums
			     (format "l %.0f" channel-user-limit))
			    (channel-key
			     (if erc-show-channel-key-p
				 (format "k %s" channel-key)
			       "k"))
			    (t ""))
		      (if lag (format ",lag:%.0f" lag) "")
		      ")"))
	     ((eq sym 'away)
	      (if away
		  (format-time-string erc-mode-line-away-status-format away)
		""))
	     ((eq sym 'topic)
	      (erc-interpret-controls channel-topic))
	     ((eq sym 'target-and/or-server)
	      (let ((server-name (erc-shorten-server-name
				  (if (boundp 'erc-announced-server-name)
				      erc-announced-server-name
				    erc-session-server))))
		(if (erc-default-target)
		    (concat (erc-string-no-properties (erc-default-target))
			    "@" server-name)
		  server-name)))
	     (t
	      sym)))
     line)))

(defun erc-update-mode-line-buffer (buffer)
  "Update the mode line in a single ERC buffer BUFFER."
  (with-current-buffer buffer
    (setq mode-line-format
	  (erc-prepare-mode-line-format erc-mode-line-format))
    (when (boundp 'header-line-format)
      (if erc-header-line-format
	  (let ((header (mapconcat 'identity
				   (erc-prepare-mode-line-format
				    erc-header-line-format)
				   "")))
	    (if erc-header-line-uses-help-echo-p
		(let ((help-echo (with-temp-buffer
				   (insert header)
				   (fill-region (point-min) (point-max))
				   (buffer-string))))
		  (setq header-line-format
			(erc-replace-regexp-in-string
			 "%"
			 "%%"
			 (propertize header 'help-echo help-echo))))
	      (setq header-line-format header)))
	(setq header-line-format nil)))))

(defun erc-update-mode-line (&optional buffer)
  "Update the mode line in BUFFER.

If BUFFER is nil, update the mode line in all ERC buffers."
  (if (and buffer (bufferp buffer))
      (erc-update-mode-line-buffer buffer)
    (dolist (buf (erc-buffer-list))
      (when (buffer-live-p buf)
	(erc-update-mode-line-buffer buf)))))

;; Miscellaneous

(defun erc-port-to-string (p)
  "Convert port P to a string.
P may be an integer or a service name."
  (if (integerp p)
      (int-to-string p)
    p))

(defun erc-string-to-port (s)
  "Convert string S to either an integer port number or a service name."
  (let ((n (string-to-number s)))
    (if (= n 0)
	s
      n)))

(defun erc-version (&optional here)
  "Show the version number of ERC in the minibuffer.
If optional argument HERE is non-nil, insert version number at point."
  (interactive "P")
  (let ((version-string
	 (format "ERC %s" erc-version-string)))
    (if here
	(insert version-string)
      (if (interactive-p)
	  (message "%s" version-string)
	version-string))))

(defun erc-version-modules (&optional here)
  "Show the version numbers of all loaded ERC modules in the minibuffer.
If optional argument HERE is non-nil, insert version number at point."
  (interactive "P")
  (let ((version-string
	 (mapconcat 'identity
		    (let (versions (case-fold-search nil))
		      (dolist (var (apropos-internal "^erc-.*version$"))
			(when (and (boundp var)
				   (stringp (symbol-value var)))
			  (setq versions (cons (format "%S: %s"
						       var (symbol-value var))
					       versions))))
		      versions) ", ")))
    (if here
	(insert version-string)
      (if (interactive-p)
	  (message "%s" version-string)
	version-string))))

(defun erc-modes (&optional here)
  "Show the active ERC modes in the minibuffer.
If optional argument HERE is non-nil, insert version number at point."
  (interactive "P")
  (let ((string
	 (mapconcat 'identity
		    (let (modes (case-fold-search nil))
		      (dolist (var (apropos-internal "^erc-.*mode$"))
			(when (and (boundp var)
				   (symbol-value var))
			  (setq modes (cons (symbol-name var)
					    modes))))
		      modes)
		    ", ")))
    (if here
	(insert string)
      (if (interactive-p)
	  (message "%s" string)
	string))))

(defun erc-latest-version ()
  "Retrieve the latest erc.el version from CVS."
  (interactive)
  (if (ignore-errors (require 'url))
      (progn
	(switch-to-buffer (get-buffer-create "*erc.el latest version*"))
	(delete-region (point-min) (point-max))
	(kill-all-local-variables)
	(url-insert-file-contents (concat
				   "http://cvs.sourceforge.net/viewcvs.py/"
				   "*checkout*/erc/erc/erc.el?content-type"
				   "=text%2Fplain&rev=HEAD"))
	(emacs-lisp-mode)
	(current-buffer))
    (error "URL needs to be installed")))

(defun erc-ediff-latest-version ()
  "Ediff your installed erc.el with the latest CVS version.
See also `erc-latest-version'."
  (interactive)
  (let ((current (locate-library "erc.el")))
    (if current
	(ediff-buffers (find-file current)
		       (erc-latest-version))
      (error "You do not appear to have the uncompiled erc.el file"))))

(defun erc-trim-string (s)
  "Trim leading and trailing spaces off S."
  (cond
   ((not (stringp s)) nil)
   ((string-match "^\\s-*$" s)
    "")
   ((string-match "^\\s-*\\(.*\\S-\\)\\s-*$" s)
    (match-string 1 s))
   (t
    s)))

(defun erc-arrange-session-in-multiple-windows ()
  "Open a window for every non-server buffer related to `erc-session-server'.

All windows are opened in the current frame."
  (interactive)
  (unless (boundp 'erc-process)
    (error "No erc-process found in current buffer"))
  (let ((bufs (erc-buffer-list nil erc-process)))
    (when bufs
      (delete-other-windows)
      (switch-to-buffer (car bufs))
      (setq bufs (cdr bufs))
      (while bufs
	(split-window)
	(switch-to-buffer-other-window (car bufs))
	(setq bufs (cdr bufs))
	(balance-windows)))))

(defun erc-popup-input-buffer ()
  "Provide a input buffer."
   (interactive)
   (let ((buffer-name (generate-new-buffer-name "*ERC input*"))
	 (mode (intern
		(completing-read
		 "Mode: "
		 (mapcar (lambda (e)
			   (list (symbol-name e)))
			 (apropos-internal "-mode$" 'commandp))
		 nil t))))
     (pop-to-buffer (make-indirect-buffer (current-buffer) buffer-name))
     (funcall mode)
     (narrow-to-region (point) (point))
     (shrink-window-if-larger-than-buffer)))

;;; Message catalog

(defun erc-make-message-variable-name (catalog entry)
  "Create a variable name corresponding to CATALOG's ENTRY."
  (intern (concat "erc-message-"
		  (symbol-name catalog) "-" (symbol-name entry))))

(defun erc-define-catalog-entry (catalog entry format-spec)
  "Set CATALOG's ENTRY to FORMAT-SPEC."
  (set (erc-make-message-variable-name catalog entry)
       format-spec))

(defun erc-define-catalog (catalog entries)
  "Define a CATALOG according to ENTRIES."
  (dolist (entry entries)
    (erc-define-catalog-entry catalog (car entry) (cdr entry))))

(erc-define-catalog
 'english
 '((bad-ping-response . "Unexpected PING response from %n (time %t)")
   (bad-syntax . "Error occurred - incorrect usage?\n%c %u\n%d")
   (incorrect-args . "Incorrect arguments. Usage:\n%c %u\n%d")
   (cannot-find-file . "Cannot find file %f")
   (cannot-read-file . "Cannot read file %f")
   (connect . "Connecting to %S:%p... ")
   (ctcp-request . "==> CTCP request from %n (%u@%h): %r")
   (flood-ctcp-off . "FLOOD PROTECTION: Automatic CTCP responses turned off.")
   (flood-strict-mode . "FLOOD PROTECTION: Switched to Strict Flood Control mode.")
   (disconnected . "Connection failed!  Re-establishing connection...")
   (disconnected-noreconnect . "Connection failed!  Not re-establishing connection.")
   (login . "Logging in as \'%n\'...")
   (nick-in-use . "%n is in use. Choose new nickname: ")
   (no-default-channel . "No default channel")
   (no-invitation . "You've got no invitation")
   (no-target . "No target")
   (undefined-ctcp . "Undefined CTCP query received. Silently ignored")
   (variable-not-bound . "Variable not bound!")
   (ACTION . "* %n %a")
   (CTCP-CLIENTINFO . "Client info for %n: %m")
   (CTCP-ECHO . "Echo %n: %m")
   (CTCP-FINGER . "Finger info for %n: %m")
   (CTCP-PING . "Ping time to %n is %t")
   (CTCP-TIME . "Time by %n is %m")
   (CTCP-UNKNOWN . "Unknown CTCP message from %n (%u@%h): %m")
   (CTCP-VERSION . "Version for %n is %m")
   (INVITE . "%n (%u@%h) invites you to channel %c")
   (JOIN   . "%n (%u@%h) has joined channel %c")
   (JOIN-you . "You have joined channel %c")
   (KICK . "%n (%u@%h) has kicked %k off channel %c: %r")
   (KICK-you . "You have been kicked off channel %c by %n (%u@%h): %r")
   (KICK-by-you . "You have kicked %k off channel %c: %r")
   (MODE   . "%n (%u@%h) has changed mode for %t to %m")
   (MODE-nick . "%n has changed mode for %t to %m")
   (NICK   . "%n (%u@%h) is now known as %N")
   (NICK-you . "Your new nickname is %N")
   (PART   . erc-message-english-PART)
   (PING   . "PING from server (last: %s sec. ago)")
   (QUIT   . "%n (%u@%h) has quit: %r")
   (TOPIC  . "%n (%u@%h) has set the topic for %c: \"%T\"")
   (WALLOPS . "Wallops from %n: %m")
   (s004   . "%s %v %U %C")
   (s221   . "User modes for %n: %m")
   (s252   . "%i operator(s) online")
   (s253   . "%i unknown connection(s)")
   (s254   . "%i channels formed")
   (s301   . "%n is AWAY: %r")
   (s303   . "Is online: %n")
   (s305   . "%m")
   (s306   . "%m")
   (s324   . "%c modes: %m")
   (s329   . "%c was created on %t")
   (s331   . "No topic is set for %c")
   (s332   . "Topic for %c: %T")
   (s333   . "%c: topic set by %n, %t")
   (s341   . "Inviting %n to channel %c")
   (s353   . "Users on %c: %u")
   (s379   . "%c: Forwarded to %f")
   (s401   . "%n: No such nick/channel")
   (s403   . "%c: No such channel")
   (s405   . "%c: You have joined too many channels")
   (s406   . "%n: There was no such nickname")
   (s412   . "No text to send")
   (s421   . "%c: Unknown command")
   (s432   . "%n is an erroneous nickname")
   (s442   . "%c: You're not on that channel")
   (s461   . "%c: not enough parameters")
   (s474   . "You can't join %c because you're banned (+b)")
   (s475   . "You must specify the correct channel key (+k) to join %c")
   (s481   . "Permission Denied - You're not an IRC operator")
   (s482   . "You need to be a channel operator of %c to do that")))

(defun erc-message-english-PART (&rest args)
  "Format a proper PART message.

This function is an example on what could be done with formatting
functions."
  (let ((nick (cadr (memq ?n args)))
	(user (cadr (memq ?u args)))
	(host (cadr (memq ?h args)))
	(channel (cadr (memq ?c args)))
	(reason (cadr (memq ?r args))))
    (if (string= nick (erc-current-nick))
	(format "You have left channel %s" channel)
      (format "%s (%s@%s) has left channel %s%s"
	      nick user host channel
	      (if (not (string= reason ""))
		  (format ": %s" reason)
		"")))))


(defvar erc-current-message-catalog 'english)
(make-variable-buffer-local 'erc-current-message-catalog)

(defun erc-retrieve-catalog-entry (entry &optional catalog)
  "Retrieve ENTRY from CATALOG.

If CATALOG is nil, `erc-current-message-catalog' is used.

If ENTRY is nil in CATALOG, it is retrieved from the fallback,
english, catalog."
  (unless catalog (setq catalog erc-current-message-catalog))
  (let ((var (erc-make-message-variable-name catalog entry)))
    (if (boundp var)
	(symbol-value var)
      (when (boundp (erc-make-message-variable-name 'english entry))
	(symbol-value (erc-make-message-variable-name 'english entry))))))

(defun erc-format-message (msg &rest args)
  "Format MSG according to ARGS.

See also `format-spec'."
  (when (oddp (length args))
    (error "Obscure usage of this function appeared"))
  (let ((entry (erc-retrieve-catalog-entry msg)))
    (when (functionp entry)
      (setq entry (apply entry args)))
    (format-spec entry (apply 'format-spec-make args))))

;;; Various hook functions

(defun erc-make-read-only ()
  "Make all the text in the current buffer read-only.
Put this function on `erc-insert-post-hook' and/or `erc-send-post-hook'."
  (put-text-property (point-min) (point-max) 'read-only t)
  (put-text-property (point-min) (point-max) 'front-nonsticky t))



;; Imenu Autoload

(add-hook 'erc-mode-hook
	  (lambda ()
	    (setq imenu-create-index-function 'erc-create-imenu-index)))
(autoload 'erc-create-imenu-index "erc-imenu" "Imenu index creation function")

(defcustom erc-kill-server-hook '(erc-kill-server)
  "*Invoked whenever a server-buffer is killed via `kill-buffer'."
  :group 'erc-hooks
  :type 'hook)

(defcustom erc-kill-channel-hook '(erc-kill-channel)
  "*Invoked whenever a channel-buffer is killed via `kill-buffer'."
  :group 'erc-hooks
  :type 'hook)

(defcustom erc-kill-buffer-hook nil
  "*Hook run whenever a non-server or channel buffer is killed.

See also `kill-buffer'."
  :group 'erc-hooks
  :type 'hook)

(defun erc-kill-buffer-function ()
  "Function to call when an ERC buffer is killed.
This function should be on `kill-buffer-hook'.
When the current buffer is in `erc-mode', this function will run
one of the following hooks:
`erc-kill-server-hook' if the server buffer was killed,
`erc-kill-channel-hook' if a channel buffer was killed,
or `erc-kill-buffer-hook' if any other buffer."
  (when (eq major-mode 'erc-mode)
    (erc-remove-channel-users)
    (cond
     ((eq (erc-server-buffer) (current-buffer))
      (run-hooks 'erc-kill-server-hook))
     ((erc-channel-p (erc-default-target))
      (run-hooks 'erc-kill-channel-hook))
     (t
      (run-hooks 'erc-kill-buffer-hook)))))

(defun erc-process-alive ()
  "Return non-nil when `erc-process' is open or running."
  (and (boundp 'erc-process)
       (processp erc-process)
       (memq (process-status erc-process) '(run open))))

(defun erc-kill-server ()
  "Sends a QUIT command to the server when the server buffer is killed.
This function should be on `erc-kill-server-hook'."
  (when (erc-process-alive)
    (erc-send-command (format "QUIT :%s" (funcall erc-quit-reason nil)))))

(defun erc-kill-channel ()
  "Sends a PART command to the server when the channel buffer is killed.
This function should be on `erc-kill-channel-hook'."
  (when (erc-process-alive)
    (erc-send-command
     (format "PART %s :%s"
	     (erc-default-target)
	     (funcall erc-part-reason nil)))))

(add-hook 'kill-buffer-hook 'erc-kill-buffer-function)

;;; Inline modules and utilities too small for their own file

;; Smiley
(define-erc-module smiley nil
  "This mode translates text-smileys such as :-) into pictures.
This requires the function `smiley-region', which is defined in
smiley.el, which is part of Gnus."
  ((add-hook 'erc-insert-modify-hook 'erc-smiley)
   (add-hook 'erc-send-modify-hook 'erc-smiley))
  ((remove-hook 'erc-insert-modify-hook 'erc-smiley)
   (remove-hook 'erc-send-modify-hook 'erc-smiley)))

(defun erc-smiley ()
  "Smilify a region.
This function should be used with `erc-insert-modify-hook'."
  (when (fboundp 'smiley-region)
    (smiley-region (point-min) (point-max))))

;; Unmorse
(defun erc-unmorse ()
  "Unmorse some text.
Add this to `erc-insert-modify-hook' if you happen to be on a
channel that has weird people talking in morse to each other.

See also `unmorse-region'."
  (goto-char (point-min))
  (when (re-search-forward "[.-]+\\([.-]+[/ ]\\)+[.-]+" nil t)
    (unmorse-region (match-beginning 0) (match-end 0))))

(defun erc-occur (string &optional proc)
  "Search for STRING in all buffers related to current server.
If called interactively and prefix argument is given, search on all connected
servers.  If called from a program, PROC specifies the server process."
  (interactive
   (list (read-string "Search for: ")
	 (if current-prefix-arg
	     nil erc-process)))
  (if (fboundp 'multi-occur)
      (multi-occur (erc-buffer-list nil proc) string)
    (error "`multi-occur' is not defined as a function")))

(provide 'erc)

;;; erc.el ends here
;;
;; Local Variables:
;; mode: outline-minor
;; outline-regexp: ";;+"
;; indent-tabs-mode: t
;; tab-width: 8
;; standard-indent: 4
;; End:
