/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
import {
  ChallengeResponse,
  HttpStatusCode,
  TalerError,
  TalerMerchantApi,
  TalerMerchantInstanceHttpClient,
  TalerMerchantManagementResultByMethod,
  assertUnreachable,
} from "@gnu-taler/taler-util";
import {
  useChallengeHandler,
  useTranslationContext,
} from "@gnu-taler/web-util/browser";
import { Fragment, VNode, h } from "preact";
import { useState } from "preact/hooks";
import { ErrorLoadingMerchant } from "../../../components/ErrorLoadingMerchant.js";
import { Loading } from "../../../components/exception/loading.js";
import { NotificationCard } from "../../../components/menu/index.js";
import { useSessionContext } from "../../../context/session.js";
import {
  useInstanceDetails,
  useManagedInstanceDetails,
} from "../../../hooks/instance.js";
import { Notification } from "../../../utils/types.js";
import { LoginPage } from "../../login/index.js";
import { NotFoundPageOrAdminCreate } from "../../notfound/index.js";
import { UpdatePage } from "./UpdatePage.js";
import { SolveMFAChallenges } from "../../../components/SolveMFA.js";
import { DeletePage } from "./DeletePage.js";

export interface Props {
  onBack: () => void;
  onConfirm: () => void;
}

export default function Update(props: Props): VNode {
  const { lib } = useSessionContext();
  const updateInstance = lib.instance.updateCurrentInstance.bind(lib.instance);
  const result = useInstanceDetails();
  return CommonUpdate(props, result, updateInstance);
}

export function AdminUpdate(props: Props & { instanceId: string }): VNode {
  const { lib } = useSessionContext();
  const t = lib.subInstanceApi(props.instanceId).instance;
  const updateInstance = t.updateCurrentInstance.bind(t);
  const result = useManagedInstanceDetails(props.instanceId);
  return CommonUpdate(props, result, updateInstance);
}

function CommonUpdate(
  { onBack, onConfirm }: Props,
  result:
    | TalerMerchantManagementResultByMethod<"getInstanceDetails">
    | TalerError
    | undefined,
  updateInstance: typeof TalerMerchantInstanceHttpClient.prototype.updateCurrentInstance,
): VNode {
  const [notif, setNotif] = useState<Notification | undefined>(undefined);
  const { i18n } = useTranslationContext();
  const { state } = useSessionContext();

  if (!result) return <Loading />;
  if (result instanceof TalerError) {
    return <ErrorLoadingMerchant error={result} />;
  }
  if (result.type === "fail") {
    switch (result.case) {
      case HttpStatusCode.Unauthorized: {
        return <LoginPage />;
      }
      case HttpStatusCode.NotFound: {
        return <NotFoundPageOrAdminCreate />;
      }
      default: {
        assertUnreachable(result);
      }
    }
  }

  const mfa = useChallengeHandler();

  const [doUpdate, repeatUpdate] = mfa.withMfaHandler(
    ({ challengeIds, onChallengeRequired }) =>
      async function doUpdateImpl(
        d: TalerMerchantApi.InstanceReconfigurationMessage,
      ) {
        if (state.status !== "loggedIn") {
          return;
        }
        try {
          const resp = await updateInstance(state.token, d, { challengeIds });
          if (resp.type === "ok") {
            return onConfirm();
          }
          switch (resp.case) {
            case HttpStatusCode.Accepted: {
              onChallengeRequired(resp.body);
              return;
            }
            case HttpStatusCode.Unauthorized:
            case HttpStatusCode.NotFound: {
              setNotif({
                message: i18n.str`Failed to update instance`,
                type: "ERROR",
                description: resp.detail?.hint,
              });
            }
          }
        } catch (error) {
          return setNotif({
            message: i18n.str`Failed to update instance`,
            type: "ERROR",
            description: error instanceof Error ? error.message : String(error),
          });
        }
      },
  );
  const [deleting, setDeleting] = useState<boolean>();

  if (mfa.pendingChallenge) {
    return (
      <SolveMFAChallenges
        currentChallenge={mfa.pendingChallenge}
        onCompleted={repeatUpdate}
        onCancel={mfa.doCancelChallenge}
      />
    );
  }

  return (
    <Fragment>
      <NotificationCard notification={notif} />
      <UpdatePage
        onBack={onBack}
        isLoading={false}
        selected={result.body}
        onUpdate={doUpdate}
      />
      <div class="columns">
        <div class="column" />
        <div class="column is-four-fifths">
          <button
            class="button "
            onClick={() => {
              setDeleting(true);
            }}
          >
            <i18n.Translate>Delete this instance</i18n.Translate>
          </button>

          {!deleting ? undefined : (
            <DeletePage instanceId={state.instance} onDeleted={() => {}} />
          )}
        </div>
        <div class="column" />
      </div>
    </Fragment>
  );
}
