/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */

import { HttpStatusCode, TalerError, TalerMerchantApi } from "@gnu-taler/taler-util";
import {
  ButtonBetterBulma,
  LocalNotificationBannerBulma,
  useLocalNotificationBetter,
  useTranslationContext,
} from "@gnu-taler/web-util/browser";
import { h, VNode } from "preact";
import { useState } from "preact/hooks";
import { ProductForm } from "../../../../components/product/ProductForm.js";
import { useSessionContext } from "../../../../context/session.js";
import { useInstanceCategories } from "../../../../hooks/category.js";
import { Loading } from "../../../../components/exception/loading.js";
import { ErrorLoadingMerchant } from "../../../../components/ErrorLoadingMerchant.js";

const TALER_SCREEN_ID = 57;

type Entity = TalerMerchantApi.ProductDetail & { product_id: string };

interface Props {
  onBack?: () => void;
  onConfirm: () => void;
  product: Entity;
}

export function UpdatePage({ product, onBack, onConfirm }: Props): VNode {
  const { state: session, lib } = useSessionContext();
  const [form, setForm] = useState<TalerMerchantApi.ProductDetail>();

  const [notification, safeFunctionHandler] = useLocalNotificationBetter();
  const update = safeFunctionHandler(
    lib.instance.updateProduct.bind(lib.instance),
    !session.token || !form
      ? undefined
      : [session.token, product.product_id, form],
  );
  update.onSuccess = onConfirm;
  update.onFail = (fail) => {
    switch (fail.case) {
      case HttpStatusCode.Unauthorized:
        return i18n.str`Unauthorized.`;
      case HttpStatusCode.NotFound:
        return i18n.str`Not found.`;
      case HttpStatusCode.Conflict:
        return i18n.str`Conflict.`;
    }
  };

  const { i18n } = useTranslationContext();

    // FIXME: if the category list is big the will bring a lot of info
    // we could find a lazy way to add up on searches
    const categoriesResult = useInstanceCategories();
    if (!categoriesResult) return <Loading  />;
    if (categoriesResult instanceof TalerError) {
      return <ErrorLoadingMerchant error={categoriesResult} />;
    }
    const categories =
      categoriesResult.type === "fail" ? [] : categoriesResult.body.categories;
  

  return (
    <div>
      <LocalNotificationBannerBulma notification={notification} />
      <section class="section">
        <section class="hero is-hero-bar">
          <div class="hero-body">
            <div class="level">
              <div class="level-left">
                <div class="level-item">
                  <span class="is-size-4">
                    <i18n.Translate>Product id:</i18n.Translate>
                    <b>{product.product_id}</b>
                  </span>
                </div>
              </div>
            </div>
          </div>
        </section>
        <hr />

        <div class="columns">
          <div class="column" />
          <div class="column is-four-fifths">
            <ProductForm initial={product} onSubscribe={setForm} alreadyExist categories={categories}/>

            <div class="buttons is-right mt-5">
              {onBack && (
                <button type="button" class="button" onClick={onBack}>
                  <i18n.Translate>Cancel</i18n.Translate>
                </button>
              )}
              <ButtonBetterBulma
                onClick={update}
                data-tooltip={
                  !update.args
                    ? i18n.str`Please complete the marked fields`
                    : "confirm operation"
                }
                type="submit"
              >
                <i18n.Translate>Confirm</i18n.Translate>
              </ButtonBetterBulma>
            </div>
          </div>
          <div class="column" />
        </div>
      </section>
    </div>
  );
}
