/*
 * This file is part of LibEuFin.
 * Copyright (C) 2024 Taler Systems S.A.

 * LibEuFin is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation; either version 3, or
 * (at your option) any later version.

 * LibEuFin is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General
 * Public License for more details.

 * You should have received a copy of the GNU Affero General Public
 * License along with LibEuFin; see the file COPYING.  If not, see
 * <http://www.gnu.org/licenses/>
 */

package tech.libeufin.nexus.cli

import com.github.ajalt.clikt.core.CliktCommand
import com.github.ajalt.clikt.parameters.arguments.argument
import com.github.ajalt.clikt.parameters.arguments.multiple
import com.github.ajalt.clikt.parameters.arguments.unique
import com.github.ajalt.clikt.parameters.groups.provideDelegate
import com.github.ajalt.clikt.parameters.options.option
import com.github.ajalt.clikt.parameters.types.enum
import kotlinx.coroutines.delay
import tech.libeufin.common.*
import tech.libeufin.nexus.*
import tech.libeufin.nexus.db.Database
import tech.libeufin.nexus.db.PaymentDAO.IncomingRegistrationResult
import tech.libeufin.nexus.ebics.EbicsClient
import tech.libeufin.nexus.ebics.SupportedDocument
import java.io.IOException
import java.io.InputStream
import java.time.Duration
import java.time.Instant
import kotlin.time.toKotlinDuration

/** Ingests an outgoing [payment] into [db] */
suspend fun ingestOutgoingPayment(
    db: Database,
    payment: OutgoingPayment
) {
    val metadata: Pair<ShortHashCode, ExchangeUrl>? = payment.wireTransferSubject?.let { 
        runCatching { parseOutgoingTxMetadata(it) }.getOrNull()
    }
    val result = db.payment.registerOutgoing(payment, metadata?.first, metadata?.second)
    if (result.new) {
        if (result.initiated)
            logger.info("$payment")
        else 
            logger.warn("$payment recovered")
    } else {
        logger.debug("{} already seen", payment)
    }
}

/** 
 * Ingest an incoming [payment] into [db]
 * Stores the payment into valid talerable ones or bounces it, according to [accountType] .
 */
suspend fun ingestIncomingPayment(
    db: Database,
    payment: IncomingPayment,
    accountType: AccountType
) {
    suspend fun bounce(msg: String) {
        if (payment.bankId == null) {
            logger.debug("$payment ignored: missing bank ID")
            return;
        }
        when (accountType) {
            AccountType.exchange -> {
                val result = db.payment.registerMalformedIncoming(
                    payment,
                    payment.amount, 
                    Instant.now()
                )
                if (result.new) {
                    logger.info("$payment bounced in '${result.bounceId}': $msg")
                } else {
                    logger.debug("{} already seen and bounced in '{}': {}", payment, result.bounceId, msg)
                }
            }
            AccountType.normal -> {
                val res = db.payment.registerIncoming(payment)
                if (res.new) {
                    logger.info("$payment")
                } else {
                    logger.debug("{} already seen", payment)
                }
            }
        }
    }
    runCatching { parseIncomingTxMetadata(payment.wireTransferSubject) }.fold(
        onSuccess = { reservePub -> 
            when (val res = db.payment.registerTalerableIncoming(payment, reservePub)) {
                IncomingRegistrationResult.ReservePubReuse -> bounce("reverse pub reuse")
                is IncomingRegistrationResult.Success -> {
                    if (res.new) {
                        logger.info("$payment")
                    } else {
                        logger.debug("{} already seen", payment)
                    }
                }
            }
        },
        onFailure = { e -> bounce(e.fmt())}
    )
}

/** Ingest an EBICS [payload] of [document] into [db] */
private suspend fun ingestPayload(
    db: Database,
    cfg: NexusEbicsConfig,
    payload: InputStream,
    document: SupportedDocument
) {
    /** Ingest a single EBICS [xml] [document] into [db] */
    suspend fun ingest(xml: InputStream) {
        when (document) {
            SupportedDocument.CAMT_052, SupportedDocument.CAMT_053, SupportedDocument.CAMT_054 -> {
                try {
                    parseTx(xml, cfg.currency, cfg.dialect).forEach {
                        if (cfg.fetch.ignoreBefore != null && it.executionTime < cfg.fetch.ignoreBefore) {
                            logger.debug("IGNORE {}", it)
                        } else {
                            when (it) {
                                is IncomingPayment -> ingestIncomingPayment(db, it, cfg.accountType)
                                is OutgoingPayment -> ingestOutgoingPayment(db, it)
                                is TxNotification.Reversal -> {
                                    logger.error("BOUNCE '${it.msgId}': ${it.reason}")
                                    db.initiated.reversal(it.msgId, "Payment bounced: ${it.reason}")
                                }
                            }
                        }
                    }
                } catch (e: Exception) {
                    throw Exception("Ingesting notifications failed", e)
                }
            }
            SupportedDocument.PAIN_002_LOGS -> {
                val acks = parseCustomerAck(xml)
                for (ack in acks) {
                    when (ack.actionType) {
                        HacAction.ORDER_HAC_FINAL_POS -> {
                            logger.debug("{}", ack)
                            db.initiated.logSuccess(ack.orderId!!)?.let { requestUID ->
                                logger.info("Payment '$requestUID' accepted at ${ack.timestamp.fmtDateTime()}")
                            }
                        }
                        HacAction.ORDER_HAC_FINAL_NEG -> {
                            logger.debug("{}", ack)
                            db.initiated.logFailure(ack.orderId!!)?.let { (requestUID, msg) ->
                                logger.error("Payment '$requestUID' refused at ${ack.timestamp.fmtDateTime()}${if (msg != null) ": $msg" else ""}")
                            }
                        }
                        else -> {
                            logger.debug("{}", ack)
                            if (ack.orderId != null) {
                                db.initiated.logMessage(ack.orderId, ack.msg())
                            }
                        }
                    }
                }
            }
            SupportedDocument.PAIN_002 -> {
                val status = parseCustomerPaymentStatusReport(xml)
                val msg = status.msg()
                logger.debug("{}", status)
                if (status.paymentCode == ExternalPaymentGroupStatusCode.RJCT) {
                    db.initiated.bankFailure(status.msgId, msg)
                    logger.error("Transaction '${status.msgId}' was rejected : $msg")
                } else {
                    db.initiated.bankMessage(status.msgId, msg)
                }
            }
        }
    }
    
    // Unzip payload if necessary
    when (document) {
        SupportedDocument.PAIN_002,
        SupportedDocument.CAMT_052,
        SupportedDocument.CAMT_053, 
        SupportedDocument.CAMT_054 -> {
            try {
                payload.unzipEach { fileName, xml ->
                    logger.trace("parse $fileName")
                    ingest(xml)
                }
            } catch (e: IOException) {
                throw Exception("Could not open any ZIP archive", e)
            }
        }
        SupportedDocument.PAIN_002_LOGS -> ingest(payload)
    }
}

/** 
 * Fetch and ingest banking records of type [docs] using EBICS [client] starting from [pinnedStart]
 * 
 * If [pinnedStart] is null fetch new records.
 * 
 * Return true if successful 
 */
private suspend fun fetchEbicsDocuments(
    client: EbicsClient,
    docs: List<EbicsDocument>,
    pinnedStart: Instant?,
): Boolean {
    val lastExecutionTime: Instant? = pinnedStart
    return docs.all { doc ->
        try {
            if (lastExecutionTime == null) {
                logger.info("Fetching new '${doc.fullDescription()}'")
            } else {
                logger.info("Fetching '${doc.fullDescription()}' from timestamp: $lastExecutionTime")
            }
            // downloading the content
            val doc = doc.doc()
            val order = client.cfg.dialect.downloadDoc(doc, false)
            client.download(
                order,
                lastExecutionTime,
                null
            ) { payload ->
                ingestPayload(client.db, client.cfg, payload, doc)
            }
            true
        } catch (e: Exception) {
            e.fmtLog(logger)
            false
        }
    }
}

enum class EbicsDocument {
    /// EBICS acknowledgement - CustomerAcknowledgement HAC pain.002
    acknowledgement,
    /// Payment status - CustomerPaymentStatusReport pain.002
    status,
    /// Account intraday reports - BankToCustomerAccountReport camt.052
    report,
    /// Account statements - BankToCustomerStatement camt.053
    statement,
    /// Debit & credit notifications - BankToCustomerDebitCreditNotification camt.054
    notification,
    ;

    fun shortDescription(): String = when (this) {
        acknowledgement -> "EBICS acknowledgement"
        status -> "Payment status"
        report -> "Account intraday reports"
        statement -> "Account statements"
        notification -> "Debit & credit notifications"
    }

    fun fullDescription(): String = when (this) {
        acknowledgement -> "EBICS acknowledgement - CustomerAcknowledgement HAC pain.002"
        status -> "Payment status - CustomerPaymentStatusReport pain.002"
        report -> "Account intraday reports - BankToCustomerAccountReport camt.052"
        statement -> "Account statements - BankToCustomerStatement camt.053"
        notification -> "Debit & credit notifications - BankToCustomerDebitCreditNotification camt.054"
    }

    fun doc(): SupportedDocument = when (this) {
        acknowledgement -> SupportedDocument.PAIN_002_LOGS
        status -> SupportedDocument.PAIN_002
        report -> SupportedDocument.CAMT_052
        statement -> SupportedDocument.CAMT_053
        notification -> SupportedDocument.CAMT_054
    }
}

class EbicsFetch: CliktCommand("Downloads and parse EBICS files from the bank and ingest them into the database") {
    private val common by CommonOption()
    private val transient by transientOption()
    private val documents: Set<EbicsDocument> by argument(
        help = "Which documents should be fetched? If none are specified, all supported documents will be fetched",
        helpTags = EbicsDocument.entries.associate { Pair(it.name, it.shortDescription()) },
    ).enum<EbicsDocument>().multiple().unique()
    private val pinnedStart by option(
        help = "Only supported in --transient mode, this option lets specify the earliest timestamp of the downloaded documents",
        metavar = "YYYY-MM-DD"
    )
    private val ebicsLog by ebicsLogOption()

    override fun run() = cliCmd(logger, common.log) {
        nexusConfig(common.config).withDb { db, nexusCgf ->
            val cfg = nexusCgf.ebics
            val (clientKeys, bankKeys) = expectFullKeys(cfg)
            val client = EbicsClient(
                cfg,
                httpClient(),
                db,
                EbicsLogger(ebicsLog),
                clientKeys,
                bankKeys
            )
            val docs = if (documents.isEmpty()) EbicsDocument.entries else documents.toList()
            if (transient) {
                logger.info("Transient mode: fetching once and returning.")
                val pinnedStartVal = pinnedStart
                val pinnedStartArg = if (pinnedStartVal != null) {
                    logger.debug("Pinning start date to: $pinnedStartVal")
                    dateToInstant(pinnedStartVal)
                } else null
                if (!fetchEbicsDocuments(client, docs, pinnedStartArg)) {
                    throw Exception("Failed to fetch documents")
                }
            } else {
                logger.debug("Running with a frequency of ${cfg.fetch.frequencyRaw}")
                if (cfg.fetch.frequency == Duration.ZERO) {
                    logger.warn("Long-polling not implemented, running therefore in transient mode")
                }
                do {
                    fetchEbicsDocuments(client, docs, null)
                    delay(cfg.fetch.frequency.toKotlinDuration())
                } while (cfg.fetch.frequency != Duration.ZERO)
            }
        }
    }
}
