/* Case and normalization insensitive comparison of Unicode strings.
   Copyright (C) 2009 Free Software Foundation, Inc.
   Written by Bruno Haible <bruno@clisp.org>, 2009.

   This program is free software: you can redistribute it and/or modify it
   under the terms of the GNU Lesser General Public License as published
   by the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.  */

int
FUNC (const SRC_UNIT *s1, size_t n1, const SRC_UNIT *s2, size_t n2,
      const char *iso639_language, uninorm_t nf, int *resultp)
{
  UNIT buf1[2048 / sizeof (UNIT)];
  UNIT buf2[2048 / sizeof (UNIT)];
  UNIT *norms1;
  size_t norms1_length;
  UNIT *norms2;
  size_t norms2_length;
  int cmp;

  /* Optimization: There is no need to do canonical composition of each string.
     Decomposition is enough.  */
  if (nf != NULL)
    nf = uninorm_decomposing_form (nf);

  /* Case-fold and normalize S1.  */
  norms1_length = sizeof (buf1) / sizeof (UNIT);
  norms1 = U_CASEFOLD (s1, n1, iso639_language, nf, buf1, &norms1_length);
  if (norms1 == NULL)
    /* errno is set here.  */
    return -1;

  /* Case-fold and normalize S2.  */
  norms2_length = sizeof (buf2) / sizeof (UNIT);
  norms2 = U_CASEFOLD (s2, n2, iso639_language, nf, buf2, &norms2_length);
  if (norms2 == NULL)
    {
      if (norms1 != buf1)
        {
          int saved_errno = errno;
          free (norms1);
          errno = saved_errno;
        }
      return -1;
    }

  /* Compare the normalized strings.  */
  cmp = U_CMP2 (norms1, norms1_length, norms2, norms2_length);
  if (cmp > 0)
    cmp = 1;
  else if (cmp < 0)
    cmp = -1;

  if (norms2 != buf2)
    free (norms2);
  if (norms1 != buf1)
    free (norms1);
  *resultp = cmp;
  return 0;
}
