/*  gnutrition - a nutrition and diet analysis program.
 *  Copyright(C) 2000, 2001 Edgar Denny(e.denny@ic.ac.uk)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gnome.h>
#include <glade/glade.h>
#include <ctype.h>

#include "text_util.h"
#include "food.h"
#include "recipe.h"
#include "plan.h"
#include "recipe_win.h"
#include "plan_win.h"
#include "edit_food_dlg.h"

static GladeXML *xml = NULL;
static void load_xml( void);
static void connect_signals( void);

/* callbacks */
static void on_ok_button_released( GtkButton *, gpointer);
static void on_cancel_button_released( GtkButton *, gpointer);
static void on_combo_entry_changed( GtkEditable *, gpointer);

static enum { RECIPE_WIN, PLAN_WIN } called_from;

/* load the glade xml if not already loaded. */
static void
load_xml()
{
	static gboolean loaded_xml = FALSE;

	/* load the glade interface. */
	if ( !loaded_xml) {
		xml = glade_xml_new(
			GNUTRITION_GLADEDIR "/edit_food_dlg.glade", NULL);
		loaded_xml = TRUE;

		if ( xml) {
			connect_signals();
		} else {
			g_log( "Gnutrition", G_LOG_LEVEL_ERROR,
				"edit_food_dlg: cannot load glade file\n");
			return;
		}
	}
}

static void
connect_signals()
{
	gtk_signal_connect( GTK_OBJECT(
		glade_xml_get_widget( xml, "ok_button")), "released", 
		GTK_SIGNAL_FUNC( on_ok_button_released), NULL);

	gtk_signal_connect( GTK_OBJECT(
		glade_xml_get_widget( xml, "cancel_button")), "released", 
		GTK_SIGNAL_FUNC( on_cancel_button_released), NULL);

	gtk_signal_connect( GTK_OBJECT(
		glade_xml_get_widget( xml, "combo_entry")), "changed", 
		GTK_SIGNAL_FUNC( on_combo_entry_changed), NULL);
}

/* reopens the search dialog window, so an ingredient can be edited,
 * when the edit button is released in the AppBar on the recipe window. */
void
gnutr_show_edit_food_dlg( char *food_no, 
                          char *msre_desc, 
                          char *amount, 
                          int from)
{
	GList *msre_list;
	char *food_desc;
	GHashTable *htbl;
	GtkWidget *widget;

	g_return_if_fail( food_no);
	g_return_if_fail( msre_desc);
	g_return_if_fail( amount);
	g_return_if_fail( from != 0 || from != 1);

	if (!xml) load_xml();

	htbl = get_htbl_fd_no_fd_desc();
	food_desc = (char *)g_hash_table_lookup( htbl, food_no);

	/* store where the dialog was called from. This is needed for the
	 * on_ok_button_released callback to determine what data needs to
	 * be edited. */
	called_from = from;

	/* create a list of measures for the selected food, put in
	 * the combo box. */
	msre_list = gnutr_get_fd_no_msre_desc_list( food_no);
	widget = glade_xml_get_widget( xml, "combo");
	gtk_combo_set_popdown_strings( GTK_COMBO( widget), msre_list);

	/* set the text entries. */
	widget =  glade_xml_get_widget( xml, "food_entry");
	gtk_entry_set_text(GTK_ENTRY( widget), food_desc);

	widget = glade_xml_get_widget( xml, "combo_entry");
	gtk_entry_set_text(GTK_ENTRY( widget), msre_desc);

	widget = glade_xml_get_widget( xml, "amount_entry");
	gtk_entry_set_text(GTK_ENTRY( widget), amount);

	/* show the dialog. */
	widget = glade_xml_get_widget( xml, "gnutr_edit_food_dialog");
	gnome_dialog_close_hides( GNOME_DIALOG( widget), TRUE);
	gtk_widget_show( widget);
}

/* The callback when the text in the measure combo entry is changed. */
static void
on_combo_entry_changed( GtkEditable *editable,
                        gpointer     user_data)
{
	/* reset the amount to one. */
	gtk_entry_set_text( GTK_ENTRY( glade_xml_get_widget(
			xml, "amount_entry")), "1.00");
}

/* The callback when the "Cancel" button is pressed in the edit food dialog. */
static void
on_cancel_button_released( GtkButton *button,
                           gpointer   user_data)
{	
	gtk_widget_hide( glade_xml_get_widget( xml, "gnutr_edit_food_dialog"));
}

/* The callback when the "OK" button is pressed in the edit food dialog. */
static void
on_ok_button_released( GtkButton *button,
                       gpointer   user_data)
{	
	char *msre_desc, *amount;
	GtkWidget *widget;

	/* get any modified entries. */
	widget = glade_xml_get_widget( xml, "amount_entry");
	amount = gtk_entry_get_text( GTK_ENTRY( widget));
	widget = glade_xml_get_widget( xml, "combo_entry");
	msre_desc = gtk_entry_get_text( GTK_ENTRY( widget));

	if (called_from == RECIPE_WIN) {
		int row;
		GtkCList *clist;
		
		row = gnutr_get_recipe_win_selected_row();
		clist = gnutr_get_recipe_win_clist();

		gtk_clist_set_text( clist, row, 0, amount);
		gtk_clist_set_text( clist, row, 1, msre_desc);
	} else {
		/* called_from == PLAN_WIN */
		char *date = gnutr_get_date();
		char *time_of_day = gnutr_get_time_of_day();
		int row;
		GtkCList *clist;
		enum { FOOD_ROW, RECIPE_ROW } table_type = FOOD_ROW;
		char *hidden_data;

		row = gnutr_get_plan_selected_row();
		clist = gnutr_get_plan_win_clist();

		gtk_clist_set_text( clist, row, 1, amount);
		gtk_clist_set_text( clist, row, 2, msre_desc);

		hidden_data = (char *)gtk_clist_get_row_data( clist, row);
		/* change the temporary plan database table. */
		gnutr_edit_plan_temp_db( date, time_of_day, table_type, 
			amount, msre_desc, NULL, hidden_data);
	}

	gtk_widget_hide( glade_xml_get_widget( xml, "gnutr_edit_food_dialog"));
}
