/*
    GNUbik -- A 3 dimensional magic cube game.
    Copyright (C) 1998, 2003  John Darrington

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
*/
static const char RCSID[]="$Id: glarea-athena.c,v 1.5 2003/10/26 04:32:51 jmd Exp $";


#include <time.h>
#include <getopt.h>
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "drwBlock.h"
#include "blocks.h"
#include "select.h"
#include "gnubik.h"
#include "version.h"
#include "menus.h"
#include "ui.h"
#include "cursors.h"
#include "glarea.h"

#include <X11/keysym.h>
#include <X11/cursorfont.h>

#include <X11/Intrinsic.h>
#include <X11/Shell.h>
#include <X11/StringDefs.h>

#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glx.h>

#include <GL/GLwDrawA.h>

#include <X11/Xutil.h>

#include <X11/Xmu/Editres.h>


#include <X11/Xaw/Form.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Command.h>

void display(Widget glarea);
void set_icon(Widget shell_widget);

void setCubeColours(char *progname, Colormap cmap);



int number_of_blocks;


void projection_init(int jitter) ;
void modelViewInit(void);


void resize (Widget w, XtPointer clientData, XtPointer call);

void cube_controls(Widget w, XtPointer clientData, XtPointer callData);

void buttons(Widget w, XtPointer clientData, XtPointer callData) ;


void graphicsInit(Widget w, XtPointer clientData, XtPointer call);

void expose(Widget w, XtPointer clientData, XtPointer call);

void detect_motion (Widget w,XEvent *event,String *params,Cardinal *num_params);

static void motion (Widget w,
			   XEvent *event,
			   String *params,Cardinal *num_params);


static void 
orientate_the_cube (Widget w,
			   XEvent *event,
		    String *params,Cardinal *num_params);


static Display *dpy;
static GLXContext glxcontext;

static Widget glwidget;

XtAppContext app;

extern Widget main_application_window;




Widget
create_gl_area(Widget containerWidget, Widget adjacentWidget)
{

  static Widget glxarea;

  Colormap cmap;

  static int attribs[] = { GLX_RGBA, 
			   GLX_DEPTH_SIZE, 12, 
			   GLX_DOUBLEBUFFER,
			   GLX_RED_SIZE, 1, 
			   GLX_ACCUM_RED_SIZE, 1, 
			   GLX_ACCUM_GREEN_SIZE, 1, 
			   GLX_ACCUM_BLUE_SIZE, 1, 
			   GLX_ACCUM_ALPHA_SIZE, 1, 
			   None };

  XVisualInfo  *visinfo;


  dpy = XtDisplay(main_application_window);

  cmap = DefaultColormap(dpy,DefaultScreen(dpy));
	
  /* select an appropriate visual */
  visinfo = glXChooseVisual(dpy, DefaultScreen(dpy), attribs);

  if ( visinfo == NULL)
    XtAppError(app,"No suitable visual available");


  /* Create the Mesa/OpenGL rendering area */
  glxarea = XtVaCreateManagedWidget("glxarea",
				    glwDrawingAreaWidgetClass, containerWidget,
				    GLwNvisualInfo, visinfo,
				    XtNcolormap, cmap,
				    NULL);



  /* set contraints on drawing area */
  XtVaSetValues(glxarea,
		XtNresizable, True,
		XtNfromHoriz, (XtArgVal) NULL ,
		XtNfromVert, adjacentWidget,
     		XtNtop, XtChainTop,
		XtNbottom, XtChainBottom ,
		XtNleft, XtChainLeft ,
		XtNright, XtChainRight,
		NULL);


  glwidget = glxarea;

  return glxarea ;
}


void
register_gl_callbacks(Widget glxarea)
{

  /* register all the drawing area callbacks */

  String translations="<Motion>: motion()\n";
  XtActionsRec actionTable[] = {{"motion", motion}};
  XtAppAddActions (app,actionTable,1);
 
  XtOverrideTranslations (glxarea, XtParseTranslationTable(translations));


  XtAddCallback(glxarea, GLwNginitCallback, graphicsInit,glxarea);
  XtAddCallback(glxarea, GLwNexposeCallback, expose,glxarea);
  XtAddCallback(glxarea, GLwNresizeCallback, resize,NULL);

  XtAddCallback(glxarea, GLwNinputCallback, buttons,NULL);

  XtAddCallback(glxarea, GLwNinputCallback, cube_controls,NULL);

}



/* Expose callback.  Just redraw the scene */
void 
expose(Widget w, XtPointer clientData, XtPointer call)
{
  Widget glxarea = (Widget) clientData;

  postRedisplay(glxarea);
}




/* init callback.  Set everything up for the first time. */
void
graphicsInit(Widget w, XtPointer clientData, XtPointer call)
{
  XVisualInfo *visinfo ;
  Widget glxarea = clientData;

  /* Set the Cursor Shape */
  XDefineCursor(dpy,XtWindow(glxarea),XCreateFontCursor(dpy, XC_crosshair));


  /* Create Mesa/OpenGL rendering Context */
  XtVaGetValues(w, GLwNvisualInfo, &visinfo, NULL);
  glxcontext = glXCreateContext(XtDisplay(w),visinfo,0,True);
	

  glXMakeCurrent(XtDisplay(w),XtWindow(w),glxcontext);

  lighting_init();
  projection_init(0);
  modelViewInit();

  resize(w,clientData, call);
  ERR_CHECK("Error in init");

}


static int button_down = 0;


/* Deal with mouse button presses */
void
buttons(Widget w, XtPointer clientData, XtPointer callData)
{

  

  GLwDrawingAreaCallbackStruct *cd = (GLwDrawingAreaCallbackStruct *) callData;


  XButtonEvent *button_event = (XButtonEvent *)cd->event ;


  switch (button_event->button) { 
  case 1: 

    if  (cd->event->type  == ButtonPress ) {
      button_down = 1;
      disableSelection();
    }

    if  (cd->event->type  == ButtonRelease ) {
      button_down = 0;
      enableSelection();
    }

    break ;
    
    /* Buttons 4 & 5 mean the mouse wheel */
  case 4:
    rotate_cube(2,1);
    break ;
  case 5:
    rotate_cube(2,0);    
    break ;
  }

}




static void 
motion (Widget w,
	XEvent *event,
	String *params,Cardinal *num_params)
{
  orientate_the_cube(w,event,params,num_params);

  /* This is needed to support the selection mechanism */
  detect_motion(w,event,params,num_params);
}


/* This callback allows the cube to be rotated using the mouse */
static void 
orientate_the_cube (Widget w,
			   XEvent *event,
			   String *params,Cardinal *num_params)
{

  static double last_mouse_x=-1; 
  static double last_mouse_y=-1;
  
  int xmotion = 0;
  int ymotion = 0;

  XMotionEvent *xme = (XMotionEvent *) event;

  if ( itemIsSelected() ) 
    return ;

  if ( ! button_down) 
    return ;

  if ( event->type != MotionNotify )
    return;

  if ( last_mouse_x >= 0 )
    xmotion = xme->x - last_mouse_x ;

  if ( last_mouse_y >= 0 )
    ymotion = xme->y - last_mouse_y ;
    

  last_mouse_x = xme->x ; 
  last_mouse_y = xme->y;

  if ( ymotion > 0 ) 
    rotate_cube(0,1);
  if ( ymotion < 0 ) 
    rotate_cube(0,0);

  if ( xmotion > 0 ) 
    rotate_cube(1,1);
  if ( xmotion < 0 ) 
    rotate_cube(1,0);

}




/* Input callback.  Despatch input events to approprite handlers */
void
cube_controls(Widget w, XtPointer clientData, XtPointer callData)
{
  GLwDrawingAreaCallbackStruct *cd = (GLwDrawingAreaCallbackStruct *) callData;

  char buffer[1];
  KeySym keysym;



  switch (cd->event->type) {
  case KeyPress:
    {
      int shifted=0;
      XKeyEvent *ke = (XKeyEvent *) cd->event;
      XLookupString(ke, buffer, 1 , &keysym, NULL );

      if ( ke->state & ShiftMask) 
	shifted=1;

      switch(keysym) {
      case XK_Right:
      case XK_Left:
      case XK_Up:
      case XK_Down:
	arrows(keysym,shifted);
	break;
      default:
	/* Ignore everything except arrow keys */
	break;
      }
      break;
    }
  case ButtonPress:
    {
    XButtonEvent *event = (XButtonEvent *)cd->event ;
    /* process mouse events */
    mouse( event->button ) ;
    }
    break;
  case ButtonRelease:
  default:
    break;
  }
}



/* Resize callback.  */
void 
resize (Widget w, XtPointer clientData, XtPointer call)
{
  GLint min_dim ;
  GLwDrawingAreaCallbackStruct *callData ;
	
  callData = (GLwDrawingAreaCallbackStruct *) call;
  glXMakeCurrent(XtDisplay(w), XtWindow(w), glxcontext);
  glXWaitX();
  min_dim = (callData->width < callData->height ) ? 
    callData->width : callData->height ;

  /* Ensure that cube is always the same proportions */
  glViewport((callData->width-min_dim) /2, 
	     (callData->height-min_dim) /2, min_dim, min_dim);
}




extern float cursorAngle;

void
set_mouse_cursor(Widget glxarea)
{

    Cursor cursor;    
    
    XColor fg = { 0, 65535, 65535, 65535 }; /* White. */
    XColor bg = { 0, 0, 0, 0 }; /* Black. */

    if ( itemIsSelected() ) { 
      int hot_x, hot_y;
      int width, height;

      unsigned char *mask_bits;
      unsigned char *data_bits;

      Pixmap source, mask;

      get_cursor(cursorAngle, &data_bits, &mask_bits, &height, &width,
		 &hot_x, &hot_y);

      source = XCreateBitmapFromData(dpy,XtWindow(glxarea), 
				     (char *) data_bits,
				     width,height);
      mask = XCreateBitmapFromData(dpy,XtWindow(glxarea), 
				   (char *) mask_bits,
				   width, height);

		
      cursor = XCreatePixmapCursor(dpy,source, mask, &fg, &bg, hot_x, hot_y);

      XFreePixmap(dpy,source);
      XFreePixmap(dpy,mask);
    }
    else { 
      cursor = XCreateFontCursor(dpy,XC_crosshair -1);
    }

    XDefineCursor(dpy,XtWindow(glxarea),cursor);
    XFreeCursor(dpy,cursor);

  }

extern int animation_in_progress;

/* Reset the bit planes, and render the scene */
void
display(Widget glxarea)
{
  int jitter;
  int samples;
  projection_init(0);
  modelViewInit();

  set_mouse_cursor(glxarea);

  samples = 8;

  glXMakeCurrent(dpy,XtWindow(glxarea), glxcontext);

  glClear(GL_ACCUM_BUFFER_BIT);

  for( jitter = 0 ; jitter < samples ; ++jitter ) { 
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
    
    projection_init(jitter);
    modelViewInit();

    ERR_CHECK("Error in display");
    drawCube();

    ERR_CHECK("Error in display");
    glAccum(GL_ACCUM,1.0/samples);
  }
  glAccum(GL_RETURN,1.0);
  glXSwapBuffers(dpy,XtWindow(glxarea));

  if ( !glXIsDirect(dpy, glxcontext) )
    glFinish();
}






/* Error string display.  This is always called by a macro
wrapper, to set the file and line_no opts parameters */
void
error_check(const char *file, int line_no, const char *string)
{
  GLenum err_state;
  if ( GL_NO_ERROR != (err_state = glGetError())  )
    fprintf(stderr,"%s:%d %s:  %s\n",file,line_no ,string, 
	   gluErrorString(err_state) );
}






/* Pair of mutually co-operating funcs to handle redisplays,
avoiding unnecessary overhead.  For example when the window
is covered by another one */

static int redisplayPending = 0 ;

Boolean
handleRedisplay(XtPointer glxarea)
{

  display((Widget)glxarea);
  redisplayPending = 0;

  return True ; /* Remove this work proc */
}

void
postRedisplay()
{

  if ( !redisplayPending ) {
    XtAppAddWorkProc(app, handleRedisplay, glwidget );
    redisplayPending = 1;
  }
}

#ifdef HAVE_LIBXPM
#include <X11/xpm.h>
#include "gnubik.xpm"
#endif

/* Set the icon for the program */
void 
set_icon(Widget shell_widget)
{

  Arg arg[1];
  Pixmap icon;

#ifdef HAVE_LIBXPM
  static  XpmColorSymbol none_color = { NULL, "None", (Pixel)0 };

  Display *dpy = XtDisplay(shell_widget);
  Window win = DefaultRootWindow(dpy);


  XpmAttributes xpm_attr;

  xpm_attr.valuemask = XpmReturnPixels|XpmColorSymbols|XpmSize;

  xpm_attr.colorsymbols = &none_color;
  xpm_attr.numsymbols = 1;


  XpmCreatePixmapFromData(dpy, win, (const char *) gnubik_xpm, &icon,NULL, &xpm_attr);
#else

#define icon_width 50
#define icon_height 50
  static const unsigned char icon_bits[] = {
    0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
    0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xaf,0xbf,
    0xaa,0xfe,0xff,0xff,0xff,0xff,0x48,0xfd,0x2d,0xc0,0xff,0xff,0xef,0xfe,0xaf,
    0xfb,0x48,0xfc,0xff,0x7f,0x69,0x75,0x95,0x03,0xf0,0xff,0x6f,0x5f,0xd5,0xaa,
    0x0e,0xc0,0xff,0x77,0xee,0xff,0xff,0xff,0xff,0xff,0xef,0xba,0xfa,0x20,0xd8,
    0xad,0xff,0x77,0xee,0xef,0x8a,0x3a,0x55,0xff,0xef,0xbe,0xfa,0x44,0xd8,0xaa,
    0xff,0x77,0xee,0xef,0x29,0x5d,0x55,0xff,0xef,0x76,0xfb,0x80,0xb8,0xaa,0xff,
    0x77,0xde,0x6e,0x2a,0x5a,0x55,0xff,0x7f,0xf6,0x7b,0x01,0xa9,0xaa,0xff,0xe7,
    0xbe,0xee,0x54,0x5a,0x55,0xff,0x67,0xea,0x7b,0x89,0xa8,0xaa,0xff,0xaf,0x7e,
    0x6f,0x42,0x5c,0xd5,0xff,0xc7,0xae,0x7b,0x28,0xaa,0xaa,0xff,0x6f,0xfb,0x6e,
    0x05,0xd9,0xd6,0xff,0x47,0xff,0xff,0xff,0xff,0xff,0xff,0x77,0xfe,0x7f,0xa8,
    0xac,0xaa,0xff,0x77,0xfe,0xff,0x02,0x5e,0xd5,0xff,0x5f,0xfe,0x7f,0xa8,0xac,
    0xaa,0xff,0x3f,0xfe,0xff,0x12,0x56,0xd5,0xff,0x1f,0xff,0x7f,0xa1,0xac,0xaa,
    0xff,0x37,0xff,0x7f,0x0a,0xb5,0xea,0xff,0x1f,0xff,0xbf,0x40,0x54,0xd5,0xff,
    0x2f,0xff,0x3f,0x94,0xae,0xea,0xff,0x2f,0xef,0xbf,0x22,0xac,0xd5,0xff,0x57,
    0xff,0x3f,0x11,0xad,0xea,0xff,0x6f,0xff,0x7f,0x8a,0x54,0xd5,0xff,0xd7,0xff,
    0x3f,0xfe,0xff,0xff,0xff,0xef,0xff,0xff,0xff,0x3f,0xc0,0xff,0xd7,0x89,0x34,
    0x00,0x00,0xc0,0xff,0xdf,0x6b,0x35,0x00,0x02,0xe0,0xff,0xef,0x55,0x35,0x00,
    0x02,0xe0,0xff,0xdf,0xab,0x3a,0x00,0x02,0xe1,0xff,0xff,0x55,0x35,0x44,0x0a,
    0xe0,0xff,0xdf,0xab,0x3a,0x00,0x02,0xe0,0xff,0xff,0xd5,0x7a,0x00,0x02,0xe0,
    0xff,0xff,0x57,0x15,0x00,0x02,0xe1,0xff,0xff,0xab,0x1a,0x02,0x0a,0xe0,0xff,
    0xff,0xaa,0x1a,0x20,0x02,0xe0,0xff,0xff,0x55,0x15,0x00,0x82,0xff,0xff,0xff,
    0xea,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
    0xff,0xff,0xff,0xff,0xff};

  icon = XCreateBitmapFromData(XtDisplay(shell_widget), XtWindow(shell_widget),
			       (char *) icon_bits, icon_width, icon_height);
#endif

  XtSetArg(arg[0],XtNiconPixmap, icon);
  XtSetValues(shell_widget,arg,1);
}




/* this is the number of frames drawn when a slice of the cube rotates 90 degrees */
extern int frameQty;



/* Increase the number of animations in a move */
void
animation_inc(Widget w, XtPointer data, XtPointer callData)
{
        frameQty++;
}




/* Decrease the number of animations in a move */
void
animation_dec(Widget w, XtPointer data, XtPointer callData)
{
        if (frameQty > 0 )
                frameQty-- ;
}


