/* Copyright (C) 1998 Aladdin Enterprises.  All rights reserved.
  
  This file is part of GNU Ghostscript.
  
  GNU Ghostscript is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY.  No author or distributor accepts responsibility
  to anyone for the consequences of using it or for whether it serves any
  particular purpose or works at all, unless he says so in writing.  Refer
  to the GNU General Public License for full details.
  
  Everyone is granted permission to copy, modify and redistribute GNU
  Ghostscript, but only under the conditions described in the GNU General
  Public License.  A copy of this license is supposed to have been given
  to you along with GNU Ghostscript so you can know your rights and
  responsibilities.  It should be in a file named COPYING.  Among other
  things, the copyright notice and this notice must be preserved on all
  copies.
  
  Aladdin Enterprises supports the work of the GNU Project, but is not
  affiliated with the Free Software Foundation or the GNU Project.  GNU
  Ghostscript, as distributed by Aladdin Enterprises, does not require any
  GNU software to build or run it.
*/

/*$Id: gdevbmpc.c $ */
/* .BMP file format driver utilities */
#include "gdevprn.h"
#include "gdevbmp.h"

/*
 * Define BMP file format structures.
 * All multi-byte quantities are stored LSB-first!
 */

typedef ushort word;
#if arch_sizeof_int == 4
typedef uint dword;
#else
#  if arch_sizeof_long == 4
typedef ulong dword;
#  endif
#endif
#if arch_is_big_endian
#  define BMP_ASSIGN_WORD(a,v) a = ((v) >> 8) + ((v) << 8)
#  define BMP_ASSIGN_DWORD(a,v)\
     a = ((v) >> 24) + (((v) >> 8) & 0xff00L) +\
	 (((dword)(v) << 8) & 0xff0000L) + ((dword)(v) << 24)
#else
#  define BMP_ASSIGN_WORD(a,v) a = (v)
#  define BMP_ASSIGN_DWORD(a,v) a = (v)
#endif

typedef struct bmp_file_header_s {

    /* BITMAPFILEHEADER */

    /*
     * This structure actually begins with two bytes
     * containing the characters 'BM', but we must omit them,
     * because some compilers would insert padding to force
     * the size member to a 32- or 64-bit boundary.
     */

    /*byte  typeB, typeM; *//* always 'BM' */
    dword size;			/* total size of file */
    word reserved1;
    word reserved2;
    dword offBits;		/* offset of bits from start of file */

} bmp_file_header;

#define sizeof_bmp_file_header (2 + sizeof(bmp_file_header))

typedef struct bmp_info_header_s {

    /* BITMAPINFOHEADER */

    dword size;			/* size of info header in bytes */
    dword width;		/* width in pixels */
    dword height;		/* height in pixels */
    word planes;		/* # of planes, always 1 */
    word bitCount;		/* bits per pixel */
    dword compression;		/* compression scheme, always 0 */
    dword sizeImage;		/* size of bits */
    dword xPelsPerMeter;	/* X pixels per meter */
    dword yPelsPerMeter;	/* Y pixels per meter */
    dword clrUsed;		/* # of colors used */
    dword clrImportant;		/* # of important colors */

    /* This is followed by (1 << bitCount) bmp_quad structures, */
    /* unless bitCount == 24. */

} bmp_info_header;

typedef struct bmp_quad_s {

    /* RGBQUAD */

    byte blue, green, red, reserved;

} bmp_quad;

/* Write the BMP file header. */
int
write_bmp_header(gx_device_printer *pdev, FILE *file)
{
    int raster = gdev_prn_raster(pdev);
    /* BMP scan lines are padded to 32 bits. */
    ulong bmp_raster = raster + (-raster & 3);
    int height = pdev->height;
    int depth = pdev->color_info.depth;
    int quads = (depth <= 8 ? sizeof(bmp_quad) << depth : 0);

    /* Write the file header. */

    fputc('B', file);
    fputc('M', file);
    {
	bmp_file_header fhdr;

	BMP_ASSIGN_DWORD(fhdr.size,
		     sizeof_bmp_file_header +
		     sizeof(bmp_info_header) + quads +
		     bmp_raster * height);
	BMP_ASSIGN_WORD(fhdr.reserved1, 0);
	BMP_ASSIGN_WORD(fhdr.reserved2, 0);
	BMP_ASSIGN_DWORD(fhdr.offBits,
		     sizeof_bmp_file_header +
		     sizeof(bmp_info_header) + quads);
	if (fwrite((const char *)&fhdr, 1, sizeof(fhdr), file) != sizeof(fhdr))
	    return_error(gs_error_ioerror);
    }

    /* Write the info header. */

    {
	bmp_info_header ihdr;

	BMP_ASSIGN_DWORD(ihdr.size, sizeof(ihdr));
	BMP_ASSIGN_DWORD(ihdr.width, pdev->width);
	BMP_ASSIGN_DWORD(ihdr.height, height);
	BMP_ASSIGN_WORD(ihdr.planes, 1);
	BMP_ASSIGN_WORD(ihdr.bitCount, depth);
	BMP_ASSIGN_DWORD(ihdr.compression, 0);
	BMP_ASSIGN_DWORD(ihdr.sizeImage, bmp_raster * height);
	/* Even though we could compute the resolution correctly, */
	/* the convention seems to be to leave it unspecified. */
	BMP_ASSIGN_DWORD(ihdr.xPelsPerMeter, 0);
	    /*(dword)(pdev->x_pixels_per_inch * (1000.0 / 30.48))); */
	BMP_ASSIGN_DWORD(ihdr.yPelsPerMeter, 0);
	    /*(dword)(pdev->y_pixels_per_inch * (1000.0 / 30.48))); */
	BMP_ASSIGN_DWORD(ihdr.clrUsed, 0);
	BMP_ASSIGN_DWORD(ihdr.clrImportant, 0);
	if (fwrite((const char *)&ihdr, 1, sizeof(ihdr), file) != sizeof(ihdr))
	    return_error(gs_error_ioerror);
    }

    /* Write the palette. */

    if (depth <= 8) {
	int i;
	gx_color_value rgb[3];
	bmp_quad q;

	q.reserved = 0;
	for (i = 0; i != 1 << depth; i++) {
	    (*dev_proc(pdev, map_color_rgb))((gx_device *)pdev,
					     (gx_color_index)i, rgb);
	    q.red = gx_color_value_to_byte(rgb[0]);
	    q.green = gx_color_value_to_byte(rgb[1]);
	    q.blue = gx_color_value_to_byte(rgb[2]);
	    fwrite((const char *)&q, sizeof(q), 1, file);
	}
    }

    return 0;
}

/* 24-bit color mappers (taken from gdevmem2.c). */
/* Note that Windows expects RGB values in the order B,G,R. */

/* Map a r-g-b color to a color index. */
gx_color_index
bmp_map_16m_rgb_color(gx_device * dev, gx_color_value r, gx_color_value g,
		  gx_color_value b)
{
    return gx_color_value_to_byte(r) +
	((uint) gx_color_value_to_byte(g) << 8) +
	((ulong) gx_color_value_to_byte(b) << 16);
}

/* Map a color index to a r-g-b color. */
int
bmp_map_16m_color_rgb(gx_device * dev, gx_color_index color,
		  gx_color_value prgb[3])
{
    prgb[2] = gx_color_value_from_byte(color >> 16);
    prgb[1] = gx_color_value_from_byte((color >> 8) & 0xff);
    prgb[0] = gx_color_value_from_byte(color & 0xff);
    return 0;
}
