/*
 * Electric(tm) VLSI Design System
 *
 * File: iocifout.c
 * Input/output aid: CIF output
 * Written by: Steven M. Rubin, Static Free Software
 *
 * Copyright (c) 2000 Static Free Software.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Static Free Software
 * 4119 Alpine Road
 * Portola Valley, California 94028
 * info@staticfreesoft.com
 */

#include "config.h"
#if IOCIF

#include "global.h"
#include "eio.h"
#include "egraphics.h"
#include "tecgen.h"

static char       io_curlay[100];
static INTBIG     io_prev_chr_sep, io_check_sum, io_nchar;	/* for checksum */
static INTSML     io_cif_resolution;		/* for resolution tests */
static INTBIG     io_cif_polypoints;		/* for points on a polygon tests */
static INTBIG     outputstate;				/* current output state */
static INTBIG     coffx, coffy;				/* offset for current facet - for highlight */
static NODEPROTO *curnp;					/* node proto of facet being written */
static INTSML     io_zero_offset;			/* Set if there is no centering */
static INTSML     io_reserr_count;			/* count the resolution errors */

INTSML io_cifwritefacet(NODEPROTO*);
void io_outputcifpoly(TECHNOLOGY*, POLYGON*, INTBIG, INTBIG, INTBIG, INTBIG, XARRAY, GEOM*);
void io_currentciflayer(char*);
void io_cifprintf(char *msg, ...);
void io_cifreserror(GEOM*, INTBIG, INTBIG, INTBIG, INTBIG);
void io_cif_write_polygon(INTSML, TECHNOLOGY*, INTBIG*, INTBIG*, INTSML, float);
void io_cif_show_reserror(INTBIG, INTBIG, INTBIG, INTBIG);
INTBIG io_cif_outscale(INTBIG value);

INTSML io_writeciflibrary(LIBRARY *lib)
{
	char file[100], *truename;
	REGISTER char *name;
	REGISTER NODEPROTO *np;
	REGISTER VARIABLE *var;
	REGISTER TECHNOLOGY *tech;
	extern AIDENTRY *us_aid;

	/* create the proper disk file for the CIF */
	if (lib->curnodeproto == NONODEPROTO)
	{
		ttyputerr("Must be editing a facet to generate CIF output");
		return(1);
	}
	(void)strcpy(file, lib->curnodeproto->cell->cellname);
	if (strcmp(&file[strlen(file)-4],".cif") != 0) (void)strcat(file, ".cif");
	name = truepath(file);
	io_fileout = xcreate(name, FILETYPECIF, "CIF File", &truename);
	if (io_fileout == NULL)
	{
		if (truename != 0) ttyputerr("Cannot write %s", truename);
		return(1);
	}

	/* get current output state */
	var = getvalkey((INTBIG)io_aid, VAID, VINTEGER, io_state);
	if (var == NOVARIABLE) outputstate = 0; else
		outputstate = var->addr;

	/* get form of output */
	if ((outputstate&CIFOUTNOTCEN) != 0) io_zero_offset = 1; else
		io_zero_offset = 0;

	/* get cif resolution */
	var = getval((INTBIG)el_curtech, VTECHNOLOGY, VINTEGER, "IO_cif_resolution");
	if (var == NOVARIABLE) io_cif_resolution = 0; else
	{
		io_cif_resolution = (INTSML)var->addr;
		if (io_cif_resolution != 0)
		{
			ttyputmsg("Resolution errors will be highlighted");
			(void)askaid(us_aid, "clear");
		}
	}

	/* get the number of allowable points on a polygon */
	var = getval((INTBIG)el_curtech, VTECHNOLOGY, VINTEGER, "IO_cif_polypoints");
	if (var == NOVARIABLE) io_cif_polypoints = HUGEINT; else
		io_cif_polypoints = var->addr;

	/* initialize cif merging facility if needed */
	if ((outputstate&CIFOUTMERGE) != 0) mrginit();

	/* initialize the checksum accumulation */
	io_prev_chr_sep = 1;
	io_check_sum = 32;
	io_nchar = 1;

	/* initialize cache of CIF layer information */
	for(tech = el_technologies; tech != NOTECHNOLOGY; tech = tech->nexttechnology)
	{
		var = getval((INTBIG)tech, VTECHNOLOGY, VSTRING|VISARRAY, "IO_cif_layer_names");
		if (var == NOVARIABLE) tech->temp1 = 0; else
		{
			tech->temp1 = var->addr;
			if (getlength(var) != tech->layercount)
			{
				ttyputerr("Warning: CIF layer information is bad for technology %s.  Use 'CIF Options' to fix it",
					tech->techname);
				tech->temp1 = 0;
			}
		}

		var = getval((INTBIG)tech, VTECHNOLOGY, VSTRING|VISARRAY, "TECH_layer_names");
		tech->temp2 = (var == NOVARIABLE ? 0 : var->addr);

		/* if ignoring DRC mask layer, delete Generic technology layer names */
		if (tech == gen_tech && (outputstate&CIFOUTADDDRC) == 0) tech->temp1 = 0;
	}

	/* write the CIF */
	io_reserr_count = 0;
	io_cifbase = 100;
	for(np = el_curlib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		np->temp1 = 0;
	io_cifwritefacet(lib->curnodeproto);

	/* clean up */
	if ((outputstate&CIFOUTNOTOPCALL) == 0)
		io_cifprintf("C %d;\n", io_cifbase);
	io_cifprintf("E\n");
	xclose(io_fileout);

	/* if doing polygon output */
	if ((outputstate&CIFOUTMERGE) != 0) mrgterm();

	/* tell the user that the file is written */
	ttyputmsg("%s written", truename);

	if (io_cif_resolution != 0)
		if (io_reserr_count != 0)
	{
		ttyputmsg("This file contains %d resolution errors", io_reserr_count);
	}

	/* complete the checksum accumulation */
	if (!io_prev_chr_sep)
	{
		io_check_sum += 32;
		io_nchar++;
	}
	ttyputverbose("(CIF checksum=%ld, nchar=%ld)", io_check_sum, io_nchar);

	return(0);
}

INTSML io_cifwritefacet(NODEPROTO *np)
{
	REGISTER NODEINST *subno;
	REGISTER ARCINST *subar;
	REGISTER NODEPROTO *subnt, *bodysubnt, *onp;
	REGISTER PORTPROTO *pp;
	REGISTER INTSML i, j;
	REGISTER INTBIG offx, offy;
	XARRAY trans;
	INTBIG rx, ry, xpos, ypos, bx, by, cx, cy;
	static POLYGON *poly = NOPOLYGON;

	/* stop if requested */
	if (stopping("CIF Output") != 0) return(1);

	/* if there are any sub-facets that have not been written, write them */
	for(subno = np->firstnodeinst; subno != NONODEINST; subno = subno->nextnodeinst)
	{
		subnt = subno->proto;
		if (subnt->primindex != 0) continue;
		if ((subno->userbits & NEXPAND) == 0 && (outputstate&CIFOUTEXACT) != 0) continue;

		/* convert body facets to contents facets */
		onp = contentsview(subnt);
		if (onp != NONODEPROTO) subnt = onp;

		/* don't recurse if this facet has already been written */
		if (subnt->temp1 != 0) continue;

		/* recurse to the bottom */
		io_cifwritefacet(subnt);
	}

	curnp = np;			/* set global for highlighting */

	/* get polygon */
	if (poly == NOPOLYGON) poly = allocstaticpolygon(4, io_aid->cluster);

	/* write this facet */
	(void)strcpy(io_curlay, "");
	np->temp1 = ++io_cifbase;
	if (io_zero_offset != 0)
	{
		offx = coffx = np->lowx;
		offy = coffy = np->lowy;
	} else
	{
		offx = coffx = (np->lowx + np->highx) / 2;
		offy = coffy = (np->lowy + np->highy) / 2;
	}
	io_cifprintf("DS %d 1 1;\n", io_cifbase);
	io_cifprintf("9 %s;\n", np->cell->cellname);

	/* write all nodes in the facet */
	for(subno = np->firstnodeinst; subno != NONODEINST; subno = subno->nextnodeinst)
	{
		subnt = subno->proto;
		i = subno->rotation;
		if (subno->transpose != 0) i = (i + 900) % 3600;
		rx = ((cosine(i)>>14) * 100) >> 16;   ry = ((sine(i)>>14) * 100) >> 16;
		makerot(subno, trans);
		if (subnt->primindex != 0)
		{
			/* don't draw anything if the node is wiped out */
			if ((subno->userbits&WIPED) != 0) continue;

			/* write a primitive nodeinst */
			i = nodepolys(subno);
			for(j=0; j<i; j++)
			{
				shapenodepoly(subno, j, poly);
				io_outputcifpoly(subnt->tech, poly, rx,
					(subno->transpose ? -ry : ry), offx, offy, trans, subno->geom);
			}
		} else if ((subno->userbits&NEXPAND) != 0 || (outputstate&CIFOUTEXACT) == 0)
		{
			/* write a call to a facet */
			if (io_zero_offset != 0)
			{
				/*
				 * find where the lower corner has been moved to,
				 * by moving the origin back to the center, transforming the lower
				 * corner, then converting back to lower left origin
				 */
				trans[2][0] = (subno->highx - subno->lowx); /* avoid roundoff */
				trans[2][1] = (subno->highy - subno->lowy);
				xform(-trans[2][0], -trans[2][1], &xpos, &ypos, trans);
				xpos /= 2; ypos /= 2;   /* now contains the correction */
				xpos += (subno->lowx - offx);
				ypos += (subno->lowy - offy);
			} else
			{
				xpos = (subno->lowx + subno->highx)/2 - offx;
				ypos = (subno->lowy + subno->highy)/2 - offy;
			}

			/* convert body facets to contents facets */
			onp = contentsview(subnt);
			if (onp != NONODEPROTO)
			{
				/* look for grab points in contents and body facets */
				bodysubnt = subnt;
				subnt = onp;

				corneroffset(subno, bodysubnt, subno->rotation, subno->transpose, &bx, &by, 0);
				corneroffset(NONODEINST, subnt, subno->rotation, subno->transpose, &cx, &cy, 0);

				if (io_zero_offset != 0)
				{
					/* This code may be incorrect */
					trans[2][0] = (subno->highx - subno->lowx); /* avoid roundoff */
					trans[2][1] = (subno->highy - subno->lowy);
					xform(-trans[2][0], -trans[2][1], &xpos, &ypos, trans);
					xpos /= 2; ypos /= 2;   /* now contains the correction */
					xpos += subno->lowx + bx - cx - offx;
					ypos += subno->lowy + by - cy - offy;
				} else
				{
					xpos = subno->lowx + bx - cx + (subnt->highx-subnt->lowx)/2 - offx;
					ypos = subno->lowy + by - cy + (subnt->highy-subnt->lowy)/2 - offy;
				}
			}

			/* resolution problem with call */
			if (io_cif_resolution != 0)
			{
				if (io_zero_offset != 0) /* Why not use xpos and ypos here ?*/
				{
					if((xpos%io_cif_resolution) != 0 || (ypos%io_cif_resolution) != 0)
					{
						io_cifprintf("(Call to symbol %ld does not resolve to grid);\n",
							subnt->temp1);
						io_reserr_count++;
					}
				} else
				{
					if ((((subno->lowx+subno->highx)/2-offx)%io_cif_resolution) != 0 ||
						(((subno->lowy+subno->highy)/2-offy)%io_cif_resolution) != 0)
					{
						io_cifprintf("(Call to symbol %ld does not resolve to grid);\n",
							subnt->temp1);
						io_reserr_count++;
					}
				}
			}
			io_cifprintf("C %ld R %ld %ld", subnt->temp1, rx, ry);
			if (subno->transpose != 0) io_cifprintf(" M Y");
			io_cifprintf(" T %ld %ld;\n", io_cif_outscale(xpos), io_cif_outscale(ypos));
		} else
		{
			/* write the vectors that describe an unexpanded facet */
			io_cifprintf("0V %ld %ld %ld %ld %ld %ld %ld %ld %ld %ld;\n",
				io_cif_outscale(subno->lowx-offx), io_cif_outscale(subno->lowy-offy),
					io_cif_outscale(subno->lowx-offx), io_cif_outscale(subno->highy-offy),
						io_cif_outscale(subno->highx-offx), io_cif_outscale(subno->highy-offy),
							io_cif_outscale(subno->highx-offx), io_cif_outscale(subno->lowy-offy),
								io_cif_outscale(subno->lowx-offx), io_cif_outscale(subno->lowy-offy));
			io_cifprintf("2C \"%s", describenodeproto(subnt));
			if (io_zero_offset != 0)
				io_cifprintf("\" T %ld %ld;\n", io_cif_outscale(subno->lowx - offx),
					io_cif_outscale(subno->lowy - offy)); else
						io_cifprintf("\" T %ld %ld;\n",
							io_cif_outscale((subno->lowx+subno->highx)/2-offx),
								io_cif_outscale((subno->lowy+subno->highy)/2-offy));
		}
	}

	for(subar=np->firstarcinst; subar!=NOARCINST; subar=subar->nextarcinst)
	{
		/* arcinst: ask the technology how to draw it */
		i = arcpolys(subar);

		/* plot each layer of the arcinst */
		for(j=0; j<i; j++)
		{
			/* write the box describing the layer */
			shapearcpoly(subar, j, poly);
			io_outputcifpoly(subar->proto->tech, poly, 1, 0, offx, offy, el_matid, subar->geom);
		}
	}

	/* write the ports as labels */
	for(pp = np->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
	{
		portposition(pp->subnodeinst, pp->subportproto, &xpos, &ypos);
		io_cifprintf("94 %s %ld %ld;\n", pp->protoname, io_cif_outscale(xpos-offx),
			io_cif_outscale(ypos-offy));
	}

	if ((outputstate&CIFOUTMERGE) != 0)
		mrgdonefacet(io_cif_write_polygon);

	io_cifprintf("DF;\n");

	return(0);
}

/*
 * routine to write polygon "poly" to the CIF file.  The polygon is from
 * technology "tech", is rotated by the factor "rx" and "ry", and is offset
 * by "offx" and "offy"
 */
void io_outputcifpoly(TECHNOLOGY *tech, POLYGON *poly, INTBIG rx, INTBIG ry,
	INTBIG offx, INTBIG offy, XARRAY trans, GEOM *geom)
{
	REGISTER INTBIG r, k, bloat;
	INTBIG xl, xh, yl, yh, xpos, ypos;
	REGISTER char *lay;

	/* determine the CIF layer name for this polygon */
	if (poly->layer < 0 || tech->temp1 == 0) lay = ""; else
		lay = ((char **)tech->temp1)[poly->layer];

	/* write polygons with valid CIF layers */
	if (lay[0] != 0)
	{
		/* get bloating for this layer */
		if (tech->temp2 == 0) bloat = 0; else
		{
			(void)initinfstr();
			(void)addstringtoinfstr(tech->techname);
			(void)addtoinfstr(':');
			(void)addstringtoinfstr(((char **)tech->temp2)[poly->layer]);
			bloat = io_getoutputbloat(returninfstr());
		}
		switch (poly->style)
		{
			case DISC:
				io_currentciflayer(lay);
				xformpoly(poly, trans);
				r = computedistance(poly->xv[0], poly->yv[0], poly->xv[1], poly->yv[1]);
				if (r <= 0) break;
				io_cifprintf(" R %ld %ld %ld;\n", r+bloat, io_cif_outscale(poly->xv[0]-offx),
					io_cif_outscale(poly->yv[0]-offy));
				return;

			default:
				if (isbox(poly, &xl,&xh, &yl,&yh) != 0)
				{
					/* rectangular shape */
					if (xh == xl || yh == yl) return;
					xform((xl+xh)/2, (yl+yh)/2, &xpos, &ypos, trans);
					if ((outputstate&CIFOUTMERGE) != 0)
					{
						/* do polygon format output */
						if (rx != 0 && ry == 0)
						{
							/* no rotation needed */
							mrgstorebox(poly->layer, tech, xh-xl+bloat, yh-yl+bloat,
								xpos-offx, ypos-offy);
							return;
						}
						if (rx == 0 && ry != 0)
						{
							/* rotate through 90 degrees */
							mrgstorebox(poly->layer, tech, yh-yl+bloat, xh-xl+bloat,
								xpos-offx, ypos-offy);
							return;
						}
						/* nonmanhattan or worse .. fall into direct output case */
					}

					/* non-merged box: highlight resolution errors */
					io_cifreserror(geom, xh-xl+bloat, yh-yl+bloat, xpos-offx, ypos-offy);

					/* want individual box output: write the layer */
					io_currentciflayer(lay);
					io_cifprintf(" B %ld %ld %ld %ld", io_cif_outscale(xh-xl+bloat),
						io_cif_outscale(yh-yl+bloat), io_cif_outscale(xpos-offx),
							io_cif_outscale(ypos-offy));
					if (rx <= 0 || ry != 0) io_cifprintf(" %ld %ld", rx, ry);
					io_cifprintf(";\n");
				} else
				{
					xformpoly(poly, trans);
					if (bloat != 0)
					{
						ttyputmsg("Warning: complex CIF polygon cannot be bloated");
						io_cifprintf("(NEXT POLYGON CANNOT BE BLOATED);\n");
					}
					if (poly->count == 1)
						io_cifprintf(" 0V %ld %ld %ld %ld;\n", io_cif_outscale(poly->xv[0]-offx),
							io_cif_outscale(poly->yv[0]-offy), io_cif_outscale(poly->xv[0]-offx),
								io_cif_outscale(poly->yv[0]-offy)); else
					if (poly->count == 2)
						io_cifprintf(" 0V %ld %ld %ld %ld;\n", io_cif_outscale(poly->xv[0]-offx),
							io_cif_outscale(poly->yv[0]-offy), io_cif_outscale(poly->xv[1]-offx),
								io_cif_outscale(poly->yv[1]-offy)); else
					{
						/*
						 * Now call routine to write the polygon:
						 *    - break long lines
						 *    - check for resolution errors
						 */
						for (k = 0; k < poly->count; k++)
						{
							/*
							 * WARNING: this changes poly!
							 */
							poly->xv[k] -= offx;
							poly->yv[k] -= offy;
						}
						io_currentciflayer(lay);
						io_cif_write_polygon(-1, NOTECHNOLOGY, poly->xv, poly->yv, poly->count, 0.0);
					}
				}
				return;
		}
	}

	/* certain polygons without valid CIF layers can still be written */
	switch (poly->style)
	{
		case CLOSEDRECT:
			xformpoly(poly, trans);
			getbbox(poly, &xl,&xh, &yl,&yh);
			io_cifprintf("0V %ld %ld", io_cif_outscale(xl-offx), io_cif_outscale(yl-offy));
			io_cifprintf(" %ld %ld", io_cif_outscale(xl-offx), io_cif_outscale(yh-offy));
			io_cifprintf(" %ld %ld", io_cif_outscale(xh-offx), io_cif_outscale(yh-offy));
			io_cifprintf(" %ld %ld;\n", io_cif_outscale(xh-offx), io_cif_outscale(yl-offy));
			return;

		case OPENED:
		case OPENEDT1:
		case OPENEDT2:
		case OPENEDT3:
		case CLOSED:
			xformpoly(poly, trans);
			io_cifprintf("0V");
			for(k=0; k<poly->count; k++)
				io_cifprintf(" %ld %ld", io_cif_outscale(poly->xv[k]-offx), io_cif_outscale(poly->yv[k]-offy));
			if (poly->style == CLOSED)
				io_cifprintf(" %ld %ld", io_cif_outscale(poly->xv[k]-offx), io_cif_outscale(poly->yv[k]-offy));
			io_cifprintf(";\n");
			return;

		case VECTORS:
			xformpoly(poly, trans);
			for(k=0; k<poly->count; k += 2)
			{
				io_cifprintf("0V %ld %ld", io_cif_outscale(poly->xv[k]-offx),
					io_cif_outscale(poly->yv[k]-offy));
				io_cifprintf(" %ld %ld;\n", io_cif_outscale(poly->xv[k+1]-offx),
					io_cif_outscale(poly->yv[k+1]-offy));
			}
			return;
	}
}

/*
 * routine to send the correct layer in "lay" to the CIF file
 */
void io_currentciflayer(char *lay)
{
	if (strcmp(lay, io_curlay) == 0) io_cifprintf("     "); else
	{
		io_cifprintf("L %s;\n", lay);
		(void)strcpy(io_curlay, lay);
	}
}

/*
 * routine to sent a line to the CIF output file and to accumulate
 * checksum information.
 */
void io_cifprintf(char *msg, ...)
{
#define NMCHR 80
	static char line[200],temp[200];
	REGISTER char *pt;
	REGISTER INTSML this_byte;
	static INTSML wait = 0;
	va_list ap;

	var_start(ap, msg);
	(void)vsprintf(temp, msg, ap);
	va_end(ap);

	if (strlen(temp) + wait > NMCHR-1)
	{
		strcat(line,"\n");
		wait++;
	}
	strcat(line,temp);
	for(pt = line; *pt != 0; pt++)
	{
		this_byte = *pt & 0177;

		if (this_byte > 32)
		{
			io_check_sum += this_byte;
			io_prev_chr_sep = 0;
			io_nchar++;
		} else if (!io_prev_chr_sep)
		{
			io_check_sum += 32;
			io_prev_chr_sep = 1;
			io_nchar++;
		}
	}
	if ((pt = strrchr(line,'\n')) != 0)
	{
		pt++;
		strcpy(temp,pt);	/* save the rest of the line */
		*pt = 0;		/* terminate the output */
		xprintf(io_fileout, "%s", line);
		strcpy(line,temp);	/* now restore it */
	}
	wait = strlen(line);
	return;
}

/* routine which highlights box which doesnt resolve */
void io_cifreserror(GEOM *pos, INTBIG length, INTBIG width, INTBIG xc, INTBIG yc)
{
	INTBIG left, right, top, bot;
	extern AIDENTRY *us_aid;

	left = xc - (length/2);
	right = xc + (length/2);
	top = yc + (width/2);
	bot = yc - (width/2);

	/* if there is a geometry module */
	if (pos != NOGEOM && io_cif_resolution != 0)
	{
		/* if doesnt resolve */
		if (((left % io_cif_resolution) != 0) || ((right % io_cif_resolution) != 0) ||
			((top % io_cif_resolution) != 0) || ((bot % io_cif_resolution) != 0))
		{
			/* highlight the object */
			(void)askaid(us_aid, "show-object", (INTBIG)pos);
			io_reserr_count++;
		}
	}
}

#define	NMCHR	80

void io_cif_write_polygon(INTSML layer, TECHNOLOGY *tech, INTBIG *xbuf, INTBIG *ybuf,
	INTSML count, float area)
{
	INTSML buflen, prlen, i, err;
	INTBIG tmpx, tmpy;
	char outbuf[NMCHR+2], prbuf[40], *lay; /* declare buffers */

	if (layer >= 0 && tech != NOTECHNOLOGY && tech->temp1 != 0)
	{
		lay = ((char **)tech->temp1)[layer];
		if (strcmp(lay, io_curlay) != 0)
		{
			io_cifprintf("L %s;\n", lay);
			(void)strcpy(io_curlay, lay);
		}
	}

	/* set up first line */
	(void)sprintf(outbuf, "P ");
	buflen = strlen(outbuf);

	/* check the number of points on the polygon */
	if (count > io_cif_polypoints)
		ttyputerr("WARNING: Polygon has too many points (%d)", count);

	/* search for any resolution errors on this polygon */
	/* highlight any edges that don't resolve */
	if (io_cif_resolution != 0)
	{
		err = 0;
		tmpx = xbuf[count-1];
		tmpy = ybuf[count-1];
		for (i = 0; i < count; i++)
		{
			if ((tmpx%io_cif_resolution) && (tmpx == xbuf[i]))
			{
				io_cif_show_reserror (tmpx+coffx, tmpy+coffy, xbuf[i]+coffx, ybuf[i]+coffy);
				err++;
			} else if ((tmpy%io_cif_resolution) && (tmpy == ybuf[i]))
			{
				io_cif_show_reserror (tmpx+coffx, tmpy+coffy, xbuf[i]+coffx, ybuf[i]+coffy);
				err++;
			}
			tmpx = xbuf[i];
			tmpy = ybuf[i];
		}
		if (err != 0) io_reserr_count++;
	}

	for(i=0; i<count; i++)
	{
		(void)sprintf(prbuf, " %ld %ld", io_cif_outscale(xbuf[i]), io_cif_outscale(ybuf[i]));
		prlen = strlen(prbuf); /* get length of coord. pair */

		/* if we have done the last point */
		if (i == count-1)
		{
			if (prlen + buflen < NMCHR-1) /* if space in buffer */
			{
				(void)strcat(outbuf, prbuf); /* concatenate strings */
				io_cifprintf("%s;\n", outbuf); /* write the buffer */
			} else
			{
				/* write as two lines */
				io_cifprintf("%s\n", outbuf);
				io_cifprintf("  %s;\n", prbuf);
			}
		} else
		{
			/* not yet done */
			if (prlen + buflen < NMCHR) /* if small enough */
			{
				/* append to buffer */
				(void)strcat(outbuf, prbuf);
				buflen = strlen(outbuf);
			} else
			{
				/* we must write the buffer out, and re-initialize it */
				io_cifprintf("%s\n", outbuf);
				(void)sprintf(outbuf, "  %s", prbuf);
				buflen = strlen(outbuf);
			}
		}
	}
}

void io_cif_show_reserror(INTBIG x1, INTBIG y1, INTBIG x2, INTBIG y2)
{
	extern AIDENTRY *us_aid;
	REGISTER NODEPROTO *np;

	/* show resolution error if it is in the current window */
	np = getcurfacet();
	if (np == curnp)
		(void)askaid(us_aid, "show-line", x1, y1, x2, y2, np); else
			ttyputmsg("Resolution error in subfacet %s", describenodeproto(curnp));
}

INTBIG io_cif_outscale(INTBIG value)
{
	float v;

	v = scaletodispunit(value, DISPUNITCMIC);
	return(roundfloat(v));
}

#endif  /* IOCIF - at top */
