/*  GUBI - Gtk+ User Interface Builder
 *  Copyright (C) 1997  Tim Janik
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
#include	"RCS.h"
RCS_ID("$Id: wdtree.c,v 1.10 1997/05/20 21:17:50 tim Exp $")


#define		__wdtree_c__

#include	"wdtree.h"
#include	"treelist.h"
#include	"browser.h"
#include	"editor.h"
#include	"widdata.h"
#include	"structures.h"
#include	"defines.h"



/* --- external global variables --- */
GList		*UD_tree_list=NULL;



/* --- prototypes --- */
static	void	widget_data_parent_add		(gb_wdat_base_S	*Parent,
						 gb_wdat_base_S	*WidDat,
						 guint		position);
static	void	tree_widget_data_insert_foreach	(gpointer	data,
						 gpointer	user_data);
static	void	tree_widget_data_unlink_foreach	(gpointer	data,
						 gpointer	user_data);
static	void	SigH_tree_window_destroy	(GtkWidget	*window);

static	gb_wdat_base_S*
		tree_widget_data_find_prev	(gb_wdat_base_S	*WidDat);
static	gint	current_widget_timeout		(gpointer	data);



/* --- variables --- */
static	gb_wdat_base_S	*current_WidDat=NULL;
static	tree_S		*current_tree=NULL;


/* --- functions --- */
tree_S*
tree_new	(gboolean	user_window,
		 gboolean	visible)
{
	register tree_S	*tree;
	
	g_assert(user_window || !visible);
	
	
	/* create and fill in current tree_S
	*/
	tree=g_new(tree_S, 1);
	tree->type=GB_STRUCT_NONE;
	tree->visible=visible;
	tree->auto_resize=TRUE;
	tree->width=-1;
	tree->height=-1;
	tree->current=NULL;
	tree->widget_data_list=NULL;
	
	
	/* append to global UD_tree_list
	*/
	if (user_window)
		UD_tree_list=g_list_append(UD_tree_list, tree);
	
	return tree;
}


void
tree_delete	(tree_S	*tree)
{
	g_assert(tree);
	
	if (g_list_length(tree->widget_data_list)) {
		register gb_wdat_base_S	*WinDat;
		
		WinDat=tree->widget_data_list->data;
		g_assert(GB_IS_WIDDAT_WINDOW(WinDat));
		
		GUBI_DATA(WinDat)->tree=NULL;
		
		if (GUBI_DATA(WinDat)->editor)
			gtk_widget_destroy(GUBI_DATA(WinDat)->editor);
		if (GUBI_DATA(WinDat)->browser)
			gtk_widget_destroy(GUBI_DATA(WinDat)->browser);
		if (WinDat->widget)
			gtk_widget_destroy(WinDat->widget);
		
		widget_data_symbol_name_unregister(WinDat);
		
		widget_data_delete_R(WinDat);
	}

	UD_tree_list=g_list_remove(UD_tree_list, tree);
	
	g_free(tree);
}


void
widget_data_parent_add	(gb_wdat_base_S	*Parent,
			 gb_wdat_base_S	*WidDat,
			 guint		position)
{
	g_assert(GB_IS_WIDDAT_CONTAINER(Parent));
	g_assert(GB_IS_WIDDAT(WidDat));
	g_assert(!WidDat->parent);
	g_assert(!GB_IS_WIDDAT_WINDOW(WidDat));
	g_assert(!(GUBI_DATA(WidDat)->category & WID_CAT_AUXILLARY));
	
	if (HAS_MAX_CHILD_COUNT(Parent))
		g_error("%s\n add <%s> to <%s>: max_child_count (%d) reached: child_count=%d",
			FUNCNAME,
			widget_data_symbol_name_get(WidDat),
			widget_data_symbol_name_get(Parent),
			GUBI_DATA(Parent)->max_child_count,
			GUBI_DATA(Parent)->child_count);
	
	position=CLAMP(position, 0, GUBI_DATA(Parent)->max_child_count);
	
	
	/* add to parent
	*/
	WidDat->parent=Parent;
	GUBI_DATA(Parent)->children=g_list_insert(GUBI_DATA(Parent)->children,
						  WidDat,
						  position);
	
	UPDATE_CHILD_COUNT(Parent);
}


void
tree_widget_data_insert_foreach	(gpointer	data,
				 gpointer	user_data)
{
	register tree_S		*tree;
	register gb_wdat_base_S	*WidDat;
	register gb_wdat_base_S	*PrevDat;
	register GList		*list;
	register GList		*new_list;
	
	tree=user_data;
	WidDat=data;
	
	g_assert(tree);
	g_assert(GB_IS_WIDDAT(WidDat));
	g_assert(!GB_IS_WIDDAT_WINDOW(WidDat));
	g_assert(GB_IS_WIDDAT_CONTAINER(WidDat->parent));
	g_assert(!WidDat->next);
	g_assert(!GUBI_DATA(WidDat)->tree);
	

	/* append to list
	*/
	GUBI_DATA(WidDat)->tree=tree;
	PrevDat=tree_widget_data_find_prev(WidDat);
	g_assert(GB_IS_WIDDAT(PrevDat));
	WidDat->next=PrevDat->next;
	PrevDat->next=WidDat;
	list=g_list_find(tree->widget_data_list, PrevDat);
	g_assert(list);
	new_list=g_list_alloc();
	new_list->data=WidDat;
	new_list->next=list->next;
	if (list->next)
		list->next->prev=new_list;
	new_list->prev=list;
	list->next=new_list;
	
	
	/* register our symbol_name
	*/
	g_assert(!widget_data_lookup(widget_data_symbol_name_get(WidDat)));
	widget_data_symbol_name_register(WidDat);
	
	
	/* link children to end of list
	*/
	g_list_foreach(GUBI_DATA(WidDat)->children,
			tree_widget_data_insert_foreach,
			tree);
}


void
tree_widget_data_insert_R	(tree_S		*tree,
				 gb_wdat_base_S	*ParDat,
				 gb_wdat_base_S	*WidDat,
				 guint	position)
{
	g_assert(tree);
	g_assert(GB_IS_WIDDAT(WidDat));
	g_assert( (GB_IS_WIDDAT_WINDOW(WidDat) && !ParDat) || 
		  (!GB_IS_WIDDAT_WINDOW(WidDat) && GB_IS_WIDDAT_CONTAINER(ParDat)) );
	g_assert(!WidDat->next);
	g_assert(!GUBI_DATA(WidDat)->tree);
	g_assert(!GUBI_DATA(WidDat)->editor);
	g_assert(!GUBI_DATA(WidDat)->browser);
	g_assert(!(GUBI_DATA(WidDat)->category & WID_CAT_AUXILLARY));
	
	if (GB_TYPE_IS_WIDDAT_WINDOW(WidDat->type)) {
		
		g_assert(!WidDat->parent);
		g_assert(!tree->widget_data_list);
		g_assert(!GUBI_DATA(WidDat)->children);
		
		
		/* link window into tree_S
		*/
		GUBI_DATA(WidDat)->tree=tree;
		tree->widget_data_list=g_list_append(tree->widget_data_list, WidDat);
		
		
		/* register our symbol_name
		*/
		g_assert(!widget_data_lookup(widget_data_symbol_name_get(WidDat)));
		widget_data_symbol_name_register(WidDat);
	
	} else if (!HAS_MAX_CHILD_COUNT(ParDat)) {
		
		g_assert(tree->widget_data_list);
		
		widget_data_parent_add(ParDat, WidDat, position);
		
		
		/* link us to end of list
		 * including children
		*/
		tree_widget_data_insert_foreach(WidDat, tree);
		
		
		/* build widget if parent exists
		*/
		if (WidDat->parent->widget) {
			gb_widget_create(WidDat);
			g_list_foreach(GUBI_DATA(WidDat)->children,
				       (GFunc)gb_widget_create,
				       NULL);
		}
	} else {
		g_error("%s\nmaximum child count reached in `%s'",
			FUNCNAME,
			widget_data_symbol_name_get(ParDat));
	}
}


void
tree_widget_data_unlink_foreach	(gpointer	data,
				 gpointer	user_data)
{
	register tree_S		*tree;
	register gb_wdat_base_S	*WidDat;	/* recursion! */
	register gb_wdat_base_S	*TmpDat;
	register GList		*list;
	
	
	tree=user_data;
	
	WidDat=data;
	
	g_assert(tree);
	g_assert(GB_IS_WIDDAT(WidDat));
	g_assert(GUBI_DATA(WidDat)->tree);
	g_assert(!GB_IS_WIDDAT_WINDOW(WidDat));
	
	
	/* delete references from other widgets in widget data
	*/
	while (GUBI_DATA(WidDat)->link_refs) {
		register gb_wdat_base_S	*RefDat;
		
		RefDat=GUBI_DATA(WidDat)->link_refs->data;
		g_assert(GB_IS_WIDDAT(RefDat));
		
		widget_data_link_remove(RefDat, WidDat);
		
		if (GUBI_DATA(RefDat)->editor)
			Editor_refresh(RefDat);
	}
	
	
	/* if we have an editor/browser then destroy it
	*/
	if (GUBI_DATA(WidDat)->editor)
		gtk_widget_destroy(GUBI_DATA(WidDat)->editor);
	if (GUBI_DATA(WidDat)->browser)
		gtk_widget_destroy(GUBI_DATA(WidDat)->browser);
	
	
	/* unregister our symbol_name
	*/
	widget_data_symbol_name_unregister(WidDat);
	
	
	/* unlink from tree
	*/
	GUBI_DATA(WidDat)->tree=NULL;
	list=g_list_find(tree->widget_data_list, WidDat);
	g_assert(list && list->prev);
	if (!WidDat->next) {
		g_assert(!list->next);
	} else {
		g_assert(list->next);
		g_assert(WidDat->next==list->next->data);
	}
	TmpDat=list->prev->data;
	g_assert(TmpDat->next==WidDat);
	if (list->next) {
		TmpDat->next=list->next->data;
	} else
		TmpDat->next=NULL;
	WidDat->next=NULL;
	tree->widget_data_list=g_list_remove(tree->widget_data_list, WidDat);
	
	
	/* unlink children from tree
	*/
	g_list_foreach(GUBI_DATA(WidDat)->children,
			tree_widget_data_unlink_foreach,
			tree);
}


void
tree_widget_data_unlink_R	(tree_S		*tree,
				 gb_wdat_base_S	*WidDat)
{
	g_assert(tree);
	g_assert(GB_IS_WIDDAT(WidDat));
	g_assert(!GB_IS_WIDDAT_WINDOW(WidDat));
	g_assert(GUBI_DATA(WidDat)->tree);
	g_assert(GB_IS_WIDDAT_CONTAINER(WidDat->parent));
	g_assert(!(GUBI_DATA(WidDat)->category & WID_CAT_AUXILLARY));
	
	
	/* destroy GtkWidget
	 * (gtk destroys our children widgets as well!)
	*/
	if (WidDat->widget) {
		gtk_widget_hide(WidDat->widget);
	}
	/* if (WidDat->parent->widget && WidDat->widget) {
	 *	gtk_container_remove(GTK_CONTAINER(WidDat->parent->widget), WidDat->widget);
	 * }
	 * FIXME: this fails on a GtkFileSelection
	*/
	if (WidDat->widget)
		gtk_widget_destroy(WidDat->widget);
	
	
	/* unlink from tree recursively
	*/
	tree_widget_data_unlink_foreach(WidDat, tree);
	
	
	/* unlink from parent
	*/
	GUBI_DATA(WidDat->parent)->children=g_list_remove(
		GUBI_DATA(WidDat->parent)->children,
		WidDat);

	UPDATE_CHILD_COUNT(WidDat->parent);
	WidDat->parent=NULL;
}


void
tree_rebuild	(tree_S	*tree)
{
	register gb_wdat_window_S	*WinDat;

	g_assert(tree);
	g_assert(tree->widget_data_list);
	WinDat=GB_wCAST(window, tree->widget_data_list->data);
	g_assert(GB_IS_WIDDAT_WINDOW(WinDat));
	
	
	/* destroy old window if we have a GtkWidget
	*/
	if (WinDat->widget && GTK_WIDGET_VISIBLE(WinDat->widget)) {
		tree->width=WinDat->widget->allocation.width;
		tree->height=WinDat->widget->allocation.height;
	}
	if (WinDat->widget) {
		register tree_S		*save_tree;
		
		save_tree=GUBI_DATA(WinDat)->tree;
		GUBI_DATA(WinDat)->tree=NULL;
		gtk_widget_destroy(WinDat->widget);
		GUBI_DATA(WinDat)->tree=save_tree;
	}
	
	
	/* build window and its widget tree
	*/
	gb_window_build(WinDat);
}


void
tree_show	(tree_S	*tree)
{
	register gb_wdat_window_S	*WinDat;

	g_assert(tree);
	g_assert(tree->widget_data_list);
	WinDat=GB_wCAST(window, tree->widget_data_list->data);
	g_assert(GB_IS_WIDDAT_WINDOW(WinDat));
	
	/* rebuild if we haven't a GtkWidget
	*/
	if (!WinDat->widget)
		tree_rebuild(tree);
	else
		gtk_widget_hide(WinDat->widget);
	
	
	/* we need an apropriate postion override!
	*/
	gtk_window_position(GTK_WINDOW(WinDat->widget), GTK_WIN_POS_NONE);
	if (WinDat->x<0 || WinDat->y<0)
		gtk_widget_set_uposition(WinDat->widget, 1, 1);
	
	
	/* we need an apropriate size override!
	*/
	if (tree->auto_resize) {
		tree->width=-1;
		tree->height=-1;
	} else if (tree->width>=0 && tree->height>=0) {
		gtk_widget_set_usize(WinDat->widget,
				     tree->width,
				     tree->height);
		gtk_window_set_policy(GTK_WINDOW(WinDat->widget),
				      TRUE,
				      TRUE,
				      (WinDat->resize_policy&GB_AUTO_SHRINK)!=0);
	}
	
	
	/* show if visible	gdk_window_get_origin
	*/
	if (tree->visible) {
		gtk_signal_connect_object(GTK_OBJECT(WinDat->widget),
					  "destroy",
					  GTK_SIGNAL_FUNC(SigH_tree_window_destroy),
					  GTK_OBJECT(WinDat->widget));
		gtk_widget_show(WinDat->widget);
	}
}


void
tree_set_hints		(gboolean	flash_current)
{
	static	 gint			current_widget_timer=0;
	
	
	/* let the current widget blink?
	*/
	if (!current_widget_timer && flash_current)
		current_widget_timer=gtk_timeout_add(SHOW_CURRENT_DELAY, current_widget_timeout, NULL);
	if (current_widget_timer && !flash_current) {
		gtk_timeout_remove(current_widget_timer);
		current_widget_timer=0;
	}
	if (current_WidDat &&
	    current_WidDat->widget &&
	    GTK_WIDGET_DRAWABLE(current_WidDat->widget) &&
	    current_WidDat->widget->window)
		_gdk_window_clear_expose(current_WidDat->widget->window);
}


void
SigH_tree_window_destroy(GtkWidget	*window)
{
	register gubi_data_S	*Gubi_Data;
	
	g_assert(Gubi_Data=gtk_object_get_user_data(GTK_OBJECT(window)));
	g_assert(GB_IS_WIDDAT_WINDOW(Gubi_Data->widget_data));
	
	if (Gubi_Data->tree) {
		Gubi_Data->tree->visible=FALSE;
		TreeList_hints_update(Gubi_Data->tree);
	}
}


gb_wdat_base_S*
tree_widget_data_find_prev	(gb_wdat_base_S	*WidDat)
{
	register GList		*list;
	
	g_assert(GB_IS_WIDDAT(WidDat));
	
	
	/* we return NULL if we don't have a parent
	*/
	if (!WidDat->parent)
		return NULL;
	
	
	/* find our previous sibling
	*/
	list=GUBI_DATA(WidDat->parent)->children;
	list=g_list_find(list, WidDat);
	g_assert(list);
	
	list=list->prev;
	
	
	/* if there is no previous sibling, our parent is
	 * the previous widget
	*/
	if (!list)
		return WidDat->parent;
	
	
	/* we found a sibling, lets get the last child in it's tree
	*/
	WidDat=widget_data_last_child(list->data);
	
	return WidDat;
}


gb_wdat_base_S*
widget_data_last_child	(gb_wdat_base_S	*WidDat)
{
	g_assert(GB_IS_WIDDAT(WidDat));
	
	while (GUBI_DATA(WidDat)->children)
		WidDat=g_list_last(GUBI_DATA(WidDat)->children)->data;
	
	return WidDat;
}


tree_S*
tree_get_current	(void)
{
	return current_tree;
}


gb_wdat_window_S*
window_data_get_current	(void)
{
	if (current_tree) {
		register gb_wdat_window_S       *WinDat;
		
		g_assert(current_tree->widget_data_list);
		WinDat=current_tree->widget_data_list->data;
		g_assert(GB_IS_WIDDAT_WINDOW(WinDat));
		
		return WinDat;
	} else
		return NULL;
}


void
tree_set_current	(tree_S		*tree)
{
	current_WidDat=NULL;

	if (tree!=current_tree) {
		if (current_tree) {
			Browser_tree_select(current_tree, FALSE);
			if (current_tree &&
			    current_tree->current &&
			    current_tree->current->widget &&
			    GTK_WIDGET_DRAWABLE(current_tree->current->widget) &&
			    current_tree->current->widget->window)
				_gdk_window_clear_expose(current_tree->current->widget->window);
		}
		if (tree)
			Browser_tree_select(tree, TRUE);
	}
	current_tree=tree;

	if (current_tree)
		current_WidDat=current_tree->current;
}


void
tree_set_current_widget_data	(tree_S		*tree,
				 gb_wdat_base_S	*WidDat)
{
	g_assert(tree);
	
	if (WidDat) {
		g_assert(GB_IS_WIDDAT(WidDat));
		g_assert(g_list_find(tree->widget_data_list, WidDat));
	}
	
	if (tree->current==current_WidDat) {
		current_WidDat=NULL;
		
		if (tree->current &&
		    tree->current->widget &&
		    GTK_WIDGET_DRAWABLE(tree->current->widget) &&
		    tree->current->widget->window)
			_gdk_window_clear_expose(tree->current->widget->window);
	}
	
	tree->current=WidDat;
	if (tree==current_tree)
		current_WidDat=WidDat;
}


gint
current_widget_timeout	(gpointer	data)
{
	static	 gboolean	current_toggle=FALSE;
	register gb_wdat_base_S	*WidDat=current_WidDat;
	register GdkGC		*GC=NULL;
	register GtkWidget	*widget;
	register GdkWindow	*window;
	
	if (!WidDat || !WidDat->widget) {
		current_toggle=FALSE;
		return TRUE;
	}
	
	switch (WidDat->type) {
	
	case	GB_WIDGET_BUTTON:
	case	GB_WIDGET_TOGGLE_BUTTON:
	case	GB_WIDGET_CHECK_BUTTON:
	case	GB_WIDGET_RADIO_BUTTON:
		widget=GTK_BUTTON(WidDat->widget)->child;
		widget=WidDat->widget;
		break;
	
	default:
		widget=WidDat->widget;
		break;
	}
	
	if ((current_toggle^=TRUE))
		GC=gtk_widget_get_default_style()->black_gc;
	else
		GC=gtk_widget_get_default_style()->white_gc;
	
	if (!widget || !GTK_WIDGET_DRAWABLE(widget)) {
		current_toggle=FALSE;
		return TRUE;
	}
	
	if (GTK_IS_BUTTON(widget))
		window=widget->parent->window;
	else
		window=widget->window;
	
	gdk_draw_rectangle	(window,
				 GC,
				 FALSE,
				 widget->allocation.x+1,
				 widget->allocation.y+1,
				 widget->allocation.width-1,
				 widget->allocation.height-1);
	gdk_draw_rectangle	(window,
				 GC,
				 FALSE,
				 widget->allocation.x,
				 widget->allocation.y,
				 widget->allocation.width,
				 widget->allocation.height);
	gdk_draw_line	(window,
			 GC,
			 widget->allocation.x,
			 widget->allocation.y,
			 widget->allocation.x +
			   widget->allocation.width,
			 widget->allocation.y +
			   widget->allocation.height);
	gdk_draw_line	(window,
			 GC,
			 widget->allocation.x +
			   widget->allocation.width,
			 widget->allocation.y,
			 widget->allocation.x,
			 widget->allocation.y +
			   widget->allocation.height);
	
	return TRUE;
}
