/* Libart_LGPL - library of basic graphic primitives
 * Copyright (C) 1998 Raph Levien
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
#include	<libart_lgpl/art_misc.h>
#include	"art_vpatho_pixbuf.h"
#include	<libart_lgpl/art_svp.h>
#include	<libart_lgpl/art_svp_vpath.h>
#include	<libart_lgpl/art_svp_point.h>

#include <math.h>

#define EPSILON 1e-6

#define POINTS_PREALLOC	     (32)
     
typedef enum
{
  D_EAST,
  D_NORTH,
  D_WEST,
  D_SOUTH
} Direction;

typedef struct
{
  int x, y;
  Direction dir;

  int width, height;
  int alpha_threshold, rowstride;
  art_u8 *pixels;

  int n_points, n_alloced_points;
  ArtVpath *vpath;
  ArtSVP *svp;
} VC;


/* --- short hand macros --- */
#define pixel_offset(vc, x, y) ((vc)->pixels + (y) * (vc)->rowstride + (x) * 4)
#define is_pixel(vc, x, y)     (pixel_offset ((vc), (x), (y)) [3] >= (vc)->alpha_threshold)
#define dir_left(dir)	       ((dir) == D_SOUTH ? D_EAST : (dir) + 1)
#define dir_right(dir)	       ((dir) == D_EAST ? D_SOUTH : (dir) - 1)
#define turn_left(vc)          ((void) ((vc)->dir = dir_left ((vc)->dir)))
#define turn_right(vc)         ((void) ((vc)->dir = dir_right ((vc)->dir)))
#define	dir_opposite(dir)      ((dir) > D_NORTH ? (dir) - 2 : (dir) + 2)
#define	dir_name(dir)	       ((dir) == D_EAST  ? "EAST" : \
                                (dir) == D_NORTH ? "NORTH" : \
                                (dir) == D_WEST  ? "WEST" : \
                                                   "SOUTH")


/* --- functions --- */
static inline int
move_towards (VC *vc, Direction dir, int *x_p, int *y_p)
{
  switch (dir)
    {
    case D_EAST:  *x_p += 1; break;
    case D_NORTH: *y_p -= 1; break;
    case D_WEST:  *x_p -= 1; break;
    case D_SOUTH: *y_p += 1; break;
    }
  return *y_p >= 0 && *x_p >= 0 && *y_p < vc->height && *x_p < vc->width;
}

static inline int
pixel_ahead (VC *vc)
{
  int x = vc->x, y = vc->y;

  return move_towards (vc, vc->dir, &x, &y) && is_pixel (vc, x, y);
}
static inline int
pixel_left (VC *vc)
{
  int x = vc->x, y = vc->y;

  return move_towards (vc, dir_left (vc->dir), &x, &y) && is_pixel (vc, x, y);
}
static inline int
pixel_right (VC *vc)
{
  int x = vc->x, y = vc->y;

  return move_towards (vc, dir_right (vc->dir), &x, &y) && is_pixel (vc, x, y);
}
static inline void
add_point (VC *vc, double x, double y)
{
  int i = vc->n_points;

  vc->n_points++;
  if (vc->n_points == vc->n_alloced_points)
    art_expand (vc->vpath, ArtVpath, vc->n_alloced_points);
  vc->vpath[i].code = ART_LINETO;
  vc->vpath[i].x = x;
  vc->vpath[i].y = y;
}

static inline void
add_rear (VC *vc)
{
  double x = vc->x, y = vc->y;
  int dir = vc->dir;

  /* depending on the direction we are facing, we have to add the coordinates
   * of the inner pixel corner at our rear.
   * offsets are as follows:
   *
   * dir facing  rear coords: x  y
   * D_EAST:		     +0 +0
   * D_NORTH:		     +0 +1
   * D_WEST:		     +1 +1
   * D_SOUTH:		     +1 +0
   */
  
  x += dir > D_NORTH;
  y += dir && dir < D_SOUTH;

  add_point (vc, x, y);
}

static void
vectorize_outline (VC *vc, int i_x, int i_y)
{
  double x = --i_x, y = i_y;
  int i, start;

  /* this function needs to be given a pixel that has no neighbour above
   * or left to it. after surrounding this pixel (plus adherents), the
   * path can only be closed with a left turn.
   */

  /* setup vc */
  vc->x = x;
  vc->y = y;
  vc->dir = D_SOUTH;
  vc->n_points -= 1;
  start = vc->n_points;
  
  /* setup starting point */
  add_rear (vc);
  vc->vpath[start].code = ART_MOVETO;

  /* vectorize until path is closed */
  do
    {
      while (pixel_left (vc))
	if (pixel_ahead (vc))
	  {
	    turn_right (vc);
	    add_rear (vc);
	    continue;
	  }
	else
	  move_towards (vc, vc->dir, &vc->x, &vc->y);

      turn_left (vc);
      move_towards (vc, vc->dir, &vc->x, &vc->y);
      i = vc->n_points;
      add_rear (vc);
    }
  while (fabs (vc->vpath[i].x - vc->vpath[start].x) >= EPSILON ||
	 fabs (vc->vpath[i].y - vc->vpath[start].y) >= EPSILON);
  
  /* end mark */
  i = vc->n_points;
  add_point (vc, 0, 0);
  vc->vpath[i].code = ART_END;
}

ArtVpath*
art_vpath_outline_from_pixbuf (const ArtPixBuf *pixbuf,
			       int              alpha_threshold)
{
  VC vectorization_context = { 0, }, *vc = &vectorization_context;
  int x, y;

  /* sanity checks */
  if (pixbuf->format != ART_PIX_RGB ||
      !pixbuf->has_alpha ||
      pixbuf->n_channels != 4 ||
      pixbuf->bits_per_sample != 8 ||
      !pixbuf->width ||
      !pixbuf->height)
    {
      /* printf ("unable to vectorize pixbuf\n"); */

      return NULL;
    }

  /* initial vc setup */
  vc->width = pixbuf->width;
  vc->height = pixbuf->height;
  vc->alpha_threshold = alpha_threshold;
  vc->rowstride = pixbuf->rowstride;
  vc->pixels = pixbuf->pixels;
  vc->n_alloced_points = 4;
  vc->n_points = 1;
  vc->vpath = art_new (ArtVpath, vc->n_alloced_points);
  vc->vpath->code = ART_END;
  vc->vpath->x = 0;
  vc->vpath->y = 0;
  vc->svp = NULL;

  /* find first pixel */
  for (y = 0; y < vc->height; y++)
    {
      for (x = 0; x < vc->width; x++)
	if (is_pixel (vc, x, y))
	  {
	    int xn;
	    
	    /* check if this point is already contained */
	    if (!vc->svp ||
		!art_svp_point_wind (vc->svp, x + 0.5, y + 0.5))
	      {
		/* pixel not yet contained, vectorize from here on */
		vectorize_outline (vc, x, y);
		if (vc->svp)
		  art_svp_free (vc->svp);
		vc->svp = art_svp_from_vpath (vc->vpath);
	      }
	    
	    /* eat contiguous pixels */
	    xn = x + 1;
	    while (xn < vc->width && is_pixel (vc, xn, y))
	      x = xn++;
	  }
    }
  
  /* if (vc->n_points == 1)
   *   printf ("unable to vectorize empty pixbuf\n");
   */

  if (vc->svp)
    {
      art_svp_free (vc->svp);
      vc->svp = NULL;
    }

  /* adjust size of closed outline path */
  vc->vpath = art_renew (vc->vpath, ArtVpath, vc->n_points);

  return vc->vpath;
}

int
art_affine_from_triplets (const ArtTriplet *src_triplet,
			  const ArtTriplet *transformed_triplet,
			  double matrix[6])
{
  double Ax = src_triplet->ax, Ay = src_triplet->ay;
  double Bx = src_triplet->bx, By = src_triplet->by;
  double Cx = src_triplet->cx, Cy = src_triplet->cy;
  double ax = transformed_triplet->ax, ay = transformed_triplet->ay;
  double bx = transformed_triplet->bx, by = transformed_triplet->by;
  double cx = transformed_triplet->cx, cy = transformed_triplet->cy;

  /* solve the linear equation system:
   *   ax = Ax * matrix[0] + Ay * matrix[2] + matrix[4];
   *   ay = Ax * matrix[1] + Ay * matrix[3] + matrix[5];
   *   bx = Bx * matrix[0] + By * matrix[2] + matrix[4];
   *   by = Bx * matrix[1] + By * matrix[3] + matrix[5];
   *   cx = Cx * matrix[0] + Cy * matrix[2] + matrix[4];
   *   cy = Cx * matrix[1] + Cy * matrix[3] + matrix[5];
   * for matrix[0..5]
   */
  double AxBy = Ax * By, AyBx = Ay * Bx;
  double AxCy = Ax * Cy, AyCx = Ay * Cx;
  double BxCy = Bx * Cy, ByCx = By * Cx;
  double com_div = AxBy - AyBx - AxCy + AyCx + BxCy - ByCx;

  if (fabs (com_div) < EPSILON)
    return ART_FALSE;

  matrix[0] = By * ax - Ay * bx + Ay * cx - Cy * ax - By * cx + Cy * bx;
  matrix[1] = By * ay - Ay * by + Ay * cy - Cy * ay - By * cy + Cy * by;
  matrix[2] = Ax * bx - Bx * ax - Ax * cx + Cx * ax + Bx * cx - Cx * bx;
  matrix[3] = Ax * by - Bx * ay - Ax * cy + Cx * ay + Bx * cy - Cx * by;
  matrix[4] = AxBy * cx - AxCy * bx - AyBx * cx + AyCx * bx + BxCy * ax - ByCx * ax;
  matrix[5] = AxBy * cy - AxCy * by - AyBx * cy + AyCx * by + BxCy * ay - ByCx * ay;
  matrix[0] /= com_div;
  matrix[1] /= com_div;
  matrix[2] /= com_div;
  matrix[3] /= com_div;
  matrix[4] /= com_div;
  matrix[5] /= com_div;

  return ART_TRUE;
}

double
art_vpath_area (const ArtVpath *vpath)
{
  double area = 0;

  while (vpath[0].code != ART_END)
    {
      int n_points, i;

      for (n_points = 1; vpath[n_points].code == ART_LINETO; n_points++)
	;
      
      for (i = 0; i < n_points; i++)
	{
	  int j = (i + 1) % n_points;
	  
	  area += vpath[j].x * vpath[i].y - vpath[i].x * vpath[j].y;
	}

      vpath += n_points;
    }

  return area * 0.5;
}
