# Conversion of files between different charsets and usages.
# Copyright (C) 1990 Free Software Foundation, Inc.
# Francois Pinard <pinard@iro.umontreal.ca>, 1990.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2, or (at your option)
# any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.


# This awk script merges several lex sources intended for recode.

BEGIN {
  TEMP_FILE = "mergelex.tmp"

  print "%{"
  print "#include \"common.h\""
  print "#ifdef USE_FPUTC"
  print "#define output(ch) fputc (ch, yyout)"
  print "#else"
  print "#define output(ch) putc (ch, yyout)"
  print "#endif"
  print "extern YY_CHAR *yytext;"
  print "extern int yyleng;"
  print "extern FILE *yyin, *yyout;"
  print "%}"
}

# A %{ comment in column 1 signals the start of a new file and of a
# C code section ended by a %} line.  This could be followed by a
# few lex definitions.
#
# A %% in column 1 signals the beginning of lex rules.

/^%\{/ {
  section = 1
  c_code = 1
  print
  next
}

/^%\}/ {
  print
  c_code = 0
  next
}

/^%%/ {
  print "%x X_" step_name
  section = 2
  next
}

# Remove block C comments.  It is assumed that, when a /* comment
# starts in column 1, there is no code following the closing */ on
# its line.  Also, remove all white lines.

/^\/\*/ {
  while (!match ($0, /\*\//))
    getline
  next
}

/^[ 	]*$/ {
  next
}

# In section 1, %{ delimited C code is copied almost verbatim.  For
# the `#define STEP' line, the step name is noted and added to the
# list of known step names, and a start declaration will be generated
# later, when the %% will be met. 

$1 == "#define" && $2 == "STEP" {
  step_name = $3
  step_names[$3]++
  next
}

c_code {
  print
  next
}

# Declarations are studied, declarations having the same name are kept
# only if their definition differ, so to generate a lex error later. 

/^[^ 	]/ && section == 1 {
  if ($1 in rules) {
    if ($0 == rules[$1]) {
      next
    }
  }
  else {
    rules[$1] = $0
  }
  print
  next
}

# In section 2, every line is copied to a temporary file.  While
# doing so, each lex rule is prefixed by a start name derived from
# the step name.

/^[ 	]/ && section == 2 {
  print > TEMP_FILE
  next
}

section == 2 {
  print "<X_" step_name ">" $0 > TEMP_FILE
  next
}

# At end, a single %% line is output followed by the contents of the
# temporary file.  Then, a second %% is output and followed by
# generated interfaces, one routine for each step name.

END {
  print "%%"

  close (TEMP_FILE)
  while (getline <TEMP_FILE)
    print
  close (TEMP_FILE)

  print "%%"
  for (step_name in step_names) {
    print "void"
    print step_name " (FILE *input_file, FILE *output_file)"
    print "{"
    print "  yy_init = 1;"
    print "  yyin = input_file;"
    print "  yyout = output_file;"
    print "  BEGIN X_" step_name ";"
    print "  yylex ();"
    print "}"
  }
}

