;;;;;;;;;;;;;;;;;;;;;;;;; -*- Mode: Emacs-Lisp -*- ;;;;;;;;;;;;;;;;;;;;;;;;;;
;; elec-ada.el --- Enhanced Ada editing mode
;; Author          : Lynn Slater
;; Created On      : Fri May 27 12:09:55 1988
;; Last Modified By: Lynn Slater
;; Last Modified On: Tue Oct 18 15:13:48 1988
;; Update Count    : 137
;; Status          : General Public Release 1.05
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; This file is part of GNU Emacs.
;; Copyright (C) 1988 Lynn Randolph Slater, Jr.
;; Copyright (C) 1988 Free Software Foundation, Inc.
;;
;; This file is distributed in the hope that it will be useful,
;; but without any warranty.  No author or distributor
;; accepts responsibility to anyone for the consequences of using it
;; or for whether it serves any particular purpose or works at all,
;; unless he says so in writing.
;;
;; Everyone is granted permission to copy, modify and redistribute
;; this file, but only under the conditions described in the
;; document "GNU Emacs copying permission notice".   An exact copy
;; of the document is supposed to have been given to you along with
;; this file so that you can know how you may redistribute it all.
;; It should be in a file named COPYING.  Among other things, the
;; copyright notice and this notice must be preserved on all copies.

;; Make this a file called elec-ada.el in your path
;; You should byte-compile it.

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; This mode tries to be independent of which Ada environment is being
;; used. It is a merger of ada.el in emacs 18.49 and electric-ada.el from
;; Steven D. Litvintchouk of Mitre Corp. This file is based in part on code
;; from the public domain and on code subject to the GNU emacs licensing
;; agreement.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; History 		
;; 13-Oct-1988		Lynn Slater	
;;    Made the auto-fill indent be indented.
;;    Greatly changed explination of indentation
;;    Greatly changed the operation of the tab key. Read the mode help
;; 11-Oct-1988		Lynn Slater	
;;    Added indent-to-previous-indentation (I LIKE this!)
;;    Made it indent better after a left paren.
;; 7-Oct-1988		Lynn Slater	
;;    Changed name to elec-ada.el so that it works on more file systems
;; 5-Oct-1988		Lynn Slater	
;;    Made the automatic indentation
;;      Sensitive to control-j's in the midst of comments
;;      Automatically indent parameters and arguments
;;      Indent broken up lines
;; 26-Sep-1988		Lynn Slater	
;;    Incorperated changes by Vincent Broman. Will call this release 1.05
;; 13-Sep-1988		Lynn Slater	
;;    Spelling changes. 
;;    Made file not care about backspace and help bindings except that
;;    the quick mode documentation is customized to the bindings I use.
;;      Thanks to jclark@src.honeywell.com (Jeff Clark)
;; 12-Sep-1988		Lynn Slater	
;;    Made the future enhancements paragraph be more detailed
;; 8-Sep-1988		Lynn Slater	
;;    Added insert-box, insert-end, and zap-crs.
;; 8-Sep-1988		Lynn Slater	
;;    Changed bug that set comment-start to "" rather than comment-end

(provide 'elec-ada) ;; lrs
(require 'dired-ada) ;; lrs
(require 'ada-tag)
;;(require 'ada-etempl) -- done later in this file

; The code in this file is partually based upon code subject to the
; following disclaimer.  The original code is public domain, but the
; changes are covered by a "copyleft".
; ----------------------------------------------------------------------------
; ada.el - Ada mode for GNU Emacs
; ----------------------------------------------------------------------------
;
;                           Ada Mode for GNU Emacs
;
;                           Steven D. Litvintchouk
;                             MITRE Corporation
;                            Bedford, MA  01730
;                              (617)271-7753
;
;                   
;
;    Ported from:  "Ada Mode for Gosling Emacs," originally created by:
;                              Steven M. Rosen
;                  Siemens Corporate Research and Support, Inc.
;                     Research and Technology Laboratories
;                             Princeton, NJ 08540
;                                (609) 734-6538
;  
; 
; ----------------------------------------------------------------------------
; 
; 	This code is public domain and may be used or modified at your site
; 	in any fashion that you choose.  No support or capabilities are
; 	implied or guaranteed by the release of this code.  This disclaimer
;	must be maintained in conjunction with the code.
; 
; ----------------------------------------------------------------------------
;
; $Header: ada.ml,v 4.1 85/04/15 08:33:06 rosen Exp $
; 
; Created 12-18-84
; Updated 04-15-85
; 
; Ported by Steven D. Litvintchouk (Mitre Corp.) to GNU Emacs
; Last updated 02-15/86
;
;
; Features: See the mode help
; ===================================================================
;
;

(defvar ada-mode-hook nil
  "List of functions to call when ada mode is invoked.
   This would be a good place to add Ada environment specific bindings.")

(defvar indent-change 2
  "*Defines the size of Ada indentation (default is 2).")

(defvar ada-mode-map nil "Local keymap for ada-mode buffers.")

(defvar ada-mode-syntax-table nil
  "Syntax table to be used for editing Ada source code.")

(defun ada-mode  ()
  ;; The doc below has no \\[] forms in it. The doc on
  ;; customized-ada-mode-help does.  I periodically take the "normal
  ;; editing" line and below from the output of customized-ada-mode-help
  ;; and place it in the string below. Sorry about this mess, but it is
  ;; needed to resolve the compromise between speed and completeness. See
  ;; the comment below the help of (defun dired-ada-mode ..) -- lrs
"Ada Mode is the major mode for editing Ada code.  

This is the STANDARD version of the help for Ada mode. The key bindings you
will see are those that are the default for your site at the time this mode
was installed. Personal customizations (from the .emacs file) or changes
made by loading other packages will not be shown. You may get an updated
and customized form of this message by typing 
'M-x customized-ada-mode-help', but this will take about 2 minutes. You
should use this version of the help text as long as it seems to work.
 
---------------------- Normal Editing ------------------

This mode automatically performs indentation while editing. Unfortunately,
Ada syntax is too awkward for the indentation to be reliable. Bindings are
as follows: (Note: 'LFD' is control-j.)

    Vanilla Newline                                    'ESC RET'
    Newline and indent in a 'smart' manner             'LFD'
      Ditto but remain in a comment if you are in one. 'ESC j'
    Newline but decrease indent level by fixed amount  'ESC LFD'
      or                                               `C-u LFD'
    Move to most recent indent level                   'TAB'
      (if BEFORE the first non blank)
    Move indentation level in by a fixed amount        'TAB'
      (If ON the first non blank)
    Tab over to under non-blanks on the above line     'TAB'
      (if AFTER the first non blank)
    Make this line be indented like the one above it.  'C-u TAB'
    Make this line be indented 1 level less than the   'ESC TAB'
      one above it.

    Indent from (dot) to (mark)			       'C-c >'
    De-dent from (dot) to (mark)		       'C-c <'
    Arbitrarly change indentation of region            'M-x change-indentation'
    Insert string in same column in all marked lines   'M-x insert-box'
    Get rid of some columns in all marked lines        'M-x kill-rectangle'

    Goto Next     Function/Procedure                   'ESC e'
    Goto Previous Function/Procedure                   'ESC a'
    Goto Next     Package                              'ESC C-e'
    Goto Previous Package                              'ESC C-a'

Comments are handled using standard GNU Emacs conventions, including:
    Start a comment                                    'ESC ;'
    Continue comment on next line                      'ESC j'

---------------------- Templates -----------------------
Ada Mode also supports a kind of template based editing. Bindings:
    Insert an Ada template                             'C-c i'
    Expand {token} surrounding or just after(?) point 
       into Ada construct template                     'C-c e'

A set of 'smart' templates (ones that use handcoded emacs routines rather
  than running from a BNF description) are available under C-ct.
  `C-c C-t h is particularly useful in that it inserts a standard
  function header. Other options are as follows:

x               ada-exit
C-x             ada-exception
w               ada-while-loop
C-w             ada-when
C-v             ada-private
C-u             ada-with-use
C-y             ada-type
t               ada-task-body
C-t             ada-task-spec
S               ada-tabsize
s               ada-select
C-s             ada-subtype
C-r             ada-record
o               ada-or
l               ada-loop
i               ada-if
f               ada-for-loop
C-f             ada-function-spec
p               ada-subprogram-body
C-p             ada-procedure-spec
C-n             ada-entry
k               ada-package-body
C-k             ada-package-spec
e               ada-else
C-e             ada-elsif
c               ada-case
d               ada-declare-block
b               ada-exception-block
a               ada-accept
C-a             ada-array
>               ada-forward-to-same-indent
<               ada-backward-to-less-indent
=               ada-show-subprogram-name
-               ada-comment
(               insert-parentheses
h               ada-header
RET             ada-openline


---------------------- Online Ada Manual ---------------
The sections of the online ada reference manual may be searched with
  'C-x .'. If this does not get the right section at first,
  'ESC ,' will get the next matching section.

All the entries in the online ada manual index for a keyword may be found
  with  'M-x index'. You may step through the entries with 'C-x `'.

If all this fails, the entire Ada reference manual may be searched 
  (using grep) by 'M-x search-adaman'.

---------------------- Debugger ------------------------
The 'M-x a-db' command invokes a.db on a binary file.

---------------------- Ada Development -----------------
Standard file headers may be made by 'M-x make-header'. The update count will
   be automatically incremented every time the file is saved. A new file in
   Ada mode and with a standard header may be made by typing 
   'C-c n'. A new set of spec and body files may be made by typing
   typing 'C-c C-n'. 'M-x make-revision' will insert a vevision
   stamp. 

Ada directories may be managed by 'M-x dired-ada'. 'Esc-? m' will give more
   help once this is done. You may invode dired on the directory of this
   file by 'C-c C-d'

On the current buffer:
       Recompilation Order                             'C-c r'
       Compile                                         'C-c C-c'
       Make this file                                  'C-c m'
       Make and load a ada program unit                'C-c RET'
       Load a program unit                             `C-c k'
       List VADS units made from this file             'C-c l'
       Remove VADS units made from this file           'C-c C-r'
       Execute a program                               'C-c x'
       Pretty Print the file                           <Not Yet Implemented>

Ada errors found during one of the compilations or makes may be scanned by
  the 'C-x `' command. This is a VERY useful facility. If the error
  is criptic, you may find the corresponding Ada Manual documentation by
  using the 'ESC `' command.  Both 'C-x `' and 'ESC `'
  work from anywhere within emacs.

Provided tags has been loaded (from dired-ada)
  you may look up Ada code by typing 'ESC .'. In case of multiple 
  entries in one file, you may find the other entries by typing 'ESC ,'.
  Typing 'ESC C-l' will pop you back from the tags lookup.
  Tag files are best made from a dired-ada ('C-c C-d').

---------------------- Customizations ------------------
The variable indent-change defines the size of indentation (the 
  default is 2).

The following local variables can be changed to taste:
  Fill-column       = 75
  Comment-column    = 40

Turning on Ada mode calls the value of the variable ada-mode-hook
  with no args, if that value is non-nil.

See the customizations section of the dired-ada mode documentation.

---------------------- More Help -----------------------
You can get more detailed help on any of these keystrokes 
  by typing 'Esc-? c' and then the keystrokes desired.
For help on functions or M-x sequences, type 'Esc-? f' and the sequence or
  function name. The help sequences are universal across modes.

---------------------- Future Enhancements -------------
  Clean up BNF notation for faster template editing
  Integrate Leif incrimental parsing minor mode
  Take ideas from outline mode to be able to skip levels of code
  Matching if statements and other block statements
  Pretty printing (not done because of no safe pretty printer)
  Integration of source control (Based on Ian G Batten's SCCS utilities)
  Block comment formatting
    we already are like indented text mode as far as auto-fill goes
    Need format paragraph like the recent lisp-fill files do
    Need to place comments such that the VADS pretty printer
       will not rearrange them.
  Mouse support for template selection and tags

Volunteers are gladly accepted.
---------------------- Known Bugs ----------------------

---------------------- Support --------------------------
For help, training, questions, or other support, contact Lynn Slater
  at extension 4482 or send a mail message to lrs@esl.

--------- Summary of Ada Mode Special Commands ----------

C-c             Prefix Command
TAB             ada-tab
LFD             indent-ada
C-x             Prefix Command
ESC             Prefix Command
DEL             backward-delete-char-untabify

C-c p           ada-db-print-value
C-c b           ada-db-break
C-c t           Prefix Command
C-c C-n         ada-new-files
C-c n           ada-new-file
C-c C-t         Prefix Command
C-c C-c         ada-compile-buffer
C-c c           ada-compile-buffer
C-c C-d         dired-ada-buffer
C-c d           dired-ada-buffer
C-c k           dired-ada-link
C-c l           ada-list-buffer
C-c RET         ada-make-unit
C-c m           ada-make-buffer
C-c C-r         ada-remove-buffer
C-c r           recompilation-order-ada-buffer
C-c x           dired-ada-execute
C-c .           ada-command
C-c e           expand-token
C-c i           insert-token
C-c <           dedent-region
C-c >           indent-region

C-x .           adaman

ESC C-a         ada-previous-package
ESC C-e         ada-next-package
ESC a           ada-previous-procedure
ESC e           ada-next-procedure
ESC RET         newline
ESC j           indent-new-comment-line
ESC LFD         ada-de-dent
ESC TAB         indent-to-previous-indentation

C-c C-t x       ada-exit
C-c C-t C-x     ada-exception
C-c C-t w       ada-while-loop
C-c C-t C-w     ada-when
C-c C-t C-v     ada-private
C-c C-t C-u     ada-with-use
C-c C-t C-y     ada-type
C-c C-t t       ada-task-body
C-c C-t C-t     ada-task-spec
C-c C-t S       ada-tabsize
C-c C-t s       ada-select
C-c C-t C-s     ada-subtype
C-c C-t C-r     ada-record
C-c C-t o       ada-or
C-c C-t l       ada-loop
C-c C-t i       ada-if
C-c C-t f       ada-for-loop
C-c C-t C-f     ada-function-spec
C-c C-t p       ada-subprogram-body
C-c C-t C-p     ada-procedure-spec
C-c C-t C-n     ada-entry
C-c C-t k       ada-package-body
C-c C-t C-k     ada-package-spec
C-c C-t e       ada-else
C-c C-t C-e     ada-elsif
C-c C-t c       ada-case
C-c C-t d       ada-declare-block
C-c C-t b       ada-exception-block
C-c C-t a       ada-accept
C-c C-t C-a     ada-array
C-c C-t >       ada-forward-to-same-indent
C-c C-t <       ada-backward-to-less-indent
C-c C-t =       ada-show-subprogram-name
C-c C-t -       ada-comment
C-c C-t (       insert-parentheses
C-c C-t h       ada-header
C-c C-t RET     ada-openline


------------- Rational for KeyBindings ------------------
In emacs, many keybindings may seem arbitrary. Ada mode tries to avoid this
as much as possible. It may be easier to learn Ada mode if the rational
behind the keymappings were understood. Below is a summary of how the
bindings got to be what they are.

C-c (control-c) is a standard emacs command prefix for language or mode
specific special things. Thus, lots of the Ada specific functions are
bound under C-c in the Ada modes.

The goto-next/previous functions are bound to the keys that are standard
goto-previous/next keys in other modes. In Ada mode, functions are roughly
equivalent to sentences, and packages are roughly equivalent to paragraphs.
Examples:
  'Esc-a' is backward sentence in many modes, but in Ada mode, it is
      previous-function.
  'Esc C-a' is beginning-of-defun in lisp mode, but in Ada mode it is
      previous-package. 
 
The comment manipulation functions are standard emacs bindings.

In most modes, 'Esc .' does a tags lookup. Online Ada documentation is
almost (but not quite) just like a tags table lookup. It is thus bound to
'C-x .' instead. All forms of lookup use 'ESC ,' to goto the next possible
match. All forms also recognize using 'Esc C-l' to backtrack. (You may
think of this binding as Esc-Last.)"
        (interactive)
	(kill-all-local-variables)
	(make-local-variable 'require-final-newline)
	(setq require-final-newline t)
	(make-local-variable 'comment-start)
	(setq comment-start "-- ")
        ;; comment end must be set because it may hold a wrong value if
	;; this buffer had been in another mode.
	(make-local-variable 'comment-end)
	(setq comment-end "")
	(make-local-variable 'comment-start-skip) ;; used by autofill
	(setq comment-start-skip "--+[ \t]*")
	(make-local-variable 'comment-multi-line)
	(setq comment-multi-line nil)
	(make-local-variable 'indent-line-function)
        ;; Define the autofill's behavior on ada code lines.
	;;(setq indent-line-function 'ada-tab)
	(setq indent-line-function 'ada-in-dent-current)
	(make-local-variable 'fill-column)
	(setq fill-column 75)
	(make-local-variable 'comment-column)
	(setq comment-column 40)
	(make-local-variable 'parse-sexp-ignore-comments)
	(setq parse-sexp-ignore-comments nil)
	(make-local-variable 'comment-indent-hook)
	(setq comment-indent-hook 'ada-comment-indent)
;
	(setq major-mode 'ada-mode)
	(setq mode-name "Ada")
	(setq blink-matching-paren t)
	(use-local-map ada-mode-map)
	;;(if (boundp 'emacs-menu)	; kludge to determine if
					; we are running as emacstool
	    ;; (ada-menu-activate))
	(if ada-mode-syntax-table
	    (set-syntax-table ada-mode-syntax-table)
	  (progn
            ;; this syntax table is a merge of two syntax tables I found
            ;; in the two ada modes in the old ada.el and the old
	    ;; electric-ada.el. 
	    (setq ada-mode-syntax-table (make-syntax-table))
	    (set-syntax-table  ada-mode-syntax-table)

	    ;; define string brackets (% is alternative string bracket)
	    (modify-syntax-entry ?%  "\"" ada-mode-syntax-table)
	    (modify-syntax-entry ?\" "\"" ada-mode-syntax-table)

	    ;; # chars match for based lits.
	    (modify-syntax-entry ?\#  "\"" ada-mode-syntax-table)

	    ;; Punctuations
	    ;;     (from old electric-ada.el)
	    (modify-syntax-entry ?:  "." ada-mode-syntax-table) 
	    (modify-syntax-entry ?\; "." ada-mode-syntax-table) 
	    (modify-syntax-entry ?&  "." ada-mode-syntax-table)
	    (modify-syntax-entry ?\|  "." ada-mode-syntax-table)
	    (modify-syntax-entry ?+  "." ada-mode-syntax-table)
	    (modify-syntax-entry ?*  "." ada-mode-syntax-table)
	    (modify-syntax-entry ?/  "." ada-mode-syntax-table)
	    (modify-syntax-entry ?=  "." ada-mode-syntax-table)
	    (modify-syntax-entry ?<  "." ada-mode-syntax-table)
	    (modify-syntax-entry ?>  "." ada-mode-syntax-table)
	    ;;     (from old ada.el)
	    (modify-syntax-entry ?$ "." ada-mode-syntax-table)
	    (modify-syntax-entry ?\[ "." ada-mode-syntax-table)
	    (modify-syntax-entry ?\] "." ada-mode-syntax-table)
	    (modify-syntax-entry ?\{ "." ada-mode-syntax-table)
	    (modify-syntax-entry ?\} "." ada-mode-syntax-table)
	    (modify-syntax-entry ?. "." ada-mode-syntax-table)
	    (modify-syntax-entry ?\\ "." ada-mode-syntax-table)
	    (modify-syntax-entry ?\' "." ada-mode-syntax-table)

            ;; a single hyphen is punctuation, but a double hyphen starts a
	    ;; comment 
	    (modify-syntax-entry ?-  ". 12" ada-mode-syntax-table)
	    ;; and \f and \n end a comment
	    (modify-syntax-entry ?\f  ">   " ada-mode-syntax-table) 
	    (modify-syntax-entry ?\n  ">   " ada-mode-syntax-table) 

	    ;; define what belongs in ada symbols
	    (modify-syntax-entry ?_ "_" ada-mode-syntax-table)

	    ;; Define parentheses to match
	    (modify-syntax-entry ?\( "()" ada-mode-syntax-table)
	    (modify-syntax-entry ?\) ")(" ada-mode-syntax-table)
	    ))
	(run-hooks 'ada-mode-hook)
)    

(defun customized-ada-mode-help ();; for documentation purposes only
  "Ada Mode is the major mode for editing Ada code.

This is the CUSTOMIZED version of the help for Ada mode. The key bindings
you will see are those that are actually in effect at the time you asked
for this help. Unfortunately, this customizing takes a few seconds for
every key sequence that needs looking up, and this message has alot of these. 
You may get faster (but not customized) mode help by typing '\\[describe-mode]'
or you may switch to this window and then print the messaage by typing 
'\\[print-buffer]'.

---------------------- Normal Editing ------------------

This mode automatically performs indentation while editing. Unfortunately,
Ada syntax is too awkward for the indentation to be reliable. Bindings are
as follows: (Note: 'LFD' is control-j.)

    Vanilla Newline                                    '\\[newline]'
    Newline and indent in a 'smart' manner             '\\[indent-ada]'
      Ditto but remain in a comment if you are in one. '\\[indent-new-comment-line]'
    Newline but decrease indent level by fixed amount  '\\[ada-de-dent]'
      or                                               `\\[universal-argument] \\[indent-ada]'
    Move to most recent indent level                   '\\[ada-tab]'
      (if BEFORE the first non blank)
    Move indentation level in by a fixed amount        '\\[ada-tab]'
      (If ON the first non blank)
    Tab over to under non-blanks on the above line     '\\[ada-tab]'
      (if AFTER the first non blank)
    Make this line be indented like the one above it.  '\\[universal-argument] \\[ada-tab]'
    Make this line be indented 1 level less than the   '\\[indent-to-previous-indentation]'
      one above it.

    Indent from (dot) to (mark)			       '\\[indent-region]'
    De-dent from (dot) to (mark)		       '\\[dedent-region]'
    Arbitrarly change indentation of region            '\\[change-indentation]'
    Insert string in same column in all marked lines   '\\[insert-box]'
    Get rid of some columns in all marked lines        '\\[kill-rectangle]'

    Goto Next     Function/Procedure                   '\\[ada-next-procedure]'
    Goto Previous Function/Procedure                   '\\[ada-previous-procedure]'
    Goto Next     Package                              '\\[ada-next-package]'
    Goto Previous Package                              '\\[ada-previous-package]'

Comments are handled using standard GNU Emacs conventions, including:
    Start a comment                                    '\\[indent-for-comment]'
    Continue comment on next line                      '\\[indent-new-comment-line]'

---------------------- Templates -----------------------
Ada Mode also supports a kind of template based editing. Bindings:
    Insert an Ada template                             '\\[insert-token]'
    Expand {token} surrounding or just after(?) point 
       into Ada construct template                     '\\[expand-token]'

A set of 'smart' templates (ones that use handcoded emacs routines rather
  than running from a BNF description) are available under C-ct.
  `\\[ada-header] is particularly useful in that it inserts a standard
  function header. Other options are as follows:
\\{ada-template-map}

---------------------- Online Ada Manual ---------------
The sections of the online ada reference manual may be searched with
  '\\[adaman]'. If this does not get the right section at first,
  \\<text-mode-map>'\\[tags-loop-continue-universal]'\\<ada-mode-map> will get the next matching section.

All the entries in the online ada manual index for a keyword may be found
  with  '\\[index]'. You may step through the entries with '\\[next-error]'.

If all this fails, the entire Ada reference manual may be searched 
  (using grep) by '\\[search-adaman]'.

---------------------- Debugger ------------------------
The '\\[a-db]' command invokes a.db on a binary file.

---------------------- Ada Development -----------------
Standard file headers may be made by '\\[make-header]'. The update count will
   be automatically incremented every time the file is saved. A new file in
   Ada mode and with a standard header may be made by typing 
   '\\[ada-new-file]'. A new set of spec and body files may be made by typing
   typing '\\[ada-new-files]'. '\\[make-revision]' will insert a vevision
   stamp. 

Ada directories may be managed by '\\[dired-ada]'. 'Esc-? m' will give more
   help once this is done. You may invode dired on the directory of this
   file by '\\[dired-ada-buffer]'

On the current buffer:
       Recompilation Order                             '\\[recompilation-order-ada-buffer]'
       Compile                                         '\\[ada-compile-buffer]'
       Make this file                                  '\\[ada-make-buffer]'
       Make and load a ada program unit                '\\[ada-make-unit]'
       Load a program unit                             `\\[dired-ada-link]'
       List VADS units made from this file             '\\[ada-list-buffer]'
       Remove VADS units made from this file           '\\[ada-remove-buffer]'
       Execute a program                               '\\[dired-ada-execute]'
       Pretty Print the file                           <Not Yet Implemented>

Ada errors found during one of the compilations or makes may be scanned by
  the '\\[next-error]' command. This is a VERY useful facility. If the error
  is criptic, you may find the corresponding Ada Manual documentation by
  using the '\\[ada-lookup-last-error]' command.  Both '\\[next-error]' and '\\[ada-lookup-last-error]'
  work from anywhere within emacs.

Provided tags has been loaded (from dired-ada)
  you may look up Ada code by typing '\\[find-tag-universal]'. In case of multiple 
  entries in one file, you may find the other entries by typing '\\[tags-loop-continue-universal]'.
  Typing '\\[backtrack-to-last-user-location]' will pop you back from the tags lookup.
  Tag files are best made from a dired-ada ('\\[dired-ada-buffer]').

---------------------- Customizations ------------------
The variable indent-change defines the size of indentation (the 
  default is 2).

The following local variables can be changed to taste:
  Fill-column       = 75
  Comment-column    = 40

Turning on Ada mode calls the value of the variable ada-mode-hook
  with no args, if that value is non-nil.

See the customizations section of the dired-ada mode documentation.

---------------------- More Help -----------------------
You can get more detailed help on any of these keystrokes 
  by typing 'Esc-? c' and then the keystrokes desired.
For help on functions or M-x sequences, type 'Esc-? f' and the sequence or
  function name. The help sequences are universal across modes.

---------------------- Future Enhancements -------------
  Clean up BNF notation for faster template editing
  Integrate Leif incrimental parsing minor mode
  Take ideas from outline mode to be able to skip levels of code
  Matching if statements and other block statements
  Pretty printing (not done because of no safe pretty printer)
  Integration of source control (Based on Ian G Batten's SCCS utilities)
  Block comment formatting
    we already are like indented text mode as far as auto-fill goes
    Need format paragraph like the recent lisp-fill files do
    Need to place comments such that the VADS pretty printer
       will not rearrange them.
  Mouse support for template selection and tags

Volunteers are gladly accepted.
---------------------- Known Bugs ----------------------

---------------------- Support --------------------------
For help, training, questions, or other support, contact Lynn Slater
  at extension 4482 or send a mail message to lrs@esl.

--------- Summary of Ada Mode Special Commands ----------
\\{ada-mode-map}

------------- Rational for KeyBindings ------------------
In emacs, many keybindings may seem arbitrary. Ada mode tries to avoid this
as much as possible. It may be easier to learn Ada mode if the rational
behind the keymappings were understood. Below is a summary of how the
bindings got to be what they are.

C-c (control-c) is a standard emacs command prefix for language or mode
specific special things. Thus, lots of the Ada specific functions are
bound under C-c in the Ada modes.

The goto-next/previous functions are bound to the keys that are standard
goto-previous/next keys in other modes. In Ada mode, functions are roughly
equivalent to sentences, and packages are roughly equivalent to paragraphs.
Examples:
  'Esc-a' is backward sentence in many modes, but in Ada mode, it is
      previous-function.
  'Esc C-a' is beginning-of-defun in lisp mode, but in Ada mode it is
      previous-package. 
 
The comment manipulation functions are standard emacs bindings.

In most modes, 'Esc .' does a tags lookup. Online Ada documentation is
almost (but not quite) just like a tags table lookup. It is thus bound to
'C-x .' instead. All forms of lookup use 'ESC ,' to goto the next possible
match. All forms also recognize using 'Esc C-l' to backtrack. (You may
think of this binding as Esc-Last.)"
  (interactive)
  (if (eq (current-local-map) ada-mode-map)
      (progn
	(require 'dired-ada)
	(require 'a-db)
	(require 'ada-tag)
	(require 'tags-ada)
	(require 'ada-etempl)
	(message "Am customizing the help message, this will take about 2 minutes.")
	(describe-function 'customized-ada-mode-help))
    (error
      "You need to be in ada-mode for the customizations to be meaningfull!")))


(defun ada-current-expression ()
  "Tries to return the expression at point"
  ;; find the beginning
  (while (progn (backward-char 1) (looking-at "[a-zA-z1-9\\._\)]"))
    (forward-char 1)
    (forward-sexp -1))
  (forward-char 1)
  (let ((start (point)))
    (while (looking-at "[a-zA-z\\._\(]")
      (forward-sexp 1))
    (buffer-substring start (point))))

(defun change-indentation (change)		; indent by arg for region
  "Indents region rigidly, by arg columns.  
A negative arg dedents the region."
  (interactive "p")
  (let ((start (dot))
	(end (mark)))
    (if (> start end)
	(indent-rigidly end start change)
      (indent-rigidly start end change))))

(defun dedent-region nil		; dedents region
  "Dedents region by amount equal to value of indent-change."
  (interactive)
  (change-indentation (- 0 indent-change)))
    
(defun indent-region nil		; indents region
  "Indents region by amount equal to value of indent-change."
  (interactive)
  (change-indentation indent-change))
    
;;(defun ada-tab  nil ;; Used by newline-and-indent, which is used by autofill
;;  "If before the first character, Tabs over to match indentation of
;;      previous line of Ada non-comment code.
;;   Otherwise, it moves the indent level in by the value of 'indent-change"   
;;  (interactive)
;;  (if (or ;; is the line blank and we are at the beginning?
;;       (and (bolp) (eolp))
;;       ;; or we are to the left of the current indentation level
;;       (< (current-column) (current-indentation)))
;;      (let ((colno
;;	     (save-excursion
;;	       (or ;; look backwards for last non-comment line, if any
;;		(re-search-backward
;;		 "^[ \t\f]*\\([^ \t\f\n---].*\\|-\\|-[^---\n].*\\)\n" nil t)
;;		(forward-line -1))
;;	       (current-indentation))))
;;	(delete-horizontal-space)
;;	(indent-to colno))
;;    (progn
;;      (back-to-indentation)
;;      (indent-to (+ indent-change (current-indentation))))))

(defun ada-tab  (&optional arg)
  "Works in 1 of three ways
     If before the first nonblank, 
       makes the line match the indentation of the previous noncomment line.
     If on the first non blank, or if there is a universal ARG,
       moves the line in an indentation level.
     If past the first nonblank, does an indent-relative"
  (interactive "P")
  (cond (arg
	 ;;out->(message "arg")
	 (back-to-indentation)
	 (indent-to (+ indent-change (current-indentation))))

	((point-on-blank-line) (indent-relative))

	((or;; is the line blank and we are at the beginning?
	  (and (bolp) (not (looking-at "[ \t]")))
	  ;; or we are to the left of the current indentation level
	  (< (current-column) (current-indentation)))
	 ;;out->(message "before")
	 (let ((colno
		(save-excursion
		  (or;; look backwards for last non-comment line, if any
		   (re-search-backward
		    "^[ \t\f]*\\([^ \t\f\n---].*\\|-\\|-[^---\n].*\\)\n" nil t)
		   (forward-line -1))
		  (current-indentation))))
	   (delete-horizontal-space)
	   (indent-to (if (= colno 0) indent-change colno))))

	((= (current-column) (current-indentation))
	 ;;out->(message "ON")
	 (indent-to (+ indent-change (current-column))))

	(t
	 ;;out->(message "after")
	 (indent-relative))))

(defun ada-in-dent  nil
  "Enters newline and in-dents for next line of Ada code.
   Sets the indentation level up irregardless of if it is appropiate to the
   current program syntax."
    (interactive)
    (let ((colno (+ (current-indentation) indent-change)))
      (beginning-of-line)
      (if 
	  (not (looking-at "[ \t]*\n"))
	  (progn 
	    (end-of-line)
	    (newline)))
      (delete-horizontal-space)
      (indent-to colno)))

(defun ada-in-dent-current  nil
  "In-dents for next line of Ada code.
   Sets the indentation level up irregardless of if it is appropiate to the
   current program syntax."
   ;; a better behavior might be to  merge the lines, and call indent-ada
    (let ((colno (+ (save-excursion (forward-line -1) (current-indentation))
		    indent-change)))
      (beginning-of-line)
      (delete-horizontal-space)
      (indent-to colno)))

(defun ada-de-dent  nil
  "Enters newline and de-dents (out-dents) for next line of Ada code."
    (interactive)
    (let ((colno (- (current-indentation) indent-change)))
      (beginning-of-line)
      (if 
	  (not (looking-at "[ \t]*\n"))
	  (progn 
	    (end-of-line)
	    (newline)))
      (delete-horizontal-space)
      (indent-to colno)))

(defun point-on-blank-line () ;; moved from ada-etempl
  "Tell whether point is on a blank line or not."
  (save-excursion
    (beginning-of-line)
    (looking-at "^[ \t\f]*$")))

(defun ada-go-to-this-indent (step indent-level) ;; moved from ada-etempl
  "Move point repeatedly by <step> lines till the current line
has given indent-level or less, or the start/end of the buffer is hit.
Ignore blank lines, statement labels, block/loop names."
  (while (and
	  (zerop (forward-line step))
	  (or (looking-at
"^[ \t\f]*$\\|^[ \t\f]*--\\|^<<[A-Za-z0-9_]+>>\\|^[A-Za-z0-9_]+:")
	      (> (current-indentation) indent-level)))
    nil))

(defun indent-to-previous-indentation ()
  "Changes the indentation of the current line to match that of the
   previous line which was not indented as far as the current line had been."
  (interactive)
  (let ((new-i (save-excursion
		 (ada-go-to-this-indent -1 (1- (current-indentation)))
		 (current-indentation))))
    (beginning-of-line)
    (delete-horizontal-space)
    (indent-to new-i)))	
    
(defun newline-indent-plus-change (col)
  (end-of-line)
  (newline)
  (indent-to (+ col indent-change)))

(defun logical-eolp (&optional deletep)
  ;; is like eolp but does not count blanks or tabs.
  ;; if deletep is true, the extra characters are deleted
  (if (looking-at "[ \t]*$")
      (progn
	(if deletep (delete-region (match-beginning 0) (match-end 0))) t)))

(defun in-ada-comment-p ()
  (save-excursion (and (re-search-backward "\\(--\\|\n\\)")
		       (looking-at "-"))))

(defun indent-ada (&optional arg)
  "Enters newline and indents for next line of Ada code.
   Increases the indent level after lines that start with certain Ada 
   keywords such as if, declare, generic, loop, etc.

   If given an argument, decreases the indentation by a fixed amount"
    (interactive "P")
    (let (colno)
	(cond
           (arg (ada-de-dent))	 
	   ((or ;; Are we inside an argument or parm list?
	     ;; I would like to automatically correctly indent parameters and
	     ;; arguments, but I cannot always tell the end of a parameter
	     ;; from the end of a statement.  So, I will conclude that it is
	     ;; a parameter if I was looking at a , or ; and THIS line had a
	     ;; starting paren without a closing paren.
	     (save-excursion
	       (skip-chars-backward " \t\(")
	       (and (looking-at "\(")
                    ;; make this the indentation
		    (setq colno (current-column))))
	     (save-excursion
	       (skip-chars-backward " \t,;\n")
	       (and (looking-at "\\(,\\|\;\\)");;; last char was one of these
		    (progn                       ;;; Have unmatched paren?
		      (re-search-backward "\\((\\|)\\|\n\\)" nil t)
		      ;; Note: this does not skip balanced parens
		      (looking-at "("))
		    ;; make this the indentation
		    (setq colno (current-column)))))
	    ;;out->(message "got arg/parm")
	    (newline)
	    (indent-to (1+ colno))
	    (while (looking-at "\\( \\|\t\\)") (delete-char 1)))

	   ((not (logical-eolp t)) ;; Are we breaking a statement?
	    ;; If we are in the midst of a comment, continue the comment
	    ;; Else indent a little
	    ;;out->(message "line spliting")
	    (cond ((in-ada-comment-p)
		   (indent-new-comment-line))
		  ((save-excursion (skip-chars-backward " \t,;")
			  (looking-at "\\(,\\|;\\)"))
		   ;; we are at a convient breaking point
		   ;;out->(message "convienance break")
		   (setq colno (current-indentation))
		   (newline)
		   (delete-horizontal-space)
		   (indent-to colno))
		  ((<= (current-column) (current-indentation))
		   ;; we are before the first text char. Add a line
		   ;;out->(message "before text")
		   (save-excursion (beginning-of-line) (newline)))
		  (t ;; we are in midst of line
		     (setq colno (current-indentation))
		     ;;out->(message "break default")
		     (newline)
		     (indent-to
		      (+ colno indent-change)))))

	   ((save-excursion ;;; Are we on a blank line?
	                    ;;; (we know that we are eolp)
	      (skip-chars-backward " \t") 
	      (backward-char 1)
	      (looking-at "\n"))
	    ;;out->(message "blankp")
	    ;; indent to match the previous non blank line
	    (newline)
	    (indent-relative t) ;; for consequative c-j's
	    ;;(indent-to-previous-indentation)
	    )

	   ((save-excursion ;;; Are we at end of a statement
	                    ;;; (we know that we are eolp)
	      (skip-chars-backward " \t") 
	      (backward-char 1)
	      (= ?; (following-char)))
	    ;;out->(message "eos")
	    ;; indent to match the previous non blank line
	    (newline)
	    (indent-relative t)
	    )

	   (t ;; start syntax sensitive checks
	     (let ((case-fold-search t))
		   (save-excursion ;; make sure that there is a terminating nl
		       (end-of-line)
		       (if (eobp)
			   (newline)))
		   (setq colno (current-indentation))
		   (beginning-of-line)
		   (cond
		       ((looking-at "[ \t]*--[ \t]*")
			;; we have a pure comment line
			(let ((prefix (buffer-substring (match-beginning 0)
							(match-end 0))))
			  ;; duplicate the indentation and comment header
			  ;; this is almost like indented text mode except
			  ;; that it maintains the comment prefix.
			  (end-of-line)
			  (newline)
			  ;;(ada-tab)
			  (insert prefix)
			  ))

		       ((and comment-start-skip ;; are we in a comment?
			     (re-search-forward comment-start-skip
						(save-excursion
						  (end-of-line) (point)) t))
			;; we are in a comment following code
			;; match indentation.
			;; Better behavior would be to act as if the
			;; comment had not been there.
			;; I.e. take the comment out, indent-ada, and put
			;; it back.
			;;out->(message "follow comment")
			;; maybe the default should indent?
			(setq colno (current-indentation))
			(end-of-line)
			(newline)
			(indent-to colno)
			)

		       ((looking-at "[ \t]*end\\b")
			(progn 
			       (end-of-line)
			       (newline)
                                (indent-to (- colno indent-change))
			       ))

		       ;; These keywords could be combined into 1 big regexp
		       ((or (looking-at "[ \t]*begin\\b")
			    (looking-at "[ \t]*when\\b") 
			    (looking-at "[ \t]*generic\\b")
			    (looking-at "[ \t]*declare\\b")
			    (looking-at "[ \t]*loop\\b")
			    (looking-at "[ \t]*if\\b")
			    (looking-at "[ \t]*elsif\\b")
			    (looking-at "[ \t]*else\\b")
			    (looking-at "[ \t]*then\\b")
			    (looking-at "[ \t]*while\\b")
			    (looking-at "[ \t]*case\\b")
			    (looking-at "[ \t]*loop\\b")
			    (looking-at "[ \t]*or\\b")
			    (looking-at "[ \t]*exception\\b")
			    (looking-at "[ \t]*record\\b")
			    (looking-at "[ \t]*private\\b")
			    (looking-at "[ \t]*<<.*>>")
			    (looking-at "[ \t]*select\\b"))
			;;out->(message "big regexp")
			(newline-indent-plus-change colno))

		       ((looking-at "[ \t]*for\\b")
			(progn 
			       (end-of-line)
			       (if (= (preceding-char) ?\;)
				   (newline-and-indent)
				   (progn 
				      (newline)
				      (indent-to
				       (+ colno indent-change))))))

		       ((or (looking-at "[ \t]*type\\b")
			   (looking-at "[ \t]*subtype\\b"))
			(progn 
			       (end-of-line)
			       (backward-word 1)
			       (cond
				((looking-at "record\\b")
				 (progn 
				   (setq colno (current-column))
				   (newline-indent-plus-change colno)))
				((looking-at "is\\b")
				 (newline-indent-plus-change colno))
				(t
				 (progn 
				   (end-of-line)
				   (newline-and-indent))))))

		       ((looking-at "[ \t]*accept\\b")
			(progn 
			       (end-of-line)
			       (backward-word 1)
			       (if (looking-at "do\\b")
				   (newline-indent-plus-change colno)
				   (progn 
					  (end-of-line)
					  (newline-and-indent)))))

		       ((looking-at "[ \t]*task\\b")
			(progn 
			       (end-of-line)
			       (backward-word 1)
			       (if (looking-at "is\\b")
				   (newline-indent-plus-change colno)
				   (progn 
					  (end-of-line)
					  (newline-and-indent)))))

		       ((or (looking-at "[ \t]*procedure\\b")
			   (looking-at "[ \t]*function\\b"))
			(progn 
			       (end-of-line)
			       (backward-word 1)
			       (if (looking-at "is\\b")
				   (newline-indent-plus-change colno)
				   (progn 
					  (end-of-line)
					  (newline-and-indent)))))

		       ((looking-at ".*:[^(]*)")
					; catches multi-line proc/func formals;
					; unfortunately, could also catch
					; multi-line discriminant part of type.
			(let ((colno (ada-indent-after-heading)))
			  (end-of-line)
			  (newline)
			  (indent-to colno)))

					;
		       ((looking-at "[ \t]*package\\b")
			(progn 
			       (end-of-line)
			       (backward-word 1)
			       (if (looking-at "is\\b")
				   (newline-indent-plus-change colno)
				   (progn 
					  (end-of-line)
					  (newline-and-indent)))))

		       (t ;; if ends in ; no indent
			  ;; else indent
			  ;;out->(message "default case")
			  ;; maybe the default should indent?
			  (setq colno (+ indent-change (current-indentation)))
			  (end-of-line)
			  (newline)
			  (indent-to colno))))))))

;; defects
;;
;; Indent right after "(" should match to the paren -- fixed
;; Indent after end of a broken statement should go back in indent levels
;;   I have no way to determine this condition.

(defun ada-tabsize (s)
  "changes spacing used for indentation. Reads spacing from minibuffer."
  (interactive "nnew indentation spacing: ")
  (setq indent-change s))

;; lrs enhancements
;; ada-procedure-start-regexp is defined in ada-etempl
(defun ada-next-procedure ()
  (interactive)
  (end-of-line)
  (if (re-search-forward ada-procedure-start-regexp nil t)
      (goto-char (match-beginning 1))
    (error "No more functions/procedures")))

(defun ada-previous-procedure ()
  (interactive)
  (beginning-of-line)
  (if (re-search-backward ada-procedure-start-regexp nil t)
      (goto-char (match-beginning 1))
    (error "No more functions/procedures")))

(defvar ada-package-start-regexp "^[ \t]*\\(package\\)")
(defun ada-next-package ()
  (interactive)
  (end-of-line)
  (if (re-search-forward ada-package-start-regexp nil t)
      (goto-char (match-beginning 1))
    (error "No more packages")))

(defun ada-previous-package ()
  (interactive)
  (beginning-of-line)
  (if (re-search-backward ada-package-start-regexp nil t)
      (goto-char (match-beginning 1))
    (error "No more packages")))

;; The below does not work yet
;;(setq ada-if-statement-regexp
;;  "[ \t]+\\(if\\([ \t]\\|(\\)\\|end[ \t]*if;\\|\\-\\-\\)")
;;(defun ada-next-if-boundry ()
;;  (if (looking-at ada-if-statement-regexp)
;;      (re-search-forward ada-if-statement-regexp))
;;  (re-search-forward ada-if-statement-regexp)
;;  ;; skip ifs in comments
;;  (goto-char (match-beginning 1))
;;  (if (looking-at "\\-\\-")
;;      (progn (end-of-line)(ada-next-if-boundry)))
;;  )
;;
;;(defun ada-forward-if-statement ()
;;  "Skips to the end if the next if statement"
;;  ;; assume are looking at an if
;;  ;; get the next stopper
;;  (ada-next-if-boundry)
;;  (message "after skip in forward") (sit-for 3)
;;  (if (looking-at "if")
;;      (while (looking-at "if")
;;	(ada-forward-if-statement)
;;	(ada-next-if-boundry);; find the matching end
;;	)
;;    (goto-char (match-end 1))))
;;
;;(defun move-forward-ada-if-statement ()
;;  (interactive)
;;  ;; find the first if
;;  ;;(set-mark-command (point))
;;  (ada-next-if-boundry)
;;  (if (looking-at "if")
;;      (ada-forward-if-statement)
;;    (error "You are already in the middle of an If statement.")))
  
(defun ada-new-files  (file-name doc)
  "Makes a spec and body file with standard headers."
  (interactive "sCore name of new spec and body files: \nsDoc String: ")
  (if (string-match "\\." file-name)
      (error "You must supply a name without suffixes!"))
  (let ((spec (concat file-name ada-spec-suffix))
	(body (concat file-name ada-body-suffix)))
    (if (or (file-exists-p spec) (file-exists-p body))
	(error "One or both spec and body files already exist!"))
    (ada-new-file body doc)
    (ada-new-file spec doc)
    (message "New files Created")
    ))

(defun ada-new-file (file-name doc)
  "Makes a new file in ada mode with a standard header."
  (interactive "sFull name of new file: \nsDoc String: ")
  (require 'header)
  (if (file-exists-p file-name)
      (error "File already exists!"))
  (let ((buf (get-buffer-create file-name)))
    (switch-to-buffer buf)
    )
  (ada-mode)
  (make-header)
  (insert doc)
  (write-file file-name)
  (end-of-buffer)
  )

(if (not ada-mode-map)
      (progn
	(setq ada-mode-map (make-keymap))
	;; Let c-h, del, and whatever the help-character might be
	;; have their global meanings in this mode.
	(define-key ada-mode-map "\C-c>"    'indent-region)
	(define-key ada-mode-map "\C-c<"    'dedent-region)
	(define-key ada-mode-map "\t"      'ada-tab)
	(define-key ada-mode-map "\e\t"    'indent-to-previous-indentation)
	(define-key ada-mode-map "\C-j"    'indent-ada)
	(define-key ada-mode-map "\e\C-j"   'ada-de-dent)
	(define-key ada-mode-map "\e\j"   'indent-new-comment-line)
	(define-key ada-mode-map "\e\C-M"     'newline)

	(define-key ada-mode-map "\C-ci"   'insert-token)
	(define-key ada-mode-map "\C-ce"   'expand-token)

	;; Movement
	(define-key ada-mode-map "\ee"   'ada-next-procedure)
	(define-key ada-mode-map "\ea"   'ada-previous-procedure)
	(define-key ada-mode-map "\e\C-e"   'ada-next-package)
	(define-key ada-mode-map "\e\C-a"   'ada-previous-package)

	;; Template support
	(define-key ada-mode-map "\C-ce" 'expand-token)

	;; Online Ada manual
	(define-key ada-mode-map "\C-x." 'adaman)
	(define-key ada-mode-map "\C-c." 'ada-command)
      
	;; now, put in dired support
	(define-key ada-mode-map "\C-cx" 'dired-ada-execute)
	(define-key ada-mode-map "\C-cr"   'recompilation-order-ada-buffer)
	(define-key ada-mode-map "\C-c\C-r" 'ada-remove-buffer)
	(define-key ada-mode-map "\C-cm"   'ada-make-buffer)
	(define-key ada-mode-map "\C-c\C-M" 'ada-make-unit)
	(define-key ada-mode-map "\C-cl"   'ada-list-buffer)
	(define-key ada-mode-map "\C-ck"    'dired-ada-link)
	(define-key ada-mode-map "\C-cd"   'dired-ada-buffer)
	(define-key ada-mode-map "\C-c\C-d"   'dired-ada-buffer)
	(define-key ada-mode-map "\C-cc"   'ada-compile-buffer)
	(define-key ada-mode-map "\C-c\C-c"   'ada-compile-buffer)

	;; tags support
	(define-key ada-mode-map "\C-c\C-t"   'get-tags-for-ada-buffer)

	;; file creation
	(define-key ada-mode-map "\C-cn" 'ada-new-file)
	(define-key ada-mode-map "\C-c\C-n" 'ada-new-files)

	;; Ada inserts lots of tabs without notice.  They should
	;;   be made to go away also without notice
        (define-key ada-mode-map "\177" 'backward-delete-char-untabify)	
	;; Disable some nasty functions
	(define-key  ada-mode-map "\eq" 'undefined)
	))

; ----------------------------------------------------------------------------
; Define some support functions
; ----------------------------------------------------------------------------
(defun insert-box (start end text)
  "Insert a text prefix at a column in all the lines in the region.
   Called from a program, takes three arguments, START, END, and TEXT.
   The column is taken from that of START.
   The rough inverse of this function is kill-rectangle."
  (interactive "r\nsText To Insert: ")
  (save-excursion
    (let (cc)
      ;; the point-marker stuff is needed to keep the edits from changing
      ;; where end is
      (goto-char end)
      (setq end (point-marker))
      (goto-char start)
      (setq cc  (current-column))
      (while (< (point) end) ;; modified 2/2/88
	;; I should here check for tab chars
	(insert text)
	(forward-line 1)
	(move-to-column cc))
      (move-marker end nil))))

(defun insert-end (start end text)
  "Insert a text prefix at the end in all the lines in the region.
   Called from a program, takes three arguments, START, END, and TEXT.
   The column is taken from that of START."
  (interactive "r\nsText To Insert: ")
  (save-excursion
    (let (cc)
      ;; the point-marker stuff is needed to keep the edits from changing
      ;; where end is
      (goto-char end)
      (setq end (point-marker))
      (goto-char start)
      (end-of-line)	
      (while (< (point) end);; modified 2/2/88
	;; I should here check for tab chars
	(insert text)
	(forward-line 1)
	(end-of-line)	
	)
      (move-marker end nil))))

(defun zap-crs ()
  "Removes control-m's at the end of lines.
   Does not change modified status or generate autosave files
   Is used if you are browsing files loaded by tape from certain systems
   (like the simtel contributed ada files viewed from a sun)
   and the control-m's mess up your terminal (like a wyse-50)."
  (interactive)
  (save-excursion
    (beginning-of-buffer)
    (let (;; Do not change the modified flag
	  (modified-flag  (buffer-modified-p))
	  ;; shadow the true local value of read-only
	  (buffer-read-only nil)
	  ;; Prevent the generation of autosave files
	  ;; Don't you just love dynamic variable binding?
	  (buffer-auto-save-file-name nil)
	  ;; we do not have to worry about backup files because
	  ;; they are not generated unless the buffer is saved.
	  )
      (message "Eliminating ending 
's . . .")
      (replace-regexp "
$" "" nil)
      (message "Eliminating ending 
's . . . DONE")
      (set-buffer-modified-p modified-flag))))


; ----------------------------------------------------------------------------
(require 'ada-etempl) 
(autoload 'expand-token "ada-templ"
  "Expands token that point is inside or just after.  A window will be
popped up containing the alternative template expansions of the token.
Once a template is selected, it will replace the token in the current
buffer.  For convenience, the mark is pushed at the beginning of the
inserted template, and point is left at the end."
	  t)

(autoload 'insert-token "ada-templ"
"Inserts a valid Ada language template and expands it."
	  t)

(autoload 'a-db "a-db"
	  "Run a.db on program FILE in buffer *ada-db-FILE*."
	  t)


;;(autoload 'ada-menu-activate "~lrs/emacs/ada-menu"
;;	  "Activates the ada mode menu.")

;; Below is code that does not yet work. I was hoping to make automatic
;; specs from a body or to make automatic body stubs from a spec.
;;(defun make-spec ()
;;  (interactive)
;;  (re-search-forward "\\(procedure\\|function\\)[ \t\n]+.*[ \t\n]+is" nil t)
;;  (goto-char (match-beginning 0))
;;  (insert
;;   (concat
;;    "  -- Taken From: " (buffer-file-name))
;;    "  --       On  : " (current-time-string)
;;    "  --       By  : " (user-full-name)
;;    "\n  "
;;    (buffer-substring (match-beginning 0) (- (match-end 0) 2))
;;    ";\n\n"))
;;  
;;
;;(defun get-spec ()
;;  "Returns a string corresponding to the spec of the next ada fcn
;;   Moves past the fcn"
;;  (save-excursion
;;    (if (re-search-forward "\\(procedure\\|function\\)[ \t\n]+.*[ \t\n]+is" nil t)
;;	(let ((start (match-beginning 0)))
;;	  (goto-char (- (match-end 0) 2))
;;	  (skip-chars-backward " \t\n")
;;	  (concat (buffer-substring start (point)) ";")
;;	  ))))
