;;!emacs
;;
;; FILE:         hinit.el
;; SUMMARY:      Standard initializations for Hyperbole hypertext system.
;; USAGE:        GNU Emacs Lisp Library
;;
;; AUTHOR:       Bob Weiner
;; ORG:          Brown U.
;;
;; ORIG-DATE:     1-Oct-91 at 02:32:51
;; LAST-MOD:     14-Dec-91 at 02:07:36 by Bob Weiner
;;
;; This file is part of Hyperbole.
;; 
;; Copyright (C) 1991, Brown University, Providence, RI
;; Developed with support from Motorola Inc.
;; 
;; Permission to use, modify and redistribute this software and its
;; documentation for any purpose other than its incorporation into a
;; commercial product is hereby granted without fee.  A distribution fee
;; may be charged with any redistribution.  Any distribution requires
;; that the above copyright notice appear in all copies, that both that
;; copyright notice and this permission notice appear in supporting
;; documentation, and that neither the name of Brown University nor the
;; author's name be used in advertising or publicity pertaining to
;; distribution of the software without specific, written prior permission.
;; 
;; Brown University makes no representations about the suitability of this
;; software for any purpose.  It is provided "as is" without express or
;; implied warranty.
;;
;;
;; DESCRIPTION:  
;; DESCRIP-END.

;;; ************************************************************************
;;; Public variables
;;; ************************************************************************

(defconst hyperb:version "2.02.1" "Hyperbole revision number.")

(defvar   hyperb:host-domain nil
  "<@domain-name> for current host.  Set automatically by 'hyperb:init'.")

;;; ************************************************************************
;;; Required Init functions
;;; ************************************************************************

(defun var:append (var-symbol-name list-to-add)
  "If VAR-SYMBOL-NAME is bound, append LIST-TO-ADD (a list) to value of
variable (a list) given by var-symbol-name.  If unbound, variable is set
to list-to-add.  Often used to append to 'hook' variables."
  (set-variable var-symbol-name
		(append (and (boundp var-symbol-name)
			     (symbol-value var-symbol-name))
			list-to-add)))

(defun first-line-p ()
  "Returns true if point is on the first line of the buffer."
  (save-excursion (beginning-of-line) (bobp)))

(defun last-line-p ()
  "Returns true if point is on the last line of the buffer."
  (save-excursion (end-of-line) (eobp)))

;;; ************************************************************************
;;; Other required Elisp libraries
;;; ************************************************************************

(if (fboundp 'smart-menu)
    (require 'hmouse-drv)  ;; Needed to load 'smart-key-help-show'.
  (require 'hui-mouse))
(require 'hypb) (require 'hui-menus)
(require 'set) (require 'hbmap) (require 'hibtypes)

;;; ************************************************************************
;;; Public functions
;;; ************************************************************************

(defun hyperb:init ()
  "Standard configuration routine for Hyperbole."
  (interactive)
  (hyperb:check-dir-user)
  (or hyperb:host-domain (setq hyperb:host-domain (hypb:domain-name)))
  (hyperb:act-set)
  ;;
  ;; Highlight explicit buttons whenever a file is read in.
  ;;
  (if hyperb:epoch-p (var:append 'find-file-hooks '(ep:but-create)))
  ;;
  ;; Save button attribute file whenever same dir file is saved and
  ;; 'ebut:hattr-save' is non-nil.
  ;;
  (var:append 'write-file-hooks '(hattr:save))
  (run-hooks 'hyperb:init-hook)
)

(defun hyperb:act-set ()
  "COORDINATION IS NOT YET OPERATIONAL.  hui-coord.el IS NOT INCLUDED.
Sets Hyperbole action command to uncoordinated or coordinated operation.
Coordinated is used when 'hcoord:hosts' is a non-nil list.
See \"hui-coord.el\"."
  (interactive)
  (fset 'hyperb:act (if (and (boundp 'hcoord:hosts) hcoord:hosts)
		     'hcoord:act 'hbut:act)))

(defun hyperb:action-v1 ()
  "Signals error if Version 1 Hyperbole link button is selected."
  (error "(hyperb:action-v1): Hyperbole version 1 buttons are not supported."))

;;; ************************************************************************
;;; Private functions
;;; ************************************************************************

(defun hyperb:check-dir-user ()
  "Ensures 'hbmap:dir-user' exists and is writable or signals an error."
  (if (or (null hbmap:dir-user) (not (stringp hbmap:dir-user))
	  (and (setq hbmap:dir-user (file-name-as-directory
				     (expand-file-name hbmap:dir-user)))
	       (file-directory-p hbmap:dir-user)
	       (not (file-writable-p hbmap:dir-user))))
      (error
       "(hyperb:init): 'hbmap:dir-user' must be a writable directory name."))
  (let ((hbmap:dir-user (directory-file-name hbmap:dir-user)))
    (or (file-directory-p hbmap:dir-user)   ;; Exists and is writable.
	(let* ((parent-dir (file-name-directory
			    (directory-file-name hbmap:dir-user))))
	  (cond
	   ((not (file-directory-p parent-dir))
	    (error
	     "(hyperb:init): 'hbmap:dir-user' parent dir does not exist."))
	   ((not (file-writable-p parent-dir))
	    (error
	     "(hyperb:init): 'hbmap:dir-user' parent directory not writable."))
	   ((hypb:call-process-p "mkdir" nil nil hbmap:dir-user)
	    (or (file-writable-p hbmap:dir-user)
		(or (progn (hypb:chmod '+ 700 hbmap:dir-user)
			   (file-writable-p hbmap:dir-user))
		    (error "(hyperb:init): Can't write to 'hbmap:dir-user'.")
		    )))
	   (t (error "(hyperb:init): 'hbmap:dir-user' create failed."))))))
  t)

(provide 'hinit)

(hyperb:init)
