/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  CreditDebitIndicator,
  Logger,
  TalerCorebankApiClient,
  TalerWireGatewayHttpClient,
  TransactionMajorState,
  TransactionMinorState,
  createEddsaKeyPair,
  encodeCrock,
  j2s,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { defaultCoinConfig } from "../harness/denomStructures.js";
import { createWalletDaemonWithClient } from "../harness/environments.js";
import {
  ExchangeService,
  GlobalTestState,
  LibeufinBankService,
  MerchantService,
  getTestHarnessPaytoForLabel,
  setupDb,
} from "../harness/harness.js";

const logger = new Logger("test-libeufin-bank.ts");

/**
 * Run test for the basic functionality of libeufin-bank.
 */
export async function runLibeufinBankTest(t: GlobalTestState) {
  // Set up test environment

  const db = await setupDb(t);

  const bank = await LibeufinBankService.create(t, {
    currency: "TESTKUDOS",
    httpPort: 8082,
    database: db.connStr,
    allowRegistrations: true,
  });

  const exchange = ExchangeService.create(t, {
    name: "testexchange-1",
    currency: "TESTKUDOS",
    httpPort: 8081,
    database: db.connStr,
  });

  const merchant = await MerchantService.create(t, {
    name: "testmerchant-1",
    httpPort: 8083,
    database: db.connStr,
  });

  const exchangeBankUsername = "exchange";
  const exchangeBankPw = "mypw-password";
  const exchangePayto = getTestHarnessPaytoForLabel(exchangeBankUsername);
  const wireGatewayApiBaseUrl = new URL(
    "accounts/exchange/taler-wire-gateway/",
    bank.baseUrl,
  ).href;

  logger.info("creating bank account for the exchange");

  exchange.addBankAccount("1", {
    wireGatewayApiBaseUrl,
    wireGatewayAuth: {
      username: exchangeBankUsername,
      password: exchangeBankPw,
    },
    accountPaytoUri: exchangePayto,
  });

  bank.setSuggestedExchange(exchange);

  await bank.start();

  await bank.pingUntilAvailable();

  exchange.addOfferedCoins(defaultCoinConfig);

  await exchange.start();
  await exchange.pingUntilAvailable();

  merchant.addExchange(exchange);

  await merchant.start();
  await merchant.pingUntilAvailable();

  const { accessToken: adminAccessToken } = await merchant.addInstanceWithWireAccount({
    id: "admin",
    name: "Default Instance",
    paytoUris: [getTestHarnessPaytoForLabel("merchant-default")],
  });

  await merchant.addInstanceWithWireAccount({
    id: "minst1",
    name: "minst1",
    paytoUris: [getTestHarnessPaytoForLabel("minst1")],
  }, {adminAccessToken});

  const { walletClient } = await createWalletDaemonWithClient(t, {
    name: "wallet",
  });

  console.log("setup done!");

  const adminUser = "admin";
  const adminPassword = "admin-password";

  const bankClient = new TalerCorebankApiClient(bank.corebankApiBaseUrl, {
    auth: {
      username: adminUser,
      password: adminPassword,
    },
  });

  // register exchange bank account
  await bankClient.registerAccountExtended({
    name: "Exchange",
    password: exchangeBankPw,
    username: exchangeBankUsername,
    is_taler_exchange: true,
    payto_uri: exchangePayto,
  });

  const bankUser = await bankClient.registerAccount("user1", "password1");
  bankClient.setAuth({
    username: "user1",
    password: "password1",
  });

  // Make sure that registering twice results in a 409 Conflict
  // {
  //   const e = await t.assertThrowsTalerErrorAsync(async () => {
  //     await bankClient.registerAccount("user1", "password2");
  //   });
  //   t.assertTrue(e.errorDetail.httpStatusCode === 409);
  // }

  let balResp = await bankClient.getAccountBalance(bankUser.username);

  console.log(balResp);

  // Check that we got the sign-up bonus.
  t.assertAmountEquals(balResp.balance.amount, "TESTKUDOS:100");
  t.assertTrue(
    balResp.balance.credit_debit_indicator === CreditDebitIndicator.Credit,
  );

  const res = createEddsaKeyPair();

  // Not a normal client, but one with admin credentials,
  // as /add-incoming is testing functionality only allowed by the admin.
  const wireGatewayApiAdminClient = new TalerWireGatewayHttpClient(
    wireGatewayApiBaseUrl,
  );

  await wireGatewayApiAdminClient.addIncoming({
    auth: {
      username: adminUser,
      password: adminPassword,
    },
    body: {
      amount: "TESTKUDOS:115",
      debit_account: bankUser.accountPaytoUri,
      reserve_pub: encodeCrock(res.eddsaPub),
    },
  });

  balResp = await bankClient.getAccountBalance(bankUser.username);
  t.assertAmountEquals(balResp.balance.amount, "TESTKUDOS:15");
  t.assertTrue(
    balResp.balance.credit_debit_indicator === CreditDebitIndicator.Debit,
  );

  const wop = await bankClient.createWithdrawalOperation(
    bankUser.username,
    "TESTKUDOS:10",
  );

  const r1 = await walletClient.client.call(
    WalletApiOperation.GetWithdrawalDetailsForUri,
    {
      talerWithdrawUri: wop.taler_withdraw_uri,
    },
  );

  console.log(j2s(r1));

  const r2 = await walletClient.client.call(
    WalletApiOperation.AcceptBankIntegratedWithdrawal,
    {
      exchangeBaseUrl: exchange.baseUrl,
      talerWithdrawUri: wop.taler_withdraw_uri,
    },
  );

  await walletClient.call(WalletApiOperation.TestingWaitTransactionState, {
    transactionId: r2.transactionId,
    txState: {
      major: TransactionMajorState.Pending,
      minor: TransactionMinorState.BankConfirmTransfer,
    },
  });

  await bankClient.confirmWithdrawalOperation(bankUser.username, {
    withdrawalOperationId: wop.withdrawal_id,
  });

  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});
}

runLibeufinBankTest.suites = ["fakebank"];
