/*
 This file is part of GNU Taler
 (C) 2022-2025 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
import {
  Button,
  FormDesign,
  InputLine,
  InternationalizationAPI,
  LocalNotificationBanner,
  useForm,
  useLocalNotificationHandler,
  useTranslationContext,
} from "@gnu-taler/web-util/browser";
import { VNode, h } from "preact";
import { useOfficer } from "../hooks/officer.js";

type FormType = {
  password: string;
};

const TALER_SCREEN_ID = 119;

const unlockAccountForm = (i18n: InternationalizationAPI): FormDesign => ({
  type: "single-column",
  fields: [
    {
      id: "password",
      type: "text",
      label: i18n.str`Password`,
      required: true,
    },
  ],
});

export function UnlockAccount(): VNode {
  const { i18n } = useTranslationContext();

  const officer = useOfficer();
  const [notification, withErrorHandler] = useLocalNotificationHandler();

  const design = unlockAccountForm(i18n);

  const { model: handler, status } = useForm<FormType>(design, {
    password: undefined,
  });

  const unlockHandler =
    status.status === "fail" || officer.state !== "locked"
      ? undefined
      : withErrorHandler(
          async () => officer.tryUnlock(status.result.password),
          () => {},
        );

  const forgetHandler =
    officer.state === "not-found"
      ? undefined
      : withErrorHandler(
          async () => officer.forget(),
          () => {},
        );

  return (
    <div class="flex min-h-full flex-col ">
      <LocalNotificationBanner notification={notification} />

      <div class="sm:mx-auto sm:w-full sm:max-w-md">
        <h1 class="mt-6 text-center text-2xl font-bold leading-9 tracking-tight text-gray-900">
          <i18n.Translate>Account locked</i18n.Translate>
        </h1>
        <p class="mt-6 text-lg leading-8 text-gray-600">
          <i18n.Translate>
            Your account is normally locked anytime you reload. To unlock type
            your password again.
          </i18n.Translate>
        </p>
      </div>

      <div class="mt-10 sm:mx-auto sm:w-full sm:max-w-[480px] ">
        <div class="bg-gray-100 px-6 py-6 shadow sm:rounded-lg sm:px-12">
          <div class="mb-4">
            <InputLine
              label={i18n.str`Password`}
              name="password"
              type="password"
              required
              handler={handler.getHandlerForAttributeKey("password")}
            />
          </div>

          <div class="mt-8">
            <Button
              type="submit"
              handler={unlockHandler}
              disabled={!unlockHandler}
              class="disabled:opacity-50 disabled:cursor-default  flex w-full justify-center rounded-md bg-indigo-600 px-3 py-1.5 text-sm font-semibold leading-6 text-white shadow-sm hover:bg-indigo-500 focus-visible:outline focus-visible:outline-2 focus-visible:outline-offset-2 focus-visible:outline-indigo-600"
            >
              <i18n.Translate>Unlock</i18n.Translate>
            </Button>
          </div>
        </div>
        <Button
          type="button"
          handler={forgetHandler}
          disabled={!forgetHandler}
          class="disabled:opacity-50 disabled:cursor-default m-4 block rounded-md bg-red-600 px-3 py-2 text-center text-sm  text-white shadow-sm hover:bg-red-500 "
        >
          <i18n.Translate>Forget account</i18n.Translate>
        </Button>
      </div>
    </div>
  );
}
