/*
  This file is part of TALER
  Copyright (C) 2024, 2025 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify it under the
  terms of the GNU Affero General Public License as published by the Free Software
  Foundation; either version 3, or (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU Affero General Public License for more details.

  You should have received a copy of the GNU Affero General Public License along with
  TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
*/
/**
 * @file donau-httpd_charity_insert.c
 * @brief Handle request to insert a charity.
 * @author Johannes Casaburi
 * @author Christian Grothoff
 */
#include <donau_config.h>
#include <gnunet/gnunet_util_lib.h>
#include <gnunet/gnunet_json_lib.h>
#include <jansson.h>
#include <microhttpd.h>
#include <pthread.h>
#include <taler/taler_json_lib.h>
#include <taler/taler_mhd_lib.h>
#include <taler/taler_signatures.h>
#include "donaudb_plugin.h"
#include "donau-httpd_charity.h"
#include "donau-httpd_db.h"


MHD_RESULT
DH_handler_charity_post (struct DH_RequestContext *rc,
                         const json_t *root,
                         const char *const args[])
{
  struct DONAU_CharityPublicKeyP charity_pub;
  const char *charity_name;
  const char *charity_url;
  struct TALER_Amount max_per_year;
  struct GNUNET_JSON_Specification spec[] = {
    GNUNET_JSON_spec_fixed_auto ("charity_pub",
                                 &charity_pub),
    GNUNET_JSON_spec_string ("charity_name",
                             &charity_name),
    GNUNET_JSON_spec_string ("charity_url",
                             &charity_url),
    TALER_JSON_spec_amount ("max_per_year",
                            DH_currency,
                            &max_per_year),
    GNUNET_JSON_spec_end ()
  };
  enum GNUNET_DB_QueryStatus qs;
  uint64_t charity_id;

  (void) args;
  {
    enum GNUNET_GenericReturnValue res;

    res = TALER_MHD_parse_json_data (rc->connection,
                                     root,
                                     spec);
    if (GNUNET_SYSERR == res)
      return MHD_NO; /* hard failure */
    if (GNUNET_NO == res)
    {
      GNUNET_break_op (0);
      return MHD_YES; /* failure */
    }
  }
  qs = DH_plugin->insert_charity (DH_plugin->cls,
                                  &charity_pub,
                                  charity_name,
                                  charity_url,
                                  &max_per_year,
                                  &charity_id);
  switch (qs)
  {
  case GNUNET_DB_STATUS_SOFT_ERROR:
    GNUNET_break (0);
    return TALER_MHD_reply_with_error (rc->connection,
                                       MHD_HTTP_INTERNAL_SERVER_ERROR,
                                       TALER_EC_GENERIC_DB_STORE_FAILED,
                                       "insert_charity");
  case GNUNET_DB_STATUS_HARD_ERROR:
    GNUNET_break (0);
    return TALER_MHD_reply_with_error (rc->connection,
                                       MHD_HTTP_INTERNAL_SERVER_ERROR,
                                       TALER_EC_GENERIC_DB_STORE_FAILED,
                                       "insert_charity");
  case GNUNET_DB_STATUS_SUCCESS_NO_RESULTS:
    GNUNET_break (0);
    return TALER_MHD_reply_with_error (rc->connection,
                                       MHD_HTTP_CONFLICT,
                                       TALER_EC_DONAU_CHARITY_PUB_EXISTS,
                                       NULL);
    return GNUNET_DB_STATUS_HARD_ERROR;
  case GNUNET_DB_STATUS_SUCCESS_ONE_RESULT:
    break;
  }
  return TALER_MHD_REPLY_JSON_PACK (
    rc->connection,
    MHD_HTTP_CREATED,
    GNUNET_JSON_pack_uint64 ("charity_id",
                             charity_id));
}


/* end of donau-httpd_charity_insert.c */
