      SUBROUTINE DLSODE (F, NEQ, Y, T, TOUT, ITOL, RTOL, ATOL, ITASK,
     1            ISTATE, IOPT, RWORK, LRW, IWORK, LIW, JAC, MF)
      EXTERNAL F, JAC
      INTEGER NEQ, ITOL, ITASK, ISTATE, IOPT, LRW, IWORK, LIW, MF
      DOUBLE PRECISION Y, T, TOUT, RTOL, ATOL, RWORK
      DIMENSION NEQ(*), Y(*), RTOL(*), ATOL(*), RWORK(LRW), IWORK(LIW)
C-----------------------------------------------------------------------
C THIS IS THE MARCH 30, 1987 VERSION OF 
C LSODE.. LIVERMORE SOLVER FOR ORDINARY DIFFERENTIAL EQUATIONS.
C THIS VERSION IS IN DOUBLE PRECISION.
C
C LSODE SOLVES THE INITIAL VALUE PROBLEM FOR STIFF OR NONSTIFF
C SYSTEMS OF FIRST ORDER ODE-S,
C     DY/DT = F(T,Y) ,  OR, IN COMPONENT FORM,
C     DY(I)/DT = F(I) = F(I,T,Y(1),Y(2),...,Y(NEQ)) (I = 1,...,NEQ).
C LSODE IS A PACKAGE BASED ON THE GEAR AND GEARB PACKAGES, AND ON THE 
C OCTOBER 23, 1978 VERSION OF THE TENTATIVE ODEPACK USER INTERFACE
C STANDARD, WITH MINOR MODIFICATIONS.
C-----------------------------------------------------------------------
C REFERENCE..
C     ALAN C. HINDMARSH,  ODEPACK, A SYSTEMATIZED COLLECTION OF ODE
C     SOLVERS, IN SCIENTIFIC COMPUTING, R. S. STEPLEMAN ET AL. (EDS.),
C     NORTH-HOLLAND, AMSTERDAM, 1983, PP. 55-64.
C-----------------------------------------------------------------------
C AUTHOR AND CONTACT.. ALAN C. HINDMARSH,
C                      COMPUTING AND MATHEMATICS RESEARCH DIV., L-316 
C                      LAWRENCE LIVERMORE NATIONAL LABORATORY
C                      LIVERMORE, CA 94550.
C-----------------------------------------------------------------------
C SUMMARY OF USAGE. 
C
C COMMUNICATION BETWEEN THE USER AND THE LSODE PACKAGE, FOR NORMAL
C SITUATIONS, IS SUMMARIZED HERE.  THIS SUMMARY DESCRIBES ONLY A SUBSET
C OF THE FULL SET OF OPTIONS AVAILABLE.  SEE THE FULL DESCRIPTION FOR 
C DETAILS, INCLUDING OPTIONAL COMMUNICATION, NONSTANDARD OPTIONS,
C AND INSTRUCTIONS FOR SPECIAL SITUATIONS.  SEE ALSO THE EXAMPLE
C PROBLEM (WITH PROGRAM AND OUTPUT) FOLLOWING THIS SUMMARY. 
C
C A. FIRST PROVIDE A SUBROUTINE OF THE FORM..
C               SUBROUTINE F (NEQ, T, Y, YDOT, IERR)
C               DIMENSION Y(NEQ), YDOT(NEQ)
C WHICH SUPPLIES THE VECTOR FUNCTION F BY LOADING YDOT(I) WITH F(I).
C
C B. NEXT DETERMINE (OR GUESS) WHETHER OR NOT THE PROBLEM IS STIFF.
C STIFFNESS OCCURS WHEN THE JACOBIAN MATRIX DF/DY HAS AN EIGENVALUE
C WHOSE REAL PART IS NEGATIVE AND LARGE IN MAGNITUDE, COMPARED TO THE 
C RECIPROCAL OF THE T SPAN OF INTEREST.  IF THE PROBLEM IS NONSTIFF,
C USE A METHOD FLAG MF = 10.  IF IT IS STIFF, THERE ARE FOUR STANDARD 
C CHOICES FOR MF, AND LSODE REQUIRES THE JACOBIAN MATRIX IN SOME FORM.
C THIS MATRIX IS REGARDED EITHER AS FULL (MF = 21 OR 22),
C OR BANDED (MF = 24 OR 25).  IN THE BANDED CASE, LSODE REQUIRES TWO
C HALF-BANDWIDTH PARAMETERS ML AND MU.  THESE ARE, RESPECTIVELY, THE
C WIDTHS OF THE LOWER AND UPPER PARTS OF THE BAND, EXCLUDING THE MAIN 
C DIAGONAL.  THUS THE BAND CONSISTS OF THE LOCATIONS (I,J) WITH
C I-ML .LE. J .LE. I+MU, AND THE FULL BANDWIDTH IS ML+MU+1. 
C
C C. IF THE PROBLEM IS STIFF, YOU ARE ENCOURAGED TO SUPPLY THE JACOBIAN
C DIRECTLY (MF = 21 OR 24), BUT IF THIS IS NOT FEASIBLE, LSODE WILL
C COMPUTE IT INTERNALLY BY DIFFERENCE QUOTIENTS (MF = 22 OR 25).
C IF YOU ARE SUPPLYING THE JACOBIAN, PROVIDE A SUBROUTINE OF THE FORM..
C               SUBROUTINE JAC (NEQ, T, Y, ML, MU, PD, NROWPD)
C               DIMENSION Y(NEQ), PD(NROWPD,NEQ)
C WHICH SUPPLIES DF/DY BY LOADING PD AS FOLLOWS.. 
C     FOR A FULL JACOBIAN (MF = 21), LOAD PD(I,J) WITH DF(I)/DY(J),
C THE PARTIAL DERIVATIVE OF F(I) WITH RESPECT TO Y(J).  (IGNORE THE
C ML AND MU ARGUMENTS IN THIS CASE.)
C     FOR A BANDED JACOBIAN (MF = 24), LOAD PD(I-J+MU+1,J) WITH
C DF(I)/DY(J), I.E. LOAD THE DIAGONAL LINES OF DF/DY INTO THE ROWS OF 
C PD FROM THE TOP DOWN.
C     IN EITHER CASE, ONLY NONZERO ELEMENTS NEED BE LOADED. 
C
C D. WRITE A MAIN PROGRAM WHICH CALLS SUBROUTINE LSODE ONCE FOR
C EACH POINT AT WHICH ANSWERS ARE DESIRED.  THIS SHOULD ALSO PROVIDE
C FOR POSSIBLE USE OF LOGICAL UNIT 6 FOR OUTPUT OF ERROR MESSAGES
C BY LSODE.  ON THE FIRST CALL TO LSODE, SUPPLY ARGUMENTS AS FOLLOWS..
C F      = NAME OF SUBROUTINE FOR RIGHT-HAND SIDE VECTOR F. 
C          THIS NAME MUST BE DECLARED EXTERNAL IN CALLING PROGRAM.
C NEQ    = NUMBER OF FIRST ORDER ODE-S. 
C Y      = ARRAY OF INITIAL VALUES, OF LENGTH NEQ.
C T      = THE INITIAL VALUE OF THE INDEPENDENT VARIABLE.
C TOUT   = FIRST POINT WHERE OUTPUT IS DESIRED (.NE. T).
C ITOL   = 1 OR 2 ACCORDING AS ATOL (BELOW) IS A SCALAR OR ARRAY.
C RTOL   = RELATIVE TOLERANCE PARAMETER (SCALAR). 
C ATOL   = ABSOLUTE TOLERANCE PARAMETER (SCALAR OR ARRAY).
C          THE ESTIMATED LOCAL ERROR IN Y(I) WILL BE CONTROLLED SO AS 
C          TO BE ROUGHLY LESS (IN MAGNITUDE) THAN 
C             EWT(I) = RTOL*ABS(Y(I)) + ATOL     IF ITOL = 1, OR
C             EWT(I) = RTOL*ABS(Y(I)) + ATOL(I)  IF ITOL = 2.
C          THUS THE LOCAL ERROR TEST PASSES IF, IN EACH COMPONENT,
C          EITHER THE ABSOLUTE ERROR IS LESS THAN ATOL (OR ATOL(I)),
C          OR THE RELATIVE ERROR IS LESS THAN RTOL.
C          USE RTOL = 0.0 FOR PURE ABSOLUTE ERROR CONTROL, AND
C          USE ATOL = 0.0 (OR ATOL(I) = 0.0) FOR PURE RELATIVE ERROR
C          CONTROL.  CAUTION.. ACTUAL (GLOBAL) ERRORS MAY EXCEED THESE
C          LOCAL TOLERANCES, SO CHOOSE THEM CONSERVATIVELY. 
C ITASK  = 1 FOR NORMAL COMPUTATION OF OUTPUT VALUES OF Y AT T = TOUT.
C ISTATE = INTEGER FLAG (INPUT AND OUTPUT).  SET ISTATE = 1.
C IOPT   = 0 TO INDICATE NO OPTIONAL INPUTS USED. 
C RWORK  = REAL WORK ARRAY OF LENGTH AT LEAST..
C             20 + 16*NEQ                    FOR MF = 10,
C             22 +  9*NEQ + NEQ**2           FOR MF = 21 OR 22,
C             22 + 10*NEQ + (2*ML + MU)*NEQ  FOR MF = 24 OR 25.
C LRW    = DECLARED LENGTH OF RWORK (IN USER-S DIMENSION).
C IWORK  = INTEGER WORK ARRAY OF LENGTH AT LEAST..
C             20        FOR MF = 10,
C             20 + NEQ  FOR MF = 21, 22, 24, OR 25.
C          IF MF = 24 OR 25, INPUT IN IWORK(1),IWORK(2) THE LOWER
C          AND UPPER HALF-BANDWIDTHS ML,MU.
C LIW    = DECLARED LENGTH OF IWORK (IN USER-S DIMENSION).
C JAC    = NAME OF SUBROUTINE FOR JACOBIAN MATRIX (MF = 21 OR 24).
C          IF USED, THIS NAME MUST BE DECLARED EXTERNAL IN CALLING
C          PROGRAM.  IF NOT USED, PASS A DUMMY NAME.
C MF     = METHOD FLAG.  STANDARD VALUES ARE..
C          10 FOR NONSTIFF (ADAMS) METHOD, NO JACOBIAN USED.
C          21 FOR STIFF (BDF) METHOD, USER-SUPPLIED FULL JACOBIAN.
C          22 FOR STIFF METHOD, INTERNALLY GENERATED FULL JACOBIAN.
C          24 FOR STIFF METHOD, USER-SUPPLIED BANDED JACOBIAN.
C          25 FOR STIFF METHOD, INTERNALLY GENERATED BANDED JACOBIAN. 
C NOTE THAT THE MAIN PROGRAM MUST DECLARE ARRAYS Y, RWORK, IWORK,
C AND POSSIBLY ATOL.
C
C E. THE OUTPUT FROM THE FIRST CALL (OR ANY CALL) IS..
C      Y = ARRAY OF COMPUTED VALUES OF Y(T) VECTOR.
C      T = CORRESPONDING VALUE OF INDEPENDENT VARIABLE (NORMALLY TOUT).
C ISTATE = 2  IF LSODE WAS SUCCESSFUL, NEGATIVE OTHERWISE.
C          -1 MEANS EXCESS WORK DONE ON THIS CALL (PERHAPS WRONG MF). 
C          -2 MEANS EXCESS ACCURACY REQUESTED (TOLERANCES TOO SMALL). 
C          -3 MEANS ILLEGAL INPUT DETECTED (SEE PRINTED MESSAGE).
C          -4 MEANS REPEATED ERROR TEST FAILURES (CHECK ALL INPUTS).
C          -5 MEANS REPEATED CONVERGENCE FAILURES (PERHAPS BAD JACOBIAN
C             SUPPLIED OR WRONG CHOICE OF MF OR TOLERANCES).
C          -6 MEANS ERROR WEIGHT BECAME ZERO DURING PROBLEM. (SOLUTION
C             COMPONENT I VANISHED, AND ATOL OR ATOL(I) = 0.)
C         -13 MEANS EXIT REQUESTED IN USER-SUPPLIED FUNCTION.
C
C F. TO CONTINUE THE INTEGRATION AFTER A SUCCESSFUL RETURN, SIMPLY
C RESET TOUT AND CALL LSODE AGAIN.  NO OTHER PARAMETERS NEED BE RESET.
C
C-----------------------------------------------------------------------
C EXAMPLE PROBLEM.
C
C THE FOLLOWING IS A SIMPLE EXAMPLE PROBLEM, WITH THE CODING
C NEEDED FOR ITS SOLUTION BY LSODE.  THE PROBLEM IS FROM CHEMICAL
C KINETICS, AND CONSISTS OF THE FOLLOWING THREE RATE EQUATIONS..
C     DY1/DT = -.04*Y1 + 1.E4*Y2*Y3
C     DY2/DT = .04*Y1 - 1.E4*Y2*Y3 - 3.E7*Y2**2
C     DY3/DT = 3.E7*Y2**2
C ON THE INTERVAL FROM T = 0.0 TO T = 4.E10, WITH INITIAL CONDITIONS
C Y1 = 1.0, Y2 = Y3 = 0.  THE PROBLEM IS STIFF.
C
C THE FOLLOWING CODING SOLVES THIS PROBLEM WITH LSODE, USING MF = 21
C AND PRINTING RESULTS AT T = .4, 4., ..., 4.E10.  IT USES
C ITOL = 2 AND ATOL MUCH SMALLER FOR Y2 THAN Y1 OR Y3 BECAUSE
C Y2 HAS MUCH SMALLER VALUES. 
C AT THE END OF THE RUN, STATISTICAL QUANTITIES OF INTEREST ARE
C PRINTED (SEE OPTIONAL OUTPUTS IN THE FULL DESCRIPTION BELOW).
C
C     EXTERNAL FEX, JEX
C     DOUBLE PRECISION ATOL, RTOL, RWORK, T, TOUT, Y
C     DIMENSION Y(3), ATOL(3), RWORK(58), IWORK(23)
C     NEQ = 3
C     Y(1) = 1.D0
C     Y(2) = 0.D0
C     Y(3) = 0.D0
C     T = 0.D0
C     TOUT = .4D0
C     ITOL = 2
C     RTOL = 1.D-4
C     ATOL(1) = 1.D-6
C     ATOL(2) = 1.D-10
C     ATOL(3) = 1.D-6
C     ITASK = 1
C     ISTATE = 1
C     IOPT = 0
C     LRW = 58
C     LIW = 23
C     MF = 21
C     DO 40 IOUT = 1,12
C       CALL LSODE(FEX,NEQ,Y,T,TOUT,ITOL,RTOL,ATOL,ITASK,ISTATE,
C    1     IOPT,RWORK,LRW,IWORK,LIW,JEX,MF)
C       WRITE(6,20)T,Y(1),Y(2),Y(3)
C 20    FORMAT(7H AT T =,E12.4,6H   Y =,3E14.6)
C       IF (ISTATE .LT. 0) GO TO 80
C 40    TOUT = TOUT*10.D0
C     WRITE(6,60)IWORK(11),IWORK(12),IWORK(13)
C 60  FORMAT(/12H NO. STEPS =,I4,11H  NO. F-S =,I4,11H  NO. J-S =,I4) 
C     STOP
C 80  WRITE(6,90)ISTATE
C 90  FORMAT(///22H ERROR HALT.. ISTATE =,I3)
C     STOP
C     END 
C
C     SUBROUTINE FEX (NEQ, T, Y, YDOT)
C     DOUBLE PRECISION T, Y, YDOT
C     DIMENSION Y(3), YDOT(3) 
C     YDOT(1) = -.04D0*Y(1) + 1.D4*Y(2)*Y(3)
C     YDOT(3) = 3.D7*Y(2)*Y(2)
C     YDOT(2) = -YDOT(1) - YDOT(3)
C     RETURN
C     END 
C
C     SUBROUTINE JEX (NEQ, T, Y, ML, MU, PD, NRPD)
C     DOUBLE PRECISION PD, T, Y
C     DIMENSION Y(3), PD(NRPD,3)
C     PD(1,1) = -.04D0
C     PD(1,2) = 1.D4*Y(3)
C     PD(1,3) = 1.D4*Y(2)
C     PD(2,1) = .04D0
C     PD(2,3) = -PD(1,3)
C     PD(3,2) = 6.D7*Y(2)
C     PD(2,2) = -PD(1,2) - PD(3,2)
C     RETURN
C     END 
C
C THE OUTPUT OF THIS PROGRAM (ON A CDC-7600 IN SINGLE PRECISION)
C IS AS FOLLOWS..
C
C   AT T =  4.0000E-01   Y =  9.851726E-01  3.386406E-05  1.479357E-02
C   AT T =  4.0000E+00   Y =  9.055142E-01  2.240418E-05  9.446344E-02
C   AT T =  4.0000E+01   Y =  7.158050E-01  9.184616E-06  2.841858E-01
C   AT T =  4.0000E+02   Y =  4.504846E-01  3.222434E-06  5.495122E-01
C   AT T =  4.0000E+03   Y =  1.831701E-01  8.940379E-07  8.168290E-01
C   AT T =  4.0000E+04   Y =  3.897016E-02  1.621193E-07  9.610297E-01
C   AT T =  4.0000E+05   Y =  4.935213E-03  1.983756E-08  9.950648E-01
C   AT T =  4.0000E+06   Y =  5.159269E-04  2.064759E-09  9.994841E-01
C   AT T =  4.0000E+07   Y =  5.306413E-05  2.122677E-10  9.999469E-01
C   AT T =  4.0000E+08   Y =  5.494529E-06  2.197824E-11  9.999945E-01
C   AT T =  4.0000E+09   Y =  5.129458E-07  2.051784E-12  9.999995E-01
C   AT T =  4.0000E+10   Y = -7.170586E-08 -2.868234E-13  1.000000E+00
C
C   NO. STEPS = 330  NO. F-S = 405  NO. J-S =  69 
C-----------------------------------------------------------------------
C FULL DESCRIPTION OF USER INTERFACE TO LSODE.
C
C THE USER INTERFACE TO LSODE CONSISTS OF THE FOLLOWING PARTS.
C
C I.   THE CALL SEQUENCE TO SUBROUTINE LSODE, WHICH IS A DRIVER
C      ROUTINE FOR THE SOLVER.  THIS INCLUDES DESCRIPTIONS OF BOTH
C      THE CALL SEQUENCE ARGUMENTS AND OF USER-SUPPLIED ROUTINES.
C      FOLLOWING THESE DESCRIPTIONS IS A DESCRIPTION OF
C      OPTIONAL INPUTS AVAILABLE THROUGH THE CALL SEQUENCE, AND THEN
C      A DESCRIPTION OF OPTIONAL OUTPUTS (IN THE WORK ARRAYS).
C
C II.  DESCRIPTIONS OF OTHER ROUTINES IN THE LSODE PACKAGE THAT MAY BE
C      (OPTIONALLY) CALLED BY THE USER.  THESE PROVIDE THE ABILITY TO 
C      ALTER ERROR MESSAGE HANDLING, SAVE AND RESTORE THE INTERNAL
C      COMMON, AND OBTAIN SPECIFIED DERIVATIVES OF THE SOLUTION Y(T). 
C
C III. DESCRIPTIONS OF COMMON BLOCKS TO BE DECLARED IN OVERLAY
C      OR SIMILAR ENVIRONMENTS, OR TO BE SAVED WHEN DOING AN INTERRUPT
C      OF THE PROBLEM AND CONTINUED SOLUTION LATER.
C
C IV.  DESCRIPTION OF TWO ROUTINES IN THE LSODE PACKAGE, EITHER OF
C      WHICH THE USER MAY REPLACE WITH HIS OWN VERSION, IF DESIRED.
C      THESE RELATE TO THE MEASUREMENT OF ERRORS. 
C
C-----------------------------------------------------------------------
C PART I.  CALL SEQUENCE.
C
C THE CALL SEQUENCE PARAMETERS USED FOR INPUT ONLY ARE
C     F, NEQ, TOUT, ITOL, RTOL, ATOL, ITASK, IOPT, LRW, LIW, JAC, MF, 
C AND THOSE USED FOR BOTH INPUT AND OUTPUT ARE
C     Y, T, ISTATE. 
C THE WORK ARRAYS RWORK AND IWORK ARE ALSO USED FOR CONDITIONAL AND
C OPTIONAL INPUTS AND OPTIONAL OUTPUTS.  (THE TERM OUTPUT HERE REFERS 
C TO THE RETURN FROM SUBROUTINE LSODE TO THE USER-S CALLING PROGRAM.) 
C
C THE LEGALITY OF INPUT PARAMETERS WILL BE THOROUGHLY CHECKED ON THE
C INITIAL CALL FOR THE PROBLEM, BUT NOT CHECKED THEREAFTER UNLESS A
C CHANGE IN INPUT PARAMETERS IS FLAGGED BY ISTATE = 3 ON INPUT.
C
C THE DESCRIPTIONS OF THE CALL ARGUMENTS ARE AS FOLLOWS.
C
C F      = THE NAME OF THE USER-SUPPLIED SUBROUTINE DEFINING THE
C          ODE SYSTEM.  THE SYSTEM MUST BE PUT IN THE FIRST-ORDER
C          FORM DY/DT = F(T,Y), WHERE F IS A VECTOR-VALUED FUNCTION
C          OF THE SCALAR T AND THE VECTOR Y.  SUBROUTINE F IS TO
C          COMPUTE THE FUNCTION F.  IT IS TO HAVE THE FORM
C               SUBROUTINE F (NEQ, T, Y, YDOT)
C               DIMENSION Y(1), YDOT(1) 
C          WHERE NEQ, T, AND Y ARE INPUT, AND THE ARRAY YDOT = F(T,Y) 
C          IS OUTPUT.  Y AND YDOT ARE ARRAYS OF LENGTH NEQ. 
C          (IN THE DIMENSION STATEMENT ABOVE, 1 IS A DUMMY
C          DIMENSION.. IT CAN BE REPLACED BY ANY VALUE.)
C          SUBROUTINE F SHOULD NOT ALTER Y(1),...,Y(NEQ).
C          F MUST BE DECLARED EXTERNAL IN THE CALLING PROGRAM.
C
C          SUBROUTINE F MAY ACCESS USER-DEFINED QUANTITIES IN
C          NEQ(2),... AND/OR IN Y(NEQ(1)+1),... IF NEQ IS AN ARRAY
C          (DIMENSIONED IN F) AND/OR Y HAS LENGTH EXCEEDING NEQ(1).
C          SEE THE DESCRIPTIONS OF NEQ AND Y BELOW.
C
C          IF QUANTITIES COMPUTED IN THE F ROUTINE ARE NEEDED
C          EXTERNALLY TO LSODE, AN EXTRA CALL TO F SHOULD BE MADE
C          FOR THIS PURPOSE, FOR CONSISTENT AND ACCURATE RESULTS.
C          IF ONLY THE DERIVATIVE DY/DT IS NEEDED, USE INTDY INSTEAD. 
C
C NEQ    = THE SIZE OF THE ODE SYSTEM (NUMBER OF FIRST ORDER
C          ORDINARY DIFFERENTIAL EQUATIONS).  USED ONLY FOR INPUT.
C          NEQ MAY BE DECREASED, BUT NOT INCREASED, DURING THE PROBLEM.
C          IF NEQ IS DECREASED (WITH ISTATE = 3 ON INPUT), THE
C          REMAINING COMPONENTS OF Y SHOULD BE LEFT UNDISTURBED, IF
C          THESE ARE TO BE ACCESSED IN F AND/OR JAC.
C
C          NORMALLY, NEQ IS A SCALAR, AND IT IS GENERALLY REFERRED TO 
C          AS A SCALAR IN THIS USER INTERFACE DESCRIPTION.  HOWEVER,
C          NEQ MAY BE AN ARRAY, WITH NEQ(1) SET TO THE SYSTEM SIZE.
C          (THE LSODE PACKAGE ACCESSES ONLY NEQ(1).)  IN EITHER CASE, 
C          THIS PARAMETER IS PASSED AS THE NEQ ARGUMENT IN ALL CALLS
C          TO F AND JAC.  HENCE, IF IT IS AN ARRAY, LOCATIONS
C          NEQ(2),... MAY BE USED TO STORE OTHER INTEGER DATA AND PASS
C          IT TO F AND/OR JAC.  SUBROUTINES F AND/OR JAC MUST INCLUDE 
C          NEQ IN A DIMENSION STATEMENT IN THAT CASE.
C
C Y      = A REAL ARRAY FOR THE VECTOR OF DEPENDENT VARIABLES, OF
C          LENGTH NEQ OR MORE.  USED FOR BOTH INPUT AND OUTPUT ON THE 
C          FIRST CALL (ISTATE = 1), AND ONLY FOR OUTPUT ON OTHER CALLS.
C          ON THE FIRST CALL, Y MUST CONTAIN THE VECTOR OF INITIAL
C          VALUES.  ON OUTPUT, Y CONTAINS THE COMPUTED SOLUTION VECTOR,
C          EVALUATED AT T.  IF DESIRED, THE Y ARRAY MAY BE USED
C          FOR OTHER PURPOSES BETWEEN CALLS TO THE SOLVER.
C
C          THIS ARRAY IS PASSED AS THE Y ARGUMENT IN ALL CALLS TO
C          F AND JAC.  HENCE ITS LENGTH MAY EXCEED NEQ, AND LOCATIONS 
C          Y(NEQ+1),... MAY BE USED TO STORE OTHER REAL DATA AND
C          PASS IT TO F AND/OR JAC.  (THE LSODE PACKAGE ACCESSES ONLY 
C          Y(1),...,Y(NEQ).)
C
C T      = THE INDEPENDENT VARIABLE.  ON INPUT, T IS USED ONLY ON THE 
C          FIRST CALL, AS THE INITIAL POINT OF THE INTEGRATION.
C          ON OUTPUT, AFTER EACH CALL, T IS THE VALUE AT WHICH A
C          COMPUTED SOLUTION Y IS EVALUATED (USUALLY THE SAME AS TOUT).
C          ON AN ERROR RETURN, T IS THE FARTHEST POINT REACHED.
C
C TOUT   = THE NEXT VALUE OF T AT WHICH A COMPUTED SOLUTION IS DESIRED.
C          USED ONLY FOR INPUT.
C
C          WHEN STARTING THE PROBLEM (ISTATE = 1), TOUT MAY BE EQUAL
C          TO T FOR ONE CALL, THEN SHOULD .NE. T FOR THE NEXT CALL.
C          FOR THE INITIAL T, AN INPUT VALUE OF TOUT .NE. T IS USED
C          IN ORDER TO DETERMINE THE DIRECTION OF THE INTEGRATION
C          (I.E. THE ALGEBRAIC SIGN OF THE STEP SIZES) AND THE ROUGH
C          SCALE OF THE PROBLEM.  INTEGRATION IN EITHER DIRECTION
C          (FORWARD OR BACKWARD IN T) IS PERMITTED.
C
C          IF ITASK = 2 OR 5 (ONE-STEP MODES), TOUT IS IGNORED AFTER
C          THE FIRST CALL (I.E. THE FIRST CALL WITH TOUT .NE. T).
C          OTHERWISE, TOUT IS REQUIRED ON EVERY CALL.
C
C          IF ITASK = 1, 3, OR 4, THE VALUES OF TOUT NEED NOT BE
C          MONOTONE, BUT A VALUE OF TOUT WHICH BACKS UP IS LIMITED
C          TO THE CURRENT INTERNAL T INTERVAL, WHOSE ENDPOINTS ARE
C          TCUR - HU AND TCUR (SEE OPTIONAL OUTPUTS, BELOW, FOR
C          TCUR AND HU).
C
C ITOL   = AN INDICATOR FOR THE TYPE OF ERROR CONTROL.  SEE 
C          DESCRIPTION BELOW UNDER ATOL.  USED ONLY FOR INPUT.
C
C RTOL   = A RELATIVE ERROR TOLERANCE PARAMETER, EITHER A SCALAR OR
C          AN ARRAY OF LENGTH NEQ.  SEE DESCRIPTION BELOW UNDER ATOL. 
C          INPUT ONLY.
C
C ATOL   = AN ABSOLUTE ERROR TOLERANCE PARAMETER, EITHER A SCALAR OR
C          AN ARRAY OF LENGTH NEQ.  INPUT ONLY.
C
C             THE INPUT PARAMETERS ITOL, RTOL, AND ATOL DETERMINE
C          THE ERROR CONTROL PERFORMED BY THE SOLVER.  THE SOLVER WILL
C          CONTROL THE VECTOR E = (E(I)) OF ESTIMATED LOCAL ERRORS
C          IN Y, ACCORDING TO AN INEQUALITY OF THE FORM
C                      RMS-NORM OF ( E(I)/EWT(I) )   .LE.   1,
C          WHERE       EWT(I) = RTOL(I)*ABS(Y(I)) + ATOL(I),
C          AND THE RMS-NORM (ROOT-MEAN-SQUARE NORM) HERE IS 
C          RMS-NORM(V) = SQRT(SUM V(I)**2 / NEQ).  HERE EWT = (EWT(I))
C          IS A VECTOR OF WEIGHTS WHICH MUST ALWAYS BE POSITIVE, AND
C          THE VALUES OF RTOL AND ATOL SHOULD ALL BE NON-NEGATIVE.
C          THE FOLLOWING TABLE GIVES THE TYPES (SCALAR/ARRAY) OF
C          RTOL AND ATOL, AND THE CORRESPONDING FORM OF EWT(I).
C
C             ITOL    RTOL       ATOL          EWT(I)
C              1     SCALAR     SCALAR     RTOL*ABS(Y(I)) + ATOL
C              2     SCALAR     ARRAY      RTOL*ABS(Y(I)) + ATOL(I)
C              3     ARRAY      SCALAR     RTOL(I)*ABS(Y(I)) + ATOL
C              4     ARRAY      ARRAY      RTOL(I)*ABS(Y(I)) + ATOL(I)
C
C          WHEN EITHER OF THESE PARAMETERS IS A SCALAR, IT NEED NOT
C          BE DIMENSIONED IN THE USER-S CALLING PROGRAM.
C
C          IF NONE OF THE ABOVE CHOICES (WITH ITOL, RTOL, AND ATOL
C          FIXED THROUGHOUT THE PROBLEM) IS SUITABLE, MORE GENERAL
C          ERROR CONTROLS CAN BE OBTAINED BY SUBSTITUTING
C          USER-SUPPLIED ROUTINES FOR THE SETTING OF EWT AND/OR FOR
C          THE NORM CALCULATION.  SEE PART IV BELOW.
C
C          IF GLOBAL ERRORS ARE TO BE ESTIMATED BY MAKING A REPEATED
C          RUN ON THE SAME PROBLEM WITH SMALLER TOLERANCES, THEN ALL
C          COMPONENTS OF RTOL AND ATOL (I.E. OF EWT) SHOULD BE SCALED 
C          DOWN UNIFORMLY.
C
C ITASK  = AN INDEX SPECIFYING THE TASK TO BE PERFORMED.
C          INPUT ONLY.  ITASK HAS THE FOLLOWING VALUES AND MEANINGS.
C          1  MEANS NORMAL COMPUTATION OF OUTPUT VALUES OF Y(T) AT
C             T = TOUT (BY OVERSHOOTING AND INTERPOLATING). 
C          2  MEANS TAKE ONE STEP ONLY AND RETURN.
C          3  MEANS STOP AT THE FIRST INTERNAL MESH POINT AT OR
C             BEYOND T = TOUT AND RETURN.
C          4  MEANS NORMAL COMPUTATION OF OUTPUT VALUES OF Y(T) AT
C             T = TOUT BUT WITHOUT OVERSHOOTING T = TCRIT.
C             TCRIT MUST BE INPUT AS RWORK(1).  TCRIT MAY BE EQUAL TO 
C             OR BEYOND TOUT, BUT NOT BEHIND IT IN THE DIRECTION OF
C             INTEGRATION.  THIS OPTION IS USEFUL IF THE PROBLEM
C             HAS A SINGULARITY AT OR BEYOND T = TCRIT.
C          5  MEANS TAKE ONE STEP, WITHOUT PASSING TCRIT, AND RETURN. 
C             TCRIT MUST BE INPUT AS RWORK(1).
C
C          NOTE..  IF ITASK = 4 OR 5 AND THE SOLVER REACHES TCRIT
C          (WITHIN ROUNDOFF), IT WILL RETURN T = TCRIT (EXACTLY) TO
C          INDICATE THIS (UNLESS ITASK = 4 AND TOUT COMES BEFORE TCRIT,
C          IN WHICH CASE ANSWERS AT T = TOUT ARE RETURNED FIRST).
C
C ISTATE = AN INDEX USED FOR INPUT AND OUTPUT TO SPECIFY THE
C          THE STATE OF THE CALCULATION.
C
C          ON INPUT, THE VALUES OF ISTATE ARE AS FOLLOWS.
C          1  MEANS THIS IS THE FIRST CALL FOR THE PROBLEM
C             (INITIALIZATIONS WILL BE DONE).  SEE NOTE BELOW.
C          2  MEANS THIS IS NOT THE FIRST CALL, AND THE CALCULATION
C             IS TO CONTINUE NORMALLY, WITH NO CHANGE IN ANY INPUT
C             PARAMETERS EXCEPT POSSIBLY TOUT AND ITASK.
C             (IF ITOL, RTOL, AND/OR ATOL ARE CHANGED BETWEEN CALLS
C             WITH ISTATE = 2, THE NEW VALUES WILL BE USED BUT NOT
C             TESTED FOR LEGALITY.)
C          3  MEANS THIS IS NOT THE FIRST CALL, AND THE
C             CALCULATION IS TO CONTINUE NORMALLY, BUT WITH 
C             A CHANGE IN INPUT PARAMETERS OTHER THAN
C             TOUT AND ITASK.  CHANGES ARE ALLOWED IN
C             NEQ, ITOL, RTOL, ATOL, IOPT, LRW, LIW, MF, ML, MU,
C             AND ANY OF THE OPTIONAL INPUTS EXCEPT H0.
C             (SEE IWORK DESCRIPTION FOR ML AND MU.)
C          NOTE..  A PRELIMINARY CALL WITH TOUT = T IS NOT COUNTED
C          AS A FIRST CALL HERE, AS NO INITIALIZATION OR CHECKING OF
C          INPUT IS DONE.  (SUCH A CALL IS SOMETIMES USEFUL FOR THE
C          PURPOSE OF OUTPUTTING THE INITIAL CONDITIONS.)
C          THUS THE FIRST CALL FOR WHICH TOUT .NE. T REQUIRES
C          ISTATE = 1 ON INPUT.
C
C          ON OUTPUT, ISTATE HAS THE FOLLOWING VALUES AND MEANINGS.
C           1  MEANS NOTHING WAS DONE, AS TOUT WAS EQUAL TO T WITH
C              ISTATE = 1 ON INPUT.  (HOWEVER, AN INTERNAL COUNTER WAS
C              SET TO DETECT AND PREVENT REPEATED CALLS OF THIS TYPE.)
C           2  MEANS THE INTEGRATION WAS PERFORMED SUCCESSFULLY.
C          -1  MEANS AN EXCESSIVE AMOUNT OF WORK (MORE THAN MXSTEP
C              STEPS) WAS DONE ON THIS CALL, BEFORE COMPLETING THE
C              REQUESTED TASK, BUT THE INTEGRATION WAS OTHERWISE
C              SUCCESSFUL AS FAR AS T.  (MXSTEP IS AN OPTIONAL INPUT
C              AND IS NORMALLY 500.)  TO CONTINUE, THE USER MAY
C              SIMPLY RESET ISTATE TO A VALUE .GT. 1 AND CALL AGAIN
C              (THE EXCESS WORK STEP COUNTER WILL BE RESET TO 0).
C              IN ADDITION, THE USER MAY INCREASE MXSTEP TO AVOID
C              THIS ERROR RETURN (SEE BELOW ON OPTIONAL INPUTS).
C          -2  MEANS TOO MUCH ACCURACY WAS REQUESTED FOR THE PRECISION
C              OF THE MACHINE BEING USED.  THIS WAS DETECTED BEFORE
C              COMPLETING THE REQUESTED TASK, BUT THE INTEGRATION
C              WAS SUCCESSFUL AS FAR AS T.  TO CONTINUE, THE TOLERANCE
C              PARAMETERS MUST BE RESET, AND ISTATE MUST BE SET
C              TO 3.  THE OPTIONAL OUTPUT TOLSF MAY BE USED FOR THIS
C              PURPOSE.  (NOTE.. IF THIS CONDITION IS DETECTED BEFORE 
C              TAKING ANY STEPS, THEN AN ILLEGAL INPUT RETURN
C              (ISTATE = -3) OCCURS INSTEAD.)
C          -3  MEANS ILLEGAL INPUT WAS DETECTED, BEFORE TAKING ANY
C              INTEGRATION STEPS.  SEE WRITTEN MESSAGE FOR DETAILS.
C              NOTE..  IF THE SOLVER DETECTS AN INFINITE LOOP OF CALLS
C              TO THE SOLVER WITH ILLEGAL INPUT, IT WILL CAUSE
C              THE RUN TO STOP.
C          -4  MEANS THERE WERE REPEATED ERROR TEST FAILURES ON
C              ONE ATTEMPTED STEP, BEFORE COMPLETING THE REQUESTED
C              TASK, BUT THE INTEGRATION WAS SUCCESSFUL AS FAR AS T.
C              THE PROBLEM MAY HAVE A SINGULARITY, OR THE INPUT
C              MAY BE INAPPROPRIATE.
C          -5  MEANS THERE WERE REPEATED CONVERGENCE TEST FAILURES ON 
C              ONE ATTEMPTED STEP, BEFORE COMPLETING THE REQUESTED
C              TASK, BUT THE INTEGRATION WAS SUCCESSFUL AS FAR AS T.
C              THIS MAY BE CAUSED BY AN INACCURATE JACOBIAN MATRIX,
C              IF ONE IS BEING USED.
C          -6  MEANS EWT(I) BECAME ZERO FOR SOME I DURING THE
C              INTEGRATION.  PURE RELATIVE ERROR CONTROL (ATOL(I)=0.0)
C              WAS REQUESTED ON A VARIABLE WHICH HAS NOW VANISHED.
C              THE INTEGRATION WAS SUCCESSFUL AS FAR AS T.
C
C          NOTE..  SINCE THE NORMAL OUTPUT VALUE OF ISTATE IS 2,
C          IT DOES NOT NEED TO BE RESET FOR NORMAL CONTINUATION.
C          ALSO, SINCE A NEGATIVE INPUT VALUE OF ISTATE WILL BE
C          REGARDED AS ILLEGAL, A NEGATIVE OUTPUT VALUE REQUIRES THE
C          USER TO CHANGE IT, AND POSSIBLY OTHER INPUTS, BEFORE
C          CALLING THE SOLVER AGAIN.
C
C IOPT   = AN INTEGER FLAG TO SPECIFY WHETHER OR NOT ANY OPTIONAL
C          INPUTS ARE BEING USED ON THIS CALL.  INPUT ONLY. 
C          THE OPTIONAL INPUTS ARE LISTED SEPARATELY BELOW. 
C          IOPT = 0 MEANS NO OPTIONAL INPUTS ARE BEING USED.
C                   DEFAULT VALUES WILL BE USED IN ALL CASES.
C          IOPT = 1 MEANS ONE OR MORE OPTIONAL INPUTS ARE BEING USED. 
C
C RWORK  = A REAL WORKING ARRAY (DOUBLE PRECISION).
C          THE LENGTH OF RWORK MUST BE AT LEAST
C             20 + NYH*(MAXORD + 1) + 3*NEQ + LWM    WHERE
C          NYH    = THE INITIAL VALUE OF NEQ,
C          MAXORD = 12 (IF METH = 1) OR 5 (IF METH = 2) (UNLESS A
C                   SMALLER VALUE IS GIVEN AS AN OPTIONAL INPUT),
C          LWM   = 0             IF MITER = 0,
C          LWM   = NEQ**2 + 2    IF MITER IS 1 OR 2,
C          LWM   = NEQ + 2       IF MITER = 3, AND
C          LWM   = (2*ML+MU+1)*NEQ + 2 IF MITER IS 4 OR 5.
C          (SEE THE MF DESCRIPTION FOR METH AND MITER.)
C          THUS IF MAXORD HAS ITS DEFAULT VALUE AND NEQ IS CONSTANT,
C          THIS LENGTH IS..
C             20 + 16*NEQ                  FOR MF = 10,
C             22 + 16*NEQ + NEQ**2         FOR MF = 11 OR 12,
C             22 + 17*NEQ                  FOR MF = 13,
C             22 + 17*NEQ + (2*ML+MU)*NEQ  FOR MF = 14 OR 15,
C             20 +  9*NEQ                  FOR MF = 20,
C             22 +  9*NEQ + NEQ**2         FOR MF = 21 OR 22,
C             22 + 10*NEQ                  FOR MF = 23,
C             22 + 10*NEQ + (2*ML+MU)*NEQ  FOR MF = 24 OR 25.
C          THE FIRST 20 WORDS OF RWORK ARE RESERVED FOR CONDITIONAL
C          AND OPTIONAL INPUTS AND OPTIONAL OUTPUTS.
C
C          THE FOLLOWING WORD IN RWORK IS A CONDITIONAL INPUT..
C            RWORK(1) = TCRIT = CRITICAL VALUE OF T WHICH THE SOLVER
C                       IS NOT TO OVERSHOOT.  REQUIRED IF ITASK IS
C                       4 OR 5, AND IGNORED OTHERWISE.  (SEE ITASK.)
C
C LRW    = THE LENGTH OF THE ARRAY RWORK, AS DECLARED BY THE USER.
C          (THIS WILL BE CHECKED BY THE SOLVER.)
C
C IWORK  = AN INTEGER WORK ARRAY.  THE LENGTH OF IWORK MUST BE AT LEAST
C             20        IF MITER = 0 OR 3 (MF = 10, 13, 20, 23), OR
C             20 + NEQ  OTHERWISE (MF = 11, 12, 14, 15, 21, 22, 24, 25).
C          THE FIRST FEW WORDS OF IWORK ARE USED FOR CONDITIONAL AND
C          OPTIONAL INPUTS AND OPTIONAL OUTPUTS.
C
C          THE FOLLOWING 2 WORDS IN IWORK ARE CONDITIONAL INPUTS..
C            IWORK(1) = ML     THESE ARE THE LOWER AND UPPER
C            IWORK(2) = MU     HALF-BANDWIDTHS, RESPECTIVELY, OF THE
C                       BANDED JACOBIAN, EXCLUDING THE MAIN DIAGONAL. 
C                       THE BAND IS DEFINED BY THE MATRIX LOCATIONS
C                       (I,J) WITH I-ML .LE. J .LE. I+MU.  ML AND MU
C                       MUST SATISFY  0 .LE.  ML,MU  .LE. NEQ-1.
C                       THESE ARE REQUIRED IF MITER IS 4 OR 5, AND
C                       IGNORED OTHERWISE.  ML AND MU MAY IN FACT BE
C                       THE BAND PARAMETERS FOR A MATRIX TO WHICH
C                       DF/DY IS ONLY APPROXIMATELY EQUAL.
C
C LIW    = THE LENGTH OF THE ARRAY IWORK, AS DECLARED BY THE USER.
C          (THIS WILL BE CHECKED BY THE SOLVER.)
C
C NOTE..  THE WORK ARRAYS MUST NOT BE ALTERED BETWEEN CALLS TO LSODE
C FOR THE SAME PROBLEM, EXCEPT POSSIBLY FOR THE CONDITIONAL AND
C OPTIONAL INPUTS, AND EXCEPT FOR THE LAST 3*NEQ WORDS OF RWORK.
C THE LATTER SPACE IS USED FOR INTERNAL SCRATCH SPACE, AND SO IS
C AVAILABLE FOR USE BY THE USER OUTSIDE LSODE BETWEEN CALLS, IF
C DESIRED (BUT NOT FOR USE BY F OR JAC).
C
C JAC    = THE NAME OF THE USER-SUPPLIED ROUTINE (MITER = 1 OR 4) TO
C          COMPUTE THE JACOBIAN MATRIX, DF/DY, AS A FUNCTION OF
C          THE SCALAR T AND THE VECTOR Y.  IT IS TO HAVE THE FORM
C               SUBROUTINE JAC (NEQ, T, Y, ML, MU, PD, NROWPD)
C               DIMENSION Y(1), PD(NROWPD,1)
C          WHERE NEQ, T, Y, ML, MU, AND NROWPD ARE INPUT AND THE ARRAY
C          PD IS TO BE LOADED WITH PARTIAL DERIVATIVES (ELEMENTS OF
C          THE JACOBIAN MATRIX) ON OUTPUT.  PD MUST BE GIVEN A FIRST
C          DIMENSION OF NROWPD.  T AND Y HAVE THE SAME MEANING AS IN
C          SUBROUTINE F.  (IN THE DIMENSION STATEMENT ABOVE, 1 IS A
C          DUMMY DIMENSION.. IT CAN BE REPLACED BY ANY VALUE.)
C               IN THE FULL MATRIX CASE (MITER = 1), ML AND MU ARE
C          IGNORED, AND THE JACOBIAN IS TO BE LOADED INTO PD IN
C          COLUMNWISE MANNER, WITH DF(I)/DY(J) LOADED INTO PD(I,J).
C               IN THE BAND MATRIX CASE (MITER = 4), THE ELEMENTS
C          WITHIN THE BAND ARE TO BE LOADED INTO PD IN COLUMNWISE
C          MANNER, WITH DIAGONAL LINES OF DF/DY LOADED INTO THE ROWS
C          OF PD.  THUS DF(I)/DY(J) IS TO BE LOADED INTO PD(I-J+MU+1,J).
C          ML AND MU ARE THE HALF-BANDWIDTH PARAMETERS (SEE IWORK).
C          THE LOCATIONS IN PD IN THE TWO TRIANGULAR AREAS WHICH
C          CORRESPOND TO NONEXISTENT MATRIX ELEMENTS CAN BE IGNORED
C          OR LOADED ARBITRARILY, AS THEY ARE OVERWRITTEN BY LSODE.
C               JAC NEED NOT PROVIDE DF/DY EXACTLY.  A CRUDE
C          APPROXIMATION (POSSIBLY WITH A SMALLER BANDWIDTH) WILL DO. 
C               IN EITHER CASE, PD IS PRESET TO ZERO BY THE SOLVER,
C          SO THAT ONLY THE NONZERO ELEMENTS NEED BE LOADED BY JAC.
C          EACH CALL TO JAC IS PRECEDED BY A CALL TO F WITH THE SAME
C          ARGUMENTS NEQ, T, AND Y.  THUS TO GAIN SOME EFFICIENCY,
C          INTERMEDIATE QUANTITIES SHARED BY BOTH CALCULATIONS MAY BE 
C          SAVED IN A USER COMMON BLOCK BY F AND NOT RECOMPUTED BY JAC,
C          IF DESIRED.  ALSO, JAC MAY ALTER THE Y ARRAY, IF DESIRED.
C          JAC MUST BE DECLARED EXTERNAL IN THE CALLING PROGRAM.
C               SUBROUTINE JAC MAY ACCESS USER-DEFINED QUANTITIES IN
C          NEQ(2),... AND/OR IN Y(NEQ(1)+1),... IF NEQ IS AN ARRAY
C          (DIMENSIONED IN JAC) AND/OR Y HAS LENGTH EXCEEDING NEQ(1). 
C          SEE THE DESCRIPTIONS OF NEQ AND Y ABOVE.
C
C MF     = THE METHOD FLAG.  USED ONLY FOR INPUT.  THE LEGAL VALUES OF
C          MF ARE 10, 11, 12, 13, 14, 15, 20, 21, 22, 23, 24, AND 25. 
C          MF HAS DECIMAL DIGITS METH AND MITER.. MF = 10*METH + MITER.
C          METH INDICATES THE BASIC LINEAR MULTISTEP METHOD..
C            METH = 1 MEANS THE IMPLICIT ADAMS METHOD.
C            METH = 2 MEANS THE METHOD BASED ON BACKWARD
C                     DIFFERENTIATION FORMULAS (BDF-S).
C          MITER INDICATES THE CORRECTOR ITERATION METHOD.. 
C            MITER = 0 MEANS FUNCTIONAL ITERATION (NO JACOBIAN MATRIX 
C                      IS INVOLVED).
C            MITER = 1 MEANS CHORD ITERATION WITH A USER-SUPPLIED
C                      FULL (NEQ BY NEQ) JACOBIAN.
C            MITER = 2 MEANS CHORD ITERATION WITH AN INTERNALLY
C                      GENERATED (DIFFERENCE QUOTIENT) FULL JACOBIAN
C                      (USING NEQ EXTRA CALLS TO F PER DF/DY VALUE).
C            MITER = 3 MEANS CHORD ITERATION WITH AN INTERNALLY
C                      GENERATED DIAGONAL JACOBIAN APPROXIMATION.
C                      (USING 1 EXTRA CALL TO F PER DF/DY EVALUATION).
C            MITER = 4 MEANS CHORD ITERATION WITH A USER-SUPPLIED
C                      BANDED JACOBIAN. 
C            MITER = 5 MEANS CHORD ITERATION WITH AN INTERNALLY
C                      GENERATED BANDED JACOBIAN (USING ML+MU+1 EXTRA 
C                      CALLS TO F PER DF/DY EVALUATION).
C          IF MITER = 1 OR 4, THE USER MUST SUPPLY A SUBROUTINE JAC
C          (THE NAME IS ARBITRARY) AS DESCRIBED ABOVE UNDER JAC.
C          FOR OTHER VALUES OF MITER, A DUMMY ARGUMENT CAN BE USED.
C-----------------------------------------------------------------------
C OPTIONAL INPUTS.
C
C THE FOLLOWING IS A LIST OF THE OPTIONAL INPUTS PROVIDED FOR IN THE
C CALL SEQUENCE.  (SEE ALSO PART II.)  FOR EACH SUCH INPUT VARIABLE,
C THIS TABLE LISTS ITS NAME AS USED IN THIS DOCUMENTATION, ITS
C LOCATION IN THE CALL SEQUENCE, ITS MEANING, AND THE DEFAULT VALUE.
C THE USE OF ANY OF THESE INPUTS REQUIRES IOPT = 1, AND IN THAT
C CASE ALL OF THESE INPUTS ARE EXAMINED.  A VALUE OF ZERO FOR ANY
C OF THESE OPTIONAL INPUTS WILL CAUSE THE DEFAULT VALUE TO BE USED.
C THUS TO USE A SUBSET OF THE OPTIONAL INPUTS, SIMPLY PRELOAD
C LOCATIONS 5 TO 10 IN RWORK AND IWORK TO 0.0 AND 0 RESPECTIVELY, AND 
C THEN SET THOSE OF INTEREST TO NONZERO VALUES.
C
C NAME    LOCATION      MEANING AND DEFAULT VALUE 
C
C H0      RWORK(5)  THE STEP SIZE TO BE ATTEMPTED ON THE FIRST STEP.
C                   THE DEFAULT VALUE IS DETERMINED BY THE SOLVER.
C
C HMAX    RWORK(6)  THE MAXIMUM ABSOLUTE STEP SIZE ALLOWED. 
C                   THE DEFAULT VALUE IS INFINITE.
C
C HMIN    RWORK(7)  THE MINIMUM ABSOLUTE STEP SIZE ALLOWED. 
C                   THE DEFAULT VALUE IS 0.  (THIS LOWER BOUND IS NOT 
C                   ENFORCED ON THE FINAL STEP BEFORE REACHING TCRIT
C                   WHEN ITASK = 4 OR 5.)
C
C MAXORD  IWORK(5)  THE MAXIMUM ORDER TO BE ALLOWED.  THE DEFAULT
C                   VALUE IS 12 IF METH = 1, AND 5 IF METH = 2.
C                   IF MAXORD EXCEEDS THE DEFAULT VALUE, IT WILL
C                   BE REDUCED TO THE DEFAULT VALUE.
C                   IF MAXORD IS CHANGED DURING THE PROBLEM, IT MAY
C                   CAUSE THE CURRENT ORDER TO BE REDUCED.
C
C MXSTEP  IWORK(6)  MAXIMUM NUMBER OF (INTERNALLY DEFINED) STEPS
C                   ALLOWED DURING ONE CALL TO THE SOLVER.
C                   THE DEFAULT VALUE IS 500.
C
C MXHNIL  IWORK(7)  MAXIMUM NUMBER OF MESSAGES PRINTED (PER PROBLEM)
C                   WARNING THAT T + H = T ON A STEP (H = STEP SIZE). 
C                   THIS MUST BE POSITIVE TO RESULT IN A NON-DEFAULT
C                   VALUE.  THE DEFAULT VALUE IS 10.
C-----------------------------------------------------------------------
C OPTIONAL OUTPUTS. 
C
C AS OPTIONAL ADDITIONAL OUTPUT FROM LSODE, THE VARIABLES LISTED
C BELOW ARE QUANTITIES RELATED TO THE PERFORMANCE OF LSODE
C WHICH ARE AVAILABLE TO THE USER.  THESE ARE COMMUNICATED BY WAY OF
C THE WORK ARRAYS, BUT ALSO HAVE INTERNAL MNEMONIC NAMES AS SHOWN.
C EXCEPT WHERE STATED OTHERWISE, ALL OF THESE OUTPUTS ARE DEFINED
C ON ANY SUCCESSFUL RETURN FROM LSODE, AND ON ANY RETURN WITH
C ISTATE = -1, -2, -4, -5, OR -6.  ON AN ILLEGAL INPUT RETURN
C (ISTATE = -3), THEY WILL BE UNCHANGED FROM THEIR EXISTING VALUES
C (IF ANY), EXCEPT POSSIBLY FOR TOLSF, LENRW, AND LENIW.
C ON ANY ERROR RETURN, OUTPUTS RELEVANT TO THE ERROR WILL BE DEFINED, 
C AS NOTED BELOW.
C
C NAME    LOCATION      MEANING
C
C HU      RWORK(11) THE STEP SIZE IN T LAST USED (SUCCESSFULLY).
C
C HCUR    RWORK(12) THE STEP SIZE TO BE ATTEMPTED ON THE NEXT STEP.
C
C TCUR    RWORK(13) THE CURRENT VALUE OF THE INDEPENDENT VARIABLE
C                   WHICH THE SOLVER HAS ACTUALLY REACHED, I.E. THE
C                   CURRENT INTERNAL MESH POINT IN T.  ON OUTPUT, TCUR
C                   WILL ALWAYS BE AT LEAST AS FAR AS THE ARGUMENT
C                   T, BUT MAY BE FARTHER (IF INTERPOLATION WAS DONE).
C
C TOLSF   RWORK(14) A TOLERANCE SCALE FACTOR, GREATER THAN 1.0,
C                   COMPUTED WHEN A REQUEST FOR TOO MUCH ACCURACY WAS 
C                   DETECTED (ISTATE = -3 IF DETECTED AT THE START OF 
C                   THE PROBLEM, ISTATE = -2 OTHERWISE).  IF ITOL IS
C                   LEFT UNALTERED BUT RTOL AND ATOL ARE UNIFORMLY
C                   SCALED UP BY A FACTOR OF TOLSF FOR THE NEXT CALL, 
C                   THEN THE SOLVER IS DEEMED LIKELY TO SUCCEED.
C                   (THE USER MAY ALSO IGNORE TOLSF AND ALTER THE
C                   TOLERANCE PARAMETERS IN ANY OTHER WAY APPROPRIATE.)
C
C NST     IWORK(11) THE NUMBER OF STEPS TAKEN FOR THE PROBLEM SO FAR. 
C
C NFE     IWORK(12) THE NUMBER OF F EVALUATIONS FOR THE PROBLEM SO FAR.
C
C NJE     IWORK(13) THE NUMBER OF JACOBIAN EVALUATIONS (AND OF MATRIX 
C                   LU DECOMPOSITIONS) FOR THE PROBLEM SO FAR.
C
C NQU     IWORK(14) THE METHOD ORDER LAST USED (SUCCESSFULLY).
C
C NQCUR   IWORK(15) THE ORDER TO BE ATTEMPTED ON THE NEXT STEP.
C
C IMXER   IWORK(16) THE INDEX OF THE COMPONENT OF LARGEST MAGNITUDE IN
C                   THE WEIGHTED LOCAL ERROR VECTOR ( E(I)/EWT(I) ),
C                   ON AN ERROR RETURN WITH ISTATE = -4 OR -5.
C
C LENRW   IWORK(17) THE LENGTH OF RWORK ACTUALLY REQUIRED.
C                   THIS IS DEFINED ON NORMAL RETURNS AND ON AN ILLEGAL
C                   INPUT RETURN FOR INSUFFICIENT STORAGE.
C
C LENIW   IWORK(18) THE LENGTH OF IWORK ACTUALLY REQUIRED.
C                   THIS IS DEFINED ON NORMAL RETURNS AND ON AN ILLEGAL
C                   INPUT RETURN FOR INSUFFICIENT STORAGE.
C
C THE FOLLOWING TWO ARRAYS ARE SEGMENTS OF THE RWORK ARRAY WHICH
C MAY ALSO BE OF INTEREST TO THE USER AS OPTIONAL OUTPUTS.
C FOR EACH ARRAY, THE TABLE BELOW GIVES ITS INTERNAL NAME,
C ITS BASE ADDRESS IN RWORK, AND ITS DESCRIPTION. 
C
C NAME    BASE ADDRESS      DESCRIPTION 
C
C YH      21             THE NORDSIECK HISTORY ARRAY, OF SIZE NYH BY
C                        (NQCUR + 1), WHERE NYH IS THE INITIAL VALUE
C                        OF NEQ.  FOR J = 0,1,...,NQCUR, COLUMN J+1
C                        OF YH CONTAINS HCUR**J/FACTORIAL(J) TIMES
C                        THE J-TH DERIVATIVE OF THE INTERPOLATING
C                        POLYNOMIAL CURRENTLY REPRESENTING THE SOLUTION,
C                        EVALUATED AT T = TCUR.
C
C ACOR     LENRW-NEQ+1   ARRAY OF SIZE NEQ USED FOR THE ACCUMULATED
C                        CORRECTIONS ON EACH STEP, SCALED ON OUTPUT
C                        TO REPRESENT THE ESTIMATED LOCAL ERROR IN Y
C                        ON THE LAST STEP.  THIS IS THE VECTOR E IN
C                        THE DESCRIPTION OF THE ERROR CONTROL.  IT IS 
C                        DEFINED ONLY ON A SUCCESSFUL RETURN FROM LSODE.
C
C-----------------------------------------------------------------------
C PART II.  OTHER ROUTINES CALLABLE.
C
C THE FOLLOWING ARE OPTIONAL CALLS WHICH THE USER MAY MAKE TO
C GAIN ADDITIONAL CAPABILITIES IN CONJUNCTION WITH LSODE.
C (THE ROUTINES XSETUN AND XSETF ARE DESIGNED TO CONFORM TO THE
C SLATEC ERROR HANDLING PACKAGE.)
C
C     FORM OF CALL                  FUNCTION
C   CALL XSETUN(LUN)          SET THE LOGICAL UNIT NUMBER, LUN, FOR
C                             OUTPUT OF MESSAGES FROM LSODE, IF
C                             THE DEFAULT IS NOT DESIRED.
C                             THE DEFAULT VALUE OF LUN IS 6.
C
C   CALL XSETF(MFLAG)         SET A FLAG TO CONTROL THE PRINTING OF
C                             MESSAGES BY LSODE.
C                             MFLAG = 0 MEANS DO NOT PRINT. (DANGER.. 
C                             THIS RISKS LOSING VALUABLE INFORMATION.)
C                             MFLAG = 1 MEANS PRINT (THE DEFAULT).
C
C                             EITHER OF THE ABOVE CALLS MAY BE MADE AT
C                             ANY TIME AND WILL TAKE EFFECT IMMEDIATELY.
C
C   CALL SRCOM(RSAV,ISAV,JOB) SAVES AND RESTORES THE CONTENTS OF
C                             THE INTERNAL COMMON BLOCKS USED BY
C                             LSODE (SEE PART III BELOW).
C                             RSAV MUST BE A REAL ARRAY OF LENGTH 218 
C                             OR MORE, AND ISAV MUST BE AN INTEGER
C                             ARRAY OF LENGTH 41 OR MORE.
C                             JOB=1 MEANS SAVE COMMON INTO RSAV/ISAV. 
C                             JOB=2 MEANS RESTORE COMMON FROM RSAV/ISAV.
C                                SRCOM IS USEFUL IF ONE IS
C                             INTERRUPTING A RUN AND RESTARTING
C                             LATER, OR ALTERNATING BETWEEN TWO OR
C                             MORE PROBLEMS SOLVED WITH LSODE.
C
C   CALL INTDY(,,,,,)         PROVIDE DERIVATIVES OF Y, OF VARIOUS
C        (SEE BELOW)          ORDERS, AT A SPECIFIED POINT T, IF
C                             DESIRED.  IT MAY BE CALLED ONLY AFTER
C                             A SUCCESSFUL RETURN FROM LSODE.
C
C THE DETAILED INSTRUCTIONS FOR USING INTDY ARE AS FOLLOWS. 
C THE FORM OF THE CALL IS..
C
C   CALL INTDY (T, K, RWORK(21), NYH, DKY, IFLAG) 
C
C THE INPUT PARAMETERS ARE..
C
C T         = VALUE OF INDEPENDENT VARIABLE WHERE ANSWERS ARE DESIRED 
C             (NORMALLY THE SAME AS THE T LAST RETURNED BY LSODE).
C             FOR VALID RESULTS, T MUST LIE BETWEEN TCUR - HU AND TCUR.
C             (SEE OPTIONAL OUTPUTS FOR TCUR AND HU.)
C K         = INTEGER ORDER OF THE DERIVATIVE DESIRED.  K MUST SATISFY
C             0 .LE. K .LE. NQCUR, WHERE NQCUR IS THE CURRENT ORDER
C             (SEE OPTIONAL OUTPUTS).  THE CAPABILITY CORRESPONDING
C             TO K = 0, I.E. COMPUTING Y(T), IS ALREADY PROVIDED
C             BY LSODE DIRECTLY.  SINCE NQCUR .GE. 1, THE FIRST
C             DERIVATIVE DY/DT IS ALWAYS AVAILABLE WITH INTDY.
C RWORK(21) = THE BASE ADDRESS OF THE HISTORY ARRAY YH.
C NYH       = COLUMN LENGTH OF YH, EQUAL TO THE INITIAL VALUE OF NEQ. 
C
C THE OUTPUT PARAMETERS ARE.. 
C
C DKY       = A REAL ARRAY OF LENGTH NEQ CONTAINING THE COMPUTED VALUE
C             OF THE K-TH DERIVATIVE OF Y(T).
C IFLAG     = INTEGER FLAG, RETURNED AS 0 IF K AND T WERE LEGAL,
C             -1 IF K WAS ILLEGAL, AND -2 IF T WAS ILLEGAL. 
C             ON AN ERROR RETURN, A MESSAGE IS ALSO WRITTEN.
C-----------------------------------------------------------------------
C PART III.  COMMON BLOCKS.
C
C IF LSODE IS TO BE USED IN AN OVERLAY SITUATION, THE USER
C MUST DECLARE, IN THE PRIMARY OVERLAY, THE VARIABLES IN..
C   (1) THE CALL SEQUENCE TO LSODE,
C   (2) THE INTERNAL COMMON BLOCK
C         /LS0001/  OF LENGTH  257  (218 DOUBLE PRECISION WORDS
C                         FOLLOWED BY 39 INTEGER WORDS),
C
C IF LSODE IS USED ON A SYSTEM IN WHICH THE CONTENTS OF INTERNAL
C COMMON BLOCKS ARE NOT PRESERVED BETWEEN CALLS, THE USER SHOULD
C DECLARE THE ABOVE TWO COMMON BLOCKS IN HIS MAIN PROGRAM TO INSURE
C THAT THEIR CONTENTS ARE PRESERVED.
C
C IF THE SOLUTION OF A GIVEN PROBLEM BY LSODE IS TO BE INTERRUPTED
C AND THEN LATER CONTINUED, SUCH AS WHEN RESTARTING AN INTERRUPTED RUN
C OR ALTERNATING BETWEEN TWO OR MORE PROBLEMS, THE USER SHOULD SAVE,
C FOLLOWING THE RETURN FROM THE LAST LSODE CALL PRIOR TO THE
C INTERRUPTION, THE CONTENTS OF THE CALL SEQUENCE VARIABLES AND THE
C INTERNAL COMMON BLOCKS, AND LATER RESTORE THESE VALUES BEFORE THE
C NEXT LSODE CALL FOR THAT PROBLEM.  TO SAVE AND RESTORE THE COMMON
C BLOCKS, USE SUBROUTINE SRCOM (SEE PART II ABOVE).
C
C-----------------------------------------------------------------------
C PART IV.  OPTIONALLY REPLACEABLE SOLVER ROUTINES.
C
C BELOW ARE DESCRIPTIONS OF TWO ROUTINES IN THE LSODE PACKAGE WHICH
C RELATE TO THE MEASUREMENT OF ERRORS.  EITHER ROUTINE CAN BE
C REPLACED BY A USER-SUPPLIED VERSION, IF DESIRED.  HOWEVER, SINCE SUCH
C A REPLACEMENT MAY HAVE A MAJOR IMPACT ON PERFORMANCE, IT SHOULD BE
C DONE ONLY WHEN ABSOLUTELY NECESSARY, AND ONLY WITH GREAT CAUTION.
C (NOTE.. THE MEANS BY WHICH THE PACKAGE VERSION OF A ROUTINE IS
C SUPERSEDED BY THE USER-S VERSION MAY BE SYSTEM-DEPENDENT.)
C
C (A) EWSET.
C THE FOLLOWING SUBROUTINE IS CALLED JUST BEFORE EACH INTERNAL
C INTEGRATION STEP, AND SETS THE ARRAY OF ERROR WEIGHTS, EWT, AS
C DESCRIBED UNDER ITOL/RTOL/ATOL ABOVE..
C     SUBROUTINE EWSET (NEQ, ITOL, RTOL, ATOL, YCUR, EWT)
C WHERE NEQ, ITOL, RTOL, AND ATOL ARE AS IN THE LSODE CALL SEQUENCE,
C YCUR CONTAINS THE CURRENT DEPENDENT VARIABLE VECTOR, AND
C EWT IS THE ARRAY OF WEIGHTS SET BY EWSET.
C
C IF THE USER SUPPLIES THIS SUBROUTINE, IT MUST RETURN IN EWT(I)
C (I = 1,...,NEQ) A POSITIVE QUANTITY SUITABLE FOR COMPARING ERRORS
C IN Y(I) TO.  THE EWT ARRAY RETURNED BY EWSET IS PASSED TO THE
C VNORM ROUTINE (SEE BELOW), AND ALSO USED BY LSODE IN THE COMPUTATION
C OF THE OPTIONAL OUTPUT IMXER, THE DIAGONAL JACOBIAN APPROXIMATION,
C AND THE INCREMENTS FOR DIFFERENCE QUOTIENT JACOBIANS.
C
C IN THE USER-SUPPLIED VERSION OF EWSET, IT MAY BE DESIRABLE TO USE
C THE CURRENT VALUES OF DERIVATIVES OF Y.  DERIVATIVES UP TO ORDER NQ 
C ARE AVAILABLE FROM THE HISTORY ARRAY YH, DESCRIBED ABOVE UNDER
C OPTIONAL OUTPUTS.  IN EWSET, YH IS IDENTICAL TO THE YCUR ARRAY,
C EXTENDED TO NQ + 1 COLUMNS WITH A COLUMN LENGTH OF NYH AND SCALE
C FACTORS OF H**J/FACTORIAL(J).  ON THE FIRST CALL FOR THE PROBLEM,
C GIVEN BY NST = 0, NQ IS 1 AND H IS TEMPORARILY SET TO 1.0.
C THE QUANTITIES NQ, NYH, H, AND NST CAN BE OBTAINED BY INCLUDING
C IN EWSET THE STATEMENTS..
C     DOUBLE PRECISION H, RLS 
C     COMMON /LS0001/ RLS(218),ILS(39)
C     NQ = ILS(35)
C     NYH = ILS(14) 
C     NST = ILS(36) 
C     H = RLS(212)
C THUS, FOR EXAMPLE, THE CURRENT VALUE OF DY/DT CAN BE OBTAINED AS
C YCUR(NYH+I)/H  (I=1,...,NEQ)  (AND THE DIVISION BY H IS
C UNNECESSARY WHEN NST = 0).
C
C (B) VNORM.
C THE FOLLOWING IS A REAL FUNCTION ROUTINE WHICH COMPUTES THE WEIGHTED
C ROOT-MEAN-SQUARE NORM OF A VECTOR V.. 
C     D = VNORM (N, V, W)
C WHERE.. 
C   N = THE LENGTH OF THE VECTOR,
C   V = REAL ARRAY OF LENGTH N CONTAINING THE VECTOR,
C   W = REAL ARRAY OF LENGTH N CONTAINING WEIGHTS,
C   D = SQRT( (1/N) * SUM(V(I)*W(I))**2 ).
C VNORM IS CALLED WITH N = NEQ AND WITH W(I) = 1.0/EWT(I), WHERE
C EWT IS AS SET BY SUBROUTINE EWSET.
C
C IF THE USER SUPPLIES THIS FUNCTION, IT SHOULD RETURN A NON-NEGATIVE 
C VALUE OF VNORM SUITABLE FOR USE IN THE ERROR CONTROL IN LSODE.
C NONE OF THE ARGUMENTS SHOULD BE ALTERED BY VNORM.
C FOR EXAMPLE, A USER-SUPPLIED VNORM ROUTINE MIGHT..
C   -SUBSTITUTE A MAX-NORM OF (V(I)*W(I)) FOR THE RMS-NORM, OR
C   -IGNORE SOME COMPONENTS OF V IN THE NORM, WITH THE EFFECT OF
C    SUPPRESSING THE ERROR CONTROL ON THOSE COMPONENTS OF Y.
C-----------------------------------------------------------------------
C-----------------------------------------------------------------------
C OTHER ROUTINES IN THE LSODE PACKAGE.
C
C IN ADDITION TO SUBROUTINE LSODE, THE LSODE PACKAGE INCLUDES THE
C FOLLOWING SUBROUTINES AND FUNCTION ROUTINES..
C  INTDY    COMPUTES AN INTERPOLATED VALUE OF THE Y VECTOR AT T = TOUT.
C  STODE    IS THE CORE INTEGRATOR, WHICH DOES ONE STEP OF THE
C           INTEGRATION AND THE ASSOCIATED ERROR CONTROL.
C  CFODE    SETS ALL METHOD COEFFICIENTS AND TEST CONSTANTS.
C  PREPJ    COMPUTES AND PREPROCESSES THE JACOBIAN MATRIX J = DF/DY
C           AND THE NEWTON ITERATION MATRIX P = I - H*L0*J. 
C  SOLSY    MANAGES SOLUTION OF LINEAR SYSTEM IN CHORD ITERATION.
C  EWSET    SETS THE ERROR WEIGHT VECTOR EWT BEFORE EACH STEP.
C  VNORM    COMPUTES THE WEIGHTED R.M.S. NORM OF A VECTOR.
C  SRCOM    IS A USER-CALLABLE ROUTINE TO SAVE AND RESTORE
C           THE CONTENTS OF THE INTERNAL COMMON BLOCKS.
C  DGETRF AND DGETRS   ARE ROUTINES FROM LAPACK FOR SOLVING FULL
C           SYSTEMS OF LINEAR ALGEBRAIC EQUATIONS.
C  DGBTRF AND DGBTRS   ARE ROUTINES FROM LAPACK FOR SOLVING BANDED
C           LINEAR SYSTEMS.
C  DAXPY, DSCAL, IDAMAX, AND DDOT   ARE BASIC LINEAR ALGEBRA MODULES
C           (BLAS) USED BY THE ABOVE LINPACK ROUTINES.
C  D1MACH   COMPUTES THE UNIT ROUNDOFF IN A MACHINE-INDEPENDENT MANNER.
C  XERRWD, XSETUN, AND XSETF   HANDLE THE PRINTING OF ALL ERROR
C           MESSAGES AND WARNINGS.  XERRWD IS MACHINE-DEPENDENT.
C NOTE..  VNORM, IDAMAX, DDOT, AND D1MACH ARE FUNCTION ROUTINES.
C ALL THE OTHERS ARE SUBROUTINES.
C
C THE INTRINSIC AND EXTERNAL ROUTINES USED BY LSODE ARE..
C DABS, DMAX1, DMIN1, DBLE, MAX0, MIN0, MOD, DSIGN, DSQRT, AND WRITE. 
C
C A BLOCK DATA SUBPROGRAM IS ALSO INCLUDED WITH THE PACKAGE,
C FOR LOADING SOME OF THE VARIABLES IN INTERNAL COMMON.
C
C-----------------------------------------------------------------------
C THE FOLLOWING CARD IS FOR OPTIMIZED COMPILATION ON LLNL COMPILERS.
CLLL. OPTIMIZE
C-----------------------------------------------------------------------
      EXTERNAL PREPJ, SOLSY
      INTEGER ILLIN, INIT, LYH, LEWT, LACOR, LSAVF, LWM, LIWM,
     1   MXSTEP, MXHNIL, NHNIL, NTREP, NSLAST, NYH, IOWNS
      INTEGER ICF, IERPJ, IERSL, JCUR, JSTART, KFLAG, L, METH, MITER, 
     1   MAXORD, MAXCOR, MSBP, MXNCF, N, NQ, NST, NFE, NJE, NQU
      INTEGER I, I1, I2, IFLAG, IMXER, KGO, LF0,
     1   LENIW, LENRW, LENWM, ML, MORD, MU, MXHNL0, MXSTP0
      DOUBLE PRECISION ROWNS, 
     1   CCMAX, EL0, H, HMIN, HMXI, HU, RC, TN, UROUND
      DOUBLE PRECISION ATOLI, AYI, BIG, EWTI, H0, HMAX, HMX, RH, RTOLI,
     1   TCRIT, TDIST, TNEXT, TOL, TOLSF, TP, SIZE, SUM, W0,
     2   D1MACH, VNORM
      DIMENSION MORD(2)
      LOGICAL IHIT
C-----------------------------------------------------------------------
C THE FOLLOWING INTERNAL COMMON BLOCK CONTAINS
C (A) VARIABLES WHICH ARE LOCAL TO ANY SUBROUTINE BUT WHOSE VALUES MUST
C     BE PRESERVED BETWEEN CALLS TO THE ROUTINE (OWN VARIABLES), AND
C (B) VARIABLES WHICH ARE COMMUNICATED BETWEEN SUBROUTINES. 
C THE STRUCTURE OF THE BLOCK IS AS FOLLOWS..  ALL REAL VARIABLES ARE
C LISTED FIRST, FOLLOWED BY ALL INTEGERS.  WITHIN EACH TYPE, THE
C VARIABLES ARE GROUPED WITH THOSE LOCAL TO SUBROUTINE LSODE FIRST,
C THEN THOSE LOCAL TO SUBROUTINE STODE, AND FINALLY THOSE USED
C FOR COMMUNICATION.  THE BLOCK IS DECLARED IN SUBROUTINES
C LSODE, INTDY, STODE, PREPJ, AND SOLSY.  GROUPS OF VARIABLES ARE
C REPLACED BY DUMMY ARRAYS IN THE COMMON DECLARATIONS IN ROUTINES
C WHERE THOSE VARIABLES ARE NOT USED.
C-----------------------------------------------------------------------
      COMMON /LS0001/ ROWNS(209),
     1   CCMAX, EL0, H, HMIN, HMXI, HU, RC, TN, UROUND,
     2   ILLIN, INIT, LYH, LEWT, LACOR, LSAVF, LWM, LIWM,
     3   MXSTEP, MXHNIL, NHNIL, NTREP, NSLAST, NYH, IOWNS(6),
     4   ICF, IERPJ, IERSL, JCUR, JSTART, KFLAG, L, METH, MITER,
     5   MAXORD, MAXCOR, MSBP, MXNCF, N, NQ, NST, NFE, NJE, NQU
C
      DATA  MORD(1),MORD(2)/12,5/, MXSTP0/500/, MXHNL0/10/
C-----------------------------------------------------------------------
C BLOCK A.
C THIS CODE BLOCK IS EXECUTED ON EVERY CALL.
C IT TESTS ISTATE AND ITASK FOR LEGALITY AND BRANCHES APPROPRIATELY.
C IF ISTATE .GT. 1 BUT THE FLAG INIT SHOWS THAT INITIALIZATION HAS
C NOT YET BEEN DONE, AN ERROR RETURN OCCURS.
C IF ISTATE = 1 AND TOUT = T, JUMP TO BLOCK G AND RETURN IMMEDIATELY. 
C-----------------------------------------------------------------------
      IF (ISTATE .LT. 1 .OR. ISTATE .GT. 3) GO TO 601
      IF (ITASK .LT. 1 .OR. ITASK .GT. 5) GO TO 602
      IF (ISTATE .EQ. 1) GO TO 10
      IF (INIT .EQ. 0) GO TO 603
      IF (ISTATE .EQ. 2) GO TO 200
      GO TO 20
 10   INIT = 0
      IF (TOUT .EQ. T) GO TO 430
 20   NTREP = 0
C-----------------------------------------------------------------------
C BLOCK B.
C THE NEXT CODE BLOCK IS EXECUTED FOR THE INITIAL CALL (ISTATE = 1),
C OR FOR A CONTINUATION CALL WITH PARAMETER CHANGES (ISTATE = 3).
C IT CONTAINS CHECKING OF ALL INPUTS AND VARIOUS INITIALIZATIONS.
C
C FIRST CHECK LEGALITY OF THE NON-OPTIONAL INPUTS NEQ, ITOL, IOPT,
C MF, ML, AND MU.
C-----------------------------------------------------------------------
      IF (NEQ(1) .LE. 0) GO TO 604
      IF (ISTATE .EQ. 1) GO TO 25
      IF (NEQ(1) .GT. N) GO TO 605
 25   N = NEQ(1)
      IF (ITOL .LT. 1 .OR. ITOL .GT. 4) GO TO 606 
      IF (IOPT .LT. 0 .OR. IOPT .GT. 1) GO TO 607 
      METH = MF/10
      MITER = MF - 10*METH
      IF (METH .LT. 1 .OR. METH .GT. 2) GO TO 608 
      IF (MITER .LT. 0 .OR. MITER .GT. 5) GO TO 608
      IF (MITER .LE. 3) GO TO 30
      ML = IWORK(1) 
      MU = IWORK(2) 
      IF (ML .LT. 0 .OR. ML .GE. N) GO TO 609
      IF (MU .LT. 0 .OR. MU .GE. N) GO TO 610
 30   CONTINUE
C NEXT PROCESS AND CHECK THE OPTIONAL INPUTS. --------------------------
      IF (IOPT .EQ. 1) GO TO 40
      MAXORD = MORD(METH)
      MXSTEP = MXSTP0
      MXHNIL = MXHNL0
      IF (ISTATE .EQ. 1) H0 = 0.0D0
      HMXI = 0.0D0
      HMIN = 0.0D0
      GO TO 60
 40   MAXORD = IWORK(5)
      IF (MAXORD .LT. 0) GO TO 611
      IF (MAXORD .EQ. 0) MAXORD = 100
      MAXORD = MIN0(MAXORD,MORD(METH))
      MXSTEP = IWORK(6)
      IF (MXSTEP .LT. 0) GO TO 612
      IF (MXSTEP .EQ. 0) MXSTEP = MXSTP0
      MXHNIL = IWORK(7)
      IF (MXHNIL .LT. 0) GO TO 613
      IF (MXHNIL .EQ. 0) MXHNIL = MXHNL0
      IF (ISTATE .NE. 1) GO TO 50
      H0 = RWORK(5) 
      IF ((TOUT - T)*H0 .LT. 0.0D0) GO TO 614
 50   HMAX = RWORK(6)
      IF (HMAX .LT. 0.0D0) GO TO 615
      HMXI = 0.0D0
      IF (HMAX .GT. 0.0D0) HMXI = 1.0D0/HMAX
      HMIN = RWORK(7)
      IF (HMIN .LT. 0.0D0) GO TO 616
C-----------------------------------------------------------------------
C SET WORK ARRAY POINTERS AND CHECK LENGTHS LRW AND LIW.
C POINTERS TO SEGMENTS OF RWORK AND IWORK ARE NAMED BY PREFIXING L TO 
C THE NAME OF THE SEGMENT.  E.G., THE SEGMENT YH STARTS AT RWORK(LYH).
C SEGMENTS OF RWORK (IN ORDER) ARE DENOTED  YH, WM, EWT, SAVF, ACOR.
C-----------------------------------------------------------------------
 60   LYH = 21
      IF (ISTATE .EQ. 1) NYH = N
      LWM = LYH + (MAXORD + 1)*NYH
      IF (MITER .EQ. 0) LENWM = 0
      IF (MITER .EQ. 1 .OR. MITER .EQ. 2) LENWM = N*N + 2
      IF (MITER .EQ. 3) LENWM = N + 2
      IF (MITER .GE. 4) LENWM = (2*ML + MU + 1)*N + 2
      LEWT = LWM + LENWM
      LSAVF = LEWT + N
      LACOR = LSAVF + N
      LENRW = LACOR + N - 1
      IWORK(17) = LENRW
      LIWM = 1
      LENIW = 20 + N
      IF (MITER .EQ. 0 .OR. MITER .EQ. 3) LENIW = 20
      IWORK(18) = LENIW
      IF (LENRW .GT. LRW) GO TO 617
      IF (LENIW .GT. LIW) GO TO 618
C CHECK RTOL AND ATOL FOR LEGALITY. ------------------------------------
      RTOLI = RTOL(1)
      ATOLI = ATOL(1)
      DO 70 I = 1,N 
        IF (ITOL .GE. 3) RTOLI = RTOL(I)
        IF (ITOL .EQ. 2 .OR. ITOL .EQ. 4) ATOLI = ATOL(I)
        IF (RTOLI .LT. 0.0D0) GO TO 619 
        IF (ATOLI .LT. 0.0D0) GO TO 620 
 70     CONTINUE
      IF (ISTATE .EQ. 1) GO TO 100
C IF ISTATE = 3, SET FLAG TO SIGNAL PARAMETER CHANGES TO STODE. --------
      JSTART = -1
      IF (NQ .LE. MAXORD) GO TO 90
C MAXORD WAS REDUCED BELOW NQ.  COPY YH(*,MAXORD+2) INTO SAVF. ---------
      DO 80 I = 1,N 
 80     RWORK(I+LSAVF-1) = RWORK(I+LWM-1)
C RELOAD WM(1) = RWORK(LWM), SINCE LWM MAY HAVE CHANGED. ---------------
 90   IF (MITER .GT. 0) RWORK(LWM) = DSQRT(UROUND)
      IF (N .EQ. NYH) GO TO 200
C NEQ WAS REDUCED.  ZERO PART OF YH TO AVOID UNDEFINED REFERENCES. -----
      I1 = LYH + L*NYH
      I2 = LYH + (MAXORD + 1)*NYH - 1
      IF (I1 .GT. I2) GO TO 200
      DO 95 I = I1,I2
 95     RWORK(I) = 0.0D0
      GO TO 200
C-----------------------------------------------------------------------
C BLOCK C.
C THE NEXT BLOCK IS FOR THE INITIAL CALL ONLY (ISTATE = 1). 
C IT CONTAINS ALL REMAINING INITIALIZATIONS, THE INITIAL CALL TO F,
C AND THE CALCULATION OF THE INITIAL STEP SIZE.
C THE ERROR WEIGHTS IN EWT ARE INVERTED AFTER BEING LOADED. 
C-----------------------------------------------------------------------
 100  UROUND = D1MACH(4)
      TN = T
      IF (ITASK .NE. 4 .AND. ITASK .NE. 5) GO TO 110
      TCRIT = RWORK(1)
      IF ((TCRIT - TOUT)*(TOUT - T) .LT. 0.0D0) GO TO 625
      IF (H0 .NE. 0.0D0 .AND. (T + H0 - TCRIT)*H0 .GT. 0.0D0)
     1   H0 = TCRIT - T
 110  JSTART = 0
      IF (MITER .GT. 0) RWORK(LWM) = DSQRT(UROUND)
      NHNIL = 0
      NST = 0
      NJE = 0
      NSLAST = 0
      HU = 0.0D0
      NQU = 0
      CCMAX = 0.3D0 
      MAXCOR = 3
      MSBP = 20
      MXNCF = 10
C INITIAL CALL TO F.  (LF0 POINTS TO YH(*,2).) -------------------------
      LF0 = LYH + NYH
      IERR = 0
      CALL F (NEQ, T, Y, RWORK(LF0), IERR)
      IF (IERR .LT. 0) THEN
        ISTATE = -13
        RETURN
      ENDIF
      NFE = 1
C LOAD THE INITIAL VALUE VECTOR IN YH. ---------------------------------
      DO 115 I = 1,N
 115    RWORK(I+LYH-1) = Y(I) 
C LOAD AND INVERT THE EWT ARRAY.  (H IS TEMPORARILY SET TO 1.0.) -------
      NQ = 1
      H = 1.0D0
      CALL EWSET (N, ITOL, RTOL, ATOL, RWORK(LYH), RWORK(LEWT))
      DO 120 I = 1,N
        IF (RWORK(I+LEWT-1) .LE. 0.0D0) GO TO 621 
 120    RWORK(I+LEWT-1) = 1.0D0/RWORK(I+LEWT-1)
C-----------------------------------------------------------------------
C THE CODING BELOW COMPUTES THE STEP SIZE, H0, TO BE ATTEMPTED ON THE 
C FIRST STEP, UNLESS THE USER HAS SUPPLIED A VALUE FOR THIS.
C FIRST CHECK THAT TOUT - T DIFFERS SIGNIFICANTLY FROM ZERO.
C A SCALAR TOLERANCE QUANTITY TOL IS COMPUTED, AS MAX(RTOL(I))
C IF THIS IS POSITIVE, OR MAX(ATOL(I)/ABS(Y(I))) OTHERWISE, ADJUSTED
C SO AS TO BE BETWEEN 100*UROUND AND 1.0E-3.
C THEN THE COMPUTED VALUE H0 IS GIVEN BY..
C                                      NEQ
C   H0**2 = TOL / ( W0**-2 + (1/NEQ) * SUM ( F(I)/YWT(I) )**2  )
C                                       1
C WHERE   W0     = MAX ( ABS(T), ABS(TOUT) ),
C         F(I)   = I-TH COMPONENT OF INITIAL VALUE OF F,
C         YWT(I) = EWT(I)/TOL  (A WEIGHT FOR Y(I)).
C THE SIGN OF H0 IS INFERRED FROM THE INITIAL VALUES OF TOUT AND T.
C-----------------------------------------------------------------------
      IF (H0 .NE. 0.0D0) GO TO 180
      TDIST = DABS(TOUT - T)
      W0 = DMAX1(DABS(T),DABS(TOUT))
      IF (TDIST .LT. 2.0D0*UROUND*W0) GO TO 622
      TOL = RTOL(1) 
      IF (ITOL .LE. 2) GO TO 140
      DO 130 I = 1,N
 130    TOL = DMAX1(TOL,RTOL(I))
 140  IF (TOL .GT. 0.0D0) GO TO 160
      ATOLI = ATOL(1)
      DO 150 I = 1,N
        IF (ITOL .EQ. 2 .OR. ITOL .EQ. 4) ATOLI = ATOL(I)
        AYI = DABS(Y(I))
        IF (AYI .NE. 0.0D0) TOL = DMAX1(TOL,ATOLI/AYI)
 150    CONTINUE
 160  TOL = DMAX1(TOL,100.0D0*UROUND)
      TOL = DMIN1(TOL,0.001D0)
      SUM = VNORM (N, RWORK(LF0), RWORK(LEWT))
      SUM = 1.0D0/(TOL*W0*W0) + TOL*SUM**2
      H0 = 1.0D0/DSQRT(SUM)
      H0 = DMIN1(H0,TDIST)
      H0 = DSIGN(H0,TOUT-T)
C ADJUST H0 IF NECESSARY TO MEET HMAX BOUND. ---------------------------
 180  RH = DABS(H0)*HMXI
      IF (RH .GT. 1.0D0) H0 = H0/RH
C LOAD H WITH H0 AND SCALE YH(*,2) BY H0. ------------------------------
      H = H0
      DO 190 I = 1,N
 190    RWORK(I+LF0-1) = H0*RWORK(I+LF0-1)
      GO TO 270
C-----------------------------------------------------------------------
C BLOCK D.
C THE NEXT CODE BLOCK IS FOR CONTINUATION CALLS ONLY (ISTATE = 2 OR 3)
C AND IS TO CHECK STOP CONDITIONS BEFORE TAKING A STEP.
C-----------------------------------------------------------------------
 200  NSLAST = NST
      GO TO (210, 250, 220, 230, 240), ITASK
 210  IF ((TN - TOUT)*H .LT. 0.0D0) GO TO 250
      CALL INTDY (TOUT, 0, RWORK(LYH), NYH, Y, IFLAG)
      IF (IFLAG .NE. 0) GO TO 627
      T = TOUT
      GO TO 420
 220  TP = TN - HU*(1.0D0 + 100.0D0*UROUND)
      IF ((TP - TOUT)*H .GT. 0.0D0) GO TO 623
      IF ((TN - TOUT)*H .LT. 0.0D0) GO TO 250
      GO TO 400
 230  TCRIT = RWORK(1)
      IF ((TN - TCRIT)*H .GT. 0.0D0) GO TO 624
      IF ((TCRIT - TOUT)*H .LT. 0.0D0) GO TO 625
      IF ((TN - TOUT)*H .LT. 0.0D0) GO TO 245
      CALL INTDY (TOUT, 0, RWORK(LYH), NYH, Y, IFLAG)
      IF (IFLAG .NE. 0) GO TO 627
      T = TOUT
      GO TO 420
 240  TCRIT = RWORK(1)
      IF ((TN - TCRIT)*H .GT. 0.0D0) GO TO 624
 245  HMX = DABS(TN) + DABS(H)
      IHIT = DABS(TN - TCRIT) .LE. 100.0D0*UROUND*HMX
      IF (IHIT) GO TO 400
      TNEXT = TN + H*(1.0D0 + 4.0D0*UROUND)
      IF ((TNEXT - TCRIT)*H .LE. 0.0D0) GO TO 250 
      H = (TCRIT - TN)*(1.0D0 - 4.0D0*UROUND)
      IF (ISTATE .EQ. 2) JSTART = -2
C-----------------------------------------------------------------------
C BLOCK E.
C THE NEXT BLOCK IS NORMALLY EXECUTED FOR ALL CALLS AND CONTAINS
C THE CALL TO THE ONE-STEP CORE INTEGRATOR STODE. 
C
C THIS IS A LOOPING POINT FOR THE INTEGRATION STEPS.
C
C FIRST CHECK FOR TOO MANY STEPS BEING TAKEN, UPDATE EWT (IF NOT AT
C START OF PROBLEM), CHECK FOR TOO MUCH ACCURACY BEING REQUESTED, AND 
C CHECK FOR H BELOW THE ROUNDOFF LEVEL IN T.
C-----------------------------------------------------------------------
 250  CONTINUE
      IF ((NST-NSLAST) .GE. MXSTEP) GO TO 500
      CALL EWSET (N, ITOL, RTOL, ATOL, RWORK(LYH), RWORK(LEWT))
      DO 260 I = 1,N
        IF (RWORK(I+LEWT-1) .LE. 0.0D0) GO TO 510 
 260    RWORK(I+LEWT-1) = 1.0D0/RWORK(I+LEWT-1)
 270  TOLSF = UROUND*VNORM (N, RWORK(LYH), RWORK(LEWT))
      IF (TOLSF .LE. 1.0D0) GO TO 280
      TOLSF = TOLSF*2.0D0
      IF (NST .EQ. 0) GO TO 626
      GO TO 520
 280  IF ((TN + H) .NE. TN) GO TO 290
      NHNIL = NHNIL + 1
      IF (NHNIL .GT. MXHNIL) GO TO 290
      CALL XERRWD('LSODE--  WARNING..INTERNAL T (=R1) AND H (=R2) ARE',
     1   50, 101, 0, 0, 0, 0, 0, 0.0D0, 0.0D0)
      CALL XERRWD(
     1  '      SUCH THAT IN THE MACHINE, T + H = T ON THE NEXT STEP  ',
     1   60, 101, 0, 0, 0, 0, 0, 0.0D0, 0.0D0)
      CALL XERRWD('      (H = STEP SIZE). SOLVER WILL CONTINUE ANYWAY',
     1   50, 101, 0, 0, 0, 0, 2, TN, H) 
      IF (NHNIL .LT. MXHNIL) GO TO 290
      CALL XERRWD('LSODE--  ABOVE WARNING HAS BEEN ISSUED I1 TIMES.  ',
     1   50, 102, 0, 0, 0, 0, 0, 0.0D0, 0.0D0)
      CALL XERRWD('      IT WILL NOT BE ISSUED AGAIN FOR THIS PROBLEM',
     1   50, 102, 0, 1, MXHNIL, 0, 0, 0.0D0, 0.0D0)
 290  CONTINUE
C-----------------------------------------------------------------------
C     CALL STODE(NEQ,Y,YH,NYH,YH,EWT,SAVF,ACOR,WM,IWM,F,JAC,PREPJ,SOLSY)
C-----------------------------------------------------------------------
      IERR = 0
      CALL STODE (NEQ, Y, RWORK(LYH), NYH, RWORK(LYH), RWORK(LEWT),
     1   RWORK(LSAVF), RWORK(LACOR), RWORK(LWM), IWORK(LIWM),
     2   F, JAC, PREPJ, SOLSY, IERR)
      IF (IERR .LT. 0) THEN
        ISTATE = -13
        RETURN
      ENDIF
      KGO = 1 - KFLAG
      GO TO (300, 530, 540), KGO
C-----------------------------------------------------------------------
C BLOCK F.
C THE FOLLOWING BLOCK HANDLES THE CASE OF A SUCCESSFUL RETURN FROM THE
C CORE INTEGRATOR (KFLAG = 0).  TEST FOR STOP CONDITIONS.
C-----------------------------------------------------------------------
 300  INIT = 1
      GO TO (310, 400, 330, 340, 350), ITASK
C ITASK = 1.  IF TOUT HAS BEEN REACHED, INTERPOLATE. -------------------
 310  IF ((TN - TOUT)*H .LT. 0.0D0) GO TO 250
      CALL INTDY (TOUT, 0, RWORK(LYH), NYH, Y, IFLAG)
      T = TOUT
      GO TO 420
C ITASK = 3.  JUMP TO EXIT IF TOUT WAS REACHED. ------------------------
 330  IF ((TN - TOUT)*H .GE. 0.0D0) GO TO 400
      GO TO 250
C ITASK = 4.  SEE IF TOUT OR TCRIT WAS REACHED.  ADJUST H IF NECESSARY.
 340  IF ((TN - TOUT)*H .LT. 0.0D0) GO TO 345
      CALL INTDY (TOUT, 0, RWORK(LYH), NYH, Y, IFLAG)
      T = TOUT
      GO TO 420
 345  HMX = DABS(TN) + DABS(H)
      IHIT = DABS(TN - TCRIT) .LE. 100.0D0*UROUND*HMX
      IF (IHIT) GO TO 400
      TNEXT = TN + H*(1.0D0 + 4.0D0*UROUND)
      IF ((TNEXT - TCRIT)*H .LE. 0.0D0) GO TO 250 
      H = (TCRIT - TN)*(1.0D0 - 4.0D0*UROUND)
      JSTART = -2
      GO TO 250
C ITASK = 5.  SEE IF TCRIT WAS REACHED AND JUMP TO EXIT. ---------------
 350  HMX = DABS(TN) + DABS(H)
      IHIT = DABS(TN - TCRIT) .LE. 100.0D0*UROUND*HMX
C-----------------------------------------------------------------------
C BLOCK G.
C THE FOLLOWING BLOCK HANDLES ALL SUCCESSFUL RETURNS FROM LSODE.
C IF ITASK .NE. 1, Y IS LOADED FROM YH AND T IS SET ACCORDINGLY.
C ISTATE IS SET TO 2, THE ILLEGAL INPUT COUNTER IS ZEROED, AND THE
C OPTIONAL OUTPUTS ARE LOADED INTO THE WORK ARRAYS BEFORE RETURNING.
C IF ISTATE = 1 AND TOUT = T, THERE IS A RETURN WITH NO ACTION TAKEN, 
C EXCEPT THAT IF THIS HAS HAPPENED REPEATEDLY, THE RUN IS TERMINATED. 
C-----------------------------------------------------------------------
 400  DO 410 I = 1,N
 410    Y(I) = RWORK(I+LYH-1) 
      T = TN
      IF (ITASK .NE. 4 .AND. ITASK .NE. 5) GO TO 420
      IF (IHIT) T = TCRIT
 420  ISTATE = 2
      ILLIN = 0
      RWORK(11) = HU
      RWORK(12) = H 
      RWORK(13) = TN
      IWORK(11) = NST
      IWORK(12) = NFE
      IWORK(13) = NJE
      IWORK(14) = NQU
      IWORK(15) = NQ
      RETURN
C
 430  NTREP = NTREP + 1
      IF (NTREP .LT. 5) RETURN
      CALL XERRWD(
     1  'LSODE--  REPEATED CALLS WITH ISTATE = 1 AND TOUT = T (=R1)  ',
     1   60, 301, 0, 0, 0, 0, 1, T, 0.0D0)
      GO TO 800
C-----------------------------------------------------------------------
C BLOCK H.
C THE FOLLOWING BLOCK HANDLES ALL UNSUCCESSFUL RETURNS OTHER THAN
C THOSE FOR ILLEGAL INPUT.  FIRST THE ERROR MESSAGE ROUTINE IS CALLED.
C IF THERE WAS AN ERROR TEST OR CONVERGENCE TEST FAILURE, IMXER IS SET.
C THEN Y IS LOADED FROM YH, T IS SET TO TN, AND THE ILLEGAL INPUT
C COUNTER ILLIN IS SET TO 0.  THE OPTIONAL OUTPUTS ARE LOADED INTO
C THE WORK ARRAYS BEFORE RETURNING.
C-----------------------------------------------------------------------
C THE MAXIMUM NUMBER OF STEPS WAS TAKEN BEFORE REACHING TOUT. ----------
 500  CALL XERRWD('LSODE--  AT CURRENT T (=R1), MXSTEP (=I1) STEPS   ',
     1   50, 201, 0, 0, 0, 0, 0, 0.0D0, 0.0D0)
      CALL XERRWD('      TAKEN ON THIS CALL BEFORE REACHING TOUT     ',
     1   50, 201, 0, 1, MXSTEP, 0, 1, TN, 0.0D0)
      ISTATE = -1
      GO TO 580
C EWT(I) .LE. 0.0 FOR SOME I (NOT AT START OF PROBLEM). ----------------
 510  EWTI = RWORK(LEWT+I-1)
      CALL XERRWD('LSODE--  AT T (=R1), EWT(I1) HAS BECOME R2 .LE. 0.',
     1   50, 202, 0, 1, I, 0, 2, TN, EWTI)
      ISTATE = -6
      GO TO 580
C TOO MUCH ACCURACY REQUESTED FOR MACHINE PRECISION. -------------------
 520  CALL XERRWD('LSODE--  AT T (=R1), TOO MUCH ACCURACY REQUESTED  ',
     1   50, 203, 0, 0, 0, 0, 0, 0.0D0, 0.0D0)
      CALL XERRWD('      FOR PRECISION OF MACHINE..  SEE TOLSF (=R2) ',
     1   50, 203, 0, 0, 0, 0, 2, TN, TOLSF)
      RWORK(14) = TOLSF
      ISTATE = -2
      GO TO 580
C KFLAG = -1.  ERROR TEST FAILED REPEATEDLY OR WITH ABS(H) = HMIN. -----
 530  CALL XERRWD('LSODE--  AT T(=R1) AND STEP SIZE H(=R2), THE ERROR',
     1   50, 204, 0, 0, 0, 0, 0, 0.0D0, 0.0D0)
      CALL XERRWD('      TEST FAILED REPEATEDLY OR WITH ABS(H) = HMIN',
     1   50, 204, 0, 0, 0, 0, 2, TN, H) 
      ISTATE = -4
      GO TO 560
C KFLAG = -2.  CONVERGENCE FAILED REPEATEDLY OR WITH ABS(H) = HMIN. ----
 540  CALL XERRWD('LSODE--  AT T (=R1) AND STEP SIZE H (=R2), THE    ',
     1   50, 205, 0, 0, 0, 0, 0, 0.0D0, 0.0D0)
      CALL XERRWD('      CORRECTOR CONVERGENCE FAILED REPEATEDLY     ',
     1   50, 205, 0, 0, 0, 0, 0, 0.0D0, 0.0D0)
      CALL XERRWD('      OR WITH ABS(H) = HMIN   ',
     1   30, 205, 0, 0, 0, 0, 2, TN, H) 
      ISTATE = -5
C COMPUTE IMXER IF RELEVANT. -------------------------------------------
 560  BIG = 0.0D0
      IMXER = 1
      DO 570 I = 1,N
        SIZE = DABS(RWORK(I+LACOR-1)*RWORK(I+LEWT-1))
        IF (BIG .GE. SIZE) GO TO 570
        BIG = SIZE
        IMXER = I
 570    CONTINUE
      IWORK(16) = IMXER
C SET Y VECTOR, T, ILLIN, AND OPTIONAL OUTPUTS. ------------------------
 580  DO 590 I = 1,N
 590    Y(I) = RWORK(I+LYH-1) 
      T = TN
      ILLIN = 0
      RWORK(11) = HU
      RWORK(12) = H 
      RWORK(13) = TN
      IWORK(11) = NST
      IWORK(12) = NFE
      IWORK(13) = NJE
      IWORK(14) = NQU
      IWORK(15) = NQ
      RETURN
C-----------------------------------------------------------------------
C BLOCK I.
C THE FOLLOWING BLOCK HANDLES ALL ERROR RETURNS DUE TO ILLEGAL INPUT
C (ISTATE = -3), AS DETECTED BEFORE CALLING THE CORE INTEGRATOR.
C FIRST THE ERROR MESSAGE ROUTINE IS CALLED.  THEN IF THERE HAVE BEEN 
C 5 CONSECUTIVE SUCH RETURNS JUST BEFORE THIS CALL TO THE SOLVER,
C THE RUN IS HALTED.
C-----------------------------------------------------------------------
 601  CALL XERRWD('LSODE--  ISTATE (=I1) ILLEGAL ',
     1   30, 1, 0, 1, ISTATE, 0, 0, 0.0D0, 0.0D0) 
      GO TO 700
 602  CALL XERRWD('LSODE--  ITASK (=I1) ILLEGAL  ',
     1   30, 2, 0, 1, ITASK, 0, 0, 0.0D0, 0.0D0)
      GO TO 700
 603  CALL XERRWD('LSODE--  ISTATE .GT. 1 BUT LSODE NOT INITIALIZED  ',
     1   50, 3, 0, 0, 0, 0, 0, 0.0D0, 0.0D0)
      GO TO 700
 604  CALL XERRWD('LSODE--  NEQ (=I1) .LT. 1     ',
     1   30, 4, 0, 1, NEQ(1), 0, 0, 0.0D0, 0.0D0) 
      GO TO 700
 605  CALL XERRWD('LSODE--  ISTATE = 3 AND NEQ INCREASED (I1 TO I2)  ',
     1   50, 5, 0, 2, N, NEQ(1), 0, 0.0D0, 0.0D0) 
      GO TO 700
 606  CALL XERRWD('LSODE--  ITOL (=I1) ILLEGAL   ',
     1   30, 6, 0, 1, ITOL, 0, 0, 0.0D0, 0.0D0)
      GO TO 700
 607  CALL XERRWD('LSODE--  IOPT (=I1) ILLEGAL   ',
     1   30, 7, 0, 1, IOPT, 0, 0, 0.0D0, 0.0D0)
      GO TO 700
 608  CALL XERRWD('LSODE--  MF (=I1) ILLEGAL     ',
     1   30, 8, 0, 1, MF, 0, 0, 0.0D0, 0.0D0)
      GO TO 700
 609  CALL XERRWD('LSODE--  ML (=I1) ILLEGAL.. .LT.0 OR .GE.NEQ (=I2)',
     1   50, 9, 0, 2, ML, NEQ(1), 0, 0.0D0, 0.0D0)
      GO TO 700
 610  CALL XERRWD('LSODE--  MU (=I1) ILLEGAL.. .LT.0 OR .GE.NEQ (=I2)',
     1   50, 10, 0, 2, MU, NEQ(1), 0, 0.0D0, 0.0D0)
      GO TO 700
 611  CALL XERRWD('LSODE--  MAXORD (=I1) .LT. 0  ',
     1   30, 11, 0, 1, MAXORD, 0, 0, 0.0D0, 0.0D0)
      GO TO 700
 612  CALL XERRWD('LSODE--  MXSTEP (=I1) .LT. 0  ',
     1   30, 12, 0, 1, MXSTEP, 0, 0, 0.0D0, 0.0D0)
      GO TO 700
 613  CALL XERRWD('LSODE--  MXHNIL (=I1) .LT. 0  ',
     1   30, 13, 0, 1, MXHNIL, 0, 0, 0.0D0, 0.0D0)
      GO TO 700
 614  CALL XERRWD('LSODE--  TOUT (=R1) BEHIND T (=R2)      ',
     1   40, 14, 0, 0, 0, 0, 2, TOUT, T)
      CALL XERRWD('      INTEGRATION DIRECTION IS GIVEN BY H0 (=R1)  ',
     1   50, 14, 0, 0, 0, 0, 1, H0, 0.0D0)
      GO TO 700
 615  CALL XERRWD('LSODE--  HMAX (=R1) .LT. 0.0  ',
     1   30, 15, 0, 0, 0, 0, 1, HMAX, 0.0D0)
      GO TO 700
 616  CALL XERRWD('LSODE--  HMIN (=R1) .LT. 0.0  ',
     1   30, 16, 0, 0, 0, 0, 1, HMIN, 0.0D0)
      GO TO 700
 617  CALL XERRWD(
     1  'LSODE--  RWORK LENGTH NEEDED, LENRW (=I1), EXCEEDS LRW (=I2)',
     1   60, 17, 0, 2, LENRW, LRW, 0, 0.0D0, 0.0D0)
      GO TO 700
 618  CALL XERRWD(
     1  'LSODE--  IWORK LENGTH NEEDED, LENIW (=I1), EXCEEDS LIW (=I2)',
     1   60, 18, 0, 2, LENIW, LIW, 0, 0.0D0, 0.0D0)
      GO TO 700
 619  CALL XERRWD('LSODE--  RTOL(I1) IS R1 .LT. 0.0        ',
     1   40, 19, 0, 1, I, 0, 1, RTOLI, 0.0D0)
      GO TO 700
 620  CALL XERRWD('LSODE--  ATOL(I1) IS R1 .LT. 0.0        ',
     1   40, 20, 0, 1, I, 0, 1, ATOLI, 0.0D0)
      GO TO 700
 621  EWTI = RWORK(LEWT+I-1)
      CALL XERRWD('LSODE--  EWT(I1) IS R1 .LE. 0.0         ',
     1   40, 21, 0, 1, I, 0, 1, EWTI, 0.0D0)
      GO TO 700
 622  CALL XERRWD(
     1  'LSODE--  TOUT (=R1) TOO CLOSE TO T(=R2) TO START INTEGRATION',
     1   60, 22, 0, 0, 0, 0, 2, TOUT, T)
      GO TO 700
 623  CALL XERRWD(
     1  'LSODE--  ITASK = I1 AND TOUT (=R1) BEHIND TCUR - HU (= R2)  ',
     1   60, 23, 0, 1, ITASK, 0, 2, TOUT, TP)
      GO TO 700
 624  CALL XERRWD(
     1  'LSODE--  ITASK = 4 OR 5 AND TCRIT (=R1) BEHIND TCUR (=R2)   ',
     1   60, 24, 0, 0, 0, 0, 2, TCRIT, TN)
      GO TO 700
 625  CALL XERRWD(
     1  'LSODE--  ITASK = 4 OR 5 AND TCRIT (=R1) BEHIND TOUT (=R2)   ',
     1   60, 25, 0, 0, 0, 0, 2, TCRIT, TOUT)
      GO TO 700
 626  CALL XERRWD('LSODE--  AT START OF PROBLEM, TOO MUCH ACCURACY   ',
     1   50, 26, 0, 0, 0, 0, 0, 0.0D0, 0.0D0)
      CALL XERRWD(
     1  '      REQUESTED FOR PRECISION OF MACHINE..  SEE TOLSF (=R1) ',
     1   60, 26, 0, 0, 0, 0, 1, TOLSF, 0.0D0)
      RWORK(14) = TOLSF
      GO TO 700
 627  CALL XERRWD('LSODE--  TROUBLE FROM INTDY. ITASK = I1, TOUT = R1',
     1   50, 27, 0, 1, ITASK, 0, 1, TOUT, 0.0D0)
C
 700  IF (ILLIN .EQ. 5) GO TO 710
      ILLIN = ILLIN + 1
      ISTATE = -3
      RETURN
 710  CALL XERRWD('LSODE--  REPEATED OCCURRENCES OF ILLEGAL INPUT    ',
     1   50, 302, 0, 0, 0, 0, 0, 0.0D0, 0.0D0)
C
 800  CALL XERRWD('LSODE--  RUN ABORTED.. APPARENT INFINITE LOOP     ',
     1   50, 303, 2, 0, 0, 0, 0, 0.0D0, 0.0D0)
      RETURN
C----------------------- END OF SUBROUTINE LSODE -----------------------
      END 
