#!/usr/bin/env python
#
# Copyright 2005 Free Software Foundation, Inc.
# 
# This file is part of GNU Radio
# 
# GNU Radio is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2, or (at your option)
# any later version.
# 
# GNU Radio is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with GNU Radio; see the file COPYING.  If not, write to
# the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.
# 

import math
import sys
from optparse import OptionParser

from gnuradio import gr, eng_notation
from gnuradio import usrp
from gnuradio import audio
from gnuradio import blks
from gnuradio.eng_option import eng_option

from gnuradio.wxgui import stdgui, fftsink
import wx


class ptt_graph(stdgui.gui_flow_graph):
    def __init__(self, frame, panel, vbox, argv):
        stdgui.gui_flow_graph.__init__ (self, frame, panel, vbox, argv)

        self.frame = frame
        self.space_bar_pressed = False
        
        parser = OptionParser (option_class=eng_option)
        parser.add_option ("-f", "--freq", type="eng_float", default=29.325e6,
                           help="set Tx and Rx frequency to FREQ", metavar="FREQ")
        parser.add_option ("-N", "--no-gui", action="store_true", default=False)
        (options, args) = parser.parse_args ()
        if len(args) != 0:
            parser.print_help()
            sys.exit(1)

        if options.freq < 1e6:
            options.freq *= 1e6
            
        self.txpath = transmit_path(self, options.freq)
        self.rxpath = receive_path(self, options.freq)
        self.build_gui(frame, panel, vbox, argv, options.no_gui)
        self.set_transmit(False)

    def set_transmit(self, enabled):
        self.txpath.set_enable(enabled)
        self.rxpath.set_enable(not(enabled))
        if enabled:
            self.frame.SetStatusText ("Transmitter ON")
        else:
            self.frame.SetStatusText ("Receiver ON")

    def build_gui(self, frame, panel, vbox, argv, no_gui):
        hbox = wx.BoxSizer(wx.HORIZONTAL)
        hbox.Add((10,0), 1.0)            # stretchy space
        self.status_msg = wx.StaticText(panel, -1, "Press Space Bar to Transmit")
        of = self.status_msg.GetFont()
        self.status_msg.SetFont(wx.Font(15, of.GetFamily(), of.GetStyle(), of.GetWeight()))
        hbox.Add(self.status_msg, 0, wx.ALIGN_CENTER | wx.EXPAND)
        hbox.Add((10,0), 1.0)            # stretchy space
        
        vbox.Add(hbox, 0, wx.EXPAND | wx.ALIGN_CENTER)

        panel.Bind(wx.EVT_KEY_DOWN, self._on_key_down)
        panel.Bind(wx.EVT_KEY_UP, self._on_key_up)
        panel.Bind(wx.EVT_KILL_FOCUS, self._on_kill_focus)
        panel.SetFocus()
        
        if 1 and not(no_gui):
            rx_fft, win1 = \
                    fftsink.make_fft_sink_c (self, panel, "Rx Input", 512,
                                             self.rxpath.IF_rate, 0, 100)
            self.connect (self.rxpath.u, rx_fft)
            vbox.Add (win1, 1, wx.EXPAND)

        if 1 and not(no_gui):
            tx_fft, win1 = \
                    fftsink.make_fft_sink_c (self, panel, "Tx Output", 512,
                                             self.txpath.usrp_rate, 20, 140)
            self.connect (self.txpath.amp, tx_fft)
            vbox.Add (win1, 1, wx.EXPAND)

    def _on_key_down(self, evt):
        # print "key_down:", evt.m_keyCode
        if evt.m_keyCode == wx.WXK_SPACE and not(self.space_bar_pressed):
            self.space_bar_pressed = True
            self.set_transmit(True)

    def _on_key_up(self, evt):
        # print "key_up", evt.m_keyCode
        if evt.m_keyCode == wx.WXK_SPACE:
            self.space_bar_pressed = False
            self.set_transmit(False)

    def _on_kill_focus(self, evt):
        # if we lose the keyboard focus, turn off the transmitter
        self.space_bar_pressed = False
        self.set_transmit(False)
        

class transmit_path(gr.hier_block):
    def __init__(self, fg, RF_freq):
        dac_rate = 128e6
        self.usrp_interp = 400
        self.usrp_rate = dac_rate // self.usrp_interp        # 320 kS/s
        self.sw_interp = 10
        self.audio_rate = self.usrp_rate // self.sw_interp   #  32 kS/s

        self.normal_gain = 4000

        RF_freq = fold_freq(RF_freq, dac_rate)
        
        self.audio = audio.source(int(self.audio_rate))
        self.fmtx = blks.nbfm_tx(fg, self.audio_rate, self.usrp_rate)
        self.amp = gr.multiply_const_cc (self.normal_gain)

        self.u = usrp.sink_c (0, self.usrp_interp)
        self.u.set_tx_freq(0, RF_freq)

        fg.connect(self.audio, self.fmtx, self.amp, self.u)
        gr.hier_block.__init__(self, fg, None, None)

    def set_enable(self, enable):
        if enable:
            self.amp.set_k (self.normal_gain)
        else:
            self.amp.set_k (0)


class receive_path(gr.hier_block):
    def __init__(self, fg, RF_freq):
        adc_rate = 64e6
        IF_freq = 40e3

        RF_freq = fold_freq (RF_freq, adc_rate)
        
        usrp_decim = 125
        IF_rate = adc_rate // usrp_decim 	# 512 kS/s
        self.IF_rate = IF_rate
        IF_decim = 8
        quad_rate = IF_rate // IF_decim         #  64 kS/s
        audio_decim = 2
        audio_rate = quad_rate // audio_decim   #  32 kS/s

        u = usrp.source_c (0, usrp_decim)
        self.u = u
        u.set_rx_freq (0, IF_freq - RF_freq)
        u.set_pga (0, 20)
        u.set_pga (1, 20)

        IF_freq = RF_freq + u.rx_freq(0)     # compute actual IF freq

        # Create filter to get actual channel we want
        chan_coeffs = gr.firdes.low_pass (1.0,                # gain
                                          IF_rate,            # sampling rate
                                          11e3,               # low pass cutoff freq
                                          4e3,                # width of trans. band
                                          gr.firdes.WIN_HANN) # filter type 

        print len(chan_coeffs)
        # Decimating Channel filter with frequency translation
        # complex in and out, float taps
        ddc = gr.freq_xlating_fir_filter_ccf(IF_decim,       # decimation rate
                                             chan_coeffs,    # taps
                                             -IF_freq,       # frequency translation amount
                                             IF_rate)        # input sample rate

        # instantiate the guts of the single channel receiver
        self.fmrx = blks.nbfm_rx(fg, audio_rate, quad_rate)

        self.set_squelch(15)
        
        # use mute block to disable Rx while transmitting
        self.m = gr.mute_ff(False)

        # sound card as final sink
        audio_sink = audio.sink (int(audio_rate))
        
        # now wire it all together
        fg.connect (u, ddc, self.fmrx, self.m, audio_sink)
        gr.hier_block.__init__(self, fg, u, audio_sink)

    def set_enable(self, enable):
        self.m.set_mute(not (enable))

    def set_squelch(self, threshold_in_db):
        self.fmrx.squelch.set_threshold(threshold_in_db)


def fold_freq(freq, fs):
    while freq > fs:
        freq = freq - fs
    if freq > fs/2:
        freq = fs - freq
    return freq

def main():
    app = stdgui.stdapp(ptt_graph, "NBFM Push to Talk")
    app.MainLoop()

if __name__ == '__main__':
    main()
