/*
 * Electric(tm) VLSI Design System
 *
 * File: netflat.c
 * Network tool: module for fully instantiating a hierarchical network
 * Written by: Steven M. Rubin, Static Free Software
 *
 * Copyright (c) 2000 Static Free Software.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Static Free Software
 * 4119 Alpine Road
 * Portola Valley, California 94028
 * info@staticfreesoft.com
 */

#include "global.h"
#include "network.h"
#include "efunction.h"
#include "tecschem.h"

static INTBIG    net_pseudonode;		/* net number for pseudonetworks */
static NODEINST *net_toplevelinst;		/* actual top-instance associated with this node */
static PCOMP   **net_tempconlist;		/* temp list of connections */
static INTSML   *net_tempconlistindex;	/* temp list of indices */
static INTSML   *net_tempportindices;	/* temp list of normalized indices */
static INTBIG    net_tempcontotal = 0;	/* size of temp list of connections */

/* prototypes for local routines */
PCOMP *net_allocpcomp(void);
PNET  *net_allocpnet(void);
PCOMP *net_buildpseudo(NODEPROTO*, PCOMP*, PNET**, PNET**, INTBIG*, PNET**,
					   INTSML, INTSML, INTSML, INTSML);
void   net_freepcomp(PCOMP*);
void   net_freepnet(PNET*);
float  net_getpartvalue(NODEINST *ni);
void   net_mergeparallel(PCOMP **pcomp, INTBIG *components);
PNET  *net_newpnet(PNET**);
void   net_removepcompfromlists(PCOMP *pcomp, PCOMP *removedpc, PCOMP *newpc);

/*
 * Routine to free all memory associated with this module.
 */
void net_freeflatmemory(void)
{
	REGISTER PCOMP *p;
	REGISTER PNET *pn;

	while (net_pcompfree != NOPCOMP)
	{
		p = net_pcompfree;
		net_pcompfree = net_pcompfree->nextpcomp;
		efree((char *)p);
	}
	while (net_pnetfree != NOPNET)
	{
		pn = net_pnetfree;
		net_pnetfree = net_pnetfree->nextpnet;
		efree((char *)pn);
	}
	if (net_tempcontotal > 0)
	{
		efree((char *)net_tempconlist);
		efree((char *)net_tempconlistindex);
		efree((char *)net_tempportindices);
	}
}

/*********************** PSEUDO-NETWORK CONSTRUCTION ***********************/

/*
 * The usage of this module is:
 *   #include "network.h"
 *   PCOMP *pcomp;
 *   PNET *pnet;
 *   INTBIG components, nets, powernets, groundnets;
 *   pcomp = net_makepseudo(facet, &components, &nets, &powernets, &groundnets,
 *	    &pnet, hierarchical, useequiv, ignorepwrgnd, mergeparallel, checkfacetoverrides);
 *   .....
 *        do something with the network in "pcomp" and "pnet"
 *   .....
 *   net_freeallpnet(pnet);
 *   net_freeallpcomp(pcomp);
 *
 * "net_makepseudo" builds a pseudonetwork structure that represents the
 * network in facet "facet".  If it returns NOPCOMP, there is an error.
 * A linked list of PCOMP objects is returned, one for every component
 * in the pseudonetwork.  A linked list of PNET objects is also returned,
 * one for every network in the pseudonetwork.  Finally, the number of
 * components, networks, power networks, and ground networks is returned
 * in the reference parameters "components", "nets", "powernets", and
 * "groundnets".
 *
 * A number of switches controls the flattening:
 * If "hierarchical"        is nonzero, the network will be fully instantiated.
 * If "useequiv"            is nonzero, user-specified equivalences are considered.
 * If "ignorepwrgnd"        is nonzero, power and ground nets are ignored.
 * If "mergeparallel"       is nonzero, parallel components are merged into one.
 * If "checkfacetoverrides" is nonzero, individual facets may override
 *                          "ignorepwrgnd", and "mergeparallel".
 */
PCOMP *net_makepseudo(NODEPROTO *facet, INTBIG *components, INTBIG *nets, INTBIG *powernets,
	INTBIG *groundnets, PNET **pnetlist, INTSML hierarchical, INTSML useequivs,
	INTSML ignorepwrgnd, INTSML mergeparallel, INTSML checkfacetoverrides)
{
	REGISTER PCOMP *pcomp, *p, **newconlist, *thepcomp;
	PCOMP *pcomplist;
	REGISTER INTSML list, i, j, k, localignorepwrgnd, theindex,
		*newconlistindex, *newportindices, localmergeparallel;
	REGISTER INTBIG tempconcount, newtotal, counter;
	PNET *power, *ground;
	REGISTER PNET *net;
	REGISTER VARIABLE *var;
	REGISTER PORTPROTO *pp, *opp;

	/* see if the current facet overrides the options */
	localignorepwrgnd = ignorepwrgnd;
	localmergeparallel = mergeparallel;
	if (checkfacetoverrides != 0)
	{
		var = getvalkey((INTBIG)facet, VNODEPROTO, VINTEGER, net_ncc_optionskey);
		if (var != NOVARIABLE)
		{
			if ((var->addr&NCCIGNOREPWRGNDOVER) != 0)
			{
				if ((var->addr&NCCIGNOREPWRGND) != 0) localignorepwrgnd = 1; else
					localignorepwrgnd = 0;
			}
			if ((var->addr&NCCNOMERGEPARALLELOVER) != 0)
			{
				if ((var->addr&NCCNOMERGEPARALLEL) == 0) localmergeparallel = 1; else
					localmergeparallel = 0;
			}
		}
	}

	/* first create net numbers external to this facet */
	net_pseudonode = 0;
	*pnetlist = NOPNET;
	power = ground = NOPNET;
	for(pp = facet->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
		pp->temp1 = (INTBIG)NOPNET;
	for(pp = facet->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
	{
		if (pp->temp1 != (INTBIG)NOPNET) continue;
		if ((pp->userbits&STATEBITS) == PWRPORT)
		{
			if (localignorepwrgnd != 0) net = net_allocpnet(); else
				net = net_newpnet(pnetlist);
			if (net == NOPNET) return(NOPCOMP);
			net->flags = POWERNET;
			power = net;
		} else if ((pp->userbits&STATEBITS) == GNDPORT)
		{
			if (localignorepwrgnd != 0) net = net_allocpnet(); else
				net = net_newpnet(pnetlist);
			if (net == NOPNET) return(NOPCOMP);
			net->flags = GROUNDNET;
			ground = net;
		} else
		{
			net = net_newpnet(pnetlist);
			if (net == NOPNET) return(NOPCOMP);
		}
		net->flags |= EXPORTEDNET;
		net->network = pp->network;
		net->realport = pp;
		pp->temp1 = (INTBIG)net;
		for(opp = pp->nextportproto; opp != NOPORTPROTO; opp = opp->nextportproto)
			if (pp->network == opp->network) opp->temp1 = pp->temp1;
	}

	/* create a list of pseudocomponents in this facet */
	*components = 0;
	begintraversehierarchy();
	net_toplevelinst = NONODEINST;
	pcomplist = net_buildpseudo(facet, NOPCOMP, &power, &ground, components,
		pnetlist, hierarchical, useequivs, ignorepwrgnd, checkfacetoverrides);
	if (pcomplist == 0) return(NOPCOMP);
	if (*components == 0)
	{
		ttyputmsg(_("There are no components in facet %s"), describenodeproto(facet));
		return(NOPCOMP);
	}

	/* now run through each network, accumulating pointers */
	counter = 0;
	for(pcomp = pcomplist; pcomp != NOPCOMP; pcomp = pcomp->nextpcomp)
	{
		counter++;
		if ((counter%5) == 0)
		{
			if (stopping(STOPREASONNCC))
			{
				net_freeallpcomp(pcomplist);
				net_freeallpnet(*pnetlist);
				*pnetlist = NOPNET;
				return(NOPCOMP);
			}
		}
		for(list=0; list<pcomp->wirecount; list++)
		{
			if (pcomp->count[list] > 0) continue;
			net = pcomp->netnumbers[list];

			/* accumulate all other nodes that have this connection */
			tempconcount = 0;
			for(p = pcomplist; p != NOPCOMP; p = p->nextpcomp)
			{
				for(i=0; i<p->wirecount; i++)
				{
					if (p->netnumbers[i] != net) continue;

					/* create a pseudo-net object for this */
					if (tempconcount >= net_tempcontotal)
					{
						newtotal = net_tempcontotal * 2;
						if (newtotal == 0) newtotal = 5;
						newconlist = (PCOMP **)emalloc(newtotal * (sizeof (PCOMP *)), net_aid->cluster);
						if (newconlist == 0) return(NOPCOMP);
						newconlistindex = (INTSML *)emalloc(newtotal * SIZEOFINTSML, net_aid->cluster);
						if (newconlistindex == 0) return(NOPCOMP);
						newportindices = (INTSML *)emalloc(newtotal * SIZEOFINTSML, net_aid->cluster);
						if (newportindices == 0) return(NOPCOMP);
						for(j=0; j<tempconcount; j++)
						{
							newconlist[j] = net_tempconlist[j];
							newconlistindex[j] = net_tempconlistindex[j];
							newportindices[j] = net_tempportindices[j];
						}
						if (net_tempcontotal > 0)
						{
							efree((char *)net_tempconlist);
							efree((char *)net_tempconlistindex);
							efree((char *)net_tempportindices);
						}
						net_tempconlist = newconlist;
						net_tempconlistindex = newconlistindex;
						net_tempportindices = newportindices;
						net_tempcontotal = newtotal;
					}
					net_tempconlist[tempconcount] = p;
					net_tempconlistindex[tempconcount] = i;
					net_tempportindices[tempconcount] = p->portindices[i];
					tempconcount++;
				}
			}
			if (tempconcount > 1)
			{
				for(j=0; j<tempconcount; j++)
				{
					thepcomp = net_tempconlist[j];
					theindex = net_tempconlistindex[j];
					thepcomp->pconnarray[theindex] = (PCOMP **)emalloc((tempconcount-1) * (sizeof (PCOMP *)),
						net_aid->cluster);
					if (thepcomp->pconnarray[theindex] == 0) return(NOPCOMP);

					thepcomp->portindexarray[theindex] = (INTSML *)emalloc((tempconcount-1) * SIZEOFINTSML,
						net_aid->cluster);
					if (thepcomp->portindexarray[theindex] == 0) return(NOPCOMP);
					k = 0;
					for(i=0; i<tempconcount; i++)
					{
						if (net_tempconlist[i] == thepcomp && net_tempconlistindex[i] == theindex)
							continue;
						thepcomp->pconnarray[theindex][k] = net_tempconlist[i];
						thepcomp->portindexarray[theindex][k] = net_tempportindices[i];
						k++;
					}
					thepcomp->count[theindex] = (INTSML)(tempconcount-1);
				}
			}
		}
	}

	/* reduce network by merging parallel components */
	if (localmergeparallel != 0)
		net_mergeparallel(&pcomplist, components);

	/* report the total number of nets */
	*nets = net_pseudonode;

	/* count the power and ground nets */
	*powernets = *groundnets = 0;
	for(net = *pnetlist; net != NOPNET; net = net->nextpnet)
	{
		if ((net->flags&POWERNET) != 0) (*powernets)++;
		if ((net->flags&GROUNDNET) != 0) (*groundnets)++;
	}

	return(pcomplist);
}

/*
 * routine to build a linked list of pseudocomponents in facet "facet".  The
 * list is appended to "initiallist" and returned.  The values of "power" and
 * "ground" are the PNETs of such components.  Routine increments the
 * integer at "components" for every component created.  If
 * "compare_hierarchically" is nonzero, net is flattened.  Returns zero on error.
 */
PCOMP *net_buildpseudo(NODEPROTO *facet, PCOMP *initiallist, PNET **power,
	PNET **ground, INTBIG *components, PNET **pnetlist, INTSML compare_hierarchically,
	INTSML useequivs, INTSML ignorepwrgnd, INTSML checkfacetoverrides)
{
	REGISTER PCOMP *pcomp;
	REGISTER PORTPROTO *pp, *opp, *realpp;
	REGISTER PORTARCINST *pi;
	REGISTER PORTEXPINST *pe;
	REGISTER ARCINST *ai;
	REGISTER NETWORK *net;
	REGISTER NODEINST *ni;
	REGISTER INTBIG fun, i, j, k, sorted, toldshort, toplevel, flattenit, lambda,
		localignorepwrgnd, hasisolated;
	INTBIG width, length;
	REGISTER NODEPROTO *realnp;
	REGISTER PNET *pn;
	REGISTER VARIABLE *var;
	REGISTER EQUIV *e;
	char *extra;

	/* get options */
	localignorepwrgnd = ignorepwrgnd;
	if (checkfacetoverrides != 0)
	{
		var = getvalkey((INTBIG)facet, VNODEPROTO, VINTEGER, net_ncc_optionskey);
		if (var != NOVARIABLE)
		{
			if ((var->addr&NCCIGNOREPWRGNDOVER) != 0)
			{
				if ((var->addr&NCCIGNOREPWRGND) != 0) localignorepwrgnd = 1; else
					localignorepwrgnd = 0;
			}
		}
	}

	if (net_toplevelinst == NONODEINST) toplevel = 1; else
		toplevel = 0;

	/* make simple checks that port characteristics match the name */
	for(pp = facet->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
	{
		if (portisnamedpower(pp) != 0)
		{
			/* port has "pwr" in its name, make sure it is marked power */
			if ((pp->userbits&STATEBITS) != PWRPORT)
				ttyputmsg(_("Warning: export %s in facet %s is named as a power port but has type %s"),
					pp->protoname, describenodeproto(facet), describeportbits(pp));
		}
		if ((pp->userbits&STATEBITS) == PWRPORT)
		{
			/* port is marked power, see if it has an appropriate name */
			if (portisnamedpower(pp) == 0)
				ttyputmsg(_("Warning: power export %s in facet %s is not named appropriately"),
					pp->protoname, describenodeproto(facet));
		}
		if (portisnamedground(pp) != 0)
		{
			/* port has "ground" in its name, make sure it is marked ground */
			if ((pp->userbits&STATEBITS) != GNDPORT)
				ttyputmsg(_("Warning: export %s in facet %s is named as a ground port but has type %s"),
					pp->protoname, describenodeproto(facet), describeportbits(pp));
		}
		if ((pp->userbits&STATEBITS) == GNDPORT)
		{
			/* port is marked ground, see if it has an appropriate name */
			if (portisnamedground(pp) == 0)
				ttyputmsg(_("Warning: ground export %s in facet %s is not named appropriately"),
					pp->protoname, describenodeproto(facet));
		}
	}

	/* place pseudo-netnumbers on all networks, taken from ports */
	for(net = facet->firstnetwork; net != NONETWORK; net = net->nextnetwork)
		net->temp1 = (INTBIG)NOPNET;
	for(pp = facet->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
	{
		for(net = facet->firstnetwork; net != NONETWORK; net = net->nextnetwork)
		{
			if (net != pp->network) continue;
			net->temp1 = pp->temp1;
			break;
		}
	}

	/* spread power and ground information from appropriate nodes */
	for(ni = facet->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
	{
		if (ni->proto->primindex == 0) continue;
		fun = net_getfunction(ni);
		if (fun != NPCONPOWER && fun != NPCONGROUND) continue;
		if (fun == NPCONPOWER)
		{
			pn = *power;
			if (pn == NOPNET)
			{
				if (localignorepwrgnd != 0) pn = net_allocpnet(); else
					pn = net_newpnet(pnetlist);
				pn->realport = NOPORTPROTO;
				pn->network = NONETWORK;
				pn->flags = POWERNET;
				*power = pn;
			}
		}
		if (fun == NPCONGROUND)
		{
			pn = *ground;
			if (pn == NOPNET)
			{
				if (localignorepwrgnd != 0) pn = net_allocpnet(); else
					pn = net_newpnet(pnetlist);
				pn->realport = NOPORTPROTO;
				pn->network = NONETWORK;
				pn->flags = GROUNDNET;
				*ground = pn;
			}
		}
		for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
		{
			ai = pi->conarcinst;
			if (ai->network->temp1 != (INTBIG)pn)
			{
				if (*power != NOPNET && *ground != NOPNET &&
					((pn == *power && (PNET *)ai->network->temp1 == *ground) ||
					 (pn == *ground && (PNET *)ai->network->temp1 == *power)))
				{
					ttyputerr(_("Warning: power/ground connected incorrectly in facet %s"),
						describenodeproto(facet));
				}
			}
			ai->network->temp1 = (INTBIG)pn;
		}
	}

	/* generate new pseudo-netnumbers for networks not connected to ports */
	for(net = facet->firstnetwork; net != NONETWORK; net = net->nextnetwork)
	{
		if (net->temp1 != (INTBIG)NOPNET) continue;
		pn = net_newpnet(pnetlist);
		if (pn == NOPNET) return(0);
		net->temp1 = (INTBIG)pn;
		pn->realport = NOPORTPROTO;
		pn->network = net;
	}

	/* search every component in the facet */
	toldshort = 0;
	for(ni = facet->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
	{
		if (toplevel != 0) net_toplevelinst = ni;

		/* if flattening the circuit, explore contents of facet instances */
		flattenit = 0;
		if (ni->proto->primindex == 0 && compare_hierarchically != 0) flattenit = 1;

		if (flattenit != 0 && useequivs != 0)
		{
			/* see if this is equivalenced */
			for(e = net_firstequiv; e != NOEQUIV; e = e->nextequiv)
			{
				/* check the validity of the equivalence */
				if (e->first == ni || e->second == ni) flattenit = 0;
			}
		}

		if (flattenit != 0)		
		{
			/* if there is an alternate contents facet, use it */
			realnp = contentsview(ni->proto);
			if (realnp == NONODEPROTO) realnp = ni->proto;

			/* put pseudo-netnumbers on the edge of this facet */
			for(realpp = realnp->firstportproto; realpp != NOPORTPROTO; realpp = realpp->nextportproto)
				realpp->temp1 = (INTBIG)NOPNET;
			for(realpp = realnp->firstportproto; realpp != NOPORTPROTO; realpp = realpp->nextportproto)
			{
				if (realpp->temp1 != (INTBIG)NOPNET) continue;

				/* if there is an alternate contents facet, compute the port */
				if (realnp == ni->proto) pp = realpp; else
				{
					pp = equivalentport(realnp, realpp, ni->proto);
					if (pp == NOPORTPROTO) pp = ni->proto->firstportproto;
				}

				/* see if an arc connects to the port */
				for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
					if (pi->proto->network == pp->network) break;
				if (pi != NOPORTARCINST) realpp->temp1 = pi->conarcinst->network->temp1; else
				{
					/* see if the port is an export */
					for(pe = ni->firstportexpinst; pe != NOPORTEXPINST; pe = pe->nextportexpinst)
						if (pe->proto->network == pp->network) break;
					if (pe != NOPORTEXPINST) realpp->temp1 = pe->exportproto->temp1; else
					{
						pn = net_newpnet(pnetlist);
						if (pn == NOPNET) return(0);
						pn->realport = NOPORTPROTO;
						pn->network = pp->network;
						if ((pp->userbits&STATEBITS) == PWRPORT)
						{
							pn->flags = POWERNET;
						} else if ((pp->userbits&STATEBITS) == GNDPORT)
						{
							pn->flags = GROUNDNET;
						}
						pn->flags |= EXPORTEDNET;
						realpp->temp1 = (INTBIG)pn;
					}
				}

				/* propagate net numbers to other connected ports */
				for(opp = realpp->nextportproto; opp != NOPORTPROTO; opp = opp->nextportproto)
					if (opp->network == realpp->network)
						opp->temp1 = realpp->temp1;
			}

			/* recurse into the facet */
			downhierarchy(ni);
			initiallist = net_buildpseudo(realnp, initiallist, power, ground,
				components, pnetlist, compare_hierarchically, useequivs,
					ignorepwrgnd, checkfacetoverrides);
			uphierarchy();
			if (initiallist == 0) return(0);
			continue;
		}

		/* nonflattenable component: add it to the pseudocomponent list */
		if (ni->proto->primindex == 0) fun = NPUNKNOWN; else
		{
			fun = net_getfunction(ni);
			if (fun == NPCONNECT || fun == NPART || fun == NPUNKNOWN) continue;

			/* allow only one power and one ground component */
			if (fun == NPCONPOWER && *power != NOPNET) continue;
			if (fun == NPCONGROUND && *ground != NOPNET) continue;
		}

		/* ignore power and ground components if requested */
		if (localignorepwrgnd != 0 &&
			(fun == NPCONPOWER || fun == NPCONGROUND)) continue;

		/* create a pseudo-component */
		pcomp = net_allocpcomp();
		if (pcomp == NOPCOMP) return(0);
		pcomp->nextpcomp = initiallist;
		initiallist = pcomp;
		pcomp->id = -1;
		pcomp->function = (INTSML)fun;

		/* special case: ignore 4th port on 4-port transistors */
		switch (fun)
		{
			case NPTRA4NMOS:  pcomp->function = NPTRANMOS;   break;
			case NPTRA4DMOS:  pcomp->function = NPTRADMOS;   break;
			case NPTRA4PMOS:  pcomp->function = NPTRAPMOS;   break;
			case NPTRA4NPN:   pcomp->function = NPTRANPN;    break;
			case NPTRA4PNP:   pcomp->function = NPTRAPNP;    break;
			case NPTRA4NJFET: pcomp->function = NPTRANJFET;  break;
			case NPTRA4PJFET: pcomp->function = NPTRAPJFET;  break;
			case NPTRA4DMES:  pcomp->function = NPTRADMES;   break;
			case NPTRA4EMES:  pcomp->function = NPTRAEMES;   break;
		}

		pcomp->hassize = 0;
		pcomp->actuallist = (void *)ni;
		pcomp->numactual = 1;
		pcomp->topactual = net_toplevelinst;
		(*components)++;
		switch (fun)
		{
			case NPTRANMOS:  case NPTRA4NMOS:
			case NPTRADMOS:  case NPTRA4DMOS:
			case NPTRAPMOS:  case NPTRA4PMOS:
			case NPTRADMES:  case NPTRA4DMES:
			case NPTRAEMES:  case NPTRA4EMES:
				/* transistors that have a length and a width */
				transistorsize(ni, &length, &width);
				lambda = facet->cell->lib->lambda[facet->tech->techindex];
				pcomp->length = (float)(length * WHOLE / lambda);
				pcomp->width = (float)(width * WHOLE / lambda);
				pcomp->hassize = 1;
				break;

			case NPTRANPN:    case NPTRA4NPN:
			case NPTRAPNP:    case NPTRA4PNP:
			case NPTRANJFET:  case NPTRA4NJFET:
			case NPTRAPJFET:  case NPTRA4PJFET:
				/* transistors that have an area */
				nodefunction(ni, &extra);
				if (extra == 0) pcomp->length = 0; else
					pcomp->length = (float)atofr(extra);
				pcomp->width = 0.0;
				pcomp->hassize = 1;
				break;

			case NPRESIST:
			case NPCAPAC:   case NPECAPAC:
			case NPDIODE:   case NPDIODEZ:
			case NPINDUCT:
				pcomp->length = net_getpartvalue(ni);
				pcomp->width = 0.0;
				pcomp->hassize = 2;
				break;
		}

		/* count the number of electrically distinct nets on the component */
		pcomp->wirecount = 0;
		hasisolated = 0;
		for(pp = ni->proto->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
		{
			/* ignore power and ground ports if requested */
			if (localignorepwrgnd != 0 &&
				((pp->userbits&STATEBITS) == PWRPORT || (pp->userbits&STATEBITS) == GNDPORT)) continue;

			/* special case for isolated ports */
			if ((pp->userbits&PORTISOLATED) != 0)
			{
				/* add one wire for each arc on the port */
				for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
					if (pi->proto == pp) pcomp->wirecount++;
				hasisolated++;
				continue;
			}

			/* see if this port is equivalent to a previous one */
			for(opp = ni->proto->firstportproto; opp != pp; opp = opp->nextportproto)
				if (opp->network == pp->network) break;
			if (opp != pp) continue;

			/* new port */
			pcomp->wirecount++;
		}

		/* special case: 4-port transistors are treated as just 3 */
		switch (fun)
		{
			case NPTRA4NMOS:
			case NPTRA4DMOS:
			case NPTRA4PMOS:
			case NPTRA4NPN:
			case NPTRA4PNP:
			case NPTRA4NJFET:
			case NPTRA4PJFET:
			case NPTRA4DMES:
			case NPTRA4EMES:
				pcomp->wirecount--;
		}

		/* allocate the port and connection lists */
		if (pcomp->wirecount != 0)
		{
			pcomp->pconnarray = (PCOMP ***)emalloc(((sizeof (PCOMP **)) * pcomp->wirecount),
				net_aid->cluster);
			if (pcomp->pconnarray == 0) return(0);
			pcomp->portindexarray = (INTSML **)emalloc(((sizeof (INTSML *)) * pcomp->wirecount),
				net_aid->cluster);
			if (pcomp->portindexarray == 0) return(0);
			pcomp->portlist = (PORTPROTO **)emalloc(((sizeof (PORTPROTO *)) * pcomp->wirecount),
				net_aid->cluster);
			if (pcomp->portlist == 0) return(0);
			pcomp->count = (INTSML *)emalloc((SIZEOFINTSML * pcomp->wirecount),
				net_aid->cluster);
			if (pcomp->count == 0) return(0);
			pcomp->state = (INTSML *)emalloc((SIZEOFINTSML * pcomp->wirecount),
				net_aid->cluster);
			if (pcomp->state == 0) return(0);
			pcomp->portindices = (INTSML *)emalloc((SIZEOFINTSML * pcomp->wirecount),
				net_aid->cluster);
			if (pcomp->portindices == 0) return(0);
			pcomp->netnumbers = (PNET **)emalloc(((sizeof (PNET *)) * pcomp->wirecount),
				net_aid->cluster);
			if (pcomp->netnumbers == 0) return(0);
			for(i=0; i<pcomp->wirecount; i++)
			{
				pcomp->count[i] = 0;
				pcomp->state[i] = 0;
			}

			/* load the ports and normalized indices */
			if (hasisolated != 0)
			{
				/* if the node has isolated ports, handle specially */
				j = 0;   k = hasisolated;   hasisolated = 0;
				for(pp = ni->proto->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
				{
					/* ignore power and ground ports if requested */
					if (localignorepwrgnd != 0 &&
						((pp->userbits&STATEBITS) == PWRPORT || (pp->userbits&STATEBITS) == GNDPORT))
							continue;

					/* special case for isolated ports */
					if ((pp->userbits&PORTISOLATED) != 0)
					{
						/* add one wire for each arc on the port */
						for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
						{
							if (pi->proto != pp) continue;
							pcomp->portlist[j] = pp;
							pcomp->portindices[j] = (INTSML)hasisolated;
							pcomp->netnumbers[j] = (PNET *)pi;
							j++;
						}
						hasisolated++;
						continue;
					}

					/* see if this port is equivalent to a previous one */
					for(opp = ni->proto->firstportproto; opp != pp; opp = opp->nextportproto)
						if (opp->network == pp->network) break;
					if (opp != pp) continue;

					/* new port */
					pcomp->portlist[j] = pp;
					pcomp->portindices[j] = (INTSML)k++;
					j++;
				}
			} else
			{
				switch (pcomp->function)
				{
					case NPTRANMOS:
					case NPTRADMOS:
					case NPTRAPMOS:
					case NPTRADMES:
					case NPTRAEMES:
					case NPTRANPN:
					case NPTRAPNP:
					case NPTRANJFET:
					case NPTRAPJFET:
						/* transistors make the active ports equivalent */
						pcomp->portindices[0] = 1;
						pcomp->portindices[1] = 2;
						pcomp->portindices[2] = 2;
						pcomp->portlist[0] = ni->proto->firstportproto;
						pcomp->portlist[1] = pcomp->portlist[0]->nextportproto;
						pcomp->portlist[2] = pcomp->portlist[1]->nextportproto;
						if (ni->proto != sch_transistorprim && ni->proto != sch_transistor4prim)
							pcomp->portlist[2] = pcomp->portlist[2]->nextportproto;
						break;

					case NPTRA4NMOS:
					case NPTRA4DMOS:
					case NPTRA4PMOS:
					case NPTRA4DMES:
					case NPTRA4EMES:
					case NPTRA4NPN:
					case NPTRA4PNP:
					case NPTRA4NJFET:
					case NPTRA4PJFET:
						/* 4-port transistors make the active two equivalent */
						pcomp->portindices[0] = 1;
						pcomp->portindices[1] = 2;
						pcomp->portindices[2] = 2;
						pcomp->portindices[3] = 3;
						pcomp->portlist[0] = ni->proto->firstportproto;
						pcomp->portlist[1] = pcomp->portlist[0]->nextportproto;
						pcomp->portlist[2] = pcomp->portlist[1]->nextportproto;
						pcomp->portlist[3] = pcomp->portlist[2]->nextportproto;
						break;

					default:
						j = 0;
						for(pp = ni->proto->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
						{
							/* ignore power and ground ports if requested */
							if (localignorepwrgnd != 0 &&
								((pp->userbits&STATEBITS) == PWRPORT || (pp->userbits&STATEBITS) == GNDPORT)) continue;

							/* see if this port is equivalent to a previous one */
							for(opp = ni->proto->firstportproto; opp != pp; opp = opp->nextportproto)
								if (opp->network == pp->network) break;
							if (opp != pp) continue;

							/* new port */
							pcomp->portindices[j] = (INTSML)j;
							pcomp->portlist[j] = pp;
							j++;
						}
						if (pcomp->function == NPUNKNOWN)
						{
							/* sort by port name for facets */
							sorted = 0;
							while (sorted == 0)
							{
								sorted = 1;
								for(i=1; i<j; i++)
								{
									if (namesame(pcomp->portlist[i-1]->protoname, pcomp->portlist[i]->protoname) <= 0)
										continue;
									pp = pcomp->portlist[i-1];
									pcomp->portlist[i-1] = pcomp->portlist[i];
									pcomp->portlist[i] = pp;
									sorted = 0;
								}
							}
						}
						break;
				}
			}
		}

		/* set the state and network numbers */
		for(j=0; j<pcomp->wirecount; j++)
		{
			pp = pcomp->portlist[j];

			/* special case for isolated ports */
			if ((pp->userbits&PORTISOLATED) != 0)
			{
				/* pickup the network number of this connection */
				pi = (PORTARCINST *)pcomp->netnumbers[j];
				ai = pi->conarcinst;
				pcomp->netnumbers[j] = (PNET *)ai->network->temp1;
				if ((ai->userbits&ISNEGATED) != 0)
				{
					if ((ai->end[0].portarcinst == pi && (ai->userbits&REVERSEEND) == 0) ||
						(ai->end[1].portarcinst == pi && (ai->userbits&REVERSEEND) != 0))
							pcomp->state[j] = NEGATEDPORT;
				}
				if ((pcomp->netnumbers[j]->flags&EXPORTEDNET) != 0)
					pcomp->state[j] |= EXPORTEDPORT;
				continue;
			}

			/* new wire */
			pcomp->netnumbers[j] = NOPNET;
			for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
				if (pi->proto->network == pp->network)
			{
				/* pickup the network number of this connection */
				ai = pi->conarcinst;
				pcomp->netnumbers[j] = (PNET *)ai->network->temp1;
				if ((ai->userbits&ISNEGATED) != 0)
				{
					if ((ai->end[0].portarcinst == pi && (ai->userbits&REVERSEEND) == 0) ||
						(ai->end[1].portarcinst == pi && (ai->userbits&REVERSEEND) != 0))
							pcomp->state[j] = NEGATEDPORT;
				}
				if (pcomp->netnumbers[j] != NOPNET)
				{
					if ((pcomp->netnumbers[j]->flags&EXPORTEDNET) != 0)
						pcomp->state[j] |= EXPORTEDPORT;
				}
				break;
			}
			if (pcomp->netnumbers[j] == NOPNET)
			{
				for(pe = ni->firstportexpinst; pe != NOPORTEXPINST; pe = pe->nextportexpinst)
					if (pe->proto->network == pp->network)
				{
					pcomp->netnumbers[j] = (PNET *)pe->exportproto->temp1;
					pcomp->state[j] |= EXPORTEDPORT;
					break;
				}
			}
			if (pcomp->netnumbers[j] == NOPNET)
			{
				pn = net_newpnet(pnetlist);
				if (pn == NOPNET) return(0);
				pn->realport = NOPORTPROTO;
				pn->network = NONETWORK;
				pcomp->netnumbers[j] = pn;
			}
		}

		/* for power or ground components, set the characteristics on the nets */
		if (fun == NPCONPOWER || fun == NPCONGROUND)
		{
			for(i=0; i<pcomp->wirecount; i++)
			{
				if (fun == NPCONPOWER)
				{
					if ((pcomp->netnumbers[i]->flags&GROUNDNET) != 0 && toldshort == 0)
					{
						ttyputerr(_("Power and ground are shorted in %s"), describenodeproto(facet));
						toldshort++;
					}
					pcomp->netnumbers[i]->flags |= POWERNET;
					*power = pcomp->netnumbers[i];
				}
				if (fun == NPCONGROUND)
				{
					if ((pcomp->netnumbers[i]->flags&POWERNET) != 0 && toldshort == 0)
					{
						ttyputerr(_("Power and ground are shorted in %s"), describenodeproto(facet));
						toldshort++;
					}
					pcomp->netnumbers[i]->flags |= GROUNDNET;
					*ground = pcomp->netnumbers[i];
				}
			}
		}
	}

	return(initiallist);
}

/*
 * Routine to reduce the network in "pcomp" to merge parallel components.
 */
void net_mergeparallel(PCOMP **pcomp, INTBIG *components)
{
	REGISTER PCOMP *pc, *opc, *nextpc, *lastpc;
	REGISTER INTBIG i, j, newnum;
	REGISTER NODEINST *ni, *oni, **newlist;

	lastpc = NOPCOMP;
	for(pc = *pcomp; pc != NOPCOMP; pc = nextpc)
	{
		nextpc = pc->nextpcomp;
		if (pc->function == NPUNKNOWN) continue;
		for(opc = pc->nextpcomp; opc != NOPCOMP; opc = opc->nextpcomp)
		{
			if (pc->function != opc->function) continue;

			/* compare the wire lists */
			if (net_diffwirelist(pc, opc, NOBUCKET, NOBUCKET, 0) != 0) continue;
#if 0
if (pc->numactual == 1) ni = (NODEINST *)pc->actuallist; else
	ni = ((NODEINST **)pc->actuallist)[0];
if (opc->numactual == 1) oni = (NODEINST *)opc->actuallist; else
	oni = ((NODEINST **)opc->actuallist)[0];
ttyputmsg("Merging nodes %s:%s (top is %s) and %s:%s (top is %s)",
 describenodeinst(ni), describenodeproto(ni->parent), describenodeinst(pc->topactual),
 describenodeinst(oni), describenodeproto(oni->parent), describenodeinst(opc->topactual));
#endif
			/* components are equivalent: delete "pc" */
			net_removepcompfromlists(*pcomp, pc, opc);

			/* add "pc"s node pointer to "opc" */
			for(j=0; j<pc->numactual; j++)
			{
				if (pc->numactual == 1) ni = (NODEINST *)pc->actuallist; else
					ni = ((NODEINST **)pc->actuallist)[j];
				newnum = opc->numactual + 1;
				if (newnum == 1) opc->actuallist = (void *)ni; else
				{
					newlist = (NODEINST **)emalloc(newnum * (sizeof (NODEINST *)),
						net_aid->cluster);
					if (newlist == 0) return;
					for(i=0; i<opc->numactual; i++)
					{
						if (opc->numactual == 1) oni = (NODEINST *)opc->actuallist; else
							oni = ((NODEINST **)opc->actuallist)[i];
						newlist[i] = oni;
					}
					newlist[i] = ni;
					if (opc->numactual > 1) efree((char *)opc->actuallist);
					opc->actuallist = (void *)newlist;
				}
				opc->numactual = newnum;
			}

			/* combine sizes (as specified by Robert Bosnyak) */
			if (pc->hassize != 0 && opc->hassize != 0)
			{
				switch (pc->function)
				{
					case NPTRANMOS:  case NPTRA4NMOS:
					case NPTRADMOS:  case NPTRA4DMOS:
					case NPTRAPMOS:  case NPTRA4PMOS:
						/* FET transistors in parallel depend on whether the length is the same */
						if (opc->length == pc->length)
						{
							/* same-length transistors: sum the width */
							opc->width += pc->width;
						} else
						{
							/* different-length transistors: more complex formula */
							if (pc->width + opc->width != 0.0)
							{
								opc->length = (pc->width * pc->length + opc->width * opc->length) /
									(pc->width + opc->width);
							}
							opc->width += pc->width;
						}
						break;
					case NPTRANPN:   case NPTRA4NPN:
					case NPTRAPNP:   case NPTRA4PNP:
					case NPTRANJFET: case NPTRA4NJFET:
					case NPTRAPJFET: case NPTRA4PJFET:
					case NPTRADMES:  case NPTRA4DMES:
					case NPTRAEMES:  case NPTRA4EMES:
						/* nonFET transistors in parallel sum the area */
						opc->length += pc->length;
						break;
					case NPRESIST:
					case NPINDUCT:
						/* resistance and capacitance in parallel take product over sum */
						if (pc->length + opc->length != 0.0)
							opc->length = (pc->length * opc->length) / (pc->length + opc->length);
						break;
					case NPCAPAC:  case NPECAPAC:
					case NPDIODE:  case NPDIODEZ:
						/* capacitance and diode in parallel sum the farads/area */
						opc->length += pc->length;
						break;
				}
			}

			/* free memory associated with component */
			for(i=0; i<pc->wirecount; i++)
			{
				if (pc->count[i] > 0)
				{
					free((char *)pc->pconnarray[i]);
					free((char *)pc->portindexarray[i]);
				}
			}
			if (pc->wirecount != 0)
			{
				efree((char *)pc->portlist);
				efree((char *)pc->count);
				efree((char *)pc->state);
				efree((char *)pc->netnumbers);
				efree((char *)pc->portindices);
			}
			if (pc->numactual > 1) efree((char *)pc->actuallist);

			if (lastpc == NOPCOMP) *pcomp = pc->nextpcomp; else
				lastpc->nextpcomp = pc->nextpcomp;
			net_freepcomp(pc);
			(*components)--;
			break;
		}
		if (opc != NOPCOMP) continue;
		lastpc = pc;
	}
}

void net_removepcompfromlists(PCOMP *pcomp, PCOMP *removedpc, PCOMP *newpc)
{
	REGISTER PCOMP *pc;
	REGISTER INTBIG i, j, k;

	/* look through every component */
	for(pc = pcomp; pc != NOPCOMP; pc = pc->nextpcomp)
	{
		/* ignore the removed one */
		if (pc == removedpc) continue;

		/* check every connection path from the component */
		for(i=0; i<pc->wirecount; i++)
		{
			/* look at every other component connected on this path */
			for(j=0; j<pc->count[i]; j++)
			{
				if (pc->pconnarray[i][j] != removedpc) continue;

				/* found reference to old component */
				for(k=0; k<pc->count[i]; k++)
					if (pc->pconnarray[i][k] == newpc) break;
				if (k >= pc->count[i] && newpc != pc)
				{
					/* new component not referenced: substitute */
					pc->pconnarray[i][j] = newpc;
				} else
				{
					/* new component is referenced: remove old */
					for(k = j; k < pc->count[i]-1; k++)
					{
						pc->pconnarray[i][k] = pc->pconnarray[i][k+1];
						pc->portindexarray[i][k] = pc->portindexarray[i][k+1];
					}
					pc->count[i]--;
				}
				break;
			}
		}
	}
}

float net_getpartvalue(NODEINST *ni)
{
	REGISTER NODEPROTO *np;
	REGISTER VARIABLE *var;
	REGISTER INTBIG i;
	float value;
	REGISTER char *pt;

	np = ni->proto;
	if (np->primindex == 0) return(0.0);

	/* diodes have area on them */
	if (np == sch_diodeprim)
	{
		var = getvalkey((INTBIG)ni, VNODEINST, -1, sch_diodekey);
		pt = describesimplevariable(var);
		return((float)atof(pt));
	}

	/* capacitors have Farads on them */
	if (np == sch_capacitorprim)
	{
		var = getvalkey((INTBIG)ni, VNODEINST, -1, sch_capacitancekey);
		pt = describesimplevariable(var);
		value = (float)atof(pt);
		i = strlen(pt);
		if (i > 0)
		{
			if (tolower(pt[i-1]) == 'f')
			{
				value = value / 1000000000000.0f;
			} else if (tolower(pt[i-1]) == 'p')
			{
				value = value / 1000000000.0f;
			} else if (tolower(pt[i-1]) == 'u')
			{
				value = value / 1000000.0f;
			} else if (tolower(pt[i-1]) == 'm')
			{
				value = value / 1000.0f;
			}
		}
		return(value);
	}

	/* resistors have Ohms on them */
	if (np == sch_resistorprim)
	{
		var = getvalkey((INTBIG)ni, VNODEINST, -1, sch_resistancekey);
		pt = describesimplevariable(var);
		value = (float)atof(pt);
		i = strlen(pt);
		if (i > 0)
		{
			if (tolower(pt[i-1]) == 'g')
			{
				value = value * 1000000000.0f;
			} else if (i > 2 && namesame(&pt[i-3], "meg") == 0)
			{
				value = value * 1000000.0f;
			} else if (tolower(pt[i-1]) == 'k')
			{
				value = value * 1000.0f;
			}
		}
		return(value);
	}

	/* inductors have Henrys on them */
	if (np == sch_inductorprim)
	{
		var = getvalkey((INTBIG)ni, VNODEINST, -1, sch_inductancekey);
		pt = describesimplevariable(var);
		value = (float)atof(pt);
		i = strlen(pt);
		if (i > 0)
		{
			if (tolower(pt[i-1]) == 'u')
			{
				value = value / 1000000.0f;
			} else if (tolower(pt[i-1]) == 'm')
			{
				value = value / 1000.0f;
			}
		}
		return(value);
	}
	return(0.0);
}

/*********************** ALLOCATION ***********************/

/*
 * routine to create a new PNET module and increase the net number count in the
 * global "net_pseudonode", add the module to the list in "pnetlist", and
 * return the module.  Returns NOPNET on error.
 */
PNET *net_newpnet(PNET **pnetlist)
{
	REGISTER PNET *p;

	p = net_allocpnet();
	if (p == 0) return(NOPNET);
	net_pseudonode++;
	p->nextpnet = *pnetlist;
	*pnetlist = p;
	return(p);
}

/*
 * routine to allocate a new pcomp module from the pool (if any) or memory
 */
PCOMP *net_allocpcomp(void)
{
	REGISTER PCOMP *p;

	if (net_pcompfree == NOPCOMP)
	{
		p = (PCOMP *)emalloc(sizeof (PCOMP), net_aid->cluster);
		if (p == 0) return(NOPCOMP);
	} else
	{
		p = net_pcompfree;
		net_pcompfree = (PCOMP *)p->nextpcomp;
	}
	return(p);
}

/*
 * routine to return pcomp module "p" to the pool of free modules
 */
void net_freepcomp(PCOMP *p)
{
	p->nextpcomp = net_pcompfree;
	net_pcompfree = p;
}

/*
 * routine to allocate a new pnet module from the pool (if any) or memory
 */
PNET *net_allocpnet(void)
{
	REGISTER PNET *p;

	if (net_pnetfree == NOPNET)
	{
		p = (PNET *)emalloc(sizeof (PNET), net_aid->cluster);
		if (p == 0) return(NOPNET);
	} else
	{
		p = net_pnetfree;
		net_pnetfree = (PNET *)p->nextpnet;
	}

	p->flags = 0;
	p->realport = NOPORTPROTO;
	return(p);
}

/*
 * routine to return pnet module "p" to the pool of free modules
 */
void net_freepnet(PNET *p)
{
	p->nextpnet = net_pnetfree;
	net_pnetfree = p;
}

/*
 * routine to free all allocated structures in the list of pseudonets
 * headed by "pnetlist"
 */
void net_freeallpnet(PNET *pnetlist)
{
	REGISTER PNET *p, *nextp;

	for(p = pnetlist; p != NOPNET; p = nextp)
	{
		nextp = p->nextpnet;
		net_freepnet(p);
	}
}

/*
 * routine to free all allocated structures in the list of pseudocomponents
 * headed by "tp"
 */
void net_freeallpcomp(PCOMP *tp)
{
	REGISTER PCOMP *p, *nextp;
	REGISTER INTSML i;

	for(p = tp; p != NOPCOMP; p = nextp)
	{
		nextp = p->nextpcomp;

		for(i=0; i<p->wirecount; i++)
		{
			if (p->count[i] > 0)
			{
				efree((char *)p->pconnarray[i]);
				efree((char *)p->portindexarray[i]);
			}
		}
		if (p->wirecount != 0)
		{
			efree((char *)p->pconnarray);
			efree((char *)p->portindexarray);
			efree((char *)p->portlist);
			efree((char *)p->count);
			efree((char *)p->state);
			efree((char *)p->netnumbers);
			efree((char *)p->portindices);
		}
		if (p->numactual > 1) efree((char *)p->actuallist);
		net_freepcomp(p);
	}
}

/*********************** COMPONENT FUNCTION ***********************/

#define NOTRANMODEL ((TRANMODEL *)-1)

typedef struct Itranmodel
{
	char *modelname;
	INTSML tmindex;
	struct Itranmodel *nexttranmodel;
} TRANMODEL;

static TRANMODEL *net_firsttranmodel = NOTRANMODEL;

/* must be larger than largest node function entry in "efunction.h" */
static INTSML net_tranmodelindex = 100;

/*
 * routine to return the function of node "ni"
 */
INTBIG net_getfunction(NODEINST *ni)
{
	REGISTER INTBIG fun;
	REGISTER PORTEXPINST *pe;
	REGISTER TRANMODEL *tm;

	fun = nodefunction(ni, 0);
	switch (fun)
	{
		case NPTRANS:
			fun = NPTRANMOS;
			break;
		case NPTRANS4:
			fun = NPTRA4NMOS;
			break;

		case NPTRANSREF:
			/* self-referential transistor: lookup the string in the table */
			for(tm = net_firsttranmodel; tm != NOTRANMODEL; tm = tm->nexttranmodel)
				if (namesame(tm->modelname, ni->proto->primname) == 0) break;
			if (tm == NOTRANMODEL)
			{
				/* new table entry */
				tm = (TRANMODEL *)emalloc(sizeof (TRANMODEL), net_aid->cluster);
				if (tm == 0) break;
				(void)allocstring(&tm->modelname, ni->proto->primname, net_aid->cluster);
				tm->tmindex = net_tranmodelindex++;
				tm->nexttranmodel = net_firsttranmodel;
				net_firsttranmodel = tm;
			}
			fun = tm->tmindex;
			break;

		case NPPIN:
		case NPNODE:
		case NPCONTACT:
		case NPWELL:
		case NPSUBSTRATE:
			fun = NPCONNECT;

		case NPCONNECT:
			/* unify the representations of power and ground */
			for(pe = ni->firstportexpinst; pe != NOPORTEXPINST; pe = pe->nextportexpinst)
			{
				if (portispower(pe->exportproto) != 0) fun = NPCONPOWER; else
					if (portisground(pe->exportproto) != 0) fun = NPCONGROUND;
			}
			break;
	}
	return(fun);
}
