;;; -*- Mode: LISP; Syntax: Common-Lisp; Package: OPAL; Base: 10 -*-
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;         The Garnet User Interface Development Environment.      ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; This code was written as part of the Garnet project at          ;;;
;;; Carnegie Mellon University, and has been placed in the public   ;;;
;;; domain.  If you are using this code or any part of Garnet,      ;;;
;;; please contact garnet@cs.cmu.edu to be put on the mailing list. ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;	Begin Change Log
;;;-------------------------------------------------------------------------
;;;
;;;     Dzg/Mickish 21-Sep-92 Added Update-Start-Fn and Update-Stop-Fn
;;;     Myers   20-Aug-92 Added running-main-event-loop-process-elsewhere-p
;;;     Almond  26-May-92 Added patch to launch-main-event-loop-process
;;;			  to handle background streams for Lapidary.
;;;     Pervin  21-Apr-92 Added main-event-loop-process-running-p
;;;     Pervin  14-Apr-92 Uncommented out process code.
;;;			  Got it to work on HP.
;;;     Pervin  30-Mar-92 Commented out process code.
;;;     Pervin  25-Mar-92 Made to be permanent part of Opal.
;;;			  Merged process-allegro and process-lucid.   
;;;     Pervin   9-Aug-90 Released for Garnet.
;;;	Stork	18-Jul-90 Created.
;;;
;;;-------------------------------------------------------------------------
;;;	End Change Log
;;;
;;;
(in-package "OPAL" :use '("LISP" "KR"))


;;;===========================================================================
;;;
;;;  Global variables
;;;
;;;===========================================================================

(defvar update-locking-p T
  "If T, uses process locks to keep Update in a process from interrupting
   itself in a different process.")

(defparameter *main-event-loop-process* nil
  "The variable which is a handle to the main-event-loop process.")

(defparameter *update-lock*
  #+ALLEGRO (mp:make-process-lock :name "UPDATE-LOCK")
  #-ALLEGRO NIL)


;;;===========================================================================
;;;
;;;  Define opal:launch-main-event-loop-process
;;;
;;;===========================================================================

#+allegro
(defun launch-main-event-loop-process ()
  "Spawn a process which is doing Garnet interaction all of the time.
   RETURN the process."
   (when (eq (type-of *main-event-loop-process*) 'mp:process)
      (mp:process-kill *main-event-loop-process*))
   (setf *main-event-loop-process*
          (mp:process-run-restartable-function
	    #+(or allegro-v4.0 allegro-v4.1)
 	    `(:name "Garnet event loop"
              :initial-bindings
              ,(acons '*terminal-io* *terminal-io*
                       excl:*cl-default-special-bindings*))
	    #-(or allegro-v4.0 allegro-v4.1)
	    "Garnet event loop"
            #'(lambda (t-io)
              ;; RGA --- This gets around a "feature" of Allegro 4.1
              ;; which does not allow background processes to do io.
              ;; Note that the main process function is now a function
              ;; of one arg which is bound to *terminal-io*
                (setq *terminal-io* t-io)
                (setq *query-io* t-io)
                (setq *standard-input* t-io)
                (setq *standard-output* t-io)
		(setq *error-output* t-io)
	      ;; Don't bind *debug-io* because RGA suggests other problems
	      ;; might arise
		;(setq *debug-io* t-io)
              ;; first, throw away any pending events
                (xlib:event-case (opal::*default-x-display* :discard-p t :timeout 1)
                  (:destroy-notify () NIL)  ; get rid of warnings
                  (otherwise () t))
                (loop
                  (inter::default-event-handler opal::*default-x-display*)))
            *terminal-io*))
    (setf (mp:process-priority *main-event-loop-process*) 1)
  *main-event-loop-process*)

;;; This was needed in order to survive the debugger on the HP.
;;; There is probably something wrong with Lucid on the HP, since this is needed.
#+lucid
(unless (fboundp 'lucid::machine-specific-restart-current-process)
  (defun lucid::machine-specific-restart-current-process ()))

#+lucid
(defun launch-main-event-loop-process ()
  "Spawn a process which is doing Garnet interaction all of the time.
   RETURN the process."
  ;; If there was already a process running, kill it.
  (when (and *main-event-loop-process*
             (lcl:processp *main-event-loop-process*))
    (lcl:kill-process *main-event-loop-process*))
  (setf *main-event-loop-process*
          (lcl:make-process
	    :name "Garnet event loop"
	    :priority 50
            :function 
	       #'(lambda ()
		   ;; first, throw away any pending events
	           (xlib:event-case (opal::*default-x-display* :discard-p t :timeout 1)
		     (:destroy-notify () NIL)  ; get rid of warnings
		     (otherwise () t))
		   (lcl:handler-bind
		     ((lcl::error #'lcl:invoke-debugger))
	 	     (loop
		       (inter::default-event-handler
			  opal::*default-x-display*)))))))


#-(or allegro lucid)
(defun launch-main-event-loop-process ())


;;;===========================================================================
;;;
;;;  Define opal:kill-main-event-loop-process
;;;
;;;===========================================================================

#+allegro
(defun kill-main-event-loop-process ()
  "
  Kill the current main-event-loop process.
  "
  (when (eq (type-of *main-event-loop-process*) 'mp:process)
    (mp:process-kill *main-event-loop-process*)
    (setf *main-event-loop-process* nil)))

#+lucid
(defun kill-main-event-loop-process ()
  "
  Kill the current main-event-loop process.
  "
  (when (and *main-event-loop-process*
             (lcl:processp *main-event-loop-process*))
    (lcl:kill-process *main-event-loop-process*)
    (setf *main-event-loop-process* nil)))


#-(or allegro lucid)
(defun kill-main-event-loop-process ())

;;;===========================================================================
;;;
;;;  Define running-p functions
;;;
;;;===========================================================================

(defun main-event-loop-process-running-p ()
  (and opal::*main-event-loop-process*
       #+lucid
       (not (equal "Run"
		   (lcl:process-whostate
			opal::*main-event-loop-process*)))
       ;;; Franz's comments about mp:process-runnable-p:  It is true of any
       ;;; process that has a stack-group (meaning that is has been reset and
       ;;; has not yet exhausted its computation), has at least one run reason,
       ;;; has zero arrest reasons, and is not blocked in a call like
       ;;; PROCESS-WAIT or any of its close relatives.  This last clause --
       ;;; testing that the process is not blocked in PROCESS-WAIT --
       ;;; perhaps isn't what you want.  If the process happens temporarily
       ;;; to be waiting for something, it won't be killed.  Perhaps you
       ;;; want to use the PROCESS-ACTIVE-P predicate instead, which
       ;;; is true whether or not the process is in a PROCESS-WAIT.
       #+allegro
       (not (mp:process-runnable-p
		opal::*main-event-loop-process*))))

(defun running-main-event-loop-process-elsewhere-p ()
  (and opal::*main-event-loop-process*
       (not (eq opal::*main-event-loop-process*
		#+allegro mp:*current-process*
		#+lucid user::*current-process*
		#-(or allegro lucid) T)
	    )))


;;;===========================================================================
;;;
;;;  Define process lock functions
;;;
;;;===========================================================================

(defun update-start-fn (window)
  (declare (ignore window))
  #+ALLEGRO
  (if update-locking-p
      (unless (eq (mp:process-lock-locker *update-lock*) mp:*current-process*)
	;; Lock only if lock is held by a different process, or unlocked.
	(mp:process-lock *update-lock*)))
  #+LUCID
  (if update-locking-p
      (lcl:process-lock *update-lock*)))

(defun update-stop-fn (window)
  (declare (ignore window))
  #+ALLEGRO
  (if (and update-locking-p
	   (eq (mp:process-lock-locker *update-lock*) mp:*current-process*))
      (mp:process-unlock *update-lock*))
  #+LUCID
  (if update-locking-p
      (lcl:process-unlock *update-lock* user::*current-process* :ignore)))
