// Copyright (C) 2005 Open Source Telecom Corp.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "engine.h"

using namespace ost;
using namespace std;

BayonneDriver *BayonneDriver::firstDriver = NULL;
BayonneDriver *BayonneDriver::lastDriver = NULL;
Semaphore BayonneDriver::oink(0);

BayonneDriver::BayonneDriver(Keydata::Define *pairs, const char *cfg, const char *id, bool pri) :
Keydata(pairs, cfg), Mutex()
{
	name = id;
	running = false;
	msgport = NULL;
	logevents = NULL;

	avail = 0;
	firstIdle = lastIdle = NULL;

	flash_timer = 250;
	pickup_timer = hangup_timer = 160;
	seize_timer = 1000;
	ring_timer = 6000;
	reset_timer = 60000;
	release_timer = 60000;
	interdigit_timer = 80;
	answer_count = 1;

	audio_priority = 0;
	audio_stack = 0;
	audio_level = 100;

	if(pri)
	{
		nextDriver = firstDriver;
		firstDriver = this;
		if(!lastDriver)
			lastDriver = this;
	}
	else
	{		
		if(firstDriver)
		{
			lastDriver->nextDriver = this;
			lastDriver = this;
		}
		else
			firstDriver = lastDriver = this;

		nextDriver = NULL;
	}

	timeslot = getTimeslotsUsed();
	count = 0;
	span = getSpansUsed();
	spans = 0;
}

BayonneDriver::~BayonneDriver()
{
	if(running)
		stopDriver();

	if(msgport)
	{
		delete msgport;
		msgport = NULL;
	}
}

void BayonneDriver::del(BayonneSession *session)
{
	BayonneDriver *driver = session->driver;

	if(!session->isAvail)
		return;

	driver->enter();
	if(driver->firstIdle == session)
		driver->firstIdle = session->nextIdle;
	if(driver->lastIdle == session)
		driver->lastIdle = session->prevIdle;
	if(session->nextIdle)
		session->nextIdle->prevIdle = session->prevIdle;
	if(session->prevIdle)
		session->prevIdle->nextIdle = session->nextIdle;
	session->isAvail = false;
	--driver->avail;
	--idle_count;
	driver->leave();
}

void BayonneDriver::add(BayonneSession *session)
{
	BayonneDriver *driver = session->driver;

	if(session->isAvail)
		return;

	session->prevIdle = driver->lastIdle;
	session->nextIdle = NULL;

	driver->enter();
	if(!driver->firstIdle)
		driver->firstIdle = driver->lastIdle = session;
	else
	{
		driver->lastIdle->nextIdle = session;
		driver->lastIdle = session;
	}
	++driver->avail;
	session->isAvail = true;
	driver->leave();
	if(++idle_count > idle_limit)
		idle_limit = idle_count;
	if(idle_count == idle_limit && shutdown_flag)
	{
		shutdown_flag = false;
#ifndef	WIN32
		raise(SIGTERM);
#endif
	}		
}

BayonneSpan *BayonneDriver::getSpan(unsigned id)
{
	if(id >= spans)
		return NULL;

	return BayonneSpan::get(span + id);
}

BayonneSession *BayonneDriver::getTimeslot(timeslot_t ts)
{
	if(ts >= count)
		return NULL;

	return getSession(timeslot + ts);
}

const char *BayonneDriver::registerScript(ScriptImage *img, Line *line)
{
	return "selected driver does not support registration";
}

const char *BayonneDriver::assignScript(ScriptImage *img, Line *line)
{
	return "selected driver does not support assignments";
}

void BayonneDriver::startDriver(void)
{
	oink.wait();		// wait for the pig to squeal
	running = true;
	time(&start_time);
}

void BayonneDriver::stopDriver(void)
{
	Event event;
	BayonneSession *session;

	if(!running)
		return;

        if(msgport)
        {
                delete msgport;
                msgport = NULL;
        }

	while(count--)
	{
		session = getSession(timeslot + count);
		if(session)
		{
			memset(&event, 0, sizeof(event));
			event.id = SYSTEM_DOWN;
//			session->putEvent(&event);
			delete session;
		}
	}

	count = 0;
	running = false;
}

BayonneDriver *BayonneDriver::get(const char *id)
{
	BayonneDriver *driver = firstDriver;

	if(!id)
		return driver;

	while(driver)
	{
		if(!stricmp(driver->name, id))
			break;
		driver = driver->nextDriver;
	}
	return driver;
}

unsigned BayonneDriver::list(char **items, unsigned max)
{
	BayonneDriver *driver = firstDriver;
	unsigned count = 0;

	while(driver && max--)
	{
		items[count++] = (char *)driver->name;
		driver = driver->nextDriver;
	}
	items[count] = NULL;
	return count;	
}
	
void BayonneDriver::start(void)
{
	BayonneDriver *driver = firstDriver;
	while(driver)
	{
		if(!driver->running)
			driver->startDriver();
		driver = driver->nextDriver;
	}
	BayonneSpan::allocate();
}

void BayonneDriver::stop(void) 
{
        BayonneDriver *driver = firstDriver;
	
        while(driver)
        {
		if(driver->running)
	                driver->stopDriver();

		driver->running = false;
                driver = driver->nextDriver; 
        }
}

BayonneSession *BayonneDriver::getIdle(void)
{
	BayonneSession *session = NULL;

	enter();
	if(firstIdle)
	{
		session	= firstIdle;
		firstIdle = session->nextIdle;
		session->isAvail = false;
	}
	leave();
	return session;
}

	
